/* SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * Copyright:
 *   2021      Kunwar Maheep Singh <kunwar.maheep@students.iiit.ac.in>
 */

#define SIMDE_TEST_X86_AVX512_INSN 4dpwssds

#include <test/x86/avx512/test-avx512.h>
#include <simde/x86/avx512/4dpwssds.h>

static int
test_simde_mm512_4dpwssds_epi32 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const int32_t src[16];
    const int32_t a0[16];
    const int32_t a1[16];
    const int32_t a2[16];
    const int32_t a3[16];
    const int32_t b[4];
    const int32_t r[16];
  } test_vec[] = {
    { {  INT32_C(  1780258707), -INT32_C(  1866417388),  INT32_C(  2059723712), -INT32_C(   131391471),  INT32_C(     2555959),  INT32_C(  1994382078),  INT32_C(  1200723961),  INT32_C(  1076071852),
         INT32_C(   967458596), -INT32_C(  1144427525),  INT32_C(  1597345358), -INT32_C(   447258194),  INT32_C(  1608941153),  INT32_C(  1389741401),  INT32_C(   144270940),  INT32_C(   407420148) },
      { -INT32_C(   145624068), -INT32_C(  1397613730),  INT32_C(  1460397993), -INT32_C(  1438883000),  INT32_C(   973677281),  INT32_C(  1133305831),  INT32_C(   961291589), -INT32_C(   565013278),
        -INT32_C(   438983801),  INT32_C(  1737590718), -INT32_C(  1212244626), -INT32_C(   513672448),  INT32_C(    85748253), -INT32_C(  1891063735), -INT32_C(  1329031986), -INT32_C(  1332864471) },
      {  INT32_C(  2090230718),  INT32_C(  1491347434), -INT32_C(  1005608252), -INT32_C(  1146720099),  INT32_C(   616612571),  INT32_C(   951257194), -INT32_C(   957842275),  INT32_C(  1417049750),
        -INT32_C(  1009709863), -INT32_C(   132336332), -INT32_C(   172151977),  INT32_C(  2008048540), -INT32_C(  1885638619),  INT32_C(   365449080),  INT32_C(  1641787595),  INT32_C(    11948583) },
      { -INT32_C(  1832614050), -INT32_C(  1819549636), -INT32_C(  1484240885), -INT32_C(   803325781),  INT32_C(   559921832), -INT32_C(   734648567), -INT32_C(    13298984), -INT32_C(  1023415452),
        -INT32_C(  1370111118), -INT32_C(  1354571613), -INT32_C(   749286872), -INT32_C(  1415351037),  INT32_C(   952893999),  INT32_C(    34341674),  INT32_C(  2030125333), -INT32_C(  1623523283) },
      {  INT32_C(  1733202116), -INT32_C(  1743351696),  INT32_C(  1567321434),  INT32_C(   285806306),  INT32_C(   994694417), -INT32_C(   314747432), -INT32_C(  1016709482),  INT32_C(    40084030),
        -INT32_C(  1553288910), -INT32_C(  1690599359), -INT32_C(   805722131), -INT32_C(   958397771), -INT32_C(  1342101033),  INT32_C(   362626686), -INT32_C(  1143470980), -INT32_C(   658687066) },
      {  INT32_C(   763045868), -INT32_C(  1781942360),  INT32_C(   325435998), -INT32_C(  1680194108) },
      {              INT32_MAX,              INT32_MIN,              INT32_MAX, -INT32_C(   913086168),  INT32_C(    45638764),              INT32_MAX,              INT32_MAX,              INT32_MAX,
                     INT32_MAX,              INT32_MIN,              INT32_MAX,              INT32_MIN,              INT32_MAX,              INT32_MAX,  INT32_C(   113633430),  INT32_C(  1473587499) } },
    { { -INT32_C(   313795729), -INT32_C(  1778194150), -INT32_C(  1823351827),  INT32_C(    57347862), -INT32_C(   567220426), -INT32_C(    59442786),  INT32_C(  2131745211), -INT32_C(  1927550690),
        -INT32_C(   545560891),  INT32_C(  1014332751),  INT32_C(  1875887960),  INT32_C(   208813014), -INT32_C(  1058299358),  INT32_C(  1471963036),  INT32_C(  1456982840),  INT32_C(  2061824693) },
      { -INT32_C(  1470537896),  INT32_C(   904187612),  INT32_C(  1805956245),  INT32_C(   293082863),  INT32_C(  1423008696), -INT32_C(    89354814),  INT32_C(   223380312), -INT32_C(   846777227),
         INT32_C(  1886773395),  INT32_C(  1151687343), -INT32_C(    38778610),  INT32_C(   386803807),  INT32_C(  1298980747), -INT32_C(  1001973652),  INT32_C(   282236827),  INT32_C(  1608341964) },
      { -INT32_C(   372288710), -INT32_C(  1154648915),  INT32_C(   481877437), -INT32_C(  1875589627),  INT32_C(   299737253),  INT32_C(  1406477496), -INT32_C(  2006734916),  INT32_C(  1005010945),
         INT32_C(  1076148115), -INT32_C(   369405653),  INT32_C(   872788783),  INT32_C(   516241785), -INT32_C(  1859149095), -INT32_C(  2082208313), -INT32_C(  1391769684),  INT32_C(   468251272) },
      { -INT32_C(   715387735), -INT32_C(  1916905634), -INT32_C(  2067676405), -INT32_C(   693926147), -INT32_C(   261565911), -INT32_C(  2072818473),  INT32_C(   473005716),  INT32_C(   439818864),
        -INT32_C(  2064674010), -INT32_C(   166613525),  INT32_C(  1836766064), -INT32_C(  2109465255), -INT32_C(   948785936), -INT32_C(  1941182984), -INT32_C(   727155357), -INT32_C(  1108418409) },
      {  INT32_C(  1581440371), -INT32_C(    95136886), -INT32_C(  2140614874), -INT32_C(   603804436),  INT32_C(  1352955224), -INT32_C(  1092751526),  INT32_C(    59934060), -INT32_C(   641630107),
        -INT32_C(   415825059),  INT32_C(  2111933526),  INT32_C(  1207781723),  INT32_C(  1294270453), -INT32_C(   811677580),  INT32_C(   613252023),  INT32_C(  1713839872), -INT32_C(    62920545) },
      {  INT32_C(  1122203371),  INT32_C(  1572848642),  INT32_C(    61193230),  INT32_C(   810666427) },
      { -INT32_C(  1005120063),              INT32_MIN,              INT32_MIN,  INT32_C(   736281318), -INT32_C(   596895566), -INT32_C(  1299493895),              INT32_MAX,              INT32_MIN,
        -INT32_C(   890523333),              INT32_MAX,              INT32_MAX,  INT32_C(  1297721585),              INT32_MIN,              INT32_MAX,              INT32_MAX,              INT32_MAX } },
    { {  INT32_C(  1224732561),  INT32_C(  1785498730),  INT32_C(  1255183531),  INT32_C(  1749421948), -INT32_C(  2002114170), -INT32_C(    68785683), -INT32_C(   520123611), -INT32_C(   451915692),
        -INT32_C(  1456664769),  INT32_C(  1175689883), -INT32_C(  1433344978),  INT32_C(  2031277811), -INT32_C(   318587649),  INT32_C(  1256777765), -INT32_C(   953424013),  INT32_C(  1990998839) },
      { -INT32_C(   434120373), -INT32_C(  1590873229),  INT32_C(   172801303), -INT32_C(  1820041580),  INT32_C(  1082164763), -INT32_C(   510957202), -INT32_C(  2002209200),  INT32_C(  1023300849),
        -INT32_C(  1591534035),  INT32_C(  1749176401), -INT32_C(  1586328051),  INT32_C(   137754349), -INT32_C(   364333700),  INT32_C(  1875628830),  INT32_C(  2079814537), -INT32_C(   172493369) },
      {  INT32_C(  1687607827),  INT32_C(   952948778),  INT32_C(  1423523687), -INT32_C(  1302589899), -INT32_C(   493050684),  INT32_C(    22112376), -INT32_C(  1552135973),  INT32_C(  1368929342),
         INT32_C(   968240654),  INT32_C(  1836155398), -INT32_C(   138327359),  INT32_C(   497622361),  INT32_C(   989808066), -INT32_C(  1992601171), -INT32_C(   684935015), -INT32_C(    97925908) },
      { -INT32_C(   130818062),  INT32_C(   593929313),  INT32_C(  1209673711),  INT32_C(   124109637), -INT32_C(  1237228536),  INT32_C(  1329560757),  INT32_C(   539388724),  INT32_C(   555437871),
        -INT32_C(  1877389778), -INT32_C(   491552781), -INT32_C(   332739161), -INT32_C(  1728868464), -INT32_C(  1454492429), -INT32_C(   436695632),  INT32_C(   654647032), -INT32_C(  1672994706) },
      {  INT32_C(  1647075951), -INT32_C(  2008752159),  INT32_C(  1014263468), -INT32_C(   254449667),  INT32_C(  1285104539), -INT32_C(  1456369231),  INT32_C(   517027504), -INT32_C(   960882345),
         INT32_C(  1562961787),  INT32_C(  1944415430), -INT32_C(   676373798),  INT32_C(  1573356737),  INT32_C(  1504272808), -INT32_C(  1576871182),  INT32_C(  1740690192),  INT32_C(  1747811309) },
      {  INT32_C(   684021090), -INT32_C(  1684296767), -INT32_C(   965588219),  INT32_C(  1998797519) },
      {              INT32_MAX,              INT32_MAX,              INT32_MAX,              INT32_MAX,              INT32_MIN, -INT32_C(  1140728141), -INT32_C(  1748060802), -INT32_C(  2079757434),
                     INT32_MIN,              INT32_MAX,              INT32_MIN,              INT32_MAX, -INT32_C(   858900495),              INT32_MAX, -INT32_C(  1531193712),              INT32_MAX } },
    { { -INT32_C(  1915695973), -INT32_C(  1221536858), -INT32_C(  1826688858), -INT32_C(   839103381),  INT32_C(  1677115809),  INT32_C(  1912508780), -INT32_C(  1405586980),  INT32_C(  1176722347),
        -INT32_C(   824970200),  INT32_C(  1837433799),  INT32_C(  1593943284), -INT32_C(  1859256848),  INT32_C(   737420222), -INT32_C(  1851984972),  INT32_C(   255710308),  INT32_C(  1481990192) },
      {  INT32_C(   455485524),  INT32_C(   545893420),  INT32_C(  1082100304),  INT32_C(  1171369095), -INT32_C(  2072983857),  INT32_C(   487919033),  INT32_C(   288117473),  INT32_C(   124354994),
        -INT32_C(   719155031), -INT32_C(  1930056900), -INT32_C(  1127451339), -INT32_C(   251486943),  INT32_C(   477459043),  INT32_C(  1631160959), -INT32_C(  1905105444), -INT32_C(  1869226778) },
      { -INT32_C(  1469728660), -INT32_C(  1724621981), -INT32_C(   246087472),  INT32_C(    31610781),  INT32_C(  1226659786), -INT32_C(  1112909855), -INT32_C(  1555292740),  INT32_C(  1697899001),
        -INT32_C(    49440615), -INT32_C(  1013562893), -INT32_C(   558503103),  INT32_C(   232757059), -INT32_C(   799605265),  INT32_C(   277741908),  INT32_C(   397662750),  INT32_C(  1434248891) },
      {  INT32_C(  1918011775),  INT32_C(   188147914),  INT32_C(   384429011),  INT32_C(  1898170754),  INT32_C(   440564422), -INT32_C(  1725247365),  INT32_C(  1706090154),  INT32_C(  1136274628),
         INT32_C(  2142637237), -INT32_C(   930419468),  INT32_C(  1507750871),  INT32_C(    46858812), -INT32_C(   149156484), -INT32_C(  2020587555), -INT32_C(   370393051),  INT32_C(   556640108) },
      { -INT32_C(  1465850957), -INT32_C(  1502598449), -INT32_C(   620802403), -INT32_C(   857945520), -INT32_C(  1262224937),  INT32_C(  1698386752), -INT32_C(    11655021), -INT32_C(  2111800369),
         INT32_C(   757776478), -INT32_C(  1999398166),  INT32_C(   962778089),  INT32_C(  1963343517),  INT32_C(  1999227191), -INT32_C(  1327733475),  INT32_C(  1555049357),  INT32_C(    98488486) },
      {  INT32_C(  2066876817), -INT32_C(  1945958749),  INT32_C(  1992713689), -INT32_C(   605303644) },
      {              INT32_MIN,              INT32_MIN,              INT32_MIN, -INT32_C(  2002705142),              INT32_MAX,              INT32_MAX,              INT32_MIN,              INT32_MAX,
                     INT32_MIN,              INT32_MAX,              INT32_MAX,              INT32_MIN,              INT32_MAX,              INT32_MIN, -INT32_C(   136205204),              INT32_MAX } },
    { { -INT32_C(  1303177835),  INT32_C(   123940730),  INT32_C(    23270234),  INT32_C(  1946567139), -INT32_C(   286246838),  INT32_C(   393933630), -INT32_C(    41009063), -INT32_C(  1562871540),
         INT32_C(   139733902), -INT32_C(  1257261221), -INT32_C(  1363774774), -INT32_C(    31277900),  INT32_C(   871109365),  INT32_C(  1598777094), -INT32_C(  1268983385), -INT32_C(   514444206) },
      { -INT32_C(  1142314400),  INT32_C(   745601378),  INT32_C(   534390635), -INT32_C(   669057565),  INT32_C(   353110543),  INT32_C(   427054961), -INT32_C(  2100441040),  INT32_C(  1700995845),
         INT32_C(   790646221), -INT32_C(  1319333818), -INT32_C(  1680787785),  INT32_C(  1131671347),  INT32_C(  1800962041),  INT32_C(   109366742), -INT32_C(  1568058979),  INT32_C(  1091038324) },
      {  INT32_C(  2138122041),  INT32_C(  1865534904),  INT32_C(   923402755), -INT32_C(   361070863), -INT32_C(   732572931),  INT32_C(  1037752735), -INT32_C(  1629527254), -INT32_C(  1981749424),
        -INT32_C(   972467954),  INT32_C(   557201950),  INT32_C(   760758332), -INT32_C(  1139289410),  INT32_C(  1150315941),  INT32_C(  1904306758),  INT32_C(   504324558),  INT32_C(  1453846344) },
      {  INT32_C(  1579003968),  INT32_C(   645944298),  INT32_C(  1381226643),  INT32_C(  1343122347),  INT32_C(   513056472), -INT32_C(   695265784), -INT32_C(  1074487433), -INT32_C(   820667250),
         INT32_C(   925708876),  INT32_C(   425569669),  INT32_C(   829141382), -INT32_C(   192841444),  INT32_C(   521278743), -INT32_C(  1560894933), -INT32_C(   815666623), -INT32_C(   744589434) },
      {  INT32_C(   789236905), -INT32_C(    12032135),  INT32_C(   875606808),  INT32_C(  1126740268), -INT32_C(   228443193),  INT32_C(   513104093), -INT32_C(   907151805),  INT32_C(   396135533),
        -INT32_C(   783899048),  INT32_C(   651267598),  INT32_C(  1834680641),  INT32_C(  2058388403), -INT32_C(  1687416130), -INT32_C(  1363607189),  INT32_C(  1702340600), -INT32_C(  1954802637) },
      { -INT32_C(   933444934), -INT32_C(  1846596272), -INT32_C(   486651345), -INT32_C(  1940082995) },
      {              INT32_MIN, -INT32_C(   801483946), -INT32_C(   906826093),              INT32_MAX, -INT32_C(  1826180388),  INT32_C(    52903383),  INT32_C(  1593655813),              INT32_MIN,
         INT32_C(  1704358571),              INT32_MIN,              INT32_MIN,  INT32_C(   394174660),              INT32_MAX,              INT32_MAX,              INT32_MIN, -INT32_C(  1450102974) } },
    { {  INT32_C(   740804801), -INT32_C(  1025842742), -INT32_C(  1155051128),  INT32_C(   541500517), -INT32_C(  1209490842), -INT32_C(    12003376), -INT32_C(   270448863), -INT32_C(  1233437195),
        -INT32_C(   807230971),  INT32_C(   194100355),  INT32_C(  1942403342), -INT32_C(  1013773219),  INT32_C(  2138733743),  INT32_C(  1971241811), -INT32_C(    10199286), -INT32_C(  1581850724) },
      {  INT32_C(    91330689),  INT32_C(  1661993557),  INT32_C(   416733115), -INT32_C(  1831048477),  INT32_C(   957437670),  INT32_C(   598642457), -INT32_C(  1977412882),  INT32_C(  1932319217),
        -INT32_C(   965173903),  INT32_C(  1529448607),  INT32_C(  1131610207),  INT32_C(  1356156778), -INT32_C(  1081481306),  INT32_C(  1709389942),  INT32_C(  1022297674),  INT32_C(  1353653215) },
      {  INT32_C(  1477912504),  INT32_C(   263405743), -INT32_C(  1437456832),  INT32_C(   486156150), -INT32_C(  2049211378),  INT32_C(   116047548), -INT32_C(  1555899964), -INT32_C(  1376456203),
        -INT32_C(   939193576), -INT32_C(  1932019637),  INT32_C(  1429613023),  INT32_C(  1601253712),  INT32_C(  1910787253), -INT32_C(   814232053), -INT32_C(  1670137177), -INT32_C(  1001756757) },
      { -INT32_C(  1114878094), -INT32_C(   414620920), -INT32_C(   600014708),  INT32_C(  1715187121),  INT32_C(    81207289), -INT32_C(  1781248275), -INT32_C(  1271838968),  INT32_C(   544766894),
        -INT32_C(   740490037), -INT32_C(   205904025),  INT32_C(  1490024103), -INT32_C(  1665266781),  INT32_C(   413242666), -INT32_C(   324176413),  INT32_C(  1788927676),  INT32_C(   629807194) },
      { -INT32_C(  2080872676),  INT32_C(   896971406),  INT32_C(  1267549608),  INT32_C(  2078755664), -INT32_C(   996964128), -INT32_C(  1179631363),  INT32_C(  2015580190), -INT32_C(  2070041240),
        -INT32_C(  1559784171), -INT32_C(   270959289),  INT32_C(   322594498), -INT32_C(  1836178766), -INT32_C(  1470750294),  INT32_C(  2137065057), -INT32_C(  1091009194),  INT32_C(  1195545906) },
      {  INT32_C(  1928022315), -INT32_C(  1990015802), -INT32_C(   593716182),  INT32_C(  1768893118) },
      {  INT32_C(   735108060),              INT32_MIN,              INT32_MIN,              INT32_MAX,              INT32_MIN, -INT32_C(   695048111), -INT32_C(   243719912),              INT32_MIN,
                     INT32_MIN,  INT32_C(  2074039483),              INT32_MAX,              INT32_MIN,              INT32_MAX,              INT32_MAX,  INT32_C(   668718559),              INT32_MIN } },
    { { -INT32_C(  1408121525),  INT32_C(   573272780),  INT32_C(   719397879), -INT32_C(   462347335),  INT32_C(   861297773),  INT32_C(  1270659104),  INT32_C(   321345621), -INT32_C(   847473022),
         INT32_C(   679054684), -INT32_C(   146103040), -INT32_C(  2128532536), -INT32_C(  1134193841),  INT32_C(   284146927), -INT32_C(   916738956), -INT32_C(  2015526396),  INT32_C(  1968462105) },
      { -INT32_C(   409088281),  INT32_C(  1004463987),  INT32_C(  1656553491), -INT32_C(  2111954285),  INT32_C(  1402080734), -INT32_C(  1105400391), -INT32_C(  1991902864),  INT32_C(   989763923),
        -INT32_C(   635331737), -INT32_C(  1793654910), -INT32_C(  1829252097), -INT32_C(   736815627), -INT32_C(   601381085),  INT32_C(    93995925), -INT32_C(  1869684931), -INT32_C(   523596679) },
      { -INT32_C(  1447302361), -INT32_C(   364916246), -INT32_C(  1719912796),  INT32_C(  1869451596), -INT32_C(   833907655),  INT32_C(   366208728),  INT32_C(  1051025861),  INT32_C(   337604589),
         INT32_C(  1153292890),  INT32_C(  1328479403),  INT32_C(  2145954611),  INT32_C(  1978553916), -INT32_C(  1035781654), -INT32_C(   438888929),  INT32_C(  1680047223),  INT32_C(  1165509355) },
      { -INT32_C(   947243492),  INT32_C(  1695987762), -INT32_C(  1595605149),  INT32_C(  1058394709),  INT32_C(   721574156), -INT32_C(   418326161),  INT32_C(  1095447638), -INT32_C(  1819818890),
         INT32_C(   744100346),  INT32_C(   764572105), -INT32_C(   976390544),  INT32_C(  1426383433), -INT32_C(  1417673157),  INT32_C(   915575264),  INT32_C(   997711301), -INT32_C(  1680933215) },
      { -INT32_C(   641193713),  INT32_C(   168188570),  INT32_C(   433050576), -INT32_C(   244395083), -INT32_C(  1164120102),  INT32_C(  1173368448), -INT32_C(  1384093941),  INT32_C(  1967673190),
         INT32_C(   307105912),  INT32_C(   991712362), -INT32_C(   581637337), -INT32_C(  1731279938),  INT32_C(   844262066), -INT32_C(  1552465256),  INT32_C(   273741738), -INT32_C(  1098540730) },
      {  INT32_C(   349230249),  INT32_C(  1347415080), -INT32_C(  1775393833),  INT32_C(   405732198) },
      {              INT32_MIN,  INT32_C(  2111107528),              INT32_MAX, -INT32_C(  1560276792),              INT32_MAX,              INT32_MAX,  INT32_C(   413902546), -INT32_C(  1783655371),
                     INT32_MAX, -INT32_C(  1242850983),              INT32_MIN,              INT32_MIN,  INT32_C(  1535807097),              INT32_MIN,              INT32_MIN,              INT32_MAX } },
    { { -INT32_C(    45448859),  INT32_C(  1839251907), -INT32_C(     8523335),  INT32_C(   868090762), -INT32_C(    12087593),  INT32_C(  1380947579), -INT32_C(  1595376582), -INT32_C(   591849609),
         INT32_C(  1540948888),  INT32_C(  2127133381), -INT32_C(   176339349),  INT32_C(   539507529),  INT32_C(  1159753930),  INT32_C(  1083666182),  INT32_C(  1675722988),  INT32_C(   792763031) },
      {  INT32_C(  1653217693), -INT32_C(    18852973), -INT32_C(   470589798),  INT32_C(  1661213849), -INT32_C(  1834474356),  INT32_C(  2144551059),  INT32_C(  1474540736), -INT32_C(   343530674),
        -INT32_C(   800190404), -INT32_C(    20042140),  INT32_C(   652329612),  INT32_C(  1787422174), -INT32_C(  1644416503),  INT32_C(   840748914), -INT32_C(   779485309),  INT32_C(  1606225698) },
      { -INT32_C(  2094068961), -INT32_C(   964559559), -INT32_C(  1645452353),  INT32_C(  1376220488),  INT32_C(   435094695),  INT32_C(  1464536019),  INT32_C(   757650443),  INT32_C(    59565539),
         INT32_C(   713473008),  INT32_C(  2028996793), -INT32_C(  1290347413), -INT32_C(   117105327), -INT32_C(   183307231),  INT32_C(   189554176),  INT32_C(   372798514),  INT32_C(  1243202906) },
      {  INT32_C(   963944320),  INT32_C(   313681063), -INT32_C(  1832466368),  INT32_C(   126602213), -INT32_C(  1057186368),  INT32_C(   785074427),  INT32_C(   373556156),  INT32_C(  1231117768),
        -INT32_C(  1534929412),  INT32_C(  2058761274), -INT32_C(   502498052),  INT32_C(   149526344),  INT32_C(   818472245), -INT32_C(   379677907),  INT32_C(  1593877142), -INT32_C(    56073729) },
      {  INT32_C(  1889544758),  INT32_C(  1542149727),  INT32_C(   457045971), -INT32_C(  1004329329),  INT32_C(   955575051),  INT32_C(   337793918), -INT32_C(   177003787), -INT32_C(  1175381117),
        -INT32_C(  1523936954), -INT32_C(  1174399513), -INT32_C(  1680523764),  INT32_C(  1885337700),  INT32_C(  1638421731), -INT32_C(  1669936473),  INT32_C(  1871833580),  INT32_C(  1261011461) },
      { -INT32_C(    68136173),  INT32_C(  1958080616), -INT32_C(  1827632338),  INT32_C(  1728278403) },
      {  INT32_C(   362820912),              INT32_MAX,  INT32_C(   967090470),              INT32_MAX,  INT32_C(  1806753744),              INT32_MAX,              INT32_MIN,              INT32_MIN,
                     INT32_MAX,              INT32_MAX,              INT32_MIN,              INT32_MAX,              INT32_MAX,              INT32_MAX,              INT32_MAX,              INT32_MAX } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i src = simde_mm512_loadu_epi32(test_vec[i].src);
    simde__m512i a0 = simde_mm512_loadu_epi32(test_vec[i].a0);
    simde__m512i a1 = simde_mm512_loadu_epi32(test_vec[i].a1);
    simde__m512i a2 = simde_mm512_loadu_epi32(test_vec[i].a2);
    simde__m512i a3 = simde_mm512_loadu_epi32(test_vec[i].a3);
    simde__m128i b = simde_x_mm_loadu_epi32(test_vec[i].b);
    simde__m512i r = simde_mm512_4dpwssds_epi32(src, a0, a1, a2, a3, &b);
    simde_test_x86_assert_equal_i32x16(r, simde_mm512_loadu_epi32(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m512i src = simde_test_x86_random_i32x16();
    simde__m512i a0 = simde_test_x86_random_i32x16();
    simde__m512i a1 = simde_test_x86_random_i32x16();
    simde__m512i a2 = simde_test_x86_random_i32x16();
    simde__m512i a3 = simde_test_x86_random_i32x16();
    simde__m128i b = simde_test_x86_random_i32x4();
    simde__m512i r = simde_mm512_4dpwssds_epi32(src, a0, a1, a2, a3, &b);

    simde_test_x86_write_i32x16(2, src, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_i32x16(2, a0, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, a1, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, a2, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, a3, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x4(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_mask_4dpwssds_epi32 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const int32_t src[16];
    const simde__mmask16 k;
    const int32_t a0[16];
    const int32_t a1[16];
    const int32_t a2[16];
    const int32_t a3[16];
    const int32_t b[4];
    const int32_t r[16];
  } test_vec[] = {
    { {  INT32_C(  1791536067),  INT32_C(  1644641910),  INT32_C(   768776232),  INT32_C(   779680539), -INT32_C(  1238800306), -INT32_C(  2027233448), -INT32_C(   317048214),  INT32_C(  1834229162),
         INT32_C(  1054350792), -INT32_C(  2086543525), -INT32_C(  1833929865), -INT32_C(  1128191890), -INT32_C(   378344816),  INT32_C(   863018185), -INT32_C(  2128508201),  INT32_C(  1877898663) },
      UINT16_C(50834),
      {  INT32_C(  1336274606),  INT32_C(   583146865),  INT32_C(  1883910319),  INT32_C(  1583078380),  INT32_C(   888808644),  INT32_C(  2042548568),  INT32_C(  1106142546), -INT32_C(  2113437228),
        -INT32_C(   523129489),  INT32_C(  2030212042), -INT32_C(  1343599165), -INT32_C(   334674648), -INT32_C(  1054865303),  INT32_C(   742055641),  INT32_C(   393029955),  INT32_C(   429552810) },
      { -INT32_C(   335975647), -INT32_C(  1033503489),  INT32_C(  1903316809), -INT32_C(    44204140),  INT32_C(  1639939463), -INT32_C(  1634862757), -INT32_C(   860489182), -INT32_C(  1864020114),
        -INT32_C(  1166286917),  INT32_C(   612163803), -INT32_C(  1013584337), -INT32_C(   171838866), -INT32_C(   900300689), -INT32_C(  1670847623),  INT32_C(  1281891805),  INT32_C(   685526637) },
      {  INT32_C(   149051181),  INT32_C(  1730960952), -INT32_C(  1154760371),  INT32_C(   582020275), -INT32_C(   437516436), -INT32_C(   931048214), -INT32_C(   569054607),  INT32_C(  1694953528),
         INT32_C(  2137909319), -INT32_C(  1796761273),  INT32_C(   223285850),  INT32_C(  1781530622), -INT32_C(   246473978), -INT32_C(   524693137), -INT32_C(   205533765),  INT32_C(    72926653) },
      { -INT32_C(   175848018), -INT32_C(  1198953634),  INT32_C(  2076563581), -INT32_C(   555354921),  INT32_C(  2127508495), -INT32_C(  1067546619),  INT32_C(   313728341), -INT32_C(  1860826141),
         INT32_C(   780573392), -INT32_C(  2098852091), -INT32_C(  1090671641), -INT32_C(  1365450081),  INT32_C(   472673047),  INT32_C(  1222478835), -INT32_C(  1956999000),  INT32_C(  1813803420) },
      {  INT32_C(   295412235), -INT32_C(  1735163471), -INT32_C(   866676435), -INT32_C(  1971588237) },
      {  INT32_C(  1791536067),              INT32_MAX,  INT32_C(   768776232),  INT32_C(   779680539),              INT32_MIN, -INT32_C(  2027233448), -INT32_C(   317048214),              INT32_MAX,
         INT32_C(  1054350792),              INT32_MIN,              INT32_MIN, -INT32_C(  1128191890), -INT32_C(   378344816),  INT32_C(   863018185),              INT32_MIN,              INT32_MAX } },
    { {  INT32_C(  1386719071), -INT32_C(   610565070), -INT32_C(  1335429868),  INT32_C(  1914536806), -INT32_C(   679233499),  INT32_C(  1718556217),  INT32_C(   456378023),  INT32_C(   430288570),
        -INT32_C(  2006234027), -INT32_C(   446495024),  INT32_C(  1653983740),  INT32_C(  1926541900), -INT32_C(  1538697366),  INT32_C(   353024110),  INT32_C(   959462783),  INT32_C(  1095948011) },
      UINT16_C(48418),
      {  INT32_C(   751104969),  INT32_C(  1844822232), -INT32_C(   148880862), -INT32_C(    45184332),  INT32_C(   968211502),  INT32_C(    41301202), -INT32_C(  1076338067),  INT32_C(  1820195747),
        -INT32_C(   963100178),  INT32_C(   590580993), -INT32_C(  2095426609),  INT32_C(   209742302), -INT32_C(   146459099), -INT32_C(   654656406), -INT32_C(  1047014882), -INT32_C(  1154673459) },
      {  INT32_C(  1451345237),  INT32_C(   578467154), -INT32_C(   425356280),  INT32_C(   603137789), -INT32_C(   971360164),  INT32_C(   312349940), -INT32_C(  1261226265), -INT32_C(  1603272630),
         INT32_C(   418836933), -INT32_C(  1355124570),  INT32_C(    43376389),  INT32_C(  1629849605), -INT32_C(  1255653183),  INT32_C(  1003013716),  INT32_C(  1206950908),  INT32_C(  1642553500) },
      { -INT32_C(   126231215),  INT32_C(  1403499342), -INT32_C(  1739244397), -INT32_C(  2030404667),  INT32_C(   255533755), -INT32_C(   448068632),  INT32_C(   992754591), -INT32_C(   325315685),
         INT32_C(  1055135472),  INT32_C(  1553042377), -INT32_C(  1913264184),  INT32_C(   487845730), -INT32_C(   114536943), -INT32_C(   237078702),  INT32_C(  1294731954),  INT32_C(   238733341) },
      { -INT32_C(  1471406370),  INT32_C(  1912921770),  INT32_C(   671087045), -INT32_C(   112979224), -INT32_C(  1292668832), -INT32_C(  1700539928), -INT32_C(   102182948),  INT32_C(  1980179096),
        -INT32_C(   367111104), -INT32_C(   144956622),  INT32_C(    85875484), -INT32_C(   822189459), -INT32_C(  1165954606), -INT32_C(  1621809981), -INT32_C(  1952956941), -INT32_C(  1627217825) },
      {  INT32_C(   646586612),  INT32_C(  1612572227), -INT32_C(  1352320190),  INT32_C(  1870488477) },
      {  INT32_C(  1386719071), -INT32_C(  2082120393), -INT32_C(  1335429868),  INT32_C(  1914536806), -INT32_C(   679233499),              INT32_MAX,  INT32_C(   456378023),  INT32_C(   430288570),
                     INT32_MIN, -INT32_C(   446495024),              INT32_MAX,              INT32_MAX,              INT32_MIN,  INT32_C(  1713136378),  INT32_C(   959462783),  INT32_C(  1910468647) } },
    { {  INT32_C(   405470805),  INT32_C(   364347170),  INT32_C(   463556540), -INT32_C(   474307601),  INT32_C(   118047939),  INT32_C(  1835476523), -INT32_C(    31667103), -INT32_C(  2073126353),
        -INT32_C(  1164142440), -INT32_C(   741387497), -INT32_C(  1829867357), -INT32_C(   680089581),  INT32_C(   417234925),  INT32_C(   126174630),  INT32_C(  1074176273), -INT32_C(   742034373) },
      UINT16_C(24844),
      {  INT32_C(  1572152205), -INT32_C(   456304394),  INT32_C(  1619845610), -INT32_C(  1763673672),  INT32_C(   383485585), -INT32_C(  1816597363), -INT32_C(   251137236),  INT32_C(  1414730695),
         INT32_C(   749799478),  INT32_C(  1242595168),  INT32_C(   413900128), -INT32_C(  1464956138), -INT32_C(  1631680239), -INT32_C(  1590528395),  INT32_C(   831732074), -INT32_C(  2088376755) },
      {  INT32_C(  1320171502),  INT32_C(   379109558),  INT32_C(  1949188957), -INT32_C(   551756594), -INT32_C(   629220763), -INT32_C(  1149521839),  INT32_C(   921439977), -INT32_C(   491097612),
         INT32_C(  1580231336), -INT32_C(  2005612246), -INT32_C(   604200436), -INT32_C(   474343298),  INT32_C(  1136474354), -INT32_C(   755090968),  INT32_C(  1007217223),  INT32_C(    69124700) },
      {  INT32_C(  1466126124),  INT32_C(   618649368), -INT32_C(   117449862), -INT32_C(   438519053), -INT32_C(   618030606),  INT32_C(   447555538),  INT32_C(  1851176210), -INT32_C(  1536002952),
        -INT32_C(   604252733),  INT32_C(   654367405), -INT32_C(  1474297931), -INT32_C(  1399915591),  INT32_C(  1736947605), -INT32_C(   259967778),  INT32_C(  1633605609),  INT32_C(   252105036) },
      {  INT32_C(  1407910310), -INT32_C(  1854215204), -INT32_C(  1539663126),  INT32_C(   709937301),  INT32_C(  1569904511), -INT32_C(   196209909),  INT32_C(   928427243),  INT32_C(   608590973),
         INT32_C(   964112733),  INT32_C(   130806300),  INT32_C(   564856204),  INT32_C(  1280113613), -INT32_C(   576069934), -INT32_C(   590219279),  INT32_C(   554903460), -INT32_C(   515548541) },
      { -INT32_C(  1474642549),  INT32_C(  1001383343), -INT32_C(  1218684182),  INT32_C(   654551125) },
      {  INT32_C(   405470805),  INT32_C(   364347170),  INT32_C(  1929225411), -INT32_C(  1750122511),  INT32_C(   118047939),  INT32_C(  1835476523), -INT32_C(    31667103), -INT32_C(  2073126353),
                     INT32_MIN, -INT32_C(   741387497), -INT32_C(  1829867357), -INT32_C(   680089581),  INT32_C(   417234925),  INT32_C(   604096557),              INT32_MAX, -INT32_C(   742034373) } },
    { {  INT32_C(  2013572486),  INT32_C(  1213519268), -INT32_C(  2140509955),  INT32_C(  1298247618),  INT32_C(   469073004),  INT32_C(  1280746593),  INT32_C(  1392751614), -INT32_C(   495319205),
         INT32_C(  1482325684),  INT32_C(  1369550420), -INT32_C(   657388778),  INT32_C(   656815034),  INT32_C(   272767919), -INT32_C(  1101227584), -INT32_C(  1491967924),  INT32_C(   461999207) },
      UINT16_C(58123),
      {  INT32_C(   345137011), -INT32_C(  2128631632), -INT32_C(  1481319807),  INT32_C(  1136812801), -INT32_C(   790855052),  INT32_C(  1395664961),  INT32_C(  1507825616),  INT32_C(   641591986),
        -INT32_C(   113586359), -INT32_C(   126133641),  INT32_C(   899624756),  INT32_C(   125330067),  INT32_C(   634869220),  INT32_C(  1299712125),  INT32_C(  1386698911), -INT32_C(  1938234302) },
      {  INT32_C(   713405107),  INT32_C(  1076035596), -INT32_C(  1015692752),  INT32_C(   164294180), -INT32_C(  1070685885),  INT32_C(  1225697193),  INT32_C(  1100723711),  INT32_C(  1288508313),
        -INT32_C(   763997243), -INT32_C(  2095933101), -INT32_C(  2142861220), -INT32_C(  1182199690),  INT32_C(  1534703794),  INT32_C(  1587840863), -INT32_C(   710983876),  INT32_C(   404843858) },
      {  INT32_C(   334141632), -INT32_C(  1919419343), -INT32_C(    99689084), -INT32_C(  1598842898), -INT32_C(  1359270577), -INT32_C(   250830668),  INT32_C(   851881183), -INT32_C(   649402343),
        -INT32_C(  1309920128), -INT32_C(  1254128848),  INT32_C(  1336888673),  INT32_C(   888103908),  INT32_C(  1155721872),  INT32_C(  1781919626), -INT32_C(  1264780133),  INT32_C(  1687021284) },
      {  INT32_C(  1242921242),  INT32_C(  1593791741), -INT32_C(  2035437663), -INT32_C(  1564828654),  INT32_C(   300326022),  INT32_C(   645602443), -INT32_C(    52816104),  INT32_C(   392194045),
        -INT32_C(   564038175),  INT32_C(  1799119306),  INT32_C(   586279184),  INT32_C(   197438341), -INT32_C(   753096120), -INT32_C(   554002490), -INT32_C(  1411722066),  INT32_C(   499268156) },
      {  INT32_C(  2063279280), -INT32_C(  1780140155), -INT32_C(  1514678496), -INT32_C(   894403710) },
      {              INT32_MAX,              INT32_MAX, -INT32_C(  2140509955),              INT32_MAX,  INT32_C(   469073004),  INT32_C(  1280746593),  INT32_C(  1392751614), -INT32_C(   495319205),
                     INT32_MAX,              INT32_MAX, -INT32_C(   657388778),  INT32_C(   656815034),  INT32_C(   272767919),              INT32_MIN,              INT32_MIN,  INT32_C(   919834342) } },
    { { -INT32_C(   325137115),  INT32_C(   332044388), -INT32_C(  1463900820), -INT32_C(  1882881569),  INT32_C(   705282213),  INT32_C(   398454775),  INT32_C(  1220310726),  INT32_C(   370372081),
        -INT32_C(  1643990726), -INT32_C(  1246638775),  INT32_C(  1365143666), -INT32_C(  1763630095), -INT32_C(   608113948), -INT32_C(  1611431975), -INT32_C(   420958219),  INT32_C(  1459419932) },
      UINT16_C(65452),
      { -INT32_C(  1496517131),  INT32_C(   135675562),  INT32_C(  1881933711),  INT32_C(  1566249117), -INT32_C(   555993877), -INT32_C(  1131490860), -INT32_C(  1263031625), -INT32_C(   156015871),
         INT32_C(    43810648),  INT32_C(  1275835325),  INT32_C(  1488795578),  INT32_C(   850729031),  INT32_C(   538022476),  INT32_C(   467443555),  INT32_C(  1255117641),  INT32_C(  1312850678) },
      { -INT32_C(  1101931263),  INT32_C(  1242193040), -INT32_C(   626866285),  INT32_C(   739006687),  INT32_C(  1296834026),  INT32_C(   107489724), -INT32_C(  1286588484), -INT32_C(  1174302279),
        -INT32_C(    25669010),  INT32_C(  1095271342),  INT32_C(   706472778),  INT32_C(   760621123),  INT32_C(    41591365), -INT32_C(  2012683317), -INT32_C(   734308326),  INT32_C(  1468939497) },
      {  INT32_C(  1028982671), -INT32_C(   729833846), -INT32_C(   855729527),  INT32_C(   150557890), -INT32_C(  1039502345),  INT32_C(  1900679766),  INT32_C(  1413842282),  INT32_C(  1370215362),
         INT32_C(  1687028186),  INT32_C(   674827679),  INT32_C(  1794389928), -INT32_C(  2089620084), -INT32_C(  1220182944), -INT32_C(   114782066), -INT32_C(   682791659),  INT32_C(   455669824) },
      { -INT32_C(  1736460295),  INT32_C(  1824569540),  INT32_C(  2094511344),  INT32_C(    50284961),  INT32_C(  1421427910), -INT32_C(   363994668), -INT32_C(  1899914674), -INT32_C(  1935021421),
         INT32_C(  1696934305), -INT32_C(   774707743),  INT32_C(   994945434), -INT32_C(  1203942158),  INT32_C(  1695413905),  INT32_C(   625957591), -INT32_C(  2001464843), -INT32_C(  1676386821) },
      {  INT32_C(  1744910726), -INT32_C(  1187392737),  INT32_C(  1878296444),  INT32_C(  1680290515) },
      { -INT32_C(   325137115),  INT32_C(   332044388),              INT32_MIN,              INT32_MIN,  INT32_C(   705282213),  INT32_C(  1251043317),  INT32_C(  1220310726),  INT32_C(  1689447640),
                     INT32_MIN,              INT32_MIN,              INT32_MAX,              INT32_MIN, -INT32_C(  2123763870),              INT32_MIN,              INT32_MIN,              INT32_MAX } },
    { {  INT32_C(    13251624), -INT32_C(  2077943409),  INT32_C(   621599018), -INT32_C(  1111416777),  INT32_C(  2032517978),  INT32_C(   322068374), -INT32_C(  1182652698), -INT32_C(  2128697000),
         INT32_C(  1837230302),  INT32_C(   754034177), -INT32_C(  1219363456),  INT32_C(  2004095773),  INT32_C(  1827707606), -INT32_C(   545316103), -INT32_C(  1583873719), -INT32_C(  1994213717) },
      UINT16_C(41886),
      { -INT32_C(   414539530),  INT32_C(   501533388), -INT32_C(   181403263),  INT32_C(  1770980985), -INT32_C(   225671053),  INT32_C(    16045415), -INT32_C(  1716084874),  INT32_C(   507270440),
        -INT32_C(  1056602379), -INT32_C(   773920432),  INT32_C(  1673990122), -INT32_C(  1999809003),  INT32_C(  1182489055), -INT32_C(  1538887890),  INT32_C(   910032142),  INT32_C(  1196718674) },
      {  INT32_C(  1359501568),  INT32_C(   740484674),  INT32_C(   194046453),  INT32_C(   529751360), -INT32_C(   463139146), -INT32_C(  1953977219), -INT32_C(    71186775),  INT32_C(  1078072640),
        -INT32_C(  1332655506),  INT32_C(   652063792), -INT32_C(   583963235), -INT32_C(  2130918198),  INT32_C(  1348756179), -INT32_C(  1210258418), -INT32_C(   223175246),  INT32_C(   557053107) },
      {  INT32_C(  1859306558),  INT32_C(   378843000), -INT32_C(   420231908),  INT32_C(  1567027338),  INT32_C(  1622002258),  INT32_C(  1763150262), -INT32_C(   631518937), -INT32_C(    67400003),
        -INT32_C(   882258605), -INT32_C(  1730019716),  INT32_C(  1300157891),  INT32_C(   397075653),  INT32_C(  1702320302),  INT32_C(   147754721),  INT32_C(   350366039),  INT32_C(   185589432) },
      {  INT32_C(   685144235),  INT32_C(   985708662),  INT32_C(  1384595341), -INT32_C(   764857821),  INT32_C(  1798824074), -INT32_C(   982252178), -INT32_C(   421964242), -INT32_C(   521017292),
        -INT32_C(   687290272),  INT32_C(   219269248),  INT32_C(   727685127),  INT32_C(  1425918154),  INT32_C(   381695144),  INT32_C(  1742419001), -INT32_C(  1085361014), -INT32_C(    56672356) },
      { -INT32_C(  2016172281),  INT32_C(  2006246511),  INT32_C(  1201861501),  INT32_C(  1671208891) },
      {  INT32_C(    13251624),              INT32_MIN,              INT32_MAX,              INT32_MIN,              INT32_MAX,  INT32_C(   322068374), -INT32_C(  1182652698),              INT32_MIN,
                     INT32_MAX,              INT32_MAX, -INT32_C(  1219363456),  INT32_C(  2004095773),  INT32_C(  1827707606),              INT32_MIN, -INT32_C(  1583873719),              INT32_MIN } },
    { {  INT32_C(   209280211),  INT32_C(   443765904), -INT32_C(  1529233144),  INT32_C(   144799745), -INT32_C(  1886358497), -INT32_C(   704240551), -INT32_C(   753031144),  INT32_C(   439859527),
        -INT32_C(  1524191211),  INT32_C(   230725893),  INT32_C(  1555208539),  INT32_C(   828658450),  INT32_C(   549516487),  INT32_C(   838256153), -INT32_C(  1258023826), -INT32_C(   472958003) },
      UINT16_C(62956),
      {  INT32_C(  1217393032), -INT32_C(  1327305986), -INT32_C(  1442581434), -INT32_C(   425735387), -INT32_C(   491996949), -INT32_C(   302638103), -INT32_C(  1607875632),  INT32_C(   798299558),
         INT32_C(    74916870),  INT32_C(  1421170958),  INT32_C(  1946138701),  INT32_C(  1868144259),  INT32_C(  1062274390), -INT32_C(   248756448), -INT32_C(  1315875574),  INT32_C(  1893738090) },
      {  INT32_C(  1500862283), -INT32_C(    22140239),  INT32_C(  1718726114), -INT32_C(  1579824565), -INT32_C(   253745456),  INT32_C(  2011237485), -INT32_C(   869764510), -INT32_C(   465762151),
         INT32_C(   289256032), -INT32_C(  1106252836), -INT32_C(   484146792),  INT32_C(   461699403), -INT32_C(  1945344993), -INT32_C(   754651791), -INT32_C(   107008928), -INT32_C(  1780622283) },
      {  INT32_C(  1789270670), -INT32_C(  1641433850), -INT32_C(  2105389770),  INT32_C(  1721566791), -INT32_C(   588011157), -INT32_C(   139462761),  INT32_C(  1492143651), -INT32_C(  1175597781),
        -INT32_C(   316369945),  INT32_C(  2139901257), -INT32_C(   520024422),  INT32_C(  2135465748), -INT32_C(   547669176),  INT32_C(  1440090674), -INT32_C(  2068920743),  INT32_C(  2067635091) },
      {  INT32_C(  2020106543),  INT32_C(  1240986798),  INT32_C(   388692226), -INT32_C(   527011176), -INT32_C(   541003091),  INT32_C(  1429575420), -INT32_C(   254155939), -INT32_C(  1385490562),
         INT32_C(   656790648), -INT32_C(   881844792), -INT32_C(  1360881130), -INT32_C(  1165068275),  INT32_C(  1738165866),  INT32_C(  1119670245),  INT32_C(   808621746),  INT32_C(   635346605) },
      {  INT32_C(   978060146),  INT32_C(   906345504),  INT32_C(  1692723031), -INT32_C(   903974048) },
      {  INT32_C(   209280211),  INT32_C(   443765904),              INT32_MIN,  INT32_C(  2021560516), -INT32_C(  1886358497),              INT32_MIN,              INT32_MIN,  INT32_C(  1897659152),
                     INT32_MIN,  INT32_C(   230725893),              INT32_MAX,  INT32_C(   828658450),  INT32_C(  1500720096),              INT32_MAX,              INT32_MIN, -INT32_C(  1560905978) } },
    { { -INT32_C(  1506690879),  INT32_C(   971566727),  INT32_C(   829037444),  INT32_C(   727075001),  INT32_C(  1818600267), -INT32_C(  1230869665), -INT32_C(  1306884270), -INT32_C(  1149421062),
         INT32_C(  2036510449),  INT32_C(   548555420),  INT32_C(   508632165), -INT32_C(  1337350044), -INT32_C(  1440960693),  INT32_C(  1818344986),  INT32_C(  1059027781), -INT32_C(  1493459788) },
      UINT16_C(23882),
      { -INT32_C(   777525473),  INT32_C(  1508773127),  INT32_C(  1963020843),  INT32_C(   505695234),  INT32_C(  1474117366), -INT32_C(   908909910),  INT32_C(  1566934882),  INT32_C(  1287303213),
        -INT32_C(  1642176105), -INT32_C(  1695085458),  INT32_C(  1628436574),  INT32_C(   998191940),  INT32_C(   462576753), -INT32_C(   505059969),  INT32_C(   440290029), -INT32_C(  1855457030) },
      { -INT32_C(   936344231), -INT32_C(   278780015),  INT32_C(  1682993440),  INT32_C(   362795172), -INT32_C(  1422904788), -INT32_C(  2071194217),  INT32_C(  1520421472),  INT32_C(   485230274),
         INT32_C(   484711563),  INT32_C(  1661683267),  INT32_C(  1539857591),  INT32_C(  1483761452),  INT32_C(   822386841),  INT32_C(   380997814),  INT32_C(   493900891), -INT32_C(   449225638) },
      { -INT32_C(  1140777608),  INT32_C(   438242659), -INT32_C(  1787435159), -INT32_C(   387062449),  INT32_C(  1008333190), -INT32_C(   581775486),  INT32_C(  2130362915), -INT32_C(  1755106273),
        -INT32_C(  1252825775), -INT32_C(   607161486), -INT32_C(  1452260006), -INT32_C(  1332585174), -INT32_C(   773018801), -INT32_C(  1632747910),  INT32_C(   538748929),  INT32_C(   783777756) },
      {  INT32_C(  1457720292), -INT32_C(   667831682),  INT32_C(   578986487),  INT32_C(  1322390783),  INT32_C(   975159231), -INT32_C(    19345923),  INT32_C(  1377760373),  INT32_C(  1484838515),
         INT32_C(  1605264353),  INT32_C(   221765653), -INT32_C(  2144355967), -INT32_C(  1915878963), -INT32_C(  1094193728),  INT32_C(   817668026),  INT32_C(   109239187), -INT32_C(  1839332687) },
      {  INT32_C(  2062617957),  INT32_C(  1854351597), -INT32_C(  1343244574),  INT32_C(  2017246648) },
      { -INT32_C(  1506690879),              INT32_MAX,  INT32_C(   829037444),              INT32_MAX,  INT32_C(  1818600267), -INT32_C(  1230869665),              INT32_MIN, -INT32_C(  1149421062),
                     INT32_MAX,  INT32_C(   548555420),  INT32_C(  1955184928),              INT32_MIN,              INT32_MIN,  INT32_C(  1818344986),              INT32_MAX, -INT32_C(  1493459788) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i src = simde_mm512_loadu_epi32(test_vec[i].src);
    simde__m512i a0 = simde_mm512_loadu_epi32(test_vec[i].a0);
    simde__m512i a1 = simde_mm512_loadu_epi32(test_vec[i].a1);
    simde__m512i a2 = simde_mm512_loadu_epi32(test_vec[i].a2);
    simde__m512i a3 = simde_mm512_loadu_epi32(test_vec[i].a3);
    simde__m128i b = simde_x_mm_loadu_epi32(test_vec[i].b);
    simde__m512i r = simde_mm512_mask_4dpwssds_epi32(src, test_vec[i].k, a0, a1, a2, a3, &b);
    simde_test_x86_assert_equal_i32x16(r, simde_mm512_loadu_epi32(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m512i src = simde_test_x86_random_i32x16();
    simde__mmask16 k = simde_test_x86_random_mmask16();
    simde__m512i a0 = simde_test_x86_random_i32x16();
    simde__m512i a1 = simde_test_x86_random_i32x16();
    simde__m512i a2 = simde_test_x86_random_i32x16();
    simde__m512i a3 = simde_test_x86_random_i32x16();
    simde__m128i b = simde_test_x86_random_i32x4();
    simde__m512i r = simde_mm512_mask_4dpwssds_epi32(src, k, a0, a1, a2, a3, &b);

    simde_test_x86_write_i32x16(2, src, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_mmask16(2, k, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, a0, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, a1, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, a2, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, a3, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x4(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_maskz_4dpwssds_epi32 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const simde__mmask16 k;
    const int32_t src[16];
    const int32_t a0[16];
    const int32_t a1[16];
    const int32_t a2[16];
    const int32_t a3[16];
    const int32_t b[4];
    const int32_t r[16];
  } test_vec[] = {
    { UINT16_C(  939),
      { -INT32_C(   207461066),  INT32_C(   399390101), -INT32_C(  1709605060),  INT32_C(    44531217), -INT32_C(  2144692999), -INT32_C(   231273469), -INT32_C(   122687556), -INT32_C(  1627628953),
         INT32_C(  1435606720),  INT32_C(   275603412), -INT32_C(   274037026), -INT32_C(    17739259), -INT32_C(   377611290), -INT32_C(   455363288),  INT32_C(   215780260), -INT32_C(  1515530011) },
      {  INT32_C(  1274755959),  INT32_C(  2019256474), -INT32_C(   211352082),  INT32_C(  1056004183), -INT32_C(  1691914125), -INT32_C(   897645787),  INT32_C(  1943428238), -INT32_C(  1424392140),
         INT32_C(  1442190523),  INT32_C(  1791840636), -INT32_C(  1369557929), -INT32_C(     1290356), -INT32_C(   442821441), -INT32_C(  1515251177), -INT32_C(  1424456330), -INT32_C(  1068093179) },
      { -INT32_C(  1055568571), -INT32_C(   181607778), -INT32_C(  1566275050), -INT32_C(  1717464871), -INT32_C(  1149354844), -INT32_C(   849334953), -INT32_C(  1216841294), -INT32_C(   260583510),
        -INT32_C(  1162769380), -INT32_C(  2068783762),  INT32_C(  1093031015), -INT32_C(  1982150684),  INT32_C(  1531205636),  INT32_C(   925410693), -INT32_C(   923885282), -INT32_C(  1934072464) },
      {  INT32_C(  1598450417), -INT32_C(  1344014777),  INT32_C(   804260170), -INT32_C(   726086960), -INT32_C(  1490027486), -INT32_C(  1075947359),  INT32_C(  1770573049),  INT32_C(   586498097),
        -INT32_C(   226411606),  INT32_C(  2090951730),  INT32_C(  1034654061),  INT32_C(  2098357083),  INT32_C(    19153248), -INT32_C(  1832844647), -INT32_C(      308786),  INT32_C(   874639497) },
      {  INT32_C(  1579590188),  INT32_C(  1943717638), -INT32_C(  1280276904),  INT32_C(  1227932393), -INT32_C(  1656007676),  INT32_C(   607128662), -INT32_C(   568120491),  INT32_C(  1209156636),
        -INT32_C(   324650778),  INT32_C(  1465876735), -INT32_C(   267776250), -INT32_C(   700892462), -INT32_C(   462191474), -INT32_C(   452418672), -INT32_C(   339465265),  INT32_C(  1429460591) },
      {  INT32_C(   239196431),  INT32_C(  1617338457), -INT32_C(  2108657488),  INT32_C(   962103723) },
      { -INT32_C(   706037269),              INT32_MAX,  INT32_C(           0),  INT32_C(   774025928),  INT32_C(           0), -INT32_C(   364018697),  INT32_C(           0),              INT32_MIN,
                     INT32_MAX,  INT32_C(  1458542252),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0) } },
    { UINT16_C(51982),
      {  INT32_C(   644849182),  INT32_C(  1213349508),  INT32_C(  1545519401),  INT32_C(  1479879959), -INT32_C(  1560703428),  INT32_C(  1041410542), -INT32_C(  2083996373),  INT32_C(   357488375),
        -INT32_C(   130302604),  INT32_C(   608210427),  INT32_C(  1719689039), -INT32_C(   927025780),  INT32_C(   845854532), -INT32_C(  1955496608),  INT32_C(   823015738), -INT32_C(  2092540657) },
      {  INT32_C(   377258522),  INT32_C(  1580907535), -INT32_C(  1463502053), -INT32_C(  1267694736), -INT32_C(  1679303878), -INT32_C(  1842980776), -INT32_C(  1597753967), -INT32_C(  1406989678),
        -INT32_C(  1664966772),  INT32_C(  2012937564),  INT32_C(   673169336),  INT32_C(  2094895170), -INT32_C(  1021852565), -INT32_C(  1386922468),  INT32_C(    88938867), -INT32_C(  1330548700) },
      {  INT32_C(  1816949776),  INT32_C(   702760817),  INT32_C(  1213268742), -INT32_C(    20697453),  INT32_C(   247586034), -INT32_C(  1917118950),  INT32_C(  1418856496), -INT32_C(  1996209287),
         INT32_C(   687165879), -INT32_C(  1638803304),  INT32_C(  1860608987), -INT32_C(   999511343), -INT32_C(  1596838522),  INT32_C(  1949142596),  INT32_C(   264814486), -INT32_C(  1164391421) },
      { -INT32_C(  1243378403),  INT32_C(  1095971942), -INT32_C(  1448068649),  INT32_C(  1785535716), -INT32_C(  1911865526),  INT32_C(  1677867213), -INT32_C(    76297480), -INT32_C(  1263203178),
        -INT32_C(     9856871), -INT32_C(  1539195443), -INT32_C(   632425994),  INT32_C(  1464187405), -INT32_C(   941272838), -INT32_C(  2144606328),  INT32_C(  1216061361),  INT32_C(  1157378475) },
      { -INT32_C(  1773902391),  INT32_C(   423331106), -INT32_C(  2081191818),  INT32_C(  1037776963),  INT32_C(   285524104),  INT32_C(  1502687400),  INT32_C(  2057375183),  INT32_C(   129932606),
         INT32_C(   631112195), -INT32_C(    46212729), -INT32_C(  1535036831), -INT32_C(   203334550), -INT32_C(  1006311908), -INT32_C(   450980586), -INT32_C(   530595934),  INT32_C(  1625824605) },
      { -INT32_C(  1484421600), -INT32_C(  1062878113),  INT32_C(  1617241846), -INT32_C(  1638709630) },
      {  INT32_C(           0),              INT32_MAX,              INT32_MAX,              INT32_MAX,  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),
        -INT32_C(   163983841),  INT32_C(  1409072950),  INT32_C(           0), -INT32_C(  2082057238),  INT32_C(           0),  INT32_C(           0),              INT32_MAX,              INT32_MIN } },
    { UINT16_C(22316),
      { -INT32_C(  2115157405), -INT32_C(  2042589401),  INT32_C(  1487183216), -INT32_C(  2082552579),  INT32_C(   273104235),  INT32_C(  1647721981), -INT32_C(   240535395), -INT32_C(  1169631913),
         INT32_C(  1060844824),  INT32_C(   902200261),  INT32_C(   378366489), -INT32_C(  1717998802), -INT32_C(  1515528024), -INT32_C(  1157046243), -INT32_C(   257117799), -INT32_C(  1632963450) },
      { -INT32_C(   270605014),  INT32_C(  2032444512),  INT32_C(  1016115470), -INT32_C(   992663268),  INT32_C(   661225226), -INT32_C(   102600352), -INT32_C(  1444311518), -INT32_C(  1387818109),
        -INT32_C(   660855432), -INT32_C(   682442551), -INT32_C(  1928076687),  INT32_C(   357689611), -INT32_C(   935478680),  INT32_C(  1321279276),  INT32_C(   838314670), -INT32_C(  1243726019) },
      {  INT32_C(   781089380), -INT32_C(  1425678278), -INT32_C(   851961406),  INT32_C(  1793296642),  INT32_C(  1865621571), -INT32_C(   306252737), -INT32_C(   601967201),  INT32_C(  1502805236),
        -INT32_C(  1333321610), -INT32_C(  1034187776), -INT32_C(  1483762779),  INT32_C(  1611821596), -INT32_C(   758168174), -INT32_C(   658535111),  INT32_C(   934665795),  INT32_C(  1368410074) },
      {  INT32_C(  1728124775),  INT32_C(  1227447716),  INT32_C(   233945328), -INT32_C(  1116929237), -INT32_C(  2121319352),  INT32_C(   224022474),  INT32_C(   121900845), -INT32_C(  1118251690),
        -INT32_C(  1876665620), -INT32_C(  1478865481),  INT32_C(   817154821),  INT32_C(   384704974),  INT32_C(   681016670), -INT32_C(   113905204),  INT32_C(  1442937088),  INT32_C(   991123790) },
      {  INT32_C(  1791702963), -INT32_C(  1978489468),  INT32_C(  1052427888),  INT32_C(  1179953384), -INT32_C(   227611610), -INT32_C(   554916899),  INT32_C(  1798630684), -INT32_C(    89765818),
         INT32_C(    73691519), -INT32_C(  2020706793),  INT32_C(   633751613),  INT32_C(   392895217), -INT32_C(   469116666), -INT32_C(  1732053636),  INT32_C(   688125666), -INT32_C(  1104959170) },
      {  INT32_C(   851609371),  INT32_C(  1002000638), -INT32_C(  1990164584), -INT32_C(  1600074854) },
      {  INT32_C(           0),  INT32_C(           0),              INT32_MAX,              INT32_MIN,  INT32_C(           0),              INT32_MAX,  INT32_C(           0),  INT32_C(           0),
                     INT32_MAX,              INT32_MAX, -INT32_C(   861755988),  INT32_C(           0),              INT32_MIN,  INT32_C(           0), -INT32_C(    29167513),  INT32_C(           0) } },
    { UINT16_C(43684),
      {  INT32_C(  1184833668), -INT32_C(  1136819528), -INT32_C(   832210005), -INT32_C(    78282695),  INT32_C(  1816875954), -INT32_C(   286530418),  INT32_C(   247039341), -INT32_C(  1430758106),
         INT32_C(   921786237), -INT32_C(  2064503079), -INT32_C(   481142871), -INT32_C(  1965119272), -INT32_C(  1963578629),  INT32_C(  2071519758), -INT32_C(  1920388505),  INT32_C(   221790607) },
      {  INT32_C(  1917004185),  INT32_C(    16201047),  INT32_C(  1692617100), -INT32_C(   303054095), -INT32_C(    92805652),  INT32_C(   796258247), -INT32_C(  1313013983), -INT32_C(   641797056),
         INT32_C(  1951138077), -INT32_C(  1032502474),  INT32_C(  2116507788),  INT32_C(   107681306), -INT32_C(  1023352069), -INT32_C(   202213679), -INT32_C(  1247498635), -INT32_C(  1064344925) },
      { -INT32_C(  1724523677), -INT32_C(  1436833250),  INT32_C(   489193986), -INT32_C(  1809607784),  INT32_C(  1196893301),  INT32_C(   255478170), -INT32_C(  1698373897), -INT32_C(  1554296000),
         INT32_C(  1279037486),  INT32_C(  1022859066), -INT32_C(  1302782182),  INT32_C(   675708339),  INT32_C(   997170593), -INT32_C(   565532186), -INT32_C(   948433273), -INT32_C(  1872047262) },
      { -INT32_C(  1646418333),  INT32_C(  1490736190), -INT32_C(  1509280781),  INT32_C(  1372541360), -INT32_C(   712229138),  INT32_C(  1874056936),  INT32_C(  1194798053),  INT32_C(  1658298879),
        -INT32_C(  2046774200),  INT32_C(  2094979720), -INT32_C(  1105008627),  INT32_C(   672133433),  INT32_C(   419273776),  INT32_C(  1468510322), -INT32_C(   627130661), -INT32_C(  1472367008) },
      { -INT32_C(  1288815318),  INT32_C(   607063063),  INT32_C(   786584052),  INT32_C(  1935077955),  INT32_C(     9130894), -INT32_C(   564653565),  INT32_C(   834270929), -INT32_C(  1747323284),
         INT32_C(  1246365491),  INT32_C(   124680467),  INT32_C(   221598154), -INT32_C(   780104893), -INT32_C(   506393634), -INT32_C(   272619234), -INT32_C(  1944028896), -INT32_C(  1574700689) },
      {  INT32_C(   334261504), -INT32_C(  1323672858), -INT32_C(   289517397), -INT32_C(  1161871397) },
      {  INT32_C(           0),  INT32_C(           0),              INT32_MIN,  INT32_C(           0),  INT32_C(           0), -INT32_C(   955777697),  INT32_C(           0),              INT32_MIN,
         INT32_C(           0),              INT32_MIN,  INT32_C(           0),              INT32_MIN,  INT32_C(           0),              INT32_MAX,  INT32_C(           0),              INT32_MAX } },
    { UINT16_C(37194),
      {  INT32_C(  1538943131),  INT32_C(  2010438231), -INT32_C(  1972354201), -INT32_C(   772247323), -INT32_C(  1641226621),  INT32_C(  1324275599), -INT32_C(  2054305338),  INT32_C(   521590659),
        -INT32_C(  1753558720),  INT32_C(   302993323),  INT32_C(  2023587730),  INT32_C(  1934202352),  INT32_C(    51475827),  INT32_C(   324140877),  INT32_C(  1285086921), -INT32_C(   177492043) },
      {  INT32_C(   730719872), -INT32_C(   952198091),  INT32_C(   188734235), -INT32_C(   478181008),  INT32_C(  1273401598),  INT32_C(  1499346832), -INT32_C(   878315755),  INT32_C(   650121638),
         INT32_C(   743525879),  INT32_C(    99848169), -INT32_C(   636472470), -INT32_C(  1161916484), -INT32_C(  1341742048), -INT32_C(   251042596),  INT32_C(    29142619), -INT32_C(  1238925889) },
      { -INT32_C(  1260226102),  INT32_C(  1941558792), -INT32_C(   984758007),  INT32_C(  2038369113), -INT32_C(  1943435856),  INT32_C(  1165832938), -INT32_C(  1605944608), -INT32_C(  2125041993),
        -INT32_C(   264947225),  INT32_C(   409202191),  INT32_C(   299741368),  INT32_C(  1821007036), -INT32_C(   856116254), -INT32_C(   988711451),  INT32_C(  1717917871), -INT32_C(  1360479033) },
      {  INT32_C(    77471221), -INT32_C(  1004797684),  INT32_C(  1859516850),  INT32_C(   937058133), -INT32_C(   150744558), -INT32_C(   138668729),  INT32_C(   878518893), -INT32_C(   740080162),
         INT32_C(  1876394339),  INT32_C(   892597123),  INT32_C(  1101203692),  INT32_C(  2038005095), -INT32_C(  1771013041), -INT32_C(     7525231),  INT32_C(   741600078), -INT32_C(  1811999184) },
      {  INT32_C(   453170840), -INT32_C(  1253034551), -INT32_C(  1527254211), -INT32_C(  1088589712),  INT32_C(  2119536108),  INT32_C(   142468026), -INT32_C(    30035762),  INT32_C(  1603417287),
        -INT32_C(   730164213),  INT32_C(   126536394),  INT32_C(   766280125), -INT32_C(   571684367),  INT32_C(   291193175), -INT32_C(   233187292),  INT32_C(  1357991561), -INT32_C(  1901100157) },
      { -INT32_C(   513660649), -INT32_C(  1326846733),  INT32_C(  1591580013), -INT32_C(  1254307490) },
      {  INT32_C(           0),              INT32_MAX,  INT32_C(           0),              INT32_MIN,  INT32_C(           0),  INT32_C(           0),              INT32_MIN,  INT32_C(           0),
                     INT32_MIN,  INT32_C(           0),  INT32_C(           0),  INT32_C(           0), -INT32_C(  1364975921),  INT32_C(           0),  INT32_C(           0),  INT32_C(   455611049) } },
    { UINT16_C(38666),
      { -INT32_C(   529518650),  INT32_C(   305068321), -INT32_C(   124407479), -INT32_C(  1574851265),  INT32_C(  2005800078), -INT32_C(  1592984380), -INT32_C(  1771410598), -INT32_C(   416386016),
        -INT32_C(   976773469), -INT32_C(   522717801), -INT32_C(   422023769), -INT32_C(  1467418342), -INT32_C(   786426099),  INT32_C(  1836198931), -INT32_C(  1207640937), -INT32_C(   190893488) },
      {  INT32_C(  1740203728),  INT32_C(    38244443),  INT32_C(   401154045),  INT32_C(   633368856), -INT32_C(  1678319480), -INT32_C(  1542887412), -INT32_C(  1805906620),  INT32_C(   260635455),
        -INT32_C(  1133100703), -INT32_C(   809517614), -INT32_C(   186144548), -INT32_C(  1575377127), -INT32_C(  1824714873), -INT32_C(  1154005385), -INT32_C(  1840278701), -INT32_C(   274605938) },
      { -INT32_C(   341043431), -INT32_C(  1329894444),  INT32_C(   748986899), -INT32_C(   791757495),  INT32_C(  1130564812), -INT32_C(  1493263534), -INT32_C(  1120383442),  INT32_C(  1068292646),
        -INT32_C(   970237711), -INT32_C(   696850749), -INT32_C(   788325496), -INT32_C(  1516121640),  INT32_C(   820511965), -INT32_C(   824776801),  INT32_C(  1535839797), -INT32_C(   627427352) },
      {  INT32_C(  1403045264),  INT32_C(   858396331),  INT32_C(   168045873), -INT32_C(   592468482),  INT32_C(  1225562026), -INT32_C(  1290280322), -INT32_C(   653352208),  INT32_C(  1790159322),
         INT32_C(   448615278), -INT32_C(  1689393303),  INT32_C(   329601556), -INT32_C(  1578150665),  INT32_C(  1793850347), -INT32_C(   853736739),  INT32_C(  2141596837),  INT32_C(  1139366357) },
      {  INT32_C(   375236268), -INT32_C(  1565414514), -INT32_C(   189442307), -INT32_C(  1785289558),  INT32_C(  2097119647),  INT32_C(   675945859),  INT32_C(   514322505), -INT32_C(   161378230),
        -INT32_C(   989020361),  INT32_C(  1734851946), -INT32_C(  1118102253),  INT32_C(  1632891329), -INT32_C(   153267598), -INT32_C(  1205983377),  INT32_C(  1658242584), -INT32_C(  1923598250) },
      {  INT32_C(  1632790007),  INT32_C(   919124514), -INT32_C(  1728896041), -INT32_C(  2013706731) },
      {  INT32_C(           0), -INT32_C(   235734375),  INT32_C(           0),              INT32_MIN,  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),
                     INT32_MIN,              INT32_MIN, -INT32_C(  1911739207),  INT32_C(           0),              INT32_MIN,  INT32_C(           0),  INT32_C(           0), -INT32_C(   608819032) } },
    { UINT16_C(55193),
      { -INT32_C(  1661073283), -INT32_C(  1755179327), -INT32_C(   774915976), -INT32_C(  1741240762), -INT32_C(   279816153), -INT32_C(  2112673394), -INT32_C(  1144510270),  INT32_C(   764567983),
         INT32_C(   734630250),  INT32_C(   549596071),  INT32_C(   703697635), -INT32_C(  2134759592),  INT32_C(   242160767),  INT32_C(     9470526),  INT32_C(  1522227370),  INT32_C(   612847290) },
      { -INT32_C(  2041622305),  INT32_C(  1587941755), -INT32_C(    58157149),  INT32_C(  1048332990), -INT32_C(  1672680610),  INT32_C(   412933230), -INT32_C(   294496204), -INT32_C(  2062354010),
        -INT32_C(  1005887159),  INT32_C(   371438195),  INT32_C(   135441225),  INT32_C(  1397133045), -INT32_C(   386887046), -INT32_C(  1577022354), -INT32_C(  1970179100), -INT32_C(  1240489108) },
      {  INT32_C(  2021268229),  INT32_C(   378445261),  INT32_C(  1025417544), -INT32_C(  1450089425),  INT32_C(  1687257590), -INT32_C(   234384883),  INT32_C(  1903990789),  INT32_C(  1076333627),
         INT32_C(  1958257319), -INT32_C(  2004203713),  INT32_C(   398830056),  INT32_C(    79779341), -INT32_C(   446147881), -INT32_C(   371757084),  INT32_C(  1130058503), -INT32_C(  2038201633) },
      {  INT32_C(  1694120996),  INT32_C(  1810662787),  INT32_C(   998420782), -INT32_C(   549502200),  INT32_C(  2059708566),  INT32_C(   526686999), -INT32_C(   832389137),  INT32_C(  1716905281),
        -INT32_C(  1530245343),  INT32_C(    34584276),  INT32_C(  1866371687),  INT32_C(  1800371669),  INT32_C(  1038488357), -INT32_C(  1638118737),  INT32_C(  1248640521), -INT32_C(   978271837) },
      { -INT32_C(   446072303), -INT32_C(  1729595088), -INT32_C(   536402421), -INT32_C(   917743965),  INT32_C(   419836522), -INT32_C(  2051579268), -INT32_C(  1009835232), -INT32_C(   158826267),
         INT32_C(   735834874),  INT32_C(  1992541035), -INT32_C(  1923691799), -INT32_C(  1957256671),  INT32_C(  1352948948), -INT32_C(   556442690),  INT32_C(  1671538046),  INT32_C(   525937189) },
      { -INT32_C(  2025179876), -INT32_C(   486732296), -INT32_C(   110144552), -INT32_C(   897268234) },
      {              INT32_MIN,  INT32_C(           0),  INT32_C(           0),              INT32_MIN, -INT32_C(  1521547527),  INT32_C(           0),  INT32_C(           0),              INT32_MAX,
                     INT32_MAX,  INT32_C(  1456901751),              INT32_MAX,  INT32_C(           0),  INT32_C(   339097854),  INT32_C(           0),              INT32_MAX,              INT32_MAX } },
    { UINT16_C(10273),
      { -INT32_C(   259793125),  INT32_C(  1586823613), -INT32_C(  1098335643),  INT32_C(   619947226),  INT32_C(   674425899),  INT32_C(  1031539406), -INT32_C(  2029882877),  INT32_C(  1471095612),
        -INT32_C(  1085787646), -INT32_C(  1742873293),  INT32_C(  1918346904),  INT32_C(  1989560906),  INT32_C(    77514806), -INT32_C(   717153582), -INT32_C(   916700276),  INT32_C(  1746930534) },
      {  INT32_C(  1898407997), -INT32_C(   569817786),  INT32_C(   894460139), -INT32_C(   508827989), -INT32_C(  2115614033), -INT32_C(   245946524), -INT32_C(   776293525), -INT32_C(    63251777),
        -INT32_C(  1989320381), -INT32_C(  1855490138), -INT32_C(  2100840489),  INT32_C(  1281651357),  INT32_C(   567167676), -INT32_C(   602790543), -INT32_C(  1750151976), -INT32_C(   376117082) },
      { -INT32_C(   260964023),  INT32_C(  1350687096),  INT32_C(   785533072),  INT32_C(  2004498107), -INT32_C(   224900992),  INT32_C(  1187949165),  INT32_C(   501054582), -INT32_C(  1375309468),
        -INT32_C(   341935757), -INT32_C(   482664622),  INT32_C(   588320360), -INT32_C(   979727548),  INT32_C(  1102525396),  INT32_C(  1418167773),  INT32_C(  1718707458),  INT32_C(  1226078166) },
      {  INT32_C(  1127592688),  INT32_C(   975597778), -INT32_C(  1017301122), -INT32_C(  1769408574),  INT32_C(   148389674), -INT32_C(   967024700), -INT32_C(  1691497020),  INT32_C(   904151364),
        -INT32_C(   965207564),  INT32_C(   134258314), -INT32_C(  1748279851),  INT32_C(  2133742420),  INT32_C(  1468466834),  INT32_C(   706601829), -INT32_C(   188396880), -INT32_C(  2144753268) },
      {  INT32_C(  1296474563),  INT32_C(   341132863), -INT32_C(   139714141),  INT32_C(   125229684),  INT32_C(  1163853280), -INT32_C(  1871741984),  INT32_C(  1384461510), -INT32_C(  1580028194),
        -INT32_C(  1880221616),  INT32_C(    27542366), -INT32_C(   638037916),  INT32_C(   182480682),  INT32_C(  1297038956),  INT32_C(  2145238969), -INT32_C(   774806797),  INT32_C(  1634902801) },
      {  INT32_C(   435183803),  INT32_C(   135959716),  INT32_C(   249631716), -INT32_C(   300367486) },
      {  INT32_C(    57018062),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),              INT32_MAX,  INT32_C(           0),  INT32_C(           0),
         INT32_C(           0),  INT32_C(           0),  INT32_C(           0),              INT32_MAX,  INT32_C(           0), -INT32_C(  1671139048),  INT32_C(           0),  INT32_C(           0) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i src = simde_mm512_loadu_epi32(test_vec[i].src);
    simde__m512i a0 = simde_mm512_loadu_epi32(test_vec[i].a0);
    simde__m512i a1 = simde_mm512_loadu_epi32(test_vec[i].a1);
    simde__m512i a2 = simde_mm512_loadu_epi32(test_vec[i].a2);
    simde__m512i a3 = simde_mm512_loadu_epi32(test_vec[i].a3);
    simde__m128i b = simde_x_mm_loadu_epi32(test_vec[i].b);
    simde__m512i r = simde_mm512_maskz_4dpwssds_epi32(test_vec[i].k, src, a0, a1, a2, a3, &b);
    simde_test_x86_assert_equal_i32x16(r, simde_mm512_loadu_epi32(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__mmask16 k = simde_test_x86_random_mmask16();
    simde__m512i src = simde_test_x86_random_i32x16();
    simde__m512i a0 = simde_test_x86_random_i32x16();
    simde__m512i a1 = simde_test_x86_random_i32x16();
    simde__m512i a2 = simde_test_x86_random_i32x16();
    simde__m512i a3 = simde_test_x86_random_i32x16();
    simde__m128i b = simde_test_x86_random_i32x4();
    simde__m512i r = simde_mm512_maskz_4dpwssds_epi32(k, src, a0, a1, a2, a3, &b);

    simde_test_x86_write_mmask16(2, k, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_i32x16(2, src, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, a0, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, a1, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, a2, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, a3, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x4(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

SIMDE_TEST_FUNC_LIST_BEGIN
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_4dpwssds_epi32)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_4dpwssds_epi32)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_4dpwssds_epi32)
SIMDE_TEST_FUNC_LIST_END

#include <test/x86/avx512/test-avx512-footer.h>
