!***********************************************************************
! This file is part of OpenMolcas.                                     *
!                                                                      *
! OpenMolcas is free software; you can redistribute it and/or modify   *
! it under the terms of the GNU Lesser General Public License, v. 2.1. *
! OpenMolcas is distributed in the hope that it will be useful, but it *
! is provided "as is" and without any express or implied warranties.   *
! For more details see the full text of the license in the file        *
! LICENSE or in <http://www.gnu.org/licenses/>.                        *
!                                                                      *
! Copyright (C) 2007, Ten-no Research Group                            *
!               2012, Thomas Bondo Pedersen                            *
!***********************************************************************

subroutine SetExp(K,InitR,RIni,Coeff,T)
!======================================================================*
!----------------------------------------------------------------------*
!
! Function : Determine the initial values for Exponential sum
!
! All data were prepared by Prof. W. Hackbusch and shown on the
! site below.
!
!     www.mis.mpg.de/scicomp/EXP_SUM/1_x/
!
! Author(s): Akio Takatsuka (2007)
!-----------------------------------------------------------------------

use ReMez_mod, only: IW
use Constants, only: Zero
use Definitions, only: wp, iwp

implicit none
integer(kind=iwp), parameter :: KMAX = 20, KMAX2 = 2*KMAX
integer(kind=iwp), intent(in) :: K, InitR
real(kind=wp), intent(out) :: RIni, Coeff(2,KMAX), T(KMAX2)
integer(kind=iwp) :: I, I_Dim
real(kind=wp) :: Alpha(KMAX), Omega(KMAX), Xi(KMAX2)
logical(kind=iwp) :: Dbg

Dbg = .false.

I_Dim = 2*K
Alpha(:) = Zero
Omega(:) = Zero
Xi(:) = Zero
Coeff(:,:) = Zero
T(:) = Zero

! ===== Check K value =====

select case (K)

  case default !(1)
    ! ===== K = 01 =====

    select case (InitR)

      case default !(1)
        ! ----- K = 01 , R = 2E0 -----

        RIni = 2.0_wp

        Omega(1) = 2.0009458905089673305330300046023239701753_wp

        Alpha(1) = 0.7151291879699048470680859679315943822075_wp

        Xi(1) = 1.088487745833368534800061899759_wp
        Xi(2) = 1.762080703894060781608023324996_wp

      case (2)
        ! ----- K = 01 , R = 5E0 -----

        RIni = 5.0_wp

        Omega(1) = 1.5216296303352122180679514840129229469312_wp

        Alpha(1) = 0.4931630669040556423090519017060628925719_wp

        Xi(1) = 1.170595465206690831828743870613_wp
        Xi(2) = 3.226441104658823615493681846722_wp

      case (3:31)
        ! ----- K = 01 , R = 1E1 (Best : R = 8.667) -----

        RIni = 1.0e1_wp

        Omega(1) = 1.4290997869805788563629667109466936381068_wp

        Alpha(1) = 0.4464926062994777657404622123182846848977_wp

        Xi(1) = 1.190833867508896858262469609091_wp
        Xi(2) = 3.774644895827677148602424805901_wp

    end select

  case (2)
    ! ===== K = 02 =====

    select case (InitR)

      case default !(1)
        ! ----- K = 02 , R = 2E0 -----

        RIni = 2.0_wp

        Omega(1) = 1.0724401884999553583380227372856552392477_wp
        Omega(2) = 3.2918896543848644533659941746606136803166_wp

        Alpha(1) = 0.4072692399097063106511596580761036534568_wp
        Alpha(2) = 2.4428035233716464261818218650645917477959_wp

        Xi(1) = 1.024669084390886258668224018109_wp
        Xi(2) = 1.221540551568295593040137936303_wp
        Xi(3) = 1.586346135830852440849920992338_wp
        Xi(4) = 1.940702043262358547767373884163_wp

      case (2)
        ! ----- K = 02 , R = 5E0 -----

        RIni = 5.0_wp

        Omega(1) = 0.6544138215791260005066258542694157540609_wp
        Omega(2) = 2.5128880395953934736782975356561564694857_wp

        Alpha(1) = 0.2412189720039558034640671951853718724124_wp
        Alpha(2) = 1.6503932424456509387919456166571308131097_wp

        Xi(1) = 1.053800078348716762481791786588_wp
        Xi(2) = 1.530809276455422984720630696920_wp
        Xi(3) = 2.701138971986026049171975005336_wp
        Xi(4) = 4.545078945463444416996701136213_wp

      case (3)
        ! ----- K = 02 , R = 1E1 -----

        RIni = 1.0e1_wp

        Omega(1) = 0.4596450781301019298055472034336688125222_wp
        Omega(2) = 2.2190206174604986150578761483487255645741_wp

        Alpha(1) = 0.1616927398937998487085578730448887085913_wp
        Alpha(2) = 1.3075141330652339404817807722025690964074_wp

        Xi(1) = 1.072446291884906891152262620182_wp
        Xi(2) = 1.752717114846459290429699084957_wp
        Xi(3) = 3.698767419816329259920836824982_wp
        Xi(4) = 8.233481847579251056146176779293_wp

      case (4)
        ! ----- K = 02 , R = 2E1 -----

        RIni = 2.0e1_wp

        Omega(1) = 0.3456883945557714279787908662200379694696_wp
        Omega(2) = 2.0588931989746813778271311567991119773069_wp

        Alpha(1) = 0.1137705533481849198753493856983709520136_wp
        Alpha(2) = 1.1122420832263719987081929119199230626691_wp

        Xi(1) = 1.086388801185830942520188502343_wp
        Xi(2) = 1.931390216502200829159531514367_wp
        Xi(3) = 4.620528669639594092268669589529_wp
        Xi(4) = 13.614404138913966666943333727602_wp

      case (5)
        ! ----- K = 02 , R = 3E1 -----

        RIni = 3.0e1_wp

        Omega(1) = 0.3089269872856084493337180951755271962611_wp
        Omega(2) = 2.0075705464402703406302796995674952995614_wp

        Alpha(1) = 0.0979831430245671137333374374478456303450_wp
        Alpha(2) = 1.0494466863388992488356071808297542702348_wp

        Xi(1) = 1.091649420750634746956134191187_wp
        Xi(2) = 2.001884214128133870425410578697_wp
        Xi(3) = 5.015778281485991329114587644611_wp
        Xi(4) = 16.815914202468247307109550092719_wp

      case (6)
        ! ----- K = 02 , R = 4E1 -----

        RIni = 4.0e1_wp

        Omega(1) = 0.2978387832240039867037588450227048042507_wp
        Omega(2) = 1.9920021768207375049194313154643509733432_wp

        Alpha(1) = 0.0931855955444948900315608152078450387990_wp
        Alpha(2) = 1.0304678137783844420281159637298173947784_wp

        Xi(1) = 1.093332292125060299864749979726_wp
        Xi(2) = 2.024812554457079488527682098464_wp
        Xi(3) = 5.148542084107910186442647049176_wp
        Xi(4) = 18.048492691174313850774324485826_wp

      case (7:31)
        ! ----- K = 02 , R = 5E1 (Best : R = 41.54) -----

        RIni = 5.0e1_wp

        Omega(1) = 0.2976555432327632915757986697968817679794_wp
        Omega(2) = 1.9917443679019896428470520755027450832131_wp

        Alpha(1) = 0.0931061725688367425719750222756765367649_wp
        Alpha(2) = 1.0301539335400714347910217094650420222024_wp

        Xi(1) = 1.093360513875665603511673495962_wp
        Xi(2) = 2.025198654658210892275452263966_wp
        Xi(3) = 5.150796128864857912049018962009_wp
        Xi(4) = 18.070172281159798970476271229302_wp

    end select

  case (3)
    ! ===== K = 03 =====

    select case (InitR)

      case default !(1)
        ! ----- K = 03 , R = 2E0 -----

        RIni = 2.0_wp

        Omega(1) = 0.7470594125158281184444220013318727069418_wp
        Omega(2) = 1.9588903206974837864322303015995885289158_wp
        Omega(3) = 4.1617883704995866428130724123946038162103_wp

        Alpha(1) = 0.2874933084171688071331227098959004706558_wp
        Alpha(2) = 1.6046370491507346881746715050454099582566_wp
        Alpha(3) = 4.5064864376399242354084839767836001556134_wp

        Xi(1) = 1.011367648358682691891487215319_wp
        Xi(2) = 1.102107335879371621334757436195_wp
        Xi(3) = 1.280275921324935759957296121936_wp
        Xi(4) = 1.528180108975737024209172465472_wp
        Xi(5) = 1.792807750855102859387051050799_wp
        Xi(6) = 1.974208219701866289202789317425_wp

      case (2)
        ! ----- K = 03 , R = 5E0 -----

        RIni = 5.0_wp

        Omega(1) = 0.4365792105033683860376356178356616055680_wp
        Omega(2) = 1.2723305210744182858640521915916110629041_wp
        Omega(3) = 3.2151540832501005597130611102230091091769_wp

        Alpha(1) = 0.1660731375014149112742277078491603958810_wp
        Alpha(2) = 0.9784372085453775896011531509888214941384_wp
        Alpha(3) = 3.0530159808682457700784035337093769157946_wp

        Xi(1) = 1.025951895784413650568669962304_wp
        Xi(2) = 1.242824075650160353407464031150_wp
        Xi(3) = 1.722934671753702002227302825244_wp
        Xi(4) = 2.541326431637942081815523254207_wp
        Xi(5) = 3.710191462368181307841102922751_wp
        Xi(6) = 4.812712621252895713562514057315_wp

      case (3)
        ! ----- K = 03 , R = 1E1 -----

        RIni = 1.0e1_wp

        Omega(1) = 0.2847348675836868480131403719157745513257_wp
        Omega(2) = 0.9583115118002354952046577263313764660779_wp
        Omega(3) = 2.8443772254878889997348895279927205592685_wp

        Alpha(1) = 0.1064455484301103369894087730118670265256_wp
        Alpha(2) = 0.6791933646735415630178896162494339705518_wp
        Alpha(3) = 2.4024163101166760476718603500856374921568_wp

        Xi(1) = 1.036540052009057201009949167325_wp
        Xi(2) = 1.350512629433860964335220589128_wp
        Xi(3) = 2.095440330945998347236119974113_wp
        Xi(4) = 3.530895274669204600285002393356_wp
        Xi(5) = 6.064676170659236835534106058176_wp
        Xi(6) = 9.312377099278923659429751946703_wp

      case (4)
        ! ----- K = 03 , R = 2E1 -----

        RIni = 2.0e1_wp

        Omega(1) = 0.1871865945351820821081342480218800972125_wp
        Omega(2) = 0.7633857188437595157155454717035070189013_wp
        Omega(3) = 2.6169711142660253494832134357750419439981_wp

        Alpha(1) = 0.0679013689405892574945246102646745356424_wp
        Alpha(2) = 0.4897948875089661041485125658745403143257_wp
        Alpha(3) = 1.9962015809195739495960839060550995327503_wp

        Xi(1) = 1.046004741949051884642926668612_wp
        Xi(2) = 1.450848248703333635638811072166_wp
        Xi(3) = 2.466950930523215088823255070061_wp
        Xi(4) = 4.633421766226166740951925016034_wp
        Xi(5) = 9.248118901369171863426077440096_wp
        Xi(6) = 17.567914206128274501664265905276_wp

      case (5)
        ! ----- K = 03 , R = 3E1 -----

        RIni = 3.0e1_wp

        Omega(1) = 0.1489350280707968614941037743815144267501_wp
        Omega(2) = 0.6872993149781066612611854504955033462466_wp
        Omega(3) = 2.5251745372914686719838239792323975052568_wp

        Alpha(1) = 0.0526373510053126044740958670159480803363_wp
        Alpha(2) = 0.4159550868841870631409125130817372451020_wp
        Alpha(3) = 1.8359809178980240516269178963071340149327_wp

        Xi(1) = 1.050731777552041640516958254103_wp
        Xi(2) = 1.502520887461407482184674866410_wp
        Xi(3) = 2.667783186472804712440626273562_wp
        Xi(4) = 5.279908463448491815882962496786_wp
        Xi(5) = 11.391839687100992576165858771731_wp
        Xi(6) = 24.939047426082355123772016369799_wp

      case (6)
        ! ----- K = 03 , R = 4E1 -----

        RIni = 4.0e1_wp

        Omega(1) = 0.1282763604743762873410797484274858959452_wp
        Omega(2) = 0.6457800255491172428698280760794858679219_wp
        Omega(3) = 2.4737466774457358498554016001946820324520_wp

        Alpha(1) = 0.0443249082477236827920660192786650455332_wp
        Alpha(2) = 0.3760267128377692993999176218622437772865_wp
        Alpha(3) = 1.7481061682069727251757906838491862799856_wp

        Xi(1) = 1.053649416445949383441815783158_wp
        Xi(2) = 1.534965201688940375458411469189_wp
        Xi(3) = 2.797338196085134732517263334017_wp
        Xi(4) = 5.717225016873238558224451510981_wp
        Xi(5) = 12.960177258090050030417406112804_wp
        Xi(6) = 31.526153474782242738805315696027_wp

      case (7)
        ! ----- K = 03 , R = 5E1 -----

        RIni = 5.0e1_wp

        Omega(1) = 0.1153310756696578369059902335891365510179_wp
        Omega(2) = 0.6194232578183114542973164429184862456168_wp
        Omega(3) = 2.4405036326787768433295433556651232720469_wp

        Alpha(1) = 0.0390820984323069169736556555116990807619_wp
        Alpha(2) = 0.3509230683778314732214603094417881834488_wp
        Alpha(3) = 1.6921947262166477795545388440601186630374_wp

        Xi(1) = 1.055646218481340204787058412172_wp
        Xi(2) = 1.557420836116662476826533190177_wp
        Xi(3) = 2.888617000023559338922271577665_wp
        Xi(4) = 6.035444707901820501731726675843_wp
        Xi(5) = 14.162824787784242307718540754280_wp
        Xi(6) = 37.407934771317043306376737277219_wp

      case (8)
        ! ----- K = 03 , R = 6E1 -----

        RIni = 6.0e1_wp

        Omega(1) = 0.1065007940761846950413838444604586186415_wp
        Omega(2) = 0.6012169550303838670810181732395705012095_wp
        Omega(3) = 2.4172446838218981427469367329052829518332_wp

        Alpha(1) = 0.0354878575633928706640498294822005931337_wp
        Alpha(2) = 0.3337331092077952527763084339396115751697_wp
        Alpha(3) = 1.6535367452293752402354976016596310728346_wp

        Xi(1) = 1.057097434208872948745855380182_wp
        Xi(2) = 1.573870966645937349978265351602_wp
        Xi(3) = 2.956339877497224052612170841492_wp
        Xi(4) = 6.277178023394458090695319230079_wp
        Xi(5) = 15.111869488781895343176842061439_wp
        Xi(6) = 42.640384216822704587651626795264_wp

      case (9)
        ! ----- K = 03 , R = 7E1 -----

        RIni = 7.0e1_wp

        Omega(1) = 0.1001553077368610132974404043626481808360_wp
        Omega(2) = 0.5879845498502223689126118533199161220182_wp
        Omega(3) = 2.4001809699912176228331350813505196128972_wp

        Alpha(1) = 0.0328950169895424992609872082843791574192_wp
        Alpha(2) = 0.3213333776732262143905564238943739496790_wp
        Alpha(3) = 1.6254323321274150328501900686362091619230_wp

        Xi(1) = 1.058191270863622694426570136628_wp
        Xi(2) = 1.586343845420391796365890135778_wp
        Xi(3) = 3.008179205772900836370065125180_wp
        Xi(4) = 6.465564686699797597186495368149_wp
        Xi(5) = 15.873238341608802483631568414069_wp
        Xi(6) = 47.261670190788973886647506006398_wp

      case (10)
        ! ----- K = 03 , R = 8E1 -----

        RIni = 8.0e1_wp

        Omega(1) = 0.0954485614946326069841066666588069722366_wp
        Omega(2) = 0.5780719785823823125598015792991901662390_wp
        Omega(3) = 2.3873083432732535235590992650500652416667_wp

        Alpha(1) = 0.0309660405726075814911553646738284761852_wp
        Alpha(2) = 0.3121038074526534121424636153419029938050_wp
        Alpha(3) = 1.6043802276101923300543825035013867363887_wp

        Xi(1) = 1.059033121789881161920778007701_wp
        Xi(2) = 1.595987137144691836248243388408_wp
        Xi(3) = 3.048551860121487124032427229459_wp
        Xi(4) = 6.614345532157845423959496855559_wp
        Xi(5) = 16.488307114705869405035798536119_wp
        Xi(6) = 51.297243347978425765526511170833_wp

      case (11)
        ! ----- K = 03 , R = 9E1 -----

        RIni = 9.0e1_wp

        Omega(1) = 0.0918992098575219402425365983200222785854_wp
        Omega(2) = 0.5705336658200345004285022887380307565763_wp
        Omega(3) = 2.3774665399131085228353471583417899637425_wp

        Alpha(1) = 0.0295080586695789376959540937367387769541_wp
        Alpha(2) = 0.3051223245003002779000573357892633907795_wp
        Alpha(3) = 1.5883739921280894394003957437178087275242_wp

        Xi(1) = 1.059686558350682067061211688586_wp
        Xi(2) = 1.603499064055321644611459830365_wp
        Xi(3) = 3.080180337433629595518189137060_wp
        Xi(4) = 6.732188839664491609906388225326_wp
        Xi(5) = 16.984271206669289473734085760270_wp
        Xi(6) = 54.763342161663440026825311335301_wp

      case (12)
        ! ----- K = 03 , R = 1E2 -----

        RIni = 1.0e2_wp

        Omega(1) = 0.0892157079089669919486071947289929084945_wp
        Omega(2) = 0.5647942501087851651861426738321370066842_wp
        Omega(3) = 2.3699428171677330734901795183944273048837_wp

        Alpha(1) = 0.0284038048199532795564959406289795573741_wp
        Alpha(2) = 0.2998300570947111501655338999805522348652_wp
        Alpha(3) = 1.5761907592561199853995260178329829159338_wp

        Xi(1) = 1.060191971776997436806982966218_wp
        Xi(2) = 1.609324357152911759254176882195_wp
        Xi(3) = 3.104816257355714646673852663561_wp
        Xi(4) = 6.824779154407094683074119778610_wp
        Xi(5) = 17.379498870857770060141533896569_wp
        Xi(6) = 57.669325212093752022757264796837_wp

      case (13:31)
        ! ----- K = 03 , R = 2E2 (Best : R = 146.8) -----

        RIni = 2.0e2_wp

        Omega(1) = 0.0842336964653581986314334444088114395299_wp
        Omega(2) = 0.5540364331533339688207195905533808399923_wp
        Omega(3) = 2.3557688429224689736704578812975796608953_wp

        Alpha(1) = 0.0263492728481437144350885744964463341944_wp
        Alpha(2) = 0.2899685548481513726706056827131519071372_wp
        Alpha(3) = 1.5533661852788560075230156476422394007386_wp

        Xi(1) = 1.061157926448131444429646974559_wp
        Xi(2) = 1.620497451354158359860917204553_wp
        Xi(3) = 3.152337144451245588925442087103_wp
        Xi(4) = 7.005381504309560792426170294789_wp
        Xi(5) = 18.164730430146035160265949315317_wp
        Xi(6) = 63.847716086263413940576905147850_wp

    end select

  case (4)
    ! ===== K = 04 =====

    select case (InitR)

      case default !(1)
        ! ----- K = 04 , R = 2E0 -----

        RIni = 2.0_wp

        Omega(1) = 0.5753086924118193929257938923349513515859_wp
        Omega(2) = 1.4324382751398495198852031662362094266427_wp
        Omega(3) = 2.6015042679469189200285039742510662108543_wp
        Omega(4) = 4.8372822744559876655644214960005911052576_wp

        Alpha(1) = 0.2225282101342608647137188276898811523097_wp
        Alpha(2) = 1.2118271769428668733874221441482177397120_wp
        Alpha(3) = 3.1874817920306483924348245340141261294775_wp
        Alpha(4) = 6.7398873748331835607794371156842316850089_wp

        Xi(1) = 1.006506167989540905796305714226_wp
        Xi(2) = 1.058464707807605930571862029588_wp
        Xi(3) = 1.161505681175087609235951902065_wp
        Xi(4) = 1.311950847515082049137349895496_wp
        Xi(5) = 1.499364570000543965826340919723_wp
        Xi(6) = 1.701144502067884147606356692872_wp
        Xi(7) = 1.879073519869262322767519690636_wp
        Xi(8) = 1.985688675481389943249486973453_wp

      case (2)
        ! ----- K = 04 , R = 5E0 -----

        RIni = 5.0_wp

        Omega(1) = 0.3311114029441486656963472556824967796274_wp
        Omega(2) = 0.8719992880247880499789900343721171793732_wp
        Omega(3) = 1.7621447067201108642887225075313040179026_wp
        Omega(4) = 3.7722154835897864035877680155550706331269_wp

        Alpha(1) = 0.1272851568278456049125666599275419343940_wp
        Alpha(2) = 0.7128840158683421319844343566618505292354_wp
        Alpha(3) = 1.9837003473579683075636928557550220375560_wp
        Alpha(4) = 4.5827403151170012138813925073321797754033_wp

        Xi(1) = 1.015209771693483679195178781551_wp
        Xi(2) = 1.139795898096606982757665604833_wp
        Xi(3) = 1.403933946114703158530509496948_wp
        Xi(4) = 1.834881569408138940859356869062_wp
        Xi(5) = 2.463725443706430423464887047302_wp
        Xi(6) = 3.294920084559301091866462263269_wp
        Xi(7) = 4.220215506742421560527206425562_wp
        Xi(8) = 4.899608368220416046347892580215_wp

      case (3)
        ! ----- K = 04 , R = 1E1 -----

        RIni = 1.0e1_wp

        Omega(1) = 0.2109714423067830878737302299774647451613_wp
        Omega(2) = 0.6012085214120437117791585779347229845371_wp
        Omega(3) = 1.3842061450697847088241870805447319980885_wp
        Omega(4) = 3.3493765842990434143926908872046510623477_wp

        Alpha(1) = 0.0804090244155528715704300650457536647764_wp
        Alpha(2) = 0.4687930946688866533453993179492158560606_wp
        Alpha(3) = 1.4098715378713611090033888673112016931555_wp
        Alpha(4) = 3.6048552773416722889727320522368358979293_wp

        Xi(1) = 1.021921225313027279247950207530_wp
        Xi(2) = 1.204384941419539613869163496851_wp
        Xi(3) = 1.607251451203153021457352422896_wp
        Xi(4) = 2.309746723025455121362792265671_wp
        Xi(5) = 3.442661796455891675284438635529_wp
        Xi(6) = 5.174135212344695517513309956925_wp
        Xi(7) = 7.518763336083220949045829994972_wp
        Xi(8) = 9.646065333930355733746409008234_wp

      case (4)
        ! ----- K = 04 , R = 2E1 -----

        RIni = 2.0e1_wp

        Omega(1) = 0.1329180070416147036311800902486091047194_wp
        Omega(2) = 0.4264908534251436216986307081722173961680_wp
        Omega(3) = 1.1416477627542353926511781159192082668596_wp
        Omega(4) = 3.0780675346606618560107992976782043115236_wp

        Alpha(1) = 0.0499698775458987140683406961289003689330_wp
        Alpha(2) = 0.3105631707014663601691250444103431505027_wp
        Alpha(3) = 1.0398249107852826248832647793030048433138_wp
        Alpha(4) = 2.9750469432139245655032783188786993378017_wp

        Xi(1) = 1.028384820056705543857462903379_wp
        Xi(2) = 1.268165682588351670712946417385_wp
        Xi(3) = 1.816842986610023565584304261478_wp
        Xi(4) = 2.829868970539194235119459519545_wp
        Xi(5) = 4.612352038396031943777442219101_wp
        Xi(6) = 7.714494741864398525896251435086_wp
        Xi(7) = 12.817017304467905825828000221378_wp
        Xi(8) = 18.811438665072096257727118917558_wp

      case (5)
        ! ----- K = 04 , R = 3E1 -----

        RIni = 3.0e1_wp

        Omega(1) = 0.1016013598614176051835515483812155679288_wp
        Omega(2) = 0.3564500076585994777433278463130150726101_wp
        Omega(3) = 1.0408682232432721238716069600549474216677_wp
        Omega(4) = 2.9620670599932430282208439820124112884514_wp

        Alpha(1) = 0.0377493790524789586313635092462703823912_wp
        Alpha(2) = 0.2471731689001984523724369038899517647678_wp
        Alpha(3) = 0.8901553582336263936887477132842150240322_wp
        Alpha(4) = 2.7154179633949424115240550703376243291132_wp

        Xi(1) = 1.031884400461319287699291091798_wp
        Xi(2) = 1.303394819838255136359300712101_wp
        Xi(3) = 1.936484594904135799484437563400_wp
        Xi(4) = 3.140840870036364630224837157790_wp
        Xi(5) = 5.359510014817230771043410353371_wp
        Xi(6) = 9.492142930507176949023184775456_wp
        Xi(7) = 17.064887801084204502904961842091_wp
        Xi(8) = 27.603035761750134504874387708639_wp

      case (6)
        ! ----- K = 04 , R = 4E1 -----

        RIni = 4.0e1_wp

        Omega(1) = 0.0842613451067411968134360580884312241778_wp
        Omega(2) = 0.3175083683168243892547700363282459079528_wp
        Omega(3) = 0.9827580985120555624170654573923400221247_wp
        Omega(4) = 2.8938728508266506878315177386795653546869_wp

        Alpha(1) = 0.0309718827633249730023221003502587045375_wp
        Alpha(2) = 0.2120774570049154385080730084167655036254_wp
        Alpha(3) = 0.8062000011279130441146055507939394146888_wp
        Alpha(4) = 2.5672235706906126311704907649868800945114_wp

        Xi(1) = 1.034189547957952938364925565828_wp
        Xi(2) = 1.326882303593381459455753657029_wp
        Xi(3) = 2.017833912902638116102643772720_wp
        Xi(4) = 3.358339765728229320143963243517_wp
        Xi(5) = 5.903755879081696503109938367615_wp
        Xi(6) = 10.860721975215827649410837985045_wp
        Xi(7) = 20.628245205359198052516567223336_wp
        Xi(8) = 36.056476263170379424616296404338_wp

      case (7)
        ! ----- K = 04 , R = 5E1 -----

        RIni = 5.0e1_wp

        Omega(1) = 0.0731190495973734073321850628068840904916_wp
        Omega(2) = 0.2923269496573136512322394625629229381047_wp
        Omega(3) = 0.9440309351700997790167620149492933023794_wp
        Omega(4) = 2.8478366618531806178475640489722309212084_wp

        Alpha(1) = 0.0266079437201842705628133858158923530368_wp
        Alpha(2) = 0.1895065774790961192112310682023057495371_wp
        Alpha(3) = 0.7515202985781507951474830764304613239801_wp
        Alpha(4) = 2.4693480571818920657912316274362751755689_wp

        Xi(1) = 1.035859420488966962437850372059_wp
        Xi(2) = 1.344041617001740170165430265925_wp
        Xi(3) = 2.078083732500034939115474141857_wp
        Xi(4) = 3.522680772883364073815026795700_wp
        Xi(5) = 6.327056844638300348981213572230_wp
        Xi(6) = 11.967990442183917604648812904600_wp
        Xi(7) = 23.694624806761000475330281034303_wp
        Xi(8) = 44.192689038985971203971070409011_wp

      case (8)
        ! ----- K = 04 , R = 6E1 -----

        RIni = 6.0e1_wp

        Omega(1) = 0.0653078213757228939496122036012515366110_wp
        Omega(2) = 0.2745449432568480962601076972307012624697_wp
        Omega(3) = 0.9160037860065624061296600533221834439246_wp
        Omega(4) = 2.8142148838640842138485165246919450510177_wp

        Alpha(1) = 0.0235419918657027563556896837884679918318_wp
        Alpha(2) = 0.1736604768046437706328520220178113220300_wp
        Alpha(3) = 0.7126891672717474266905354762791091616236_wp
        Alpha(4) = 2.3990539647469760751227058070966791092360_wp

        Xi(1) = 1.037140461619688747412632023881_wp
        Xi(2) = 1.357289114402989183786468752047_wp
        Xi(3) = 2.125078140520427659714702839366_wp
        Xi(4) = 3.652834342225301128646158810298_wp
        Xi(5) = 6.669801867650820922320958894502_wp
        Xi(6) = 12.892172317412252485931134504682_wp
        Xi(7) = 26.378389732187363631021881360539_wp
        Xi(8) = 52.027795795803441894489260732826_wp

      case (9)
        ! ----- K = 04 , R = 7E1 -----

        RIni = 7.0e1_wp

        Omega(1) = 0.0595075285913887735555084096406019966707_wp
        Omega(2) = 0.2612399364166240053278938981762280491239_wp
        Omega(3) = 0.8946057701502559431701484815580016629610_wp
        Omega(4) = 2.7883700207922745009040538244349249907827_wp

        Alpha(1) = 0.0212604165557323362225922592366744723336_wp
        Alpha(2) = 0.1618724211569130696873998823126505897108_wp
        Alpha(3) = 0.6835043049845140389537447023204919105410_wp
        Alpha(4) = 2.3457307234433385698373769612246064752981_wp

        Xi(1) = 1.038161965305633614828524635687_wp
        Xi(2) = 1.367906084839922344650701979685_wp
        Xi(3) = 2.163045651450423545025109062223_wp
        Xi(4) = 3.759269913511129595096865729786_wp
        Xi(5) = 6.955090352815260643353639791187_wp
        Xi(6) = 13.680462783788660021712380032710_wp
        Xi(7) = 28.756568839488641399840251899889_wp
        Xi(8) = 59.575852601901405913853437823491_wp

      case (10)
        ! ----- K = 04 , R = 8E1 -----

        RIni = 8.0e1_wp

        Omega(1) = 0.0550202496658656947137210676407326204185_wp
        Omega(2) = 0.2508680510132605612151797391806695713967_wp
        Omega(3) = 0.8776426020163860330729720116149650266379_wp
        Omega(4) = 2.7677729356352664641209571083635410104762_wp

        Alpha(1) = 0.0194916432536069991175730915095032358408_wp
        Alpha(2) = 0.1527342674760213229605320187087791339309_wp
        Alpha(3) = 0.6606717166460769025495593231855906424244_wp
        Alpha(4) = 2.3036894119803122266132433537677570711821_wp

        Xi(1) = 1.038999699185351887459859143714_wp
        Xi(2) = 1.376648661571191919659150337818_wp
        Xi(3) = 2.194515905695126582828496930588_wp
        Xi(4) = 3.848374497746649226153151102103_wp
        Xi(5) = 7.197426589624710588366518493331_wp
        Xi(6) = 14.363811748530706862911343613121_wp
        Xi(7) = 30.884378731911624308553721007797_wp
        Xi(8) = 66.849598004065109602134242550164_wp

      case (11)
        ! ----- K = 04 , R = 9E1 -----

        RIni = 9.0e1_wp

        Omega(1) = 0.0514404134505259737501683264815355300925_wp
        Omega(2) = 0.2425316805672637418404054970855199258040_wp
        Omega(3) = 0.8638140458215766392642900872722577787499_wp
        Omega(4) = 2.7509105528330038540653357248544352842146_wp

        Alpha(1) = 0.0180777829160191627153871996372402897180_wp
        Alpha(2) = 0.1454283956184184608531519827812061862460_wp
        Alpha(3) = 0.6422668197625267019735227636001440032487_wp
        Alpha(4) = 2.2695770442654134364975082593929300855962_wp

        Xi(1) = 1.039701559473601906805062533579_wp
        Xi(2) = 1.383998181274938783931992281051_wp
        Xi(3) = 2.221116333865750902369640140677_wp
        Xi(4) = 3.924322959823527990578378998876_wp
        Xi(5) = 7.406527255831039190710901598536_wp
        Xi(6) = 14.963700097020325378849303632478_wp
        Xi(7) = 32.803053350819499158624870105427_wp
        Xi(8) = 73.860605145714235443554773752339_wp

      case (12)
        ! ----- K = 04 , R = 1E2 -----

        RIni = 1.0e2_wp

        Omega(1) = 0.0485154483001083335682321676096684370805_wp
        Omega(2) = 0.2356710456997001232359086747569776321143_wp
        Omega(3) = 0.8522950664164703286885226007907334633273_wp
        Omega(4) = 2.7368154329484132645643884584885086042050_wp

        Alpha(1) = 0.0169204412159692850155435659906255629892_wp
        Alpha(2) = 0.1394458611981022595169591934971542457333_wp
        Alpha(3) = 0.6270840649394218112842348811231119043441_wp
        Alpha(4) = 2.2412769022182530667848532335284517102991_wp

        Xi(1) = 1.040299549550007821561471710314_wp
        Xi(2) = 1.390278043830974921083593054405_wp
        Xi(3) = 2.243950554331804970228242401831_wp
        Xi(4) = 3.989985712671379214496955589730_wp
        Xi(5) = 7.589215465942349175560571739041_wp
        Xi(6) = 15.495677450862050677712900004224_wp
        Xi(7) = 34.544307705980290587599590423906_wp
        Xi(8) = 80.619317886311765174178756154788_wp

      case (13)
        ! ----- K = 04 , R = 2E2 -----

        RIni = 2.0e2_wp

        Omega(1) = 0.0346528544291562165083103903839845116863_wp
        Omega(2) = 0.2022620232531529587587451612940014911146_wp
        Omega(3) = 0.7941993308902318025530031597725155734224_wp
        Omega(4) = 2.6650476801253452636790430840818544311333_wp

        Alpha(1) = 0.0114012537962333943318828338546466749648_wp
        Alpha(2) = 0.1108289024285984015197803129182396730812_wp
        Alpha(3) = 0.5527015496667354365950206029900471094152_wp
        Alpha(4) = 2.1002718748225481968597977422419376125617_wp

        Xi(1) = 1.043510941463284021849672322801_wp
        Xi(2) = 1.424292069942840605165264300513_wp
        Xi(3) = 2.369336520202418435061819668164_wp
        Xi(4) = 4.358428194352952914601695422192_wp
        Xi(5) = 8.647755843713323312300922385276_wp
        Xi(6) = 18.726822746608632615228362539028_wp
        Xi(7) = 45.936015344901233076885249673182_wp
        Xi(8) = 135.930079665824886978042762564201_wp

      case (14)
        ! ----- K = 04 , R = 3E2 -----

        RIni = 3.0e2_wp

        Omega(1) = 0.0300842254221815340984474453078778566351_wp
        Omega(2) = 0.1907535375929733114611520222725360440563_wp
        Omega(3) = 0.7733295903723639011994803182936664143199_wp
        Omega(4) = 2.6389901708906205511726700274621748576465_wp

        Alpha(1) = 0.0095667747961039157790762718983068602085_wp
        Alpha(2) = 0.1012379805187263745680720586350709311318_wp
        Alpha(3) = 0.5269302725083039626134684407787744930829_wp
        Alpha(4) = 2.0503770061024033061029825386611946669291_wp

        Xi(1) = 1.044749147786660410067334503470_wp
        Xi(2) = 1.437539613008931760579182934645_wp
        Xi(3) = 2.418961360192959209481494697869_wp
        Xi(4) = 4.508037017931998435057300289586_wp
        Xi(5) = 9.094182415437122085286048012165_wp
        Xi(6) = 20.168709377634733440065595289070_wp
        Xi(7) = 51.485939374184318526039838559427_wp
        Xi(8) = 171.534260231812210592550194121486_wp

      case (15)
        ! ----- K = 04 , R = 4E2 -----

        RIni = 4.0e2_wp

        Omega(1) = 0.0284661205929678002576150783187181048106_wp
        Omega(2) = 0.1865878163370500782299293146548890831582_wp
        Omega(3) = 0.7656572669069769046914022969652080519154_wp
        Omega(4) = 2.6293741631156146041845000915770924621029_wp

        Alpha(1) = 0.0089149584966167103430493114449295610768_wp
        Alpha(2) = 0.0978104436659742490270375535332814109779_wp
        Alpha(3) = 0.5175885566543470744449786935525992248586_wp
        Alpha(4) = 2.0321413590981910171542196286154080553388_wp

        Xi(1) = 1.045216293509242250106376637131_wp
        Xi(2) = 1.442556485572168388509251002461_wp
        Xi(3) = 2.437872235678494421511064427222_wp
        Xi(4) = 4.565631989514494578404762981094_wp
        Xi(5) = 9.268663264519454292669764505774_wp
        Xi(6) = 20.745603718304171844341587949145_wp
        Xi(7) = 53.790818525643893238674664303289_wp
        Xi(8) = 188.385827462242781263346280695714_wp

      case (16:31)
        ! ----- K = 04 , R = 5E2 (Best : R = 436.1) -----

        RIni = 5.0e2_wp

        Omega(1) = 0.0283509824145518462858693315570279835924_wp
        Omega(2) = 0.1862893386105706895712131113129750303870_wp
        Omega(3) = 0.7651050730792038023189878537255737001033_wp
        Omega(4) = 2.6286812528458147286778562312647977705637_wp

        Alpha(1) = 0.0088685367946663247640828832071835385875_wp
        Alpha(2) = 0.0975658350270974381366175312568778110744_wp
        Alpha(3) = 0.5169189990170087889041414253021855529369_wp
        Alpha(4) = 2.0308311260118173089978671486122152600728_wp

        Xi(1) = 1.045250181977026276611796029581_wp
        Xi(2) = 1.442920714529975515167184441090_wp
        Xi(3) = 2.439247862946276155723945855414_wp
        Xi(4) = 4.569832780139895886221956811113_wp
        Xi(5) = 9.281448986761759875618871262759_wp
        Xi(6) = 20.788176902012078735987321564949_wp
        Xi(7) = 53.962878608716110463011661479982_wp
        Xi(8) = 189.696990422674594614638188261324_wp

    end select

  case (5)
    ! ===== K = 05 =====

    select case (InitR)

      case default !(1)
        ! ----- K = 05 , R = 2E0 -----

        RIni = 2.0_wp

        Omega(1) = 0.4683556115061091129983267961067738838210_wp
        Omega(2) = 1.1387420740801897163681222213682531219092_wp
        Omega(3) = 1.9531818897084401308490708704113103522104_wp
        Omega(4) = 3.1153038431409946034094521349899764572910_wp
        Omega(5) = 5.3979418510235632052904342081944832898444_wp

        Alpha(1) = 0.1816122344478981142344448801440037755128_wp
        Alpha(2) = 0.9776830415195706051986919427054800735277_wp
        Alpha(3) = 2.5060472629612186804687334884533811418805_wp
        Alpha(4) = 4.9959238094351378477869785044163108977955_wp
        Alpha(5) = 9.0782980472197038765772014734523054357851_wp

        Xi(1) = 1.004205961424876742218471881163_wp
        Xi(2) = 1.037808810177569494902563573824_wp
        Xi(3) = 1.104673526155082756558187262463_wp
        Xi(4) = 1.203620646526363327142271353321_wp
        Xi(5) = 1.331606110593612308330355165076_wp
        Xi(6) = 1.482173116895686719907306028166_wp
        Xi(7) = 1.643636751837373089933300174259_wp
        Xi(8) = 1.797882208171468448811188001191_wp
        Xi(9) = 1.921494689550123476790011167736_wp
        Xi(10) = 1.990922307578263482060257349993_wp

      case (2)
        ! ----- K = 05 , R = 5E0 -----

        RIni = 5.0_wp

        Omega(1) = 0.2675144938257100701254221492009222060915_wp
        Omega(2) = 0.6731648185410648106067992013645806537170_wp
        Omega(3) = 1.2341495375414908785378573230495646839699_wp
        Omega(4) = 2.1730749987606481090328391925226014791406_wp
        Omega(5) = 4.2400491045957718516841439271303215718945_wp

        Alpha(1) = 0.1033351211759554590466098772383318760149_wp
        Alpha(2) = 0.5658998554798738632692738093066253668439_wp
        Alpha(3) = 1.5002594864446179758615818045086598431226_wp
        Alpha(4) = 3.1563889823905420255363563075690080950153_wp
        Alpha(5) = 6.1935841722085905663650051700130916287890_wp

        Xi(1) = 1.009973918929146431255200466737_wp
        Xi(2) = 1.090947626405043171474856800174_wp
        Xi(3) = 1.259063365124301121355382404587_wp
        Xi(4) = 1.526000210062745005099721362640_wp
        Xi(5) = 1.907288936960188738592957002904_wp
        Xi(6) = 2.417693978255027923308234649724_wp
        Xi(7) = 3.058580838005749557607926281122_wp
        Xi(8) = 3.789900418969845993441558640313_wp
        Xi(9) = 4.488508901774702612979400795368_wp
        Xi(10) = 4.937710663142177220773926249464_wp

      case (3)
        ! ----- K = 05 , R = 1E1 -----

        RIni = 1.0e1_wp

        Omega(1) = 0.1686100332421062595255559918294707699715_wp
        Omega(2) = 0.4447462226391396488648245083341592476245_wp
        Omega(3) = 0.8914300222454683441828879320034673128248_wp
        Omega(4) = 1.7540776690871601983497465870875942073326_wp
        Omega(5) = 3.7773593670209078805664187905577477977204_wp

        Alpha(1) = 0.0647973162949741815670734457199841926922_wp
        Alpha(2) = 0.3632942397911341266415318623783647922210_wp
        Alpha(3) = 1.0098084995269309119183404055419828182494_wp
        Alpha(4) = 2.2813077174805524267257178516743465479522_wp
        Alpha(5) = 4.8771539519428148820243129613061228155857_wp

        Xi(1) = 1.014582863653057650654004673285_wp
        Xi(2) = 1.134203389307848182355163701995_wp
        Xi(3) = 1.389246760143581714357606526900_wp
        Xi(4) = 1.812037270091620028882387072322_wp
        Xi(5) = 2.454117398216207501576366367679_wp
        Xi(6) = 3.388587251028423504711772906717_wp
        Xi(7) = 4.699016817830402586476934434501_wp
        Xi(8) = 6.418823685933021436830075989732_wp
        Xi(9) = 8.349475129996430378988281351127_wp
        Xi(10) = 9.786083246339030821889237721223_wp

      case (4)
        ! ----- K = 05 , R = 2E1 -----

        RIni = 2.0e1_wp

        Omega(1) = 0.1043419845604544110779284687984258539473_wp
        Omega(2) = 0.2955585665005043083473218468171239692310_wp
        Omega(3) = 0.6678313388646436865592193043994484469295_wp
        Omega(4) = 1.4788816616452081139634552875428141760494_wp
        Omega(5) = 3.4728708916802090379740558523380400401948_wp

        Alpha(1) = 0.0397910355186319709875524593817308627308_wp
        Alpha(2) = 0.2313133225382769353847172214000416090585_wp
        Alpha(3) = 0.6899245534043173482918569205946113243044_wp
        Alpha(4) = 1.7098435082055959542040105003657401994133_wp
        Alpha(5) = 4.0149225768154399888662520190507621009601_wp

        Xi(1) = 1.019217117302840040836235668831_wp
        Xi(2) = 1.178415662855246872275580272138_wp
        Xi(3) = 1.526280432408703061791278854376_wp
        Xi(4) = 2.125613075941608395673376730173_wp
        Xi(5) = 3.086751187750071800464449212065_wp
        Xi(6) = 4.593720184650979501857215314686_wp
        Xi(7) = 6.928231715123500258392275075181_wp
        Xi(8) = 10.429602304922355690899871127897_wp
        Xi(9) = 15.109104672966703414999090071547_wp
        Xi(10) = 19.306177258864197996898748321826_wp

      case (5)
        ! ----- K = 05 , R = 3E1 -----

        RIni = 3.0e1_wp

        Omega(1) = 0.0785202296725237321788569942737634477226_wp
        Omega(2) = 0.2352291127246095037714793130056989411969_wp
        Omega(3) = 0.5755235459038332647744261316713476617224_wp
        Omega(4) = 1.3603613273128701396854495220267722288554_wp
        Omega(5) = 3.3389033426820007048260730009303642873419_wp

        Alpha(1) = 0.0297577867799910187276724591112775364365_wp
        Alpha(2) = 0.1781464556286160730389639406112678443606_wp
        Alpha(3) = 0.5601138852938733217684888487930550127203_wp
        Alpha(4) = 1.4739028120126258845998951096945006611350_wp
        Alpha(5) = 3.6519366545145374282690942102647113642888_wp

        Xi(1) = 1.021836627027043814075306371603_wp
        Xi(2) = 1.203748836653624470203506613419_wp
        Xi(3) = 1.606670345753720860427701810913_wp
        Xi(4) = 2.315548608377475355837513126822_wp
        Xi(5) = 3.486487534924729528957795809951_wp
        Xi(6) = 5.397491666427624752450792344760_wp
        Xi(7) = 8.519390416720351725046733726288_wp
        Xi(8) = 13.555879139999029356182524974628_wp
        Xi(9) = 21.021496749683150869966730134308_wp
        Xi(10) = 28.634423521169729146343407855113_wp

      case (6)
        ! ----- K = 05 , R = 4E1 -----

        RIni = 4.0e1_wp

        Omega(1) = 0.0641814838050678704967598464659062074134_wp
        Omega(2) = 0.2015553177348803795505350136552102924270_wp
        Omega(3) = 0.5226910879403149711537254762383497563860_wp
        Omega(4) = 1.2901011954205592439665767345857716463797_wp
        Omega(5) = 3.2584115181337518739324021765924044302665_wp

        Alpha(1) = 0.0241901803500972818551558720864824714170_wp
        Alpha(2) = 0.1485733749263374468200897549063910219047_wp
        Alpha(3) = 0.4872759750389151346509908346860129313427_wp
        Alpha(4) = 1.3391490990318978887216316842057040048530_wp
        Alpha(5) = 3.4411575727997536413432788648947280307766_wp

        Xi(1) = 1.023618696885570287786852705647_wp
        Xi(2) = 1.221131920310683900633870468955_wp
        Xi(3) = 1.662645702391932947322369940046_wp
        Xi(4) = 2.450464176589166269372085937839_wp
        Xi(5) = 3.778112484856030209908611672276_wp
        Xi(6) = 6.004079808102726620484823616408_wp
        Xi(7) = 9.771896848110523801922422393318_wp
        Xi(8) = 16.155999486316149824777221866157_wp
        Xi(9) = 26.333922981430985232539909723926_wp
        Xi(10) = 37.796347893531772609704288967691_wp

      case (7)
        ! ----- K = 05 , R = 5E1 -----

        RIni = 5.0e1_wp

        Omega(1) = 0.0549289684719608573105023054117257430562_wp
        Omega(2) = 0.1797226084604662145184595864466636783163_wp
        Omega(3) = 0.4876092684357195764356749517753897293915_wp
        Omega(4) = 1.2421875649935035342474529129752625067340_wp
        Omega(5) = 3.2030368198904093346955634924455580403446_wp

        Alpha(1) = 0.0205985169364180585578296594009839459716_wp
        Alpha(2) = 0.1294650714429877070990940191030915684678_wp
        Alpha(3) = 0.4397972396705089706309306507581169398691_wp
        Alpha(4) = 1.2499372840249322914873975065574995824136_wp
        Alpha(5) = 3.2997722744581938507293233930894871264172_wp

        Xi(1) = 1.024945306274655407348384006383_wp
        Xi(2) = 1.234152929071583455950845975568_wp
        Xi(3) = 1.705016340573330133512271189478_wp
        Xi(4) = 2.554066360145842692896955838933_wp
        Xi(5) = 4.006449188957825125866757387527_wp
        Xi(6) = 6.490851924497973947456724852501_wp
        Xi(7) = 10.808236636812674757827934524101_wp
        Xi(8) = 18.395005103541284728618787625010_wp
        Xi(9) = 31.182470293965814951003823196629_wp
        Xi(10) = 46.806410907619687457315027501181_wp

      case (8)
        ! ----- K = 05 , R = 6E1 -----

        RIni = 6.0e1_wp

        Omega(1) = 0.0484092394476768007366320866607933837145_wp
        Omega(2) = 0.1642667164220850362451762857740611423196_wp
        Omega(3) = 0.4622317543222329255900738403939342902049_wp
        Omega(4) = 1.2068045801673655563132636636680672381772_wp
        Omega(5) = 3.1618887677979012988931789562840890539519_wp

        Alpha(1) = 0.0180677099136463318526386444967668509776_wp
        Alpha(2) = 0.1159843500861629782565289520368523312754_wp
        Alpha(3) = 0.4060170289500016420044040121872797044489_wp
        Alpha(4) = 1.1856093034676985442871580134216458191077_wp
        Alpha(5) = 3.1967392715623493253812581471606790728401_wp

        Xi(1) = 1.025987986599939304102439840971_wp
        Xi(2) = 1.244436379710849673897732348138_wp
        Xi(3) = 1.738749585862048788209056893184_wp
        Xi(4) = 2.637470542163933028435157668312_wp
        Xi(5) = 4.193080245286467884740372324970_wp
        Xi(6) = 6.896419867232555861707304911690_wp
        Xi(7) = 11.692614370719187257517679423557_wp
        Xi(8) = 20.366298558617970998821378181276_wp
        Xi(9) = 35.654447253787359656262978546692_wp
        Xi(10) = 55.674065914067402998516298495701_wp

      case (9)
        ! ----- K = 05 , R = 7E1 -----

        RIni = 7.0e1_wp

        Omega(1) = 0.0435404641423749599441785345516064964499_wp
        Omega(2) = 0.1526714077865754897296796457895062815169_wp
        Omega(3) = 0.4428218963958594054004542367453112206022_wp
        Omega(4) = 1.1792923812761709411681826376039339265844_wp
        Omega(5) = 3.1297433721720418944422376084091297343548_wp

        Alpha(1) = 0.0161774474636488094969117472812780711422_wp
        Alpha(2) = 0.1059055746618679526608545332644339964645_wp
        Alpha(3) = 0.3805585874490105481766921702302397534368_wp
        Alpha(4) = 1.1365624686784197416161701355008517566603_wp
        Alpha(5) = 3.1174919552876026438738288337759740898036_wp

        Xi(1) = 1.026838038680242269021919421235_wp
        Xi(2) = 1.252852622804190629098204656788_wp
        Xi(3) = 1.766537099233853070679757524264_wp
        Xi(4) = 2.706794351109581448465435382644_wp
        Xi(5) = 4.350130814145329771498033055721_wp
        Xi(6) = 7.243079059817089169429765727060_wp
        Xi(7) = 12.463487647364055717000286271912_wp
        Xi(8) = 22.129085741641334232621129984864_wp
        Xi(9) = 39.810779880025723560105221920935_wp
        Xi(10) = 64.406047341502177097649628478848_wp

      case (10)
        ! ----- K = 05 , R = 8E1 -----

        RIni = 8.0e1_wp

        Omega(1) = 0.0397514331536537414319993754413529174485_wp
        Omega(2) = 0.1436062765137879609363786778675198263500_wp
        Omega(3) = 0.4273835064255168622604236922102316498240_wp
        Omega(4) = 1.1571122334202288380652218213739956809150_wp
        Omega(5) = 3.1037318963686114029875556807880343512807_wp

        Alpha(1) = 0.0147059497531036244120876258813235271106_wp
        Alpha(2) = 0.0980528681678504911183773609661962211703_wp
        Alpha(3) = 0.3605733280656044851652641120010045483468_wp
        Alpha(4) = 1.0976668403036145611100307672991505114624_wp
        Alpha(5) = 3.0541811446566613800959050228911451085878_wp

        Xi(1) = 1.027549586211971455510481854301_wp
        Xi(2) = 1.259920380026032329847524016131_wp
        Xi(3) = 1.789997919671034310416801726973_wp
        Xi(4) = 2.765763801097122743686759771542_wp
        Xi(5) = 4.485113281421210089296852663310_wp
        Xi(6) = 7.544956076967540107257315096234_wp
        Xi(7) = 13.145950170095980844529870434911_wp
        Xi(8) = 23.723757443087620236030210740807_wp
        Xi(9) = 43.696432384087426959579092411445_wp
        Xi(10) = 73.007490012850690647150386780595_wp

      case (11)
        ! ----- K = 05 , R = 9E1 -----

        RIni = 9.0e1_wp

        Omega(1) = 0.0367103115933988785408103542456936807525_wp
        Omega(2) = 0.1362975085252602825992016347123048092271_wp
        Omega(3) = 0.4147426140491629501059079460079104251236_wp
        Omega(4) = 1.1387451937085927495315468283010318373272_wp
        Omega(5) = 3.0821271502266368218312525550928171469423_wp

        Alpha(1) = 0.0135244470648898468016302571240161434218_wp
        Alpha(2) = 0.0917428214601229520200201608681744858131_wp
        Alpha(3) = 0.3444004900958717987143227479140961122539_wp
        Alpha(4) = 1.0659073186356656392668670885193193953455_wp
        Alpha(5) = 3.0021576019667892893435373613542083148786_wp

        Xi(1) = 1.028157239821055244765463099199_wp
        Xi(2) = 1.265972618547575588121394118968_wp
        Xi(3) = 1.810179597639107708308310595857_wp
        Xi(4) = 2.816815239727872043030629245486_wp
        Xi(5) = 4.603010284514818944989755777897_wp
        Xi(6) = 7.811600560233624539370478334632_wp
        Xi(7) = 13.757390685449223435496413081580_wp
        Xi(8) = 25.179356479133221431362943398824_wp
        Xi(9) = 47.345971610642393925111326424826_wp
        Xi(10) = 81.482534713832055844884294515396_wp

      case (12)
        ! ----- K = 05 , R = 1E2 -----

        RIni = 1.0e2_wp

        Omega(1) = 0.0342103034874631526696152957617291789916_wp
        Omega(2) = 0.1302622960820672937630720217283553097332_wp
        Omega(3) = 0.4041584156458790294866832704112624696791_wp
        Omega(4) = 1.1232181675451095866731174299779638658947_wp
        Omega(5) = 3.0638169728620764668738235547706949546409_wp

        Alpha(1) = 0.0125527275404679892651334575069677068626_wp
        Alpha(2) = 0.0865493948772892533749523380492973956279_wp
        Alpha(3) = 0.3310012429371387315162403308654859301896_wp
        Alpha(4) = 1.0393832416814095931769171277103680495202_wp
        Alpha(5) = 2.9584699518773349188188737102223058172967_wp

        Xi(1) = 1.028684362432210875628289803441_wp
        Xi(2) = 1.271235368477093939892745566578_wp
        Xi(3) = 1.827797553096407015267403639580_wp
        Xi(4) = 2.861628476901701719934886791741_wp
        Xi(5) = 4.707300182002275256310658280157_wp
        Xi(6) = 8.049789021787946660321799718929_wp
        Xi(7) = 14.310414647355005401084537819401_wp
        Xi(8) = 26.517614005906436927539893133599_wp
        Xi(9) = 50.786829027389754267346466676258_wp
        Xi(10) = 89.834677621284348428276622655630_wp

      case (13)
        ! ----- K = 05 , R = 2E2 -----

        RIni = 2.0e2_wp

        Omega(1) = 0.0219924131992907643790133211808557334166_wp
        Omega(2) = 0.1002064224819224335166702351263001702364_wp
        Omega(3) = 0.3489637351854245363700929988270971193742_wp
        Omega(4) = 1.0398862719837947781158921101152259325318_wp
        Omega(5) = 2.9648211490348502911412048588246648250788_wp

        Alpha(1) = 0.0077919805414365443251355311960609784094_wp
        Alpha(2) = 0.0610302875027291444151751869523492288749_wp
        Alpha(3) = 0.2635451761362904776547170376810313996430_wp
        Alpha(4) = 0.9023059551184773100754483998731103611135_wp
        Alpha(5) = 2.7287535886135676362583557530427924575633_wp

        Xi(1) = 1.031738582081012241178225030680_wp
        Xi(2) = 1.301959934154840365511107691532_wp
        Xi(3) = 1.931949578030785384056416953413_wp
        Xi(4) = 3.131318808130420078359584801220_wp
        Xi(5) = 5.350957513524513323426717859022_wp
        Xi(6) = 9.568335189349001663680627238762_wp
        Xi(7) = 17.987957016817579344869493240822_wp
        Xi(8) = 35.937509974005508950334331075283_wp
        Xi(9) = 77.269299127612637412276797022059_wp
        Xi(10) = 167.164290422268234273528442201950_wp

      case (14)
        ! ----- K = 05 , R = 3E2 -----

        RIni = 3.0e2_wp

        Omega(1) = 0.0174282312374317400131801909443640674624_wp
        Omega(2) = 0.0885516446827289778186164705131577790098_wp
        Omega(3) = 0.3261206213119612461344941356511384356054_wp
        Omega(4) = 1.0041414563483471091443616218796819339332_wp
        Omega(5) = 2.9219747649465219435215707077801994273614_wp

        Alpha(1) = 0.0060033834494476276132450510373411711740_wp
        Alpha(2) = 0.0513810282138952673963618999336544490575_wp
        Alpha(3) = 0.2369999919944957109688371890698910249284_wp
        Alpha(4) = 0.8463234388251620139622527494438486428407_wp
        Alpha(5) = 2.6327559231103213198584012122438480218989_wp

        Xi(1) = 1.033174749276737282299709619338_wp
        Xi(2) = 1.316546435265119147366326413096_wp
        Xi(3) = 1.982179306549416015082593078844_wp
        Xi(4) = 3.264351293528295230471011900697_wp
        Xi(5) = 5.678771289831076665276937687210_wp
        Xi(6) = 10.374156421715174907459411368649_wp
        Xi(7) = 20.046686141095781039794809785803_wp
        Xi(8) = 41.607779583810231966573001116672_wp
        Xi(9) = 95.084831518769201169627436343035_wp
        Xi(10) = 234.568627387691309854611532159652_wp

      case (15)
        ! ----- K = 05 , R = 4E2 -----

        RIni = 4.0e2_wp

        Omega(1) = 0.0150187614602143699419223993318794896368_wp
        Omega(2) = 0.0822228451037669558342933957828524427214_wp
        Omega(3) = 0.3132830923567541781904564196503670814309_wp
        Omega(4) = 0.9837032818773332431801573183527409582894_wp
        Omega(5) = 2.8973695770360148279195383613782155407534_wp

        Alpha(1) = 0.0050550447094746876118002336473328250932_wp
        Alpha(2) = 0.0462342261395064916236290718243306940849_wp
        Alpha(3) = 0.2224845241613079064560594411137373072052_wp
        Alpha(4) = 0.8150922595119977967221928938368336048370_wp
        Alpha(5) = 2.5785670834577289001546263058628483122448_wp

        Xi(1) = 1.034032920030843131142458957150_wp
        Xi(2) = 1.325306119866962213421808514102_wp
        Xi(3) = 2.012588769819031008683277161886_wp
        Xi(4) = 3.345847426858509376110598076970_wp
        Xi(5) = 5.883019761153540387252358634385_wp
        Xi(6) = 10.887461813180383892254998734472_wp
        Xi(7) = 21.397302457233739180278431213367_wp
        Xi(8) = 45.485650878612636598824492395465_wp
        Xi(9) = 108.077596856357892887712246476895_wp
        Xi(10) = 293.377329734328583604741069734700_wp

      case (16)
        ! ----- K = 05 , R = 5E2 -----

        RIni = 5.0e2_wp

        Omega(1) = 0.0135371869189180438887112238946497377867_wp
        Omega(2) = 0.0782451713630320120227627615772103553127_wp
        Omega(3) = 0.3050398442664156748805118213319076403423_wp
        Omega(4) = 0.9704412982176837114303528941672283281150_wp
        Omega(5) = 2.8813614470814399848557635364088014284789_wp

        Alpha(1) = 0.0044700910206532437740403729785343411862_wp
        Alpha(2) = 0.0430422335189548189549968524453271712105_wp
        Alpha(3) = 0.2133256975578133257876378356399094116114_wp
        Alpha(4) = 0.7951322401352172105932353263213485661254_wp
        Alpha(5) = 2.5436793036816291951971585882930071420560_wp

        Xi(1) = 1.034604842237904686209210014525_wp
        Xi(2) = 1.331161827039902273081550809763_wp
        Xi(3) = 2.033020735682391787017830875328_wp
        Xi(4) = 3.401014082257426986632617427198_wp
        Xi(5) = 6.022775779578891303719240246650_wp
        Xi(6) = 11.243704457368707444117617022528_wp
        Xi(7) = 22.352756423276303403280262926955_wp
        Xi(8) = 48.306073236663569628324133020669_wp
        Xi(9) = 117.945145859368349001983222734680_wp
        Xi(10) = 344.399657536421490683364865503790_wp

      case (17)
        ! ----- K = 05 , R = 6E2 -----

        RIni = 6.0e2_wp

        Omega(1) = 0.0125497022369018914496793782285821716727_wp
        Omega(2) = 0.0755480277998204573136121521559349645258_wp
        Omega(3) = 0.2993684653512476888869656838076949156857_wp
        Omega(4) = 0.9612526221486810956294750962758399737140_wp
        Omega(5) = 2.8702502162736308354874792403421679409803_wp

        Alpha(1) = 0.0040793511611987108165482531028228629921_wp
        Alpha(2) = 0.0408997178514718597193682708690332106016_wp
        Alpha(3) = 0.2071003853755610939370431985984133405054_wp
        Alpha(4) = 0.7814442554474885124478691911242833612050_wp
        Alpha(5) = 2.5196339666338295893831950023056265308696_wp

        Xi(1) = 1.035008238780495264021869195048_wp
        Xi(2) = 1.335300992017198816116940740528_wp
        Xi(3) = 2.047513221463767489919846620516_wp
        Xi(4) = 3.440345054969623298608480399530_wp
        Xi(5) = 6.123156777924488606591746364671_wp
        Xi(6) = 11.502110566433446770147341897683_wp
        Xi(7) = 23.055158337444412440866980951171_wp
        Xi(8) = 50.420991181853802340362413758612_wp
        Xi(9) = 125.578805960313507290859735832100_wp
        Xi(10) = 388.108156802750548258762464115534_wp

      case (18)
        ! ----- K = 05 , R = 7E2 -----

        RIni = 7.0e2_wp

        Omega(1) = 0.0118636637624195178692006451964813251720_wp
        Omega(2) = 0.0736487157459496412766292508522703030849_wp
        Omega(3) = 0.2953333596223997398124046231959738406658_wp
        Omega(4) = 0.9546824601007417951544463396995610082740_wp
        Omega(5) = 2.8622952932595032279547431208754915132886_wp

        Alpha(1) = 0.0038074691411251674333998201932641181067_wp
        Alpha(2) = 0.0394027284673490241780448913727630255721_wp
        Alpha(3) = 0.2027096957655570047296704136452483169251_wp
        Alpha(4) = 0.7717281393047814657369654067764486171654_wp
        Alpha(5) = 2.5025046733436876863956171979097575786000_wp

        Xi(1) = 1.035300325762522878733813691543_wp
        Xi(2) = 1.338302645324408399912265166254_wp
        Xi(3) = 2.058049151069528705498140608299_wp
        Xi(4) = 3.469043393629417977889681234949_wp
        Xi(5) = 6.196792486997452185926160694862_wp
        Xi(6) = 11.693019319216821878121281841345_wp
        Xi(7) = 23.579165309594961322942063830510_wp
        Xi(8) = 52.021973375747542937125311723889_wp
        Xi(9) = 131.493131827179792911475253447406_wp
        Xi(10) = 424.753380875778627501215822803715_wp

      case (19)
        ! ----- K = 05 , R = 8E2 -----

        RIni = 8.0e2_wp

        Omega(1) = 0.0113813497171839778759997200904829917079_wp
        Omega(2) = 0.0722993078519351827732693401007413314119_wp
        Omega(3) = 0.2924451938945003215083486003411650244743_wp
        Omega(4) = 0.9499630028765620042311909609900055784237_wp
        Omega(5) = 2.8565759204045159773410755832401264342479_wp

        Alpha(1) = 0.0036161175535254347276672282204476593925_wp
        Alpha(2) = 0.0383455049806827012912025746255162417242_wp
        Alpha(3) = 0.1995869766812679076551772378023130727343_wp
        Alpha(4) = 0.7647855301606951202580964022104126343038_wp
        Alpha(5) = 2.4902332388884395458772402776048693340272_wp

        Xi(1) = 1.035512044375376673852225117844_wp
        Xi(2) = 1.340480804514106308159872293917_wp
        Xi(3) = 2.065708029999885967905107198028_wp
        Xi(4) = 3.489961455007227071091480130605_wp
        Xi(5) = 6.250673885791233096772406829800_wp
        Xi(6) = 11.833441977084792261394607937319_wp
        Xi(7) = 23.967358721928112862131765581353_wp
        Xi(8) = 53.220943640742990547731761097339_wp
        Xi(9) = 135.999940734977351364243247644481_wp
        Xi(10) = 454.425212167193654899666199753483_wp

      case (20)
        ! ----- K = 05 , R = 9E2 -----

        RIni = 9.0e2_wp

        Omega(1) = 0.0110493996986944845713091178408815640211_wp
        Omega(2) = 0.0713632054457008080852590571618865311621_wp
        Omega(3) = 0.2904310341712171869663506457159840579152_wp
        Omega(4) = 0.9466633544503644491496796575802363804542_wp
        Omega(5) = 2.8525745409845265157262189648079697690264_wp

        Alpha(1) = 0.0034843237838238312649514669648576070671_wp
        Alpha(2) = 0.0376153534981393007195370117162758205609_wp
        Alpha(3) = 0.1974191937931761697681051076491698381687_wp
        Alpha(4) = 0.7599497569265250643288402143671333988095_wp
        Alpha(5) = 2.4816698505783679781604156300289787395741_wp

        Xi(1) = 1.035661029320602322481049972769_wp
        Xi(2) = 1.342014708503715602716348243018_wp
        Xi(3) = 2.071108717563173735399031927251_wp
        Xi(4) = 3.504739759714249971306321285169_wp
        Xi(5) = 6.288846628111508652372863004132_wp
        Xi(6) = 11.933300415226327029563635484521_wp
        Xi(7) = 24.244845393727907296652079871535_wp
        Xi(8) = 54.084778335166424953484032300821_wp
        Xi(9) = 139.288693645026466591541058903658_wp
        Xi(10) = 477.083147859152222031164924942459_wp

      case (21)
        ! ----- K = 05 , R = 1E3 -----

        RIni = 1.0e3_wp

        Omega(1) = 0.0108382726570232800350219570101328869782_wp
        Omega(2) = 0.0707644978086732066545881149333130366585_wp
        Omega(3) = 0.2891382091047487843294035159136257107093_wp
        Omega(4) = 0.9445417262916081805245935298920301192993_wp
        Omega(5) = 2.8500005633404065035450825060792112708441_wp

        Alpha(1) = 0.0034004600489073895373085156109901894261_wp
        Alpha(2) = 0.0371498167966149987156249893255344218801_wp
        Alpha(3) = 0.1960321090378673603580109152666466343362_wp
        Alpha(4) = 0.7568483876518314330643366627526802403736_wp
        Alpha(5) = 2.4761708329681671751556354399426140844298_wp

        Xi(1) = 1.035757223385584084281510475201_wp
        Xi(2) = 1.343005822739786468667487484563_wp
        Xi(3) = 2.074601499880166638077258478567_wp
        Xi(4) = 3.514309494963886897024238353815_wp
        Xi(5) = 6.313612662509091697123075892506_wp
        Xi(6) = 11.998250870322400191830825022521_wp
        Xi(7) = 24.425972041287355445912665707908_wp
        Xi(8) = 54.651711438412047962265294387407_wp
        Xi(9) = 141.466224071874645137891768342797_wp
        Xi(10) = 492.570529270241514563366536094691_wp

      case (22:31)
        ! ----- K = 05 , R = 2E3 (Best : R = 1154) -----

        RIni = 2.0e3_wp

        Omega(1) = 0.0107186010624974955309879299722441992060_wp
        Omega(2) = 0.0704239357038686291254884867807373183268_wp
        Omega(3) = 0.2884012053308492404702506606284373447124_wp
        Omega(4) = 0.9433310104981166196980689464535174693083_wp
        Omega(5) = 2.8485313369244249009568092922961568547180_wp

        Alpha(1) = 0.0033529112764591385087846226072116895267_wp
        Alpha(2) = 0.0368855238086302728956974734259022241645_wp
        Alpha(3) = 0.1952428895765193203392729240364111831241_wp
        Alpha(4) = 0.7550812995393414493814493448908820028009_wp
        Alpha(5) = 2.4730352459564652644382481971874199189187_wp

        Xi(1) = 1.035812293717382617536798550262_wp
        Xi(2) = 1.343573108837654108403160402307_wp
        Xi(3) = 2.076601508202342995451986085165_wp
        Xi(4) = 3.519794398851761027102041934356_wp
        Xi(5) = 6.327824256171820903934394086932_wp
        Xi(6) = 12.035582649906387529316342810404_wp
        Xi(7) = 24.530303291186305682247681403396_wp
        Xi(8) = 54.979390713200053542736300826732_wp
        Xi(9) = 142.731756929721936510979674039845_wp
        Xi(10) = 501.753271542624243922725568722853_wp

    end select

  case (6)
    ! ===== K = 06 =====

    select case (InitR)

      case default !(1)
        ! ----- K = 06 , R = 2E0 -----

        RIni = 2.0_wp

        Omega(1) = 0.3951503586666182849006841854011540249303_wp
        Omega(2) = 0.9483584803953558999261436168737304797105_wp
        Omega(3) = 1.5813287669193051950466619159918479908811_wp
        Omega(4) = 2.3797403876283135313977890046288621306303_wp
        Omega(5) = 3.5481339355732610745496058424208740689210_wp
        Omega(6) = 5.8817260741536767109684247234469012255431_wp

        Alpha(1) = 0.1534423273269616295463391511533224331743_wp
        Alpha(2) = 0.8208030144142543005673050282577918324023_wp
        Alpha(3) = 2.0763266377595740032162285881867092030006_wp
        Alpha(4) = 4.0375449448252483082913433243277268047677_wp
        Alpha(5) = 6.9549672561485968930519718789184935303638_wp
        Alpha(6) = 11.4891014499758782963292902401519768318394_wp

        Xi(1) = 1.002939863002787006869727381986_wp
        Xi(2) = 1.026434552149436049629967204755_wp
        Xi(3) = 1.073260845596178922312612102896_wp
        Xi(4) = 1.142927410284706687547580383768_wp
        Xi(5) = 1.234275411331628588876806185137_wp
        Xi(6) = 1.344957165532231756774686481926_wp
        Xi(7) = 1.470755151352145535001252063534_wp
        Xi(8) = 1.604854336886718575651451790254_wp
        Xi(9) = 1.737351219132925049439680753594_wp
        Xi(10) = 1.855486880309094952531782929395_wp
        Xi(11) = 1.945142207156052932765302743157_wp
        Xi(12) = 1.993735287513832045501287193456_wp

      case (2)
        ! ----- K = 06 , R = 5E0 -----

        RIni = 5.0_wp

        Omega(1) = 0.2246903767486849122763520325563035839878_wp
        Omega(2) = 0.5519095844142891326106030447373740344119_wp
        Omega(3) = 0.9622019394266135803054848518733876971964_wp
        Omega(4) = 1.5504627855961601076272698973568253677513_wp
        Omega(5) = 2.5290208569930264208974746154012791521382_wp
        Omega(6) = 4.6466141585747064604410005994594712319667_wp

        Alpha(1) = 0.0870216340647555737652257303038627611613_wp
        Alpha(2) = 0.4709136752047922974001321200221426011012_wp
        Alpha(3) = 1.2180676458000685170304441817812346471328_wp
        Alpha(4) = 2.4535517473926842261719438731404352438403_wp
        Alpha(5) = 4.4452949405703532036703007346289950874052_wp
        Alpha(6) = 7.8614295814483422927473887931171248055762_wp

        Xi(1) = 1.007037883811018135664719364009_wp
        Xi(2) = 1.063908019632764205227894172001_wp
        Xi(3) = 1.180628980475084192761948964634_wp
        Xi(4) = 1.362952739862027088735621505045_wp
        Xi(5) = 1.618921217783021904005648361480_wp
        Xi(6) = 1.957758248114852028462506972062_wp
        Xi(7) = 2.387184641951110330646002033461_wp
        Xi(8) = 2.907461913380709053050798829076_wp
        Xi(9) = 3.500166704810058573979145157828_wp
        Xi(10) = 4.111882968405614833130173879638_wp
        Xi(11) = 4.641862576083718421298546230602_wp
        Xi(12) = 4.957655499604545906288527445938_wp

      case (3)
        ! ----- K = 06 , R = 1E1 -----

        RIni = 1.0e1_wp

        Omega(1) = 0.1407385838051433856106010866771072187476_wp
        Omega(2) = 0.3565694200685690354226536902260136230325_wp
        Omega(3) = 0.6603802616738012804843693803746873527416_wp
        Omega(4) = 1.1604682915072341640571468301956770119432_wp
        Omega(5) = 2.0803301182531367932628668659944537466799_wp
        Omega(6) = 4.1513349861478130481580450794609760123421_wp

        Alpha(1) = 0.0543212598500059367314274534621620915686_wp
        Alpha(2) = 0.2985187647358117048385951763922463442213_wp
        Alpha(3) = 0.7962225930216238800108709272596030359637_wp
        Alpha(4) = 1.6836229524774230601608104862521031463984_wp
        Alpha(5) = 3.2535534765211933906029623253530758120178_wp
        Alpha(6) = 6.1993107601849360499893237808066714933375_wp

        Xi(1) = 1.010390027445351043833277060457_wp
        Xi(2) = 1.094952075027758760716531383217_wp
        Xi(3) = 1.271782267388700225991052250585_wp
        Xi(4) = 1.556518752320054307936025850267_wp
        Xi(5) = 1.973568404102859693400283391806_wp
        Xi(6) = 2.557355324903913973899269307388_wp
        Xi(7) = 3.352030882203745386273968653867_wp
        Xi(8) = 4.404595560535857340627580169823_wp
        Xi(9) = 5.740477777935007439296527520511_wp
        Xi(10) = 7.301940614229004384433396879217_wp
        Xi(11) = 8.840257599451475464640004542360_wp
        Xi(12) = 9.857134216221037015942529890822_wp

      case (4)
        ! ----- K = 06 , R = 2E1 -----

        RIni = 2.0e1_wp

        Omega(1) = 0.0862496741180035379067893470950156853405_wp
        Omega(2) = 0.2287552313838759988283118665397530833161_wp
        Omega(3) = 0.4619231653435139031303388601479298358754_wp
        Omega(4) = 0.9029137915898263100529519409320755585213_wp
        Omega(5) = 1.7807264424526320117014507871111561598809_wp
        Omega(6) = 3.8200570280772443360669393674911020752916_wp

        Alpha(1) = 0.0331253840720488536691890318203190801682_wp
        Alpha(2) = 0.1862453189927838743364689119696997465780_wp
        Alpha(3) = 0.5202262124793207121137110504616174466719_wp
        Alpha(4) = 1.1783399761432671823966136792627423801605_wp
        Alpha(5) = 2.4682032832613331014767754645333752705483_wp
        Alpha(6) = 5.0989823083682598225604976072133922571084_wp

        Xi(1) = 1.013856805828272228940135912811_wp
        Xi(2) = 1.127427449637189030957028224034_wp
        Xi(3) = 1.369180650217897420007316655610_wp
        Xi(4) = 1.769539581176343221585733900270_wp
        Xi(5) = 2.378972498614929419104985219846_wp
        Xi(6) = 3.276246661733991947206001693615_wp
        Xi(7) = 4.578788142114554839817719411599_wp
        Xi(8) = 6.449052322730253097859964839245_wp
        Xi(9) = 9.076047734421376751712262631600_wp
        Xi(10) = 12.560202401726437062418062495794_wp
        Xi(11) = 16.536631228932167728476465562437_wp
        Xi(12) = 19.547499375541998470340310234405_wp

      case (5)
        ! ----- K = 06 , R = 3E1 -----

        RIni = 3.0e1_wp

        Omega(1) = 0.0643875392260774473662246139560316748884_wp
        Omega(2) = 0.1769925533203732905728139748746663428847_wp
        Omega(3) = 0.3803989314000021870147000280226734503231_wp
        Omega(4) = 0.7936758834219182629486112612315906744698_wp
        Omega(5) = 1.6486087888192316119276986663422235324106_wp
        Omega(6) = 3.6716777532566438266292685455027822172269_wp

        Alpha(1) = 0.0246338601868061877442589273357054224789_wp
        Alpha(2) = 0.1410255763456897599880624991075528384954_wp
        Alpha(3) = 0.4082545329312478092306272492439944699072_wp
        Alpha(4) = 0.9707499275233410204527170761501508877700_wp
        Alpha(5) = 2.1393343031375326331305214422329186163552_wp
        Alpha(6) = 4.6296955440744570115831069134770814343938_wp

        Xi(1) = 1.015871521114546384184813831997_wp
        Xi(2) = 1.146484930769694886125431576573_wp
        Xi(3) = 1.427344744422456389006462207369_wp
        Xi(4) = 1.899818296204869565537744213746_wp
        Xi(5) = 2.634570814368591317594647804867_wp
        Xi(6) = 3.746933318713922312064718012081_wp
        Xi(7) = 5.419224037294607949043212524742_wp
        Xi(8) = 7.927168045245041083472770093721_wp
        Xi(9) = 11.649874700288855315123692779622_wp
        Xi(10) = 16.955935530593848448166482967281_wp
        Xi(11) = 23.594132485605730422534298540427_wp
        Xi(12) = 29.124355626658757639074748269081_wp

      case (6)
        ! ----- K = 06 , R = 4E1 -----

        RIni = 4.0e1_wp

        Omega(1) = 0.0522574742342978113803029626555085584982_wp
        Omega(2) = 0.1480796993930270559432010521461187124714_wp
        Omega(3) = 0.3341395292959524739658146164078544870790_wp
        Omega(4) = 0.7296921875931181159023568294497152919575_wp
        Omega(5) = 1.5689274489411984757044130067704656994465_wp
        Omega(6) = 3.5813517206284184537547210513963591438369_wp

        Alpha(1) = 0.0199272921588992815234223088260989698028_wp
        Alpha(2) = 0.1158687609653954539857666911073508231311_wp
        Alpha(3) = 0.3455473250255029791578372660998397236654_wp
        Alpha(4) = 0.8529186964399603203962341169130922935437_wp
        Alpha(5) = 1.9493007795284760018481723387040460693242_wp
        Alpha(6) = 4.3543982689127892935984642797109245293541_wp

        Xi(1) = 1.017270403178994494716724794614_wp
        Xi(2) = 1.159800883046806106312261175351_wp
        Xi(3) = 1.468442034643479504585870942801_wp
        Xi(4) = 1.993273757141346267458999197419_wp
        Xi(5) = 2.821531744056516985289309595863_wp
        Xi(6) = 4.099651161191333565564365315481_wp
        Xi(7) = 6.067370770734152255442006573816_wp
        Xi(8) = 9.106291550278170173794711761417_wp
        Xi(9) = 13.788581281739392109426944887929_wp
        Xi(10) = 20.801839405309055643783500144650_wp
        Xi(11) = 30.180838554184981833983236754193_wp
        Xi(12) = 38.605808888195780537860857251076_wp

      case (7)
        ! ----- K = 06 , R = 5E1 -----

        RIni = 5.0e1_wp

        Omega(1) = 0.0444334182386757780728681402337976180661_wp
        Omega(2) = 0.1293308348039938405240863519840566198127_wp
        Omega(3) = 0.3036711754764852394777429472894070272559_wp
        Omega(4) = 0.6863985328085999401193963154188537600930_wp
        Omega(5) = 1.5138503087235758502553106819021877527121_wp
        Omega(6) = 3.5185424210031163373864349086517222531256_wp

        Alpha(1) = 0.0168938375522182825927815447500379697487_wp
        Alpha(2) = 0.0996091777682384279030331809823284316963_wp
        Alpha(3) = 0.3047652371799653853450158547122583740929_wp
        Alpha(4) = 0.7753183769485387444244002541626770153016_wp
        Alpha(5) = 1.8222505520282237026757216069583478201821_wp
        Alpha(6) = 4.1681458423137244346731722366428130044369_wp

        Xi(1) = 1.018329136868172673938484629197_wp
        Xi(2) = 1.169925964521422761719336713693_wp
        Xi(3) = 1.499946922777239079064073856706_wp
        Xi(4) = 2.065711648691398642004729557797_wp
        Xi(5) = 2.968537148823861617896041620313_wp
        Xi(6) = 4.381973458167951779307358606275_wp
        Xi(7) = 6.597215529572176407994976710114_wp
        Xi(8) = 10.094351252177452381818267834745_wp
        Xi(9) = 15.634955098062942901657179639585_wp
        Xi(10) = 24.250794196703078522969754260430_wp
        Xi(11) = 36.390287374282789891605238707939_wp
        Xi(12) = 48.002121522630162350070426313664_wp

      case (8)
        ! ----- K = 06 , R = 6E1 -----

        RIni = 6.0e1_wp

        Omega(1) = 0.0389207539214614393984598920278594746947_wp
        Omega(2) = 0.1160606055099792352353874243242604080706_wp
        Omega(3) = 0.2817827958041292704650070122252003557151_wp
        Omega(4) = 0.6545864705958214315481384637518402769274_wp
        Omega(5) = 1.4727168297337244618882418079053309156734_wp
        Omega(6) = 3.4714354715327135230719390879272623351426_wp

        Alpha(1) = 0.0147577944494537626858251828237955649570_wp
        Alpha(2) = 0.0881339680501746974836184238266678647733_wp
        Alpha(3) = 0.2758135968716968830646233540315748200555_wp
        Alpha(4) = 0.7195968816061252057566730022664103216812_wp
        Alpha(5) = 1.7298527418488277009002082595934268738347_wp
        Alpha(6) = 4.0313890355775024482327950625659696015646_wp

        Xi(1) = 1.019173107753838788806267923182_wp
        Xi(2) = 1.178026728564142044789787855397_wp
        Xi(3) = 1.525313988635216473003890225169_wp
        Xi(4) = 2.124542779770625053510635371623_wp
        Xi(5) = 3.089282425702473901208897077986_wp
        Xi(6) = 4.617141829792998202527248841065_wp
        Xi(7) = 7.045958766593616910255820640074_wp
        Xi(8) = 10.947651740140429289967205495060_wp
        Xi(9) = 17.267385931674823157724874000962_wp
        Xi(10) = 27.393277391962152302745669985029_wp
        Xi(11) = 42.283685047310163161726759284420_wp
        Xi(12) = 57.320130973726621487901455864744_wp

      case (9)
        ! ----- K = 06 , R = 7E1 -----

        RIni = 7.0e1_wp

        Omega(1) = 0.0348032175661934841130653456309218540810_wp
        Omega(2) = 0.1061085857260316455183803488471205866972_wp
        Omega(3) = 0.2651359722976122245839594032679187307622_wp
        Omega(4) = 0.6299268945664218586569932378793623684032_wp
        Omega(5) = 1.4404181621079914719625938990255065164092_wp
        Omega(6) = 3.4343276793451510164097845168029721207859_wp

        Alpha(1) = 0.0131630672750795466269084499355557227318_wp
        Alpha(2) = 0.0795507056312876082931700126577023723939_wp
        Alpha(3) = 0.2540370048152601382225254655899959743692_wp
        Alpha(4) = 0.6772494438031879543122421027945279092819_wp
        Alpha(5) = 1.6588625672007166433853317388802395271341_wp
        Alpha(6) = 3.9254796374566991239932894908193361516169_wp

        Xi(1) = 1.019869801124880972402822476575_wp
        Xi(2) = 1.184733879548496024566514583931_wp
        Xi(3) = 1.546426224341374471861997041966_wp
        Xi(4) = 2.173852206399786536139898118769_wp
        Xi(5) = 3.191425632925343441491025653356_wp
        Xi(6) = 4.818387366773663990724557715239_wp
        Xi(7) = 7.435254134314602479909755983201_wp
        Xi(8) = 11.699895029075140608261496044662_wp
        Xi(9) = 18.734606364728364477634237061920_wp
        Xi(10) = 30.288827417441373105952417965625_wp
        Xi(11) = 47.904785539258988878719147663787_wp
        Xi(12) = 66.564805255975383981026372026690_wp

      case (10)
        ! ----- K = 06 , R = 8E1 -----

        RIni = 8.0e1_wp

        Omega(1) = 0.0315973344518909737188756565312575119719_wp
        Omega(2) = 0.0983312504597541692870528440284516591419_wp
        Omega(3) = 0.2519549389039057720135559442775630145661_wp
        Omega(4) = 0.6100807113831754208426068342152603918294_wp
        Omega(5) = 1.4141485422756007230568955912453077417013_wp
        Omega(6) = 3.4040697463532191200091120419912726902112_wp

        Alpha(1) = 0.0119218682970624811615680807216910430668_wp
        Alpha(2) = 0.0728594690057079250061376415881220225401_wp
        Alpha(3) = 0.2369701576901333403899060567332490734316_wp
        Alpha(4) = 0.6437479496493271354895342184487816439287_wp
        Alpha(5) = 1.6021676951562195459748821901868609529629_wp
        Alpha(6) = 3.8403261825125701504728137347655092526111_wp

        Xi(1) = 1.020459608471411983455362504358_wp
        Xi(2) = 1.190426416844526231217198108947_wp
        Xi(3) = 1.564422761724479715196578499814_wp
        Xi(4) = 2.216133937261998612189631274738_wp
        Xi(5) = 3.279697509222747613946355449421_wp
        Xi(6) = 4.994006773068674855022702763208_wp
        Xi(7) = 7.778930158075274756569045253585_wp
        Xi(8) = 12.373095970732255066346083616224_wp
        Xi(9) = 20.069427291837646079586399139316_wp
        Xi(10) = 32.979406739200383538029859309404_wp
        Xi(11) = 53.286665854063588545347363023552_wp
        Xi(12) = 75.739948898847376648757467876294_wp

      case (11)
        ! ----- K = 06 , R = 9E1 -----

        RIni = 9.0e1_wp

        Omega(1) = 0.0290224356155021399123123456570416678346_wp
        Omega(2) = 0.0920629962514437429516337904611411602218_wp
        Omega(3) = 0.2412003862634107494074828595409520914927_wp
        Omega(4) = 0.5936581793011396837047265184761357659227_wp
        Omega(5) = 1.3922177274028809855705227094269105236890_wp
        Omega(6) = 3.3787565044853692037326448049938676376769_wp

        Alpha(1) = 0.0109252384908035184792304329338963331253_wp
        Alpha(2) = 0.0674791066323320984753487798701865330031_wp
        Alpha(3) = 0.2231770678139233854745358820448331016451_wp
        Alpha(4) = 0.6164401736272702965815928344284202466952_wp
        Alpha(5) = 1.5555685911402398983321981873473305313382_wp
        Alpha(6) = 3.7699289128884461914145015404287164528796_wp

        Xi(1) = 1.020968577308356571250493038949_wp
        Xi(2) = 1.195349326545488175899470406627_wp
        Xi(3) = 1.580044347500419270107982483431_wp
        Xi(4) = 2.253022698082180482999706261360_wp
        Xi(5) = 3.357230602733887736771958842930_wp
        Xi(6) = 5.149563677934596625258151236793_wp
        Xi(7) = 8.086405713116187474881479957389_wp
        Xi(8) = 12.982536846305238532536807660733_wp
        Xi(9) = 21.295206267189717373386481646946_wp
        Xi(10) = 35.496053516596015268286246069351_wp
        Xi(11) = 58.455305927451702747293937889594_wp
        Xi(12) = 84.848581717880271528542390768735_wp

      case (12)
        ! ----- K = 06 , R = 1E2 -----

        RIni = 1.0e2_wp

        Omega(1) = 0.0269037503472777864172587897889643571148_wp
        Omega(2) = 0.0868884034646989601314973940271446650740_wp
        Omega(3) = 0.2322198468998249781331455129929075553719_wp
        Omega(4) = 0.5797744666263794617393732833132702353396_wp
        Omega(5) = 1.3735366550622797052541579354567602422321_wp
        Omega(6) = 3.3571561544965228026781572934211794745352_wp

        Alpha(1) = 0.0101053613446297862534198233434856728508_wp
        Alpha(2) = 0.0630473793038213744022494298968606329936_wp
        Alpha(3) = 0.2117607008763383316597879446607777254030_wp
        Alpha(4) = 0.5936601207247772421017086019734421142857_wp
        Alpha(5) = 1.5164074607971299795501282647869345510117_wp
        Alpha(6) = 3.7104671222364464652429538737976599804824_wp

        Xi(1) = 1.021414424800761986929578672978_wp
        Xi(2) = 1.199669935580648808329251586713_wp
        Xi(3) = 1.593799340204969835197747196620_wp
        Xi(4) = 2.285647994232349912218857368451_wp
        Xi(5) = 3.426207707105316589745250266397_wp
        Xi(6) = 5.288978769298995855335093763117_wp
        Xi(7) = 8.364407269306575645750889691499_wp
        Xi(8) = 13.539315320810248635501060743280_wp
        Xi(9) = 22.429276179440487052133001988352_wp
        Xi(10) = 37.862557259468652044037684234468_wp
        Xi(11) = 63.431700995949399989742856220687_wp
        Xi(12) = 93.893166381930095822083703893668_wp

      case (13)
        ! ----- K = 06 , R = 2E2 -----

        RIni = 2.0e2_wp

        Omega(1) = 0.0164846314068179163341601152839577260067_wp
        Omega(2) = 0.0611067628119328394039168686480145709083_wp
        Omega(3) = 0.1855559934147548801965793563439888202993_wp
        Omega(4) = 0.5047457845186150892030987891168081205251_wp
        Omega(5) = 1.2702004011589922115310624661788097000681_wp
        Omega(6) = 3.2370283689529729053650092263083593024930_wp

        Alpha(1) = 0.0060745561605835059879668618321862938458_wp
        Alpha(2) = 0.0411656892946592426047976772535141876119_wp
        Alpha(3) = 0.1543149999726211490301509485223441231483_wp
        Alpha(4) = 0.4758340265306222515633103459231634246862_wp
        Alpha(5) = 1.3088413484383164279902661553123266457987_wp
        Alpha(6) = 3.3901114378459397153287718795766636503686_wp

        Xi(1) = 1.024101300602834063355715821242_wp
        Xi(2) = 1.225872067300112482112275924528_wp
        Xi(3) = 1.678117393118015734172938924207_wp
        Xi(4) = 2.488617257603293679265113458321_wp
        Xi(5) = 3.863957253547394488785160504563_wp
        Xi(6) = 6.196242623729781210870082297504_wp
        Xi(7) = 10.229266424355462202985811437372_wp
        Xi(8) = 17.412774327048138035128888034819_wp
        Xi(9) = 30.682210829818703335530871711967_wp
        Xi(10) = 56.130245727702952542037539984676_wp
        Xi(11) = 105.433492840975274018600060799145_wp
        Xi(12) = 181.154627443317280025003590537835_wp

      case (14)
        ! ----- K = 06 , R = 3E2 -----

        RIni = 3.0e2_wp

        Omega(1) = 0.0125220873991649013673126546002112302602_wp
        Omega(2) = 0.0510449651372189538865904463033063009902_wp
        Omega(3) = 0.1660665774537122245792068742087366217675_wp
        Omega(4) = 0.4717249694785838887698715959917450391004_wp
        Omega(5) = 1.2233401903869242636395744949773245480174_wp
        Omega(6) = 3.1821812879622630178787223709235831847764_wp

        Alpha(1) = 0.0045403772662585912767843686170532446766_wp
        Alpha(2) = 0.0327772446201638313597389196712983405746_wp
        Alpha(3) = 0.1315230758019218048579827493038685304327_wp
        Alpha(4) = 0.4270458936595012981913849908766067642318_wp
        Alpha(5) = 1.2198581690530462475104134245640352673945_wp
        Alpha(6) = 3.2496928841682374872364136297875347736408_wp

        Xi(1) = 1.025454614360495746291576657505_wp
        Xi(2) = 1.239178295945885844762078542569_wp
        Xi(3) = 1.721535746309017900212978702257_wp
        Xi(4) = 2.595149849331321221211632543913_wp
        Xi(5) = 4.099740166302397086431352768443_wp
        Xi(6) = 6.701036214670878926602526348688_wp
        Xi(7) = 11.308385676268067748621859447411_wp
        Xi(8) = 19.762232804543666981125782378115_wp
        Xi(9) = 35.988217681413820079466825418990_wp
        Xi(10) = 68.797082696545387298092677497152_wp
        Xi(11) = 138.073633010752780642116377407547_wp
        Xi(12) = 263.244689306263335931701519143644_wp

      case (15)
        ! ----- K = 06 , R = 4E2 -----

        RIni = 4.0e2_wp

        Omega(1) = 0.0103875843304171820269982020983057235952_wp
        Omega(2) = 0.0455094300305711951095202511680115620152_wp
        Omega(3) = 0.1548849634653172308842647431070815855492_wp
        Omega(4) = 0.4522533544737275122324914045091048819813_wp
        Omega(5) = 1.1952746785512347834409124791577028190659_wp
        Omega(6) = 3.1492145201891648162006304145776880432095_wp

        Alpha(1) = 0.0037126126629825554528679758519577047693_wp
        Alpha(2) = 0.0282275965399766230694815353011173364450_wp
        Alpha(3) = 0.1188555545634441366689162088154940732920_wp
        Alpha(4) = 0.3992215234884476781219552615764811775989_wp
        Alpha(5) = 1.1681141741170859201784035241722392584052_wp
        Alpha(6) = 3.1670518413818134278395893810298389325908_wp

        Xi(1) = 1.026309589054521672854027036781_wp
        Xi(2) = 1.247623140493916250176598126131_wp
        Xi(3) = 1.749302523234483264453001383210_wp
        Xi(4) = 2.664006702747980768119714922726_wp
        Xi(5) = 4.254371915215652049692407477011_wp
        Xi(6) = 7.038230898778517470679855216886_wp
        Xi(7) = 12.045687217398791613899577956914_wp
        Xi(8) = 21.412379743666467133214603713043_wp
        Xi(9) = 39.847404817516257326259765392251_wp
        Xi(10) = 78.447323801851049439315399780526_wp
        Xi(11) = 164.775747948596678463251130608569_wp
        Xi(12) = 340.704993926550859129065429442562_wp

      case (16)
        ! ----- K = 06 , R = 5E2 -----

        RIni = 5.0e2_wp

        Omega(1) = 0.0090407515643733420463126391163066752910_wp
        Omega(2) = 0.0419530265693814966586653539964979486854_wp
        Omega(3) = 0.1474881550985332631474719264352835068621_wp
        Omega(4) = 0.4391435240680694682146884694207145116707_wp
        Omega(5) = 1.1761879348950756306226711478402080501837_wp
        Omega(6) = 3.1267420244356869851905245116796550064464_wp

        Alpha(1) = 0.0031893814523236114041891876456566445874_wp
        Alpha(2) = 0.0253389996114558638162154828588068511408_wp
        Alpha(3) = 0.1106604325079244116039264844708611690294_wp
        Alpha(4) = 0.3808951402331243825578627887340488200607_wp
        Alpha(5) = 1.1335878719950364830731243115380380004353_wp
        Alpha(6) = 3.1114724532084870984530861059980111349432_wp

        Xi(1) = 1.026911118547184017645718112632_wp
        Xi(2) = 1.253582552323086940580104675647_wp
        Xi(3) = 1.768996911522499009016863580346_wp
        Xi(4) = 2.713191598845857231215097926658_wp
        Xi(5) = 4.365907597138108712630372165719_wp
        Xi(6) = 7.284472214720572345002247427459_wp
        Xi(7) = 12.592436525112847781754377152197_wp
        Xi(8) = 22.659479034882555902041811890513_wp
        Xi(9) = 42.835993321550367429750849623815_wp
        Xi(10) = 86.171248709276722130279146938392_wp
        Xi(11) = 187.263674144445891420196481647054_wp
        Xi(12) = 413.913484907066588913693294671248_wp

      case (17)
        ! ----- K = 06 , R = 6E2 -----

        RIni = 6.0e2_wp

        Omega(1) = 0.0081091385540135693656027429492005875744_wp
        Omega(2) = 0.0394541019605209582625016985490962184713_wp
        Omega(3) = 0.1421765733804508024330824980663301460027_wp
        Omega(4) = 0.4296107974766591557622998076571008141400_wp
        Omega(5) = 1.1622091695018335619563054628144982416416_wp
        Omega(6) = 3.1102558315953168074571855150622923247283_wp

        Alpha(1) = 0.0028268458406437786379807047690629340231_wp
        Alpha(2) = 0.0233296022544698573519074196944145249688_wp
        Alpha(3) = 0.1048731918356399272484742561661885673630_wp
        Alpha(4) = 0.3677786713597671238969513685113099654700_wp
        Alpha(5) = 1.1086419641112550304095848985319605617406_wp
        Alpha(6) = 3.0710862053749614941113021959750994938076_wp

        Xi(1) = 1.027362349375534096769004011040_wp
        Xi(2) = 1.258062845629194640849519981618_wp
        Xi(3) = 1.783857473947565594691103130209_wp
        Xi(4) = 2.750495517298796569740856887698_wp
        Xi(5) = 4.451106009008991984746139713991_wp
        Xi(6) = 7.474277175325931010536201215544_wp
        Xi(7) = 13.018669023256232651283847978618_wp
        Xi(8) = 23.645500520094804292395296307916_wp
        Xi(9) = 45.242679959784439926856469682548_wp
        Xi(10) = 92.550254538427181048854119893576_wp
        Xi(11) = 206.575346884772450523071185557455_wp
        Xi(12) = 483.170485098160643666620472913564_wp

      case (18)
        ! ----- K = 06 , R = 7E2 -----

        RIni = 7.0e2_wp

        Omega(1) = 0.0074247323082149151767562551395424064182_wp
        Omega(2) = 0.0375927111082512853721488907399184142832_wp
        Omega(3) = 0.1381527530049599144834670053083591767518_wp
        Omega(4) = 0.4223205987016474642354712382141812554437_wp
        Omega(5) = 1.1514604701601748779592193638876551631256_wp
        Omega(6) = 3.0975627662452193883998313728156404067704_wp

        Alpha(1) = 0.0025601001960133667433892277467619158493_wp
        Alpha(2) = 0.0218457448223638122831329923256848957180_wp
        Alpha(3) = 0.1005461602165559508416316399082024446443_wp
        Alpha(4) = 0.3578681828337668482410954123862367737274_wp
        Alpha(5) = 1.0896558648067596280053018209521553671948_wp
        Alpha(6) = 3.0402149545555030530838425129047664086102_wp

        Xi(1) = 1.027715621821195243930369134233_wp
        Xi(2) = 1.261576389565768080143308427221_wp
        Xi(3) = 1.795544353439286316509275465148_wp
        Xi(4) = 2.779948400327085405222410363635_wp
        Xi(5) = 4.518743573786206716835084362671_wp
        Xi(6) = 7.626014676975904829549918195752_wp
        Xi(7) = 13.362419624020622798722668367333_wp
        Xi(8) = 24.449525117501510517240004283224_wp
        Xi(9) = 47.233724318332957828325824323201_wp
        Xi(10) = 97.935128153593729548320734323852_wp
        Xi(11) = 223.395485833209753306904943315203_wp
        Xi(12) = 548.719796409537993453486848238754_wp

      case (19)
        ! ----- K = 06 , R = 8E2 -----

        RIni = 8.0e2_wp

        Omega(1) = 0.0069001546543077985549532039726006171776_wp
        Omega(2) = 0.0361483829711729105784953110236390472210_wp
        Omega(3) = 0.1349880105474074203292852297542125938890_wp
        Omega(4) = 0.4165440392673443189099687783416570141526_wp
        Omega(5) = 1.1429068035001628213156305013953328852949_wp
        Omega(6) = 3.0874514747660918537620206025806623983954_wp

        Alpha(1) = 0.0023553678339827302485676510657164328677_wp
        Alpha(2) = 0.0207030056277595794300103474935847991389_wp
        Alpha(3) = 0.0971788429385239001901753211343226723784_wp
        Alpha(4) = 0.3500900854010225815668867072050218780532_wp
        Alpha(5) = 1.0746683097025738517268228511269967384578_wp
        Alpha(6) = 3.0157614053961606392657152087011240837455_wp

        Xi(1) = 1.028000752156941029246198238134_wp
        Xi(2) = 1.264415988943018486130066835305_wp
        Xi(3) = 1.805010490262082701240151094702_wp
        Xi(4) = 2.803879916986824682220008631006_wp
        Xi(5) = 4.573942469778034288722251554660_wp
        Xi(6) = 7.750540580820214526212852401699_wp
        Xi(7) = 13.646518061899365275614626202838_wp
        Xi(8) = 25.119963251522863285566944391292_wp
        Xi(9) = 48.913594408691437643438293392251_wp
        Xi(10) = 102.554530536405437556035646906594_wp
        Xi(11) = 238.202187815886369665063782008474_wp
        Xi(12) = 610.759132377441485761693229505909_wp

      case (20)
        ! ----- K = 06 , R = 9E2 -----

        RIni = 9.0e2_wp

        Omega(1) = 0.0064853094461266979653356052996915348530_wp
        Omega(2) = 0.0349935860300550033064764278434299882292_wp
        Omega(3) = 0.1324294705506935298291179906038372848798_wp
        Omega(4) = 0.4118457658248256582898295941053135038601_wp
        Omega(5) = 1.1359254669154330633345367784770019170537_wp
        Omega(6) = 3.0791919318758857709427551041514448115777_wp

        Alpha(1) = 0.0021932677980895974418018107244751635498_wp
        Alpha(2) = 0.0197953716360736805643274864585645644866_wp
        Alpha(3) = 0.0944803091081423382480261770333118676035_wp
        Alpha(4) = 0.3438129085859721877115654370404840278752_wp
        Alpha(5) = 1.0625153853630918895922796707509405678138_wp
        Alpha(6) = 2.9958772123567106030524870297071515778953_wp

        Xi(1) = 1.028236134913692806066090601469_wp
        Xi(2) = 1.266762739177218750441338002144_wp
        Xi(3) = 1.812848047722443427004664073809_wp
        Xi(4) = 2.823744785906744068698301375342_wp
        Xi(5) = 4.619926215583368029737182380856_wp
        Xi(6) = 7.854753049594836031115085983600_wp
        Xi(7) = 13.885655519336573675628909718682_wp
        Xi(8) = 25.688468482267040883873510104252_wp
        Xi(9) = 50.352049626865415805909709234811_wp
        Xi(10) = 106.565717689397061546108247398479_wp
        Xi(11) = 251.342760645662371368769250068453_wp
        Xi(12) = 669.448234681487855890491545096666_wp

      case (21)
        ! ----- K = 06 , R = 1E3 -----

        RIni = 1.0e3_wp

        Omega(1) = 0.0061493505796972659139391947143854499913_wp
        Omega(2) = 0.0340491288320566085672728810773701724202_wp
        Omega(3) = 0.1303175043656297746362884848458918440883_wp
        Omega(4) = 0.4079481557970084787390335073764546791608_wp
        Omega(5) = 1.1301170634334537682492635712883100040926_wp
        Omega(6) = 3.0723153218216233070487347633914509970054_wp

        Alpha(1) = 0.0020618552752413614870631087313407325823_wp
        Alpha(2) = 0.0190573955732373034512451125815912789108_wp
        Alpha(3) = 0.0922691708416082665949044076814100634465_wp
        Alpha(4) = 0.3386390121450568758694043419721353416207_wp
        Alpha(5) = 1.0524587572217026424205646439880013076618_wp
        Alpha(6) = 2.9793845872747349067324873495010706392350_wp

        Xi(1) = 1.028433820474111892490261255251_wp
        Xi(2) = 1.268735476408904108095582197180_wp
        Xi(3) = 1.819446326348997650503361289687_wp
        Xi(4) = 2.840504832213112635947577233608_wp
        Xi(5) = 4.658839268464775823768569962269_wp
        Xi(6) = 7.943279025846940839643284260063_wp
        Xi(7) = 14.089786680061612422418249046530_wp
        Xi(8) = 26.176766994187056540599467524544_wp
        Xi(9) = 51.597819895252218635323115947244_wp
        Xi(10) = 110.081315607349564660022434026132_wp
        Xi(11) = 263.077018330371225407482427272043_wp
        Xi(12) = 724.915711823892740806041246059976_wp

      case (22)
        ! ----- K = 06 , R = 2E3 -----

        RIni = 2.0e3_wp

        Omega(1) = 0.0046569405523143578406867516827905006949_wp
        Omega(2) = 0.0297242860906955031989184080193355796951_wp
        Omega(3) = 0.1204049483744157716992866265226513888820_wp
        Omega(4) = 0.3894115976155329942069369120183353061293_wp
        Omega(5) = 1.1022784500339817690328300670010719386482_wp
        Omega(6) = 3.0392951653690305389380471590143883986457_wp

        Alpha(1) = 0.0014764458510144259402872805191954397319_wp
        Alpha(2) = 0.0157368046519649566180259299061350297677_wp
        Alpha(3) = 0.0820975255837409421109419904727388939136_wp
        Alpha(4) = 0.3144524324480677776084708990866900535366_wp
        Alpha(5) = 1.0049446929502894142053129322711413351499_wp
        Alpha(6) = 2.9009766057570200677769717279019801026152_wp

        Xi(1) = 1.029405214778790805764049909321_wp
        Xi(2) = 1.278453273819321666239338941562_wp
        Xi(3) = 1.852083750419863797821024009593_wp
        Xi(4) = 2.923888032492990372152030786879_wp
        Xi(5) = 4.854025758190635494223469370567_wp
        Xi(6) = 8.392005549262923054000540279418_wp
        Xi(7) = 15.138500474576849719188231802036_wp
        Xi(8) = 28.729264082951998496642809222124_wp
        Xi(9) = 58.264641065642921217981875514624_wp
        Xi(10) = 129.562709698379522213484982273712_wp
        Xi(11) = 331.833089656248126891879834943211_wp
        Xi(12) = 1116.683301472208890703363692864514_wp

      case (23:31)
        ! ----- K = 06 , R = 3E3 (R = 2802.1) -----

        RIni = 3.0e3_wp

        Omega(1) = 0.0044122353526579834677601589087297950798_wp
        Omega(2) = 0.0289892203219406956332546360446955624468_wp
        Omega(3) = 0.1186779113929924570088980488591356987627_wp
        Omega(4) = 0.3861395125485350843393649034052117485771_wp
        Omega(5) = 1.0973265611814315093758276375979221484158_wp
        Omega(6) = 3.0334106111925839318731873417434030670847_wp

        Alpha(1) = 0.0013802019452414498485521955422304640759_wp
        Alpha(2) = 0.0151836143815727678219742818009618812169_wp
        Alpha(3) = 0.0803615118552532432872931474066025003822_wp
        Alpha(4) = 0.3102556086327740258025507241734786134657_wp
        Alpha(5) = 0.9966114862688711845714432946419236714064_wp
        Alpha(6) = 2.8871396539466742396824738836613732928527_wp

        Xi(1) = 1.029582279599890563067006443720_wp
        Xi(2) = 1.280228685956502250381169871751_wp
        Xi(3) = 1.858070579019150707117576104555_wp
        Xi(4) = 2.939270843402635112993903643286_wp
        Xi(5) = 4.890324755598177758428996009243_wp
        Xi(6) = 8.476319446291319014558096611722_wp
        Xi(7) = 15.338163216545078493605047675885_wp
        Xi(8) = 29.223576767850096037870688547855_wp
        Xi(9) = 59.585839834692867395676829644913_wp
        Xi(10) = 133.559268625906095689925834335554_wp
        Xi(11) = 346.736477454457815289279309922676_wp
        Xi(12) = 1218.904339756573323549915244257136_wp

    end select

  case (7)
    ! ===== K = 07 =====

    select case (InitR)

      case default !(1)
        ! ----- K = 07 , R = 2E0 -----

        RIni = 2.0_wp

        Omega(1) = 0.3418302741901420474621587503039066291421_wp
        Omega(2) = 0.8139373975876501259149251032987137932651_wp
        Omega(3) = 1.3352973461439932906149458125000251129677_wp
        Omega(4) = 1.9505008993945933530068936878620888819569_wp
        Omega(5) = 2.7444403923927759290143230108682814716303_wp
        Omega(6) = 3.9247848246178932362336622130527530316613_wp
        Omega(7) = 6.3099568403220470053441304436603331851074_wp

        Alpha(1) = 0.1328537955998787967458766787198776171408_wp
        Alpha(2) = 0.7079130219116617702703429804156343152499_wp
        Alpha(3) = 1.7769359317465696517825135547496984145255_wp
        Alpha(4) = 3.4093982762602818997483417051697074384720_wp
        Alpha(5) = 5.7364147937170302084600170822170639439719_wp
        Alpha(6) = 9.0227759271360481047041801616614975500852_wp
        Alpha(7) = 13.9533183597427484952391285588646496762522_wp

        Xi(1) = 1.002169736045136631320297271319_wp
        Xi(2) = 1.019513465126059823515937419192_wp
        Xi(3) = 1.054112462151100426720543024040_wp
        Xi(4) = 1.105723906980923610765370213027_wp
        Xi(5) = 1.173819232993989099852853719153_wp
        Xi(6) = 1.257377121511579858600775316813_wp
        Xi(7) = 1.354606744408410648819462984704_wp
        Xi(8) = 1.462620903503482741050099880109_wp
        Xi(9) = 1.577101049248088766060842125238_wp
        Xi(10) = 1.692071173794360988110359322878_wp
        Xi(11) = 1.799942384597369494621602048490_wp
        Xi(12) = 1.892020132924478879833316002923_wp
        Xi(13) = 1.959581528128465564039632773596_wp
        Xi(14) = 1.995418384645174889589248901078_wp

      case (2)
        ! ----- K = 07 , R = 5E0 -----

        RIni = 5.0_wp

        Omega(1) = 0.1938010079679485214818220051302688489159_wp
        Omega(2) = 0.4692435586300529009988209000470149590001_wp
        Omega(3) = 0.7946376596686955935222411440133072346725_wp
        Omega(4) = 1.2188241997775221336442266761856956236443_wp
        Omega(5) = 1.8330663504453953155251891282695453355700_wp
        Omega(6) = 2.8441417219183059963884957488389204627310_wp
        Omega(7) = 5.0081428000286361285758873762574694410432_wp

        Alpha(1) = 0.0751780120539355767539336714011533047142_wp
        Alpha(2) = 0.4039405737710808214351466266522905357306_wp
        Alpha(3) = 1.0300953839123784970457226028983654941840_wp
        Alpha(4) = 2.0257208844822332206511705798845923709450_wp
        Alpha(5) = 3.5300416178849783098783743362503173557343_wp
        Alpha(6) = 5.8202600945046032879064135112656686033006_wp
        Alpha(7) = 9.5718793022987917473720642647094791755080_wp

        Xi(1) = 1.005229217356591901169279856809_wp
        Xi(2) = 1.047367901529659821226875271449_wp
        Xi(3) = 1.133255240513607687907401966321_wp
        Xi(4) = 1.266031972392625048920260233398_wp
        Xi(5) = 1.450194606485410606338919137226_wp
        Xi(6) = 1.691237676145176444431235396326_wp
        Xi(7) = 1.994883041225001147787529443001_wp
        Xi(8) = 2.365467028461262406508619493550_wp
        Xi(9) = 2.802861052159663947814377538137_wp
        Xi(10) = 3.297286131697760815399342315679_wp
        Xi(11) = 3.822053774311960143168082071341_wp
        Xi(12) = 4.326580054007320654766571843552_wp
        Xi(13) = 4.736373451605665986594712046909_wp
        Xi(14) = 4.969367264862313466472487633041_wp

      case (3)
        ! ----- K = 07 , R = 1E1 -----

        RIni = 1.0e1_wp

        Omega(1) = 0.1209024409395209843302420454902446067535_wp
        Omega(2) = 0.2992121471107249075992943337931606606617_wp
        Omega(3) = 0.5287623608132956787406149523622644892384_wp
        Omega(4) = 0.8652325406126138992264282268074282455927_wp
        Omega(5) = 1.4096599453909605129337326290261955819005_wp
        Omega(6) = 2.3721982300004752701287624727655156675610_wp
        Omega(7) = 4.4850705608595406738946242963805843828595_wp

        Alpha(1) = 0.0467849719196950606124379631878262841838_wp
        Alpha(2) = 0.2541310917703071038133404851988572659138_wp
        Alpha(3) = 0.6620206366128168775329887463243494494236_wp
        Alpha(4) = 1.3468236945914095781815303376482972907979_wp
        Alpha(5) = 2.4608175652259984735742426309457187016960_wp
        Alpha(6) = 4.3018718689144129466335464417880984910880_wp
        Alpha(7) = 7.5591271744403052404090437299544191773748_wp

        Xi(1) = 1.007773548460501324597511640935_wp
        Xi(2) = 1.070746162242091784034224433153_wp
        Xi(3) = 1.200884006431706493194314089301_wp
        Xi(4) = 1.406664527874591639599460846100_wp
        Xi(5) = 1.701161285541335777644553206578_wp
        Xi(6) = 2.102581382471403875604465394389_wp
        Xi(7) = 2.634629620557902439500694291574_wp
        Xi(8) = 3.325656924246834897970076516671_wp
        Xi(9) = 4.204512745060396103607275319902_wp
        Xi(10) = 5.289117016413992876551486421732_wp
        Xi(11) = 6.561239747493596864529041834402_wp
        Xi(12) = 7.922448746536713549883795826290_wp
        Xi(13) = 9.146670152521306849935367466742_wp
        Xi(14) = 9.897948626231446764278665284564_wp

      case (4)
        ! ----- K = 07 , R = 2E1 -----

        RIni = 2.0e1_wp

        Omega(1) = 0.0736409437937410912873993466820632392000_wp
        Omega(2) = 0.1880960775887743137598211856653662721328_wp
        Omega(3) = 0.3535856122925518073838498794847495787508_wp
        Omega(4) = 0.6310263698265568733545768931314512428798_wp
        Omega(5) = 1.1268723268357688314239561933050026709680_wp
        Omega(6) = 2.0532076958089364882714678017805454146583_wp
        Omega(7) = 4.1312146140908651273035423479029759619152_wp

        Alpha(1) = 0.0283978750193578094886787902689784246490_wp
        Alpha(2) = 0.1566946545197924593886091779171287896588_wp
        Alpha(3) = 0.4212835366409847733014729098677264573780_wp
        Alpha(4) = 0.9007103394850066366002054596417991660928_wp
        Alpha(5) = 1.7551920884002582784293167161848714385997_wp
        Alpha(6) = 3.2943879648814646311262566147348707090714_wp
        Alpha(7) = 6.2167521169331160300723648770571116983774_wp

        Xi(1) = 1.010457578611366577406131816996_wp
        Xi(2) = 1.095616183002617286261666518232_wp
        Xi(3) = 1.273980560723253206830904438185_wp
        Xi(4) = 1.562112152636461765136538515364_wp
        Xi(5) = 1.986628611083233840158349114979_wp
        Xi(6) = 2.587264263840953612419784612975_wp
        Xi(7) = 3.421112813296283916997958529649_wp
        Xi(8) = 4.566870080787266725479839335478_wp
        Xi(9) = 6.126331365088566714011131253415_wp
        Xi(10) = 8.214734908047492183309767899146_wp
        Xi(11) = 10.916644469830227914992903937019_wp
        Xi(12) = 14.157909730233225860385193328739_wp
        Xi(13) = 17.448466798972224843858325193935_wp
        Xi(14) = 19.682176912149916015240536104614_wp

      case (5)
        ! ----- K = 07 , R = 3E1 -----

        RIni = 3.0e1_wp

        Omega(1) = 0.0547081236603999407457348414657305468722_wp
        Omega(2) = 0.1431673361803440539338381706779834701138_wp
        Omega(3) = 0.2818519061350474052586589923663495937944_wp
        Omega(4) = 0.5330147207850684359367252063144348994683_wp
        Omega(5) = 1.0042678371684878323210135531162734423560_wp
        Omega(6) = 1.9102105077080432205633916953324558107852_wp
        Omega(7) = 3.9707839278055398883095217232863660683506_wp

        Alpha(1) = 0.0210415974965753205954983179326589493030_wp
        Alpha(2) = 0.1175159106746453374671966020381663753369_wp
        Alpha(3) = 0.3237851994719231859306673299014889266800_wp
        Alpha(4) = 0.7182511074334368187353783152815367429866_wp
        Alpha(5) = 1.4622083631638707608250971059504763616133_wp
        Alpha(6) = 2.8681290607978677652690341348673541688186_wp
        Alpha(7) = 5.6391282837140386504162781200477638776647_wp

        Xi(1) = 1.012047889671287567302411514181_wp
        Xi(2) = 1.110459031424489875165134700197_wp
        Xi(3) = 1.318193740074288370246052837231_wp
        Xi(4) = 1.657889794272495439489585600246_wp
        Xi(5) = 2.166670828411587028600529536426_wp
        Xi(6) = 2.901774920327571963274415622891_wp
        Xi(7) = 3.948924790532304569335222965698_wp
        Xi(8) = 5.432838744548646986443074524686_wp
        Xi(9) = 7.528256253676515277663966463351_wp
        Xi(10) = 10.462117475713087632119280279142_wp
        Xi(11) = 14.471943652043618301628047362328_wp
        Xi(12) = 19.619191297398738436236209636832_wp
        Xi(13) = 25.270403304102207706419158661504_wp
        Xi(14) = 29.392338201320215115841527619978_wp

      case (6)
        ! ----- K = 07 , R = 4E1 -----

        RIni = 4.0e1_wp

        Omega(1) = 0.0442163443975152241595058272977114555147_wp
        Omega(2) = 0.1180995323059765671812955563357849797512_wp
        Omega(3) = 0.2413652023935609491770218348882792724908_wp
        Omega(4) = 0.4763952040553355030605073294758255997294_wp
        Omega(5) = 0.9312065231173353566963835081793376957648_wp
        Omega(6) = 1.8229592256705783305482590495039119105058_wp
        Omega(7) = 3.8722630501939474274158969890891057730187_wp

        Alpha(1) = 0.0169689443476473593619315970854044550720_wp
        Alpha(2) = 0.0957454021439034121850595628056002084350_wp
        Alpha(3) = 0.2692943329972380760823583850660511984643_wp
        Alpha(4) = 0.6152724295093568839128649050618236060473_wp
        Alpha(5) = 1.2943422432026802725664113569159496819339_wp
        Alpha(6) = 2.6197801492130474551502800073521370904928_wp
        Alpha(7) = 5.2979645334238242792256057267508140284917_wp

        Xi(1) = 1.013167810074629683896214882299_wp
        Xi(2) = 1.120961719456940277922681048484_wp
        Xi(3) = 1.349752481091188788231947925933_wp
        Xi(4) = 1.727074796288967656202398281984_wp
        Xi(5) = 2.298702839347466489114402143912_wp
        Xi(6) = 3.136713029630052694714248406882_wp
        Xi(7) = 4.351843312869864519899787413948_wp
        Xi(8) = 6.110484393461254443678798420336_wp
        Xi(9) = 8.656634765265322829265459692039_wp
        Xi(10) = 12.330714532485533930193655205443_wp
        Xi(11) = 17.544287001608629954912954573842_wp
        Xi(12) = 24.563692529334879703292737751497_wp
        Xi(13) = 32.732631020313671822014045531546_wp
        Xi(14) = 39.041770580631060420795375165426_wp

      case (7)
        ! ----- K = 07 , R = 5E1 -----

        RIni = 5.0e1_wp

        Omega(1) = 0.0374556861766770008238325914001753957905_wp
        Omega(2) = 0.1018584053448467243574568005803904213735_wp
        Omega(3) = 0.2148511025892223438697476584069434579760_wp
        Omega(4) = 0.4385126249624096627874065418284388329084_wp
        Omega(5) = 0.8811012601672848720498788799293521378786_wp
        Omega(6) = 1.7621025260651733441042560146527762299229_wp
        Omega(7) = 3.8032676122312763994978007664471419957408_wp

        Alpha(1) = 0.0143466169541190409781331515856606095838_wp
        Alpha(2) = 0.0816866369651240595850809518563284683523_wp
        Alpha(3) = 0.2339291825665959799257151430218470977707_wp
        Alpha(4) = 0.5478173696161150821597358995251880742217_wp
        Alpha(5) = 1.1828967187208399169549496998676829662145_wp
        Alpha(6) = 2.4526048384096859368563858794232146465220_wp
        Alpha(7) = 5.0658409539581052498900815006521725081257_wp

        Xi(1) = 1.014025144133170897075071903437_wp
        Xi(2) = 1.129030581189734573849606236173_wp
        Xi(3) = 1.374154955229312968139135375800_wp
        Xi(4) = 1.781044048069493173345502479155_wp
        Xi(5) = 2.402854810318760382702521494203_wp
        Xi(6) = 3.324598199402773961120757806320_wp
        Xi(7) = 4.679287478364376717723110976621_wp
        Xi(8) = 6.671367587349018773990039221644_wp
        Xi(9) = 9.610141382797516037064800009659_wp
        Xi(10) = 13.947791102397180343190619122851_wp
        Xi(11) = 20.279511536792605675208478466232_wp
        Xi(12) = 29.122085171822439941813343011745_wp
        Xi(13) = 39.901986900788082385238553229101_wp
        Xi(14) = 48.637992964763487789559004426110_wp

      case (8)
        ! ----- K = 07 , R = 6E1 -----

        RIni = 6.0e1_wp

        Omega(1) = 0.0326960774209725745576341198600500703719_wp
        Omega(2) = 0.0903720874058653475289029123607154048159_wp
        Omega(3) = 0.1959082814526107491255330766977138523544_wp
        Omega(4) = 0.4109239117501715364837340527248699117990_wp
        Omega(5) = 0.8438800600320581920773567452531693788842_wp
        Omega(6) = 1.7163114229596175923635986704240963263146_wp
        Omega(7) = 3.7512051733025360371592132890938842137984_wp

        Alpha(1) = 0.0125016247905536379320489532261662102286_wp
        Alpha(2) = 0.0717712321052866914869341856408890834018_wp
        Alpha(3) = 0.2088738414319771117188748177362356273079_wp
        Alpha(4) = 0.4996135480465156085655258066946515782547_wp
        Alpha(5) = 1.1023091213725187257656268347005834584706_wp
        Alpha(6) = 2.3303116968910064584104280260845598604647_wp
        Alpha(7) = 4.8945625765439792422169251739916262522456_wp

        Xi(1) = 1.014715224361727711533710805103_wp
        Xi(2) = 1.135543811863650034239478503473_wp
        Xi(3) = 1.393953321295632137338978140395_wp
        Xi(4) = 1.825135379699663323014614813022_wp
        Xi(5) = 2.488699823601178217008858539749_wp
        Xi(6) = 3.481151315361104700606478812297_wp
        Xi(7) = 4.955626374217314921051535936414_wp
        Xi(8) = 7.151627151292512403132906051439_wp
        Xi(9) = 10.440090096247071853492738302549_wp
        Xi(10) = 15.382079466381195343056831337236_wp
        Xi(11) = 22.760534155163923900402900279971_wp
        Xi(12) = 33.373972944163259365946139212156_wp
        Xi(13) = 46.822227620804437678986298365658_wp
        Xi(14) = 58.185982218651912008244853069527_wp

      case (9)
        ! ----- K = 07 , R = 7E1 -----

        RIni = 7.0e1_wp

        Omega(1) = 0.0291433464945640642622870161715376013944_wp
        Omega(2) = 0.0817642335427500664324818209494072362986_wp
        Omega(3) = 0.1815754004803848480574784279850675261514_wp
        Omega(4) = 0.3896901972802879141066518392522155522784_wp
        Omega(5) = 0.8147620352906383847462444547371518410728_wp
        Omega(6) = 1.6801229596135672934549901147427419800806_wp
        Omega(7) = 3.7099722173278536427321050439331884263083_wp

        Alpha(1) = 0.0111252038160296457427713345475495110293_wp
        Alpha(2) = 0.0643586014972104123637100525634480163717_wp
        Alpha(3) = 0.1900644749502537916602696449475118356531_wp
        Alpha(4) = 0.4631352947176304528224301260141970715267_wp
        Alpha(5) = 1.0406826317548054407213786731745130964555_wp
        Alpha(6) = 2.2358669464308131998551976149514075586922_wp
        Alpha(7) = 4.7613343074885285197833661996469345467631_wp

        Xi(1) = 1.015289737852678177056769959208_wp
        Xi(2) = 1.140979005247219047109764888948_wp
        Xi(3) = 1.410544090266070947148682734884_wp
        Xi(4) = 1.862294490029174263436527458904_wp
        Xi(5) = 2.561577244847484791058153663634_wp
        Xi(6) = 3.615255468026502653393697284123_wp
        Xi(7) = 5.194845325019531162482472330666_wp
        Xi(8) = 7.572390344480325552467547822744_wp
        Xi(9) = 11.177153038806722059467224639207_wp
        Xi(10) = 16.675834464013250280425837424048_wp
        Xi(11) = 25.040315400228195095999605968018_wp
        Xi(12) = 37.372988442251295473917060974145_wp
        Xi(13) = 53.524677150646856726295963468942_wp
        Xi(14) = 67.689337530528587344447188911545_wp

      case (10)
        ! ----- K = 07 , R = 8E1 -----

        RIni = 8.0e1_wp

        Omega(1) = 0.0263787173200564479601888581011825962719_wp
        Omega(2) = 0.0750421509086382211394091953926555049748_wp
        Omega(3) = 0.1702797881866766488298753703944932169634_wp
        Omega(4) = 0.3726996354512291501385416042646525625059_wp
        Omega(5) = 0.7911419754631662469546680982457331765545_wp
        Omega(6) = 1.6505212211957283195869206071648704892141_wp
        Omega(7) = 3.6761865418821602685597177417875514038315_wp

        Alpha(1) = 0.0100546153997157513998472223859526053502_wp
        Alpha(2) = 0.0585824764836521650425210303025913560759_wp
        Alpha(3) = 0.1753502950297512436001563448928486366185_wp
        Alpha(4) = 0.4343856131368536278131945588265594437871_wp
        Alpha(5) = 0.9916572813188190211371869642320575621852_wp
        Alpha(6) = 2.1600909301499331841359130557123080507154_wp
        Alpha(7) = 4.6537869270461901666235959851292136590928_wp

        Xi(1) = 1.015779815962351235580224395605_wp
        Xi(2) = 1.145624668844172087999247477974_wp
        Xi(3) = 1.424775149344744374084632820843_wp
        Xi(4) = 1.894322217755564204230873948109_wp
        Xi(5) = 2.624780635964467335411592907590_wp
        Xi(6) = 3.732449332910154737429769822299_wp
        Xi(7) = 5.405778563515919540698184686534_wp
        Xi(8) = 7.947205775677921247169416263390_wp
        Xi(9) = 11.841357539307096187648771223877_wp
        Xi(10) = 17.857273402076895882362173040292_wp
        Xi(11) = 27.155320486195248888897091354977_wp
        Xi(12) = 41.157692449794122908218740874986_wp
        Xi(13) = 60.033087433123451526206793005258_wp
        Xi(14) = 77.150815062068650709847794644247_wp

      case (11)
        ! ----- K = 07 , R = 9E1 -----

        RIni = 9.0e1_wp

        Omega(1) = 0.0241592217050219513695098142586070721904_wp
        Omega(2) = 0.0696282194693923318982824166578637914427_wp
        Omega(3) = 0.1611029161809211525906290404797083226640_wp
        Omega(4) = 0.3587065401325918283790324808668614764429_wp
        Omega(5) = 0.7714604530426776657575996254401218266139_wp
        Omega(6) = 1.6256811668432707961582175948400674769800_wp
        Omega(7) = 3.6477955931137628568360770087153355234477_wp

        Alpha(1) = 0.0091954804054835872354338476876434160800_wp
        Alpha(2) = 0.0539396604718906497494984066121137544769_wp
        Alpha(3) = 0.1634793191158028323581221513682848467397_wp
        Alpha(4) = 0.4110293908202165037984055634412428048563_wp
        Alpha(5) = 0.9514935033869456469830805378506966007990_wp
        Alpha(6) = 2.0975458645195081742486353792997988421121_wp
        Alpha(7) = 4.5645482583921669532747711350850750022801_wp

        Xi(1) = 1.016205645890911096708084104812_wp
        Xi(2) = 1.149668216519620178562265111477_wp
        Xi(3) = 1.437199773265788005512620661985_wp
        Xi(4) = 1.922401009668877397938702156210_wp
        Xi(5) = 2.680488469883554993608912320191_wp
        Xi(6) = 3.836430600122321613997672384855_wp
        Xi(7) = 5.594389492923190437909802819050_wp
        Xi(8) = 8.285339363704412511332786372975_wp
        Xi(9) = 12.446610927164962057470121159497_wp
        Xi(10) = 18.946365628555859900172819720865_wp
        Xi(11) = 29.132014537375160499396709035125_wp
        Xi(12) = 44.757116842193479560457758381631_wp
        Xi(13) = 66.366208781268083836957227816811_wp
        Xi(14) = 86.572612087553766122005605154754_wp

      case (12)
        ! ----- K = 07 , R = 1E2 -----

        RIni = 1.0e2_wp

        Omega(1) = 0.0223336425511763742141036447053092928172_wp
        Omega(2) = 0.0651620107304038612404959232571322047534_wp
        Omega(3) = 0.1534694844777638628207424437155381724551_wp
        Omega(4) = 0.3469229620284127962143942008887975703146_wp
        Omega(5) = 0.7547179370389609007045185473661064179396_wp
        Omega(6) = 1.6044221299260411336745288379823648483580_wp
        Omega(7) = 3.6234684847661182520439648868659787694924_wp

        Alpha(1) = 0.0084890821340208078627211844661981654525_wp
        Alpha(2) = 0.0501166420149617110350252335154719318666_wp
        Alpha(3) = 0.1536699828592926572490588374431119689234_wp
        Alpha(4) = 0.3916035531845051195505135605001711951445_wp
        Alpha(5) = 0.9178336466850517870669570741348763931455_wp
        Alpha(6) = 2.0447798621093103792461381784661966776184_wp
        Alpha(7) = 4.4889127432163770567950800138845579567715_wp

        Xi(1) = 1.016581033751988272229027643334_wp
        Xi(2) = 1.153238255900683739696432894561_wp
        Xi(3) = 1.448198864476133218559539272974_wp
        Xi(4) = 1.947348932807454913174183275260_wp
        Xi(5) = 2.730218361459509690547520222026_wp
        Xi(6) = 3.929796491821510695530150769628_wp
        Xi(7) = 5.764907707920420065761391015258_wp
        Xi(8) = 8.593437345984634490037024079356_wp
        Xi(9) = 13.003022433206446351050544762984_wp
        Xi(10) = 19.957861788850703277375941979699_wp
        Xi(11) = 30.990380197874582535177978570573_wp
        Xi(12) = 48.193880797171424035241216188297_wp
        Xi(13) = 72.539294925666396807173441629857_wp
        Xi(14) = 95.956531905924009606678737327456_wp

      case (13)
        ! ----- K = 07 , R = 2E2 -----

        RIni = 2.0e2_wp

        Omega(1) = 0.0133615547183825570028305575534521842940_wp
        Omega(2) = 0.0429728469424360175410925952177443321034_wp
        Omega(3) = 0.1143029399081515586560726591147663100401_wp
        Omega(4) = 0.2838881266934189482611071431161775535656_wp
        Omega(5) = 0.6622322841999484042811198458711174907876_wp
        Omega(6) = 1.4847175320092703810050463464342840325116_wp
        Omega(7) = 3.4859753729916252771962870138366952232900_wp

        Alpha(1) = 0.0050213411684266507485648978019454613531_wp
        Alpha(2) = 0.0312546410994290844202411500801774835168_wp
        Alpha(3) = 0.1045970270084145620410366606112262388706_wp
        Alpha(4) = 0.2920522758702768403556507270657505159761_wp
        Alpha(5) = 0.7407504784499061527671195936939341208927_wp
        Alpha(6) = 1.7609744335543204401530945069076494746696_wp
        Alpha(7) = 4.0759036969145123916954953635638503328664_wp

        Xi(1) = 1.018899180639725698428732503231_wp
        Xi(2) = 1.175398537435454030561192417803_wp
        Xi(3) = 1.517098188949008791422644326641_wp
        Xi(4) = 2.105575536480536643453229106449_wp
        Xi(5) = 3.050761571414857339359580357119_wp
        Xi(6) = 4.543870834868153870452628373400_wp
        Xi(7) = 6.913496021851973589392864605330_wp
        Xi(8) = 10.726771563318656134249085010168_wp
        Xi(9) = 16.979968567486211656475392217658_wp
        Xi(10) = 27.460101193699478661081059271964_wp
        Xi(11) = 45.403849726057555587915848249736_wp
        Xi(12) = 76.434664319683272064309687010564_wp
        Xi(13) = 127.557313717779402660601473229462_wp
        Xi(14) = 187.964656376434722406809640915526_wp

      case (14)
        ! ----- K = 07 , R = 3E2 -----

        RIni = 3.0e2_wp

        Omega(1) = 0.0099434776685598156043417164204334390831_wp
        Omega(2) = 0.0343511106592218869145501525946828635938_wp
        Omega(3) = 0.0981869859630001312231998428226020791953_wp
        Omega(4) = 0.2563375496711152665965313679485859665874_wp
        Omega(5) = 0.6200522686047906112676193768340482392887_wp
        Omega(6) = 1.4287348681193340442507899723523223656230_wp
        Omega(7) = 3.4213578343787924831639890577150708850240_wp

        Alpha(1) = 0.0037021290962675962693202485944560242714_wp
        Alpha(2) = 0.0240203563378520316728596080066426843302_wp
        Alpha(3) = 0.0852762937444175518725514133222542767498_wp
        Alpha(4) = 0.2512427085458493746265504209524976886314_wp
        Alpha(5) = 0.6652246780099562398647777827775939840649_wp
        Alpha(6) = 1.6360585970149727252963625656612123293598_wp
        Alpha(7) = 3.8902964899531037119752990838250639171747_wp

        Xi(1) = 1.020114753497002453054565140889_wp
        Xi(2) = 1.187099191216807305706247566146_wp
        Xi(3) = 1.553915186630408594393656929178_wp
        Xi(4) = 2.191512615183995796221821938943_wp
        Xi(5) = 3.228604039450749353841463307191_wp
        Xi(6) = 4.893713679900430141896799929313_wp
        Xi(7) = 7.588583297277757381844170980756_wp
        Xi(8) = 12.026587110560332955985773928376_wp
        Xi(9) = 19.505469250220939298259836824911_wp
        Xi(10) = 32.459052347340309092882382202561_wp
        Xi(11) = 55.579138018720356601704013499443_wp
        Xi(12) = 97.897786866246901023302928734893_wp
        Xi(13) = 174.028024411478690586241313553728_wp
        Xi(14) = 277.096563357526212451054092866798_wp

      case (15)
        ! ----- K = 07 , R = 4E2 -----

        RIni = 4.0e2_wp

        Omega(1) = 0.0080939178056842052675398224009795100642_wp
        Omega(2) = 0.0296137712027515596681381555149714657205_wp
        Omega(3) = 0.0889710554315036591940877956641298140994_wp
        Omega(4) = 0.2400346223587189581734823926040256480974_wp
        Omega(5) = 0.5945144839955654431193528275390036696990_wp
        Omega(6) = 1.3943733291175232474326181608681451962184_wp
        Omega(7) = 3.3815881302823882889520973327179831358080_wp

        Alpha(1) = 0.0029885877621226805003736622535354117147_wp
        Alpha(2) = 0.0200860577922968259766053076079139749766_wp
        Alpha(3) = 0.0745569090055890414765402129937399422488_wp
        Alpha(4) = 0.2279899926003866874278918491780743238451_wp
        Alpha(5) = 0.6211585820580099310248868571537883553901_wp
        Alpha(6) = 1.5618502570945419418314142889769868816074_wp
        Alpha(7) = 3.7787455933745262991359237236821400074405_wp

        Xi(1) = 1.020907027261065597428107976086_wp
        Xi(2) = 1.194755421073434828059187107208_wp
        Xi(3) = 1.578171835966652915761096465097_wp
        Xi(4) = 2.248663631211762458468259029409_wp
        Xi(5) = 3.348346364020586124520331705590_wp
        Xi(6) = 5.132934571193718323138532877437_wp
        Xi(7) = 8.058748380630870102037111024629_wp
        Xi(8) = 12.951435148119643400721479054027_wp
        Xi(9) = 21.347969131813360109395372177232_wp
        Xi(10) = 36.215466708309418209649210851353_wp
        Xi(11) = 63.506453953796701269313151883011_wp
        Xi(12) = 115.417602470391644095548677739771_wp
        Xi(13) = 214.691514081141909806027534557415_wp
        Xi(14) = 363.703742836510989511200619972442_wp

      case (16)
        ! ----- K = 07 , R = 5E2 -----

        RIni = 5.0e2_wp

        Omega(1) = 0.0069203048135253428019736431434094647841_wp
        Omega(2) = 0.0265678981798603566214964869646175671392_wp
        Omega(3) = 0.0828637474252807121509402550618705873831_wp
        Omega(4) = 0.2289794140563879391050928699402611243841_wp
        Omega(5) = 0.5769324424498350087852326095738675348912_wp
        Omega(6) = 1.3704988341020230761639620009617601681384_wp
        Omega(7) = 3.3539045747733188186819053289156045138952_wp

        Alpha(1) = 0.0025357978643651919453790228361350633790_wp
        Alpha(2) = 0.0175787590913981830039277591895552355084_wp
        Alpha(3) = 0.0676131686107736120728336354224197179974_wp
        Alpha(4) = 0.2126283417185967508991617513913752191002_wp
        Alpha(5) = 0.5915602300563041923654392095777154736425_wp
        Alpha(6) = 1.5113889410817196977138837654663916509890_wp
        Alpha(7) = 3.7022956650360443564638096569652248035709_wp

        Xi(1) = 1.021479441200213300524474169251_wp
        Xi(2) = 1.200302232522448429331840547007_wp
        Xi(3) = 1.595827735605926870027064523594_wp
        Xi(4) = 2.290529108175524496300579091468_wp
        Xi(5) = 3.436812160845001003738052869352_wp
        Xi(6) = 5.311563852341080136862383476171_wp
        Xi(7) = 8.414316926197732417549990380223_wp
        Xi(8) = 13.661402656813166562113637247933_wp
        Xi(9) = 22.787609388017212220037266412476_wp
        Xi(10) = 39.213126744657628864471066165720_wp
        Xi(11) = 69.999978856728368538375750773639_wp
        Xi(12) = 130.265809858301510423816971240285_wp
        Xi(13) = 250.980726330417617891965775811514_wp
        Xi(14) = 447.983276787880737329627933718257_wp

      case (17)
        ! ----- K = 07 , R = 6E2 -----

        RIni = 6.0e2_wp

        Omega(1) = 0.0061035796916500113814640741949357760632_wp
        Omega(2) = 0.0244232192416007003895835926476909349958_wp
        Omega(3) = 0.0784589868370574932828170715282123381940_wp
        Omega(4) = 0.2208702419251439254743382689660791129427_wp
        Omega(5) = 0.5638921330027344504579565886981384892351_wp
        Omega(6) = 1.3526720019223379766525397482901382772980_wp
        Omega(7) = 3.3332048728001092039677916156747983222886_wp

        Alpha(1) = 0.0022206030882728451253574190622092077163_wp
        Alpha(2) = 0.0158270515451072657162057133461985358736_wp
        Alpha(3) = 0.0626946503162819286951099040372525905696_wp
        Alpha(4) = 0.2015781143691273810301077540341196936424_wp
        Alpha(5) = 0.5700007444812679664497730158689847712594_wp
        Alpha(6) = 1.4742950212653216273198109043107706384035_wp
        Alpha(7) = 3.6457725385269232113404080930507689117803_wp

        Xi(1) = 1.021919091049763004728188897641_wp
        Xi(2) = 1.204571103144114048617840939670_wp
        Xi(3) = 1.609463268497709121816405253291_wp
        Xi(4) = 2.323015621981505911654644624953_wp
        Xi(5) = 3.505896984626772737523442424745_wp
        Xi(6) = 5.452178620159553543850217138811_wp
        Xi(7) = 8.696906558515406310272410905782_wp
        Xi(8) = 14.232062660029646540896242079555_wp
        Xi(9) = 23.960469947688607286750972491518_wp
        Xi(10) = 41.695231511953006958010048421670_wp
        Xi(11) = 75.486978336182950784227774221335_wp
        Xi(12) = 143.151997159489598895887318974474_wp
        Xi(13) = 283.787903272999503967444212548799_wp
        Xi(14) = 530.071360403332262234421534685680_wp

      case (18)
        ! ----- K = 07 , R = 7E2 -----

        RIni = 7.0e2_wp

        Omega(1) = 0.0054997334786715528114468842270368131864_wp
        Omega(2) = 0.0228205582831145280442084171540484405227_wp
        Omega(3) = 0.0751020958906572246355681961305084115565_wp
        Omega(4) = 0.2146086969873823602510034408430428243264_wp
        Omega(5) = 0.5537362828757469355414784495383884177500_wp
        Omega(6) = 1.3387155190228089959641674666634969526058_wp
        Omega(7) = 3.3169815640530344305267340665466235805070_wp

        Alpha(1) = 0.0019874597606895922284125043360408779947_wp
        Alpha(2) = 0.0145272077069423383971261110568506946805_wp
        Alpha(3) = 0.0590012024024202301324961653788969684342_wp
        Alpha(4) = 0.1931754652858629902690071333815069465345_wp
        Alpha(5) = 0.5534433535878024911706767052788080718528_wp
        Alpha(6) = 1.4456024138634961035462106471172205601761_wp
        Alpha(7) = 3.6018543623381220743304848008747853782552_wp

        Xi(1) = 1.022270812436681717877591524246_wp
        Xi(2) = 1.207991628120255596184028090700_wp
        Xi(3) = 1.620418778730794495636022278795_wp
        Xi(4) = 2.349214970193795601512143367273_wp
        Xi(5) = 3.561890991607034621154598275083_wp
        Xi(6) = 5.566867419566254913607106225015_wp
        Xi(7) = 8.929140258862125879499305192155_wp
        Xi(8) = 14.705262472528805760968451110671_wp
        Xi(9) = 24.943550746590106043099055455059_wp
        Xi(10) = 43.803025607890732375654385677421_wp
        Xi(11) = 80.223932633082059941165287142439_wp
        Xi(12) = 154.522575945131116492703426956723_wp
        Xi(13) = 313.725737845372681794620106643379_wp
        Xi(14) = 610.074245598087845920964866763825_wp

      case (19)
        ! ----- K = 07 , R = 8E2 -----

        RIni = 8.0e2_wp

        Omega(1) = 0.0050337173255694819542215847577720788664_wp
        Omega(2) = 0.0215715572186778580597268017635115278097_wp
        Omega(3) = 0.0724425268654193898633223726302166056712_wp
        Omega(4) = 0.2095952536416926865374746324910404382535_wp
        Omega(5) = 0.5455484720427803431832061709094716661639_wp
        Omega(6) = 1.3274159182138670886588402431094380062859_wp
        Omega(7) = 3.3038349430915769215123811708423318123096_wp

        Alpha(1) = 0.0018074376308293873225900948837306658756_wp
        Alpha(2) = 0.0135206342248096353230596162325882758637_wp
        Alpha(3) = 0.0561110636131199456631791448041335357289_wp
        Alpha(4) = 0.1865310578633183984223259041512754663472_wp
        Alpha(5) = 0.5402437912418699069187419392612525825825_wp
        Alpha(6) = 1.4225951349686722217403878065766775762313_wp
        Alpha(7) = 3.5665103665179134825174989131824077048805_wp

        Xi(1) = 1.022560533027442646698476624856_wp
        Xi(2) = 1.210812880720948947802471928892_wp
        Xi(3) = 1.629474819752517520191070765634_wp
        Xi(4) = 2.370937131634333485058832580172_wp
        Xi(5) = 3.608506609082108268493627178941_wp
        Xi(6) = 5.662837814619105219777345228138_wp
        Xi(7) = 9.124675675335249740854903777176_wp
        Xi(8) = 15.106626870729963952461050169518_wp
        Xi(9) = 25.784828768779630973895233836402_wp
        Xi(10) = 45.626411447423677481688431001317_wp
        Xi(11) = 84.378553476106636921372228243854_wp
        Xi(12) = 164.680771008311224751219370432409_wp
        Xi(13) = 341.242194277617065245600258549530_wp
        Xi(14) = 688.080995614329459397762178696212_wp

      case (20)
        ! ----- K = 07 , R = 9E2 -----

        RIni = 9.0e2_wp

        Omega(1) = 0.0046623733840567663782794506806272938348_wp
        Omega(2) = 0.0205672519359959761756968419851854257274_wp
        Omega(3) = 0.0702737396603948621029152137029694102921_wp
        Omega(4) = 0.2054711368999697145915550242079028464559_wp
        Omega(5) = 0.5387744815015728043591965168435109490019_wp
        Omega(6) = 1.3180345908931347003508610660205135900469_wp
        Omega(7) = 3.2929120497385742784117457215842250661808_wp

        Alpha(1) = 0.0016639061581903090382578440289268462138_wp
        Alpha(2) = 0.0127159638686803215795124489574663417812_wp
        Alpha(3) = 0.0537791314332657295004772469299014758803_wp
        Alpha(4) = 0.1811218120543096948355443406852849363986_wp
        Alpha(5) = 0.5294244944099308792118813160598733702500_wp
        Alpha(6) = 1.4036450526560744079661996464025719433266_wp
        Alpha(7) = 3.5373112546752497732747283487597655948775_wp

        Xi(1) = 1.022804500014966087081264012326_wp
        Xi(2) = 1.213191167955960221135763843492_wp
        Xi(3) = 1.637123071334967660292285251966_wp
        Xi(4) = 2.389329226512886018406883748000_wp
        Xi(5) = 3.648110078430141154292795646619_wp
        Xi(6) = 5.744722848990831974829579520403_wp
        Xi(7) = 9.292377415951423910885076384858_wp
        Xi(8) = 15.452991735382019830233324420021_wp
        Xi(9) = 26.516284754150251320309328306735_wp
        Xi(10) = 47.226412734817914820112383367245_wp
        Xi(11) = 88.067292490806068507025461400417_wp
        Xi(12) = 173.844074280575919919100691402036_wp
        Xi(13) = 366.679168982654130809484271935617_wp
        Xi(14) = 764.169197413019728071237324229514_wp

      case (21)
        ! ----- K = 07 , R = 1E3 -----

        RIni = 1.0e3_wp

        Omega(1) = 0.0043590321081121139760861777322220600439_wp
        Omega(2) = 0.0197399376432508969001888149419343676527_wp
        Omega(3) = 0.0684653140928303800778536788995687345505_wp
        Omega(4) = 0.2020068374139948520101014300953679025952_wp
        Omega(5) = 0.5330567089683808103500566033527974241224_wp
        Omega(6) = 1.3100924544159886419789073919162092352053_wp
        Omega(7) = 3.2836589740207806184697780782499876295333_wp

        Alpha(1) = 0.0015465907100920743574390230079540486940_wp
        Alpha(2) = 0.0120566431726508442136752479942329863150_wp
        Alpha(3) = 0.0518524899332544268603262683364762963834_wp
        Alpha(4) = 0.1766179546713693626684661339143467273516_wp
        Alpha(5) = 0.5203635254756012724994956863966422133672_wp
        Alpha(6) = 1.3877091798262134707134599076461256572657_wp
        Alpha(7) = 3.5126938632453743125566569815987350011710_wp

        Xi(1) = 1.023013513345144816463032355625_wp
        Xi(2) = 1.215230582716563716714000542574_wp
        Xi(3) = 1.643691936875600002583922010224_wp
        Xi(4) = 2.405159331170766592216184176500_wp
        Xi(5) = 3.682296258249919650379702873444_wp
        Xi(6) = 5.815666045838884868832141528117_wp
        Xi(7) = 9.438312805633033496927053462855_wp
        Xi(8) = 15.755998129717939035959739513260_wp
        Xi(9) = 27.160318797056292632746687942813_wp
        Xi(10) = 48.646378109863155934922129830511_wp
        Xi(11) = 91.374590421681239627016424265094_wp
        Xi(12) = 182.174592894364896014658405931641_wp
        Xi(13) = 390.305983622066960142538150080327_wp
        Xi(14) = 838.408036003843683214853399476851_wp

      case (22)
        ! ----- K = 07 , R = 2E3 -----

        RIni = 2.0e3_wp

        Omega(1) = 0.0029076880267603933005398324359788708726_wp
        Omega(2) = 0.0156597956896735144358535993941439201649_wp
        Omega(3) = 0.0592095502043018142181824651354427047067_wp
        Omega(4) = 0.1838875600867601159041754901868337412907_wp
        Omega(5) = 0.5027200930136521302951831124961046270982_wp
        Omega(6) = 1.2675787498436361208815903456326168452506_wp
        Omega(7) = 3.2340325901955293901816196333598441015056_wp

        Alpha(1) = 0.0009839513388777788122618579800399141178_wp
        Alpha(2) = 0.0088654691035194740209818644628703143695_wp
        Alpha(3) = 0.0422681213886127304387536549046266998175_wp
        Alpha(4) = 0.1536724501306380062459018207010608136898_wp
        Alpha(5) = 0.4733864179916387237624132711122371119927_wp
        Alpha(6) = 1.3040672497163430146966886513659744650795_wp
        Alpha(7) = 3.3825029684199556978714540367292329392512_wp

        Xi(1) = 1.024172280366627781934829588550_wp
        Xi(2) = 1.226568594740137924472590802427_wp
        Xi(3) = 1.680383653786395965409883324071_wp
        Xi(4) = 2.494162545601589924502972106524_wp
        Xi(5) = 3.876214624517233026026324926860_wp
        Xi(6) = 6.222616866269559330205468627994_wp
        Xi(7) = 10.286950020887491977564232037423_wp
        Xi(8) = 17.547328549187459604402627988406_wp
        Xi(9) = 31.045965513023039257511914712495_wp
        Xi(10) = 57.434448623674729736410249358869_wp
        Xi(11) = 112.539312812692807910797121451196_wp
        Xi(12) = 238.042729684642031182106869380277_wp
        Xi(13) = 560.898336854662458939024816118035_wp
        Xi(14) = 1490.286275794594197607878527378489_wp

      case (23)
        ! ----- K = 07 , R = 3E3 -----

        RIni = 3.0e3_wp

        Omega(1) = 0.0023891359913667532623401029484230817523_wp
        Omega(2) = 0.0141270708183685676429170753609643185200_wp
        Omega(3) = 0.0555607731872176977069453709251467543595_wp
        Omega(4) = 0.1765507108276267910564353777469470685446_wp
        Omega(5) = 0.4902176680956567550581301495604691353947_wp
        Omega(6) = 1.2498655911588724229822205957773917361919_wp
        Omega(7) = 3.2133062542046689007927545711496009062103_wp

        Alpha(1) = 0.0007821171870305443173034760429886136368_wp
        Alpha(2) = 0.0077017029989496356004449804759370845630_wp
        Alpha(3) = 0.0386294252067045345620343969883503376650_wp
        Alpha(4) = 0.1446819307132389343853841670584081668949_wp
        Alpha(5) = 0.4545657667811826265153243853411879626947_wp
        Alpha(6) = 1.2700397189863825259895785402264323238342_wp
        Alpha(7) = 3.3290409499662179716207577317987897913554_wp

        Xi(1) = 1.024675716539555973241988617861_wp
        Xi(2) = 1.231511459124680835488033214808_wp
        Xi(3) = 1.696471376869980616391465899628_wp
        Xi(4) = 2.533496948444540662224652716183_wp
        Xi(5) = 3.962846228259305424732397704624_wp
        Xi(6) = 6.406910222305436564300962221097_wp
        Xi(7) = 10.677689748845748881310446787296_wp
        Xi(8) = 18.388813827747888991065439157069_wp
        Xi(9) = 32.917116563310722682916820147625_wp
        Xi(10) = 61.800599438868931787688376644496_wp
        Xi(11) = 123.497094864236621504505198032575_wp
        Xi(12) = 268.713219652258063802063858815927_wp
        Xi(13) = 663.478795889117814399416062087766_wp
        Xi(14) = 1999.469575084823482424845053628815_wp

      case (24)
        ! ----- K = 07 , R = 4E3 -----

        RIni = 4.0e3_wp

        Omega(1) = 0.0021345887454745103229132954271440869576_wp
        Omega(2) = 0.0133516115008142197954869455529813837558_wp
        Omega(3) = 0.0536724306078645814847285448051517064982_wp
        Omega(4) = 0.1727066836549376267545205521081186361698_wp
        Omega(5) = 0.4836135983756697330070324636341894120051_wp
        Omega(6) = 1.2404615898649592524395288140048876357469_wp
        Omega(7) = 3.2022901936660388647754027502401186211500_wp

        Alpha(1) = 0.0006828418611695833685875723414321347704_wp
        Alpha(2) = 0.0071228279773048582995505152888615629792_wp
        Alpha(3) = 0.0367803190813663724561134293206121625985_wp
        Alpha(4) = 0.1400427289025622261988942088284382236907_wp
        Alpha(5) = 0.4447519660204219866823233120456393407949_wp
        Alpha(6) = 1.2521695310166051146776380886826984806248_wp
        Alpha(7) = 3.3008429720439399334547259101668714720290_wp

        Xi(1) = 1.024948138888778894684851439312_wp
        Xi(2) = 1.234190219000572436014451227493_wp
        Xi(3) = 1.705213827210675740656334320455_wp
        Xi(4) = 2.554952491376573394006588291205_wp
        Xi(5) = 4.010342010249804854324362501927_wp
        Xi(6) = 6.508599720176197863524825137915_wp
        Xi(7) = 10.895005443340458234878498355869_wp
        Xi(8) = 18.861337129879108520311148389226_wp
        Xi(9) = 33.980614487849755446513544399068_wp
        Xi(10) = 64.320822143869004941518774387532_wp
        Xi(11) = 129.955360699156203016513266845777_wp
        Xi(12) = 287.355331978838669043652132017996_wp
        Xi(13) = 728.957088334483430391852465390912_wp
        Xi(14) = 2380.807754337805310029807515093125_wp

      case (25)
        ! ----- K = 07 , R = 5E3 -----

        RIni = 5.0e3_wp

        Omega(1) = 0.0020013110880352670689172903334754338722_wp
        Omega(2) = 0.0129378908695508702795281923136894164372_wp
        Omega(3) = 0.0526524725928330593509372851979133400846_wp
        Omega(4) = 0.1706164960435682915551678512366962081614_wp
        Omega(5) = 0.4800066743351757202824987846856430451226_wp
        Omega(6) = 1.2353112694406893889422990873150354218524_wp
        Omega(7) = 3.1962532851668917213347476113938228081679_wp

        Alpha(1) = 0.0006308136329875133503749580097738625106_wp
        Alpha(2) = 0.0068171406204782821707966887665730570944_wp
        Alpha(3) = 0.0357916075110693214643040830497122684051_wp
        Alpha(4) = 0.1375409863779743612616484477184641832537_wp
        Alpha(5) = 0.4394293305525136815502598719440641161782_wp
        Alpha(6) = 1.2424396843361854057092782954718757082446_wp
        Alpha(7) = 3.2854539018961136489019314588233555696206_wp

        Xi(1) = 1.025098873577052448636398596093_wp
        Xi(2) = 1.235673707244880951635658539178_wp
        Xi(3) = 1.710062509020929812368801148370_wp
        Xi(4) = 2.566876297161696589841164972690_wp
        Xi(5) = 4.036811428959137008735946761107_wp
        Xi(6) = 6.565473574729242831147374426237_wp
        Xi(7) = 11.017072395598808961375558723716_wp
        Xi(8) = 19.128174733489920446988241664599_wp
        Xi(9) = 34.585218590953022612527467671839_wp
        Xi(10) = 65.766092894947603769706390153260_wp
        Xi(11) = 133.702927996013346795267651145878_wp
        Xi(12) = 298.367673115269793904236195203339_wp
        Xi(13) = 768.766113582342625665244639776574_wp
        Xi(14) = 2636.957864008649196518874191497162_wp

      case (26)
        ! ----- K = 07 , R = 6E3 -----

        RIni = 6.0e3_wp

        Omega(1) = 0.0019443655262837805540929227176606686189_wp
        Omega(2) = 0.0127592807278033828906170380271256625804_wp
        Omega(3) = 0.0522093509652323378418623672780896072254_wp
        Omega(4) = 0.1697052986393442418718751144024459165394_wp
        Omega(5) = 0.4784306996286163934429570832040923278328_wp
        Omega(6) = 1.2330577486624195179083438023148744377977_wp
        Omega(7) = 3.1936109524714979715324936204723371702130_wp

        Alpha(1) = 0.0006085749350024496484217942501512843556_wp
        Alpha(2) = 0.0066859021591416628164316834654484766887_wp
        Alpha(3) = 0.0353643071742100135552912699293814924317_wp
        Alpha(4) = 0.1364550050371723066521120401595190685384_wp
        Alpha(5) = 0.4371119961322447686495520219329691258281_wp
        Alpha(6) = 1.2381951172423498714432119061079617949872_wp
        Alpha(7) = 3.2787324516562921345678299767101293582527_wp

        Xi(1) = 1.025165174452603894690583263838_wp
        Xi(2) = 1.236326517914779124572392798864_wp
        Xi(3) = 1.712197846342166541056678541466_wp
        Xi(4) = 2.572132749436840402955731232915_wp
        Xi(5) = 4.048496441962004332472402357013_wp
        Xi(6) = 6.590626762178348379064840578678_wp
        Xi(7) = 11.071180942174621612827944971968_wp
        Xi(8) = 19.246778242553667320410837326961_wp
        Xi(9) = 34.854893906749322175786520716656_wp
        Xi(10) = 66.413622087835886102780147410840_wp
        Xi(11) = 135.392437669718992576672889072142_wp
        Xi(12) = 303.379269717156431346305467400271_wp
        Xi(13) = 787.164708503534546246704906025116_wp
        Xi(14) = 2762.028952486066240679107863797981_wp

      case (27:31)
        ! ----- K = 07 , R = 7E3 (Best : R = 6373) -----

        RIni = 7.0e3_wp

        Omega(1) = 0.0019398222440315960905672225964360699813_wp
        Omega(2) = 0.0127449810742930615739735666740173325451_wp
        Omega(3) = 0.0521738008587097122689798991274079398295_wp
        Omega(4) = 0.1696321211629053736074648414677845664755_wp
        Omega(5) = 0.4783040507603607306979110608269323279274_wp
        Omega(6) = 1.2328765453712284030614657059921057680185_wp
        Omega(7) = 3.1933984156189349494676943397308832572890_wp

        Alpha(1) = 0.0006068004605425493834683378191301172944_wp
        Alpha(2) = 0.0066754145503086015556908419665838039236_wp
        Alpha(3) = 0.0353300853116583099772316650666326154351_wp
        Alpha(4) = 0.1363679064445000981291473973744388104024_wp
        Alpha(5) = 0.4369259718550706375703868156179510151560_wp
        Alpha(6) = 1.2378541669818243195532250777723959345167_wp
        Alpha(7) = 3.2781922981409610852194386199087716704526_wp

        Xi(1) = 1.025170524052779595907541887456_wp
        Xi(2) = 1.236379131356275595647088716422_wp
        Xi(3) = 1.712369980532019143231250990489_wp
        Xi(4) = 2.572556450402184307106254190778_wp
        Xi(5) = 4.049438838950574166830609357248_wp
        Xi(6) = 6.592655571675112157956077929555_wp
        Xi(7) = 11.075549856600157824781938131764_wp
        Xi(8) = 19.256364499899104640590064008165_wp
        Xi(9) = 34.876717865489795487865531953275_wp
        Xi(10) = 66.466108155508556797064212418036_wp
        Xi(11) = 135.529643829296833873465644160206_wp
        Xi(12) = 303.787584679777685398027742280647_wp
        Xi(13) = 788.671587835625583551291128969751_wp
        Xi(14) = 2772.466801786170626620275925233727_wp

    end select

  case (8)
    ! ===== K = 08 =====

    select case (InitR)

      case default !(1:3)
        ! ----- K = 08 , R = 1E1 -----

        RIni = 1.0e1_wp

        Omega(1) = 0.1060264691056738337355419034280235024426_wp
        Omega(2) = 0.2585290566968715258728672007348947659011_wp
        Omega(3) = 0.4435891050076717719745974383149800246429_wp
        Omega(4) = 0.6929754149010114514916497396868066971365_wp
        Omega(5) = 1.0604512558549935925984730733517835687962_wp
        Omega(6) = 1.6406263617959378568180936475151554532204_wp
        Omega(7) = 2.6364875094068331538854632611901251948439_wp
        Omega(8) = 4.7875207637643221363424328451685596519383_wp

        Alpha(1) = 0.0410961677249340287632430913922210180544_wp
        Alpha(2) = 0.2215921003939924626704397608856211832062_wp
        Alpha(3) = 0.5688643520073686326546553992944410538257_wp
        Alpha(4) = 1.1299764765876945176933943137953519908478_wp
        Alpha(5) = 1.9937655342879093120283737206044349932199_wp
        Alpha(6) = 3.3208105201716792307872760936504619166953_wp
        Alpha(7) = 5.4098231210814420621448006532006047564209_wp
        Alpha(8) = 8.9485547791067179199120396049238479463384_wp

        Xi(1) = 1.006032718863063354138714933228_wp
        Xi(2) = 1.054756906560646363473984776782_wp
        Xi(3) = 1.154677820077761327588646012021_wp
        Xi(4) = 1.310785006622422077461782852215_wp
        Xi(5) = 1.530719648521820868422929629027_wp
        Xi(6) = 1.825017031800220838346458795964_wp
        Xi(7) = 2.207357900716805350046015310461_wp
        Xi(8) = 2.694552124162489622127489297121_wp
        Xi(9) = 3.305727151173377462937269521959_wp
        Xi(10) = 4.059779988418272830916905258292_wp
        Xi(11) = 4.969437980590544347292425442220_wp
        Xi(12) = 6.029429134977404547077184382431_wp
        Xi(13) = 7.196510027041024338753660849477_wp
        Xi(14) = 8.364013108639125808861647382031_wp
        Xi(15) = 9.348328430558432060662465978851_wp
        Xi(16) = 9.923503335592442870655149889103_wp

      case (4)
        ! ----- K = 08 , R = 2E1 -----

        RIni = 2.0e1_wp

        Omega(1) = 0.0643091646223874215151082057995157725827_wp
        Omega(2) = 0.1604657999272340102763292901677516510972_wp
        Omega(3) = 0.2880804447393661645070508936505859765020_wp
        Omega(4) = 0.4816504189697576558540966645827197112339_wp
        Omega(5) = 0.7991696608085075711298481127009551983065_wp
        Omega(6) = 1.3381266056699614649949658939931396162137_wp
        Omega(7) = 2.3014174412873261213723780072015756559267_wp
        Omega(8) = 4.4140627077983801873801994464230347148259_wp

        Alpha(1) = 0.0248626641496695616997454235039266556839_wp
        Alpha(2) = 0.1356048286078502581323201522933707963148_wp
        Alpha(3) = 0.3561108775384803358553861135638030077644_wp
        Alpha(4) = 0.7334936098496531953704578865593788350452_wp
        Alpha(5) = 1.3603355122458833637542896521921420571744_wp
        Alpha(6) = 2.4047037240593269399879794434582436224446_wp
        Alpha(7) = 4.1743696392540901437748224189761003799504_wp
        Alpha(8) = 7.3612293134220797129890767607207635592204_wp

        Xi(1) = 1.008169052232099786869934376909_wp
        Xi(2) = 1.074413773288232067234723721150_wp
        Xi(3) = 1.211692604293373886530441052578_wp
        Xi(4) = 1.429797599193880932363226454740_wp
        Xi(5) = 1.744176486994071265720224195306_wp
        Xi(6) = 2.177223636374015249038763042400_wp
        Xi(7) = 2.760158261984511549134296837238_wp
        Xi(8) = 3.535169509939452733978879073184_wp
        Xi(9) = 4.557231078278236897829284179906_wp
        Xi(10) = 5.894169856949790875164096881633_wp
        Xi(11) = 7.621200059709250569910804484763_wp
        Xi(12) = 9.801003964955703424109800803166_wp
        Xi(13) = 12.432003598376196459342202249587_wp
        Xi(14) = 15.343339384741938492481838629544_wp
        Xi(15) = 18.054131542215802208942165840710_wp
        Xi(16) = 19.764723853693879527040899724000_wp

      case (5)
        ! ----- K = 08 , R = 3E1 -----

        RIni = 3.0e1_wp

        Omega(1) = 0.0476193831333583450525004984038579536332_wp
        Omega(2) = 0.1209066723887145041542424094005436785437_wp
        Omega(3) = 0.2245974893577788025821138223614426010499_wp
        Omega(4) = 0.3939685643048650259248535249190226181781_wp
        Omega(5) = 0.6878199173590028391955635000964974778981_wp
        Omega(6) = 1.2046845144004612635597536995923917402251_wp
        Omega(7) = 2.1494044055658569707632776779604455441586_wp
        Omega(8) = 4.2432459544114843929055180815623771195533_wp

        Alpha(1) = 0.0183751523863023500047613476363062012808_wp
        Alpha(2) = 0.1010910396253187920628411819445879871182_wp
        Alpha(3) = 0.2701415651193927010753279005506044541107_wp
        Alpha(4) = 0.5719241549530381922713216658049617535653_wp
        Alpha(5) = 1.0991792961155158183869651189468186203158_wp
        Alpha(6) = 2.0209412748621891147641355113151462319365_wp
        Alpha(7) = 3.6475369117856662377636506322531317891844_wp
        Alpha(8) = 6.6737616671756857531297435492945169244194_wp

        Xi(1) = 1.009453069531725518575074329597_wp
        Xi(2) = 1.086294465453709110424294903563_wp
        Xi(3) = 1.246517015355910598770235175881_wp
        Xi(4) = 1.503568330084838722401560651143_wp
        Xi(5) = 1.878971864677563542462418455514_wp
        Xi(6) = 2.404690851105510255355385274001_wp
        Xi(7) = 3.126710352324097630730137531430_wp
        Xi(8) = 4.109621855206218431471459906490_wp
        Xi(9) = 5.441960414658121477672614307863_wp
        Xi(10) = 7.241175263151872247384399639003_wp
        Xi(11) = 9.653682038183337798045724476026_wp
        Xi(12) = 12.836502406889897921413168369043_wp
        Xi(13) = 16.886690073653670472395016410871_wp
        Xi(14) = 21.653659797237005603387505203727_wp
        Xi(15) = 26.393779632799863620745695058645_wp
        Xi(16) = 29.554146068708933157292984006403_wp

      case (6)
        ! ----- K = 08 , R = 4E1 -----

        RIni = 4.0e1_wp

        Omega(1) = 0.0383807437367189014720077113590468442794_wp
        Omega(2) = 0.0988727723491150179601283029540148561409_wp
        Omega(3) = 0.1888951020665660617106661536834799619555_wp
        Omega(4) = 0.3438325244242168562481057336732703788584_wp
        Omega(5) = 0.6224537875740374240698948860028849594528_wp
        Omega(6) = 1.1241351163372292887559361629357113088190_wp
        Omega(7) = 2.0558777824169633277693464146196333786065_wp
        Omega(8) = 4.1376874677588956052270985352947718638461_wp

        Alpha(1) = 0.0147868844915852609137710857271919007161_wp
        Alpha(2) = 0.0819390877546529709812448902250814342096_wp
        Alpha(3) = 0.2221894844955850291406572113994677408755_wp
        Alpha(4) = 0.4811176249313395669314919605996649920598_wp
        Alpha(5) = 0.9506792305778743405048070158613882085774_wp
        Alpha(6) = 1.7993973054918034820215747005001105662814_wp
        Alpha(7) = 3.3386937535433484879054955030142082250677_wp
        Alpha(8) = 6.2657281268480889825471524812172674501198_wp

        Xi(1) = 1.010366740445417035334195465790_wp
        Xi(2) = 1.094780045401361425697756013520_wp
        Xi(3) = 1.271563130120029259440862789976_wp
        Xi(4) = 1.557138585674537680143304785307_wp
        Xi(5) = 1.978051847176626979450154986129_wp
        Xi(6) = 2.574362080522365413834742775556_wp
        Xi(7) = 3.404866893413130592279877695283_wp
        Xi(8) = 4.554125907078944354971355990003_wp
        Xi(9) = 6.141583216364771941834371782321_wp
        Xi(10) = 8.332339656152746481913851095413_wp
        Xi(11) = 11.345203586933031587521969463239_wp
        Xi(12) = 15.441954294911313443246225762806_wp
        Xi(13) = 20.849989559360751762220886895705_wp
        Xi(14) = 27.503663248036671212098935868084_wp
        Xi(15) = 34.457132831177792381405211585843_wp
        Xi(16) = 39.301898615146465043590495724857_wp

      case (7)
        ! ----- K = 08 , R = 5E1 -----

        RIni = 5.0e1_wp

        Omega(1) = 0.0324331424753547827576646931518888905543_wp
        Omega(2) = 0.0846168152459831565353576103560850896201_wp
        Omega(3) = 0.1656001458179835813551818504696377942764_wp
        Omega(4) = 0.3106025687325338813151581751625940341910_wp
        Omega(5) = 0.5781375786016099380419316910995064517920_wp
        Omega(6) = 1.0683503216704084423172363416121299906081_wp
        Omega(7) = 1.9902248502356169701336952027048710078816_wp
        Omega(8) = 4.0633908158171135622388492336654053360689_wp

        Alpha(1) = 0.0124783628250833338653203201037977798649_wp
        Alpha(2) = 0.0695851236918722079302017206881014743658_wp
        Alpha(3) = 0.1911241780349972787915197730557181898803_wp
        Alpha(4) = 0.4218794617526348468170786093489610379947_wp
        Alpha(5) = 0.8527570534377367394876338912990121343682_wp
        Alpha(6) = 1.6513816100170211501139277965322094132716_wp
        Alpha(7) = 3.1297492657062539102991172157430810329970_wp
        Alpha(8) = 5.9869906376636193414200737272778951592045_wp

        Xi(1) = 1.011072047923903786492520751050_wp
        Xi(2) = 1.101348923582572341653591785349_wp
        Xi(3) = 1.291052883774184086604179533264_wp
        Xi(4) = 1.599124392338631489031940124956_wp
        Xi(5) = 2.056410952563171109673187397782_wp
        Xi(6) = 2.710029243561897579692376858418_wp
        Xi(7) = 3.630148625993221634918398099678_wp
        Xi(8) = 4.919394629851516541427930429009_wp
        Xi(9) = 6.725824433335095013999915813940_wp
        Xi(10) = 9.259909167783606280532904264646_wp
        Xi(11) = 12.811960610130180616451911479459_wp
        Xi(12) = 17.752979007771977057461576521291_wp
        Xi(13) = 24.460071329031597824402166629909_wp
        Xi(14) = 33.001617176073367095068267218494_wp
        Xi(15) = 42.293955496997097776024077120383_wp
        Xi(16) = 49.013661838087606636288340311580_wp

      case (8)
        ! ----- K = 08 , R = 6E1 -----

        RIni = 6.0e1_wp

        Omega(1) = 0.0282493396917258093250667396606479186971_wp
        Omega(2) = 0.0745460391371821543519829780799490492882_wp
        Omega(3) = 0.1490178202155422004619722553542260357062_wp
        Omega(4) = 0.2866012944928036406517200818111490434603_wp
        Omega(5) = 0.5455078206488269725515580232944046201737_wp
        Omega(6) = 1.0265829920889139290117800240587087046151_wp
        Omega(7) = 1.9405634730900531873887607248185815933539_wp
        Omega(8) = 4.0070879826910095447729809325920768969809_wp

        Alpha(1) = 0.0108553510974204830253448880728051051392_wp
        Alpha(2) = 0.0608802488511118427343626023640066335929_wp
        Alpha(3) = 0.1691520259232484054237079240667895874140_wp
        Alpha(4) = 0.3797103414106192527887294330124490215894_wp
        Alpha(5) = 0.7823660354159745778388282411164311724860_wp
        Alpha(6) = 1.5437686299849278366395910477848474329221_wp
        Alpha(7) = 2.9762479203500894029289086661904661923472_wp
        Alpha(8) = 5.7806062770457643908497691320746980636613_wp

        Xi(1) = 1.011643782347894183711901894274_wp
        Xi(2) = 1.106685789782311103214042236065_wp
        Xi(3) = 1.306952921678010662580489420481_wp
        Xi(4) = 1.633572383294769909912666494556_wp
        Xi(5) = 2.121165649437425079499447422293_wp
        Xi(6) = 2.823126617453443450850292251175_wp
        Xi(7) = 3.819881754177334538249630191409_wp
        Xi(8) = 5.230598170176873665335876006921_wp
        Xi(9) = 7.230001575562373054145703932249_wp
        Xi(10) = 10.071751936538247290155168034431_wp
        Xi(11) = 14.116078363430355207576438747452_wp
        Xi(12) = 19.844920052593694769979548730987_wp
        Xi(13) = 27.797619290786595202050190955845_wp
        Xi(14) = 38.214659335640939443606844960755_wp
        Xi(15) = 49.936766005697233127696987864397_wp
        Xi(16) = 58.693184247146195215566466529822_wp

      case (9)
        ! ----- K = 08 , R = 7E1 -----

        RIni = 7.0e1_wp

        Omega(1) = 0.0251286437058089860943941662788092550329_wp
        Omega(2) = 0.0670064982182152067786084191636231111033_wp
        Omega(3) = 0.1365152448171116218549048237917986625689_wp
        Omega(4) = 0.2682605973514779726699867185013559378604_wp
        Omega(5) = 0.5201604752923396644327148907738234129283_wp
        Omega(6) = 0.9936935209922501315103066099609918637725_wp
        Omega(7) = 1.9011380241938253567429012447220770809508_wp
        Omega(8) = 3.9623285503691931192307706943722678261111_wp

        Alpha(1) = 0.0096453351395891204763473249922389740618_wp
        Alpha(2) = 0.0543778231378707960550324476689532104956_wp
        Alpha(3) = 0.1526831785356098382909484367786490111030_wp
        Alpha(4) = 0.3479126051459739386872722227694865182457_wp
        Alpha(5) = 0.7288141658794775911475033458586381129862_wp
        Alpha(6) = 1.4610854083998511351470336872004907036171_wp
        Alpha(7) = 2.8572596946046264153838689647812998373411_wp
        Alpha(8) = 5.6195795452967103279437455221057007292984_wp

        Xi(1) = 1.012122710671435961058382824174_wp
        Xi(2) = 1.111164724011816449752466928746_wp
        Xi(3) = 1.320342732736776704904023416187_wp
        Xi(4) = 1.662718595953324588160131269898_wp
        Xi(5) = 2.176281095847154413739851785969_wp
        Xi(6) = 2.920088909828480301655861950394_wp
        Xi(7) = 3.983932103718470257424505431842_wp
        Xi(8) = 5.502265284245295174150869321750_wp
        Xi(9) = 7.674822924632206803171458053825_wp
        Xi(10) = 10.796457643348444377373596125835_wp
        Xi(11) = 15.295496764972239377849971830869_wp
        Xi(12) = 21.765117983615827092214178861695_wp
        Xi(13) = 30.915267742492391381606275935212_wp
        Xi(14) = 43.188723957921283339023155178893_wp
        Xi(15) = 57.408826043273802532812544185958_wp
        Xi(16) = 68.343169229978600588693016248953_wp

      case (10)
        ! ----- K = 08 , R = 8E1 -----

        RIni = 8.0e1_wp

        Omega(1) = 0.0227017559634129721878547086139077393341_wp
        Omega(2) = 0.0611239009072996963401575801905218909837_wp
        Omega(3) = 0.1266953565686958380293123532789323348879_wp
        Omega(4) = 0.2536760883721414111286341502848884488230_wp
        Omega(5) = 0.4997158187726234234262486649802426086353_wp
        Omega(6) = 0.9668633389980441755740502018934279249152_wp
        Omega(7) = 1.8687591398468625841821147415267034830322_wp
        Omega(8) = 3.9255289517027899676149260699276055675000_wp

        Alpha(1) = 0.0087047417233341318281537391313151807992_wp
        Alpha(2) = 0.0493145028931534314920142984817186260216_wp
        Alpha(3) = 0.1398190631110847891754086352866792708483_wp
        Alpha(4) = 0.3229342398078029359443068962987055670055_wp
        Alpha(5) = 0.6864049880172785394680246007226287474623_wp
        Alpha(6) = 1.3950320723685946309967562117826389567199_wp
        Alpha(7) = 2.7614744811077356516967190591671510446758_wp
        Alpha(8) = 5.4892323987691283215459292499360799411079_wp

        Xi(1) = 1.012533502490892778542287688825_wp
        Xi(2) = 1.115012603425935855958398001953_wp
        Xi(3) = 1.331879579513728790161365689304_wp
        Xi(4) = 1.687931917130258218902663647221_wp
        Xi(5) = 2.224201391792045462945937028820_wp
        Xi(6) = 3.004915982444246826772379921344_wp
        Xi(7) = 4.128492671460314285096504516304_wp
        Xi(8) = 5.743620554542309518050724470584_wp
        Xi(9) = 8.073607998075502993617913460866_wp
        Xi(10) = 11.452690616934524448812526831176_wp
        Xi(11) = 16.375430771618280048618210287259_wp
        Xi(12) = 23.545756770140535349411514687290_wp
        Xi(13) = 33.849897118730002693737235475169_wp
        Xi(14) = 47.957291188064149884545672719582_wp
        Xi(15) = 64.727769266696378293313163254652_wp
        Xi(16) = 77.965680890470509610545857270836_wp

      case (11)
        ! ----- K = 08 , R = 9E1 -----

        RIni = 9.0e1_wp

        Omega(1) = 0.0207545161695637046829612587170443838147_wp
        Omega(2) = 0.0563898985798704578030699288704319371845_wp
        Omega(3) = 0.1187430252961514762452565653616520791047_wp
        Omega(4) = 0.2417297705831963939448634934303861143690_wp
        Omega(5) = 0.4827599837096196503359330637339397185315_wp
        Omega(6) = 0.9443959761885041913733254692608909408591_wp
        Omega(7) = 1.8414905485795193190246346603977656286588_wp
        Omega(8) = 3.8945094081684952511731440116804492390656_wp

        Alpha(1) = 0.0079503396990795410480889018368932941172_wp
        Alpha(2) = 0.0452471530544417981952029159109507361336_wp
        Alpha(3) = 0.1294552708085345075057727079470559772290_wp
        Alpha(4) = 0.3027034338865617590615467169534724689584_wp
        Alpha(5) = 0.6518003973409469030885973750422124339821_wp
        Alpha(6) = 1.3407130798147101330810085362266192987590_wp
        Alpha(7) = 2.6821760146617595517177373709571952531405_wp
        Alpha(8) = 5.3808006207281632595450238110146301551140_wp

        Xi(1) = 1.012892217775423017332367403576_wp
        Xi(2) = 1.118377380194879730447750365574_wp
        Xi(3) = 1.341993546337162129746152505749_wp
        Xi(4) = 1.710112273452325181224975558791_wp
        Xi(5) = 2.266542972591529318338379739295_wp
        Xi(6) = 3.080271879844941867532606694091_wp
        Xi(7) = 4.257723429170776089855432466891_wp
        Xi(8) = 5.960920914601750644681160684613_wp
        Xi(9) = 8.435492033735638570633719179881_wp
        Xi(10) = 12.053415046951222512963686206433_wp
        Xi(11) = 17.373660061890040428450587839393_wp
        Xi(12) = 25.209960095336718662734076801257_wp
        Xi(13) = 36.628681045053516098647383714138_wp
        Xi(14) = 52.545874290794805278742574117246_wp
        Xi(15) = 71.907509186043684815126120213336_wp
        Xi(16) = 87.562358890709824886522394393751_wp

      case (12)
        ! ----- K = 08 , R = 1E2 -----

        RIni = 1.0e2_wp

        Omega(1) = 0.0191536972157081944805691103503320071866_wp
        Omega(2) = 0.0524874991701344780364337210681879852814_wp
        Omega(3) = 0.1121483176832860124096298039264585355568_wp
        Omega(4) = 0.2317178192093658137794016244059847053904_wp
        Omega(5) = 0.4683927366669551456933306976537778609782_wp
        Omega(6) = 0.9251988674368323040423864322967517637153_wp
        Omega(7) = 1.8180761380444023894405528030127072725008_wp
        Omega(8) = 3.8678537676116576005547892380675989443262_wp

        Alpha(1) = 0.0073303678892273275250013949040639449350_wp
        Alpha(2) = 0.0418998419489961924599388982948244830595_wp
        Alpha(3) = 0.1209028306256034907491844591909124773110_wp
        Alpha(4) = 0.2859242916917437047036147645950876494680_wp
        Alpha(5) = 0.6229028130797220469665881337206769785553_wp
        Alpha(6) = 1.2950331295281306881374966399889103740861_wp
        Alpha(7) = 2.6150915210491833956529816873626259621233_wp
        Alpha(8) = 5.2886811792799689560008735700336046647863_wp

        Xi(1) = 1.013209889188098188333303495412_wp
        Xi(2) = 1.121360848910737284967603644148_wp
        Xi(3) = 1.350981373984641096446315811086_wp
        Xi(4) = 1.729883192048452760121446392105_wp
        Xi(5) = 2.304431601657352488878027774177_wp
        Xi(6) = 3.148024141512270462732459019861_wp
        Xi(7) = 4.374562374757779956418318745293_wp
        Xi(8) = 6.158623105037639874209665080684_wp
        Xi(9) = 8.767045170776013054275699332152_wp
        Xi(10) = 12.608057826417582214517665306630_wp
        Xi(11) = 18.303279924216017092442054803314_wp
        Xi(12) = 26.775037004656948386172032883223_wp
        Xi(13) = 39.272402299385986434304696146569_wp
        Xi(14) = 56.974567404222613137970698815593_wp
        Xi(15) = 78.959365725034720430441925742571_wp
        Xi(16) = 97.134544711358374660670200739787_wp

      case (13)
        ! ----- K = 08 , R = 2E2 -----

        RIni = 2.0e2_wp

        Omega(1) = 0.0112997080726952972012938495108280712032_wp
        Omega(2) = 0.0331538459933014433800801089302634494516_wp
        Omega(3) = 0.0786965757234113262867832017111169307100_wp
        Omega(4) = 0.1789636493609061990991233523362602397810_wp
        Omega(5) = 0.3899113717033011928972483645106095195842_wp
        Omega(6) = 0.8174921432572277218628342687622989615193_wp
        Omega(7) = 1.6846079041638185670769559121318081906793_wp
        Omega(8) = 3.7155370264539169717667166370489439941593_wp

        Alpha(1) = 0.0042923690505157349716684159375568174966_wp
        Alpha(2) = 0.0254154224653902906883522856348878171673_wp
        Alpha(3) = 0.0783393362576383196028756081275012945753_wp
        Alpha(4) = 0.2007925992125950392100863436950497487032_wp
        Alpha(5) = 0.4726395102313897322930495714654597350091_wp
        Alpha(6) = 1.0517208224751955088736854460051972637302_wp
        Alpha(7) = 2.2505781328076893770505362279976679928950_wp
        Alpha(8) = 4.7811047016943063726022999926357215372263_wp

        Xi(1) = 1.015206143343586549059043289134_wp
        Xi(2) = 1.140189268684429005460030659425_wp
        Xi(3) = 1.408141644285239425229624976588_wp
        Xi(4) = 1.856952607098087805991004639594_wp
        Xi(5) = 2.551250894168072993421075844722_wp
        Xi(6) = 3.596788903236659127758342857639_wp
        Xi(7) = 5.163755900602811575207662375675_wp
        Xi(8) = 7.524095349419460813152754630195_wp
        Xi(9) = 11.114903730582899794333828769055_wp
        Xi(10) = 16.646935196204675398712335976370_wp
        Xi(11) = 25.288912071546789949588740498143_wp
        Xi(12) = 38.968652370577076770458146981468_wp
        Xi(13) = 60.779458366592280044171658914820_wp
        Xi(14) = 95.031428293897582691707182078744_wp
        Xi(15) = 143.996271130142976066301763182764_wp
        Xi(16) = 191.696182955146930640411007118473_wp

      case (14)
        ! ----- K = 08 , R = 3E2 -----

        RIni = 3.0e2_wp

        Omega(1) = 0.0083147700732372714089635679264589374071_wp
        Omega(2) = 0.0256810205584535582997796412940694210647_wp
        Omega(3) = 0.0651854465211138051674568880122784264586_wp
        Omega(4) = 0.1563410800780707693744897915366642848767_wp
        Omega(5) = 0.3545343631542285954061973635154103590139_wp
        Omega(6) = 0.7671776007943373896016958646271177713061_wp
        Omega(7) = 1.6209236463502489251084850985229479647387_wp
        Omega(8) = 3.6426182431755443347221046224149176850915_wp

        Alpha(1) = 0.0031400004160420679633702276687193255533_wp
        Alpha(2) = 0.0191108173471338873873208484902019854701_wp
        Alpha(3) = 0.0617360444099322960463064463576543516865_wp
        Alpha(4) = 0.1664281836158163361329706889102375555467_wp
        Alpha(5) = 0.4095542480247392515540095364601214100730_wp
        Alpha(6) = 0.9458455055554492088413544303993063522285_wp
        Alpha(7) = 2.0873834230907897340206308411936220181815_wp
        Alpha(8) = 4.5494170928592982520478893437854139847332_wp

        Xi(1) = 1.016281976238305648414274318103_wp
        Xi(2) = 1.150394802721959229985931871809_wp
        Xi(3) = 1.439444124816771421095790151057_wp
        Xi(4) = 1.927518194641683521887640107018_wp
        Xi(5) = 2.690794755644557892425641765222_wp
        Xi(6) = 3.856167060340970678287628170544_wp
        Xi(7) = 5.631827990273911388723754312124_wp
        Xi(8) = 8.358023927568324704627522603317_wp
        Xi(9) = 12.596532565686797791246198663373_wp
        Xi(10) = 19.290531183175411522254250051844_wp
        Xi(11) = 30.052574545636373200968582963810_wp
        Xi(12) = 47.682345416800974432364190747080_wp
        Xi(13) = 77.024725923281626246974518323896_wp
        Xi(14) = 125.858082461702271405423747552277_wp
        Xi(15) = 201.808154923244766287826479356227_wp
        Xi(16) = 284.486122795369502369577929812294_wp

      case (15)
        ! ----- K = 08 , R = 4E2 -----

        RIni = 4.0e2_wp

        Omega(1) = 0.0067008875889055753375121758999566345949_wp
        Omega(2) = 0.0215908496440356932643997141599290223724_wp
        Omega(3) = 0.0575423155787227755035507983233378226373_wp
        Omega(4) = 0.1430624602634638181891755387420417378053_wp
        Omega(5) = 0.3331842059183278898014217428436012369275_wp
        Omega(6) = 0.7362076092535837336188440926942888609119_wp
        Omega(7) = 1.5812540713021125379821388290402239817922_wp
        Omega(8) = 3.5971080384364662310345273077416550222551_wp

        Alpha(1) = 0.0025176525567014111136737313000402704866_wp
        Alpha(2) = 0.0156870855266922170677206492539479398829_wp
        Alpha(3) = 0.0525805310064843255993228491457225359795_wp
        Alpha(4) = 0.1470151831911213632464539152744364969294_wp
        Alpha(5) = 0.3730161746691587561534312472710794850173_wp
        Alpha(6) = 0.8831960386953907907940479471431416413907_wp
        Alpha(7) = 1.9892072955102488654183928717955609499768_wp
        Alpha(8) = 4.4084930703152622123892900862784927085158_wp

        Xi(1) = 1.016997180607474292844683017023_wp
        Xi(2) = 1.157202679675031171914120764477_wp
        Xi(3) = 1.460451278101122088329093251513_wp
        Xi(4) = 1.975265489965839091498177315653_wp
        Xi(5) = 2.786220940640250699543364887312_wp
        Xi(6) = 4.035887149943814944733078142747_wp
        Xi(7) = 5.961190076668444562304344236381_wp
        Xi(8) = 8.955253699668341887431266545860_wp
        Xi(9) = 13.678943432092563078222857697597_wp
        Xi(10) = 21.265716999283591331568188387280_wp
        Xi(11) = 33.703880926196548187961932185885_wp
        Xi(12) = 54.561382722672526549012594898613_wp
        Xi(13) = 90.310496138781121207295132791160_wp
        Xi(14) = 152.230375307364352008288044260098_wp
        Xi(15) = 254.493575469728197963048543783771_wp
        Xi(16) = 375.770376438034457905468244121039_wp

      case (16)
        ! ----- K = 08 , R = 5E2 -----

        RIni = 5.0e2_wp

        Omega(1) = 0.0056765954727616951858215415486347277252_wp
        Omega(2) = 0.0189684285531276371907465202668041648337_wp
        Omega(3) = 0.0525102584623810031392329537858199017819_wp
        Omega(4) = 0.1340894633572563902608812402839078004035_wp
        Omega(5) = 0.3184836575206165450461961302641711313299_wp
        Omega(6) = 0.7145967548360081613746663808939274531440_wp
        Omega(7) = 1.5533459981559690892558359132102907551598_wp
        Omega(8) = 3.5650469439391760267422937236148072770447_wp

        Alpha(1) = 0.0021229535274735288473389613556410737694_wp
        Alpha(2) = 0.0135064199433631837297302771827567369201_wp
        Alpha(3) = 0.0466737186745842354107353684017356343361_wp
        Alpha(4) = 0.1342539621944193918437818191513599686004_wp
        Alpha(5) = 0.3485587576491086894984929717145050176441_wp
        Alpha(6) = 0.8406244096896624128523872743201650337141_wp
        Alpha(7) = 1.9217275963995139784722121500948333050474_wp
        Alpha(8) = 4.3108989561714496580382971480105425143847_wp

        Xi(1) = 1.017522457222688383104633458309_wp
        Xi(2) = 1.162214480881127272250859239211_wp
        Xi(3) = 1.475981656325268065888672852637_wp
        Xi(4) = 2.010766171804123481721068844585_wp
        Xi(5) = 2.857699018743528042614532314047_wp
        Xi(6) = 4.171748635930965416057647487946_wp
        Xi(7) = 6.212887708931987642158728535335_wp
        Xi(8) = 9.417362631740984755743251177762_wp
        Xi(9) = 14.528380793754103720953618772427_wp
        Xi(10) = 22.840921073627481002743522253695_wp
        Xi(11) = 36.670066045180380594897817658762_wp
        Xi(12) = 60.271227894864704381638897956464_wp
        Xi(13) = 101.628326632819418605391259546877_wp
        Xi(14) = 175.457508957763793452921952109591_wp
        Xi(15) = 303.185783856004199443567870275729_wp
        Xi(16) = 465.696952975465335022864366010253_wp

      case (17)
        ! ----- K = 08 , R = 6E2 -----

        RIni = 6.0e2_wp

        Omega(1) = 0.0049631227560872719340512640700396884696_wp
        Omega(2) = 0.0171254524636490031161900138637066426384_wp
        Omega(3) = 0.0488948450803400766644427576090858877933_wp
        Omega(4) = 0.1275139287006728620876838675712861004286_wp
        Omega(5) = 0.3075600247602412360640317112725483639224_wp
        Omega(6) = 0.6983780043252394544375794005475199810462_wp
        Omega(7) = 1.5322730134232326378723731963304999226239_wp
        Omega(8) = 3.5408125259521600184699219315831442145281_wp

        Alpha(1) = 0.0018481578123667529228029920789454221275_wp
        Alpha(2) = 0.0119828198121361834915281287423927114588_wp
        Alpha(3) = 0.0425003466075095890485270166275766356989_wp
        Alpha(4) = 0.1250994074781486592218325906711307027308_wp
        Alpha(5) = 0.3307649289127311088907872071995797114141_wp
        Alpha(6) = 0.8092949962970130067330468137676291462412_wp
        Alpha(7) = 1.8716385878240684829953988099759953911416_wp
        Alpha(8) = 4.2380476718029878352607398017681816781987_wp

        Xi(1) = 1.017931737489983994677730794187_wp
        Xi(2) = 1.166126687088432865944259697599_wp
        Xi(3) = 1.488142831743968411421959185947_wp
        Xi(4) = 2.038685505878753312109658013540_wp
        Xi(5) = 2.914229161373346585223870208736_wp
        Xi(6) = 4.279952233877284060482859562313_wp
        Xi(7) = 6.415003888346393525442257210756_wp
        Xi(8) = 9.791979041278517329784103484513_wp
        Xi(9) = 15.224499415759320633187434701483_wp
        Xi(10) = 24.147941963990778684034932410896_wp
        Xi(11) = 39.166863288595609133430075488036_wp
        Xi(12) = 65.159262885039493602701288210710_wp
        Xi(13) = 111.517879335856661342230022881949_wp
        Xi(14) = 196.296994258655902676480309310136_wp
        Xi(15) = 348.607063705276868759153074961432_wp
        Xi(16) = 554.362393276937058528286428327192_wp

      case (18)
        ! ----- K = 08 , R = 7E2 -----

        RIni = 7.0e2_wp

        Omega(1) = 0.0044348751043990017105410794857889289489_wp
        Omega(2) = 0.0157499318410671428728766184368392799797_wp
        Omega(3) = 0.0461450866606818998462370608887317402491_wp
        Omega(4) = 0.1224336447000104455669561645536624894248_wp
        Omega(5) = 0.2990278393358638452592061091062092259563_wp
        Omega(6) = 0.6856106320477458669995253492945863627028_wp
        Omega(7) = 1.5156041931287733760033015206758477688709_wp
        Omega(8) = 3.5216267135009914567532474105604478609166_wp

        Alpha(1) = 0.0016447651044251732014945396810785827046_wp
        Alpha(2) = 0.0108516614681113453641541537544878970323_wp
        Alpha(3) = 0.0393710657262846921716368999571988851471_wp
        Alpha(4) = 0.1181468832597833753259367051136052850779_wp
        Alpha(5) = 0.3170960230793336978930008207555601984495_wp
        Alpha(6) = 0.7850072328236101285546533201653573996737_wp
        Alpha(7) = 1.8325419240849563618477036097687005167245_wp
        Alpha(8) = 4.1809312247825503000391023089576947313617_wp

        Xi(1) = 1.018263452788968317116839734648_wp
        Xi(2) = 1.169302024710275848297380552054_wp
        Xi(3) = 1.498038435178118336628221018980_wp
        Xi(4) = 2.061481544340214850383544153978_wp
        Xi(5) = 2.960592752162711259640570649054_wp
        Xi(6) = 4.369192178959472600517643936513_wp
        Xi(7) = 6.582798084819636397473779920730_wp
        Xi(8) = 10.105353697117153155722590884835_wp
        Xi(9) = 15.811915705435575898833711061542_wp
        Xi(10) = 25.261995455184047860638929527966_wp
        Xi(11) = 41.320061386100109580149508659019_wp
        Xi(12) = 69.433193047246356169321934714844_wp
        Xi(13) = 120.312329102978800024481564889811_wp
        Xi(14) = 215.238606077263635044105605231834_wp
        Xi(15) = 391.261408758174212108027489875894_wp
        Xi(16) = 641.835825018210841652610554319836_wp

      case (19)
        ! ----- K = 08 , R = 8E2 -----

        RIni = 8.0e2_wp

        Omega(1) = 0.0040264999918426540221549566906517059195_wp
        Omega(2) = 0.0146786769119747113790630218058130829206_wp
        Omega(3) = 0.0439682178014438697386619039203514702763_wp
        Omega(4) = 0.1183595924121215987276671197103716792753_wp
        Omega(5) = 0.2921245499269274579492387117207741198399_wp
        Omega(6) = 0.6752146704956449841349142015900497426628_wp
        Omega(7) = 1.5019777460020226257262840330319875192799_wp
        Omega(8) = 3.5059316086191087472306165961910551231995_wp

        Alpha(1) = 0.0014875565232259687857860485271817296837_wp
        Alpha(2) = 0.0099749829069579409847811583139653990315_wp
        Alpha(3) = 0.0369240609001513219396858715704023978788_wp
        Alpha(4) = 0.1126502931174692987160968447868381048238_wp
        Alpha(5) = 0.3061859528751703799301104452768029773324_wp
        Alpha(6) = 0.7654746988259726943760324902044089867559_wp
        Alpha(7) = 1.8009234408139323254478478264495322491712_wp
        Alpha(8) = 4.1345719764847616165340249416715323604876_wp

        Xi(1) = 1.018540004076203246089621368142_wp
        Xi(2) = 1.171952366106674925231613537679_wp
        Xi(3) = 1.506315092686870306631745364268_wp
        Xi(4) = 2.080601695298627345403000288293_wp
        Xi(5) = 2.999624216167487557359663807866_wp
        Xi(6) = 4.444665879532511173176378971306_wp
        Xi(7) = 6.725481320767191097381026976265_wp
        Xi(8) = 10.373512238455663971720133176291_wp
        Xi(9) = 16.318228897540168238419755475377_wp
        Xi(10) = 26.230309144769280512918263603694_wp
        Xi(11) = 43.210003744964286895996474768822_wp
        Xi(12) = 73.228488043359466513892996886170_wp
        Xi(13) = 128.234621569760617182387463230953_wp
        Xi(14) = 232.622632419814380777545181899768_wp
        Xi(15) = 431.521930984642145351370956518622_wp
        Xi(16) = 728.170044478462002657259688476188_wp

      case (20)
        ! ----- K = 08 , R = 9E2 -----

        RIni = 9.0e2_wp

        Omega(1) = 0.0037004598585148988313034407312759888065_wp
        Omega(2) = 0.0138175172883332139873169817391451719146_wp
        Omega(3) = 0.0421928680892260580071807026664165718444_wp
        Omega(4) = 0.1150007485811967754960245292894516211391_wp
        Omega(5) = 0.2863906977665336270528519124756172686830_wp
        Omega(6) = 0.6665335644303945399156280382513273252698_wp
        Omega(7) = 1.4905611878984845606243778148858325494075_wp
        Omega(8) = 3.4927739456075436869086914271065325010568_wp

        Alpha(1) = 0.0013620554429901594556538322548094877007_wp
        Alpha(2) = 0.0092734079691412496733424783261490187414_wp
        Alpha(3) = 0.0349498666613168215373320165534343217928_wp
        Alpha(4) = 0.1081731334469284820388448528638569712257_wp
        Alpha(5) = 0.2972266643940104389803522066948460178537_wp
        Alpha(6) = 0.7493320377090901402957270893701036129642_wp
        Alpha(7) = 1.7746689959691647210096240860899285962660_wp
        Alpha(8) = 4.0959603100621581276558069717452781333122_wp

        Xi(1) = 1.018775508491156873958592699037_wp
        Xi(2) = 1.174211612730809165200743238167_wp
        Xi(3) = 1.513382632847937119707250330780_wp
        Xi(4) = 2.096967615958879766228581398302_wp
        Xi(5) = 3.033136975610514185800956066252_wp
        Xi(6) = 4.509721095769690817067765209458_wp
        Xi(7) = 6.849034886445259618162340053971_wp
        Xi(8) = 10.606960214946192893638310383864_wp
        Xi(9) = 16.761727614323374801683086410264_wp
        Xi(10) = 27.084571457613766028507673588877_wp
        Xi(11) = 44.891417230264682947105114863007_wp
        Xi(12) = 76.639019392290796128808239018326_wp
        Xi(13) = 135.442688929812149994935666086349_wp
        Xi(14) = 248.697760513201823903695064643671_wp
        Xi(15) = 469.676669712967440134532282058899_wp
        Xi(16) = 813.407470316859758374672395575544_wp

      case (21)
        ! ----- K = 08 , R = 1E3 -----

        RIni = 1.0e3_wp

        Omega(1) = 0.0034335760503910689907179870465625537435_wp
        Omega(2) = 0.0131080577194026246206680157667390229648_wp
        Omega(3) = 0.0407113742175260309220409526975714697983_wp
        Omega(4) = 0.1121716653092656144037673852931202844729_wp
        Omega(5) = 0.2815304358415968095431282380847193280715_wp
        Omega(6) = 0.6591413831361904632319577423960055284624_wp
        Omega(7) = 1.4808119960150753685384703017824392645707_wp
        Omega(8) = 3.4815321032916017986516554749698570958572_wp

        Alpha(1) = 0.0012593274957499998663477814415077860044_wp
        Alpha(2) = 0.0086978504539733445175800526094622711071_wp
        Alpha(3) = 0.0333181980142722798886561596382049899034_wp
        Alpha(4) = 0.1044413334133183721434603302702370797306_wp
        Alpha(5) = 0.2897059260448647628944686405194630651749_wp
        Alpha(6) = 0.7357066637534231497890305218056994362996_wp
        Alpha(7) = 1.7524189550363361240819734532259133175103_wp
        Alpha(8) = 4.0631526124948221172054896310754656951758_wp

        Xi(1) = 1.018979407447829179018429501813_wp
        Xi(2) = 1.176169329208864889084115712414_wp
        Xi(3) = 1.519516377580273219349143998258_wp
        Xi(4) = 2.111199937117118880566118965803_wp
        Xi(5) = 3.062359396681441598125211767112_wp
        Xi(6) = 4.566636996217601827171422845808_wp
        Xi(7) = 6.957560060020085674660494934329_wp
        Xi(8) = 10.812960773503130699688123961266_wp
        Xi(9) = 17.155169793462104483455776815504_wp
        Xi(10) = 27.847134518765816187796380276609_wp
        Xi(11) = 46.403372070943336071685569876877_wp
        Xi(12) = 79.732837463404732533667207405870_wp
        Xi(13) = 142.053194706536473054803160209758_wp
        Xi(14) = 263.652774992788203028970528407626_wp
        Xi(15) = 505.955334415113799101382241474312_wp
        Xi(16) = 897.583498348762517882182265793745_wp

      case (22)
        ! ----- K = 08 , R = 2E3 -----

        RIni = 2.0e3_wp

        Omega(1) = 0.0021430462268822023309589085496296156919_wp
        Omega(2) = 0.0095934513264009831643493869668404272488_wp
        Omega(3) = 0.0330567547313259784287067452601682759905_wp
        Omega(4) = 0.0971319794098423857755587353113302384600_wp
        Omega(5) = 0.2551846773649667515662751299254651371484_wp
        Omega(6) = 0.6185015005689335353296354202967677338165_wp
        Omega(7) = 1.4267377848347798636021235840765086777537_wp
        Omega(8) = 3.4190751095195143275089905143815371957317_wp

        Alpha(1) = 0.0007623416133137365415047053069166893557_wp
        Alpha(2) = 0.0058908109757378770075222831209549312348_wp
        Alpha(3) = 0.0251495451866708728604304879869490996214_wp
        Alpha(4) = 0.0852376686243077630886558204714731346030_wp
        Alpha(5) = 0.2501340726547855392654547135311560168702_wp
        Alpha(6) = 0.6627794284196205648477043248512075024337_wp
        Alpha(7) = 1.6318321257986886528481423885850176702661_wp
        Alpha(8) = 3.8839165956082179677767013403766327428457_wp

        Xi(1) = 1.020159001593107734868863534228_wp
        Xi(2) = 1.187526285869552161852599758696_wp
        Xi(3) = 1.555265978199967590446355092393_wp
        Xi(4) = 2.194689107792471268213937984903_wp
        Xi(5) = 3.235249322883711310368595737152_wp
        Xi(6) = 4.906996158365866961097168097439_wp
        Xi(7) = 7.614850924736838379876940896551_wp
        Xi(8) = 12.079316566765041890754717179846_wp
        Xi(9) = 19.616275207518496383196637822266_wp
        Xi(10) = 32.715915456623407751163279399975_wp
        Xi(11) = 56.297129941781333100497830557174_wp
        Xi(12) = 100.593458953313436066157038339952_wp
        Xi(13) = 188.357478859669821333322481393679_wp
        Xi(14) = 373.907275932987620992165034294885_wp
        Xi(15) = 795.634013655747957927122371302175_wp
        Xi(16) = 1686.377993447082192846053771972947_wp

      case (23)
        ! ----- K = 08 , R = 3E3 -----

        RIni = 3.0e3_wp

        Omega(1) = 0.0016651967973811464223829755079871184353_wp
        Omega(2) = 0.0082344958996268037770835195553187357831_wp
        Omega(3) = 0.0299132583617116829806614709472145641200_wp
        Omega(4) = 0.0907228736405908252229783605979918092999_wp
        Omega(5) = 0.2436730155314357148480658432632139920315_wp
        Omega(6) = 0.6004207194957538095076224338164649907412_wp
        Omega(7) = 1.4024065780531140042045393223624216716416_wp
        Omega(8) = 3.3909109012617059081100279183118573200773_wp

        Alpha(1) = 0.0005779647722766572749244552188729873166_wp
        Alpha(2) = 0.0048345080412755526751872900501766361359_wp
        Alpha(3) = 0.0219435969317702805491434843790377051675_wp
        Alpha(4) = 0.0773991677369858432824351046619426597317_wp
        Alpha(5) = 0.2334900109680693839101731784646531764338_wp
        Alpha(6) = 0.6314114180649208010482481556735478989140_wp
        Alpha(7) = 1.5791206489394280554393670823465356534143_wp
        Alpha(8) = 3.8047625323126017483563177101757446507690_wp

        Xi(1) = 1.020717974317504311582843756767_wp
        Xi(2) = 1.192926375345656953897745256032_wp
        Xi(3) = 1.572365758565759568505214305301_wp
        Xi(4) = 2.234949192792201978991287192322_wp
        Xi(5) = 3.319519259832642186667725492377_wp
        Xi(6) = 5.075134836263528399172978433640_wp
        Xi(7) = 7.944772461828507571605928472280_wp
        Xi(8) = 12.726911915245122794947130184795_wp
        Xi(9) = 20.902626152924975227662129917405_wp
        Xi(10) = 35.327057585343022875351559619617_wp
        Xi(11) = 61.770511965615394315426422977566_wp
        Xi(12) = 112.580787376352591942774150624018_wp
        Xi(13) = 216.291139842679021737525957291837_wp
        Xi(14) = 444.932477510570292433822459088333_wp
        Xi(15) = 1002.025485610155657867359479951119_wp
        Xi(16) = 2390.559013053741090448767181442236_wp

      case (24)
        ! ----- K = 08 , R = 4E3 -----

        RIni = 4.0e3_wp

        Omega(1) = 0.0014125661829260472694481433082042132199_wp
        Omega(2) = 0.0074937692680186526238192543016708446757_wp
        Omega(3) = 0.0281426446017923193173519008647431860481_wp
        Omega(4) = 0.0870442922664944302666443012861385852830_wp
        Omega(5) = 0.2369808061228626872355123439417567965393_wp
        Omega(6) = 0.5898121473669076002743424935736982206436_wp
        Omega(7) = 1.3880475721468792164826552837375572835299_wp
        Omega(8) = 3.3742707589985145493014645490248426540347_wp

        Alpha(1) = 0.0004803056376499482843359545048171274639_wp
        Alpha(2) = 0.0042692087887265292081189805867974529363_wp
        Alpha(3) = 0.0201825342118137213973306414230290961598_wp
        Alpha(4) = 0.0729992859041493389140632663814045599793_wp
        Alpha(5) = 0.2239987472474404483303639032398812958036_wp
        Alpha(6) = 0.6133156517874289083034830549756577511289_wp
        Alpha(7) = 1.5484597079648389776791728156268845850718_wp
        Alpha(8) = 3.7584802839867816838439334992116869216261_wp

        Xi(1) = 1.021056476561818582402159372169_wp
        Xi(2) = 1.196202442602344475633055487229_wp
        Xi(3) = 1.582771745869115058219164793307_wp
        Xi(4) = 2.259551932099742987264390237101_wp
        Xi(5) = 3.371305780872138385369576885076_wp
        Xi(6) = 5.179188258767591076636022107138_wp
        Xi(7) = 8.150662048084573192002766361952_wp
        Xi(8) = 13.135051401130087644388011458574_wp
        Xi(9) = 21.722841000299467618256388234954_wp
        Xi(10) = 37.015307537045970150829665712422_wp
        Xi(11) = 65.370220308190230787404040313504_wp
        Xi(12) = 120.634159819952608179438602320488_wp
        Xi(13) = 235.588202886921525197605475909768_wp
        Xi(14) = 495.947551325105920855262908730765_wp
        Xi(15) = 1159.523072215187794564705825450801_wp
        Xi(16) = 3021.329814481966200334639438551676_wp

      case (25)
        ! ----- K = 08 , R = 5E3 -----

        RIni = 5.0e3_wp

        Omega(1) = 0.0012560128074441001169983661285393938822_wp
        Omega(2) = 0.0070239046575416292430630292066562336828_wp
        Omega(3) = 0.0269953790374489263546685510405942487466_wp
        Omega(4) = 0.0846323403990839879843205255383065122032_wp
        Omega(5) = 0.2325573562750163136630607296773121106526_wp
        Omega(6) = 0.5827590540484507476183539953495937879779_wp
        Omega(7) = 1.3784658861922865615099592639936076921003_wp
        Omega(8) = 3.3631587036004961274553159977784844159032_wp

        Alpha(1) = 0.0004196991604112789323126704230364908410_wp
        Alpha(2) = 0.0039154579369630628607557650162906970337_wp
        Alpha(3) = 0.0190601015881723848084195252483574734015_wp
        Alpha(4) = 0.0701548744514945170271547764118125201094_wp
        Alpha(5) = 0.2178004795859835697544992089147442015928_wp
        Alpha(6) = 0.6014111142968013818213036947302185808439_wp
        Alpha(7) = 1.5281833987584513796309862443401073051064_wp
        Alpha(8) = 3.7277730360685648396904673385066075752547_wp

        Xi(1) = 1.021286012308503826965210004030_wp
        Xi(2) = 1.198426449965374330683919035234_wp
        Xi(3) = 1.589849845991704522228139817575_wp
        Xi(4) = 2.276331635832671055957518446711_wp
        Xi(5) = 3.406750436624172942434718436289_wp
        Xi(6) = 5.250725632191437263970867910245_wp
        Xi(7) = 8.292971227012546952921112630719_wp
        Xi(8) = 13.418939586929014879009003369248_wp
        Xi(9) = 22.297650276840474209416598228017_wp
        Xi(10) = 38.209159167489258602384838425792_wp
        Xi(11) = 67.944318550299602946229704514280_wp
        Xi(12) = 126.474664437222455208587046371349_wp
        Xi(13) = 249.846997526762718458415157840591_wp
        Xi(14) = 534.665541448239452937407634181000_wp
        Xi(15) = 1284.180367206866845375401453566155_wp
        Xi(16) = 3585.705870058596230975922480865847_wp

      case (26)
        ! ----- K = 08 , R = 6E3 -----

        RIni = 6.0e3_wp

        Omega(1) = 0.0011500481004688015065847933138080960713_wp
        Omega(2) = 0.0066999330935280467300949091072337360231_wp
        Omega(3) = 0.0261924342913283282166704438523319176113_wp
        Omega(4) = 0.0829303042986406247909529285133878317993_wp
        Omega(5) = 0.2294182983616035149894473235687719636644_wp
        Omega(6) = 0.5777335833512779340176424014430267561693_wp
        Omega(7) = 1.3716213004158607933948704760140913094801_wp
        Omega(8) = 3.3552168348270158923681016904794205402141_wp

        Alpha(1) = 0.0003786335074948956111030892008294246320_wp
        Alpha(2) = 0.0036740934531544308959442678803528070652_wp
        Alpha(3) = 0.0182837045590645175482968388946947158047_wp
        Alpha(4) = 0.0681673475707708599458773367218977767834_wp
        Alpha(5) = 0.2134385607409870259064399444781656711712_wp
        Alpha(6) = 0.5929904812716678494540455124361955085988_wp
        Alpha(7) = 1.5137888616481399616442091171641948221804_wp
        Alpha(8) = 3.7059238120115058068553687276036612274766_wp

        Xi(1) = 1.021451800449782277646973427387_wp
        Xi(2) = 1.200034072905669535404234171683_wp
        Xi(3) = 1.594973221740001397320717069217_wp
        Xi(4) = 2.288500077504828713508347459715_wp
        Xi(5) = 3.432518210254069016863776031556_wp
        Xi(6) = 5.302895618439775006925512856881_wp
        Xi(7) = 8.397140788545276907615944050889_wp
        Xi(8) = 13.627664939714634267410586776492_wp
        Xi(9) = 22.722519618169650721176133778556_wp
        Xi(10) = 39.097212007429638102457003512313_wp
        Xi(11) = 69.874316127822566215022082047881_wp
        Xi(12) = 130.898155782560052906582370724209_wp
        Xi(13) = 260.793276038229136143398179115138_wp
        Xi(14) = 564.980498947044184665511323828468_wp
        Xi(15) = 1384.874279477222157042071160049090_wp
        Xi(16) = 4088.104869330484655742807831302343_wp

      case (27)
        ! ----- K = 08 , R = 7E3 -----

        RIni = 7.0e3_wp

        Omega(1) = 0.0010744332166565012706056802573029163028_wp
        Omega(2) = 0.0064652733490262209884080399618228041092_wp
        Omega(3) = 0.0256044015068254107644749777497605869314_wp
        Omega(4) = 0.0816762846342895764326064687865791569266_wp
        Omega(5) = 0.2270959749376227446724177405634748083685_wp
        Omega(6) = 0.5740045808031631187308072017394522390532_wp
        Omega(7) = 1.3665329240577759860059831042988776061975_wp
        Omega(8) = 3.3493104419351631469582014632990762947884_wp

        Alpha(1) = 0.0003493074543081077457771562854316887980_wp
        Alpha(2) = 0.0035007144790761561605025670379856550518_wp
        Alpha(3) = 0.0177200698247082988648451766164337861653_wp
        Alpha(4) = 0.0667135377029118788362463746133279585138_wp
        Alpha(5) = 0.2102311912125660162451541365657980975357_wp
        Alpha(6) = 0.5867753361711184133904170567852531803510_wp
        Alpha(7) = 1.5031361158199717780598048122442378371488_wp
        Alpha(8) = 3.6897272261912295819846069333536320300482_wp

        Xi(1) = 1.021576047090998875534716139324_wp
        Xi(2) = 1.201239584206932473216776491309_wp
        Xi(3) = 1.598818959507872945031113631753_wp
        Xi(4) = 2.297646168074588161629040472533_wp
        Xi(5) = 3.451922062136160265820652992552_wp
        Xi(6) = 5.342269967332008041328850200813_wp
        Xi(7) = 8.475977879712730296210554392644_wp
        Xi(8) = 13.786149094289558133218098756601_wp
        Xi(9) = 23.046374048145767864104183075824_wp
        Xi(10) = 39.777345296839743717109971754553_wp
        Xi(11) = 71.361214735857919258532611195278_wp
        Xi(12) = 134.331918860858340325492576994293_wp
        Xi(13) = 269.377559056215929556010735268501_wp
        Xi(14) = 589.114902585990282690708852442185_wp
        Xi(15) = 1466.989064506464314763256595597340_wp
        Xi(16) = 4531.215030129676081482870131367235_wp

      case (28)
        ! ----- K = 08 , R = 8E3 -----

        RIni = 8.0e3_wp

        Omega(1) = 0.0010187920024821150578778111207324530429_wp
        Omega(2) = 0.0062904921216730113587560122793279626663_wp
        Omega(3) = 0.0251627516939175241693924314767549788030_wp
        Omega(4) = 0.0807301242767707417244079683149582393753_wp
        Omega(5) = 0.2253383067673434065665170542014372756512_wp
        Omega(6) = 0.5711759383022554021399480594922692944237_wp
        Omega(7) = 1.3626676872789057304129201053655151554267_wp
        Omega(8) = 3.3448225248609738599614438214757683454081_wp

        Alpha(1) = 0.0003277166785761017410492756111355452942_wp
        Alpha(2) = 0.0033724318575932676636188492459364485221_wp
        Alpha(3) = 0.0172995667533701635395262377267733278075_wp
        Alpha(4) = 0.0656226259651636144700284636321896414302_wp
        Alpha(5) = 0.2078148268124083186713432075398699794277_wp
        Alpha(6) = 0.5820796197507841215966310766116720287755_wp
        Alpha(7) = 1.4950714931789085518788057216355014134024_wp
        Alpha(8) = 3.6774503027702362826026338815665894799167_wp

        Xi(1) = 1.021671002042388205223975850000_wp
        Xi(2) = 1.202161314562530556150979355046_wp
        Xi(3) = 1.601761466027823694051497593893_wp
        Xi(4) = 2.304651398713548862695610353057_wp
        Xi(5) = 3.466804726006286386684115963241_wp
        Xi(6) = 5.372522613929845453073497196783_wp
        Xi(7) = 8.536677921424592142309040188053_wp
        Xi(8) = 13.908476252112584528250471826993_wp
        Xi(9) = 23.297092629209026914266500440931_wp
        Xi(10) = 40.305769961082365378246539222573_wp
        Xi(11) = 72.521641272860303741032428348490_wp
        Xi(12) = 137.027405284593694581585232583620_wp
        Xi(13) = 276.169416913870757579463344200121_wp
        Xi(14) = 608.436098962609436813231411633751_wp
        Xi(15) = 1533.982633106747890172627535321226_wp
        Xi(16) = 4916.479141479043034834717218473088_wp

      case (29)
        ! ----- K = 08 , R = 9E3 -----

        RIni = 9.0e3_wp

        Omega(1) = 0.0009772948930853391284634293194322207654_wp
        Omega(2) = 0.0061588545859857373910286541373454438641_wp
        Omega(3) = 0.0248279792632893715773956862941096446207_wp
        Omega(4) = 0.0800104155964547076993918436893515888642_wp
        Omega(5) = 0.2239981341538947426952855732085190254566_wp
        Omega(6) = 0.5690154793075477250444485499691893437557_wp
        Omega(7) = 1.3597123047826839243668387013919129913120_wp
        Omega(8) = 3.3413902799279070789106871330886860960163_wp

        Alpha(1) = 0.0003116085515013381649146047259156533382_wp
        Alpha(2) = 0.0032763270270228551517249077334320572774_wp
        Alpha(3) = 0.0169824721145694412746424573762427101542_wp
        Alpha(4) = 0.0647962903315198952947204409646975165060_wp
        Alpha(5) = 0.2059788824482929637675653403161568633095_wp
        Alpha(6) = 0.5785040412974863724415719445914874086156_wp
        Alpha(7) = 1.4889211961369413103574085854852171451057_wp
        Alpha(8) = 3.6680786204192270931207647821636896878772_wp

        Xi(1) = 1.021743940791255173482741935853_wp
        Xi(2) = 1.202869572669394368350827273506_wp
        Xi(3) = 1.604023797288146306822334552500_wp
        Xi(4) = 2.310041602167562402878400740214_wp
        Xi(5) = 3.478268425986425059753121002970_wp
        Xi(6) = 5.395856515936656412537469895696_wp
        Xi(7) = 8.583571074372407718779121044150_wp
        Xi(8) = 14.003158945111380723029526063073_wp
        Xi(9) = 23.491596446149704716793715775225_wp
        Xi(10) = 40.716854890305933873523303034858_wp
        Xi(11) = 73.427573061585111519056212614487_wp
        Xi(12) = 139.141130298612416882342301960307_wp
        Xi(13) = 281.528042976889119886063639341955_wp
        Xi(14) = 623.821571913905158934809946913447_wp
        Xi(15) = 1588.132591724796786158968586732954_wp
        Xi(16) = 5244.360558124685915970530913909897_wp

      case (30)
        ! ----- K = 08 , R = 1E4 -----

        RIni = 1.0e4_wp

        Omega(1) = 0.0009464736592531918982037029551990930942_wp
        Omega(2) = 0.0060603142125795529876743582796405629409_wp
        Omega(3) = 0.0245761508893177345417288895412788463091_wp
        Omega(4) = 0.0794675791825153871774235596336932729855_wp
        Omega(5) = 0.2229854619672547855061591620673411284770_wp
        Omega(6) = 0.5673808117367401690706377181960817779327_wp
        Omega(7) = 1.3574743205521424079971690157542241195188_wp
        Omega(8) = 3.3387907492178366939718225925304295742535_wp

        Alpha(1) = 0.0002996417442897515258706464888341716724_wp
        Alpha(2) = 0.0032046869025454530459728701673183248744_wp
        Alpha(3) = 0.0167448876770150786602602997407696072685_wp
        Alpha(4) = 0.0641750239049883675144635847287233332281_wp
        Alpha(5) = 0.2045953172880951720781321501174687682578_wp
        Alpha(6) = 0.5758049688355264809263081571266695846134_wp
        Alpha(7) = 1.4842730758101718564713131232046805507707_wp
        Alpha(8) = 3.6609907150717963281606054382066872676660_wp

        Xi(1) = 1.021799368551998172280174947968_wp
        Xi(2) = 1.203407933153845310266828738666_wp
        Xi(3) = 1.605744200736071323793070775299_wp
        Xi(4) = 2.314143118448317799169464126230_wp
        Xi(5) = 3.486998390044023478622631229129_wp
        Xi(6) = 5.413644324178145450510157843382_wp
        Xi(7) = 8.619363150613084765573612688883_wp
        Xi(8) = 14.075530346751954986067589814525_wp
        Xi(9) = 23.640522751564629041001741605044_wp
        Xi(10) = 41.032293548734703098751497307717_wp
        Xi(11) = 74.124568878431914588367224183685_wp
        Xi(12) = 140.773073238400151588334274777026_wp
        Xi(13) = 285.685022748084026883441666200270_wp
        Xi(14) = 635.842924952911650204878668546371_wp
        Xi(15) = 1630.941436333689831705839878850384_wp
        Xi(16) = 5514.459053591922825354743054049322_wp

      case (31)
        ! ----- K = 08 , R = 2E4 (Best : R = 13750) -----

        RIni = 2.0e4_wp

        Omega(1) = 0.0008991679845521025146213091349452900536_wp
        Omega(2) = 0.0059076940528835928267816846645027473173_wp
        Omega(3) = 0.0241839922144492679877196507007286463420_wp
        Omega(4) = 0.0786197299999429678366664672739538843871_wp
        Omega(5) = 0.2214005640066087043894230840246351021960_wp
        Omega(6) = 0.5648187066649001726484184693344303695994_wp
        Omega(7) = 1.3539633386811781257065209760348523104767_wp
        Omega(8) = 3.3347117690388683390167318121299899758014_wp

        Alpha(1) = 0.0002812708994976841560705178499606632370_wp
        Alpha(2) = 0.0030942621869177655448272020594080444766_wp
        Alpha(3) = 0.0163765529725487554336287900069724088326_wp
        Alpha(4) = 0.0632081354614848777052774422835490497619_wp
        Alpha(5) = 0.2024364017926240634856352787473987575595_wp
        Alpha(6) = 0.5715854881795667998684029598166489449795_wp
        Alpha(7) = 1.4769970948293096661392900803555505717668_wp
        Alpha(8) = 3.6498865034832642553590253964301837186213_wp

        Xi(1) = 1.021886671863076921249954831517_wp
        Xi(2) = 1.204256055576628144164301470465_wp
        Xi(3) = 1.608455843152621252459787137834_wp
        Xi(4) = 2.320612438458095723274599020414_wp
        Xi(5) = 3.500779688469234837353005751481_wp
        Xi(6) = 5.441757327447499788513479712648_wp
        Xi(7) = 8.676006252118678827370268358266_wp
        Xi(8) = 14.190251215366537212471398365921_wp
        Xi(9) = 23.877064381777278509466633238389_wp
        Xi(10) = 41.534481087341492515363494675285_wp
        Xi(11) = 75.237562784543763741429422964302_wp
        Xi(12) = 143.389208759031316908760800288292_wp
        Xi(13) = 292.384601788253456366994953441463_wp
        Xi(14) = 655.376702446691242420762080200802_wp
        Xi(15) = 1701.442526845445518124400052784040_wp
        Xi(16) = 5981.188006841379713574013976540300_wp

    end select

  case (9)
    ! ===== K = 09 =====

    select case (InitR)

      case default !(1:3)
        ! ----- K = 09 , R = 1E1 -----

        RIni = 1.0e1_wp

        Omega(1) = 0.0944408179910022914211827939345855753572_wp
        Omega(2) = 0.2279868254700224364391568376841945564593_wp
        Omega(3) = 0.3835833446278580757457781630348492285520_wp
        Omega(4) = 0.5808769624236879772758747331184991935515_wp
        Omega(5) = 0.8513259963968657339497200786659902860265_wp
        Omega(6) = 1.2461698737081927510677095694013871707284_wp
        Omega(7) = 1.8552456496972611524982552100304644682183_wp
        Omega(8) = 2.8782542842146350474483856851648511110398_wp
        Omega(9) = 5.0648647749669027666626075490086122954381_wp

        Alpha(1) = 0.0366466399628543326388667965287648797812_wp
        Alpha(2) = 0.1966210450881259960855375681920698127669_wp
        Alpha(3) = 0.4998872452506653510785443461639232509697_wp
        Alpha(4) = 0.9775594233646502536597307220223029844419_wp
        Alpha(5) = 1.6858162243103592955540195874597486636048_wp
        Alpha(6) = 2.7212944265320903013127828984352163388394_wp
        Alpha(7) = 4.2485871297667448256117628258010654462851_wp
        Alpha(8) = 6.5659021657465178678850237226072295015911_wp
        Alpha(9) = 10.3619750572942871608486670176318966696272_wp

        Xi(1) = 1.004816667567330690951678262568_wp
        Xi(2) = 1.043640164615983558925489671498_wp
        Xi(3) = 1.122838292110773394764933197365_wp
        Xi(4) = 1.245537281040695091961952034598_wp
        Xi(5) = 1.416503870858663658945965357905_wp
        Xi(6) = 1.642263421803972880713869131331_wp
        Xi(7) = 1.931242857291945032001748150385_wp
        Xi(8) = 2.293851689305635893449675233313_wp
        Xi(9) = 2.742336706136967312526578344212_wp
        Xi(10) = 3.290135009447957957322283828461_wp
        Xi(11) = 3.950260267941195376894397695011_wp
        Xi(12) = 4.731964228677754497661561705613_wp
        Xi(13) = 5.634602288469151520206024130744_wp
        Xi(14) = 6.637671877203011474612159847908_wp
        Xi(15) = 7.687408987534133705983585560517_wp
        Xi(16) = 8.684756878114089741378667230975_wp
        Xi(17) = 9.487188575731532130461132190646_wp
        Xi(18) = 9.940546114292697808638432643846_wp

      case (4)
        ! ----- K = 09 , R = 2E1 -----

        RIni = 2.0e1_wp

        Omega(1) = 0.0571068968380650208214283861801741082331_wp
        Omega(2) = 0.1403046002401256655449986870953793527406_wp
        Omega(3) = 0.2442888425192801570354217410852015746059_wp
        Omega(4) = 0.3900028093978512700292296533355340670823_wp
        Omega(5) = 0.6117598936983984954656075327950048858838_wp
        Omega(6) = 0.9636093286970012084923647988077277659613_wp
        Omega(7) = 1.5367428589142293419555346245530813575897_wp
        Omega(8) = 2.5294131477214145515240717454474861369818_wp
        Omega(9) = 4.6740157375771901292205168232385403825901_wp

        Alpha(1) = 0.0221160161671775412491568451461843025641_wp
        Alpha(2) = 0.1197005617264399827869773666744190876443_wp
        Alpha(3) = 0.3095577652107071025687663745618039001783_wp
        Alpha(4) = 0.6220398529933125745974668852067424040797_wp
        Alpha(5) = 1.1147022114834890278841847410795651285298_wp
        Alpha(6) = 1.8885600218294077823734736232275110978662_wp
        Alpha(7) = 3.1147249042730755213409521431344728625845_wp
        Alpha(8) = 5.0981371610313976662498536285283989855088_wp
        Alpha(9) = 8.5274716594293656964781424889565641933586_wp

        Xi(1) = 1.006555872841583084060547437222_wp
        Xi(2) = 1.059567009314663162903441540053_wp
        Xi(3) = 1.168615284932517771484226931644_wp
        Xi(4) = 1.339862039200077573975269928930_wp
        Xi(5) = 1.582914108413751619482655319526_wp
        Xi(6) = 1.911428455969247629849666614277_wp
        Xi(7) = 2.344012560312972394752084603908_wp
        Xi(8) = 2.905321623723958540490286406310_wp
        Xi(9) = 3.627168437279999827415818969278_wp
        Xi(10) = 4.549312527505255060976396830341_wp
        Xi(11) = 5.719136926761125761485560214936_wp
        Xi(12) = 7.188365599284229816648417177305_wp
        Xi(13) = 9.002970775840758896682580747495_wp
        Xi(14) = 11.179329521976524121525120580856_wp
        Xi(15) = 13.657279910840638271866953612488_wp
        Xi(16) = 16.227987378030576691434472280662_wp
        Xi(17) = 18.472407258722503753042842866705_wp
        Xi(18) = 19.818889952580328229006512152921_wp

      case (5)
        ! ----- K = 09 , R = 3E1 -----

        RIni = 3.0e1_wp

        Omega(1) = 0.0421868132593801466490870087644449171194_wp
        Omega(2) = 0.1050116169989357095166672938224561306697_wp
        Omega(3) = 0.1876064252395335822036319395333947568361_wp
        Omega(4) = 0.3112629685874389948304578196180969484885_wp
        Omega(5) = 0.5109461867562471753438993471441165183933_wp
        Omega(6) = 0.8412092785410361046654834082936247341422_wp
        Omega(7) = 1.3943134568732370930999389657678477760783_wp
        Omega(8) = 2.3697175189404877360642431849413469535648_wp
        Omega(9) = 4.4940473157040011671636836609877718728967_wp

        Alpha(1) = 0.0163142165710595383079330655470839950283_wp
        Alpha(2) = 0.0888739577095855045956677292773306930940_wp
        Alpha(3) = 0.2328241238721762329164349097920805320427_wp
        Alpha(4) = 0.4776215751713829359528015044977777847635_wp
        Alpha(5) = 0.8804879707385602953333657294532343939863_wp
        Alpha(6) = 1.5426690213493910481135779066974578199734_wp
        Alpha(7) = 2.6362216580612873393762302098863870014611_wp
        Alpha(8) = 4.4683557670936419926877025687872446724214_wp
        Alpha(9) = 7.7290029467926875373327832985381746766507_wp

        Xi(1) = 1.007612743465411295597580765282_wp
        Xi(2) = 1.069288322758420971270165722444_wp
        Xi(3) = 1.196793605127912162879554036632_wp
        Xi(4) = 1.398623977355137069597014465305_wp
        Xi(5) = 1.688169721369932256018440985823_wp
        Xi(6) = 2.084819481200011412769940166712_wp
        Xi(7) = 2.615652456460386988132751917568_wp
        Xi(8) = 3.317638945726810187353272851851_wp
        Xi(9) = 4.240261748474029180732552957167_wp
        Xi(10) = 5.448388874927914446010673854559_wp
        Xi(11) = 7.024638234516258417268474989470_wp
        Xi(12) = 9.068900307006999057628238158912_wp
        Xi(13) = 11.689140847001693195028604321806_wp
        Xi(14) = 14.970329231989185316507862122393_wp
        Xi(15) = 18.896983906883134981050109679757_wp
        Xi(16) = 23.201684329870220000954783579061_wp
        Xi(17) = 27.172322404323542907947364177801_wp
        Xi(18) = 29.659116605127599357663581436384_wp

      case (6)
        ! ----- K = 09 , R = 4E1 -----

        RIni = 4.0e1_wp

        Omega(1) = 0.0339352597594549956029142193103531610632_wp
        Omega(2) = 0.0853875404125285589817064869910812063836_wp
        Omega(3) = 0.1558201128416026146370843355470903190962_wp
        Omega(4) = 0.2665467533840560541123516219430555906911_wp
        Omega(5) = 0.4525182570791364093112062183532628978355_wp
        Omega(6) = 0.7683860176045937915851984090220838652385_wp
        Omega(7) = 1.3074948146510294393323609041246413653425_wp
        Omega(8) = 2.2708533075573004104269481118194562441204_wp
        Omega(9) = 4.3823089011892681450197062353879573493032_wp

        Alpha(1) = 0.0131076927865854293291480287507455582130_wp
        Alpha(2) = 0.0717893279105617697030163049110029760413_wp
        Alpha(3) = 0.1901024737453172946342319291068712061588_wp
        Alpha(4) = 0.3967095145193068460472401309813150760419_wp
        Alpha(5) = 0.7480739001355727603405007530845693963784_wp
        Alpha(6) = 1.3446599602615252448418320385670199357264_wp
        Alpha(7) = 2.3582852576121468594558261422378109273268_wp
        Alpha(8) = 4.0973891923794437705357018852936334951664_wp
        Alpha(9) = 7.2533276338963031727204155973964816439548_wp

        Xi(1) = 1.008370802207068996059723120506_wp
        Xi(2) = 1.076281916485633244188503676586_wp
        Xi(3) = 1.217180087021721421173571509033_wp
        Xi(4) = 1.441475326386836438912961155800_wp
        Xi(5) = 1.765697188344513895216086885043_wp
        Xi(6) = 2.214079990555948211682427140268_wp
        Xi(7) = 2.821020282627014928592337761337_wp
        Xi(8) = 3.634363517681795004588690689573_wp
        Xi(9) = 4.719590811785275055820904066550_wp
        Xi(10) = 6.164995925690650534038705909978_wp
        Xi(11) = 8.087321179937766064092785622819_wp
        Xi(12) = 10.635473568815829112539639300650_wp
        Xi(13) = 13.985214370229430863069863821835_wp
        Xi(14) = 18.306471237533177499196002102622_wp
        Xi(15) = 23.662468079847879753874195785102_wp
        Xi(16) = 29.775621021240134683738531862218_wp
        Xi(17) = 35.656631518931774505604126446201_wp
        Xi(18) = 39.469135361457544960983589987791_wp

      case (7)
        ! ----- K = 09 , R = 5E1 -----

        RIni = 5.0e1_wp

        Omega(1) = 0.0286272689853100972254734734104286708600_wp
        Omega(2) = 0.0727085059279048214692987329998974388445_wp
        Omega(3) = 0.1351357560506628231771679773953742653703_wp
        Omega(4) = 0.2371065680463565181662665207196205585660_wp
        Omega(5) = 0.4133327297773413407479866504834475904318_wp
        Omega(6) = 0.7184889744300452265571461929472008023367_wp
        Omega(7) = 1.2469241956836626880611440526536171091720_wp
        Omega(8) = 2.2011224300594862497357229180749982333509_wp
        Omega(9) = 4.3033668024855845596170211830866492164205_wp

        Alpha(1) = 0.0110461620270614760939376585401061614533_wp
        Alpha(2) = 0.0607803585908880953167640268097748901255_wp
        Alpha(3) = 0.1624695706747093375434251236177907173897_wp
        Alpha(4) = 0.3440842840849093881819449614045680618801_wp
        Alpha(5) = 0.6612286910567070099512161640564755771265_wp
        Alpha(6) = 1.2133287214430294528736201109531123165652_wp
        Alpha(7) = 2.1716507047182063833406368003053898974031_wp
        Alpha(8) = 3.8454420774551983344853589574796615124797_wp
        Alpha(9) = 6.9274020285996225332857123557772638378083_wp

        Xi(1) = 1.008959735058196170785597967079_wp
        Xi(2) = 1.081727547054457924820790315490_wp
        Xi(3) = 1.233121852783218921423627478795_wp
        Xi(4) = 1.475183691902814312737687862942_wp
        Xi(5) = 1.827141305274111254060460940440_wp
        Xi(6) = 2.317454167204209215281951461662_wp
        Xi(7) = 2.986999140777004608073366687115_wp
        Xi(8) = 3.893408271618215431243356650093_wp
        Xi(9) = 5.116816594572857777867280759665_wp
        Xi(10) = 6.767428367328306293339984511803_wp
        Xi(11) = 8.994713612110818832037772985188_wp
        Xi(12) = 11.996105690184234046635758375032_wp
        Xi(13) = 16.017531079781982929557959671740_wp
        Xi(14) = 21.323363339601217870261651299302_wp
        Xi(15) = 28.079299750913681289060663814894_wp
        Xi(16) = 36.038988461443780951520832189772_wp
        Xi(17) = 43.963748828900448661904931668687_wp
        Xi(18) = 49.253357175193826684306275254244_wp

      case (8)
        ! ----- K = 09 , R = 6E1 -----

        RIni = 6.0e1_wp

        Omega(1) = 0.0248959721764766027179375390554483260530_wp
        Omega(2) = 0.0637623571241729140057427627219510668510_wp
        Omega(3) = 0.1204491051105075692019704368396215699022_wp
        Omega(4) = 0.2159745478825277155882121984942223491544_wp
        Omega(5) = 0.3847391931442429500694665106541592081157_wp
        Omega(6) = 0.6814342485253308534780963823251909161627_wp
        Omega(7) = 1.2013063905514669937966282242669535662571_wp
        Omega(8) = 2.1481696096724234316868024086133459604753_wp
        Omega(9) = 4.2433533131659190401445413964864883382688_wp

        Alpha(1) = 0.0095976729315090804611749044153445886707_wp
        Alpha(2) = 0.0530301581506099356459857548096170631879_wp
        Alpha(3) = 0.1429528058023013083942210840748820999124_wp
        Alpha(4) = 0.3067286838593251691443934692227202276626_wp
        Alpha(5) = 0.5991056928552873535320788533908853423782_wp
        Alpha(6) = 1.1184415035302334036869387201562631162233_wp
        Alpha(7) = 2.0353775012644672000924200849780731914507_wp
        Alpha(8) = 3.6597441611647041148983050540266503958264_wp
        Alpha(9) = 6.6854573090516597900362372186577886168379_wp

        Xi(1) = 1.009439712975342740008954933728_wp
        Xi(2) = 1.086173843961676821479045429442_wp
        Xi(3) = 1.246182533203064046745112136705_wp
        Xi(4) = 1.502931295016856072521704845713_wp
        Xi(5) = 1.878023333089494872893965293770_wp
        Xi(6) = 2.403678824259185109468717200798_wp
        Xi(7) = 3.126614448205453200557712989394_wp
        Xi(8) = 4.113390795506283836857619462535_wp
        Xi(9) = 5.457701372264594397165637928993_wp
        Xi(10) = 7.290351469819014162965231662383_wp
        Xi(11) = 9.792165891831702495551326137502_wp
        Xi(12) = 13.208156738942577135276223998517_wp
        Xi(13) = 17.855245368191081212216886342503_wp
        Xi(14) = 24.098110732540835243753463146277_wp
        Xi(15) = 32.222282275814727370288892416283_wp
        Xi(16) = 42.047677291122653286947752349079_wp
        Xi(17) = 52.118697621441455680724175181240_wp
        Xi(18) = 59.014681121470907498860469786450_wp

      case (9)
        ! ----- K = 09 , R = 7E1 -----

        RIni = 7.0e1_wp

        Omega(1) = 0.0221145238599984708285472578132013765639_wp
        Omega(2) = 0.0570717255426385474206558284793189272932_wp
        Omega(3) = 0.1094026520368098471004138748374767686755_wp
        Omega(4) = 0.1999182472556715514091715824029904524650_wp
        Omega(5) = 0.3626940710347193269039341168014090044380_wp
        Omega(6) = 0.6524437931310793758327148950115770276170_wp
        Omega(7) = 1.1652087065303460656573789067103064098774_wp
        Omega(8) = 2.1059899831229173835893397903618051714147_wp
        Omega(9) = 4.1955111714606977495710393721850550718955_wp

        Alpha(1) = 0.0085183608716480930129208784529204478986_wp
        Alpha(2) = 0.0472454375504523413084134817602199873932_wp
        Alpha(3) = 0.1283432881500915206924142222821849657066_wp
        Alpha(4) = 0.2786356835219511445694319518029402615866_wp
        Alpha(5) = 0.5520531959128402951265945919345767833875_wp
        Alpha(6) = 1.0459297104212418192065517374977900999511_wp
        Alpha(7) = 1.9302841753638598767603418160376804735279_wp
        Alpha(8) = 3.5153907951084480490184375200968247554556_wp
        Alpha(9) = 6.4962582918876183627442111312433326020255_wp

        Xi(1) = 1.009843670644150163007851661590_wp
        Xi(2) = 1.089921584463474284599875108182_wp
        Xi(3) = 1.257222614712810082149112289596_wp
        Xi(4) = 1.526478640844834332927597297935_wp
        Xi(5) = 1.921418080233361886256099326786_wp
        Xi(6) = 2.477658685557917726512491829460_wp
        Xi(7) = 3.247246099484508796142664954765_wp
        Xi(8) = 4.304973006142101080943179747251_wp
        Xi(9) = 5.757178149665511641443205181901_wp
        Xi(10) = 7.754134190414868861747355044400_wp
        Xi(11) = 10.506743335083980274598236537287_wp
        Xi(12) = 14.306497198812701568577943067595_wp
        Xi(13) = 19.541352837094909644308216023489_wp
        Xi(14) = 26.680033709516938091391158316412_wp
        Xi(15) = 36.141002166271460668139914318431_wp
        Xi(16) = 47.840286390914921566680506614944_wp
        Xi(17) = 60.139334259777297103655158139190_wp
        Xi(18) = 68.755191896088548135501650193646_wp

      case (10)
        ! ----- K = 09 , R = 8E1 -----

        RIni = 8.0e1_wp

        Omega(1) = 0.0199526784301063577451595563647113174000_wp
        Omega(2) = 0.0518563003933426289609489017590204795738_wp
        Omega(3) = 0.1007466086972549770961059441287677174159_wp
        Omega(4) = 0.1872164941269185330232445105269611929089_wp
        Omega(5) = 0.3450262461891316848462511746031822212899_wp
        Omega(6) = 0.6289183401844695380346686452988080873183_wp
        Omega(7) = 1.1356374020567791251418512943338612330990_wp
        Omega(8) = 2.0712467277560673076502312239810521532490_wp
        Omega(9) = 4.1560789671987043254405835845943784079282_wp

        Alpha(1) = 0.0076797894839484662018849520006419373530_wp
        Alpha(2) = 0.0427441668967045518829467841415148399165_wp
        Alpha(3) = 0.1169452389354414968946618488820909931292_wp
        Alpha(4) = 0.2566228998618464373596571165281332582708_wp
        Alpha(5) = 0.5149406599350990045905508696311159155812_wp
        Alpha(6) = 0.9882757618211388782867611590976508750828_wp
        Alpha(7) = 1.8460526692834495029365074469929197675810_wp
        Alpha(8) = 3.3988951192899249120239757449013495715917_wp
        Alpha(9) = 6.3427950576266634284162970747189547182643_wp

        Xi(1) = 1.010191606375282269425497361226_wp
        Xi(2) = 1.093153804208305990286476117035_wp
        Xi(3) = 1.266767168213191796510841413692_wp
        Xi(4) = 1.546904808150974197644700325327_wp
        Xi(5) = 1.959220781480741635505182784982_wp
        Xi(6) = 2.542436945678284931915413835668_wp
        Xi(7) = 3.353509288478506267239254157175_wp
        Xi(8) = 4.474882125703746105216396244941_wp
        Xi(9) = 6.024763681615760368970313720638_wp
        Xi(10) = 8.171904787613485154906811658293_wp
        Xi(11) = 11.156115140955559894564430578612_wp
        Xi(12) = 15.314228304191129126365922274999_wp
        Xi(13) = 21.104851155624359648754762019962_wp
        Xi(14) = 29.103157066959354892787814605981_wp
        Xi(15) = 39.870672818322979935601324541494_wp
        Xi(16) = 53.445175507062735675845033256337_wp
        Xi(17) = 68.039170680158321147246169857681_wp
        Xi(18) = 78.476477920522839326622488442808_wp

      case (11)
        ! ----- K = 09 , R = 9E1 -----

        RIni = 9.0e1_wp

        Omega(1) = 0.0182189939054983629658344577648132478487_wp
        Omega(2) = 0.0476627110113685720830481885174068068523_wp
        Omega(3) = 0.0937524260468592182812212071363333620866_wp
        Omega(4) = 0.1768615462829543274748652151306949065201_wp
        Omega(5) = 0.3304536625757883221111369659306333801396_wp
        Omega(6) = 0.6093043202241387087383581167010504486825_wp
        Omega(7) = 1.1107832014454115858535093730452558702382_wp
        Omega(8) = 2.0419093040247857171864270808825381209317_wp
        Omega(9) = 4.1227649447721858360846325375348442321410_wp

        Alpha(1) = 0.0070075258403555020703036993606760418629_wp
        Alpha(2) = 0.0391306245919652705714470233844259894340_wp
        Alpha(3) = 0.1077729099627231377348294226459213263070_wp
        Alpha(4) = 0.2388361570386172796128659845393826799409_wp
        Alpha(5) = 0.4847687047212494723745068031517391204943_wp
        Alpha(6) = 0.9410620539784709467651807601740188147232_wp
        Alpha(7) = 1.7765815928488142974106880900286853375292_wp
        Alpha(8) = 3.3022331320416157920179939244675892950909_wp
        Alpha(9) = 6.2148972696211912656628906237443743520998_wp

        Xi(1) = 1.010496581083280220880737942935_wp
        Xi(2) = 1.095990183864951506009788317897_wp
        Xi(3) = 1.275160572005126224937043843966_wp
        Xi(4) = 1.564919881246217947534660197562_wp
        Xi(5) = 1.992684549902902002571636330686_wp
        Xi(6) = 2.600037625374358896124249440618_wp
        Xi(7) = 3.448494194251035427356555373990_wp
        Xi(8) = 4.627657101784136806088554294547_wp
        Xi(9) = 6.266935365306330218970742862439_wp
        Xi(10) = 8.552684926856656488780572544783_wp
        Xi(11) = 11.752554209802655194039289199281_wp
        Xi(12) = 16.247597185827061139207216911018_wp
        Xi(13) = 22.566441385940585462321905652061_wp
        Xi(14) = 31.392232420985656915490835672244_wp
        Xi(15) = 43.437546052548135833148990059271_wp
        Xi(16) = 58.884123231982044899268657900393_wp
        Xi(17) = 75.828863188570153397449757903814_wp
        Xi(18) = 88.179798689784689713633270002902_wp

      case (12)
        ! ----- K = 09 , R = 1E2 -----

        RIni = 1.0e2_wp

        Omega(1) = 0.0167944177457944369643606796760337829255_wp
        Omega(2) = 0.0442084363213146724151648113421142483048_wp
        Omega(3) = 0.0879647289700077565909605917249258766333_wp
        Omega(4) = 0.1682207519727353216011092248982627950227_wp
        Omega(5) = 0.3181646277188681652041499919869593782096_wp
        Omega(6) = 0.5926071686898262811282918349764514687195_wp
        Omega(7) = 1.0894766158992761576757013486371761246119_wp
        Omega(8) = 2.0166575312935179065381569030179775836586_wp
        Omega(9) = 4.0940777439727191787735560701833037455799_wp

        Alpha(1) = 0.0064552940709981691840570444240049452489_wp
        Alpha(2) = 0.0361585229138805650089794584134406107978_wp
        Alpha(3) = 0.1002117521668118395314415851327094486578_wp
        Alpha(4) = 0.2241169715425050551775489754424697252944_wp
        Alpha(5) = 0.4596569999787066861996788491584453595351_wp
        Alpha(6) = 0.9015052273114635050170299901406423259687_wp
        Alpha(7) = 1.7180040401346316646016137985597538317961_wp
        Alpha(8) = 3.2202910181231106476629905133179931908671_wp
        Alpha(9) = 6.1060543748685659035287109919210024600034_wp

        Xi(1) = 1.010767590706046830870135089242_wp
        Xi(2) = 1.098513253256041984550969714540_wp
        Xi(3) = 1.282640843522771026668366500800_wp
        Xi(4) = 1.581016658720702851423020596222_wp
        Xi(5) = 2.022682780613193844919048269837_wp
        Xi(6) = 2.651877998803424621053134724313_wp
        Xi(7) = 3.534377760344112353891460331923_wp
        Xi(8) = 4.766517190711064898105131515393_wp
        Xi(9) = 6.488322663266226617267279364043_wp
        Xi(10) = 8.902976339778169532525320395422_wp
        Xi(11) = 12.304988817151161595124519987365_wp
        Xi(12) = 17.118551029943605555280750074587_wp
        Xi(13) = 23.941540277556665305969119117435_wp
        Xi(14) = 33.565990820897510290427012691339_wp
        Xi(15) = 46.861900712050953334703029540265_wp
        Xi(16) = 64.174408587402146174338835038498_wp
        Xi(17) = 83.517069627790160295244259458514_wp
        Xi(18) = 97.866183331819870887258172587053_wp

      case (13)
        ! ----- K = 09 , R = 2E2 -----

        RIni = 2.0e2_wp

        Omega(1) = 0.0098170280410918193187669420103624862151_wp
        Omega(2) = 0.0271418312236300197459824140588935392771_wp
        Omega(3) = 0.0588555405499913494831571141724380069604_wp
        Omega(4) = 0.1233686937014436572622217043460057794846_wp
        Omega(5) = 0.2520334255785888517701395089565252760622_wp
        Omega(6) = 0.4999564200366363836923726954308833114737_wp
        Omega(7) = 0.9685402483680957653956918040893953047998_wp
        Omega(8) = 1.8714190637073344861563556351669035393570_wp
        Omega(9) = 3.9288430703647423862870380339984421880217_wp

        Alpha(1) = 0.0037535059072584840494327812487895812410_wp
        Alpha(2) = 0.0215518207542155084664736459237888155371_wp
        Alpha(3) = 0.0627363115303287840557539337127401779526_wp
        Alpha(4) = 0.1500627747875535932740561143039492719709_wp
        Alpha(5) = 0.3306054926762187856322579987411458546376_wp
        Alpha(6) = 0.6934050066019539150150539918993075616527_wp
        Alpha(7) = 1.4030303812801839501980469271380513873737_wp
        Alpha(8) = 2.7716770734119855206130378366680133694899_wp
        Alpha(9) = 5.5024333947554721585375403503093139079283_wp

        Xi(1) = 1.012493178977090445302446097831_wp
        Xi(2) = 1.114635600388191899131652695676_wp
        Xi(3) = 1.330753639161060988111394409739_wp
        Xi(4) = 1.685489635820716217791057078834_wp
        Xi(5) = 2.219623428939024205059499239567_wp
        Xi(6) = 2.997013539656331337869207831481_wp
        Xi(7) = 4.115637502885650771990155849522_wp
        Xi(8) = 5.723997886549447128637368997772_wp
        Xi(9) = 8.046725109427186248094165943456_wp
        Xi(10) = 11.425345310044984696411229929680_wp
        Xi(11) = 16.382984293605163969229088216650_wp
        Xi(12) = 23.725841312075762377931109625706_wp
        Xi(13) = 34.695755476516356146515773772876_wp
        Xi(14) = 51.167672329122699286657605100004_wp
        Xi(15) = 75.757973024436673144255127709812_wp
        Xi(16) = 111.141141004968368143079882059965_wp
        Xi(17) = 156.003502086901291792675472436258_wp
        Xi(18) = 193.941202758701799216400907255320_wp

      case (14)
        ! ----- K = 09 , R = 3E2 -----

        RIni = 3.0e2_wp

        Omega(1) = 0.0071726984278288066010701145430727976304_wp
        Omega(2) = 0.0205765888728709433145848363075347542406_wp
        Omega(3) = 0.0472784447759684887763415115508047747994_wp
        Omega(4) = 0.1045511950832360717830596744115645435613_wp
        Omega(5) = 0.2227845285766869881552247611988804010252_wp
        Omega(6) = 0.4572258464535360821489034383047211917983_wp
        Omega(7) = 0.9110357207216383647677138135634322679834_wp
        Omega(8) = 1.8011087998289808716906312824335145705845_wp
        Omega(9) = 3.8486906070706142239581243336132843069208_wp

        Alpha(1) = 0.0027314605081480598387174042610339719239_wp
        Alpha(2) = 0.0159839368530351834281333627679755338136_wp
        Alpha(3) = 0.0482285546093564691738587914965341596485_wp
        Alpha(4) = 0.1205956314577905223286492276957293512396_wp
        Alpha(5) = 0.2773862359659811593578550292793494236321_wp
        Alpha(6) = 0.6044051058271434355060290621874941052738_wp
        Alpha(7) = 1.2638987330530245045212903165499085389456_wp
        Alpha(8) = 2.5683342984551687556826066183290890876378_wp
        Alpha(9) = 5.2238613500063365430170492231809475924820_wp

        Xi(1) = 1.013442243452601123016379447428_wp
        Xi(2) = 1.123545736564435914097738467010_wp
        Xi(3) = 1.357579144166563902564569266129_wp
        Xi(4) = 1.744447270564926417198842178991_wp
        Xi(5) = 2.332479735856684662865037638291_wp
        Xi(6) = 3.198527988114039162514945502380_wp
        Xi(7) = 4.462522014311318948893286817636_wp
        Xi(8) = 6.309665798547319267089439343010_wp
        Xi(9) = 9.026287646419280932472140044354_wp
        Xi(10) = 13.058806753397421812276646413409_wp
        Xi(11) = 19.111213672692908200831385379104_wp
        Xi(12) = 28.306646857575006329121869441678_wp
        Xi(13) = 42.451836058384489782507875688111_wp
        Xi(14) = 64.442554456834715372048894010959_wp
        Xi(15) = 98.726758168282922258074396637539_wp
        Xi(16) = 150.976388946992133244240363865174_wp
        Xi(17) = 222.590737029752850958952414828218_wp
        Xi(18) = 288.838153884362069473423773047216_wp

      case (15)
        ! ----- K = 09 , R = 4E2 -----

        RIni = 4.0e2_wp

        Omega(1) = 0.0057455219493550986707511589370728222192_wp
        Omega(2) = 0.0169957484843711026029143378035524136749_wp
        Omega(3) = 0.0407999991108632573202387710392669006865_wp
        Omega(4) = 0.0936390097268970759833905601055015210932_wp
        Omega(5) = 0.2052932850347774952756360611338948274351_wp
        Omega(6) = 0.4310646867374589646330212966418571340910_wp
        Omega(7) = 0.8752159980258892398767914255763855635450_wp
        Omega(8) = 1.7568568252994703761090192206673066266376_wp
        Omega(9) = 3.7981800919499917636168451195999296032824_wp

        Alpha(1) = 0.0021805301099834817776713274169869682950_wp
        Alpha(2) = 0.0129668297960735722897319044041863911332_wp
        Alpha(3) = 0.0402738064105043472570085068273915140935_wp
        Alpha(4) = 0.1041067480547515475663013055640515602818_wp
        Alpha(5) = 0.2468847433368982753301061963679963184859_wp
        Alpha(6) = 0.5522250628274896434768276298754585695860_wp
        Alpha(7) = 1.1807304388190987321817504440701895873644_wp
        Alpha(8) = 2.4449277513093071833964742278766379968147_wp
        Alpha(9) = 5.0530353436462703249978933861541463556932_wp

        Xi(1) = 1.014082320008305114359195564511_wp
        Xi(2) = 1.129572660367690155216839620334_wp
        Xi(3) = 1.375820336165024656551944537419_wp
        Xi(4) = 1.784828830160060559066256824501_wp
        Xi(5) = 2.410491847853145446194805812112_wp
        Xi(6) = 3.339407540750610744540241570100_wp
        Xi(7) = 4.708260852050882275168092183648_wp
        Xi(8) = 6.730819490899820774741646456363_wp
        Xi(9) = 9.742509792408877671368583328615_wp
        Xi(10) = 14.275279067561807030267706508653_wp
        Xi(11) = 21.184531719626648712390126760852_wp
        Xi(12) = 31.866723607990150701069209659977_wp
        Xi(13) = 48.632487294712915149030862949076_wp
        Xi(14) = 75.327499865931313749456421646755_wp
        Xi(15) = 118.207952411799593513130890443108_wp
        Xi(16) = 186.230305258828872594367709325525_wp
        Xi(17) = 284.882646179396948948880918806026_wp
        Xi(18) = 382.757902531871198242852116777613_wp

      case (16)
        ! ----- K = 09 , R = 5E2 -----

        RIni = 5.0e2_wp

        Omega(1) = 0.0048408422622110301172535852011480361767_wp
        Omega(2) = 0.0147065498776455584619584477344689243949_wp
        Omega(3) = 0.0365691795723620899713360646310356116828_wp
        Omega(4) = 0.0863219618381844940111901481882838993442_wp
        Omega(5) = 0.1933129417331675636105908430706357137296_wp
        Omega(6) = 0.4128560803248047374820924038107960996058_wp
        Omega(7) = 0.8499869026869296652283887472112411387570_wp
        Omega(8) = 1.7254635478924916411856693709836463312968_wp
        Omega(9) = 3.7623132069881963044505335247436050849501_wp

        Alpha(1) = 0.0018316193162704232806072771749805738750_wp
        Alpha(2) = 0.0110482476764524436467582862997360848567_wp
        Alpha(3) = 0.0351648885678759096644785831636825790270_wp
        Alpha(4) = 0.0933425211230552068260714164427138683777_wp
        Alpha(5) = 0.2266113854977841110267377383591380635153_wp
        Alpha(6) = 0.5169700044462441223419059410648657149068_wp
        Alpha(7) = 1.1237641613889137305690862911333738338726_wp
        Alpha(8) = 2.3595017117435727388094490741821118717780_wp
        Alpha(9) = 4.9339327011643232204912701277521591691766_wp

        Xi(1) = 1.014557844970328790747127190031_wp
        Xi(2) = 1.134059522948674922230086203889_wp
        Xi(3) = 1.389451040287651769223470354930_wp
        Xi(4) = 1.815157261209636782063603366311_wp
        Xi(5) = 2.469464621121473972293475229733_wp
        Xi(6) = 3.446760036341784853809386279799_wp
        Xi(7) = 4.897283105684400378483644988847_wp
        Xi(8) = 7.058237777334755448741671068902_wp
        Xi(9) = 10.305980129936124188583246308326_wp
        Xi(10) = 15.245008648988065745426279118391_wp
        Xi(11) = 22.861666274799535308881459627628_wp
        Xi(12) = 34.793888874816124672645356596945_wp
        Xi(13) = 53.808524630705320771634214693790_wp
        Xi(14) = 84.637269347141066234974093873689_wp
        Xi(15) = 135.292823009987981955348779194992_wp
        Xi(16) = 218.145883117094211575692774829349_wp
        Xi(17) = 343.752713393957227056008107979324_wp
        Xi(18) = 475.812907782750572627250562618428_wp

      case (17)
        ! ----- K = 09 , R = 6E2 -----

        RIni = 6.0e2_wp

        Omega(1) = 0.0042112164997987732153974691721304290937_wp
        Omega(2) = 0.0131017443419484174921678409399261600043_wp
        Omega(3) = 0.0335483667275851311346905863491896226947_wp
        Omega(4) = 0.0809880866937857405520083745043002920738_wp
        Omega(5) = 0.1844390570715671000409045104939664838639_wp
        Omega(6) = 0.3992056659940321296274959306238372391817_wp
        Omega(7) = 0.8309034361257299758688442403720131324008_wp
        Omega(8) = 1.7015875252701152502013762224386539401166_wp
        Omega(9) = 3.7350145975046860522764818890451010702236_wp

        Alpha(1) = 0.0015889645189828853134388664020311410230_wp
        Alpha(2) = 0.0097094235867913892325478815957529032943_wp
        Alpha(3) = 0.0315686052432080812238802524499758916932_wp
        Alpha(4) = 0.0856610590267349970372903755766458289145_wp
        Alpha(5) = 0.2119351219922484596478980470313224770962_wp
        Alpha(6) = 0.4911220300723124560155537915440149276947_wp
        Alpha(7) = 1.0815587767859619615666219583083318411809_wp
        Alpha(8) = 2.2957009271064489539480868440790573004051_wp
        Alpha(9) = 4.8444980562273129774501778666007112406078_wp

        Xi(1) = 1.014931989558186863813540878887_wp
        Xi(2) = 1.137595427276255482867614587938_wp
        Xi(3) = 1.400223484478566062872220698843_wp
        Xi(4) = 1.839219415469965509591677987888_wp
        Xi(5) = 2.516486149705543408925181858571_wp
        Xi(6) = 3.532883133055617747323925992298_wp
        Xi(7) = 5.050020403199381573491888453020_wp
        Xi(8) = 7.324977296251430986407521261938_wp
        Xi(9) = 10.769247260418148980261332514630_wp
        Xi(10) = 16.050472350587626915036087993371_wp
        Xi(11) = 24.270655484484105677417398183593_wp
        Xi(12) = 37.284660631758730655466171910462_wp
        Xi(13) = 58.277068934025140340554616003033_wp
        Xi(14) = 92.809724042426474277522885358849_wp
        Xi(15) = 150.592468673002287851692138076487_wp
        Xi(16) = 247.461463866105427042429987238847_wp
        Xi(17) = 399.765074289033366328194318839451_wp
        Xi(18) = 568.077352225634511329044329386306_wp

      case (18)
        ! ----- K = 09 , R = 7E2 -----

        RIni = 7.0e2_wp

        Omega(1) = 0.0037452979413056402919975806280541985593_wp
        Omega(2) = 0.0119065364988764364785814981427536363157_wp
        Omega(3) = 0.0312620574896743206805217914212580154754_wp
        Omega(4) = 0.0768822495241898944451949642064914058892_wp
        Omega(5) = 0.1775211901125933941356831516356074018859_wp
        Omega(6) = 0.3884622797054659326697941684747306112513_wp
        Omega(7) = 0.8157768943467678142010719721799461012779_wp
        Omega(8) = 1.6825795696161266758704433410542833371437_wp
        Omega(9) = 3.7132685721630956357432878922608665561711_wp

        Alpha(1) = 0.0014095059934701930788443244670950227349_wp
        Alpha(2) = 0.0087163949435635453861511518988680613518_wp
        Alpha(3) = 0.0288802601170618445065656990783375945853_wp
        Alpha(4) = 0.0798510207914467600695583929248932975042_wp
        Alpha(5) = 0.2007018612184257304212157968503760230305_wp
        Alpha(6) = 0.4711331220524638425386446810572138588213_wp
        Alpha(7) = 1.0486449545129898443035151855973197143612_wp
        Alpha(8) = 2.2456262013296283809680314869083872508781_wp
        Alpha(9) = 4.7740022239383786352602545566270464405534_wp

        Xi(1) = 1.015237855473021364284894052510_wp
        Xi(2) = 1.140489695132720713915616683742_wp
        Xi(3) = 1.409061234383698115726554767058_wp
        Xi(4) = 1.859021466956424458653700237942_wp
        Xi(5) = 2.555337142598838082825293827050_wp
        Xi(6) = 3.604394232696263654803364717694_wp
        Xi(7) = 5.177580777145438844111940968062_wp
        Xi(8) = 7.549221981124213971727637817466_wp
        Xi(9) = 11.161601790690508574296901933565_wp
        Xi(10) = 16.738314212686065972357507547486_wp
        Xi(11) = 25.485098092299234071381186517158_wp
        Xi(12) = 39.454079013578476423096397240897_wp
        Xi(13) = 62.215656701418109827267377553994_wp
        Xi(14) = 100.112967315529583313404859978846_wp
        Xi(15) = 164.492889647548323692194394141097_wp
        Xi(16) = 274.665312278384626365479448395490_wp
        Xi(17) = 453.319129539005994583567016320558_wp
        Xi(18) = 659.604619900343409066856281697255_wp

      case (19)
        ! ----- K = 09 , R = 8E2 -----

        RIni = 8.0e2_wp

        Omega(1) = 0.0033852098812278768834514551403294668752_wp
        Omega(2) = 0.0109774080736766307342131628217718475327_wp
        Omega(3) = 0.0294590554580619398994140601538704871132_wp
        Omega(4) = 0.0735981065963514141640045181091966952636_wp
        Omega(5) = 0.1719299109739136701433068893374311869593_wp
        Omega(6) = 0.3797108500746642855382118053597650941811_wp
        Omega(7) = 0.8033826246878963353733292540592714203740_wp
        Omega(8) = 1.6669487573286433863047756309683222752938_wp
        Omega(9) = 3.6953768800115142395979611622536253889848_wp

        Alpha(1) = 0.0012708763612674633672627278152371388398_wp
        Alpha(2) = 0.0079473213793494062269487572015835841199_wp
        Alpha(3) = 0.0267833433812802426014028269601885412499_wp
        Alpha(4) = 0.0752722660784924992309792568523363343047_wp
        Alpha(5) = 0.1917594605878600712647614329187106818608_wp
        Alpha(6) = 0.4550830556615172917860404067091906199494_wp
        Alpha(7) = 1.0220320315021205132346993060821205290267_wp
        Alpha(8) = 2.2049227056952739652734285513702161551919_wp
        Alpha(9) = 4.7164964053301330262207846644173514505383_wp

        Xi(1) = 1.015494846734757918623598405006_wp
        Xi(2) = 1.142924036879366089781116799795_wp
        Xi(3) = 1.416508618588864020534749676994_wp
        Xi(4) = 1.875751259452837171729823462218_wp
        Xi(5) = 2.588270227246376346558892622696_wp
        Xi(6) = 3.665260371207954169384471421189_wp
        Xi(7) = 5.286676676004319297592914939088_wp
        Xi(8) = 7.742061977409237162172511714786_wp
        Xi(9) = 11.501099651012934720235858510051_wp
        Xi(10) = 17.337630763701917932170792902014_wp
        Xi(11) = 26.551512591897243413183171689695_wp
        Xi(12) = 41.375925788490504571293104874030_wp
        Xi(13) = 65.740140071894493468962483717632_wp
        Xi(14) = 106.725407085254902904425389209564_wp
        Xi(15) = 177.257642594410686792683051749009_wp
        Xi(16) = 300.103328267503049187059360747298_wp
        Xi(17) = 504.714364302674262402481630829243_wp
        Xi(18) = 750.435276622015730974002423181446_wp

      case (20)
        ! ----- K = 09 , R = 9E2 -----

        RIni = 9.0e2_wp

        Omega(1) = 0.0030977424691614607340063927698459167459_wp
        Omega(2) = 0.0102316638905545772966616227712954856344_wp
        Omega(3) = 0.0279930901026207251671101914475281446926_wp
        Omega(4) = 0.0708952689284168618234554734569563549940_wp
        Omega(5) = 0.1672877491668182620230403978567323974858_wp
        Omega(6) = 0.3723968344810377277521139249838455498320_wp
        Omega(7) = 0.7929727209927074070040797781899755136692_wp
        Omega(8) = 1.6537803753979151391746704735652429008042_wp
        Omega(9) = 3.6802970160533538548051912808745100846863_wp

        Alpha(1) = 0.0011602484211060985717217235472717429268_wp
        Alpha(2) = 0.0073321764357024223340904170269360928813_wp
        Alpha(3) = 0.0250950982980019906870734758275176545794_wp
        Alpha(4) = 0.0715519741026311104838022840446409844617_wp
        Alpha(5) = 0.1844298926633792353379712181116190095054_wp
        Alpha(6) = 0.4418305063609447410808393763570478540714_wp
        Alpha(7) = 0.9999272003534077926873227526449738888914_wp
        Alpha(8) = 2.1709623019090606504347823468847877848020_wp
        Alpha(9) = 4.6683738668041492225410649119510253512999_wp

        Xi(1) = 1.015715269904661042610176380219_wp
        Xi(2) = 1.145013877896202366670971861673_wp
        Xi(3) = 1.422912376064371018724932782540_wp
        Xi(4) = 1.890168336484572671955632638952_wp
        Xi(5) = 2.616730005799423755237856359379_wp
        Xi(6) = 3.718043628867526893388290032405_wp
        Xi(7) = 5.381674116079378136345140459795_wp
        Xi(8) = 7.910768261877659578238525250882_wp
        Xi(9) = 11.799680026576712552240089326716_wp
        Xi(10) = 17.867848238659794161270477630410_wp
        Xi(11) = 27.501306142315619481658606204633_wp
        Xi(12) = 43.100648356764131651636118292004_wp
        Xi(13) = 68.930828688179012590786509662166_wp
        Xi(14) = 112.772932024889469160289223736982_wp
        Xi(15) = 189.076604838093731836279332014783_wp
        Xi(16) = 324.032486664164790096664958696238_wp
        Xi(17) = 554.184538502184091124380671544714_wp
        Xi(18) = 840.601406203501388902932234259424_wp

      case (21)
        ! ----- K = 09 , R = 1E3 -----

        RIni = 1.0e3_wp

        Omega(1) = 0.0028624090304025796460983327093585693035_wp
        Omega(2) = 0.0096180997032853235358422147593948636768_wp
        Omega(3) = 0.0267727106569027455297690373803298768962_wp
        Omega(4) = 0.0686213572099158147516839018065226696308_wp
        Omega(5) = 0.1633526775549459982288420800689010548012_wp
        Omega(6) = 0.3661615356237251258893824340523437399497_wp
        Omega(7) = 0.7840601806864651043913611028202126362885_wp
        Omega(8) = 1.6424763479445014240521577564280164551747_wp
        Omega(9) = 3.6673470190152673215452233668187886905798_wp

        Alpha(1) = 0.0010697132697980306932461928089324321167_wp
        Alpha(2) = 0.0068276991894412194727555210325975476593_wp
        Alpha(3) = 0.0237021379736091671702849807343294052941_wp
        Alpha(4) = 0.0684570314572635938460301412156194800218_wp
        Alpha(5) = 0.1782853335724231008961636682497164940742_wp
        Alpha(6) = 0.4306491453842748466022787023899098812763_wp
        Alpha(7) = 0.9811811804065308630713911075549304996457_wp
        Alpha(8) = 2.1420504811752469024200423719506147790526_wp
        Alpha(9) = 4.6272998769299419703944498127867746006814_wp

        Xi(1) = 1.015907400071548715553802766376_wp
        Xi(2) = 1.146836899869768950548445940463_wp
        Xi(3) = 1.428506324684849970426195331363_wp
        Xi(4) = 1.902786202134254691319646679570_wp
        Xi(5) = 2.641698869403860285140148156380_wp
        Xi(6) = 3.764492866971857517720961849861_wp
        Xi(7) = 5.465568235491984114878677436522_wp
        Xi(8) = 8.060363154364896828493747271693_wp
        Xi(9) = 12.065651317312179709000108385197_wp
        Xi(10) = 18.342605257009002163448574407312_wp
        Xi(11) = 28.356729072394385633010260860942_wp
        Xi(12) = 44.664383316634532559985526489754_wp
        Xi(13) = 71.845980142920193092792580102923_wp
        Xi(14) = 118.348121088373596204723625646693_wp
        Xi(15) = 200.092162779904987002232985560113_wp
        Xi(16) = 346.650998970642015284404635622195_wp
        Xi(17) = 601.917721942256615852429035840032_wp
        Xi(18) = 930.129063527018485946218362414584_wp

      case (22)
        ! ----- K = 09 , R = 2E3 -----

        RIni = 2.0e3_wp

        Omega(1) = 0.0017219984012311614023473319012857984944_wp
        Omega(2) = 0.0065884426254116958339659862291248781929_wp
        Omega(3) = 0.0204962747939638545263652761457828788139_wp
        Omega(4) = 0.0565294509352286292890293201152118740538_wp
        Omega(5) = 0.1419270750951637811022431473473304208710_wp
        Omega(6) = 0.3315985863927448277859409697665427074753_wp
        Omega(7) = 0.7339849281745116267323264525046511153050_wp
        Omega(8) = 1.5784277751334237942845012403481064211519_wp
        Omega(9) = 3.5938760022078233764050020360159010124335_wp

        Alpha(1) = 0.0006313454889490319018397847304711195671_wp
        Alpha(2) = 0.0043668469857874161270711748684958397604_wp
        Alpha(3) = 0.0167545678178722372817405520881162139801_wp
        Alpha(4) = 0.0525840159291544383493056946665689110887_wp
        Alpha(5) = 0.1459750048379582043313806502538021447890_wp
        Alpha(6) = 0.3706381471395138468101146389521716173476_wp
        Alpha(7) = 0.8789214132697631501897964079184077945683_wp
        Alpha(8) = 1.9823948779811632675204591258477648807457_wp
        Alpha(9) = 4.3986398388485846411809221390143420649110_wp

        Xi(1) = 1.017049176702895268119275495877_wp
        Xi(2) = 1.157698409290087672310690103394_wp
        Xi(3) = 1.461985362290557343041552673668_wp
        Xi(4) = 1.978766373725918028452300245768_wp
        Xi(5) = 2.793256426785199025971881403585_wp
        Xi(6) = 4.049236578360383199005206877885_wp
        Xi(7) = 5.985905338379571236865489547441_wp
        Xi(8) = 9.000728837074756394191765718205_wp
        Xi(9) = 13.763279935388228015363809575433_wp
        Xi(10) = 21.425935302543523047758000643626_wp
        Xi(11) = 34.023698458379380317462281624330_wp
        Xi(12) = 55.263913287691619299901724104984_wp
        Xi(13) = 92.143612152471854152835373241714_wp
        Xi(14) = 158.428881638126718631176537144256_wp
        Xi(15) = 282.483090323835139001662497548750_wp
        Xi(16) = 524.843340620022861309212913738520_wp
        Xi(17) = 1008.768050953667418712100811717391_wp
        Xi(18) = 1793.900715959898712004161325239693_wp

      case (23)
        ! ----- K = 09 , R = 3E3 -----

        RIni = 3.0e3_wp

        Omega(1) = 0.0012962831308852621829681126030056614873_wp
        Omega(2) = 0.0054180644503889189698053654303810944270_wp
        Omega(3) = 0.0179136340878167857434044487469071427199_wp
        Omega(4) = 0.0513232221538696522669480409167830359252_wp
        Omega(5) = 0.1324094929296019472031008951118380423395_wp
        Omega(6) = 0.3158801151655616752330930224745841883305_wp
        Omega(7) = 0.7108062282245082569428692997881569226593_wp
        Omega(8) = 1.5484532018565422108527518396670075162547_wp
        Omega(9) = 3.5594302119364723932699867692264206198161_wp

        Alpha(1) = 0.0004677826770463522486164677222216998209_wp
        Alpha(2) = 0.0034369131730817087311227362670523755028_wp
        Alpha(3) = 0.0140274848449730914044667265394572552850_wp
        Alpha(4) = 0.0460846626837133622854461925677505007570_wp
        Alpha(5) = 0.1322703437013994585806381937276299254336_wp
        Alpha(6) = 0.3444646922297356469088933095257232253061_wp
        Alpha(7) = 0.8333436224059946878105382961354763438067_wp
        Alpha(8) = 1.9100798947374523063889167895368359495478_wp
        Alpha(9) = 4.2939725970702426905561965853053152386565_wp

        Xi(1) = 1.017616265156594057691638588459_wp
        Xi(2) = 1.163110781763274042535302732482_wp
        Xi(3) = 1.478765111309849790942327896204_wp
        Xi(4) = 2.017148114569818525805902642567_wp
        Xi(5) = 2.870600924325366836960607419549_wp
        Xi(6) = 4.196401930766708810328219358965_wp
        Xi(7) = 6.258870094387393507272265846808_wp
        Xi(8) = 9.502540637572148627190693481737_wp
        Xi(9) = 14.686968839704956300319660389420_wp
        Xi(10) = 23.141109520577700826848843718153_wp
        Xi(11) = 37.257002672376713498653133527228_wp
        Xi(12) = 61.491445268627430022212454474584_wp
        Xi(13) = 104.487770788361471641092492745884_wp
        Xi(14) = 183.830182742659324748868421295356_wp
        Xi(15) = 337.441768310797018143309955462428_wp
        Xi(16) = 651.933533159956624059017116223913_wp
        Xi(17) = 1329.806380711968678443035685177165_wp
        Xi(18) = 2607.084823743725834788165229838341_wp

      case (24)
        ! ----- K = 09 , R = 4E3 -----

        RIni = 4.0e3_wp

        Omega(1) = 0.0010688850352942277345185288013780511296_wp
        Omega(2) = 0.0047768063467026780887263930162195402218_wp
        Omega(3) = 0.0164437571556601970830269751969177072226_wp
        Omega(4) = 0.0482872060920564158153134217549284912252_wp
        Omega(5) = 0.1267659789113546653697116306513947847634_wp
        Omega(6) = 0.3064418543767718427985220774889540962249_wp
        Omega(7) = 0.6967562347704147392309613484595587351578_wp
        Omega(8) = 1.5301758899530688042261300507718146945990_wp
        Omega(9) = 3.5384053951731580640004126703246356555610_wp

        Alpha(1) = 0.0003803762514503546170662433837566032691_wp
        Alpha(2) = 0.0029354867547009724771518262816452637942_wp
        Alpha(3) = 0.0125188405641583705003004340500227797861_wp
        Alpha(4) = 0.0423977620853282027837060589453566095131_wp
        Alpha(5) = 0.1243410987679817497871871158354739606011_wp
        Alpha(6) = 0.3290894034731464541068853002503402649381_wp
        Alpha(7) = 0.8062550457285698337446186445554019428528_wp
        Alpha(8) = 1.8667285745423568974398972897255077896261_wp
        Alpha(9) = 4.2308745838119774058602717836663487105398_wp

        Xi(1) = 1.017972913409440976432313030742_wp
        Xi(2) = 1.166520663629237118219354740756_wp
        Xi(3) = 1.489369630213643176512051946592_wp
        Xi(4) = 2.041508815880282935858666681916_wp
        Xi(5) = 2.919964474799958625535548084606_wp
        Xi(6) = 4.290980070192719836568578939406_wp
        Xi(7) = 6.435732295665031909258357867287_wp
        Xi(8) = 9.830756490223758554437838430573_wp
        Xi(9) = 15.297664105121758036827384064082_wp
        Xi(10) = 24.289225229657723223453125172000_wp
        Xi(11) = 39.452631126727291607098058179304_wp
        Xi(12) = 65.792145778324712890317016444897_wp
        Xi(13) = 113.186204073241912321234892147004_wp
        Xi(14) = 202.174048148890269122057716799645_wp
        Xi(15) = 378.387710310245006184448612884808_wp
        Xi(16) = 750.653484216962727126887244821773_wp
        Xi(17) = 1595.695540737967720867906962212146_wp
        Xi(18) = 3375.521036011437759905007283123268_wp

      case (25)
        ! ----- K = 09 , R = 5E3 -----

        RIni = 5.0e3_wp

        Omega(1) = 0.0009260627770325222336527758290516260109_wp
        Omega(2) = 0.0043656639391461567718954344709259007828_wp
        Omega(3) = 0.0154761374940805984633387460194797746738_wp
        Omega(4) = 0.0462564321296459549713955307359558588587_wp
        Omega(5) = 0.1229495139573777406216811489136908619457_wp
        Omega(6) = 0.3000063118609164515841920828354005834626_wp
        Omega(7) = 0.6871166160166532573721001608824110462592_wp
        Omega(8) = 1.5175870789580829839544862114308898526360_wp
        Omega(9) = 3.5239144130064043321758082694117319988436_wp

        Alpha(1) = 0.0003254404360101518881557299786881284898_wp
        Alpha(2) = 0.0026180514674484443450536228902708613120_wp
        Alpha(3) = 0.0115451773269337696752609781492004970005_wp
        Alpha(4) = 0.0399764124320777946288384390588488415119_wp
        Alpha(5) = 0.1190640832195332738838947778114984998865_wp
        Alpha(6) = 0.3187534379658245386546591704224340446672_wp
        Alpha(7) = 0.7879045136257066413686123151904894257314_wp
        Alpha(8) = 1.8371950908189229402555156522858226253447_wp
        Alpha(9) = 4.1877312846909169937598127475553155818488_wp

        Xi(1) = 1.018223491055936285850161415745_wp
        Xi(2) = 1.168919254817437955126270543449_wp
        Xi(3) = 1.496844550309035365475221479592_wp
        Xi(4) = 2.058728171771944849208213490499_wp
        Xi(5) = 2.954985959115944692569485918732_wp
        Xi(6) = 4.358387310953027374196139254181_wp
        Xi(7) = 6.562468157680101060388877520779_wp
        Xi(8) = 10.067425096070548431147206880354_wp
        Xi(9) = 15.741193797883095929421304326468_wp
        Xi(10) = 25.130010411199288482622660190202_wp
        Xi(11) = 41.076166787470114780123076059226_wp
        Xi(12) = 69.008812964846139359487153797090_wp
        Xi(13) = 119.782851976330407949011824797481_wp
        Xi(14) = 216.324815207572608483088671960104_wp
        Xi(15) = 410.676067869934812465393392244550_wp
        Xi(16) = 830.866962130459026758533269685358_wp
        Xi(17) = 1821.993947312338340438486739003565_wp
        Xi(18) = 4103.125862949434461413034114229959_wp

      case (26)
        ! ----- K = 09 , R = 6E3 -----

        RIni = 6.0e3_wp

        Omega(1) = 0.0008275446967731432786926443147170612269_wp
        Omega(2) = 0.0040770928209128811744981358677203386875_wp
        Omega(3) = 0.0147834768067865480155659491968977992116_wp
        Omega(4) = 0.0447858538699645466399227363568646964609_wp
        Omega(5) = 0.1201638766104322943376720399244472758937_wp
        Omega(6) = 0.2952809718970280440132614163672997165122_wp
        Omega(7) = 0.6800069561584170269724625157437003508676_wp
        Omega(8) = 1.5082761105851120459874631363206276546407_wp
        Omega(9) = 3.5131912499637148285854676421635645056085_wp

        Alpha(1) = 0.0002875179253957972909773293786282050144_wp
        Alpha(2) = 0.0023975729163532282753073832408038934538_wp
        Alpha(3) = 0.0108584821309771691825234673188516065778_wp
        Alpha(4) = 0.0382462256420431207152676837679461385733_wp
        Alpha(5) = 0.1152564953442573633031497694345635096624_wp
        Alpha(6) = 0.3112408729002721530937109611913982121223_wp
        Alpha(7) = 0.7744924306712252477856266164035758947648_wp
        Alpha(8) = 1.8155216145202244967761170957309957429970_wp
        Alpha(9) = 4.1559868654318949401059446469020031145192_wp

        Xi(1) = 1.018411464449650201402608473700_wp
        Xi(2) = 1.170720123158882433568248049305_wp
        Xi(3) = 1.502465139934904708317493782221_wp
        Xi(4) = 2.071702288831712671264562164453_wp
        Xi(5) = 2.981443669360396931788254049067_wp
        Xi(6) = 4.409480226859837182894902296582_wp
        Xi(7) = 6.658908402585880750845781506086_wp
        Xi(8) = 10.248336902673067018779140546769_wp
        Xi(9) = 16.082016847958395967116418390219_wp
        Xi(10) = 25.780032612769895003848596992668_wp
        Xi(11) = 42.340290267741560462783079188398_wp
        Xi(12) = 71.534627945986173996884005532593_wp
        Xi(13) = 125.016349036148568474313691467614_wp
        Xi(14) = 227.695969317093756867365073048859_wp
        Xi(15) = 437.058285546168342500239489822889_wp
        Xi(16) = 897.938755526446165500242813095610_wp
        Xi(17) = 2018.127927928557927450547992975771_wp
        Xi(18) = 4792.881465381862065910922865441535_wp

      case (27)
        ! ----- K = 09 , R = 7E3 -----

        RIni = 7.0e3_wp

        Omega(1) = 0.0007552936323448430041703487061287985993_wp
        Omega(2) = 0.0038622717168615726822444232354050730471_wp
        Omega(3) = 0.0142598450027955562363895375410649535297_wp
        Omega(4) = 0.0436642852027035399011651271850009692344_wp
        Omega(5) = 0.1180264413253419070705331657411552015446_wp
        Omega(6) = 0.2916386422069088975826678389324797535664_wp
        Omega(7) = 0.6745080482776006745187537261276844446911_wp
        Omega(8) = 1.5010591549296192516709663733465163204528_wp
        Omega(9) = 3.5048765058635483835708379629636510799173_wp

        Alpha(1) = 0.0002596868790846979322642372917410644173_wp
        Alpha(2) = 0.0022348903483392374692937683679574978335_wp
        Alpha(3) = 0.0103453966022495648471076889335362736233_wp
        Alpha(4) = 0.0369401084153355743395249870253405077847_wp
        Alpha(5) = 0.1123604405679129684885266850236096303206_wp
        Alpha(6) = 0.3054946294132511537728919687184614417674_wp
        Alpha(7) = 0.7641900750527943561455149201755432386562_wp
        Alpha(8) = 1.7988216034442439343740346990863088194601_wp
        Alpha(9) = 4.1314779669129050722158058572830441335100_wp

        Xi(1) = 1.018558731978950639162663505122_wp
        Xi(2) = 1.172131936927446008915414121798_wp
        Xi(3) = 1.506876523430133237350068975413_wp
        Xi(4) = 2.081901165189758657518906415085_wp
        Xi(5) = 3.002283519752387932252174485015_wp
        Xi(6) = 4.449827618791080872072407625950_wp
        Xi(7) = 6.735296143463850805557097123355_wp
        Xi(8) = 10.392136042058479286284700116028_wp
        Xi(9) = 16.354011451083167446446386961156_wp
        Xi(10) = 26.301210973404164810904304339090_wp
        Xi(11) = 43.359455691751780596898679931428_wp
        Xi(12) = 73.584391609781571862158866537129_wp
        Xi(13) = 129.297798148595386447112787209335_wp
        Xi(14) = 237.092743920405970126097372485674_wp
        Xi(15) = 459.150124798627372768589083307234_wp
        Xi(16) = 955.154559206768593582204118774825_wp
        Xi(17) = 2190.358162303570911166872292596963_wp
        Xi(18) = 5447.138394667314925001733172393870_wp

      case (28)
        ! ----- K = 09 , R = 8E3 -----

        RIni = 8.0e3_wp

        Omega(1) = 0.0006999750349259222250828226387191211000_wp
        Omega(2) = 0.0036956292786038210907891552511189692254_wp
        Omega(3) = 0.0138485816729835371586079546118996930204_wp
        Omega(4) = 0.0427771976018919282652545887923878353831_wp
        Omega(5) = 0.1163277118663094138832354677848268664775_wp
        Omega(6) = 0.2887334228096178135637445844041693021609_wp
        Omega(7) = 0.6701100499979678797664479905638046375316_wp
        Omega(8) = 1.4952771916809653082172312799080771128502_wp
        Omega(9) = 3.4982130048175731091425477714551561803091_wp

        Alpha(1) = 0.0002383647610636543066935782918757368021_wp
        Alpha(2) = 0.0021096524885159010384065933355404731664_wp
        Alpha(3) = 0.0099462230796899336260355408041722302670_wp
        Alpha(4) = 0.0359154574427805942811160831024119843846_wp
        Alpha(5) = 0.1100747420247687572523597084039881366380_wp
        Alpha(6) = 0.3009391008073506232193584930545426914250_wp
        Alpha(7) = 0.7559949292957711996566484702331223388683_wp
        Alpha(8) = 1.7855045682887562425713787184200498359132_wp
        Alpha(9) = 4.1119029053730469726486707049417645976064_wp

        Xi(1) = 1.018677716815463165224257824359_wp
        Xi(2) = 1.173273207018096588190834983489_wp
        Xi(3) = 1.510445803105953123802833448774_wp
        Xi(4) = 2.090163365282045846837294389431_wp
        Xi(5) = 3.019193541099971872087442958943_wp
        Xi(6) = 4.482633558072083487452963446174_wp
        Xi(7) = 6.797553391683034926338408121538_wp
        Xi(8) = 10.509657889464081687372343232312_wp
        Xi(9) = 16.577024137863919182184213596543_wp
        Xi(10) = 26.730143399664715267438630519337_wp
        Xi(11) = 44.201935389617642624404680873340_wp
        Xi(12) = 75.287795912764834453667628366702_wp
        Xi(13) = 132.878975976798816921453649797513_wp
        Xi(14) = 245.017212574918897008413587457198_wp
        Xi(15) = 477.983292761807701531928671556670_wp
        Xi(16) = 1004.685293736425898469821760272680_wp
        Xi(17) = 2343.085202243852703141158144717338_wp
        Xi(18) = 6067.768060278267428397924732053070_wp

      case (29)
        ! ----- K = 09 , R = 9E3 -----

        RIni = 9.0e3_wp

        Omega(1) = 0.0006562555949134862097456894652110909183_wp
        Omega(2) = 0.0035623967101711022665845959205357940114_wp
        Omega(3) = 0.0135163916656435436341326689763109314413_wp
        Omega(4) = 0.0420565524273398218171177154189877001045_wp
        Omega(5) = 0.1149422647181856013597826653149791553687_wp
        Omega(6) = 0.2863569692532348407516991362076197447095_wp
        Omega(7) = 0.6665045419456762991429130127407631789538_wp
        Omega(8) = 1.4905304766796539943287530638471594102157_wp
        Omega(9) = 3.4927411942424347814258045463731150448439_wp

        Alpha(1) = 0.0002215041451664834586830238490322980693_wp
        Alpha(2) = 0.0020101870345634199486422168199478699435_wp
        Alpha(3) = 0.0096263287145913500154076488345145090619_wp
        Alpha(4) = 0.0350886056219235608138952604850846483941_wp
        Alpha(5) = 0.1082211098172572829307475604510013766912_wp
        Alpha(6) = 0.2972311371099197367467124875073380962931_wp
        Alpha(7) = 0.7493060755473707331363562855752746827420_wp
        Alpha(8) = 1.7746133568189385439030011926142549327778_wp
        Alpha(9) = 4.0958729118449167887726358738831322625629_wp

        Xi(1) = 1.018776069157263152419046203789_wp
        Xi(2) = 1.174216979156755687858058878614_wp
        Xi(3) = 1.513399629431814344404649830267_wp
        Xi(4) = 2.097007699319862089040386599059_wp
        Xi(5) = 3.033220597771280707200328152773_wp
        Xi(6) = 4.509890311786309907964714893680_wp
        Xi(7) = 6.849382021323657984603755499720_wp
        Xi(8) = 10.607718344733282039848987210462_wp
        Xi(9) = 16.763597330244027922355187776304_wp
        Xi(10) = 27.090082887005839542768348415791_wp
        Xi(11) = 44.911495157432774393002627988380_wp
        Xi(12) = 76.728678917868063699692093848626_wp
        Xi(13) = 135.924664766140963156892951246846_wp
        Xi(14) = 251.802780753667186458555349304334_wp
        Xi(15) = 494.256277307559753447785411140103_wp
        Xi(16) = 1048.041669105673348250107324020064_wp
        Xi(17) = 2479.529732791401696267996612732532_wp
        Xi(18) = 6656.264013706788183810658665606752_wp

      case (30)
        ! ----- K = 09 , R = 1E4 -----

        RIni = 1.0e4_wp

        Omega(1) = 0.0006208594515187709415282896912614829521_wp
        Omega(2) = 0.0034534126161566192116306856473018083165_wp
        Omega(3) = 0.0132423168362171444399849953854619855065_wp
        Omega(4) = 0.0414591277165024014841659873858992213513_wp
        Omega(5) = 0.1137899306668965527603314298599190834693_wp
        Omega(6) = 0.2843755027087346596648478880275945357425_wp
        Omega(7) = 0.6634927640743865600138698401799786097399_wp
        Omega(8) = 1.4865608254662938828103563304594558758254_wp
        Omega(9) = 3.4881641795154761623472261522493909069453_wp

        Alpha(1) = 0.0002078469652528859301345068560495227272_wp
        Alpha(2) = 0.0019292989209037795216477166365849627017_wp
        Alpha(3) = 0.0093641492864657089188382653455189341685_wp
        Alpha(4) = 0.0344069545580813640534196835157887406353_wp
        Alpha(5) = 0.1066866208021499817933889359133914354061_wp
        Alpha(6) = 0.2941521685207762780499100851994143113188_wp
        Alpha(7) = 0.7437390948258375387058989602895309189989_wp
        Alpha(8) = 1.7655336768516124521660737300976506958250_wp
        Alpha(9) = 4.0824948352969070938280060634184565060423_wp

        Xi(1) = 1.018858789375025899808999141349_wp
        Xi(2) = 1.175011031484274259969156906092_wp
        Xi(3) = 1.515886425956611331731897773789_wp
        Xi(4) = 2.102774530086370540936033246560_wp
        Xi(5) = 3.045052594955341328242237608492_wp
        Xi(6) = 4.532913593446208804383507073865_wp
        Xi(7) = 6.893232572595227878112233987906_wp
        Xi(8) = 10.690843024689531597613834090765_wp
        Xi(9) = 16.922099277879264508581136361443_wp
        Xi(10) = 27.396661436507848682436749854219_wp
        Xi(11) = 45.517687762936599963486949960156_wp
        Xi(12) = 77.964215296603784070839093800487_wp
        Xi(13) = 138.548135610549745161401702375770_wp
        Xi(14) = 257.681395599821175701471531738207_wp
        Xi(15) = 508.463092995186691336328976831282_wp
        Xi(16) = 1086.316738391781835049165749751410_wp
        Xi(17) = 2602.119680662271022031362122106657_wp
        Xi(18) = 7213.819733283016149893995816455572_wp

      case (31)
        ! ----- K = 09 , R = 3E4 (Best : R = 28393) -----

        RIni = 3.0e4_wp

        Omega(1) = 0.0004354299056155451189326690961860569229_wp
        Omega(2) = 0.0028608520316584030092596681186037366196_wp
        Omega(3) = 0.0117112913825153417034176557554037145792_wp
        Omega(4) = 0.0380713875095065937734207205112024752225_wp
        Omega(5) = 0.1071875873349240602293774501929668474531_wp
        Omega(6) = 0.2729343308454240926079308049434501981523_wp
        Omega(7) = 0.6460013499115237401077861678633240671843_wp
        Omega(8) = 1.4634218815677372793051733146363346804719_wp
        Omega(9) = 3.4614670015318758573060614613936536443362_wp

        Alpha(1) = 0.0001362078758772922180869218623951687197_wp
        Alpha(2) = 0.0014984233172990983195404520551471949652_wp
        Alpha(3) = 0.0079304843136686811123963131500702594678_wp
        Alpha(4) = 0.0306088525260754423692717909877281101672_wp
        Alpha(5) = 0.0980233175125910484790222806927850740522_wp
        Alpha(6) = 0.2766007114067537993448361866022366939433_wp
        Alpha(7) = 0.7117747229082216156863423262013412795568_wp
        Alpha(8) = 1.7131253759530024614919987979710924719257_wp
        Alpha(9) = 4.0050154343014464928054985115579711418832_wp

        Xi(1) = 1.019349605390273929466163349566_wp
        Xi(2) = 1.179727839421699866128898093187_wp
        Xi(3) = 1.530686767100153370941943709038_wp
        Xi(4) = 2.137188978850325381123390067195_wp
        Xi(5) = 3.115909405473552945846013595421_wp
        Xi(6) = 4.671400470121846554196615164400_wp
        Xi(7) = 7.158377525361323866562313078532_wp
        Xi(8) = 11.196526468244599649727588630554_wp
        Xi(9) = 17.893206861623897827029416873756_wp
        Xi(10) = 29.290661647905315439349704931260_wp
        Xi(11) = 49.300006109312959838575363136215_wp
        Xi(12) = 85.766036314244301040632389998564_wp
        Xi(13) = 155.365014625779427434593138457330_wp
        Xi(14) = 296.099822100602221353060627961895_wp
        Xi(15) = 603.777482619712320743854405691309_wp
        Xi(16) = 1353.360691141729413522476477282908_wp
        Xi(17) = 3513.499234321727268337554050958715_wp
        Xi(18) = 12351.225064419754213318469737714622_wp

    end select

  case (10)
    ! ===== K = 10 =====

    select case (InitR)

      case default !(1:3)
        ! ----- K = 10 , R = 1E1 -----

        RIni = 1.0e1_wp

        Omega(1) = 0.0851548884512817157106711121739017755772_wp
        Omega(2) = 0.2041221377375185690611911551450852186917_wp
        Omega(3) = 0.3387624704041514381172919834206425093726_wp
        Omega(4) = 0.5020218012767448245844623122557948136091_wp
        Omega(5) = 0.7136389791481699146086931095922523127228_wp
        Omega(6) = 1.0042283383685008889794898490066543672583_wp
        Omega(7) = 1.4225951252649079001484644368957788174157_wp
        Omega(8) = 2.0553775439606230667627384534057455311995_wp
        Omega(9) = 3.1013168392332142260427868540872964331356_wp
        Omega(10) = 5.3215604240635308100348943582957872422412_wp

        Alpha(1) = 0.0330697624272925114554972508829067123770_wp
        Alpha(2) = 0.1768078437824842392430508761225205205392_wp
        Alpha(3) = 0.4464853448677116219906260696781430397095_wp
        Alpha(4) = 0.8637994095087873673501251137185619199954_wp
        Alpha(5) = 1.4665304201175292089440183040949250425911_wp
        Alpha(6) = 2.3172114001372251723340212059554232837399_wp
        Alpha(7) = 3.5172089429912433802046661801554705562012_wp
        Alpha(8) = 5.2328732180584053819763501458339760574745_wp
        Alpha(9) = 7.7617287680176058257443261201302675544866_wp
        Alpha(10) = 11.7952949791468539881417187942247437604237_wp

        Xi(1) = 1.003934043651302852277057420238_wp
        Xi(2) = 1.035597560330198496896664250855_wp
        Xi(3) = 1.099946515856572986790003210178_wp
        Xi(4) = 1.199038529418928672736106844088_wp
        Xi(5) = 1.336001895201351444384015032529_wp
        Xi(6) = 1.515097369569101782649581111606_wp
        Xi(7) = 1.741798718468895162004433929681_wp
        Xi(8) = 2.022861086752043427612612891409_wp
        Xi(9) = 2.366318054679312713438343873662_wp
        Xi(10) = 2.781311568499321894504597296738_wp
        Xi(11) = 3.277602366254750317336313925054_wp
        Xi(12) = 3.864515159405216115267103771380_wp
        Xi(13) = 4.548945094352016048008802151514_wp
        Xi(14) = 5.331920668070182002442475299731_wp
        Xi(15) = 6.203227682007736412846121698550_wp
        Xi(16) = 7.134066726905839409336729239897_wp
        Xi(17) = 8.069204892498868964857694496828_wp
        Xi(18) = 8.922965548049153544293843332014_wp
        Xi(19) = 9.586495357975749287068878423668_wp
        Xi(20) = 9.952473091634462124335791521901_wp

      case (4)
        ! ----- K = 10 , R = 2E1 -----

        RIni = 2.0e1_wp

        Omega(1) = 0.0513720958134025681927881633315191578504_wp
        Omega(2) = 0.1248594580087193300568662937544139879265_wp
        Omega(3) = 0.2128344130226482852554382302345814537148_wp
        Omega(4) = 0.3287926283808711603196678341376468779345_wp
        Omega(5) = 0.4942264205009505737215338982970536108041_wp
        Omega(6) = 0.7420669272415499498969822245175720354382_wp
        Omega(7) = 1.1227892497020226877544810117903750779078_wp
        Omega(8) = 1.7234803772519538559099619434711314625019_wp
        Omega(9) = 2.7403911047158000403173305770110346202273_wp
        Omega(10) = 4.9150273553728394682060809373069787397981_wp

        Alpha(1) = 0.0199190782019326982112128465729483650648_wp
        Alpha(2) = 0.1072334417383124932971112143464953447847_wp
        Alpha(3) = 0.2744085978174513274971662440959718765043_wp
        Alpha(4) = 0.5421416040798437781158408688142458231596_wp
        Alpha(5) = 0.9483605618497397696719727888670092852408_wp
        Alpha(6) = 1.5578821538499603195637721597677227691747_wp
        Alpha(7) = 2.4764307129347023839812530487947128676751_wp
        Alpha(8) = 3.8758067491779373346227205443526031558577_wp
        Alpha(9) = 6.0582950800444962447845187458739246721962_wp
        Alpha(10) = 9.7118480399069814258672383999737576232292_wp

        Xi(1) = 1.005376604124994693195173367517_wp
        Xi(2) = 1.048763421907059670040891774523_wp
        Xi(3) = 1.137541925740631291530738367079_wp
        Xi(4) = 1.275782535377121582531242949266_wp
        Xi(5) = 1.469783188682559988444235876681_wp
        Xi(6) = 1.728376440612933768624023822547_wp
        Xi(7) = 2.063393850121227532639495150235_wp
        Xi(8) = 2.490252281796120580742437411459_wp
        Xi(9) = 3.028594043714212232952173731171_wp
        Xi(10) = 3.702877357639770333199921714673_wp
        Xi(11) = 4.542709189548428958763809371035_wp
        Xi(12) = 5.582457295094456607795901925684_wp
        Xi(13) = 6.859182461836509940925066253214_wp
        Xi(14) = 8.407072372916607403726951819323_wp
        Xi(15) = 10.245285079537717073602798301124_wp
        Xi(16) = 12.354965727196505731813602757541_wp
        Xi(17) = 14.642575027180748620941885818070_wp
        Xi(18) = 16.896736001875174576492000255712_wp
        Xi(19) = 18.771523113469569298342176644212_wp
        Xi(20) = 19.856318263968247872161554390402_wp

      case (5)
        ! ----- K = 10 , R = 3E1 -----

        RIni = 3.0e1_wp

        Omega(1) = 0.0378829891316186436572269104899879366144_wp
        Omega(2) = 0.0930163245037145861547622782694588750019_wp
        Omega(3) = 0.1617466336454914180433906426004320167067_wp
        Omega(4) = 0.2576551952740756494650200408846885125058_wp
        Omega(5) = 0.4026941949745052461324956727395374400658_wp
        Omega(6) = 0.6301925993044387373483508629767868569616_wp
        Omega(7) = 0.9910799607619160759013239581083354323710_wp
        Omega(8) = 1.5734470164829399977312815450680716367060_wp
        Omega(9) = 2.5740012849105933854902683588505851730588_wp
        Omega(10) = 4.7268570117366334642008962596548826695653_wp

        Alpha(1) = 0.0146720893474109634543403883795431585924_wp
        Alpha(2) = 0.0793869117946712017544041301020385503762_wp
        Alpha(3) = 0.2051749781339107614426065309687352566925_wp
        Alpha(4) = 0.4118459242697055932040314796260815910500_wp
        Alpha(5) = 0.7367388171009950032040031198121710076521_wp
        Alpha(6) = 1.2445399693315783262552443044768324398319_wp
        Alpha(7) = 2.0412209987653119546650165583301372862479_wp
        Alpha(8) = 3.2995507853921115116319073834461050864775_wp
        Alpha(9) = 5.3237296721066788398116220726308256416814_wp
        Alpha(10) = 8.8014613611571692972784797248664290236775_wp

        Xi(1) = 1.006260876037949373641190720097_wp
        Xi(2) = 1.056862976860428054949256337913_wp
        Xi(3) = 1.160831214850607703750057231318_wp
        Xi(4) = 1.323797135747777582617627778472_wp
        Xi(5) = 1.554547200491692039627661392487_wp
        Xi(6) = 1.865583007695159876906249030171_wp
        Xi(7) = 2.273985700647530612452523746114_wp
        Xi(8) = 2.802556054918465656879827951897_wp
        Xi(9) = 3.481186170552955607380984925925_wp
        Xi(10) = 4.348427674928144312429395546005_wp
        Xi(11) = 5.453102342622122880935742195163_wp
        Xi(12) = 6.855472150034190314399351606767_wp
        Xi(13) = 8.626697477281858152456384125628_wp
        Xi(14) = 10.843790080905853050186049557624_wp
        Xi(15) = 13.574302827800757838619394313184_wp
        Xi(16) = 16.840507670545135452538909603959_wp
        Xi(17) = 20.549543164941929868083647647836_wp
        Xi(18) = 24.387232452140358875125603077549_wp
        Xi(19) = 27.729559759629049972215475428072_wp
        Xi(20) = 29.731006308352625576787997552231_wp

      case (6)
        ! ----- K = 10 , R = 4E1 -----

        RIni = 4.0e1_wp

        Omega(1) = 0.0304283542679826195679411611483278221613_wp
        Omega(2) = 0.0753370830029148128009893956025244676766_wp
        Omega(3) = 0.1331690520655589057678631209613584474027_wp
        Omega(4) = 0.2174481193685754137342536554400318493663_wp
        Omega(5) = 0.3501479372604658767023470822277175784620_wp
        Omega(6) = 0.5645330070305193969379224661064853307835_wp
        Omega(7) = 0.9118556551925829048335474247277687709357_wp
        Omega(8) = 1.4812981052443766873333644285004595531063_wp
        Omega(9) = 2.4704999769201472898348670437229657181888_wp
        Omega(10) = 4.6095978515266311219961137268796846910845_wp

        Alpha(1) = 0.0117740251702337547500765916989873272058_wp
        Alpha(2) = 0.0639700951282584003517481079092998896840_wp
        Alpha(3) = 0.1666917616889322231515588312866782416677_wp
        Alpha(4) = 0.3390289700500114554242939851613769519645_wp
        Alpha(5) = 0.6176072358757308952085379738505821478611_wp
        Alpha(6) = 1.0663680454451372958072502550308513491473_wp
        Alpha(7) = 1.7906093738966222766832714352602806684445_wp
        Alpha(8) = 2.9631424352797224503787421046041572481045_wp
        Alpha(9) = 4.8893793130761661856753530486230374663137_wp
        Alpha(10) = 8.2575316823540801510969422416508223250275_wp

        Xi(1) = 1.006899144780989947909471959875_wp
        Xi(2) = 1.062723543495707518853447826057_wp
        Xi(3) = 1.177761340350092555961936113818_wp
        Xi(4) = 1.358932941180709253370557931806_wp
        Xi(5) = 1.617097610798962527950926870712_wp
        Xi(6) = 1.967858618396710338057341294071_wp
        Xi(7) = 2.432817040973496266770964568948_wp
        Xi(8) = 3.041253416943164042526501211761_wp
        Xi(9) = 3.832241560710106011329589836834_wp
        Xi(10) = 4.857251484427018241307738577461_wp
        Xi(11) = 6.183217811492185554851641615670_wp
        Xi(12) = 7.895646709574505373897823634621_wp
        Xi(13) = 10.100429441955376910766695175425_wp
        Xi(14) = 12.920925598076232635423588046564_wp
        Xi(15) = 16.482361549301608256296036714161_wp
        Xi(16) = 20.867101081829256971345021298703_wp
        Xi(17) = 26.013304987079913349265480881911_wp
        Xi(18) = 31.534033822387357665073537305034_wp
        Xi(19) = 36.516222241626955218735117725259_wp
        Xi(20) = 39.582863294995880239918184528847_wp

      case (7)
        ! ----- K = 10 , R = 5E1 -----

        RIni = 5.0e1_wp

        Omega(1) = 0.0256360764998129773236843712697075048368_wp
        Omega(2) = 0.0639288721978045180823380273793521411108_wp
        Omega(3) = 0.1146131231768703616989810330217824940746_wp
        Omega(4) = 0.1910990220813975077864205545652431794679_wp
        Omega(5) = 0.3152112267330760065394551494577513039985_wp
        Omega(6) = 0.5200333749196062467891679692577611149318_wp
        Omega(7) = 0.8571142202739547632855607317559076818725_wp
        Omega(8) = 1.4166433883580459902494161350539059185394_wp
        Omega(9) = 2.3972309879960180553724186092168224604393_wp
        Omega(10) = 4.5265143423641330715570529896751850174041_wp

        Alpha(1) = 0.0099118398530477141314073454003708008031_wp
        Alpha(2) = 0.0540446717644748341431780273846907647339_wp
        Alpha(3) = 0.1418346518220869974035791907795989175156_wp
        Alpha(4) = 0.2917769967006175313559059730783573627377_wp
        Alpha(5) = 0.5397886064780781045194420209032415414185_wp
        Alpha(6) = 0.9489117072037067764827059090926297812985_wp
        Alpha(7) = 1.6235559881076618259088117923738536774181_wp
        Alpha(8) = 2.7363110644426438824126923687884982427931_wp
        Alpha(9) = 4.5934743313426938901046503449521196671412_wp
        Alpha(10) = 7.8839620327921848055852116488040337571874_wp

        Xi(1) = 1.007397524921629083536830928658_wp
        Xi(2) = 1.067308174745297444981767531758_wp
        Xi(3) = 1.191052363596173063612151943325_wp
        Xi(4) = 1.386654455349809759155156607680_wp
        Xi(5) = 1.666761037061785107179709419789_wp
        Xi(6) = 2.049681437360226300995955095452_wp
        Xi(7) = 2.561013762777291654683387012170_wp
        Xi(8) = 3.235848668315498850378900130842_wp
        Xi(9) = 4.121612128283250076499905079075_wp
        Xi(10) = 5.281726961622831101501873707349_wp
        Xi(11) = 6.800205251162270153695438290598_wp
        Xi(12) = 8.786900637551380829903813207693_wp
        Xi(13) = 11.382148526181198749246403778557_wp
        Xi(14) = 14.757044191321142191826443212932_wp
        Xi(15) = 19.099655989294517563492625811250_wp
        Xi(16) = 24.564947661652084267072670975551_wp
        Xi(17) = 31.145527462279573546397037908662_wp
        Xi(18) = 38.410994668580101859406417119658_wp
        Xi(19) = 45.162035093038253464464704123316_wp
        Xi(20) = 49.415398342449165687562118165488_wp

      case (8)
        ! ----- K = 10 , R = 6E1 -----

        RIni = 6.0e1_wp

        Omega(1) = 0.0222692249040388280981964813642903777691_wp
        Omega(2) = 0.0558880663046408164463683867356769141566_wp
        Omega(3) = 0.1014636556552278832694230503719268199347_wp
        Omega(4) = 0.1722689305543891316927331350261631826015_wp
        Omega(5) = 0.2899119020042486984762783452929113536811_wp
        Omega(6) = 0.4872765569924375810494879962542214002497_wp
        Omega(7) = 0.8161879928937111060597772715929920650524_wp
        Omega(8) = 1.3677291102282897400713168600816516118357_wp
        Omega(9) = 2.3414239803231609166056514936826715711504_wp
        Omega(10) = 4.4631975323559974601697608598271926894085_wp

        Alpha(1) = 0.0086040646080336160600617306098980563434_wp
        Alpha(2) = 0.0470627087261883153550430774342272499666_wp
        Alpha(3) = 0.1242996631171070397297857661189635081200_wp
        Alpha(4) = 0.2583066428451812494902859601575428882825_wp
        Alpha(5) = 0.4843311800515679819837452230180119272518_wp
        Alpha(6) = 0.8645073678078467050963154538756327838200_wp
        Alpha(7) = 1.5023398624500846272903850642244094615307_wp
        Alpha(8) = 2.5701128719246770133483731557433316083916_wp
        Alpha(9) = 4.3748135182289043707513920278273644726141_wp
        Alpha(10) = 7.6060959473532058099966290942717250800342_wp

        Xi(1) = 1.007805439364083889943260463617_wp
        Xi(2) = 1.071066197625785616364020580438_wp
        Xi(3) = 1.201978120630536587599458686615_wp
        Xi(4) = 1.409534003621606906863673869257_wp
        Xi(5) = 1.707957933545320009394857807550_wp
        Xi(6) = 2.117969744747726989131039743164_wp
        Xi(7) = 2.668766664277351642348828919538_wp
        Xi(8) = 3.400728088857348286634663114825_wp
        Xi(9) = 4.368971642621038846063252769625_wp
        Xi(10) = 5.648069605065367200773945866032_wp
        Xi(11) = 7.338201033536233721787239714374_wp
        Xi(12) = 9.572659536432813427719867860333_wp
        Xi(13) = 12.525630984383066084814073448683_wp
        Xi(14) = 16.416380028637400057336948044906_wp
        Xi(15) = 21.498893484353131022590588372623_wp
        Xi(16) = 28.009491746673664291575334139850_wp
        Xi(17) = 36.013714095210152738024689256235_wp
        Xi(18) = 45.064676508108325404811100867164_wp
        Xi(19) = 53.686820000316714349142266371473_wp
        Xi(20) = 59.230909829919811071530233448357_wp

      case (9)
        ! ----- K = 10 , R = 7E1 -----

        RIni = 7.0e1_wp

        Omega(1) = 0.0197607384753493559913451122281302652084_wp
        Omega(2) = 0.0498802316575754729679071514114596652689_wp
        Omega(3) = 0.0915913970615895810505752150376679310284_wp
        Omega(4) = 0.1580208841785472934673680359130010231183_wp
        Omega(5) = 0.2705366518430575556019169330967955033884_wp
        Omega(6) = 0.4618333919085841772738786764618268421145_wp
        Omega(7) = 0.7839904120304717415165969562895753597331_wp
        Omega(8) = 1.3288768434592462668606197007115099495422_wp
        Omega(9) = 2.2968563750443367990107523812781664673821_wp
        Omega(10) = 4.4126139217534362389934454729001345185679_wp

        Alpha(1) = 0.0076300405792137536050681514282478801192_wp
        Alpha(2) = 0.0418550058743523850566224404207149589752_wp
        Alpha(3) = 0.1111879397803506057252460013928807569528_wp
        Alpha(4) = 0.2331854554385689059001597958631357698778_wp
        Alpha(5) = 0.4424728513495298915230314468205463640516_wp
        Alpha(6) = 0.8003165994032521342056024638011990646191_wp
        Alpha(7) = 1.4093598104639633360277845008923236491682_wp
        Alpha(8) = 2.4415582485058356129563328362763741097297_wp
        Alpha(9) = 4.2044568362569176688675609554479706275743_wp
        Alpha(10) = 7.3884269231485020785858697234260716868448_wp

        Xi(1) = 1.008150014524303845044397864861_wp
        Xi(2) = 1.074244696826699676161717689205_wp
        Xi(3) = 1.211241091789887549981345749028_wp
        Xi(4) = 1.428996356505864060186181130341_wp
        Xi(5) = 1.743149796067134401082905637015_wp
        Xi(6) = 2.176601195846597373894162896768_wp
        Xi(7) = 2.761831104162296402501977632404_wp
        Xi(8) = 3.544087200266895613682410770551_wp
        Xi(9) = 4.585635727511376983755314595825_wp
        Xi(10) = 5.971522290488774669780375203842_wp
        Xi(11) = 7.817294699650083355125379513950_wp
        Xi(12) = 10.278832317284462163561875591711_wp
        Xi(13) = 13.563459418203007213546418263661_wp
        Xi(14) = 17.938583992132021224449878182838_wp
        Xi(15) = 23.725978660722990391113151886771_wp
        Xi(16) = 31.249706910884162639613359502277_wp
        Xi(17) = 40.663158133347690439268751205759_wp
        Xi(18) = 51.527416320312516485879772432099_wp
        Xi(19) = 62.104689349878719990122322158754_wp
        Xi(20) = 69.031045631153914177158270604195_wp

      case (10)
        ! ----- K = 10 , R = 8E1 -----

        RIni = 8.0e1_wp

        Omega(1) = 0.0178119645779939205493930774737432187749_wp
        Omega(2) = 0.0452010429992078838318486470881918393161_wp
        Omega(3) = 0.0838686860067898385246972295381517881196_wp
        Omega(4) = 0.1467933491672222748726343855008025229836_wp
        Omega(5) = 0.2550997772518251667029417356635789815300_wp
        Omega(6) = 0.4413114002778232506502258808422922697901_wp
        Omega(7) = 0.7577390458093131011879303726708201338624_wp
        Omega(8) = 1.2969458549711413536094720133640123549412_wp
        Omega(9) = 2.2600630049093749383750523440639312866551_wp
        Omega(10) = 4.3708423865441739823364808437133888219250_wp

        Alpha(1) = 0.0068735846308498890855135895257971290206_wp
        Alpha(2) = 0.0378052651777488056238627801841634124003_wp
        Alpha(3) = 0.1009686477595863478192708337366489956821_wp
        Alpha(4) = 0.2135380826836232647045893395798543679120_wp
        Alpha(5) = 0.4095631635599569478529708571601375410864_wp
        Alpha(6) = 0.7494985795520889645558379466017129288957_wp
        Alpha(7) = 1.3351857641416567897864331460056064315722_wp
        Alpha(8) = 2.3382521089266642320102229657408088314696_wp
        Alpha(9) = 4.0667052456180286103097631933422917427379_wp
        Alpha(10) = 7.2115952445320296061714859803259969339706_wp

        Xi(1) = 1.008447777913406502949857090901_wp
        Xi(2) = 1.076994369922411397700462432070_wp
        Xi(3) = 1.219270759862891835112290450649_wp
        Xi(4) = 1.445915848636912761325523657518_wp
        Xi(5) = 1.773854178956438908403188403629_wp
        Xi(6) = 2.227978804389550490780078084718_wp
        Xi(7) = 2.843796420245597696835337941934_wp
        Xi(8) = 3.671075196579713069769401734277_wp
        Xi(9) = 4.778771840255771170651033497023_wp
        Xi(10) = 6.261827428427214680371759580879_wp
        Xi(11) = 8.250454285898860409299848672759_wp
        Xi(12) = 10.922326929336661857465606839668_wp
        Xi(13) = 14.517172348889335643795650554466_wp
        Xi(14) = 19.350245010161896179468055123607_wp
        Xi(15) = 25.812262812914589486454342015520_wp
        Xi(16) = 34.319794162066901945656871575352_wp
        Xi(17) = 45.126350830342776623838041416548_wp
        Xi(18) = 57.823101718997251711323892564565_wp
        Xi(19) = 70.426307830426652024924383255211_wp
        Xi(20) = 78.817056210728345402860561819125_wp

      case (11)
        ! ----- K = 10 , R = 9E1 -----

        RIni = 9.0e1_wp

        Omega(1) = 0.0162498298873089111255316576223894919906_wp
        Omega(2) = 0.0414415425972865049722947128341132838614_wp
        Omega(3) = 0.0776387647291330781353872053651699047805_wp
        Omega(4) = 0.1376735143936578371714031868155991844560_wp
        Omega(5) = 0.2424335958616652716941320191978981313241_wp
        Omega(6) = 0.4242897948094856199201425905886964073943_wp
        Omega(7) = 0.7357627394488079689547667394666774498546_wp
        Omega(8) = 1.2700319870608946222576701723383507669496_wp
        Omega(9) = 2.2289313489910051518591571628746805799892_wp
        Omega(10) = 4.3354907748034642064709209829942437863792_wp

        Alpha(1) = 0.0062673829680089525943469932686702428271_wp
        Alpha(2) = 0.0345560641043288385510564176411607562045_wp
        Alpha(3) = 0.0927525124197384534383741330754169496231_wp
        Alpha(4) = 0.1976907281888568097740627171887695112673_wp
        Alpha(5) = 0.3828877720426687085257793502490741843758_wp
        Alpha(6) = 0.7080447357505527855082473609460436136942_wp
        Alpha(7) = 1.2742625091008633055836979375285977766907_wp
        Alpha(8) = 2.2528493904765100892383794528583962346602_wp
        Alpha(9) = 3.9522041796110818614919807378882410375809_wp
        Alpha(10) = 7.0640128564798895837231751393403555994155_wp

        Xi(1) = 1.008709550799474295615203167653_wp
        Xi(2) = 1.079414015510788574540509965338_wp
        Xi(3) = 1.226349205084010790772590382414_wp
        Xi(4) = 1.460868137240908749417862833031_wp
        Xi(5) = 1.801074201916648020462517754137_wp
        Xi(6) = 2.273700068589444158980231169842_wp
        Xi(7) = 2.917062160450168891876951415831_wp
        Xi(8) = 3.785154229053986666645587044755_wp
        Xi(9) = 4.953234030557328548306783577715_wp
        Xi(10) = 6.525632508391738927349540189127_wp
        Xi(11) = 8.646604435005628950389189313341_wp
        Xi(12) = 11.514889775118924520463936644887_wp
        Xi(13) = 15.401887752234283438142192768083_wp
        Xi(14) = 20.670270248651332094941279748923_wp
        Xi(15) = 27.780358779898185700094059846776_wp
        Xi(16) = 37.244886866785996006046222106534_wp
        Xi(17) = 49.427722187554696847699986506086_wp
        Xi(18) = 63.970150072867874503634855187784_wp
        Xi(19) = 78.660058441117475559745120250454_wp
        Xi(20) = 88.589934153812753173107275017628_wp

      case (12)
        ! ----- K = 10 , R = 1E2 -----

        RIni = 1.0e2_wp

        Omega(1) = 0.0149667333303671774160669197782647099615_wp
        Omega(2) = 0.0383470287811951085504445432827758111216_wp
        Omega(3) = 0.0724914643973222100692215506823057324937_wp
        Omega(4) = 0.1300892818709850394054503702756520056028_wp
        Omega(5) = 0.2318018689487819192392159539650720034842_wp
        Omega(6) = 0.4098643132422220099664224995938965889763_wp
        Omega(7) = 0.7169870732466540024906825889772221671592_wp
        Omega(8) = 1.2469010750155313586558583627272867033753_wp
        Omega(9) = 2.2020857411608898814361912066317472636001_wp
        Omega(10) = 4.3050004878976240942252406984636081688222_wp

        Alpha(1) = 0.0057695950967900246555054735806550292310_wp
        Alpha(2) = 0.0318850373523618157524438330302496424906_wp
        Alpha(3) = 0.0859854511862230953773761008007703310341_wp
        Alpha(4) = 0.1845984841571087922094127412031561163985_wp
        Alpha(5) = 0.3607477727164535910564773651953984767715_wp
        Alpha(6) = 0.6734363415369500948455712407980655598294_wp
        Alpha(7) = 1.2230817507171462061232522500198172110686_wp
        Alpha(8) = 2.1806860194693055696787270525405233456695_wp
        Alpha(9) = 3.8549835889359444251241887391401519380452_wp
        Alpha(10) = 6.9382543120618274915152590143208044537460_wp

        Xi(1) = 1.008942800663505860595328678109_wp
        Xi(2) = 1.081571842562595240776375515868_wp
        Xi(3) = 1.232671760475624759558453680253_wp
        Xi(4) = 1.474253298620731967671741713399_wp
        Xi(5) = 1.825509665946051017482686107396_wp
        Xi(6) = 2.314882219289935853844333846219_wp
        Xi(7) = 2.983314428037550270107758354499_wp
        Xi(8) = 3.888772807784422466627002878958_wp
        Xi(9) = 5.112476240397841844437243352672_wp
        Xi(10) = 6.767702546743285968357944915041_wp
        Xi(11) = 9.012185726495320496447216651603_wp
        Xi(12) = 12.065068387750407239052130581314_wp
        Xi(13) = 16.228708919855096235243530067294_wp
        Xi(14) = 21.912686087031235649541516252725_wp
        Xi(15) = 29.647268804459919185750882419939_wp
        Xi(16) = 40.044179128258341238827400232481_wp
        Xi(17) = 53.586313012247669108900627321646_wp
        Xi(18) = 69.983253115838594211883805229490_wp
        Xi(19) = 86.812756991639693997242499534650_wp
        Xi(20) = 98.350485614697928909988533519027_wp

      case (13)
        ! ----- K = 10 , R = 2E2 -----

        RIni = 2.0e2_wp

        Omega(1) = 0.0086915765856702453875327267322770108393_wp
        Omega(2) = 0.0230969157217989523684319695673305972150_wp
        Omega(3) = 0.0467619442529359648299560042045142083111_wp
        Omega(4) = 0.0912152273160531483554608507546834772484_wp
        Omega(5) = 0.1754696414861878131220785867472322649974_wp
        Omega(6) = 0.3309435153894120798729380078517792185266_wp
        Omega(7) = 0.6115252378444786909069547098383168304281_wp
        Omega(8) = 1.1144275238224498491721933812748090986133_wp
        Omega(9) = 2.0466210507551132822667144806061401141051_wp
        Omega(10) = 4.1283173287555058033476695911190290644299_wp

        Alpha(1) = 0.0033373898429685276293693579766422296906_wp
        Alpha(2) = 0.0187829201025006288636507280689902366078_wp
        Alpha(3) = 0.0525560175455382537018861680011561077208_wp
        Alpha(4) = 0.1191576576823400992268279716124190770188_wp
        Alpha(5) = 0.2481220755782996451618924865778481603229_wp
        Alpha(6) = 0.4935977722085605890313484561682244589065_wp
        Alpha(7) = 0.9512698505332244500913921325668809458875_wp
        Alpha(8) = 1.7897496715257404193907819767161981872050_wp
        Alpha(9) = 3.3196100790475945492303405659484383249946_wp
        Alpha(10) = 6.2374001617059706020253229130645422628731_wp

        Xi(1) = 1.010443336959742160639438846026_wp
        Xi(2) = 1.095495276973907675674771433627_wp
        Xi(3) = 1.273697812479731929485166075366_wp
        Xi(4) = 1.561788189507271183698033922838_wp
        Xi(5) = 1.986893817861121977886955380921_wp
        Xi(6) = 2.590149845576304480417734188435_wp
        Xi(7) = 3.432417010925453394352621638852_wp
        Xi(8) = 4.602431687100476793116088369828_wp
        Xi(9) = 6.228688173480221028539827177184_wp
        Xi(10) = 8.497306434165050245503614245735_wp
        Xi(11) = 11.678924840204528626599889129523_wp
        Xi(12) = 16.168921976974195827900238064956_wp
        Xi(13) = 22.546738410295669833433018780511_wp
        Xi(14) = 31.659698516862932566109933851806_wp
        Xi(15) = 44.727628056492924835119140425377_wp
        Xi(16) = 63.420391313750978063562557807131_wp
        Xi(17) = 89.695670287561551518862312804004_wp
        Xi(18) = 124.670261729711519121943652521622_wp
        Xi(19) = 164.812157020091428330510474609127_wp
        Xi(20) = 195.389707710008929869571758786151_wp

      case (14)
        ! ----- K = 10 , R = 3E2 -----

        RIni = 3.0e2_wp

        Omega(1) = 0.0063194806278197052362891983183501576349_wp
        Omega(2) = 0.0172560047833030945709404139695730684423_wp
        Omega(3) = 0.0366472048428242284041141552802800518407_wp
        Omega(4) = 0.0752354171213032440356958605032211551134_wp
        Omega(5) = 0.1510870545398002325309652280060923601468_wp
        Omega(6) = 0.2952035614099551131639213863433468532094_wp
        Omega(7) = 0.5620186886646714090960859355750045551758_wp
        Omega(8) = 1.0505752343277138402760986779505003596569_wp
        Omega(9) = 1.9705361918601552558482845056708754327701_wp
        Omega(10) = 4.0417630245914110847550715366338636158616_wp

        Alpha(1) = 0.0024194723389312347069082440292192570652_wp
        Alpha(2) = 0.0138046334311983167548093303853007363102_wp
        Alpha(3) = 0.0396925971499723762111360488707978433354_wp
        Alpha(4) = 0.0934222984881429462702398178186591337635_wp
        Alpha(5) = 0.2024464290319135188056650850874085278974_wp
        Alpha(6) = 0.4181018464748283449924134286268539995035_wp
        Alpha(7) = 0.8332962399975895222744431967143441397639_wp
        Alpha(8) = 1.6150345745196448490757848426291332089022_wp
        Alpha(9) = 3.0746551660908783176924530256002299211104_wp
        Alpha(10) = 5.9113092020828519575562087773334951634752_wp

        Xi(1) = 1.011281696362410611089895084991_wp
        Xi(2) = 1.103306398274419302838952883228_wp
        Xi(3) = 1.296889679570163099865502676078_wp
        Xi(4) = 1.611794637885313709367920975701_wp
        Xi(5) = 2.080320702954512547024068513934_wp
        Xi(6) = 2.752091846962221399200063109980_wp
        Xi(7) = 3.701643236381298505792425412864_wp
        Xi(8) = 5.039415138300416063723424775489_wp
        Xi(9) = 6.928276783724347475095645121357_wp
        Xi(10) = 9.609094668843139655858842651526_wp
        Xi(11) = 13.440420023547740535976857678691_wp
        Xi(12) = 18.960193433730371682238713404445_wp
        Xi(13) = 26.981585834209268605007880914570_wp
        Xi(14) = 38.739364903289162822552782472485_wp
        Xi(15) = 56.100879741652382719230285346157_wp
        Xi(16) = 81.813805024382479445799187800503_wp
        Xi(17) = 119.549667143381229877929783356194_wp
        Xi(18) = 172.649731371664969442614179229167_wp
        Xi(19) = 238.035976555240286464676380262517_wp
        Xi(20) = 291.598770626314468118467360113755_wp

      case (15)
        ! ----- K = 10 , R = 4E2 -----

        RIni = 4.0e2_wp

        Omega(1) = 0.0050414887113661361037624832320130918717_wp
        Omega(2) = 0.0140800054500074398739956657404526207955_wp
        Omega(3) = 0.0310367263079246550432213147888149684661_wp
        Omega(4) = 0.0660863849445842620891427253437822031401_wp
        Omega(5) = 0.1366765837125404271286009325048205198527_wp
        Omega(6) = 0.2735256663780587675692621690837214032399_wp
        Omega(7) = 0.5313705135370472408692064936186483237179_wp
        Omega(8) = 1.0104385988466745368999713527990991224215_wp
        Omega(9) = 1.9222796166689597670328495948588454211858_wp
        Omega(10) = 3.9868277184968530305053668882919737370685_wp

        Alpha(1) = 0.0019254909170795910657038692770023913248_wp
        Alpha(2) = 0.0111128914345121174053033314896299366126_wp
        Alpha(3) = 0.0326715796695796239417029088736299513585_wp
        Alpha(4) = 0.0791423112744457488463385372301495124248_wp
        Alpha(5) = 0.1765489795507469377686430786478233301295_wp
        Alpha(6) = 0.3743260077419205072524286542767768537487_wp
        Alpha(7) = 0.7634651418561960165075179485238976440087_wp
        Alpha(8) = 1.5097794912507614131184224826043305256462_wp
        Alpha(9) = 2.9250174225165053239379464589120516393450_wp
        Alpha(10) = 5.7101524172963629357065162839290906049428_wp

        Xi(1) = 1.011853390856991647133063250141_wp
        Xi(2) = 1.108646476666356957751710665683_wp
        Xi(3) = 1.312818245875861967899948801186_wp
        Xi(4) = 1.646359283871812145328729359850_wp
        Xi(5) = 2.145420700521415087068041738227_wp
        Xi(6) = 2.866047574432412507836748227774_wp
        Xi(7) = 3.893283466834573220164902873641_wp
        Xi(8) = 5.354529532742844356429240404438_wp
        Xi(9) = 7.440034709028623380582673219052_wp
        Xi(10) = 10.435182827798015401526976475566_wp
        Xi(11) = 14.771615580191396934449377109644_wp
        Xi(12) = 21.108666358851921747777358184806_wp
        Xi(13) = 30.463805408628496824638420825693_wp
        Xi(14) = 44.420598469816782581653935579880_wp
        Xi(15) = 65.450058006619073823484367125047_wp
        Xi(16) = 97.351899309879051984395470498157_wp
        Xi(17) = 145.591442086941418629364441983398_wp
        Xi(18) = 216.187417830411720928718288803339_wp
        Xi(19) = 307.737175259318556874710282045271_wp
        Xi(20) = 387.133073207267179111079613562652_wp

      case (16)
        ! ----- K = 10 , R = 5E2 -----

        RIni = 5.0e2_wp

        Omega(1) = 0.0042324685609204748920790582593365325437_wp
        Omega(2) = 0.0120546370748870361851306522681104294037_wp
        Omega(3) = 0.0273986415737904902802037250542088742122_wp
        Omega(4) = 0.0600063698043882148383293580196085059697_wp
        Omega(5) = 0.1268815804111528272178806286962782223782_wp
        Omega(6) = 0.2585247942583658795339274921731842482586_wp
        Omega(7) = 0.5098606010576789215687186496150928860516_wp
        Omega(8) = 0.9819685015827939782379410460499968849035_wp
        Omega(9) = 1.8878328906613232435716953627746761412709_wp
        Omega(10) = 3.9475916782101656337874984181723903020611_wp

        Alpha(1) = 0.0016130554450903338090927353529711874014_wp
        Alpha(2) = 0.0094040709862812426859680339438773977179_wp
        Alpha(3) = 0.0281792896456298726902417331810490885857_wp
        Alpha(4) = 0.0698808951661137112344084412751676893549_wp
        Alpha(5) = 0.1594696974511107449390227841501399552726_wp
        Alpha(6) = 0.3449733568032465537314505688026855523276_wp
        Alpha(7) = 0.7159420334350728644823985691303391831752_wp
        Alpha(8) = 1.4372483830045267496483551994401750562247_wp
        Alpha(9) = 2.8208885751505575071110265561458163574571_wp
        Alpha(10) = 5.5692173765763346026010094380609416475636_wp

        Xi(1) = 1.012281851750328150402538640584_wp
        Xi(2) = 1.112655829018262414881190736082_wp
        Xi(3) = 1.324817075799919270065636844880_wp
        Xi(4) = 1.672514848849586383385326660012_wp
        Xi(5) = 2.194966480380893188899607548770_wp
        Xi(6) = 2.953386524695425789611633038412_wp
        Xi(7) = 4.041374209184313866585758212580_wp
        Xi(8) = 5.600307202630111662756268176011_wp
        Xi(9) = 7.843307547048567834879762017408_wp
        Xi(10) = 11.093509658933041624705173600063_wp
        Xi(11) = 15.845547108056645307753995677302_wp
        Xi(12) = 22.865211568856501675450765276310_wp
        Xi(13) = 33.352580843064771526362832787527_wp
        Xi(14) = 49.209705705642749529876711633847_wp
        Xi(15) = 73.472708373403977248627949592219_wp
        Xi(16) = 110.957993990505837049453941034471_wp
        Xi(17) = 168.946615691431218028717431423047_wp
        Xi(18) = 256.413208477791467743145403801464_wp
        Xi(19) = 374.596819945121224632034184764962_wp
        Xi(20) = 482.078964309526471265776592645125_wp

      case (17)
        ! ----- K = 10 , R = 6E2 -----

        RIni = 6.0e2_wp

        Omega(1) = 0.0036700303900818465139500916755502246680_wp
        Omega(2) = 0.0106377997883372633690940585980688748435_wp
        Omega(3) = 0.0248163213140422915740365217632246164214_wp
        Omega(4) = 0.0556039406929946345510464705830244724893_wp
        Omega(5) = 0.1196656466589770147608768127711620365972_wp
        Omega(6) = 0.2473234458266989885580865171643694111481_wp
        Omega(7) = 0.4936263432805592477157097947015884642497_wp
        Omega(8) = 0.9603068597394762664258907602921055968181_wp
        Omega(9) = 1.8614967394761155464926252744994883414620_wp
        Omega(10) = 3.9175799473154924604332421944619113673980_wp

        Alpha(1) = 0.0013960043685149540151937456685694860514_wp
        Alpha(2) = 0.0082132548876076855998078587485500268173_wp
        Alpha(3) = 0.0250272674132974799887917669374837181806_wp
        Alpha(4) = 0.0633069328322906192752840028037386588267_wp
        Alpha(5) = 0.1471799227081377646676090334421083127836_wp
        Alpha(6) = 0.3235737232994003323774401259527166985208_wp
        Alpha(7) = 0.6808942948679821898861144746639695313206_wp
        Alpha(8) = 1.3832418428333822428642069235493750056776_wp
        Alpha(9) = 2.7427730207536506401230780305411371955415_wp
        Alpha(10) = 5.4629433264669168906178808864382290266803_wp

        Xi(1) = 1.012621454662819640043895652592_wp
        Xi(2) = 1.115838165453791844586826920160_wp
        Xi(3) = 1.334365050983588460919646023761_wp
        Xi(4) = 1.693400697466688524192059139839_wp
        Xi(5) = 2.234705295692805331313895988998_wp
        Xi(6) = 3.023818437237688030744947598194_wp
        Xi(7) = 4.161557714823225141535789628477_wp
        Xi(8) = 5.801204286139057726603279752098_wp
        Xi(9) = 8.175571683591240763032403915389_wp
        Xi(10) = 11.640672377038550208520895079545_wp
        Xi(11) = 16.746696098108507603022254706104_wp
        Xi(12) = 24.354621648984547767347041968833_wp
        Xi(13) = 35.830254270563893500417540494141_wp
        Xi(14) = 53.369584665590483127384224815160_wp
        Xi(15) = 80.540436987329664747514712530574_wp
        Xi(16) = 123.139323981935882947047478097602_wp
        Xi(17) = 190.258853088247873563965129051212_wp
        Xi(18) = 294.009996425292684962826683658932_wp
        Xi(19) = 439.055243665943953190744508674470_wp
        Xi(20) = 576.493245747046950178127389108340_wp

      case (18)
        ! ----- K = 10 , R = 7E2 -----

        RIni = 7.0e2_wp

        Omega(1) = 0.0032542049889380949915407986467591472390_wp
        Omega(2) = 0.0095845715277716938660811010070039728248_wp
        Omega(3) = 0.0228716368797110218745124478323393901746_wp
        Omega(4) = 0.0522328218464833619212042085250269352059_wp
        Omega(5) = 0.1140629704981057233851454192630203721137_wp
        Omega(6) = 0.2385322997992477096290903429420815484718_wp
        Omega(7) = 0.4807761161686558414333960559394753886409_wp
        Omega(8) = 0.9430492834876176160593778952101473578296_wp
        Omega(9) = 1.8404331172312120606026838998126038404735_wp
        Omega(10) = 3.8935670915039185988829745221906364349707_wp

        Alpha(1) = 0.0012356310955025927425262089495543171846_wp
        Alpha(2) = 0.0073310324723258707728122175219243050748_wp
        Alpha(3) = 0.0226776888586415631861574904865053170511_wp
        Alpha(4) = 0.0583566335293559620051072681090609250987_wp
        Alpha(5) = 0.1378184890156283799739081810664664828892_wp
        Alpha(6) = 0.3070966214157072800034365550603254746420_wp
        Alpha(7) = 0.6536553453439855522629463313588615847038_wp
        Alpha(8) = 1.3409422950604044433081316056721732365986_wp
        Alpha(9) = 2.6812225946438997526127118176120234238624_wp
        Alpha(10) = 5.3788596769382154365833625941917262025527_wp

        Xi(1) = 1.012900857627860401314646987547_wp
        Xi(2) = 1.118459326721759505883849183938_wp
        Xi(3) = 1.342245589900514743842037390653_wp
        Xi(4) = 1.710687375875769244211628550278_wp
        Xi(5) = 2.267714201987363026357183426462_wp
        Xi(6) = 3.082579276498763421257565120470_wp
        Xi(7) = 4.262342355741563575902836058518_wp
        Xi(8) = 5.970655739022618686915461205800_wp
        Xi(9) = 8.457640888002688175223431343852_wp
        Xi(10) = 12.108483930024105924838639936070_wp
        Xi(11) = 17.523177212053987253023712433730_wp
        Xi(12) = 25.648992276300671777286566133114_wp
        Xi(13) = 38.003832043215744079106555020076_wp
        Xi(14) = 57.057203179280471496553905907945_wp
        Xi(15) = 86.879552209645019580919189650103_wp
        Xi(16) = 134.212137799403177715329427144297_wp
        Xi(17) = 209.941935034556916031545092948818_wp
        Xi(18) = 329.436073714199028183990947127313_wp
        Xi(19) = 501.424828964952372711838535224160_wp
        Xi(20) = 670.416864260303763611847926995324_wp

      case (19)
        ! ----- K = 10 , R = 8E2 -----

        RIni = 8.0e2_wp

        Omega(1) = 0.0029330749052116580108893871157646482395_wp
        Omega(2) = 0.0087672082685003208973357905278711577068_wp
        Omega(3) = 0.0213446274367968236463239664266189343778_wp
        Omega(4) = 0.0495476801027003352572385379976038777272_wp
        Omega(5) = 0.1095487420615464476506841461078689548003_wp
        Omega(6) = 0.2313858405917663885528875664099324183098_wp
        Omega(7) = 0.4702560460707860016330971592024567939916_wp
        Omega(8) = 0.9288450627750289506981418163888974959264_wp
        Omega(9) = 1.8230399046731256502756543524945698209194_wp
        Omega(10) = 3.8737316934718114202432920345486877522490_wp

        Alpha(1) = 0.0011118450661519535918240580681451423750_wp
        Alpha(2) = 0.0066484639002268138156620752663578111097_wp
        Alpha(3) = 0.0208496197653120780107457218205402327271_wp
        Alpha(4) = 0.0544701624352677744922478306893731669902_wp
        Alpha(5) = 0.1303955767535651947050924850945996524842_wp
        Alpha(6) = 0.2939119642783786198780068138480814354807_wp
        Alpha(7) = 0.6316881490532415380266886839599749237095_wp
        Alpha(8) = 1.3066088455436533010637495388905904292187_wp
        Alpha(9) = 2.6310144253057876860806690855199008183263_wp
        Alpha(10) = 5.3100363236689658024032578698125917071593_wp

        Xi(1) = 1.013136955479273742511056588533_wp
        Xi(2) = 1.120676286477326305339972944974_wp
        Xi(3) = 1.348922273258674577652710402376_wp
        Xi(4) = 1.725367830002335719076327069654_wp
        Xi(5) = 2.295830009766047126668411793027_wp
        Xi(6) = 3.132813074819947013862145301921_wp
        Xi(7) = 4.348872548296264884358508506779_wp
        Xi(8) = 6.116849595751287185351613429063_wp
        Xi(9) = 8.702312553797890090807876095624_wp
        Xi(10) = 12.516694687068216307324064473505_wp
        Xi(11) = 18.205181659072122620987577157337_wp
        Xi(12) = 26.794097712685306787727679278532_wp
        Xi(13) = 39.942137325331399523575814924925_wp
        Xi(14) = 60.374968778913925184642641497135_wp
        Xi(15) = 92.640055205432798930698101713688_wp
        Xi(16) = 144.390157487883737968958897113225_wp
        Xi(17) = 228.282573541115812940094365046662_wp
        Xi(18) = 363.020362127012237968282093447669_wp
        Xi(19) = 561.941036587875227037436332011566_wp
        Xi(20) = 763.881072280655437523044781755743_wp

      case (20)
        ! ----- K = 10 , R = 9E2 -----

        RIni = 9.0e2_wp

        Omega(1) = 0.0026768722975028171728839701841093368095_wp
        Omega(2) = 0.0081121894164151209087417122335639252384_wp
        Omega(3) = 0.0201076998512748532391228870400134987051_wp
        Omega(4) = 0.0473454129573102471413006617217966720546_wp
        Omega(5) = 0.1058099448966131957098756218027535425108_wp
        Omega(6) = 0.2254222361170528210309611100159443708435_wp
        Omega(7) = 0.4614245274188429461227622940810988438898_wp
        Omega(8) = 0.9168662209932350510600829185392512954422_wp
        Omega(9) = 1.8083310522241474101482958114495147583511_wp
        Omega(10) = 3.8569524630728154492020909316352117457427_wp

        Alpha(1) = 0.0010131325949005472269497042832331334949_wp
        Alpha(2) = 0.0061029931115648970293762832009520913346_wp
        Alpha(3) = 0.0193811446162218188716495079551571656395_wp
        Alpha(4) = 0.0513226634278318588158773229379594482680_wp
        Alpha(5) = 0.1243314878775564602441587418180102275755_wp
        Alpha(6) = 0.2830557631852255335571603295230502794766_wp
        Alpha(7) = 0.6134788317812593461749066936494756419052_wp
        Alpha(8) = 1.2779919332611943687434685745785145627451_wp
        Alpha(9) = 2.5889883766039190108286527358671946785762_wp
        Alpha(10) = 5.2522618822502933752169318459834812529152_wp

        Xi(1) = 1.013340506069180610372766626615_wp
        Xi(2) = 1.122589180203960424787867911078_wp
        Xi(3) = 1.354691599735056251529660698019_wp
        Xi(4) = 1.738078713455105556834855151660_wp
        Xi(5) = 2.320235898215292067326559477536_wp
        Xi(6) = 3.176555398741145603983340728327_wp
        Xi(7) = 4.424497164492826078736192885188_wp
        Xi(8) = 6.245151350824582261057199561449_wp
        Xi(9) = 8.918035302609499784948221545733_wp
        Xi(10) = 12.878450829603987428523259595181_wp
        Xi(11) = 18.812984656048486938398345280632_wp
        Xi(12) = 27.820980821978816168241199768829_wp
        Xi(13) = 41.692348392723356135386447718361_wp
        Xi(14) = 63.393913446195932721860533121117_wp
        Xi(15) = 97.927487715505802341742214167652_wp
        Xi(16) = 153.826154826311490259049818973835_wp
        Xi(17) = 245.489989054823343678224389918796_wp
        Xi(18) = 395.010353992267319478415288358519_wp
        Xi(19) = 620.788700367442668193174171165083_wp
        Xi(20) = 856.910708122485446525118391036813_wp

      case (21)
        ! ----- K = 10 , R = 1E3 -----

        RIni = 1.0e3_wp

        Omega(1) = 0.0024672464939504652393320415079344676279_wp
        Omega(2) = 0.0075740393679528136391924269567310634699_wp
        Omega(3) = 0.0190813545239317744910730999946535391132_wp
        Omega(4) = 0.0454979068375473446648820340099117487398_wp
        Omega(5) = 0.1026467684099852501974654145344700850728_wp
        Omega(6) = 0.2203438450309990335784825557507993210038_wp
        Omega(7) = 0.4538648781925274396217753064552979935797_wp
        Omega(8) = 0.9065719038631264092235190199531302823743_wp
        Omega(9) = 1.7956602125219735557838315886947100352700_wp
        Omega(10) = 3.8424941566456491484106428035971703138785_wp

        Alpha(1) = 0.0009323990731474346103138267389132815000_wp
        Alpha(2) = 0.0056560052513406161738321840937171458563_wp
        Alpha(3) = 0.0181719764811097294023625172906810654183_wp
        Alpha(4) = 0.0487116706076877454791506857900706961573_wp
        Alpha(5) = 0.1192619458576450608360106260541488865101_wp
        Alpha(6) = 0.2739171355714497416503072635407534107799_wp
        Alpha(7) = 0.5980606335541609821517504419485788957900_wp
        Alpha(8) = 1.2536454845296414365536305512982551135792_wp
        Alpha(9) = 2.5531023689261667071817574203862477588700_wp
        Alpha(10) = 5.2028048639403972550065979874034383101389_wp

        Xi(1) = 1.013518771925512285039111537444_wp
        Xi(2) = 1.124265631526049065186563524765_wp
        Xi(3) = 1.359754378828921279800251431524_wp
        Xi(4) = 1.749252064327726823608592787718_wp
        Xi(5) = 2.341737351042612545618318287133_wp
        Xi(6) = 3.215197438739649914238999395266_wp
        Xi(7) = 4.491518208540886354651622269785_wp
        Xi(8) = 6.359268176451460747537064444046_wp
        Xi(9) = 9.110684492029050861316985532312_wp
        Xi(10) = 13.202955683170126036096725385960_wp
        Xi(11) = 19.360892828771091835060591535012_wp
        Xi(12) = 28.751732205868852112334854709275_wp
        Xi(13) = 43.288356603745349986123169827579_wp
        Xi(14) = 66.165637340636030923335386688677_wp
        Xi(15) = 102.819359616610749581788741835453_wp
        Xi(16) = 162.634010646464310198555658359965_wp
        Xi(17) = 261.723013102670544638428395955998_wp
        Xi(18) = 425.598861167297845137236222967658_wp
        Xi(19) = 678.117414638977752505688556539098_wp
        Xi(20) = 949.526067440132668984187347405168_wp

      case (22)
        ! ----- K = 10 , R = 2E3 -----

        RIni = 2.0e3_wp

        Omega(1) = 0.0014526807816713179251917649882457073129_wp
        Omega(2) = 0.0049297471955163571493195942292003808305_wp
        Omega(3) = 0.0138551513028217050564071840916546829447_wp
        Omega(4) = 0.0357468212429620454283945088502782994055_wp
        Omega(5) = 0.0854970858922482741348684853743833400586_wp
        Omega(6) = 0.1922337452291708614644100139301485796750_wp
        Omega(7) = 0.4113201462352773910328106177214380068108_wp
        Omega(8) = 0.8478938111702770348337629413570937231270_wp
        Omega(9) = 1.7228690042716290663563083462861413863720_wp
        Omega(10) = 3.7593552654949775578344106641281996417092_wp

        Alpha(1) = 0.0005421696478208119686995916880565410390_wp
        Alpha(2) = 0.0034806730732080148226928955441580870733_wp
        Alpha(3) = 0.0121808446627809141984838674395430491160_wp
        Alpha(4) = 0.0354347079465288640331293874147222666693_wp
        Alpha(5) = 0.0928087866746072898794335179206704822263_wp
        Alpha(6) = 0.2251455517645407656819427077987683105675_wp
        Alpha(7) = 0.5142100271408126002073597782260350186334_wp
        Alpha(8) = 1.1191922258880422136112162134757852527400_wp
        Alpha(9) = 2.3525743224761141579766399178907931855065_wp
        Alpha(10) = 4.9242248867504468650552196251624081924092_wp

        Xi(1) = 1.014597792749656095289204071896_wp
        Xi(2) = 1.134436845099320007208831828649_wp
        Xi(3) = 1.390599483503960958718885521090_wp
        Xi(4) = 1.817719358871181273176569070937_wp
        Xi(5) = 2.474464282290303179497395480446_wp
        Xi(6) = 3.455904036003200488188016881175_wp
        Xi(7) = 4.913482237359631337163951014801_wp
        Xi(8) = 7.086528454205757836855544828936_wp
        Xi(9) = 10.355243055451960874392691724921_wp
        Xi(10) = 15.331357867549850092919738742125_wp
        Xi(11) = 23.015738839648443813529210544289_wp
        Xi(12) = 35.078615893121463394088710430196_wp
        Xi(13) = 54.370071151073339515980009650775_wp
        Xi(14) = 85.879927506270057162296449604355_wp
        Xi(15) = 138.589860380804692760459140288276_wp
        Xi(16) = 229.159680799216293481390671615827_wp
        Xi(17) = 389.249173017283270181732035553068_wp
        Xi(18) = 678.447113000007039851357859561176_wp
        Xi(19) = 1188.115962211603321430075652642699_wp
        Xi(20) = 1855.702942249214593473105594512162_wp

      case (23)
        ! ----- K = 10 , R = 3E3 -----

        RIni = 3.0e3_wp

        Omega(1) = 0.0010738804618956168932148597825945723905_wp
        Omega(2) = 0.0039152313311347269926616563272270266083_wp
        Omega(3) = 0.0117284581530090255200150881759180165886_wp
        Omega(4) = 0.0315715502645784553185058507462912302799_wp
        Omega(5) = 0.0778819587605281879913693247885486670157_wp
        Omega(6) = 0.1794002209408897666648207216977262135060_wp
        Omega(7) = 0.3914628342219864378054381554017382427446_wp
        Omega(8) = 0.8200399624470611673163374100159472845917_wp
        Omega(9) = 1.6879552117417538126308354518023691070994_wp
        Omega(10) = 3.7194242684483121124793675305042484069418_wp

        Alpha(1) = 0.0003967302317420766487669641541446638455_wp
        Alpha(2) = 0.0026604835814450934334069928926397308189_wp
        Alpha(3) = 0.0098489286120088280087307578206101688068_wp
        Alpha(4) = 0.0300488167483034730690792842022762698662_wp
        Alpha(5) = 0.0816621938621181960049101589138587087291_wp
        Alpha(6) = 0.2039328886821691243984126795663058828723_wp
        Alpha(7) = 0.4767849144334121988867119384947201865543_wp
        Alpha(8) = 1.0579265306400313928492096238187514245510_wp
        Alpha(9) = 2.2597542577639734308597024847742318343080_wp
        Alpha(10) = 4.7939093124023811104579551045645757767488_wp

        Xi(1) = 1.015150512195101957795169422827_wp
        Xi(2) = 1.139662893625249794190765206814_wp
        Xi(3) = 1.406534849554869738802280465517_wp
        Xi(4) = 1.853355656995012800520271356586_wp
        Xi(5) = 2.544208360205743575969614123089_wp
        Xi(6) = 3.583883042504509402461851386157_wp
        Xi(7) = 5.140944517623982631688434041806_wp
        Xi(8) = 7.484744461305179324701192244973_wp
        Xi(9) = 11.048733490328873131820774933232_wp
        Xi(10) = 16.540679503715717895209746934881_wp
        Xi(11) = 25.137923544129997780252616657037_wp
        Xi(12) = 38.842667988550012486692608604244_wp
        Xi(13) = 61.145845410447943829579386942896_wp
        Xi(14) = 98.315172188442081627202107085850_wp
        Xi(15) = 161.976990552996012631692401839700_wp
        Xi(16) = 274.521366554113617375332268011334_wp
        Xi(17) = 480.753414990375269988387429975774_wp
        Xi(18) = 872.152150704680091064524560806603_wp
        Xi(19) = 1617.989737536047364030444839499978_wp
        Xi(20) = 2730.560653699768991486962477210909_wp

      case (24)
        ! ----- K = 10 , R = 4E3 -----

        RIni = 4.0e3_wp

        Omega(1) = 0.0008710258738679722909390542499950133015_wp
        Omega(2) = 0.0033608319216372058163594838554183441559_wp
        Omega(3) = 0.0105209015614479294216962543956350728536_wp
        Omega(4) = 0.0291318792682550152024377118631370819912_wp
        Omega(5) = 0.0733425279782349960833393379133759992783_wp
        Omega(6) = 0.1716321958691351225108649367068203162034_wp
        Omega(7) = 0.3792963222960632354121331827334628883364_wp
        Omega(8) = 0.8028142821879096244306467378137881496514_wp
        Omega(9) = 1.6662385945818383603143561511039649758459_wp
        Omega(10) = 3.6945668905108900519854847610545789393655_wp

        Alpha(1) = 0.0003189013513191434614517324352316388070_wp
        Alpha(2) = 0.0022180408379182630683261988335797099126_wp
        Alpha(3) = 0.0085622083802234129730279006340043501666_wp
        Alpha(4) = 0.0269984516983066408202083684073069314024_wp
        Alpha(5) = 0.0752068311142086706595180807533029820888_wp
        Alpha(6) = 0.1914251600871076061422156044944387076612_wp
        Alpha(7) = 0.4543973763628831388783346767468174220994_wp
        Alpha(8) = 1.0208561605370438546977820970518280319084_wp
        Alpha(9) = 2.2031043684761797731008758916004808270372_wp
        Alpha(10) = 4.7139161935991886124945604574776325534913_wp

        Xi(1) = 1.015506565858864634942573812459_wp
        Xi(2) = 1.143035114280607693749806930139_wp
        Xi(3) = 1.416848829360455259648333969924_wp
        Xi(4) = 1.876516805143232096515434892403_wp
        Xi(5) = 2.589780887553195976198694427950_wp
        Xi(6) = 3.668062082444905891431830013971_wp
        Xi(7) = 5.291724324827579940475064113059_wp
        Xi(8) = 7.751056290437090507978307174497_wp
        Xi(9) = 11.517149648139810796595994268898_wp
        Xi(10) = 17.366666817265994425187325411741_wp
        Xi(11) = 26.605671471427667244824544212634_wp
        Xi(12) = 41.483096270017238978849993813469_wp
        Xi(13) = 65.976340454713680826193122896939_wp
        Xi(14) = 107.346974824950474040885506354925_wp
        Xi(15) = 179.337929725681243811608034377514_wp
        Xi(16) = 309.080899972430674593759647450497_wp
        Xi(17) = 552.743379803463856347889304743148_wp
        Xi(18) = 1031.064031042961785633593763122917_wp
        Xi(19) = 1993.591017126431915174045172989281_wp
        Xi(20) = 3578.011512315118128224611382393050_wp

      case (25)
        ! ----- K = 10 , R = 5E3 -----

        RIni = 5.0e3_wp

        Omega(1) = 0.0007431794401095743669437292025069935475_wp
        Omega(2) = 0.0030055420983453206422728796977025811366_wp
        Omega(3) = 0.0097249569761531400332958945814665341345_wp
        Omega(4) = 0.0274923139468489620731419999927203079437_wp
        Omega(5) = 0.0702505967829803888220112709217524127325_wp
        Omega(6) = 0.1662865450907947724451399831857578703875_wp
        Omega(7) = 0.3708552468418009445650599525956536695048_wp
        Omega(8) = 0.7907883037645155538744590195676664734492_wp
        Omega(9) = 1.6510183817369136722663994221171890330879_wp
        Omega(10) = 3.6771356408478858242890735530394863417314_wp

        Alpha(1) = 0.0002698635605074361434595552571144225218_wp
        Alpha(2) = 0.0019374881805125896826202009119374802815_wp
        Alpha(3) = 0.0077316581226501765259868437837192933770_wp
        Alpha(4) = 0.0249919540890399625044510090177141137247_wp
        Alpha(5) = 0.0708942348818701222867822396567216713947_wp
        Alpha(6) = 0.1829661537692895389973189723686886054566_wp
        Alpha(7) = 0.4391085307768620177998331044477708928753_wp
        Alpha(8) = 0.9953449891370984821750700355380558903562_wp
        Alpha(9) = 2.1638931456033836848735357749617946865328_wp
        Alpha(10) = 4.6583347845772465202311696241821437070030_wp

        Xi(1) = 1.015761905522425848232524747683_wp
        Xi(2) = 1.145456265549093649732358135562_wp
        Xi(3) = 1.424269242260172810038594548576_wp
        Xi(4) = 1.893227150506627856434670176267_wp
        Xi(5) = 2.622778685669105851706317533711_wp
        Xi(6) = 3.729287436722984407219336211448_wp
        Xi(7) = 5.401967884976654192541456467325_wp
        Xi(8) = 7.946941855860498587181550700365_wp
        Xi(9) = 11.864030887071025224810405518383_wp
        Xi(10) = 17.983017475809047037646482003481_wp
        Xi(11) = 27.710351355899081444988940603480_wp
        Xi(12) = 43.489879502946803307289469486818_wp
        Xi(13) = 69.689071791289023578841455019983_wp
        Xi(14) = 114.380016461326117188646023237197_wp
        Xi(15) = 193.067490232188291657267065204451_wp
        Xi(16) = 336.924967640876272423344772732889_wp
        Xi(17) = 612.112609457937406176597505691461_wp
        Xi(18) = 1166.208947569273627342667509765306_wp
        Xi(19) = 2328.357007035189155885746004059911_wp
        Xi(20) = 4400.318794246789840141076410873211_wp

      case (26)
        ! ----- K = 10 , R = 6E3 -----

        RIni = 6.0e3_wp

        Omega(1) = 0.0006546460868159010347083298598722111983_wp
        Omega(2) = 0.0027559343285720026859823156024226209482_wp
        Omega(3) = 0.0091533655076862242054661902498913228499_wp
        Omega(4) = 0.0262978801544948501235979525930686051538_wp
        Omega(5) = 0.0679756840176229069013544119293168677132_wp
        Omega(6) = 0.1623235380963315463529340193504779676914_wp
        Omega(7) = 0.3645597507129552442266956502336938683584_wp
        Omega(8) = 0.7817777963990977949321550810601877401496_wp
        Omega(9) = 1.6395819121798266358291246147516062592331_wp
        Omega(10) = 3.6640322363638034132773957374240580975311_wp

        Alpha(1) = 0.0002359054779922201880353551623295017825_wp
        Alpha(2) = 0.0017421610274382283694814347045805302816_wp
        Alpha(3) = 0.0071448724389958750840038618351579557952_wp
        Alpha(4) = 0.0235534337944883915382909399943711648007_wp
        Alpha(5) = 0.0677660658354808540797969453051141641708_wp
        Alpha(6) = 0.1767741391135355176440291727346654226949_wp
        Alpha(7) = 0.4278362266663702313991297226936083575310_wp
        Alpha(8) = 0.9764291736433913701498875370443641941165_wp
        Alpha(9) = 2.1346954872130372662421271545696299654082_wp
        Alpha(10) = 4.6168309999158093559143423156854169064900_wp

        Xi(1) = 1.015956987048296602019724355070_wp
        Xi(2) = 1.147307628546274119174519512399_wp
        Xi(3) = 1.429951990212024972920143439659_wp
        Xi(4) = 1.906050941262453609389808384389_wp
        Xi(5) = 2.648169446245551599677936494004_wp
        Xi(6) = 3.776553877910799129945587648827_wp
        Xi(7) = 5.487407583017900029268704154006_wp
        Xi(8) = 8.099432497604121565168422414160_wp
        Xi(9) = 12.135421059281113738935464763102_wp
        Xi(10) = 18.467966908870437714104695281492_wp
        Xi(11) = 28.585123074481174617472722943035_wp
        Xi(12) = 45.090685285555316397926306493105_wp
        Xi(13) = 72.675858909161066459536648665107_wp
        Xi(14) = 120.094158812983521085471227962671_wp
        Xi(15) = 204.354962067814596077264255313821_wp
        Xi(16) = 360.146971185465807924197534362065_wp
        Xi(17) = 662.533546906581793267676516734355_wp
        Xi(18) = 1283.793870989625666045874652354541_wp
        Xi(19) = 2630.654446316201495292830259131733_wp
        Xi(20) = 5199.087330578308051798330780002289_wp

      case (27)
        ! ----- K = 10 , R = 7E3 -----

        RIni = 7.0e3_wp

        Omega(1) = 0.0005894337117817897871925262719573074266_wp
        Omega(2) = 0.0025697219209575387010189239343771472335_wp
        Omega(3) = 0.0087192982240147648271186265299392825767_wp
        Omega(4) = 0.0253806036987043458811138434486065484919_wp
        Omega(5) = 0.0662150923425636726442128007774634568250_wp
        Omega(6) = 0.1592383500598885868158611361922183391471_wp
        Omega(7) = 0.3596358039121044572461721039458026893954_wp
        Omega(8) = 0.7747051418395835614347233466503439558437_wp
        Omega(9) = 1.6305850484217997494341273467810538022604_wp
        Omega(10) = 3.6537205359673349575229395203024296279182_wp

        Alpha(1) = 0.0002108892791676530489045705884024184407_wp
        Alpha(2) = 0.0015975896285734081348582562667227924269_wp
        Alpha(3) = 0.0067051290145370598709271604299930746862_wp
        Alpha(4) = 0.0224625370876326169567013726208193347134_wp
        Alpha(5) = 0.0653717748314939917488129023093179625903_wp
        Alpha(6) = 0.1720007540745488256789990158535097464210_wp
        Alpha(7) = 0.4190975398123096649504078226033954024388_wp
        Alpha(8) = 0.9617003167186997781125737394525287982106_wp
        Alpha(9) = 2.1118856540803401934495997505081277267891_wp
        Alpha(10) = 4.5843367189279305151661247919037123210728_wp

        Xi(1) = 1.016112431848799921166662252325_wp
        Xi(2) = 1.148783819519298703311360743129_wp
        Xi(3) = 1.434488586048307485597969046509_wp
        Xi(4) = 1.916304352019664957286829154448_wp
        Xi(5) = 2.668513636604800630652140069898_wp
        Xi(6) = 3.814523150955451240512808497307_wp
        Xi(7) = 5.556251095647487081834875466413_wp
        Xi(8) = 8.222727966325468763365835744850_wp
        Xi(9) = 12.355717871798836730856674392953_wp
        Xi(10) = 18.863374909541553076097741836747_wp
        Xi(11) = 29.301965962163520094257984638375_wp
        Xi(12) = 46.410094597696795501451383358926_wp
        Xi(13) = 75.154152734836061185152988883829_wp
        Xi(14) = 124.872938892471809987916309125922_wp
        Xi(15) = 213.884402134001678094654685935438_wp
        Xi(16) = 379.979871483282035271589194280750_wp
        Xi(17) = 706.235618502009305785804826882668_wp
        Xi(18) = 1387.751699638245933554614452987153_wp
        Xi(19) = 2906.228803626507716018068094854243_wp
        Xi(20) = 5975.583040521315707938043715330423_wp

      case (28)
        ! ----- K = 10 , R = 8E3 -----

        RIni = 8.0e3_wp

        Omega(1) = 0.0005392552127602062085177379348061715847_wp
        Omega(2) = 0.0024247926212958021032482688579892671221_wp
        Omega(3) = 0.0083764286534713166290939804527193590644_wp
        Omega(4) = 0.0246494419210904408322975562770085211639_wp
        Omega(5) = 0.0648029218064254492278784477998776480945_wp
        Omega(6) = 0.1567518884914159084978087579931305128866_wp
        Omega(7) = 0.3556524160150506188183411326875216218468_wp
        Omega(8) = 0.7689669348259188687961206143395287426756_wp
        Omega(9) = 1.6232725366290187618209528430668342480203_wp
        Omega(10) = 3.6453370534672855701654320403903852820804_wp

        Alpha(1) = 0.0001916362889846043065853641168007868989_wp
        Alpha(2) = 0.0014858556424114500268540757402062346682_wp
        Alpha(3) = 0.0063615903805791639624878260322188294751_wp
        Alpha(4) = 0.0216018516951527043170173903938180437478_wp
        Alpha(5) = 0.0634683844611125438155640153536918290911_wp
        Alpha(6) = 0.1681839281196474854062706053015041618437_wp
        Alpha(7) = 0.4120781965585719024689533712679789800859_wp
        Alpha(8) = 0.9498272260709862341906267735325286594161_wp
        Alpha(9) = 2.0934494387115830259151610936996235068364_wp
        Alpha(10) = 4.5580268727262413749369562854241166860447_wp

        Xi(1) = 1.016240074424546586972541162286_wp
        Xi(2) = 1.149996640003310312294106110276_wp
        Xi(3) = 1.438219369245532938900514197123_wp
        Xi(4) = 1.924747481555941825532261846288_wp
        Xi(5) = 2.685294247801549651988653466539_wp
        Xi(6) = 3.845907382537659943049293542749_wp
        Xi(7) = 5.613293949872002155075667584327_wp
        Xi(8) = 8.325176650420320748069435978778_wp
        Xi(9) = 12.539349560191624636737439679735_wp
        Xi(10) = 19.194156819094702701861132609906_wp
        Xi(11) = 29.904126804560926226536632377417_wp
        Xi(12) = 47.523652979244679680354002115905_wp
        Xi(13) = 77.257305678500895959603944618266_wp
        Xi(14) = 128.954673044338415363596617169151_wp
        Xi(15) = 222.087597862233788223451647070306_wp
        Xi(16) = 397.217217380373791707492969749183_wp
        Xi(17) = 744.689647863577452946426404878366_wp
        Xi(18) = 1480.770111530073871342061408995505_wp
        Xi(19) = 3159.281593163338799579165083741827_wp
        Xi(20) = 6730.860033210824243887770990113495_wp

      case (29)
        ! ----- K = 10 , R = 9E3 -----

        RIni = 9.0e3_wp

        Omega(1) = 0.0004993672899010212969344114431521214348_wp
        Omega(2) = 0.0023083851082641087179457485990125009678_wp
        Omega(3) = 0.0080975548740404425835813419926045853003_wp
        Omega(4) = 0.0240502434224015340449989242346817697182_wp
        Omega(5) = 0.0636395853625634186795942751069254939011_wp
        Omega(6) = 0.1546954156882665470701572474343521435003_wp
        Omega(7) = 0.3523475646338251464513829230851271745451_wp
        Omega(8) = 0.7641947978821455716427410442825873815309_wp
        Omega(9) = 1.6171820561365553377188714057055562989262_wp
        Omega(10) = 3.6383529487866477763330363970339931256603_wp

        Alpha(1) = 0.0001763278642964950008481394511735025621_wp
        Alpha(2) = 0.0013966738489163576343957964224357093386_wp
        Alpha(3) = 0.0060847863738081463643708594359049701694_wp
        Alpha(4) = 0.0209025184439057381995591194598360651469_wp
        Alpha(5) = 0.0619119562592124370580817186576760846606_wp
        Alpha(6) = 0.1650476738341768858053760712922830578009_wp
        Alpha(7) = 0.4062886092368531241320050995113888347987_wp
        Alpha(8) = 0.9400053232670322994851512532488158058186_wp
        Alpha(9) = 2.0781646409684503076526623788211622922972_wp
        Alpha(10) = 4.5361826641383027796175508861153957695933_wp

        Xi(1) = 1.016347286163623462804106445390_wp
        Xi(2) = 1.151015790821104004676422860598_wp
        Xi(3) = 1.441356899587345926781918603599_wp
        Xi(4) = 1.931855777473684393171805828704_wp
        Xi(5) = 2.699441181486060780964769678647_wp
        Xi(6) = 3.872412335802562820177147306921_wp
        Xi(7) = 5.661566270315299894810151526414_wp
        Xi(8) = 8.412077454143638951568340278442_wp
        Xi(9) = 12.695525341059824706892167345273_wp
        Xi(10) = 19.476333699099736636653479315839_wp
        Xi(11) = 30.419560793419684807323277730973_wp
        Xi(12) = 48.480602311172952819523507272947_wp
        Xi(13) = 79.073001800190617187336084015215_wp
        Xi(14) = 132.497705706562297706119224471877_wp
        Xi(15) = 229.255331266340414531068603309905_wp
        Xi(16) = 412.402010579539868545539960109636_wp
        Xi(17) = 778.923674562525614584185262856408_wp
        Xi(18) = 1564.783963043916336643768261183141_wp
        Xi(19) = 3393.010515303080970150872985868773_wp
        Xi(20) = 7465.825669118156909132011378460447_wp

      case (30)
        ! ----- K = 10 , R = 1E4 -----

        RIni = 1.0e4_wp

        Omega(1) = 0.0004668500190071136037058161189026300741_wp
        Omega(2) = 0.0022125830949226374988479443175343197581_wp
        Omega(3) = 0.0078655446011830261521978946079314276574_wp
        Omega(4) = 0.0235485349216305695394885416985020221148_wp
        Omega(5) = 0.0626612189845160618488733965392167846176_wp
        Omega(6) = 0.1529600992483176926246520380137106087659_wp
        Omega(7) = 0.3495514360883487606104867012613013343980_wp
        Omega(8) = 0.7601490813086587507488821324752592545337_wp
        Omega(9) = 1.6120121524724761282376314652786675196694_wp
        Omega(10) = 3.6324233181727452956458895849678469858191_wp

        Alpha(1) = 0.0001638448168262916940862950973285122602_wp
        Alpha(2) = 0.0013236958800674285022613046645077738983_wp
        Alpha(3) = 0.0058563634806415089364502364350367757950_wp
        Alpha(4) = 0.0203212175461642492981981754363030034938_wp
        Alpha(5) = 0.0606111723824622499480734971050166493001_wp
        Alpha(6) = 0.1624157193093745306253445140343139030392_wp
        Alpha(7) = 0.4014143723529730106919840432189161560927_wp
        Alpha(8) = 0.9317156300794014760126987573229939698649_wp
        Alpha(9) = 2.0652402323623246008753306934835336505785_wp
        Alpha(10) = 4.5176891627407891488012181557820667876513_wp

        Xi(1) = 1.016438941992472549339515675015_wp
        Xi(2) = 1.151887410843916417150893660715_wp
        Xi(3) = 1.444041938087961746682054420265_wp
        Xi(4) = 1.937944601327246360090474086579_wp
        Xi(5) = 2.711573469892460428809846662013_wp
        Xi(6) = 3.895176051201984182168966408533_wp
        Xi(7) = 5.703096914696195845247833622338_wp
        Xi(8) = 8.486990466098886826129543692332_wp
        Xi(9) = 12.830460806532057196210905303246_wp
        Xi(10) = 19.720759648459364998460130458113_wp
        Xi(11) = 30.867341651348655255648889550457_wp
        Xi(12) = 49.314738074799366471939743306052_wp
        Xi(13) = 80.661929483772102499050671298164_wp
        Xi(14) = 135.612742585992632399216617500315_wp
        Xi(15) = 235.593109019815892993254813347903_wp
        Xi(16) = 425.923429412591868059578814609267_wp
        Xi(17) = 809.687804313004781064933013112750_wp
        Xi(18) = 1641.240732349594282579730020188435_wp
        Xi(19) = 3609.930576169334359937224121495092_wp
        Xi(20) = 8181.271691613231036743769664099091_wp

      case (31)
        ! ----- K = 10 , R = 1E5 (Best : R = 56503) -----

        RIni = 1.0e5_wp

        Omega(1) = 0.0002188017493762318239417294425826730764_wp
        Omega(2) = 0.0014375664500890244128473657598404900071_wp
        Omega(3) = 0.0058848743519528664629849115403487092024_wp
        Omega(4) = 0.0191306240822222516468974857703488900995_wp
        Omega(5) = 0.0538584874422464578561152524077693115601_wp
        Omega(6) = 0.1370884682117943122853752357936940597938_wp
        Omega(7) = 0.3236444502032899059047309914260992513846_wp
        Omega(8) = 0.7222915374704975218400800074380896376169_wp
        Omega(9) = 1.5633339391232095614897232294104867378337_wp
        Omega(10) = 3.5765361929512171882024340607841850214754_wp

        Alpha(1) = 0.0000684439013883050839929452714076145436_wp
        Alpha(2) = 0.0007529515955878675215771287905844322275_wp
        Alpha(3) = 0.0039850355998364926202256629163972467822_wp
        Alpha(4) = 0.0153807990762694658962519717732009283395_wp
        Alpha(5) = 0.0492554652626580150358520746728396844105_wp
        Alpha(6) = 0.1389707382811857045096584628574021280656_wp
        Alpha(7) = 0.3573119289310187454190587824642832970312_wp
        Alpha(8) = 0.8557928463574533264955450373445700051889_wp
        Alpha(9) = 1.9457922045681717026794493863306456660212_wp
        Alpha(10) = 4.3457510815955149355814535194753034375026_wp

        Xi(1) = 1.017332022553130616709937683595_wp
        Xi(2) = 1.160396353209442369671874906167_wp
        Xi(3) = 1.470341637704521359350171016978_wp
        Xi(4) = 1.997855369163787341403940023277_wp
        Xi(5) = 2.831657524964081751707792311556_wp
        Xi(6) = 4.122147772655313542090882439339_wp
        Xi(7) = 6.120802466554915003110054483315_wp
        Xi(8) = 9.248005859104586343436005257956_wp
        Xi(9) = 14.216952520405455036127395640477_wp
        Xi(10) = 22.265232010172739753744997059925_wp
        Xi(11) = 35.599346873582316214817966582018_wp
        Xi(12) = 58.285299504795977050541910813308_wp
        Xi(13) = 98.107643930079989784409644215657_wp
        Xi(14) = 170.678774856488366634676978605967_wp
        Xi(15) = 309.186027541697107778873032657430_wp
        Xi(16) = 589.257915242668196764697086109663_wp
        Xi(17) = 1201.556899697256635350939291129180_wp
        Xi(18) = 2693.276966514136449903915604409121_wp
        Xi(19) = 6992.095093521639840528081322190701_wp
        Xi(20) = 24579.752122908689996094722118868958_wp

    end select

  case (11)
    ! ===== K = 11 =====

    select case (InitR)

      case default !(1:3)
        ! ----- K = 11 , R = 1E1 -----

        RIni = 1.0e1_wp

        Omega(1) = 0.0775418031116303837967595156965838754104_wp
        Omega(2) = 0.1849123891997807165503506926507171215235_wp
        Omega(3) = 0.3038503048558095734777567686890620279883_wp
        Omega(4) = 0.4433256105757104311240485855494242173336_wp
        Omega(5) = 0.6164377958322638620695596567422569478367_wp
        Omega(6) = 0.8427130345849869985067233379982809537978_wp
        Omega(7) = 1.1516837447426178303111835021788067479065_wp
        Omega(8) = 1.5901310218673417025501276578225429148006_wp
        Omega(9) = 2.2427211748928490614354153143139569692721_wp
        Omega(10) = 3.3086113852215574260545033480340748610615_wp
        Omega(11) = 5.5609371683070527290528028974136987017118_wp

        Alpha(1) = 0.0301309885271363365100741023641650606635_wp
        Alpha(2) = 0.1606809945496051955719481532658399203228_wp
        Alpha(3) = 0.4037750551486719004321696013803943969833_wp
        Alpha(4) = 0.7751832496328281628091352861709140142921_wp
        Alpha(5) = 1.3015684469973897338353555075407541608001_wp
        Alpha(6) = 2.0256723922926537207432617915259243090986_wp
        Alpha(7) = 3.0143817180020071619470783952365877667035_wp
        Alpha(8) = 4.3718289090112480007807582715884109347826_wp
        Alpha(9) = 6.2650017998353700757543927046810949832434_wp
        Alpha(10) = 8.9909992515811430506197932110978854325367_wp
        Alpha(11) = 13.2454300125589543215415266885770506632980_wp

        Xi(1) = 1.003273313513527648056442020952_wp
        Xi(2) = 1.029590869049646218238991401961_wp
        Xi(3) = 1.082926499973093964869844119114_wp
        Xi(4) = 1.164689536100922592743410477478_wp
        Xi(5) = 1.277018303784519149537621252044_wp
        Xi(6) = 1.422814405817666808320987037462_wp
        Xi(7) = 1.605789617280226246032702464017_wp
        Xi(8) = 1.830512445012425343715198322592_wp
        Xi(9) = 2.102430837927628287827019892831_wp
        Xi(10) = 2.427834409249072842359312707217_wp
        Xi(11) = 2.813696452389779884776382923306_wp
        Xi(12) = 3.267308650574832364944788398731_wp
        Xi(13) = 3.795571461478587432228012921165_wp
        Xi(14) = 4.403743151478856171499032257088_wp
        Xi(15) = 5.093394497119079525251222895221_wp
        Xi(16) = 5.859312922846066839718814911464_wp
        Xi(17) = 6.685280709353001782780517858740_wp
        Xi(18) = 7.539203178158188994872901389499_wp
        Xi(19) = 8.369184439007884772354639579817_wp
        Xi(20) = 9.103695956024776096693340576849_wp
        Xi(21) = 9.659796125678525681619579490444_wp
        Xi(22) = 9.961142800362639472544490359951_wp

      case (4)
        ! ----- K = 11 , R = 2E1 -----

        RIni = 2.0e1_wp

        Omega(1) = 0.0466937996822860070988069850633772617243_wp
        Omega(2) = 0.1126033802262315067067399536326899323058_wp
        Omega(3) = 0.1890391440443657539844084930491519003226_wp
        Omega(4) = 0.2851482966031922318276073391452030136861_wp
        Omega(5) = 0.4150403333396616663882993519818498384666_wp
        Omega(6) = 0.5999483249149235138055098748299798216976_wp
        Omega(7) = 0.8710338480743368342988641739310651246342_wp
        Omega(8) = 1.2759883242857371767964638764603080289817_wp
        Omega(9) = 1.8993330622314406922693935664270270535781_wp
        Omega(10) = 2.9368796146680767613322549092202962128795_wp
        Omega(11) = 5.1400782864171481395879070852572567673633_wp

        Alpha(1) = 0.0181210797127243208924727430053636112461_wp
        Alpha(2) = 0.0971750380905392633493276692557838458697_wp
        Alpha(3) = 0.2467985172992078721931506410380841032293_wp
        Alpha(4) = 0.4817566785490354173688946903819640965594_wp
        Alpha(5) = 0.8282771166045613832600719161902702580846_wp
        Alpha(6) = 1.3300696805608714204426437643924430176412_wp
        Alpha(7) = 2.0567305766295164660886068963563388933835_wp
        Alpha(8) = 3.1164004963694246137274890262602866641828_wp
        Alpha(9) = 4.6805076135644237333641171083797871688148_wp
        Alpha(10) = 7.0492221261381141847457421079070627456531_wp
        Alpha(11) = 10.9116044227192424539638060387858331523603_wp

        Xi(1) = 1.004488687272064465568106872428_wp
        Xi(2) = 1.040655880503802272613045332150_wp
        Xi(3) = 1.114370866869753685913201068924_wp
        Xi(4) = 1.228430737034884367444263764124_wp
        Xi(5) = 1.387139795240555471041860036152_wp
        Xi(6) = 1.596477373296791676471687027039_wp
        Xi(7) = 1.864351775240325198123962635588_wp
        Xi(8) = 2.200928373803418052097324864036_wp
        Xi(9) = 2.619005200711196410575526405395_wp
        Xi(10) = 3.134392695918814298466914536334_wp
        Xi(11) = 3.766234046800330914411478189230_wp
        Xi(12) = 4.537127178375117797764354277135_wp
        Xi(13) = 5.472770514940351053306821782485_wp
        Xi(14) = 6.600612423468904096848092200034_wp
        Xi(15) = 7.946569514961566195465625384031_wp
        Xi(16) = 9.528324635071206902270368743757_wp
        Xi(17) = 11.343159668603970342652675906692_wp
        Xi(18) = 13.348482476294802900190772376732_wp
        Xi(19) = 15.436126389431109655510365730180_wp
        Xi(20) = 17.410130365170752750683669241027_wp
        Xi(21) = 18.991985340400373383912269176221_wp
        Xi(22) = 19.883248872480938938822614403534_wp

      case (5)
        ! ----- K = 11 , R = 3E1 -----

        RIni = 3.0e1_wp

        Omega(1) = 0.0343853828194110494539958728515971486672_wp
        Omega(2) = 0.0836001053696773024971632994051251586143_wp
        Omega(3) = 0.1425876773323265062450764320556295672304_wp
        Omega(4) = 0.2204440782713079492494438402228063011989_wp
        Omega(5) = 0.3315518760784969383925094993381676999888_wp
        Omega(6) = 0.4974967407138457998094956125090249088316_wp
        Omega(7) = 0.7496946753853811701642784193211355159292_wp
        Omega(8) = 1.1363279712558223806875834505092370818602_wp
        Omega(9) = 1.7427500950293133717607790833525882590038_wp
        Omega(10) = 2.7645508616331733189007413553639480596757_wp
        Omega(11) = 4.9444585797781260795222268988879932294367_wp

        Alpha(1) = 0.0133321421683787708297889767691812856754_wp
        Alpha(2) = 0.0717844674758855171126212431353952325708_wp
        Alpha(3) = 0.1837505079175499636242110956807493948872_wp
        Alpha(4) = 0.3631906488145699331954695254776055435286_wp
        Alpha(5) = 0.6356426510512829221245974242737730719455_wp
        Alpha(6) = 1.0444692700724251522258201330117799443542_wp
        Alpha(7) = 1.6592152251340612241882824240413185634679_wp
        Alpha(8) = 2.5883048394719798256536497094160154119891_wp
        Alpha(9) = 4.0041452021372941614575646784857099191868_wp
        Alpha(10) = 6.2084129877437377363971759969274444301846_wp
        Alpha(11) = 9.8885546801249998207644642533864498545881_wp

        Xi(1) = 1.005238982501608224241762545859_wp
        Xi(2) = 1.047506997148030853444708454525_wp
        Xi(3) = 1.133953004614822940255766670070_wp
        Xi(4) = 1.268458006813831051920460346594_wp
        Xi(5) = 1.457034379328764806763724104499_wp
        Xi(6) = 1.708130758025110139787863805516_wp
        Xi(7) = 2.033102713009531136269594786548_wp
        Xi(8) = 2.446840637749924481612884274995_wp
        Xi(9) = 2.968531831765823256327405110788_wp
        Xi(10) = 3.622542791092627238215551077616_wp
        Xi(11) = 4.439389335501653323281384677657_wp
        Xi(12) = 5.456670502894996907169988809017_wp
        Xi(13) = 6.719657910243575584235270481415_wp
        Xi(14) = 8.280823895686300801372015101265_wp
        Xi(15) = 10.196868876272738388237548701198_wp
        Xi(16) = 12.520501353935608300840809370413_wp
        Xi(17) = 15.282277337325394486224061640911_wp
        Xi(18) = 18.455974983409653485122436933352_wp
        Xi(19) = 21.902872540956904594960930054981_wp
        Xi(20) = 25.305408442239061496750740509754_wp
        Xi(21) = 28.140092899406508115262015579106_wp
        Xi(22) = 29.782364787011379817990852636456_wp

      case (6)
        ! ----- K = 11 , R = 4E1 -----

        RIni = 4.0e1_wp

        Omega(1) = 0.0275873899402055327081950686363143177005_wp
        Omega(2) = 0.0675179313170176000007236156996714981915_wp
        Omega(3) = 0.1166597880630956601271546770048903596262_wp
        Omega(4) = 0.1840049234146371288326372167576394645039_wp
        Omega(5) = 0.2839493541216279612023672823672981735399_wp
        Omega(6) = 0.4380335127182523924686211730383789131338_wp
        Omega(7) = 0.6776838734799814061167920475892145759644_wp
        Omega(8) = 1.0515697426467314990958246134766795876203_wp
        Omega(9) = 1.6460013694707742813033712891979121195618_wp
        Omega(10) = 2.6569503323667347158536927720362541549548_wp
        Omega(11) = 4.8222004072349348761071519131604645735933_wp

        Alpha(1) = 0.0106884539456373632020807530607076074602_wp
        Alpha(2) = 0.0577396059898886379488326323072833190508_wp
        Alpha(3) = 0.1487543597431635365247251656661831020756_wp
        Alpha(4) = 0.2970663897990963662326929090040650294213_wp
        Alpha(5) = 0.5275519950057114952103598515265048263245_wp
        Alpha(6) = 0.8829075520464834679916152349044011771184_wp
        Alpha(7) = 1.4319631560494335048485531336481813013961_wp
        Alpha(8) = 2.2826484117443071109650881700403601826110_wp
        Alpha(9) = 3.6076327856256673880415430932799836227787_wp
        Alpha(10) = 5.7096768822342786236026523916109454148682_wp
        Alpha(11) = 9.2758730438427994067357440499677068146411_wp

        Xi(1) = 1.005783326006271300085727349316_wp
        Xi(2) = 1.052487601408567548114461609465_wp
        Xi(3) = 1.148244607841531594546492789366_wp
        Xi(4) = 1.297835051320845218083589733826_wp
        Xi(5) = 1.508698769283583550923459604132_wp
        Xi(6) = 1.791373445471515501786416435515_wp
        Xi(7) = 2.160176288700313776587613734392_wp
        Xi(8) = 2.634125137047316175245223468515_wp
        Xi(9) = 3.238087431930898431667320558347_wp
        Xi(10) = 4.004184086936066402830775023958_wp
        Xi(11) = 4.973462649229208518229100599584_wp
        Xi(12) = 6.197780262115873076405464514593_wp
        Xi(13) = 7.741592558633213706081876992471_wp
        Xi(14) = 9.682877078701319828804572331649_wp
        Xi(15) = 12.111404184952905989207383452566_wp
        Xi(16) = 15.120567916287438012430677236608_wp
        Xi(17) = 18.785452511544115491298745368454_wp
        Xi(18) = 23.114867093063940418906643259334_wp
        Xi(19) = 27.963223265502905141507983266536_wp
        Xi(20) = 32.905737217259679330827459864395_wp
        Xi(21) = 37.149516720183049747761039327543_wp
        Xi(22) = 39.663654492691002611398642940799_wp

      case (7)
        ! ----- K = 11 , R = 5E1 -----

        RIni = 5.0e1_wp

        Omega(1) = 0.0232195566793358917775057497639989989580_wp
        Omega(2) = 0.0571513181735215659321164369949253014624_wp
        Omega(3) = 0.0998552854507156438591580945030035110221_wp
        Omega(4) = 0.1602055619046999012056132702386257449234_wp
        Omega(5) = 0.2525035438948522461502944197464515241336_wp
        Omega(6) = 0.3981171244943776991954877825241609912155_wp
        Omega(7) = 0.6284424002596707273856496689390382925922_wp
        Omega(8) = 0.9926088267135148024964720614438107304522_wp
        Omega(9) = 1.5778155724565829864328853071597791313252_wp
        Omega(10) = 2.5805592790528422068689617940862035538885_wp
        Omega(11) = 4.7353741399348671268706989501140469656093_wp

        Alpha(1) = 0.0089905001583754167007777116649158699602_wp
        Alpha(2) = 0.0487041770923081894940215308409392491740_wp
        Alpha(3) = 0.1261762122605637203871103979579970655323_wp
        Alpha(4) = 0.2542357904458434050744969345814183725452_wp
        Alpha(5) = 0.4571594292572615855362212766710428013539_wp
        Alpha(6) = 0.7769092949171712697564728955512691754848_wp
        Alpha(7) = 1.2814468596702209815443168627879799714719_wp
        Alpha(8) = 2.0780301498130218532198881797867784371192_wp
        Alpha(9) = 3.3393616918465426032010950896378176366852_wp
        Alpha(10) = 5.3690541246141367530582411138340148681891_wp
        Alpha(11) = 8.8542885137191006629173162245649564283667_wp

        Xi(1) = 1.006210109757504921426142507901_wp
        Xi(2) = 1.056398646441784029150444401690_wp
        Xi(3) = 1.159500533992196700345742033100_wp
        Xi(4) = 1.321070476658640847097796189313_wp
        Xi(5) = 1.549783157108070042362775453615_wp
        Xi(6) = 1.857999745604813185938242181550_wp
        Xi(7) = 2.262653987453266012618013913738_wp
        Xi(8) = 2.786454618005919255802060874316_wp
        Xi(9) = 3.459413085995143127306106212160_wp
        Xi(10) = 4.320768676526033707455815013887_wp
        Xi(11) = 5.421392081002391185717143029521_wp
        Xi(12) = 6.826671040936758972240278042776_wp
        Xi(13) = 8.619645170878245339640688627014_wp
        Xi(14) = 10.903625654862894262496497699289_wp
        Xi(15) = 13.802322586481959816595566081787_wp
        Xi(16) = 17.452867307040970041784322397405_wp
        Xi(17) = 21.981989252163744585563676103845_wp
        Xi(18) = 27.447010430433434611391385615775_wp
        Xi(19) = 33.715502395166521192038100451782_wp
        Xi(20) = 40.272848099043959756160315777151_wp
        Xi(21) = 46.045079807836921025054888190198_wp
        Xi(22) = 49.529969194038888841147283415012_wp

      case (8)
        ! ----- K = 11 , R = 6E1 -----

        RIni = 6.0e1_wp

        Omega(1) = 0.0201523470175281204055153230422781085451_wp
        Omega(2) = 0.0498513967434951159696883329214278646191_wp
        Omega(3) = 0.0879662646198283677617061476761151084247_wp
        Omega(4) = 0.1432515275765687200420876734607844582570_wp
        Omega(5) = 0.2298666555952093790837894790590745230929_wp
        Omega(6) = 0.3689707967994120758393507275574307868737_wp
        Omega(7) = 0.5919302613549222069535410417628895629605_wp
        Omega(8) = 0.9482919217093114909998032013849211807610_wp
        Omega(9) = 1.5260458052374904870185715055619368740736_wp
        Omega(10) = 2.5222361903145047716968268902704153333616_wp
        Omega(11) = 4.6690760135659036588265391465313314256491_wp

        Alpha(1) = 0.0077985503072843401969374522390149317985_wp
        Alpha(2) = 0.0423524201109992350250692174501443432177_wp
        Alpha(3) = 0.1102652551111842159071952870763588805403_wp
        Alpha(4) = 0.2239475061574449626645945052305641809198_wp
        Alpha(5) = 0.4071358003831677819096650228214429034779_wp
        Alpha(6) = 0.7010697302865118014881737340537881664204_wp
        Alpha(7) = 1.1728396207014495131832829133422535505815_wp
        Alpha(8) = 1.9290152408304663420252414440803079287434_wp
        Alpha(9) = 3.1422391785388204986026461362769168772502_wp
        Alpha(10) = 5.1168202530289228812179769256829331425251_wp
        Alpha(11) = 8.5402070139792344233634069361471574666211_wp

        Xi(1) = 1.006560623915288661504863210183_wp
        Xi(2) = 1.059614804442524975619868699006_wp
        Xi(3) = 1.168778918256115415875463769968_wp
        Xi(4) = 1.340289416578526023580454629158_wp
        Xi(5) = 1.583913090945669306501032702617_wp
        Xi(6) = 1.913637316052279297102015065413_wp
        Xi(7) = 2.348750746943249423370758677265_wp
        Xi(8) = 2.915315924424118237002351294862_wp
        Xi(9) = 3.648065668999952153956842226812_wp
        Xi(10) = 4.592844553199314636213884677218_wp
        Xi(11) = 5.809747422460572900682063135491_wp
        Xi(12) = 7.377041094945992064872003490805_wp
        Xi(13) = 9.395727948083074930127334134511_wp
        Xi(14) = 11.994057570426518195480514172857_wp
        Xi(15) = 15.329926577711685170384015020062_wp
        Xi(16) = 19.585963444057799367689631786504_wp
        Xi(17) = 24.945356665340155425183987780713_wp
        Xi(18) = 31.524101840103794855765806381953_wp
        Xi(19) = 39.219801161246302452501044655264_wp
        Xi(20) = 47.445806203028849497294050507890_wp
        Xi(21) = 54.842855007443132461103507679923_wp
        Xi(22) = 59.383169485026616251505116039766_wp

      case (9)
        ! ----- K = 11 , R = 7E1 -----

        RIni = 7.0e1_wp

        Omega(1) = 0.0178680837427477367867698077041893611749_wp
        Omega(2) = 0.0444015974630119514095422973698479296445_wp
        Omega(3) = 0.0790534342226611162648920380635608751163_wp
        Omega(4) = 0.1304614133244965352645275760701260026053_wp
        Omega(5) = 0.2126234626501610114779772936488555501455_wp
        Omega(6) = 0.3464873698648142582049921922582313982275_wp
        Omega(7) = 0.5633974185429598534346519456139645853909_wp
        Omega(8) = 0.9132730201360275555286759308071253826711_wp
        Omega(9) = 1.4848031156704543130154042018808979719324_wp
        Omega(10) = 2.4755646905106925954008489920710189835518_wp
        Omega(11) = 4.6160207288556499025458657037290777225280_wp

        Alpha(1) = 0.0069111231519738776855104537936991260239_wp
        Alpha(2) = 0.0376175443778602281864995200699741673134_wp
        Alpha(3) = 0.0983788554316176733961925825189265282233_wp
        Alpha(4) = 0.2012493936056623861145754503243487931741_wp
        Alpha(5) = 0.3694783097483234116608068892118055259743_wp
        Alpha(6) = 0.6436206330275095796708724471546503309582_wp
        Alpha(7) = 1.0899407695335288343398486010649151012331_wp
        Alpha(8) = 1.8143508270266151124420345164267587279028_wp
        Alpha(9) = 2.9893905748368822881521483791900095638994_wp
        Alpha(10) = 4.9199519381062117805679390070139334056876_wp
        Alpha(11) = 8.2938255368030769556358938743301223439630_wp

        Xi(1) = 1.006857599192093716614082588023_wp
        Xi(2) = 1.062342605357837705478026013584_wp
        Xi(3) = 1.176664399235137160783931653540_wp
        Xi(4) = 1.356669684735823086670601644865_wp
        Xi(5) = 1.613107790238305128604698324146_wp
        Xi(6) = 1.961437595963733477992009535118_wp
        Xi(7) = 2.423095507358677820532871782788_wp
        Xi(8) = 3.027228083368775823826929416427_wp
        Xi(9) = 3.812945912708375304943017591874_wp
        Xi(10) = 4.832271776130559904922218139589_wp
        Xi(11) = 6.154016021838048258307124482869_wp
        Xi(12) = 7.868747300855569675945289098706_wp
        Xi(13) = 10.094835069599521508096340305372_wp
        Xi(14) = 12.984989087355007315351118624136_wp
        Xi(15) = 16.731241509380213740298959379516_wp
        Xi(16) = 21.562745035293026557229878470423_wp
        Xi(17) = 27.722554796770373059661141112997_wp
        Xi(18) = 35.393240225111605889996013729615_wp
        Xi(19) = 44.516783567365161258105876740387_wp
        Xi(20) = 54.451824494240437701114698754168_wp
        Xi(21) = 63.554269878117245724108430593446_wp
        Xi(22) = 69.224585099912302292579635576431_wp

      case (10)
        ! ----- K = 11 , R = 8E1 -----

        RIni = 8.0e1_wp

        Omega(1) = 0.0160941951431383953638438273278654655485_wp
        Omega(2) = 0.0401601741924849507256662425186410914080_wp
        Omega(3) = 0.0720906972120942371015021692126634178521_wp
        Omega(4) = 0.1204112625174464150731726622134409865339_wp
        Omega(5) = 0.1989522181966777404301966597488515731129_wp
        Omega(6) = 0.3284599168601711009763104848202530661183_wp
        Omega(7) = 0.5402638660330096453542243761791752376666_wp
        Omega(8) = 0.8846145384436353546904995004407368242028_wp
        Omega(9) = 1.4508208171080411968533602218300870845269_wp
        Omega(10) = 2.4369656699328211739517119349685003726336_wp
        Omega(11) = 4.5721418768127472043519277100642739242176_wp

        Alpha(1) = 0.0062221568462019169464015996146022402513_wp
        Alpha(2) = 0.0339374511936524437029913999624985976311_wp
        Alpha(3) = 0.0891223549172805987108204189839355535696_wp
        Alpha(4) = 0.1835224902391902816436086362528179449782_wp
        Alpha(5) = 0.3399444112314101982587084133102806049465_wp
        Alpha(6) = 0.5983039637414345515687395804516057751243_wp
        Alpha(7) = 1.0240978685121112761726921980276472368132_wp
        Alpha(8) = 1.7226238644716444793061546625345670236129_wp
        Alpha(9) = 2.8662979670373295178718853204458127947873_wp
        Alpha(10) = 4.7605096085039891777926368554574310110183_wp
        Alpha(11) = 8.0934197853927114085087479722346870403271_wp

        Xi(1) = 1.007114911323129302742236668156_wp
        Xi(2) = 1.064708254106896957421127258847_wp
        Xi(3) = 1.183514884140128698439622945138_wp
        Xi(4) = 1.370935080634906710244161343049_wp
        Xi(5) = 1.638612033508315807034567967992_wp
        Xi(6) = 2.003351900306294064001838561850_wp
        Xi(7) = 2.488570524563886765144135604366_wp
        Xi(8) = 3.126275779556943167037871833536_wp
        Xi(9) = 3.959667397510368679969597338619_wp
        Xi(10) = 5.046586336691558532119988855769_wp
        Xi(11) = 6.464118428179701520124716962457_wp
        Xi(12) = 8.314620795618188192448172468829_wp
        Xi(13) = 10.733276889443528670682603820552_wp
        Xi(14) = 13.896753202124445932853780139027_wp
        Xi(15) = 18.031001346196642312613711567337_wp
        Xi(16) = 23.412291370075801024211448186918_wp
        Xi(17) = 30.346026098270278199831029297684_wp
        Xi(18) = 39.087647695362775564220658708336_wp
        Xi(19) = 49.635855874445203150119310109289_wp
        Xi(20) = 61.311065980470514071021259638883_wp
        Xi(21) = 72.187931301061125745344160264949_wp
        Xi(22) = 79.055224610952105873673545488600_wp

      case (11)
        ! ----- K = 11 , R = 9E1 -----

        RIni = 9.0e1_wp

        Omega(1) = 0.0146727601814103851884249431376805450356_wp
        Omega(2) = 0.0367546982817821544715951916776353058935_wp
        Omega(3) = 0.0664809481604872105456089000086183915528_wp
        Omega(4) = 0.1122696684291449136910414838275151794278_wp
        Omega(5) = 0.1877844975152047632752094763475270156050_wp
        Omega(6) = 0.3135844709409314501086674803476128658986_wp
        Omega(7) = 0.5209897018134802104249309051553495919507_wp
        Omega(8) = 0.8605452139568929573664943233346491524571_wp
        Omega(9) = 1.4221134659936936344174981949883829202008_wp
        Omega(10) = 2.4042539239717671687227018573196346551413_wp
        Omega(11) = 4.5349558958883104250667250845197031594580_wp

        Alpha(1) = 0.0056702141892506984487967117841700215308_wp
        Alpha(2) = 0.0309862655699217783785727724915615155510_wp
        Alpha(3) = 0.0816860387399204145020530781762246874678_wp
        Alpha(4) = 0.1692434066400484682065122332694784290652_wp
        Alpha(5) = 0.3160607081649595111595522911596134463252_wp
        Alpha(6) = 0.5614600806021925772119549047367570437927_wp
        Alpha(7) = 0.9702293534113280694072302401398388838061_wp
        Alpha(8) = 1.6470963177289165131709364198187017791497_wp
        Alpha(9) = 2.7643431184907410428242685584976356949483_wp
        Alpha(10) = 4.6277899426647656983338507963310348714003_wp
        Alpha(11) = 7.9259744054270826574097597028156769738416_wp

        Xi(1) = 1.007341663573080700199054160482_wp
        Xi(2) = 1.066794619685856562035786154752_wp
        Xi(3) = 1.189565740566661882176950626455_wp
        Xi(4) = 1.383562565681869578225124395310_wp
        Xi(5) = 1.661249550979290727135105620871_wp
        Xi(6) = 2.040676979969849535361328674021_wp
        Xi(7) = 2.547099296104168583611965237523_wp
        Xi(8) = 3.215197927414357416658202692972_wp
        Xi(9) = 4.092017650220201428149652267408_wp
        Xi(10) = 5.240906303037136232478993003170_wp
        Xi(11) = 6.746840074156372169585788034496_wp
        Xi(12) = 8.723505522613407395064732963164_wp
        Xi(13) = 11.322386074322037554512609958124_wp
        Xi(14) = 14.743607175741540703424459213267_wp
        Xi(15) = 19.246733999116489780348282501166_wp
        Xi(16) = 25.155461057089254839347347569856_wp
        Xi(17) = 32.839384846893302876552622748818_wp
        Xi(18) = 42.631954001327561701306656516408_wp
        Xi(19) = 54.599273429896236945607634893918_wp
        Xi(20) = 68.039133671991829224734438241740_wp
        Xi(21) = 80.750605961361787560326419566081_wp
        Xi(22) = 88.875884517700640997939220966373_wp

      case (12)
        ! ----- K = 11 , R = 1E2 -----

        RIni = 1.0e2_wp

        Omega(1) = 0.0135056228913994452729794171902399924079_wp
        Omega(2) = 0.0339533364876027449583584683773240797677_wp
        Omega(3) = 0.0618515718361504429331830610255815727783_wp
        Omega(4) = 0.1055161868702422083980988973261894159350_wp
        Omega(5) = 0.1784485164513013136943940170620059859630_wp
        Omega(6) = 0.3010351533892806076249740176997704566020_wp
        Omega(7) = 0.5045904598592531176809959647666659066090_wp
        Omega(8) = 0.8399224594117821994791912854072535310479_wp
        Omega(9) = 1.3973915223155492387231702000605082503171_wp
        Omega(10) = 2.3760047579964309286560536493837503257964_wp
        Omega(11) = 4.5028432521787169753521573589694071415579_wp

        Alpha(1) = 0.0052171157900164305121616745847257234914_wp
        Alpha(2) = 0.0285613209731876607790204474976447635015_wp
        Alpha(3) = 0.0755656714563188581487750486442855901714_wp
        Alpha(4) = 0.1574618406901295399800922922284307503560_wp
        Alpha(5) = 0.2962809494169539860049852192380726023657_wp
        Alpha(6) = 0.5307942425023796681840017319675695262049_wp
        Alpha(7) = 0.9251356226523106539728208419504085213703_wp
        Alpha(8) = 1.5835045879490817360719240314281819337339_wp
        Alpha(9) = 2.6780450591320426989138531181922076029878_wp
        Alpha(10) = 4.5149547171879260030677438741975038283272_wp
        Alpha(11) = 7.7831431411206166304933562472001540299971_wp

        Xi(1) = 1.007544151737991758224083360673_wp
        Xi(2) = 1.068659033452348684181849869379_wp
        Xi(3) = 1.194980270598195126806155552668_wp
        Xi(4) = 1.394883709874343532099730258444_wp
        Xi(5) = 1.681594294760566527142942039319_wp
        Xi(6) = 2.074319473268862619712987749310_wp
        Xi(7) = 2.600032748539986276288191180406_wp
        Xi(8) = 3.295928120544334691916985025095_wp
        Xi(9) = 4.212683432651104258569174643156_wp
        Xi(10) = 5.418881494632919341557925063668_wp
        Xi(11) = 7.007049298105176418204748101104_wp
        Xi(12) = 9.101789195740375220181062054792_wp
        Xi(13) = 11.870409633437578280322488577525_wp
        Xi(14) = 15.536009144308547636099959721179_wp
        Xi(15) = 20.391410969809726090942736398404_wp
        Xi(16) = 26.807857506229526480673319710490_wp
        Xi(17) = 35.220527963930422440941292450134_wp
        Xi(18) = 46.045097106276776009747830897822_wp
        Xi(19) = 59.424485080890632523326067016001_wp
        Xi(20) = 74.648490737217016943727543321074_wp
        Xi(21) = 89.247781400980597130145888229436_wp
        Xi(22) = 98.687213059750639010392347927336_wp

      case (13)
        ! ----- K = 11 , R = 2E2 -----

        RIni = 2.0e2_wp

        Omega(1) = 0.0078046543400422635145728894948136122878_wp
        Omega(2) = 0.0201790394732283336855016087421632597909_wp
        Omega(3) = 0.0388177376499393517197641391425211310207_wp
        Omega(4) = 0.0712382820959274406944156457499595092031_wp
        Omega(5) = 0.1296799649960446518400739718268610545238_wp
        Omega(6) = 0.2333959079982976153372267041907761608854_wp
        Omega(7) = 0.4136818764549498482409762428746446971672_wp
        Omega(8) = 0.7229529177297216195260953774948120553745_wp
        Omega(9) = 1.2547944496604302322016397108761509571195_wp
        Omega(10) = 2.2115303605695885785179660709687254893652_wp
        Omega(11) = 4.3158752146954668738390581417263547336916_wp

        Alpha(1) = 0.0030056959919188562807105142291388966136_wp
        Alpha(2) = 0.0166858950303621396647486396086490501034_wp
        Alpha(3) = 0.0454121585139150468891477540472889629086_wp
        Alpha(4) = 0.0988641643548232611750045152609889242967_wp
        Alpha(5) = 0.1964885269077900034113580082784089597681_wp
        Alpha(6) = 0.3731861006919959337360323536891826279316_wp
        Alpha(7) = 0.6885800478141612706770545437873209948521_wp
        Alpha(8) = 1.2431291028181766957069406820401979985036_wp
        Alpha(9) = 2.2076885241065623104173010204931415501051_wp
        Alpha(10) = 3.8906895639880266766531813349416779601597_wp
        Alpha(11) = 6.9840797596545937373814505466640412123525_wp

        Xi(1) = 1.008857660934746233814124360961_wp
        Xi(2) = 1.080784273809255970695726578601_wp
        Xi(3) = 1.230364761635588597733781401899_wp
        Xi(4) = 1.469372260898910083869173259785_wp
        Xi(5) = 1.816609709388716484041066678845_wp
        Xi(6) = 2.299915446269172837048463953202_wp
        Xi(7) = 2.959328043960797540801524463383_wp
        Xi(8) = 3.851503823852511351625085844752_wp
        Xi(9) = 5.055838276384788621594518209790_wp
        Xi(10) = 6.683232597826647388220000500780_wp
        Xi(11) = 8.888742086174264016990020920161_wp
        Xi(12) = 11.889757156156335888251562860063_wp
        Xi(13) = 15.991943396208497255722713603632_wp
        Xi(14) = 21.625657965206819011508287964318_wp
        Xi(15) = 29.395022467630873463234619791251_wp
        Xi(16) = 40.137368984784588022557416664426_wp
        Xi(17) = 54.973256143442616388633537027886_wp
        Xi(18) = 75.270314164360716667478978791905_wp
        Xi(19) = 102.281561292680015257117176474821_wp
        Xi(20) = 135.856224216334324300659019968407_wp
        Xi(21) = 171.364500755064211867662038457638_wp
        Xi(22) = 196.376475465801705613810845818534_wp

      case (14)
        ! ----- K = 11 , R = 3E2 -----

        RIni = 3.0e2_wp

        Omega(1) = 0.0056543295301813358414011055803893857075_wp
        Omega(2) = 0.0149238456038072123943028965929735729645_wp
        Omega(3) = 0.0298410617506146968957271093172978204677_wp
        Omega(4) = 0.0573864042420610532420062552955153378775_wp
        Omega(5) = 0.1090165155437165930358200602312557236928_wp
        Omega(6) = 0.2033835540589411647522700976598031274989_wp
        Omega(7) = 0.3717369601140614015079115250994590269329_wp
        Omega(8) = 0.6672646318758590398508880048300540011041_wp
        Omega(9) = 1.1853197469033224647214433411512857219350_wp
        Omega(10) = 2.1303494112152294552315484610360840633803_wp
        Omega(11) = 4.2235788700889298776484925834751038564718_wp

        Alpha(1) = 0.0021727378031523375944273458000230436760_wp
        Alpha(2) = 0.0121865964288996270824291085453638938674_wp
        Alpha(3) = 0.0338656294644117483577108133202537842976_wp
        Alpha(4) = 0.0760307668100809659503204127051212424249_wp
        Alpha(5) = 0.1565903606314177170491692969722663519860_wp
        Alpha(6) = 0.3081737637671490040143854050036509306665_wp
        Alpha(7) = 0.5877869036139700349209193108723781051594_wp
        Alpha(8) = 1.0936113751271432683849840761247662612732_wp
        Alpha(9) = 1.9954974609163636708455621948665736908879_wp
        Alpha(10) = 3.6029405911917065715322633367989624275651_wp
        Alpha(11) = 6.6099436566276548320371386768812271839124_wp

        Xi(1) = 1.009600799991315490770904528262_wp
        Xi(2) = 1.087668626770462309769071929111_wp
        Xi(3) = 1.250588229153527876618551273324_wp
        Xi(4) = 1.512339286585170538852046506761_wp
        Xi(5) = 1.895400881497233290832668539672_wp
        Xi(6) = 2.433427157673583628508251486267_wp
        Xi(7) = 3.175475198888437448592816925164_wp
        Xi(8) = 4.191958353626640123874153021077_wp
        Xi(9) = 5.583113562397886250193712598389_wp
        Xi(10) = 7.491445875865887293193479123943_wp
        Xi(11) = 10.120153278409915897743942547748_wp
        Xi(12) = 13.760407643010427368446269014157_wp
        Xi(13) = 18.831750256110034424592014623556_wp
        Xi(14) = 25.941631829303968593408713516624_wp
        Xi(15) = 35.971610122283145787730473941224_wp
        Xi(16) = 50.195840854900154449219629171353_wp
        Xi(17) = 70.420611278164901382370377547204_wp
        Xi(18) = 99.061839343598339889529480473129_wp
        Xi(19) = 138.826312319355694380185184400034_wp
        Xi(20) = 190.933104513611828043084805983653_wp
        Xi(21) = 249.601758117843898668319901901214_wp
        Xi(22) = 293.453465815889291107287917270696_wp

      case (15)
        ! ----- K = 11 , R = 4E2 -----

        RIni = 4.0e2_wp

        Omega(1) = 0.0044976260521517319245515960889070239404_wp
        Omega(2) = 0.0120741168056918430547008215525384144939_wp
        Omega(3) = 0.0248952788581032363513678056302080854323_wp
        Omega(4) = 0.0495477165718859348641492021392274125446_wp
        Omega(5) = 0.0969568792731942327023245478678781239523_wp
        Omega(6) = 0.1853828690454563002214249944898938338156_wp
        Omega(7) = 0.3460084439065048523631942800626859479962_wp
        Omega(8) = 0.6324824500366259017207146153616292849620_wp
        Omega(9) = 1.1413356299301324647040706272704824186803_wp
        Omega(10) = 2.0785533474380979603989699766586340956565_wp
        Omega(11) = 4.1646774569138222208547484015639383869711_wp

        Alpha(1) = 0.0017251108842001406274932291868597078466_wp
        Alpha(2) = 0.0097587036631565804408185399280994953841_wp
        Alpha(3) = 0.0275863189794949365422823414975561640006_wp
        Alpha(4) = 0.0634477708115134323541657243583635761297_wp
        Alpha(5) = 0.1341890869411384295140782230149056886148_wp
        Alpha(6) = 0.2708946251364908589561852209071091124315_wp
        Alpha(7) = 0.5287828806322352404511652479435213081160_wp
        Alpha(8) = 1.0044247033581155623691363865113146403019_wp
        Alpha(9) = 1.8668735332583040877974314697418378727889_wp
        Alpha(10) = 3.4262637969634559642444215921130989954690_wp
        Alpha(11) = 6.3781033361686373488186330149574132519774_wp

        Xi(1) = 1.010112021550399436756358106937_wp
        Xi(2) = 1.092414881681432928250963243499_wp
        Xi(3) = 1.264587592050053307447289585586_wp
        Xi(4) = 1.542250938775270479035202353391_wp
        Xi(5) = 1.950643916300210820025107238962_wp
        Xi(6) = 2.527848505185540314626688185307_wp
        Xi(7) = 3.329887321827251532753433327372_wp
        Xi(8) = 4.437960282291271765179457853279_wp
        Xi(9) = 5.968915119305805363266598240202_wp
        Xi(10) = 8.090901694363919286097364125254_wp
        Xi(11) = 11.046947046654886710054244147727_wp
        Xi(12) = 15.190517045401885963193389095238_wp
        Xi(13) = 21.039401899348980558637767224184_wp
        Xi(14) = 29.357464499375479619405759912354_wp
        Xi(15) = 41.277759212052887664856637428556_wp
        Xi(16) = 58.482446156036501781760872376026_wp
        Xi(17) = 83.442682564773062829643901494592_wp
        Xi(18) = 119.645028020573583910080017744804_wp
        Xi(19) = 171.414887890938018963349342982383_wp
        Xi(20) = 241.858695941404226781301645132771_wp
        Xi(21) = 324.959591817497274235382320739518_wp
        Xi(22) = 390.041181302548662090945441605072_wp

      case (16)
        ! ----- K = 11 , R = 5E2 -----

        RIni = 5.0e2_wp

        Omega(1) = 0.0037662887410819571916741684437096582272_wp
        Omega(2) = 0.0102607532707418736217397522113292662738_wp
        Omega(3) = 0.0217061831502438882615679295595700892818_wp
        Omega(4) = 0.0443837939780572404802430099002297225752_wp
        Omega(5) = 0.0888304726922509532856774006059907833333_wp
        Omega(6) = 0.1730191077007708891810806861588645233496_wp
        Omega(7) = 0.3280594190240620734114879164522449173091_wp
        Omega(8) = 0.6079089945236058984898153134679432696430_wp
        Omega(9) = 1.1099636389059035398432437324878208073642_wp
        Omega(10) = 2.0414044190001449170963898627206845048931_wp
        Omega(11) = 4.1224225984546086082316307397377386223525_wp

        Alpha(1) = 0.0014423138924763025197926043403928719755_wp
        Alpha(2) = 0.0082197985891394618109204455065619487897_wp
        Alpha(3) = 0.0235807105099614116229154395926181919663_wp
        Alpha(4) = 0.0553321513921753326662486376086302897193_wp
        Alpha(5) = 0.1195250489508127474734913188447738008335_wp
        Alpha(6) = 0.2460990065813065139977474510279975561389_wp
        Alpha(7) = 0.4889378463987340393164578861151525757123_wp
        Alpha(8) = 0.9433780372993746744293176720308480298627_wp
        Alpha(9) = 1.7778182811830074268539050863502382071601_wp
        Alpha(10) = 3.3028236886633358783491726251568820771354_wp
        Alpha(11) = 6.2150744684586401466765137602266122485162_wp

        Xi(1) = 1.010497815491529190363995116542_wp
        Xi(2) = 1.096002240606849505622118201575_wp
        Xi(3) = 1.275199737243853945439613084556_wp
        Xi(4) = 1.565017259187337682140549188770_wp
        Xi(5) = 1.992906227574651268578259655229_wp
        Xi(6) = 2.600532897616855750478476183574_wp
        Xi(7) = 3.449617410857001624513440418696_wp
        Xi(8) = 4.630278431259522230623720817633_wp
        Xi(9) = 6.273261560193874468698699287117_wp
        Xi(10) = 8.568459619920009329337162728990_wp
        Xi(11) = 11.793138496154818108119233244224_wp
        Xi(12) = 16.355117405512885283419555193518_wp
        Xi(13) = 22.859290892700708325291358846698_wp
        Xi(14) = 32.210581239743349135640215763488_wp
        Xi(15) = 45.773156189807511291106623119163_wp
        Xi(16) = 65.612262749568161603053173536182_wp
        Xi(17) = 94.839737306547094809749953725486_wp
        Xi(18) = 138.010810624332375340372891514562_wp
        Xi(19) = 201.158564261307254705801739191884_wp
        Xi(20) = 289.635851036345949777173203187886_wp
        Xi(21) = 397.980376028435770652258440804871_wp
        Xi(22) = 486.207578566478576365117447721786_wp

      case (17)
        ! ----- K = 11 , R = 6E2 -----

        RIni = 6.0e2_wp

        Omega(1) = 0.0032583762595687854266888154727496718088_wp
        Omega(2) = 0.0089945425144352905224046871440757300320_wp
        Omega(3) = 0.0194534746653866053971984259596261779279_wp
        Omega(4) = 0.0406703100825870640662289007966068155042_wp
        Omega(5) = 0.0828822448445453864693428791543539091435_wp
        Omega(6) = 0.1638367468622252963076453211033189916179_wp
        Omega(7) = 0.3145704196522117309037583354758282894181_wp
        Omega(8) = 0.5892630713497090854053120367872509177687_wp
        Omega(9) = 1.0859847078241911882360432040073305870465_wp
        Omega(10) = 2.0128882604504029184278035558008923544548_wp
        Omega(11) = 4.0899797614649650006474201902051390788984_wp

        Alpha(1) = 0.0012460394897290092060417434141617043508_wp
        Alpha(2) = 0.0071487801396430483380248590598304847532_wp
        Alpha(3) = 0.0207774988093894703261279221714885601102_wp
        Alpha(4) = 0.0495983068314865137162330329528847627785_wp
        Alpha(5) = 0.1090359089997250805799270181872806517731_wp
        Alpha(6) = 0.2281340299783249546611892805136889705864_wp
        Alpha(7) = 0.4597232649902577561722843979552521886944_wp
        Alpha(8) = 0.8981462859609064504958636510068004099594_wp
        Alpha(9) = 1.7112485746366723854362465462308762198518_wp
        Alpha(10) = 3.2099066335151224178302842293675212204107_wp
        Alpha(11) = 6.0917531992763160613064965520635496432078_wp

        Xi(1) = 1.010805372459477708096779735580_wp
        Xi(2) = 1.098865619000078383549201532787_wp
        Xi(3) = 1.283689514573291367335178270181_wp
        Xi(4) = 1.583287498127780903219713470875_wp
        Xi(5) = 2.026956412345776807702041333314_wp
        Xi(6) = 2.659376188697842394994749759540_wp
        Xi(7) = 3.547095792989746083955296396262_wp
        Xi(8) = 4.787854381541439793053965834346_wp
        Xi(9) = 6.524384052416819790298774250914_wp
        Xi(10) = 8.965525477268821448635638393654_wp
        Xi(11) = 12.418707643050627490062531865789_wp
        Xi(12) = 17.340195095127145944249869025100_wp
        Xi(13) = 24.413492412775800864677133716896_wp
        Xi(14) = 34.672557359549868657988547226978_wp
        Xi(15) = 49.696120687816508886686106549746_wp
        Xi(16) = 71.911084122990895563243274324350_wp
        Xi(17) = 105.046282059082216490952799148317_wp
        Xi(18) = 154.713563805560001346295351254412_wp
        Xi(19) = 228.703813614806019172132067751591_wp
        Xi(20) = 334.878404555376608997541154622013_wp
        Xi(21) = 469.014081344866218648004263513940_wp
        Xi(22) = 581.997135078390529783209927927601_wp

      case (18)
        ! ----- K = 11 , R = 7E2 -----

        RIni = 7.0e2_wp

        Omega(1) = 0.0028831912480045087212277655835598877410_wp
        Omega(2) = 0.0080547896960305267873784628579469924148_wp
        Omega(3) = 0.0177641918678628270847818723665589857319_wp
        Omega(4) = 0.0378426247694864904115932564138136129372_wp
        Omega(5) = 0.0782870521182277218151350506314400945485_wp
        Omega(6) = 0.1566599587223711810546594030979505873802_wp
        Omega(7) = 0.3039273296837429993424908175603249560481_wp
        Omega(8) = 0.5744369494539767600225013688408637335669_wp
        Omega(9) = 1.0668055886728782982774202392484141910245_wp
        Omega(10) = 1.9900009359991648603179642540439431286359_wp
        Omega(11) = 4.0639355181090332552913679808881397548248_wp

        Alpha(1) = 0.0011011368472607193091320547883270319911_wp
        Alpha(2) = 0.0063561904304476650575254676085734661939_wp
        Alpha(3) = 0.0186927643171768198647865553280178474438_wp
        Alpha(4) = 0.0452979299110936900636930485530839973762_wp
        Alpha(5) = 0.1010854056292192224152820013982623237325_wp
        Alpha(6) = 0.2143707645737023147116534493861195187492_wp
        Alpha(7) = 0.4371216892824491518049039029403246559013_wp
        Alpha(8) = 0.8628534639195982823404486306007044049693_wp
        Alpha(9) = 1.6589340653420582790723847299574345015571_wp
        Alpha(10) = 3.1364760958078274867208423692943597416161_wp
        Alpha(11) = 5.9939098333794121674193700055610634080949_wp

        Xi(1) = 1.011059676264312853452044527192_wp
        Xi(2) = 1.101235574587819892418069644968_wp
        Xi(3) = 1.290729288973113345214381586512_wp
        Xi(4) = 1.598475864989992153552406106698_wp
        Xi(5) = 2.055354171858966342038496644129_wp
        Xi(6) = 2.708643847270304997037343452959_wp
        Xi(7) = 3.629086150599954327302526513321_wp
        Xi(8) = 4.921080606943753329909213567461_wp
        Xi(9) = 6.737918074576284085779043220299_wp
        Xi(10) = 9.305267659806523622423313213403_wp
        Xi(11) = 12.957587106833398172926752334178_wp
        Xi(12) = 18.194973643455962528450720050444_wp
        Xi(13) = 25.772788989918090972117137305020_wp
        Xi(14) = 36.844257238445314590891133121886_wp
        Xi(15) = 53.188858434996097189484398626291_wp
        Xi(16) = 77.576545348772702706541259942696_wp
        Xi(17) = 114.330947293240342728382064052539_wp
        Xi(18) = 170.104003977413448703281950713517_wp
        Xi(19) = 254.471866625116703020115238587096_wp
        Xi(20) = 378.002620022668104127205879194662_wp
        Xi(21) = 538.308905487260507150004684717715_wp
        Xi(22) = 677.441769430224394354933536988028_wp

      case (19)
        ! ----- K = 11 , R = 8E2 -----

        RIni = 8.0e2_wp

        Omega(1) = 0.0025936679815209045092815352611878565448_wp
        Omega(2) = 0.0073265355590078712569803592348505194565_wp
        Omega(3) = 0.0164427238548874570983523426756511831570_wp
        Omega(4) = 0.0356008498361247634063874150517348127210_wp
        Omega(5) = 0.0745996200705102757499498702942197780885_wp
        Omega(6) = 0.1508449808442598329518476857336040097834_wp
        Omega(7) = 0.2952357646445974116104077333844202257751_wp
        Omega(8) = 0.5622513167694076248589660216303798279114_wp
        Omega(9) = 1.0509646969145498192083759692039279798337_wp
        Omega(10) = 1.9710423046491469765148854542680112444941_wp
        Omega(11) = 4.0423577589097630046420517313521258984110_wp

        Alpha(1) = 0.0009893735940206601844169881235582786339_wp
        Alpha(2) = 0.0057435672506857186733056418008469545811_wp
        Alpha(3) = 0.0170741427403647941375252738288592624372_wp
        Alpha(4) = 0.0419335759461701156943912605773450152924_wp
        Alpha(5) = 0.0948076122813876336905495462170634368704_wp
        Alpha(6) = 0.2034033714513136402901767596507198732070_wp
        Alpha(7) = 0.4189621037780286376243660956264491801448_wp
        Alpha(8) = 0.8342932954015259300350505766719066969017_wp
        Alpha(9) = 1.6163458927661897599837095840236145249946_wp
        Alpha(10) = 3.0764177556541122311468267058742753761180_wp
        Alpha(11) = 5.9136220183418721921611571623600411840016_wp

        Xi(1) = 1.011275517891623727307660873809_wp
        Xi(2) = 1.103248794090624789756248380623_wp
        Xi(3) = 1.296718497999818083574012639314_wp
        Xi(4) = 1.611425340338749509342709809179_wp
        Xi(5) = 2.079631241991189730513103772758_wp
        Xi(6) = 2.750900718173535138087043572241_wp
        Xi(7) = 3.699679890254345520413797876458_wp
        Xi(8) = 5.036287746054576338738256735539_wp
        Xi(9) = 6.923459855102234069331584853657_wp
        Xi(10) = 9.602020876295634602658191081304_wp
        Xi(11) = 13.430962838312866545163737352908_wp
        Xi(12) = 18.950483820982974941110610345163_wp
        Xi(13) = 26.982267075158729401074264586669_wp
        Xi(14) = 38.790660332355037492879645810717_wp
        Xi(15) = 56.344113096241490102472626588792_wp
        Xi(16) = 82.739291247479588242708103962286_wp
        Xi(17) = 122.874004490899161136374662106618_wp
        Xi(18) = 184.421970762005453678411548423810_wp
        Xi(19) = 278.758005143637350276009101435193_wp
        Xi(20) = 419.309994116410967468500103905171_wp
        Xi(21) = 606.051488249862153390168373334745_wp
        Xi(22) = 772.565802767476019097259154477797_wp

      case (20)
        ! ----- K = 11 , R = 9E2 -----

        RIni = 9.0e2_wp

        Omega(1) = 0.0023628382210883959192670945772402069096_wp
        Omega(2) = 0.0067436900257026460985495023257862978738_wp
        Omega(3) = 0.0153759124610560547947820026794263359449_wp
        Omega(4) = 0.0337695236021534751132415659580221500846_wp
        Omega(5) = 0.0715558676321224931195122236082806210788_wp
        Omega(6) = 0.1460054304956700081414427971893132962578_wp
        Omega(7) = 0.2879536312425977612856290571707873482410_wp
        Omega(8) = 0.5519856851543989045881868915621737414767_wp
        Omega(9) = 1.0375637395027244186960016514298388301540_wp
        Omega(10) = 1.9549638627492123649177432898049744380842_wp
        Omega(11) = 4.0240548291653549630234043554821710131364_wp

        Alpha(1) = 0.0009003070544349668756826245250447088297_wp
        Alpha(2) = 0.0052544185410532049085930220416046365628_wp
        Alpha(3) = 0.0157763948086487224303656271701123969819_wp
        Alpha(4) = 0.0392174231619657099484101137054636865287_wp
        Alpha(5) = 0.0896975387361955071486457370666389721237_wp
        Alpha(6) = 0.1944045560423672174753207883557415414089_wp
        Alpha(7) = 0.4039555012406253642776726708873979987402_wp
        Alpha(8) = 0.8105465859229441190673091932605842657722_wp
        Alpha(9) = 1.5807541895705499766051557730683896352275_wp
        Alpha(10) = 3.0260253637311591221048523436820687493309_wp
        Alpha(11) = 5.8460680636665212756529486082257562884479_wp

        Xi(1) = 1.011462352317341486618133827857_wp
        Xi(2) = 1.104992721003111662845044826931_wp
        Xi(3) = 1.301913484870989259748787214566_wp
        Xi(4) = 1.622677887976778621421415638171_wp
        Xi(5) = 2.100775626259635315727300475253_wp
        Xi(6) = 2.787808747240925408067066526563_wp
        Xi(7) = 3.761541879546799615032307007034_wp
        Xi(8) = 5.137621988706277868005217746017_wp
        Xi(9) = 7.087332602485865488961547242752_wp
        Xi(10) = 9.865302075733673028978820696011_wp
        Xi(11) = 13.853006800327013128305841949128_wp
        Xi(12) = 19.627657504269309289915423910600_wp
        Xi(13) = 28.072607486447898102882581028439_wp
        Xi(14) = 40.556398044558649679941497723235_wp
        Xi(15) = 59.226255018542725845620555702453_wp
        Xi(16) = 87.491043814091919862685475806074_wp
        Xi(17) = 130.803769290163998964549030290527_wp
        Xi(18) = 197.840738696161377085935839659214_wp
        Xi(19) = 301.779756698642040280144982489219_wp
        Xi(20) = 459.029228277024053728183616840397_wp
        Xi(21) = 672.388667147153316072660800273297_wp
        Xi(22) = 867.388489463458042361132527275913_wp

      case (21)
        ! ----- K = 11 , R = 1E3 -----

        RIni = 1.0e3_wp

        Omega(1) = 0.0021740872264102924137251822292796199498_wp
        Omega(2) = 0.0062654140734613558082574018628929657382_wp
        Omega(3) = 0.0144934344125449496880266454096330974721_wp
        Omega(4) = 0.0322384802162817058210429310011191361696_wp
        Omega(5) = 0.0689880349533423857129039828695038494288_wp
        Omega(6) = 0.1418933622404710467179581381835795639290_wp
        Omega(7) = 0.2817301334836233389050282049925755245567_wp
        Omega(8) = 0.5431705675223087211446297561945328880029_wp
        Omega(9) = 1.0260142895533087236108887863039740295790_wp
        Omega(10) = 1.9410767529159068538234869660819015280140_wp
        Omega(11) = 4.0082438733806685813784720373575964913471_wp

        Alpha(1) = 0.0008275060405979828267253562920375414835_wp
        Alpha(2) = 0.0048538992490044590716160904843995416158_wp
        Alpha(3) = 0.0147096837575472349300117818823863125033_wp
        Alpha(4) = 0.0369705659049513704169829218793008607236_wp
        Alpha(5) = 0.0854390615716348159294660051521486110460_wp
        Alpha(6) = 0.1868523116342620025531577609978484133535_wp
        Alpha(7) = 0.3912822366416272664075594323040974131800_wp
        Alpha(8) = 0.7903843504629096339752711286585906691471_wp
        Alpha(9) = 1.5504001921788530224703347992765145590965_wp
        Alpha(10) = 2.9828994649313298984528303181562591817055_wp
        Alpha(11) = 5.7881154950766118133520521382706647273153_wp

        Xi(1) = 1.011626577944182569612600886000_wp
        Xi(2) = 1.106526584008231192429901057483_wp
        Xi(3) = 1.306488004306895040059974255620_wp
        Xi(4) = 1.632602251589605259554649485665_wp
        Xi(5) = 2.119461908127519490958878822795_wp
        Xi(6) = 2.820506515514542811209203199141_wp
        Xi(7) = 3.816505442352685583328950058579_wp
        Xi(8) = 5.227950254525374241664631425053_wp
        Xi(9) = 7.233934065422368251980361764097_wp
        Xi(10) = 10.101762872371427856230119424197_wp
        Xi(11) = 14.233684181896668193186006323714_wp
        Xi(12) = 20.241312228725278458962844041480_wp
        Xi(13) = 29.065709461086927115069933336144_wp
        Xi(14) = 42.173589286440474402373190798698_wp
        Xi(15) = 61.882031162953867478582203887072_wp
        Xi(16) = 91.899086735024803594684694729722_wp
        Xi(17) = 138.215380021578030739215492417316_wp
        Xi(18) = 210.490640457978115673842722799236_wp
        Xi(19) = 323.703942267917895547757467511474_wp
        Xi(20) = 497.339829521204033480064055083858_wp
        Xi(21) = 737.439254546925192723705322350725_wp
        Xi(22) = 961.925573460294248806512484861742_wp

      case (22)
        ! ----- K = 11 , R = 2E3 -----

        RIni = 2.0e3_wp

        Omega(1) = 0.0012623682047825584772750883767213070374_wp
        Omega(2) = 0.0039256526753984839104248943247865710759_wp
        Omega(3) = 0.0100465837533007697967448834490411524456_wp
        Omega(4) = 0.0242425014592881167561005473568003232288_wp
        Omega(5) = 0.0551805151273874833863831729924287472500_wp
        Omega(6) = 0.1192702380940837676869437262994200210642_wp
        Omega(7) = 0.2468429160149552121882162922239523084045_wp
        Omega(8) = 0.4929860865537508678060722722147346530619_wp
        Omega(9) = 0.9594751280677348143619866116527106214562_wp
        Omega(10) = 1.8604943398396530794981079681704727590841_wp
        Omega(11) = 3.9164423268801717532231615459537010792701_wp

        Alpha(1) = 0.0004763386763845254530552357019948517269_wp
        Alpha(2) = 0.0029099646551678040284737028833839111996_wp
        Alpha(3) = 0.0094574935145634442861732431306887036726_wp
        Alpha(4) = 0.0256521647921702371382255856643861413602_wp
        Alpha(5) = 0.0634401600811532823473307476924709646937_wp
        Alpha(6) = 0.1469147278490494683213414550682074377619_wp
        Alpha(7) = 0.3228783614455336845267072881648928728282_wp
        Alpha(8) = 0.6796431373233775072176601761420045022533_wp
        Alpha(9) = 1.3812552178932358922435960635333174195694_wp
        Alpha(10) = 2.7398655433520379364181057679772379742644_wp
        Alpha(11) = 5.4589664557360760804675947621689147126745_wp

        Xi(1) = 1.012634485295069944484590285239_wp
        Xi(2) = 1.115960417122124936450781873543_wp
        Xi(3) = 1.334732406200042509596820350382_wp
        Xi(4) = 1.694205723556559923170458659136_wp
        Xi(5) = 2.236241696464598578296054109948_wp
        Xi(6) = 3.026551793939026981138454286402_wp
        Xi(7) = 4.166247455693754736527623094844_wp
        Xi(8) = 5.809102145284259851795716222966_wp
        Xi(9) = 8.188781606285571878015017155406_wp
        Xi(10) = 11.662817491591790737895384388878_wp
        Xi(11) = 16.784270306190127018675095627032_wp
        Xi(12) = 24.420053833854110511211721323832_wp
        Xi(13) = 35.949725216187675389672984138656_wp
        Xi(14) = 53.606023489636841609234929606487_wp
        Xi(15) = 81.069835718520333420200607577044_wp
        Xi(16) = 124.532655436403525327226926577850_wp
        Xi(17) = 194.619603518130789329698693279624_wp
        Xi(18) = 309.879071731328180172582165141648_wp
        Xi(19) = 502.674500056099144235099984712178_wp
        Xi(20) = 825.737819569139915920619188227647_wp
        Xi(21) = 1333.340188188764327881230542516278_wp
        Xi(22) = 1893.776396022996856172149193753285_wp

      case (23)
        ! ----- K = 11 , R = 3E3 -----

        RIni = 3.0e3_wp

        Omega(1) = 0.0009228838994159140362278747997393235636_wp
        Omega(2) = 0.0030346810581029129714051889750436874493_wp
        Omega(3) = 0.0082645523392666875460888138704707017723_wp
        Omega(4) = 0.0208629228147157623313135647175053932756_wp
        Omega(5) = 0.0491040452568306093184924247807554564815_wp
        Omega(6) = 0.1089988086210330150874030230578881628389_wp
        Omega(7) = 0.2305969093077353990705029990193608568916_wp
        Omega(8) = 0.4691265413835654474552540327270122588743_wp
        Omega(9) = 0.9273310285024501460247013662208104278761_wp
        Omega(10) = 1.8211892895434077157407193325866501254495_wp
        Omega(11) = 3.8716236620173660497919859713178425408842_wp

        Alpha(1) = 0.0003458600493587631532409070893091182497_wp
        Alpha(2) = 0.0021800684483446950346975147256439742094_wp
        Alpha(3) = 0.0074338002448057351524994453700889707193_wp
        Alpha(4) = 0.0211226719585344040053666130052345550894_wp
        Alpha(5) = 0.0542911478650846585830623634155722712080_wp
        Alpha(6) = 0.1297318847429075853197556408846757847186_wp
        Alpha(7) = 0.2925888019277061919461851990931222644576_wp
        Alpha(8) = 0.6294124318896298635617886363391448867333_wp
        Alpha(9) = 1.3030115311584663801824229834025459240365_wp
        Alpha(10) = 2.6257268942687263856092944047659898387792_wp
        Alpha(11) = 5.3027692366205613138871455269907073670765_wp

        Xi(1) = 1.013162307629898129250796412482_wp
        Xi(2) = 1.120914469518859819590836379710_wp
        Xi(3) = 1.349640187754250304752977329681_wp
        Xi(4) = 1.726948755314575472503621500486_wp
        Xi(5) = 2.298863022302009189454588544699_wp
        Xi(6) = 3.138244439814425477573364697825_wp
        Xi(7) = 4.358256474678705915358489786371_wp
        Xi(8) = 6.132764794237344253401839067052_wp
        Xi(9) = 8.729089111644693055672150627800_wp
        Xi(10) = 12.561715766963198824546221743503_wp
        Xi(11) = 18.281335377765054883356099502123_wp
        Xi(12) = 26.924710149403320686226082081305_wp
        Xi(13) = 40.172007476294265639060743211530_wp
        Xi(14) = 60.798697236948366733749749002413_wp
        Xi(15) = 93.487711881290384327569675804170_wp
        Xi(16) = 146.331520562003058835709445872908_wp
        Xi(17) = 233.672616773833143694050207272994_wp
        Xi(18) = 381.600754617379850003322516727167_wp
        Xi(19) = 638.368852255887035962444997494458_wp
        Xi(20) = 1090.853238414367184838660307377722_wp
        Xi(21) = 1858.189718002260019669158452870761_wp
        Xi(22) = 2804.971625837513768741970920927997_wp

      case (24)
        ! ----- K = 11 , R = 4E3 -----

        RIni = 4.0e3_wp

        Omega(1) = 0.0007412355914717758371035817772411180560_wp
        Omega(2) = 0.0025500918289050421850566258358608551227_wp
        Omega(3) = 0.0072606523430177817511419347939333190922_wp
        Omega(4) = 0.0188983051896847964339808600628400903076_wp
        Omega(5) = 0.0454905661909004588254576591233035998130_wp
        Omega(6) = 0.1027828884911571632042206862756028584727_wp
        Omega(7) = 0.2206244947739454568354146996034259586850_wp
        Omega(8) = 0.4543089394242018984706229373093933077143_wp
        Omega(9) = 0.9071881653619918974224348162671560658055_wp
        Omega(10) = 1.7964239050734691755233790599888266115158_wp
        Omega(11) = 3.8433675617984309610605797580262787960237_wp

        Alpha(1) = 0.0002761320530126240699661397732427373097_wp
        Alpha(2) = 0.0017871801758091304107081822910227386636_wp
        Alpha(3) = 0.0063237062667975813191577881024196816639_wp
        Alpha(4) = 0.0185749572332312158613005712015642245660_wp
        Alpha(5) = 0.0490227902621349213037446525959017762375_wp
        Alpha(6) = 0.1196387864534846704318805585276486169732_wp
        Alpha(7) = 0.2745011177460962377883287033331072279907_wp
        Alpha(8) = 0.5990053817796525529085736871959255722686_wp
        Alpha(9) = 1.2551208194212885533918785951890129126696_wp
        Alpha(10) = 2.5552713219979284167606564670549573747849_wp
        Alpha(11) = 5.2057929169179171491853297393248567459523_wp

        Xi(1) = 1.013507909133304014655078961749_wp
        Xi(2) = 1.124163449694987797081516256448_wp
        Xi(3) = 1.359445665438874723592727056243_wp
        Xi(4) = 1.748570385623043480015186001975_wp
        Xi(5) = 2.340424869223935786747420317866_wp
        Xi(6) = 3.212837585309808281167770482867_wp
        Xi(7) = 4.487424836087381988577321711986_wp
        Xi(8) = 6.352303794614191673195691123510_wp
        Xi(9) = 9.098958048284406790497114148053_wp
        Xi(10) = 13.183328635620850772906809122986_wp
        Xi(11) = 19.328223806317943620977639618275_wp
        Xi(12) = 28.697951916635848879089110141649_wp
        Xi(13) = 43.202468477442194458998381279002_wp
        Xi(14) = 66.040542432351887432162573787764_wp
        Xi(15) = 102.694192864624376272664196818596_wp
        Xi(16) = 162.810990503060741912966236100146_wp
        Xi(17) = 263.864668019880545618560674370201_wp
        Xi(18) = 438.521112893176594121902311940175_wp
        Xi(19) = 749.532175553113846067088132940626_wp
        Xi(20) = 1317.076815743827621907335867490474_wp
        Xi(21) = 2333.268953940222266929538363910979_wp
        Xi(22) = 3698.566202684434597314222514796711_wp

      case (25)
        ! ----- K = 11 , R = 5E3 -----

        RIni = 5.0e3_wp

        Omega(1) = 0.0006267332758852317722579485079117261881_wp
        Omega(2) = 0.0022404994313095141259883842947339682716_wp
        Omega(3) = 0.0066017708522960727291522728919781481949_wp
        Omega(4) = 0.0175803778127506669060281605659890047377_wp
        Omega(5) = 0.0430286752670226813875548489002342833487_wp
        Omega(6) = 0.0984970458414388567585122033487543546926_wp
        Omega(7) = 0.2136815596076955863706452187766871020358_wp
        Omega(8) = 0.4439105540875763014380551291226950638702_wp
        Omega(9) = 0.8929659232509779923656415001609332193766_wp
        Omega(10) = 1.7788722459640630017873100254810481146706_wp
        Omega(11) = 3.8233333460474906823848423309541999515204_wp

        Alpha(1) = 0.0002322146395909955911310504293710264834_wp
        Alpha(2) = 0.0015383078777110473123499812322934054976_wp
        Alpha(3) = 0.0056097155499255455726426665679501537554_wp
        Alpha(4) = 0.0169052401931734672522236176059329260113_wp
        Alpha(5) = 0.0455116261284354011137681834953117032683_wp
        Alpha(6) = 0.1128182709493343310874899007578697762710_wp
        Alpha(7) = 0.2621385842803852858769967509866560817500_wp
        Alpha(8) = 0.5780283644600492836741235935793525868576_wp
        Alpha(9) = 1.2218328763756418393763084062619839187391_wp
        Alpha(10) = 2.5060164008391585472104723386621571989963_wp
        Alpha(11) = 5.1377321548383988246248843534402794830385_wp

        Xi(1) = 1.013759180858394429625320143362_wp
        Xi(2) = 1.126528252383917901827299623552_wp
        Xi(3) = 1.366596914335719902358103061690_wp
        Xi(4) = 1.764382443729622758224820366912_wp
        Xi(5) = 2.370924666432016908023533430239_wp
        Xi(6) = 3.267810182254802050258804535510_wp
        Xi(7) = 4.583096387201138603457312159506_wp
        Xi(8) = 6.515835113413934717285086462013_wp
        Xi(9) = 9.376216422625220800468603954414_wp
        Xi(10) = 13.652573425612661578085194058474_wp
        Xi(11) = 20.124650489238526481286517544333_wp
        Xi(12) = 30.058601331363003265895184057399_wp
        Xi(13) = 45.550203837520550555206799714369_wp
        Xi(14) = 70.145402727739678563900760366323_wp
        Xi(15) = 109.992188118941259349081729368436_wp
        Xi(16) = 176.057872207031846437108590919252_wp
        Xi(17) = 288.530732333027965463845276872235_wp
        Xi(18) = 485.921466125530878754634755978259_wp
        Xi(19) = 844.296017988520174801791995378153_wp
        Xi(20) = 1515.858811846124858280226987972128_wp
        Xi(21) = 2769.870549675472343631454918977397_wp
        Xi(22) = 4576.245877897764160469051830659737_wp

      case (26)
        ! ----- K = 11 , R = 6E3 -----

        RIni = 6.0e3_wp

        Omega(1) = 0.0005473782929676091589050948712450589406_wp
        Omega(2) = 0.0020234317344929258391381305705325477362_wp
        Omega(3) = 0.0061296388124654699446628358314170803567_wp
        Omega(4) = 0.0166202678647742327646509097674454302762_wp
        Omega(5) = 0.0412142813057742613662979332816993505162_wp
        Omega(6) = 0.0953100824256778086240740957557626700236_wp
        Omega(7) = 0.2084812698209122131751949366651466277744_wp
        Omega(8) = 0.4360759297357788821509864962688851619532_wp
        Omega(9) = 0.8822012112666169261873845564014828823929_wp
        Omega(10) = 1.7655503222027704023511934794399280690413_wp
        Omega(11) = 3.8081221072884642157331536171938068946474_wp

        Alpha(1) = 0.0002017946813401587113227188651355496762_wp
        Alpha(2) = 0.0013650957954943520156917383286121481234_wp
        Alpha(3) = 0.0051063430539639230514999362724182230266_wp
        Alpha(4) = 0.0157103155764604293100853565912089493395_wp
        Alpha(5) = 0.0429662099418053528877836644284116118797_wp
        Alpha(6) = 0.1078215546982082305435611362109948885291_wp
        Alpha(7) = 0.2530041706297298811326624418116537640344_wp
        Alpha(8) = 0.5624205865907779740850772065563489832130_wp
        Alpha(9) = 1.1969260235286672264861523862222725256288_wp
        Alpha(10) = 2.4690046401427888621022549431316406298720_wp
        Alpha(11) = 5.0864407388447149165403016191078222618671_wp

        Xi(1) = 1.013953504011853172452235771761_wp
        Xi(2) = 1.128358627594889074187596567089_wp
        Xi(3) = 1.372140064760185083512708525966_wp
        Xi(4) = 1.776663964374088323537256850493_wp
        Xi(5) = 2.394676147487823382551536122165_wp
        Xi(6) = 3.310757150500711268456197067245_wp
        Xi(7) = 4.658118960797535882626535519435_wp
        Xi(8) = 6.644618065181252250690768779151_wp
        Xi(9) = 9.595598690279814975065408866328_wp
        Xi(10) = 14.025829661216053410692505831747_wp
        Xi(11) = 20.761870967492430821443871735710_wp
        Xi(12) = 31.154365044786872313703485204428_wp
        Xi(13) = 47.454701976670969162941382535337_wp
        Xi(14) = 73.502772137602551523472005357007_wp
        Xi(15) = 116.017300440218688671600855144561_wp
        Xi(16) = 187.112745425990175437980234107727_wp
        Xi(17) = 309.376229073982038336732003358520_wp
        Xi(18) = 526.584806978543925104485623478467_wp
        Xi(19) = 927.109321573275227124089070684931_wp
        Xi(20) = 1693.810124284694755392166598539916_wp
        Xi(21) = 3175.115660441574632955052948091179_wp
        Xi(22) = 5439.118719873584542945366138155805_wp

      case (27)
        ! ----- K = 11 , R = 7E3 -----

        RIni = 7.0e3_wp

        Omega(1) = 0.0004888589654993823415061000240821459384_wp
        Omega(2) = 0.0018616953743137684792181802762370268489_wp
        Omega(3) = 0.0057714059692291130183748788535363916097_wp
        Omega(4) = 0.0158821609122009120908280196265982553427_wp
        Omega(5) = 0.0398065979662624536909287104036181403899_wp
        Omega(6) = 0.0928200129151813189718280873441447198502_wp
        Omega(7) = 0.2043948992766658286535270960815324770010_wp
        Omega(8) = 0.4298908173319880503910825342295254358760_wp
        Omega(9) = 0.8736723641609264178224322772159382566315_wp
        Omega(10) = 1.7549721269953933305840720890955708455294_wp
        Omega(11) = 3.7960404785939114290661128281456626609724_wp

        Alpha(1) = 0.0001793701948587827513269946831881221372_wp
        Alpha(2) = 0.0012368771763232082847653708337562483699_wp
        Alpha(3) = 0.0047295378501987948447370308565490537589_wp
        Alpha(4) = 0.0148046949969108685156272769110896714295_wp
        Alpha(5) = 0.0410168641143887723388475850805345856998_wp
        Alpha(6) = 0.1039627887046145387749140377720014072338_wp
        Alpha(7) = 0.2459022093684296517352490946128895643596_wp
        Alpha(8) = 0.5502188036604130099869666703504833549232_wp
        Alpha(9) = 1.1773684055533485784709665789016241888021_wp
        Alpha(10) = 2.4398440817448355183785779765059942292282_wp
        Alpha(11) = 5.0459378421517919626783366116029583281488_wp

        Xi(1) = 1.014110063993542450080237449228_wp
        Xi(2) = 1.129834266023652116221927255157_wp
        Xi(3) = 1.376614175376544919296613322945_wp
        Xi(4) = 1.786592754684194123463401859819_wp
        Xi(5) = 2.413917056239944505672059826829_wp
        Xi(6) = 3.345636074011955358098110990817_wp
        Xi(7) = 4.719228876055178750932583842115_wp
        Xi(8) = 6.749870651720982652962965708809_wp
        Xi(9) = 9.775572634850614675387781016269_wp
        Xi(10) = 14.333314418138938924063285895727_wp
        Xi(11) = 21.289252861263578554817943455646_wp
        Xi(12) = 32.065959035318667681713211692340_wp
        Xi(13) = 49.048341237057613624383822781283_wp
        Xi(14) = 76.330652603958345600587698243089_wp
        Xi(15) = 121.130517723825521826663820235126_wp
        Xi(16) = 196.576460619874407456664222593190_wp
        Xi(17) = 327.405204366704011637034810178193_wp
        Xi(18) = 562.187777745814405094826327058399_wp
        Xi(19) = 1000.730898143708903658133380076833_wp
        Xi(20) = 1855.202325197516517896190180181293_wp
        Xi(21) = 3553.944782316665923271870042299270_wp
        Xi(22) = 6287.992938677223824228690318705048_wp

      case (28)
        ! ----- K = 11 , R = 8E3 -----

        RIni = 8.0e3_wp

        Omega(1) = 0.0004437681784262299374822296159619899880_wp
        Omega(2) = 0.0017359012756834154712376973761674137098_wp
        Omega(3) = 0.0054884264315541892356410051245080294890_wp
        Omega(4) = 0.0152927906015183914024177080220878721661_wp
        Omega(5) = 0.0386741261492350979778198378361553366744_wp
        Omega(6) = 0.0908051858607855987034324606044233263447_wp
        Omega(7) = 0.2010729870218403514886074470724430796054_wp
        Omega(8) = 0.4248436695705187065837202947671258357332_wp
        Omega(9) = 0.8666922993395973932656203841951025879098_wp
        Omega(10) = 1.7462992361734780941443997970274892850284_wp
        Omega(11) = 3.7861326498591262976374516302513484333758_wp

        Alpha(1) = 0.0001620957580603920404350047751169776322_wp
        Alpha(2) = 0.0011377383639357606060877822502092060475_wp
        Alpha(3) = 0.0044352997606686039615651190055195529638_wp
        Alpha(4) = 0.0140900437920721600546582500876335863893_wp
        Alpha(5) = 0.0394651913802239503377433139930108296767_wp
        Alpha(6) = 0.1008699769730821261219796611263710417461_wp
        Alpha(7) = 0.2401783505512109114215374416900772303052_wp
        Alpha(8) = 0.5403404704563012579177951055786621736843_wp
        Alpha(9) = 1.1614778624872791111619779735875113146903_wp
        Alpha(10) = 2.4160862126637673468528694220580632645579_wp
        Alpha(11) = 5.0128780625618468073760591874332703810069_wp

        Xi(1) = 1.014239942221573816377028220970_wp
        Xi(2) = 1.131059076026157462959570443317_wp
        Xi(3) = 1.380331395988050338543944028924_wp
        Xi(4) = 1.794852498112521126615546973593_wp
        Xi(5) = 2.429950200208132349722484066845_wp
        Xi(6) = 3.374759709016794407112141418459_wp
        Xi(7) = 4.770378813680489580011795380798_wp
        Xi(8) = 6.838212765075277659379549000107_wp
        Xi(9) = 9.927096669501406712987634506362_wp
        Xi(10) = 14.593081887978388286493391934329_wp
        Xi(11) = 21.736488130016903172828102164260_wp
        Xi(12) = 32.842314094798293326593618957077_wp
        Xi(13) = 50.412076426022924838243888245870_wp
        Xi(14) = 78.763800701877293090891196669645_wp
        Xi(15) = 125.557583029224057967476735342416_wp
        Xi(16) = 204.829876278781896445368992942804_wp
        Xi(17) = 343.264066898434656999183900438766_wp
        Xi(18) = 593.830362490899102856367619551747_wp
        Xi(19) = 1067.014697285996970399502004056558_wp
        Xi(20) = 2003.005390344890530140986584228813_wp
        Xi(21) = 3910.001707684114783170059581607347_wp
        Xi(22) = 7123.500668713473708226757707961951_wp

      case (29)
        ! ----- K = 11 , R = 9E3 -----

        RIni = 9.0e3_wp

        Omega(1) = 0.0004078691872318507710981469102911461944_wp
        Omega(2) = 0.0016348875607537805668889520794434533357_wp
        Omega(3) = 0.0052581081186506578951953894737053607500_wp
        Omega(4) = 0.0148087257998724855491222808859252957348_wp
        Omega(5) = 0.0377381223895507049116086317058771371080_wp
        Omega(6) = 0.0891318209355365257209619783695941919177_wp
        Omega(7) = 0.1983032656250016002305114684589604223675_wp
        Omega(8) = 0.4206221039756080757492703176003701059926_wp
        Omega(9) = 0.8608396444681421005233439014414642542761_wp
        Omega(10) = 1.7390161743430621089662874156900329580822_wp
        Omega(11) = 3.7778109552097616187157913536509568075417_wp

        Alpha(1) = 0.0001483448807384251019126060000628136715_wp
        Alpha(2) = 0.0010585558771085235814493063512785331337_wp
        Alpha(3) = 0.0041982030737258507678037679385074643079_wp
        Alpha(4) = 0.0135089149642617613732045560304140252583_wp
        Alpha(5) = 0.0381940850164093613008117842483324366754_wp
        Alpha(6) = 0.0983215967127150371909316377283527543796_wp
        Alpha(7) = 0.2354400512436616309838548211086362016431_wp
        Alpha(8) = 0.5321321943865780585629192411811771989960_wp
        Alpha(9) = 1.1482339764255602875724429634551881918014_wp
        Alpha(10) = 2.3962399342344530084672471481610500632087_wp
        Alpha(11) = 4.9852187457262847206443390213337352179224_wp

        Xi(1) = 1.014350075600351303397629598724_wp
        Xi(2) = 1.132098149109364029003994833733_wp
        Xi(3) = 1.383487455589817327443469263670_wp
        Xi(4) = 1.801873039568553124738052584330_wp
        Xi(5) = 2.443597017918493595782167271402_wp
        Xi(6) = 3.399591860500832806975604616362_wp
        Xi(7) = 4.814079903885744832637422208066_wp
        Xi(8) = 6.913864059193249945838022352262_wp
        Xi(9) = 10.057187199621232224708289315629_wp
        Xi(10) = 14.816750600359321647300769875244_wp
        Xi(11) = 22.122812980254363109111626073400_wp
        Xi(12) = 33.515353358244164720669555634913_wp
        Xi(13) = 51.599133570115167723296734081373_wp
        Xi(14) = 80.891508404224807551874398825476_wp
        Xi(15) = 129.449498973704297075104641123744_wp
        Xi(16) = 212.130668776373502368293166853164_wp
        Xi(17) = 357.396008057185506134878494322038_wp
        Xi(18) = 622.279099349365191984340128783515_wp
        Xi(19) = 1127.278119364376255506776658421586_wp
        Xi(20) = 2139.390923877392672647346216763253_wp
        Xi(21) = 4246.099010265820874021613917648210_wp
        Xi(22) = 7946.161467249157825065708493639249_wp

      case (30)
        ! ----- K = 11 , R = 1E4 -----

        RIni = 1.0e4_wp

        Omega(1) = 0.0003785550695185704905331018800670669910_wp
        Omega(2) = 0.0015517451432116935316520689837685442747_wp
        Omega(3) = 0.0050662753610968026142610148073591735951_wp
        Omega(4) = 0.0144023942219292172832968896602212627300_wp
        Omega(5) = 0.0369481737121121147226652983025507737125_wp
        Omega(6) = 0.0877136968671946349224198893890225292580_wp
        Omega(7) = 0.1959481538043601072551129840815242744156_wp
        Omega(8) = 0.4170227140801808644366994310859197980790_wp
        Omega(9) = 0.8558390892074689600002926348487619634398_wp
        Omega(10) = 1.7327854324885193371000649542601479424775_wp
        Omega(11) = 3.7706904785971858274761653317952436736959_wp

        Alpha(1) = 0.0001371172950178501341687661597154926363_wp
        Alpha(2) = 0.0009937039401902874349137946216603545224_wp
        Alpha(3) = 0.0040024563087556733581913622193465540633_wp
        Alpha(4) = 0.0130252872174303745869984312404088466764_wp
        Alpha(5) = 0.0371294655008602045503181190294617053382_wp
        Alpha(6) = 0.0961764698255383165804120737529814277877_wp
        Alpha(7) = 0.2314355666053775903228621052543090286235_wp
        Alpha(8) = 0.5251727289213194890793966196351760800098_wp
        Alpha(9) = 1.1369760622900190238709558498975127349695_wp
        Alpha(10) = 2.3793366605690113944199248763311516086105_wp
        Alpha(11) = 4.9616300421987320260451548303848312571063_wp

        Xi(1) = 1.014445078090098826813052201867_wp
        Xi(2) = 1.132994811083856317285821491136_wp
        Xi(3) = 1.386212832419826497727145808714_wp
        Xi(4) = 1.807941230908676616189953434422_wp
        Xi(5) = 2.455406801772977036730019606559_wp
        Xi(6) = 3.421113194397476642890504328065_wp
        Xi(7) = 4.852020621598441331347867366830_wp
        Xi(8) = 6.979672539648318155876965951379_wp
        Xi(9) = 10.170604636139685588228087631535_wp
        Xi(10) = 15.012230965278301672398419963628_wp
        Xi(11) = 22.461386446764169332393490385869_wp
        Xi(12) = 34.107029147621249549893196473249_wp
        Xi(13) = 52.646324094495085033768022597656_wp
        Xi(14) = 82.776011367184239880812057776893_wp
        Xi(15) = 132.912412796370639544996095082752_wp
        Xi(16) = 218.661691303756028137428302215994_wp
        Xi(17) = 370.119212246490729811831954521040_wp
        Xi(18) = 648.091812080745762469913273662314_wp
        Xi(19) = 1182.497918253986295522572902427783_wp
        Xi(20) = 2266.010502890212877868947316528647_wp
        Xi(21) = 4564.472467287974115635051930439658_wp
        Xi(22) = 8756.418598288730433587545576301636_wp

      case (31)
        ! ----- K = 11 , R = 2E5 (Best : R = 109000) -----

        RIni = 2.0e5_wp

        Omega(1) = 0.0001134976878119174444673697307829282610_wp
        Omega(2) = 0.0007457000167406750592463748688546593124_wp
        Omega(3) = 0.0030526245415056300336642704359654820756_wp
        Omega(4) = 0.0099235004487624246363231851619790546692_wp
        Omega(5) = 0.0279373537885593099774266066109840167542_wp
        Omega(6) = 0.0711041222257731397104010308274357754499_wp
        Omega(7) = 0.1677705024084778823205898676262926016989_wp
        Omega(8) = 0.3732056083739767542541750783291476523118_wp
        Omega(9) = 0.7941436626469130445247178451406711019445_wp
        Omega(10) = 1.6552701124330565916236215895374073170387_wp
        Omega(11) = 3.6820059233978352326507876846051203756360_wp

        Alpha(1) = 0.0000355034846592468107132831054925398845_wp
        Alpha(2) = 0.0003905739575885531102960276960463920082_wp
        Alpha(3) = 0.0020671330057150484115067632599066493881_wp
        Alpha(4) = 0.0079783846601041099849384525058426653743_wp
        Alpha(5) = 0.0255498864500882378069962930916547705351_wp
        Alpha(6) = 0.0720851227548212716324249615951202940778_wp
        Alpha(7) = 0.1853055788144508502635196198204958761835_wp
        Alpha(8) = 0.4433472301960455170871319707792679309932_wp
        Alpha(9) = 1.0024332764988156885743383894826763480523_wp
        Alpha(10) = 2.1748065663060481430245152179026035810239_wp
        Alpha(11) = 4.6738220578828596602459499731452297055512_wp

        Xi(1) = 1.015690089382477667647071417623_wp
        Xi(2) = 1.144775026302406757206311938013_wp
        Xi(3) = 1.422180113927765727826688091362_wp
        Xi(4) = 1.888518573347652048733183449691_wp
        Xi(5) = 2.613470594874711714662718353708_wp
        Xi(6) = 3.711993684861224562093751089265_wp
        Xi(7) = 5.370779642078565153506114482340_wp
        Xi(8) = 7.891427121501472941120930437719_wp
        Xi(9) = 11.765526964896047277912838335823_wp
        Xi(10) = 17.807606050638013578391039004600_wp
        Xi(11) = 27.395214005522875326537057816267_wp
        Xi(12) = 42.915949154636157513581506961486_wp
        Xi(13) = 68.624694514972746979541984302386_wp
        Xi(14) = 112.360691745346326406163051103704_wp
        Xi(15) = 189.131571697762011746712929038949_wp
        Xi(16) = 329.035348174625998651787739390784_wp
        Xi(17) = 596.051036400452403984928650970687_wp
        Xi(18) = 1135.976006601578366161930944144842_wp
        Xi(19) = 2316.370944283527541607270450185752_wp
        Xi(20) = 5192.120834502648774044786250669858_wp
        Xi(21) = 13479.416793377781868201736870105378_wp
        Xi(22) = 47385.042527437597300377092324197292_wp

    end select

  case (12)
    ! ===== K = 12 =====

    select case (InitR)

      case default !(1:3)
        ! ----- K = 12 , R = 1E1 -----

        RIni = 1.0e1_wp

        Omega(1) = 0.0711846766353695929835615312242236285556_wp
        Omega(2) = 0.1690895322362734570071068213037435157275_wp
        Omega(3) = 0.2757926387649254820061406334730236267205_wp
        Omega(4) = 0.3977636094386997021865592927269972278737_wp
        Omega(5) = 0.5441057434880599478588085460284418104493_wp
        Omega(6) = 0.7279147631051287236844671291180475236615_wp
        Omega(7) = 0.9682939484624312369894644225709612328501_wp
        Omega(8) = 1.2936786640404163718212279809982589995343_wp
        Omega(9) = 1.7493131133110281919592482458014615076536_wp
        Omega(10) = 2.4187704026228852782157835443754834159336_wp
        Omega(11) = 3.5024366140575561975791657465251205394452_wp
        Omega(12) = 5.7855524079892674525239792959752094247960_wp

        Alpha(1) = 0.0276731292622081597226468612990557538467_wp
        Alpha(2) = 0.1472866131015313441898935409302140442378_wp
        Alpha(3) = 0.3687581752417785056644226188149460199384_wp
        Alpha(4) = 0.7039332404419826932288721532682274073522_wp
        Alpha(5) = 1.1723654412494890637288710855123952114809_wp
        Alpha(6) = 1.8045683154882165159836951429106477462483_wp
        Alpha(7) = 2.6469524304892961248745347768718261249887_wp
        Alpha(8) = 3.7693192226293995227785066370884692332766_wp
        Alpha(9) = 5.2773524398830704345772502072975385090103_wp
        Alpha(10) = 7.3381749735963551352184441878279130833107_wp
        Alpha(11) = 10.2488461667396625765641404726125074375886_wp
        Alpha(12) = 14.7099922873718894451508520937466073519317_wp

        Xi(1) = 1.002765923971119948229606400059_wp
        Xi(2) = 1.024986275526004813121727643477_wp
        Xi(3) = 1.069923061571954508281422080440_wp
        Xi(4) = 1.138574075278952991139604089543_wp
        Xi(5) = 1.232450417327966717367764004631_wp
        Xi(6) = 1.353597504629638669632583913849_wp
        Xi(7) = 1.504622215175725945452735177188_wp
        Xi(8) = 1.688724456972242326537568535283_wp
        Xi(9) = 1.909718822962894374246738127265_wp
        Xi(10) = 2.172034255460995098938742353667_wp
        Xi(11) = 2.480662653811127114304030394365_wp
        Xi(12) = 2.841026290314171571073711963784_wp
        Xi(13) = 3.258702872965208679426757365505_wp
        Xi(14) = 3.738936981267856123891876740650_wp
        Xi(15) = 4.285822352545366410898897147064_wp
        Xi(16) = 4.901021586284760217771400148123_wp
        Xi(17) = 5.581889846714328228590579206170_wp
        Xi(18) = 6.318929117702670764707834072382_wp
        Xi(19) = 7.092747329803447991613402079381_wp
        Xi(20) = 7.871138577292408604162143870298_wp
        Xi(21) = 8.607632829144091446416098278860_wp
        Xi(22) = 9.243536184503822437062703354371_wp
        Xi(23) = 9.715357748524472440189436728275_wp
        Xi(24) = 9.967640980357458236130185902368_wp

      case (4)
        ! ----- K = 12 , R = 2E1 -----

        RIni = 2.0e1_wp

        Omega(1) = 0.0428025346628120542797864885725678263384_wp
        Omega(2) = 0.1026156437369573046664321044896261980739_wp
        Omega(3) = 0.1703341135440457080398339237414795377390_wp
        Omega(4) = 0.2524250366464257813270237462699441266523_wp
        Omega(5) = 0.3585406407629632903914448927018554513779_wp
        Omega(6) = 0.5029449307365428544934135568489352863253_wp
        Omega(7) = 0.7061343335450083705143002610338243130172_wp
        Omega(8) = 0.9975940043005185541252592684724476157498_wp
        Omega(9) = 1.4229943025732463503154473505318833304045_wp
        Omega(10) = 2.0653230769823831772941113982255956216250_wp
        Omega(11) = 3.1208980779060891430111424060811486924649_wp
        Omega(12) = 5.3514751896835863285289969226710127259139_wp

        Alpha(1) = 0.0166219868333402929036517929953231664797_wp
        Alpha(2) = 0.0888764190139657434596505598822790972235_wp
        Alpha(3) = 0.2244621641849088582339188935321772078169_wp
        Alpha(4) = 0.4343008089118843558320594289146043820438_wp
        Alpha(5) = 0.7372692134906314825708306681661952097784_wp
        Alpha(6) = 1.1640767884289861903166288237265746374760_wp
        Alpha(7) = 1.7626592744339100066490108598138419893075_wp
        Alpha(8) = 2.6056037700656303733332680439005457628809_wp
        Alpha(9) = 3.8021293786557150470167831812773329147603_wp
        Alpha(10) = 5.5228868604047138988685783367316162184579_wp
        Alpha(11) = 8.0665421480846943892706324419350494281389_wp
        Alpha(12) = 12.1245972500037078271178936361707201285753_wp

        Xi(1) = 1.003803596370103570420417860021_wp
        Xi(2) = 1.034415659365927397353619732190_wp
        Xi(3) = 1.096621417100223667496008972400_wp
        Xi(4) = 1.192406516394687885436638741243_wp
        Xi(5) = 1.324814247094212635584738191330_wp
        Xi(6) = 1.498042874009141811977342817475_wp
        Xi(7) = 1.717591997493018617679076676108_wp
        Xi(8) = 1.990454740061872790672301003490_wp
        Xi(9) = 2.325343203762643022602940590104_wp
        Xi(10) = 2.732929767195396435815885061160_wp
        Xi(11) = 3.226077701749926812105004336750_wp
        Xi(12) = 3.820012877579278808134674583208_wp
        Xi(13) = 4.532351036147622822067987380379_wp
        Xi(14) = 5.382800296171293342756963840046_wp
        Xi(15) = 6.392248841049624770004750073937_wp
        Xi(16) = 7.580727023437338037929050793551_wp
        Xi(17) = 8.963474656843955691402914265353_wp
        Xi(18) = 10.544065759858001988386044800450_wp
        Xi(19) = 12.303519685154400178407618771104_wp
        Xi(20) = 14.185251542330838449464114336962_wp
        Xi(21) = 16.078706928215402250509358239583_wp
        Xi(22) = 17.810527882264631542238997319316_wp
        Xi(23) = 19.158744744777995581280771730803_wp
        Xi(24) = 19.903266289867562605087059068865_wp

      case (5)
        ! ----- K = 12 , R = 3E1 -----

        RIni = 3.0e1_wp

        Omega(1) = 0.0314847850052882651287426002406633074315_wp
        Omega(2) = 0.0759880787044356675646785553857043282733_wp
        Omega(3) = 0.1277666945808081383489799090491523259061_wp
        Omega(4) = 0.1931751032420674228553669066199560688801_wp
        Omega(5) = 0.2819959712025565720780375528198291590343_wp
        Omega(6) = 0.4087743922620433270642172074449405272389_wp
        Omega(7) = 0.5942255934432832301756929882508018181397_wp
        Omega(8) = 0.8680605358456985755463870824133465475825_wp
        Omega(9) = 1.2764587208952323604902595599597248110513_wp
        Omega(10) = 1.9030088361974479215735917003726740404090_wp
        Omega(11) = 2.9432226543582369869284792729402511213266_wp
        Omega(12) = 5.1490168368308802796830259129734486123198_wp

        Alpha(1) = 0.0122175904856221182198541905933319995370_wp
        Alpha(2) = 0.0655438352296026405680593254765575750298_wp
        Alpha(3) = 0.1665923720397374805578206632961091315792_wp
        Alpha(4) = 0.3255827057148261954432760800770552123140_wp
        Alpha(5) = 0.5606935393689712887610454572495655156672_wp
        Alpha(6) = 0.9021601691438875129176906686456760553483_wp
        Alpha(7) = 1.3977054457316340485822231398671533497691_wp
        Alpha(8) = 2.1199036671113539323854646490019604243571_wp
        Alpha(9) = 3.1783136457891825786634815154130251357856_wp
        Alpha(10) = 4.7445349934637615109400687884999570087530_wp
        Alpha(11) = 7.1183009631362091041914541555968298780499_wp
        Alpha(12) = 10.9882665995376886869036581195757662499091_wp

        Xi(1) = 1.004447934429248711295755425166_wp
        Xi(2) = 1.040285530941391825554553918209_wp
        Xi(3) = 1.113322679833070314174732484691_wp
        Xi(4) = 1.226321105027237134163335674497_wp
        Xi(5) = 1.383537201353583936665019027412_wp
        Xi(6) = 1.590897874604819522731564973572_wp
        Xi(7) = 1.856271132776980358988251629260_wp
        Xi(8) = 2.189828918204150003242944300119_wp
        Xi(9) = 2.604493478509141737766569013601_wp
        Xi(10) = 3.116455392284870871922805579679_wp
        Xi(11) = 3.745757447736137540211798224732_wp
        Xi(12) = 4.516908253769600511181442348274_wp
        Xi(13) = 5.459441990165533564044464398535_wp
        Xi(14) = 6.608214497704099591336018848509_wp
        Xi(15) = 8.003018535318867998987202305727_wp
        Xi(16) = 9.686740724612728241707526599669_wp
        Xi(17) = 11.700636306985161984479948404925_wp
        Xi(18) = 14.074413061266897988048352186041_wp
        Xi(19) = 16.807854304494409820869282867761_wp
        Xi(20) = 19.840822097852637208303949734045_wp
        Xi(21) = 23.012711666017984169646526382280_wp
        Xi(22) = 26.025801437599993062160796952753_wp
        Xi(23) = 28.450293676293621054906868295120_wp
        Xi(24) = 29.820317503622383885700242167616_wp

      case (6)
        ! ----- K = 12 , R = 4E1 -----

        RIni = 4.0e1_wp

        Omega(1) = 0.0252371170645091488656224908843084264731_wp
        Omega(2) = 0.0612388028802558352680415060254048853494_wp
        Omega(3) = 0.1040508555604487832515667698751737901830_wp
        Omega(4) = 0.1599052433098304811453452961567123224995_wp
        Omega(5) = 0.2385711867493230491088507250196926179342_wp
        Omega(6) = 0.3545821249596775316106564424067926211137_wp
        Omega(7) = 0.5285932206532508617451149302013391206856_wp
        Omega(8) = 0.7904471850597390465286047611659370204507_wp
        Omega(9) = 1.1868782571026183060916570077836240670877_wp
        Omega(10) = 1.8022353684717613577469241814554834490991_wp
        Omega(11) = 2.8319457784449118414610985228208051012189_wp
        Omega(12) = 5.0221823554408107077688672159609950540471_wp

        Alpha(1) = 0.0097872502132220878271146122934220912271_wp
        Alpha(2) = 0.0526466975146047576613141973089771141758_wp
        Alpha(3) = 0.1345082558306065039079946546451971300939_wp
        Alpha(4) = 0.2650565360294568865312265365519195370325_wp
        Alpha(5) = 0.4618694245718520602968660410114765113576_wp
        Alpha(6) = 0.7545813463702144361223804469585019205624_wp
        Alpha(7) = 1.1902704560564051072812929987776442430913_wp
        Alpha(8) = 1.8408744447358315109251197805306787813606_wp
        Alpha(9) = 2.8156533824155635659103324996976880356669_wp
        Alpha(10) = 4.2866097725398282212473655139461925500655_wp
        Alpha(11) = 6.5543641021523252536244708021939686659607_wp
        Alpha(12) = 10.3064204128060784605713795514247976825573_wp

        Xi(1) = 1.004917393904752588975129445892_wp
        Xi(2) = 1.044569558409484608335962563253_wp
        Xi(3) = 1.125552323994750328188009513841_wp
        Xi(4) = 1.251274713961436201286464564575_wp
        Xi(5) = 1.427010101392351663299878916646_wp
        Xi(6) = 1.660149052252017745504378276600_wp
        Xi(7) = 1.960591483574498210826556054975_wp
        Xi(8) = 2.341277156839835152726814193613_wp
        Xi(9) = 2.818850155076643253281368228791_wp
        Xi(10) = 3.414460705474536696179058137268_wp
        Xi(11) = 4.154720695961834792900996271570_wp
        Xi(12) = 5.072811401955001301983944062002_wp
        Xi(13) = 6.209671722714000902283715266350_wp
        Xi(14) = 7.615067047492341609747634745275_wp
        Xi(15) = 9.348072787550938980011272505521_wp
        Xi(16) = 11.475990199432322107046811066056_wp
        Xi(17) = 14.069749403353831805618678885139_wp
        Xi(18) = 17.192254390006447204580775078853_wp
        Xi(19) = 20.873827331033322897077542101840_wp
        Xi(20) = 25.067204801490871780497715626268_wp
        Xi(21) = 29.578089799215747926139363777054_wp
        Xi(22) = 33.987322349454254028638944262752_wp
        Xi(23) = 37.627748199051418528937595908701_wp
        Xi(24) = 39.723083912498277824043579897761_wp

      case (7)
        ! ----- K = 12 , R = 5E1 -----

        RIni = 5.0e1_wp

        Omega(1) = 0.0212246513027600224522744220764791656109_wp
        Omega(2) = 0.0517398368864022841059815277509059683325_wp
        Omega(3) = 0.0887040340301526822420282065995156273175_wp
        Omega(4) = 0.1382327771396707316017620978398916520291_wp
        Omega(5) = 0.2100216299707748596143968775051824593447_wp
        Omega(6) = 0.3184838878625700616721051761093264076408_wp
        Omega(7) = 0.4841478354476556527426385490020521729093_wp
        Omega(8) = 0.7369760980033169930946204684296674258803_wp
        Omega(9) = 1.1242221404269488538654742093569893768290_wp
        Omega(10) = 1.7309580289551620669533799068418034039496_wp
        Omega(11) = 2.7527605274972598524679118225932938912592_wp
        Omega(12) = 4.9319359578631724503293420891480991485878_wp

        Alpha(1) = 0.0082269231802614559072336203847386482124_wp
        Alpha(2) = 0.0443547657196888649133151207720748843144_wp
        Alpha(3) = 0.1138293811452207303662378240494135184235_wp
        Alpha(4) = 0.2259109878434789831300058896013105425027_wp
        Alpha(5) = 0.3976625203871757075271831105567343911389_wp
        Alpha(6) = 0.6581133163982475239948272127676887066627_wp
        Alpha(7) = 1.0535979897201334513131415482334318767244_wp
        Alpha(8) = 1.6552845367932827516915111898399004530802_wp
        Alpha(9) = 2.5719827479339771442428536074586986615031_wp
        Alpha(10) = 3.9759001427243164585468432337123090292152_wp
        Alpha(11) = 6.1684054473566169670729897323013801724301_wp
        Alpha(12) = 9.8365133085386865204394202244486677955138_wp

        Xi(1) = 1.005286713345526050981081078728_wp
        Xi(2) = 1.047944183621078399456030827430_wp
        Xi(3) = 1.135210433843728024933504572314_wp
        Xi(4) = 1.271053249752736233921072972475_wp
        Xi(5) = 1.461628112483563147485331190634_wp
        Xi(6) = 1.715604595616679065570898965465_wp
        Xi(7) = 2.044676138922400543103743975060_wp
        Xi(8) = 2.464253002358599976655273433579_wp
        Xi(9) = 2.994337852047697069290260740715_wp
        Xi(10) = 3.660609812792010766017633538461_wp
        Xi(11) = 4.495759111326378142103887824987_wp
        Xi(12) = 5.541107071300827739690486595237_wp
        Xi(13) = 6.848473528743382710698428539331_wp
        Xi(14) = 8.482121150474734669159893041268_wp
        Xi(15) = 10.520303395708345418034512785965_wp
        Xi(16) = 13.055308816817881779892562832401_wp
        Xi(17) = 16.189663296095421140055603537178_wp
        Xi(18) = 20.023783993073709482762856382010_wp
        Xi(19) = 24.626643041042886221950181102613_wp
        Xi(20) = 29.976661609065542288612982169127_wp
        Xi(21) = 35.861161955219889561213841844278_wp
        Xi(22) = 41.747243713964529397697500456843_wp
        Xi(23) = 46.711681987399490362938836440776_wp
        Xi(24) = 49.613923452080444308903706129854_wp

      case (8)
        ! ----- K = 12 , R = 6E1 -----

        RIni = 6.0e1_wp

        Omega(1) = 0.0184080983839379267157781994157517146959_wp
        Omega(2) = 0.0450560812760502163758973091095549889928_wp
        Omega(3) = 0.0778611314389885509642088082249422598125_wp
        Omega(4) = 0.1228310593868079889846747602966114953915_wp
        Omega(5) = 0.1895609344573431226393814622022127025502_wp
        Omega(6) = 0.2923043033618055340429765093235303652364_wp
        Omega(7) = 0.4514544295919487180892420985500734786910_wp
        Omega(8) = 0.6970886541216345195170166204601258641560_wp
        Omega(9) = 1.0769243506632811455646370202288153450354_wp
        Omega(10) = 1.6766859343620682565615884196397189498384_wp
        Omega(11) = 2.6921885124990737729079592899239514736109_wp
        Omega(12) = 4.8629171815900725591060027586109981712070_wp

        Alpha(1) = 0.0071319631843329120966111414180871319601_wp
        Alpha(2) = 0.0385288520026886962092113695504957959770_wp
        Alpha(3) = 0.0992694979607395232008675012291831762923_wp
        Alpha(4) = 0.1982658028996117775436891852991472262602_wp
        Alpha(5) = 0.3521334070004843228962176232554881494252_wp
        Alpha(6) = 0.5893267643752341043814242660392466177655_wp
        Alpha(7) = 0.9554417205575109880359534197946658196088_wp
        Alpha(8) = 1.5208802148863778551423145413146187365783_wp
        Alpha(9) = 2.3939769277419587804271033926184486517741_wp
        Alpha(10) = 3.7470487790881047123620062322402191057336_wp
        Alpha(11) = 5.8821023096538253248366512337952372035943_wp
        Alpha(12) = 9.4859726615850346702341067128827489796095_wp

        Xi(1) = 1.005590897534543594229843044285_wp
        Xi(2) = 1.050726610075362197922049789511_wp
        Xi(3) = 1.143190120587738044482910282351_wp
        Xi(4) = 1.287442936812643280264194245444_wp
        Xi(5) = 1.490422409980629544698493837984_wp
        Xi(6) = 1.761939900711092939341803620401_wp
        Xi(7) = 2.115302543429060096007165991949_wp
        Xi(8) = 2.568162320285456149153088811765_wp
        Xi(9) = 3.143597672735349753516190252078_wp
        Xi(10) = 3.871472651209925308103676488969_wp
        Xi(11) = 4.790153673573017386547873153191_wp
        Xi(12) = 5.948648593343000728125152809511_wp
        Xi(13) = 7.409180553411453687331217343726_wp
        Xi(14) = 9.250068242662710976849382848997_wp
        Xi(15) = 11.568467124960551785546603031918_wp
        Xi(16) = 14.481811087291866183002220225262_wp
        Xi(17) = 18.125290118076745983499709069520_wp
        Xi(18) = 22.639673152777500846760672104097_wp
        Xi(19) = 28.138486427565130274658367071083_wp
        Xi(20) = 34.636063633484886807589742119262_wp
        Xi(21) = 41.915097966903564364465895053513_wp
        Xi(22) = 49.338638589697195634825588328454_wp
        Xi(23) = 55.715382113648056268406794089287_wp
        Xi(24) = 59.494370263074479988907317107305_wp

      case (9)
        ! ----- K = 12 , R = 7E1 -----

        RIni = 7.0e1_wp

        Omega(1) = 0.0163112546177454600382497129987258155381_wp
        Omega(2) = 0.0400697356456504853207400973502672059112_wp
        Omega(3) = 0.0697424933664369035060106791212852783701_wp
        Omega(4) = 0.1112380896871291995238462124717671031249_wp
        Omega(5) = 0.1740396434997305038887283057391641705181_wp
        Omega(6) = 0.2722299633243371610688034201164242631421_wp
        Omega(7) = 0.4260759427480407430805908058912834235343_wp
        Omega(8) = 0.6657626611697179455900269529067259099975_wp
        Omega(9) = 1.0394169828252716300445029906640570516174_wp
        Omega(10) = 1.6333462645016366815691605252069962261885_wp
        Omega(11) = 2.6436380689380596005228635236861123303242_wp
        Omega(12) = 4.8076086410744625688519493511563496213057_wp

        Alpha(1) = 0.0063170013560629504831786822007660964218_wp
        Alpha(2) = 0.0341880877391648044947766987150528095185_wp
        Alpha(3) = 0.0884008694023858860409491484955157858394_wp
        Alpha(4) = 0.1775737848854596828896594898350436153578_wp
        Alpha(5) = 0.3179284457802149525178312533535418538122_wp
        Alpha(6) = 0.5373835811949758285970685234289589971013_wp
        Alpha(7) = 0.8808348910539448565386669298682420503610_wp
        Alpha(8) = 1.4179620136198704534284675826683042032528_wp
        Alpha(9) = 2.2566378506322052439284769898009130884020_wp
        Alpha(10) = 3.5692344888793875149098078214393581220065_wp
        Alpha(11) = 5.6583087139239065569383535514447203240707_wp
        Alpha(12) = 9.2106775903010455952210633334686917805811_wp

        Xi(1) = 1.005849257632735276350374742371_wp
        Xi(2) = 1.053092012352318902301644143460_wp
        Xi(3) = 1.149985578372599481257465559025_wp
        Xi(4) = 1.301434763600383661069984087000_wp
        Xi(5) = 1.515081610792431805029810421015_wp
        Xi(6) = 1.801771543866024114345994466824_wp
        Xi(7) = 2.176283703217635774997476549864_wp
        Xi(8) = 2.658329448751052618334783650411_wp
        Xi(9) = 3.273833265818200818257979745063_wp
        Xi(10) = 4.056563155826173982469035061982_wp
        Xi(11) = 5.050223975979483648397760919835_wp
        Xi(12) = 6.311120439927974264202742427088_wp
        Xi(13) = 7.911451037221845648556711738664_wp
        Xi(14) = 9.943163719482029876302331405924_wp
        Xi(15) = 12.521980802474876036736728401877_wp
        Xi(16) = 15.790426333215222919406639512374_wp
        Xi(17) = 19.916953022029388532809779377430_wp
        Xi(18) = 25.084653646156171050846106496124_wp
        Xi(19) = 31.456063148345611419492984239810_wp
        Xi(20) = 39.089771616338169758081422244800_wp
        Xi(21) = 47.776116506157187392711538365120_wp
        Xi(22) = 56.784586987926429016426688178854_wp
        Xi(23) = 64.648275342701393565703060772876_wp
        Xi(24) = 69.365518178125603089567707826291_wp

      case (10)
        ! ----- K = 12 , R = 8E1 -----

        RIni = 8.0e1_wp

        Omega(1) = 0.0146834403786877239651904761101597340911_wp
        Omega(2) = 0.0361914353823823021965373943065191753021_wp
        Omega(3) = 0.0634072655822392595469159763676980645641_wp
        Omega(4) = 0.1021478552799039412541976142023703033601_wp
        Omega(5) = 0.1617806182160848998003712870263903766954_wp
        Omega(6) = 0.2562191165902423766050858416898350355950_wp
        Omega(7) = 0.4056164627770592757797802480057569596283_wp
        Omega(8) = 0.6402570726581067900870861131501499130536_wp
        Omega(9) = 1.0086292613465813007090146613542458453594_wp
        Omega(10) = 1.5975625195441151351087236132109126174328_wp
        Omega(11) = 2.6034273539351463129000452401129450663575_wp
        Omega(12) = 4.7618102577925454188360887552988742754678_wp

        Alpha(1) = 0.0056844758581019654055792875504971028988_wp
        Alpha(2) = 0.0308158126566199542090130052834906493331_wp
        Alpha(3) = 0.0799429238320965633604023477254418850180_wp
        Alpha(4) = 0.1614319011604487479982859149396645648267_wp
        Alpha(5) = 0.2911529715886109996993234144690987363902_wp
        Alpha(6) = 0.4965287577366228496630103789177823614409_wp
        Alpha(7) = 0.8218028957748642600398408797346405663120_wp
        Alpha(8) = 1.3359859877786959548836079325617731683451_wp
        Alpha(9) = 2.1465138972910534464599158210873497409921_wp
        Alpha(10) = 3.4257800602112389596123898094504056643927_wp
        Alpha(11) = 5.4768215137600342588310164959608528079116_wp
        Alpha(12) = 8.9865269269342936426375745284644835919607_wp

        Xi(1) = 1.006073602738567478002973398787_wp
        Xi(2) = 1.055147598456383129352301186721_wp
        Xi(3) = 1.155899772802454390091542180130_wp
        Xi(4) = 1.313638049284265273022702380512_wp
        Xi(5) = 1.536647076774095812840698060953_wp
        Xi(6) = 1.836719343566482923993561648945_wp
        Xi(7) = 2.229990798652839172240558429650_wp
        Xi(8) = 2.738082525376893865435776187489_wp
        Xi(9) = 3.389574080726167901959217099161_wp
        Xi(10) = 4.221900745674179864213443336141_wp
        Xi(11) = 5.283817592672097366093830750344_wp
        Xi(12) = 6.638587627006000431613419454635_wp
        Xi(13) = 8.368001849960878511976158922181_wp
        Xi(14) = 10.577235893869860045013453486717_wp
        Xi(15) = 13.400224985438609952292932092632_wp
        Xi(16) = 17.004416677055524467765401652741_wp
        Xi(17) = 21.591855315333097315236288160634_wp
        Xi(18) = 27.389345058557642623858763286648_wp
        Xi(19) = 34.611908204340102266499679117828_wp
        Xi(20) = 43.369330502997703969542131474668_wp
        Xi(21) = 53.470561241119019674217227588997_wp
        Xi(22) = 64.102226806568861176838236559661_wp
        Xi(23) = 73.517454101237934040169452742930_wp
        Xi(24) = 79.228194435685670563018767609265_wp

      case (11)
        ! ----- K = 12 , R = 9E1 -----

        RIni = 9.0e1_wp

        Omega(1) = 0.0133794494576575845756776948422550610474_wp
        Omega(2) = 0.0330793026375962093940706967853948583524_wp
        Omega(3) = 0.0583083622425075984217843899637401250402_wp
        Omega(4) = 0.0947986056263172510938591966422883672294_wp
        Omega(5) = 0.1518020216293596127748923713118855971516_wp
        Omega(6) = 0.2430698353940594271123919190014106561648_wp
        Omega(7) = 0.3886538069394757499014358970157445583027_wp
        Omega(8) = 0.6189288727611131342518785425355076768028_wp
        Omega(9) = 0.9827038031723337512478820554573388790232_wp
        Omega(10) = 1.5672784492918067736659179511882200586115_wp
        Omega(11) = 2.5693057474657722766622769405842063861201_wp
        Omega(12) = 4.7229543998458347066858598051197759559727_wp

        Alpha(1) = 0.0051778836534097643948459314521687302246_wp
        Alpha(2) = 0.0281125858476668863738048582262551988720_wp
        Alpha(3) = 0.0731525476894912956787680145820385035904_wp
        Alpha(4) = 0.1484433217602887751442475999796855035129_wp
        Alpha(5) = 0.2695385351817635452911781079876618605340_wp
        Alpha(6) = 0.4634018779090883400240564321537206637913_wp
        Alpha(7) = 0.7736733807857686019845110114712127824532_wp
        Alpha(8) = 1.2687470578926418802480358127837689607986_wp
        Alpha(9) = 2.0556487983990030488871264546801853612124_wp
        Alpha(10) = 3.3067709916274913978500443745289771868556_wp
        Alpha(11) = 5.3255742673944999232733821248331196329673_wp
        Alpha(12) = 8.7990705510743595623573964914498901634943_wp

        Xi(1) = 1.006271692633365518279066341822_wp
        Xi(2) = 1.056963864704113705179910898124_wp
        Xi(3) = 1.161132291472661585084689972458_wp
        Xi(4) = 1.324455033512620702180855036811_wp
        Xi(5) = 1.555807813919135473845756223454_wp
        Xi(6) = 1.867859209744250026559442279250_wp
        Xi(7) = 2.278005022350426610330473287114_wp
        Xi(8) = 2.809650170707710874684268076429_wp
        Xi(9) = 3.493868344334070932940758402729_wp
        Xi(10) = 4.371557818077144736985256701089_wp
        Xi(11) = 5.496275775368103647111617249266_wp
        Xi(12) = 6.937944306517945176546180441335_wp
        Xi(13) = 8.787602244058466935232309946713_wp
        Xi(14) = 11.163280732117624457523652470314_wp
        Xi(15) = 14.216772640700797801678600951192_wp
        Xi(16) = 18.140225091983027677980389036350_wp
        Xi(17) = 23.169423359036196138904806218761_wp
        Xi(18) = 29.575886628057655853424323666445_wp
        Xi(19) = 37.629897665946181042201379440826_wp
        Xi(20) = 47.498288763048672721234222393605_wp
        Xi(21) = 59.018463345375563759709347522175_wp
        Xi(22) = 71.304831430491172651608966504000_wp
        Xi(23) = 82.328487884719088740759218580934_wp
        Xi(24) = 89.083051291139891222914393154042_wp

      case (12)
        ! ----- K = 12 , R = 1E2 -----

        RIni = 1.0e2_wp

        Omega(1) = 0.0123090459207837410891199329820988594975_wp
        Omega(2) = 0.0305206065457638403504104975677568845072_wp
        Omega(3) = 0.0541046151353604127178941081421914560678_wp
        Omega(4) = 0.0887139631277540403485239867142908565256_wp
        Omega(5) = 0.1434876391615258704037702205957849344031_wp
        Omega(6) = 0.2320235858589071699863929515439409101418_wp
        Omega(7) = 0.3742832847705676747721882657859993059901_wp
        Omega(8) = 0.6007236370303555422225059989838058527312_wp
        Omega(9) = 0.9604392175073996686953357870830672027296_wp
        Omega(10) = 1.5411563493400488027585543138187063050282_wp
        Omega(11) = 2.5398045743466134197257461080354801197245_wp
        Omega(12) = 4.6893656358078771848908739006134283044958_wp

        Alpha(1) = 0.0047621173572975827785759539954868468925_wp
        Alpha(2) = 0.0258922311893780472792723611891174684274_wp
        Alpha(3) = 0.0675671840652446778192983384281333769650_wp
        Alpha(4) = 0.1377372357479949906979791590666550860078_wp
        Alpha(5) = 0.2516682772723749776536215222444425876347_wp
        Alpha(6) = 0.4358989325752495758051034691060721115718_wp
        Alpha(7) = 0.7335111188980595020819701346237451389243_wp
        Alpha(8) = 1.2123305594792852413633249542002090493042_wp
        Alpha(9) = 1.9789993733384250202030074650139113145997_wp
        Alpha(10) = 3.2058935052153564042483335772004693353665_wp
        Alpha(11) = 5.1968507098720845721086247781528300038190_wp
        Alpha(12) = 8.6390362946276927349695262670081774558639_wp

        Xi(1) = 1.006448902079904277376797006038_wp
        Xi(2) = 1.058589676293567137109648268467_wp
        Xi(3) = 1.165821620001373944228856283090_wp
        Xi(4) = 1.334165259123700095602108040005_wp
        Xi(5) = 1.573044468555178850683283486855_wp
        Xi(6) = 1.895942984412371630866846194952_wp
        Xi(7) = 2.321435517447334613821280369095_wp
        Xi(8) = 2.874602436329419588838945953668_wp
        Xi(9) = 3.588872098063118456492612273756_wp
        Xi(10) = 4.508430174224262648730032765343_wp
        Xi(11) = 5.691416210890235599043768255356_wp
        Xi(12) = 7.214146415187596048188695130321_wp
        Xi(13) = 9.176595487619671475364169221933_wp
        Xi(14) = 11.709302775310305939730770585783_wp
        Xi(15) = 14.981573410594507048121393211204_wp
        Xi(16) = 19.209998861148907067439295381917_wp
        Xi(17) = 24.664136319335358502394250912459_wp
        Xi(18) = 31.660955242341479728279685090797_wp
        Xi(19) = 40.528232960375789543233659983912_wp
        Xi(20) = 51.494869298111737847267743717339_wp
        Xi(21) = 64.435607046864416112830209470985_wp
        Xi(22) = 78.403035185486382022124196566892_wp
        Xi(23) = 91.085884791545486179054069708627_wp
        Xi(24) = 98.930619049291083751118680567060_wp

      case (13)
        ! ----- K = 12 , R = 2E2 -----

        RIni = 2.0e2_wp

        Omega(1) = 0.0070860134344731771125858264120944696174_wp
        Omega(2) = 0.0179638474505374610739656280819498412882_wp
        Omega(3) = 0.0332648386084722072207381544928228578328_wp
        Omega(4) = 0.0580597117881842633590919015851294915365_wp
        Omega(5) = 0.1005771448110310824520792320757589521918_wp
        Omega(6) = 0.1733333929652051708802809951159851209468_wp
        Omega(7) = 0.2957310433985090605238550687250409509943_wp
        Omega(8) = 0.4987160340350490686360009073840870996719_wp
        Omega(9) = 0.8331516142667499745348702600011847607675_wp
        Omega(10) = 1.3896105342194977049505646982119344556850_wp
        Omega(11) = 2.3672949507125710425729098185243515217735_wp
        Omega(12) = 4.4930548591707977260610584746558515689685_wp

        Alpha(1) = 0.0027347725482160792943084998921771111569_wp
        Alpha(2) = 0.0150339160352873828102875271295335579680_wp
        Alpha(3) = 0.0401109195501032844837787337544110499721_wp
        Alpha(4) = 0.0846931137323594645930410068634941467280_wp
        Alpha(5) = 0.1620925487078466827506242203815745028805_wp
        Alpha(6) = 0.2958529950954934552818931248197031891323_wp
        Alpha(7) = 0.5251822397506850208519398048867543593587_wp
        Alpha(8) = 0.9139677614786228681837548792898928695649_wp
        Alpha(9) = 1.5660334617940182692138323394814847233647_wp
        Alpha(10) = 2.6532976247909677335393152652898152155103_wp
        Alpha(11) = 4.4819518632296585563742907787343483505538_wp
        Alpha(12) = 7.7409496329154443465883628228141333238455_wp

        Xi(1) = 1.007606335947693436188785098828_wp
        Xi(2) = 1.069232017552700409715633889096_wp
        Xi(3) = 1.196646752951617210668396051876_wp
        Xi(4) = 1.398375867761524281394219793029_wp
        Xi(5) = 1.687889530462865563822071057754_wp
        Xi(6) = 2.084773905860931427803939275378_wp
        Xi(7) = 2.616577766581499571571164941552_wp
        Xi(8) = 3.321363400960225862375979621888_wp
        Xi(9) = 4.251126249734825880005617548818_wp
        Xi(10) = 5.476484773152846827985046340714_wp
        Xi(11) = 7.093214112109250613340954982888_wp
        Xi(12) = 9.231335345132891039704459146265_wp
        Xi(13) = 12.067697332434482497542926893175_wp
        Xi(14) = 15.843238509328470788560527804378_wp
        Xi(15) = 20.886295899388681220759655943198_wp
        Xi(16) = 27.642822895472207384362506310538_wp
        Xi(17) = 36.712144547343591401145079800017_wp
        Xi(18) = 48.879107075080135788591739753883_wp
        Xi(19) = 65.110967290470128980561792531034_wp
        Xi(20) = 86.428747757007699487052931885955_wp
        Xi(21) = 113.430340398014508088153640130713_wp
        Xi(22) = 145.049438578274902361675557926901_wp
        Xi(23) = 176.322305548129662736145384371866_wp
        Xi(24) = 197.078080187922438468173247372306_wp

      case (14)
        ! ----- K = 12 , R = 3E2 -----

        RIni = 3.0e2_wp

        Omega(1) = 0.0051195983342860145902886647665402941598_wp
        Omega(2) = 0.0131892853745070899709111679129480698691_wp
        Omega(3) = 0.0251976096018116801238985458244457049659_wp
        Omega(4) = 0.0458385770750394970663051368714446809349_wp
        Omega(5) = 0.0827442424008601229071637253587034877000_wp
        Omega(6) = 0.1478233441962730683889295710264910610476_wp
        Omega(7) = 0.2601727262941121533506780294509042050777_wp
        Omega(8) = 0.4509304574403051471076595696496625009786_wp
        Omega(9) = 0.7718498864664846378143975291763467794226_wp
        Omega(10) = 1.3151307967790183471416820548682835578802_wp
        Omega(11) = 2.2815665390532840575178730180283537265495_wp
        Omega(12) = 4.3955515211933913343085589264802592879278_wp

        Alpha(1) = 0.0019724053061639556097514082173144389643_wp
        Alpha(2) = 0.0109300683169841590296897317242941216620_wp
        Alpha(3) = 0.0296391592056178483410768670358592657976_wp
        Alpha(4) = 0.0641708378072586352788820272807779332425_wp
        Alpha(5) = 0.1266927676410249966438507224220089142364_wp
        Alpha(6) = 0.2389728369268007877960482629453586866930_wp
        Alpha(7) = 0.4379665566520968119018988873314413012849_wp
        Alpha(8) = 0.7852367571901352752612203433901783000692_wp
        Alpha(9) = 1.3828078771058658968462831695411807686469_wp
        Alpha(10) = 2.4020735504281130311826641943184768024366_wp
        Alpha(11) = 4.1504387185145925299664670937715982290683_wp
        Alpha(12) = 7.3183269826043906326751287139131818548776_wp

        Xi(1) = 1.008267959695940689372230247844_wp
        Xi(2) = 1.075334014570839409954960963489_wp
        Xi(3) = 1.214423438364915396893949084411_wp
        Xi(4) = 1.435707676780266528926595370752_wp
        Xi(5) = 1.755348731706775427694952607105_wp
        Xi(6) = 2.197070763825094897496795209868_wp
        Xi(7) = 2.794641811616995056167384037238_wp
        Xi(8) = 3.595319636244884395435278401365_wp
        Xi(9) = 4.664555667667056663028973817475_wp
        Xi(10) = 6.092613434832621599392815658902_wp
        Xi(11) = 8.004000654849777432638113783270_wp
        Xi(12) = 10.570906296406753927148713945172_wp
        Xi(13) = 14.032353148255792877306669108606_wp
        Xi(14) = 18.721414303265493240049921475077_wp
        Xi(15) = 25.103699919348321019374470353114_wp
        Xi(16) = 33.830718480144582167856714960408_wp
        Xi(17) = 45.810545997389025451446142866985_wp
        Xi(18) = 62.290592020751041651971391388543_wp
        Xi(19) = 84.919463929991559805743328581684_wp
        Xi(20) = 115.667867926532999825472991517472_wp
        Xi(21) = 156.249955261180685589517125322345_wp
        Xi(22) = 206.183554648235459069716135616090_wp
        Xi(23) = 258.379990278291688771128420398782_wp
        Xi(24) = 294.757469626102057425054425721100_wp

      case (15)
        ! ----- K = 12 , R = 4E2 -----

        RIni = 4.0e2_wp

        Omega(1) = 0.0040632299910078106759216186403410597805_wp
        Omega(2) = 0.0106063465272962994428892742962422524045_wp
        Omega(3) = 0.0207757312971020101062205208061750560944_wp
        Omega(4) = 0.0389902821279679968334054302597779795292_wp
        Omega(5) = 0.0724629194976112335485748591756838266065_wp
        Omega(6) = 0.1327041092315043892269077474566074670292_wp
        Omega(7) = 0.2385928737227557325901398238565676024336_wp
        Omega(8) = 0.4213494413144101220281868835915162208039_wp
        Omega(9) = 0.7332827906426043373835747807287788191388_wp
        Omega(10) = 1.2677049097445011802262268552077273398027_wp
        Omega(11) = 2.2266091411905947496617519387740458114422_wp
        Omega(12) = 4.3330561116531063439206994392094429713325_wp

        Alpha(1) = 0.0015632017689609898157798153692232823175_wp
        Alpha(2) = 0.0087194292824458665820168347566909705648_wp
        Alpha(3) = 0.0239611098766580661549173598967588105779_wp
        Alpha(4) = 0.0529226964629682000570097804431624055610_wp
        Alpha(5) = 0.1069822237653646031718332465310083989607_wp
        Alpha(6) = 0.2066916845880041533352724017480461782270_wp
        Alpha(7) = 0.3874734399418142787572573448873214374544_wp
        Alpha(8) = 0.7092770597551052706611178499951364528897_wp
        Alpha(9) = 1.2728176104506660636008635867355565096659_wp
        Alpha(10) = 2.2490192304692428984947555337114977191959_wp
        Alpha(11) = 3.9460569038339200339735035605315260909265_wp
        Alpha(12) = 7.0555049835035526799471850001310713196290_wp

        Xi(1) = 1.008726364096767681775672353961_wp
        Xi(2) = 1.079569865239148009376443648755_wp
        Xi(3) = 1.226807850253259740951755629723_wp
        Xi(4) = 1.461846526871927482967364331223_wp
        Xi(5) = 1.802882368342192318986204335740_wp
        Xi(6) = 2.276806611874737694542134014419_wp
        Xi(7) = 2.922209999747150059590250492647_wp
        Xi(8) = 3.793578015320768035290727171471_wp
        Xi(9) = 4.967093522904663892118815038401_wp
        Xi(10) = 6.548938380413871829244243372159_wp
        Xi(11) = 8.687296182516121927606744801409_wp
        Xi(12) = 11.589738085971347570507505864157_wp
        Xi(13) = 15.548427601878168852547745792947_wp
        Xi(14) = 20.976776646222467454308557854858_wp
        Xi(15) = 28.462646901987146044743504091912_wp
        Xi(16) = 38.844884001942801776685598724725_wp
        Xi(17) = 53.320433416294026763432301052603_wp
        Xi(18) = 73.583309351268848384763909820094_wp
        Xi(19) = 101.967629986026167489276961219957_wp
        Xi(20) = 141.459971877096662859485398655579_wp
        Xi(21) = 195.111381434438942938935923621102_wp
        Xi(22) = 263.524373433416917222871234116610_wp
        Xi(23) = 338.072865592173264936981524897419_wp
        Xi(24) = 392.068013409484955378303538964246_wp

      case (16)
        ! ----- K = 12 , R = 5E2 -----

        RIni = 5.0e2_wp

        Omega(1) = 0.0033960422874595711797197125202812939548_wp
        Omega(2) = 0.0089658613194300815344673044431089703465_wp
        Omega(3) = 0.0179367737694282584733686822589526776994_wp
        Omega(4) = 0.0345134149083668092621013639281768448086_wp
        Omega(5) = 0.0655952830806395807914104152669398928310_wp
        Omega(6) = 0.1224040530863569261612548016993073929370_wp
        Omega(7) = 0.2236461872101807314401709231299619773381_wp
        Omega(8) = 0.4005751196598171588627960992834431408482_wp
        Omega(9) = 0.7058874382446316435367299202408020164512_wp
        Omega(10) = 1.2337273932034403278214240717858274365426_wp
        Omega(11) = 2.1870446405759955243574416128282678073447_wp
        Omega(12) = 4.2880666798091348198770922373057601362234_wp

        Alpha(1) = 0.0013049277071097104255392994673692919072_wp
        Alpha(2) = 0.0073201722780452361153633597870560301146_wp
        Alpha(3) = 0.0203478395613008850024403017034413387876_wp
        Alpha(4) = 0.0457004677472745743920673217736139193335_wp
        Alpha(5) = 0.0941638306719849646012347917090057336509_wp
        Alpha(6) = 0.1853856908569083935172999363927281990527_wp
        Alpha(7) = 0.3536480269813660527473270223097756570496_wp
        Alpha(8) = 0.6576800518814915979225188547019342877320_wp
        Alpha(9) = 1.1971736850908161353288991501919724669278_wp
        Alpha(10) = 2.1426416636948536359203593137934262813360_wp
        Alpha(11) = 3.8028020862221158767194073568873591284500_wp
        Alpha(12) = 6.8701597226829623989326212285533301837859_wp

        Xi(1) = 1.009074237506387751147143849728_wp
        Xi(2) = 1.082788834124289475795144266357_wp
        Xi(3) = 1.236243635746990285275044574309_wp
        Xi(4) = 1.481834199212673094112405203227_wp
        Xi(5) = 1.839397126819089084794923982802_wp
        Xi(6) = 2.338397763170883258791732384196_wp
        Xi(7) = 3.021387942382548134076861168218_wp
        Xi(8) = 3.948843286226582439827140214028_wp
        Xi(9) = 5.205937621410271708057709849626_wp
        Xi(10) = 6.912343228558149717862674199509_wp
        Xi(11) = 9.236561551875578097992036674668_wp
        Xi(12) = 12.416918625646503052638303099542_wp
        Xi(13) = 16.792417588408565404592409642248_wp
        Xi(14) = 22.848305341171556122684127565492_wp
        Xi(15) = 31.283472283025091970412590036688_wp
        Xi(16) = 43.109804876488710723841935035239_wp
        Xi(17) = 59.796093856544525677043644407149_wp
        Xi(18) = 83.466063442960214159571652459135_wp
        Xi(19) = 117.132188040261245025319691848154_wp
        Xi(20) = 164.827560596713928939283100305602_wp
        Xi(21) = 231.079740044725963465932494500521_wp
        Xi(22) = 317.951780134916631187502744637641_wp
        Xi(23) = 415.840851811601635301673951516932_wp
        Xi(24) = 489.064245362743104550418138387613_wp

      case (17)
        ! ----- K = 12 , R = 6E2 -----

        RIni = 6.0e2_wp

        Omega(1) = 0.0029330978418186225229015098728627131131_wp
        Omega(2) = 0.0078221933123818168768042456175104693017_wp
        Omega(3) = 0.0159389660701572009094708053805744540909_wp
        Omega(4) = 0.0313142278421377730498000903358590818470_wp
        Omega(5) = 0.0606021829072125219372614308294133778077_wp
        Omega(6) = 0.1148010491641904454824738013307694473042_wp
        Omega(7) = 0.2124732222970532045155239719846917978430_wp
        Omega(8) = 0.3848807027475980752659421441075338066184_wp
        Omega(9) = 0.6850094387100437870745557600304920242706_wp
        Omega(10) = 1.2076616461896024742102978555280401451455_wp
        Omega(11) = 2.1565782421799108423739882445246962561214_wp
        Omega(12) = 4.2534223060838955649334836017771976912627_wp

        Alpha(1) = 0.0011258192253550014499241239919122281599_wp
        Alpha(2) = 0.0063474759189304904574433830767576925069_wp
        Alpha(3) = 0.0178245049430911581278964535626774257082_wp
        Alpha(4) = 0.0406174118666135568191338876126161849811_wp
        Alpha(5) = 0.0850442268691548172035248563671583355017_wp
        Alpha(6) = 0.1700436027524185550685803872128509794948_wp
        Alpha(7) = 0.3290007972164096816420231911282101577854_wp
        Alpha(8) = 0.6196717314799680028356877736772645448582_wp
        Alpha(9) = 1.1409124547374645714274341012917091120471_wp
        Alpha(10) = 2.0628733271752983560298061460613894269045_wp
        Alpha(11) = 3.6946810222503385231444111846954569955415_wp
        Alpha(12) = 6.7296153572336118308012953281860291099292_wp

        Xi(1) = 1.009352858684480677242505108904_wp
        Xi(2) = 1.085369791743565834438341177748_wp
        Xi(3) = 1.243824575125177332463738411050_wp
        Xi(4) = 1.497938282004267463191557463453_wp
        Xi(5) = 1.868921519321079305125596348169_wp
        Xi(6) = 2.388413777333400952454164900551_wp
        Xi(7) = 3.102332381575931267341308950058_wp
        Xi(8) = 4.076287495485813287040416375007_wp
        Xi(9) = 5.403217826091134451654135473220_wp
        Xi(10) = 7.214554549252716551463460170623_wp
        Xi(11) = 9.696684339680174313270966113265_wp
        Xi(12) = 13.115277401711676166611020288855_wp
        Xi(13) = 17.851442912697049459486153821786_wp
        Xi(14) = 24.455742830752111850040053653288_wp
        Xi(15) = 33.729262324603299694897318161679_wp
        Xi(16) = 46.845267992600511424566844809192_wp
        Xi(17) = 65.529878294449921419251303689180_wp
        Xi(18) = 92.320485313405437011180509898622_wp
        Xi(19) = 130.896657375562962877246775406093_wp
        Xi(20) = 186.351426250791749744672998190254_wp
        Xi(21) = 264.783104229940797202935698351212_wp
        Xi(22) = 370.008871873476376246347996357144_wp
        Xi(23) = 491.967307881478889547421928796211_wp
        Xi(24) = 585.781742831649248459680023870533_wp

      case (18)
        ! ----- K = 12 , R = 7E2 -----

        RIni = 7.0e2_wp

        Omega(1) = 0.0025913964430986229023616985306377458187_wp
        Omega(2) = 0.0069745687472657970477579281003144284767_wp
        Omega(3) = 0.0144458782481699179102050626527509180619_wp
        Omega(4) = 0.0288910389174395462479787762694050101686_wp
        Omega(5) = 0.0567656733522657164232709265522247576996_wp
        Omega(6) = 0.1088872602145011116267676876123715601352_wp
        Omega(7) = 0.2036941460477392089637675015589834970342_wp
        Omega(8) = 0.3724435069704642134089632443672002182211_wp
        Omega(9) = 0.6683474147663148126770858714618128715301_wp
        Omega(10) = 1.1867479034725879530722328358827155625477_wp
        Omega(11) = 2.1320583937567418447971839823296136273711_wp
        Omega(12) = 4.2255387852958599535953265746712759209913_wp

        Alpha(1) = 0.0009936834721502659399096799518712186394_wp
        Alpha(2) = 0.0056283719522631914286091411783047311346_wp
        Alpha(3) = 0.0159514188285338891451384073227881543744_wp
        Alpha(4) = 0.0368179787624991746786736607210954730363_wp
        Alpha(5) = 0.0781633783526307162708946993379122147871_wp
        Alpha(6) = 0.1583492582236514436698666952185732270664_wp
        Alpha(7) = 0.3100284584585823945850936722568036429948_wp
        Alpha(8) = 0.5901523820112894313941534485401518850267_wp
        Alpha(9) = 1.0968730441192009743452129288954211006057_wp
        Alpha(10) = 2.0000185378949157531582730973340744640154_wp
        Alpha(11) = 3.6090366250071584601055024732474407755944_wp
        Alpha(12) = 6.6178680810696371172342777899899601834477_wp

        Xi(1) = 1.009584167126639444580712967614_wp
        Xi(2) = 1.087514380980905951638165740647_wp
        Xi(3) = 1.250134264708169984717245348804_wp
        Xi(4) = 1.511372453379046270531896323241_wp
        Xi(5) = 1.893623009760089895042907415235_wp
        Xi(6) = 2.430406020502554801428901631866_wp
        Xi(7) = 3.170572493340563957403208106278_wp
        Xi(8) = 4.184228167584695687233192762378_wp
        Xi(9) = 5.571163680861715695753072496998_wp
        Xi(10) = 7.473262964238113993751394525944_wp
        Xi(11) = 10.092930148516707195872899038847_wp
        Xi(12) = 13.720542848569237331414483183067_wp
        Xi(13) = 18.775586499438879755849440833515_wp
        Xi(14) = 25.868720497977782796922285868391_wp
        Xi(15) = 35.896015764496048767312474936375_wp
        Xi(16) = 50.182383787779369900550197058919_wp
        Xi(17) = 70.698658082982278487438332348347_wp
        Xi(18) = 100.381169430693210645189061835936_wp
        Xi(19) = 143.563697999821433773370848996365_wp
        Xi(20) = 206.403391174944324415463370314683_wp
        Xi(21) = 296.635299425856340693741231007152_wp
        Xi(22) = 420.066826894266429737179180392559_wp
        Xi(23) = 566.652856082373333757029598700683_wp
        Xi(24) = 682.245952844929716807342856554897_wp

      case (19)
        ! ----- K = 12 , R = 8E2 -----

        RIni = 8.0e2_wp

        Omega(1) = 0.0023278936489411329762620967223640811028_wp
        Omega(2) = 0.0063185218754966287458375373900509863745_wp
        Omega(3) = 0.0132814453404489143366560004351495383190_wp
        Omega(4) = 0.0269786605441758097134123239174696706755_wp
        Omega(5) = 0.0537007996946042293508595619894174788556_wp
        Omega(6) = 0.1041144250073139989408552326388690545400_wp
        Omega(7) = 0.1965488527774496979729857218388566764133_wp
        Omega(8) = 0.3622488239008863353252110073299263603985_wp
        Omega(9) = 0.6546090522727519354201884682620260491603_wp
        Omega(10) = 1.1694264755675532816000736224815170771762_wp
        Omega(11) = 2.1116976718956828160036337527216687703913_wp
        Omega(12) = 4.2023836966363481912334343348902621073648_wp

        Alpha(1) = 0.0008918316487520093297028874758901328956_wp
        Alpha(2) = 0.0050730424161482702214215540804931325880_wp
        Alpha(3) = 0.0144995827015344837641555196180997988620_wp
        Alpha(4) = 0.0338544679809992165817891188878130392936_wp
        Alpha(5) = 0.0727516916962798764045698175773058125060_wp
        Alpha(6) = 0.1490704377177037137896641744161385645384_wp
        Alpha(7) = 0.2948485991588116589788570731656136558740_wp
        Alpha(8) = 0.5663551610182622583116243686340851581917_wp
        Alpha(9) = 1.0611357875910290857277701315197759868170_wp
        Alpha(10) = 1.9487292245922343926012657022717178278981_wp
        Alpha(11) = 3.5388436890305506690203307673669996802346_wp
        Alpha(12) = 6.5259940885676365634505968760237237802357_wp

        Xi(1) = 1.009781189422376461309674433586_wp
        Xi(2) = 1.089342453375589435697265439096_wp
        Xi(3) = 1.255520216420012828435065488364_wp
        Xi(4) = 1.522861997141996169818217721215_wp
        Xi(5) = 1.914800539616287154351392807428_wp
        Xi(6) = 2.466514573148087141173795400206_wp
        Xi(7) = 3.229453337277839216311325909459_wp
        Xi(8) = 4.277730525222177454178207334223_wp
        Xi(9) = 5.717273132463409214109456568664_wp
        Xi(10) = 7.699387389298664483711087669349_wp
        Xi(11) = 10.441018635400417615723134279193_wp
        Xi(12) = 14.255128687173797280215992167030_wp
        Xi(13) = 19.596544917191017438906630587780_wp
        Xi(14) = 27.131721338938801776213827210427_wp
        Xi(15) = 37.845675962046086616974438499028_wp
        Xi(16) = 53.206649423236871805642511645829_wp
        Xi(17) = 75.419151614884124459892156266960_wp
        Xi(18) = 107.804895712109601112194123828658_wp
        Xi(19) = 155.338725466247233222838453059467_wp
        Xi(20) = 225.240339361376605026299202449991_wp
        Xi(21) = 326.928991424194883730347527261983_wp
        Xi(22) = 468.396188718868363570768664772004_wp
        Xi(23) = 640.048365516021274701508048110554_wp
        Xi(24) = 778.476211396508393225435185058814_wp

      case (20)
        ! ----- K = 12 , R = 9E2 -----

        RIni = 9.0e2_wp

        Omega(1) = 0.0021179393643737881895871855717286558196_wp
        Omega(2) = 0.0057940548940096001734946810565740626942_wp
        Omega(3) = 0.0123440249542606461606643352725554141358_wp
        Omega(4) = 0.0254226054514351960271001992092143595414_wp
        Omega(5) = 0.0511805111153068904816223951698139771338_wp
        Omega(6) = 0.1001552558913137193962535238800359849165_wp
        Omega(7) = 0.1905788931856397850245514111189315542561_wp
        Omega(8) = 0.3536793060741086684685072216716150705906_wp
        Omega(9) = 0.6430024712651347888314051770564105936501_wp
        Omega(10) = 1.1547365191625175191844285271791648028739_wp
        Omega(11) = 2.0943916499248837598517319902313715829223_wp
        Omega(12) = 4.1827013514862671538299676576855290477397_wp

        Alpha(1) = 0.0008107099970118842197798566422736049608_wp
        Alpha(2) = 0.0046299931135256944031852331580129122912_wp
        Alpha(3) = 0.0133373609699492468702364039376262461190_wp
        Alpha(4) = 0.0314684237600592341910720438347137672963_wp
        Alpha(5) = 0.0683619730740917786696465158202684619937_wp
        Alpha(6) = 0.1414853034908993946401666377821193520958_wp
        Alpha(7) = 0.2823490730713291792155356096927221187798_wp
        Alpha(8) = 0.5466319853569378457995749098774496133046_wp
        Alpha(9) = 1.0313485189672434379258916803401291417686_wp
        Alpha(10) = 1.9057753817936694537083300304836086525029_wp
        Alpha(11) = 3.4798372032584188881867981457673977274681_wp
        Alpha(12) = 6.4485549634885074733042786832015735853929_wp

        Xi(1) = 1.009952281380811104868379379784_wp
        Xi(2) = 1.090930967917693870028114644377_wp
        Xi(3) = 1.260205860792128242748796895789_wp
        Xi(4) = 1.532874411119366155498849879635_wp
        Xi(5) = 1.933293810743679655775317804967_wp
        Xi(6) = 2.498127091446271916480928454440_wp
        Xi(7) = 3.281156979832339873684454722991_wp
        Xi(8) = 4.360110951557405069897027649617_wp
        Xi(9) = 5.846481208596198264094978558347_wp
        Xi(10) = 7.900164152090885672646836113131_wp
        Xi(11) = 10.751433973032685802694030652304_wp
        Xi(12) = 14.734082922574942209857640218473_wp
        Xi(13) = 20.335756299552270846484058886006_wp
        Xi(14) = 28.275075403277745666325015960751_wp
        Xi(15) = 39.620843458520804584305219364637_wp
        Xi(16) = 55.977406661451159352455020368211_wp
        Xi(17) = 79.773154247089361597311540919009_wp
        Xi(18) = 114.702775670644112206153497623973_wp
        Xi(19) = 166.369080825248077573630389736081_wp
        Xi(20) = 243.049485971892347824230284913938_wp
        Xi(21) = 355.881817085389865445987922498716_wp
        Xi(22) = 515.203501803473302822489188201871_wp
        Xi(23) = 712.272365609997750501847235682362_wp
        Xi(24) = 874.487760295694255829879182329023_wp

      case (21)
        ! ----- K = 12 , R = 1E3 -----

        RIni = 1.0e3_wp

        Omega(1) = 0.0019463548977259609882261675835660175160_wp
        Omega(2) = 0.0053641265236251452123256936582729181850_wp
        Omega(3) = 0.0115705678700995050980931564896247998320_wp
        Omega(4) = 0.0241262389665478289319807498822245683812_wp
        Omega(5) = 0.0490612219221091198371250286652811567478_wp
        Omega(6) = 0.0968006411727045761107563089098437814073_wp
        Omega(7) = 0.1854888010464084995205548411778551098905_wp
        Omega(8) = 0.3463341660098125813485245294565118001628_wp
        Omega(9) = 0.6330104554036071431879877102755216355945_wp
        Omega(10) = 1.1420476000340242452198158407128403268871_wp
        Omega(11) = 2.0794138334222343020719209194524523809378_wp
        Omega(12) = 4.1656658270972997198076936964383776285104_wp

        Alpha(1) = 0.0007444374958604834432202152752402340985_wp
        Alpha(2) = 0.0042674831635267779901862078822255552879_wp
        Alpha(3) = 0.0123834190491907699616523272877210981591_wp
        Alpha(4) = 0.0294995182851700339077928103917458635408_wp
        Alpha(5) = 0.0647151525530052908758284029022700423184_wp
        Alpha(6) = 0.1351401304661587269689273713146882016645_wp
        Alpha(7) = 0.2718256002117980587326571267903219109030_wp
        Alpha(8) = 0.5299318074134546454104742474466860358007_wp
        Alpha(9) = 1.0060016003098821085271277753214747008315_wp
        Alpha(10) = 1.8690725806928342152615774018897809582995_wp
        Alpha(11) = 3.4292526283354700571224388250257675281318_wp
        Alpha(12) = 6.3820141367267550843063450560777027931181_wp

        Xi(1) = 1.010103112092257265333845872579_wp
        Xi(2) = 1.092332114783997979996603422403_wp
        Xi(3) = 1.264343204757084850888525229262_wp
        Xi(4) = 1.541728095174602886172771087914_wp
        Xi(5) = 1.949677080149472249855600869406_wp
        Xi(6) = 2.526194557119787677647651524815_wp
        Xi(7) = 3.327182911423273878282966253828_wp
        Xi(8) = 4.433661683368288571136656139871_wp
        Xi(9) = 5.962217011861906646418007849064_wp
        Xi(10) = 8.080640332344100968536837470424_wp
        Xi(11) = 11.031528435262277587289303504292_wp
        Xi(12) = 15.168030557469529144391795494329_wp
        Xi(13) = 21.008459485960092431866552686870_wp
        Xi(14) = 29.320491489101491479341010659709_wp
        Xi(15) = 41.252235313912665522206868473631_wp
        Xi(16) = 58.537788685829933197496410812732_wp
        Xi(17) = 83.820647941553667853442632917904_wp
        Xi(18) = 121.157060388289714747411807138633_wp
        Xi(19) = 176.765049793261642255282062308197_wp
        Xi(20) = 259.972767569124886100073723582682_wp
        Xi(21) = 383.661620605873604800128973124629_wp
        Xi(22) = 560.651672202403613232135626276431_wp
        Xi(23) = 783.420520368272808864329448397257_wp
        Xi(24) = 970.293091739903134307354548582225_wp

      case (22)
        ! ----- K = 12 , R = 2E3 -----

        RIni = 2.0e3_wp

        Omega(1) = 0.0011191579213804319974210639944758871600_wp
        Omega(2) = 0.0032686761628313806416415020390797696592_wp
        Omega(3) = 0.0077087399912689356829477058900140784559_wp
        Omega(4) = 0.0174324625651019033129170924395490160919_wp
        Omega(5) = 0.0377804662828506910933804539831637470115_wp
        Omega(6) = 0.0785010102683294701204951351880545473705_wp
        Omega(7) = 0.1571527226341211399098570650201267540069_wp
        Omega(8) = 0.3047343816013479167481049769694578799317_wp
        Omega(9) = 0.5755987681276236801785131658171223989484_wp
        Omega(10) = 1.0683278448896231092217837255198276125157_wp
        Omega(11) = 1.9918273202232110990345104184484625875484_wp
        Omega(12) = 4.0660174427838312121406827426284280591062_wp

        Alpha(1) = 0.0004253461425896496173415485612641709356_wp
        Alpha(2) = 0.0025124573159036588932423487378203796894_wp
        Alpha(3) = 0.0077110331233492505094457441800620545180_wp
        Alpha(4) = 0.0196667688917383451300792884461343668079_wp
        Alpha(5) = 0.0460699664597410117796848650489893728377_wp
        Alpha(6) = 0.1019345617495361915912218883639717148526_wp
        Alpha(7) = 0.2155714528451965541220106126196753848490_wp
        Alpha(8) = 0.4389685497524493766878950912202483891633_wp
        Alpha(9) = 0.8656822115642424942121187225740897019932_wp
        Alpha(10) = 1.6631068381675097116959710641381775531045_wp
        Alpha(11) = 3.1423301713480807483697132553857045422774_wp
        Alpha(12) = 6.0017142252759861143808495231866118047037_wp

        Xi(1) = 1.011039072006112649049214269326_wp
        Xi(2) = 1.101043496799447574119312776020_wp
        Xi(3) = 1.290158227396799084374379162909_wp
        Xi(4) = 1.597242793944516847659542690963_wp
        Xi(5) = 2.053046323150903518239829281811_wp
        Xi(6) = 2.704635222047308662496334363823_wp
        Xi(7) = 3.622407478466833641306266144966_wp
        Xi(8) = 4.910219350075307076508579484653_wp
        Xi(9) = 6.720507534150138795325801766012_wp
        Xi(10) = 9.277601521601525291771395131946_wp
        Xi(11) = 12.913884274682290609348456200678_wp
        Xi(12) = 18.126309733916155408905979129486_wp
        Xi(13) = 25.665795866861291329996674903668_wp
        Xi(14) = 36.680441203648435077866452047601_wp
        Xi(15) = 52.948441202086392820547189330682_wp
        Xi(16) = 77.262366058087110332053271122277_wp
        Xi(17) = 114.072385534359611369836784433573_wp
        Xi(18) = 170.573730887064442640621564351022_wp
        Xi(19) = 258.538057694210294812364736571908_wp
        Xi(20) = 397.270057505410022713476791977882_wp
        Xi(21) = 617.487588820508449316548649221659_wp
        Xi(22) = 961.300852052352183818584308028221_wp
        Xi(23) = 1448.751627700116159758181311190128_wp
        Xi(24) = 1918.726179869732959559769369661808_wp

      case (23)
        ! ----- K = 12 , R = 3E3 -----

        RIni = 3.0e3_wp

        Omega(1) = 0.0008121033978678005086831820597079861779_wp
        Omega(2) = 0.0024758914491879611614147562023355297889_wp
        Omega(3) = 0.0061840387301675452017574066525262832528_wp
        Omega(4) = 0.0146476159436097251154813240342500435531_wp
        Omega(5) = 0.0328799244389716317279591818040218598185_wp
        Omega(6) = 0.0702772393108532951038229172335825012397_wp
        Omega(7) = 0.1440596376824298634336052674087547131876_wp
        Omega(8) = 0.2850571929983484794485226743177008756902_wp
        Omega(9) = 0.5479078906024392761034587373547566357956_wp
        Omega(10) = 1.0322349251234491782818272764821188047790_wp
        Omega(11) = 1.9485640056433970451884793440022747290641_wp
        Omega(12) = 4.0167705581257705758913001403698217472993_wp

        Alpha(1) = 0.0003071487155038260791730359526704660644_wp
        Alpha(2) = 0.0018562367167545763176936772639522010842_wp
        Alpha(3) = 0.0059269627132180731568844903565940196444_wp
        Alpha(4) = 0.0157853122542906193461669048933027070802_wp
        Alpha(5) = 0.0384313670603751672376346970594340746175_wp
        Alpha(6) = 0.0878495253001757317210342924929111774190_wp
        Alpha(7) = 0.1909682470184396789957773782342442814297_wp
        Alpha(8) = 0.3981208018572957993072569671255678258603_wp
        Alpha(9) = 0.8012392220997077544892422951150479093485_wp
        Alpha(10) = 1.5667393798490310214202581029496741393814_wp
        Alpha(11) = 3.0061217791432274326710732959000438313524_wp
        Alpha(12) = 5.8193331245322036022506184682612229153165_wp

        Xi(1) = 1.011537667270811324251797436169_wp
        Xi(2) = 1.105696052958149435524980164214_wp
        Xi(3) = 1.304010485718364174031851099311_wp
        Xi(4) = 1.627225596146438388257988993946_wp
        Xi(5) = 2.109334385478168750116187424304_wp
        Xi(6) = 2.802776896249266310725753448274_wp
        Xi(7) = 3.786687504015108891473537222616_wp
        Xi(8) = 5.178921891244562241123777779350_wp
        Xi(9) = 7.154325833633077038210129616758_wp
        Xi(10) = 9.973323332104322347033131812566_wp
        Xi(11) = 14.026931720695269942451866018729_wp
        Xi(12) = 19.908322302108640984921739125468_wp
        Xi(13) = 28.528155044368060091683481083180_wp
        Xi(14) = 41.303228107103747646089431100336_wp
        Xi(15) = 60.470502289662372164685821829266_wp
        Xi(16) = 89.619228828159433752897911062973_wp
        Xi(17) = 134.609726544418902852151553872773_wp
        Xi(18) = 205.189443803730656584738412107072_wp
        Xi(19) = 317.862303811160296501681088443547_wp
        Xi(20) = 500.942847350866574590577684489290_wp
        Xi(21) = 802.607963614237230498549280355292_wp
        Xi(22) = 1297.890370986454122492759211127122_wp
        Xi(23) = 2052.941393499940959888050429071882_wp
        Xi(24) = 2852.754713529029453988883346937655_wp

      case (24)
        ! ----- K = 12 , R = 4E3 -----

        RIni = 4.0e3_wp

        Omega(1) = 0.0006481129780097903614989966194440823566_wp
        Omega(2) = 0.0020466770735909012241228119039632149523_wp
        Omega(3) = 0.0053330587438768426823312146772132491535_wp
        Omega(4) = 0.0130421841198512183248274650879117153579_wp
        Omega(5) = 0.0299838018313827793310448553004698979407_wp
        Omega(6) = 0.0653224926526793809930114716177929068408_wp
        Omega(7) = 0.1360452092785600851604192740307297526670_wp
        Omega(8) = 0.2728507539383111943560086126714381293823_wp
        Omega(9) = 0.5305383007341930598575047151044259408081_wp
        Omega(10) = 1.0094001089269477237746197584300489324960_wp
        Omega(11) = 1.9210522466051994238019073879542020222289_wp
        Omega(12) = 3.9854419726506038249534691342290670945658_wp

        Alpha(1) = 0.0002441084588471243577005574969165241361_wp
        Alpha(2) = 0.0015039659540854241270781306571979740738_wp
        Alpha(3) = 0.0049543174187313296201879949179347484289_wp
        Alpha(4) = 0.0136203241766619226683682122164276151466_wp
        Alpha(5) = 0.0340694986532491067115742747195561435092_wp
        Alpha(6) = 0.0796364298725844749237464531566033087984_wp
        Alpha(7) = 0.1763621932285932128648851466290459200081_wp
        Alpha(8) = 0.3734982400670228677372342040996500145411_wp
        Alpha(9) = 0.7618906435501059938000985305439627381929_wp
        Alpha(10) = 1.5072708912101296354008397870849478294986_wp
        Alpha(11) = 2.9213720831136537564674121592034339300881_wp
        Alpha(12) = 5.7052025799370194676304879344286291598110_wp

        Xi(1) = 1.011868124782982881842106903658_wp
        Xi(2) = 1.108784281261581288781370435714_wp
        Xi(3) = 1.313230305611500008285130736141_wp
        Xi(4) = 1.647256661318397008477582010855_wp
        Xi(5) = 2.147119232044813253107787276441_wp
        Xi(6) = 2.869040857831678182406770494239_wp
        Xi(7) = 3.898364032497771645589465516757_wp
        Xi(8) = 5.362992949926492766265367917455_wp
        Xi(9) = 7.454045205298210262367941680850_wp
        Xi(10) = 10.458479260954887335444141172047_wp
        Xi(11) = 14.811003284264805978862056257128_wp
        Xi(12) = 21.177532366231453608493295881487_wp
        Xi(13) = 30.591410896226637256001268383443_wp
        Xi(14) = 44.679355356495013291279194689309_wp
        Xi(15) = 66.043498670043934588302914079350_wp
        Xi(16) = 98.920467704516685955584698319853_wp
        Xi(17) = 150.342748961565701254494697991504_wp
        Xi(18) = 232.234614185135523775627142128997_wp
        Xi(19) = 365.256778760870198602406233590045_wp
        Xi(20) = 585.925054037377420645960057754564_wp
        Xi(21) = 959.087740800150882758412507200774_wp
        Xi(22) = 1593.727443434642945141455072644021_wp
        Xi(23) = 2613.271812605958078101764385792194_wp
        Xi(24) = 3774.791876342242130659698773342825_wp

      case (25)
        ! ----- K = 12 , R = 5E3 -----

        RIni = 5.0e3_wp

        Omega(1) = 0.0005448622032481286372600947547420457973_wp
        Omega(2) = 0.0017734323890971321100885741729735745054_wp
        Omega(3) = 0.0047781227439880182135425426732944220731_wp
        Omega(4) = 0.0119707005213017274373966600253815784072_wp
        Omega(5) = 0.0280174152450522154481120590610587761660_wp
        Omega(6) = 0.0619132750049512343984360780713149807752_wp
        Omega(7) = 0.1304701295911892246838807360442835658887_wp
        Omega(8) = 0.2642812084313892949610867028065186445929_wp
        Omega(9) = 0.5182503659100501419080934961680640071791_wp
        Omega(10) = 0.9931501845600177424120726110690071664067_wp
        Omega(11) = 1.9014045488724224262011217034817889270926_wp
        Omega(12) = 3.9630619605274942168163937461500268000236_wp

        Alpha(1) = 0.0002044570226688318376350054358871373428_wp
        Alpha(2) = 0.0012812569582845620733450595963610896266_wp
        Alpha(3) = 0.0043315761250334632171618022816045323609_wp
        Alpha(4) = 0.0122095813310535008168426361792324730260_wp
        Alpha(5) = 0.0311780207666068839313360666620766892265_wp
        Alpha(6) = 0.0741104015575603983884914877089400420118_wp
        Alpha(7) = 0.1664105913111946309087619058630913571051_wp
        Alpha(8) = 0.3565435846098175170560719282608630464892_wp
        Alpha(9) = 0.7345542696373138491765936164412664766132_wp
        Alpha(10) = 1.4656545188883580361268593739154653121659_wp
        Alpha(11) = 2.8617277624721939989580615781861183677393_wp
        Alpha(12) = 5.6245660855886989468004066861084311312879_wp

        Xi(1) = 1.012110763042167530293378996387_wp
        Xi(2) = 1.111054165000070571411167064113_wp
        Xi(3) = 1.320019842610480564752163668008_wp
        Xi(4) = 1.662046220801892847302762279416_wp
        Xi(5) = 2.175108770476030362875027868164_wp
        Xi(6) = 2.918324544001868842309221907527_wp
        Xi(7) = 3.981816971017502796662684838580_wp
        Xi(8) = 5.501281333895242910136946790089_wp
        Xi(9) = 7.680552305253643530832258212726_wp
        Xi(10) = 10.827509524563446416910161840264_wp
        Xi(11) = 15.411638503587643841789833754063_wp
        Xi(12) = 22.157344588140915005863851305890_wp
        Xi(13) = 32.197716288236957044716080034163_wp
        Xi(14) = 47.332251889737306068534072878151_wp
        Xi(15) = 70.467652321923360765609789524433_wp
        Xi(16) = 106.388566350058342439843794835497_wp
        Xi(17) = 163.136249882802584793006772656554_wp
        Xi(18) = 254.543858205361176835768155513051_wp
        Xi(19) = 404.995912311389834215002991868460_wp
        Xi(20) = 658.547479369785563552142093612929_wp
        Xi(21) = 1095.902157239230646057492890577123_wp
        Xi(22) = 1860.061478154625132841459844712517_wp
        Xi(23) = 3139.042099606381820198208743022406_wp
        Xi(24) = 4686.177830762634208827677184672211_wp

      case (26)
        ! ----- K = 12 , R = 6E3 -----

        RIni = 6.0e3_wp

        Omega(1) = 0.0004733547189603010971959035112829189407_wp
        Omega(2) = 0.0015823881371287818824851266865175181131_wp
        Omega(3) = 0.0043823111403624851655164498809291728776_wp
        Omega(4) = 0.0111926736918029304569891370195466873128_wp
        Omega(5) = 0.0265709338401388661733454506219676005685_wp
        Omega(6) = 0.0593801198165422312119589211970582809386_wp
        Omega(7) = 0.1262934109001463779865413392111683776875_wp
        Omega(8) = 0.2578165632814459010948779371652506142709_wp
        Omega(9) = 0.5089272241359717529425241155927039926610_wp
        Omega(10) = 0.9807660789125268947286941101371127160746_wp
        Omega(11) = 1.8863908554847012133794462163116634201288_wp
        Omega(12) = 3.9459563742834082044654009369111236082972_wp

        Alpha(1) = 0.0001770171299286721908510122436850378613_wp
        Alpha(2) = 0.0011264749626048635599989632106142367629_wp
        Alpha(3) = 0.0038940614761975710174005089287893599526_wp
        Alpha(4) = 0.0112041479869499801548030884675022367247_wp
        Alpha(5) = 0.0290893432383205473994880084151470889253_wp
        Alpha(6) = 0.0700727250077648080423507737091171065913_wp
        Alpha(7) = 0.1590694792498797117502552442291907652816_wp
        Alpha(8) = 0.3439359505070394979462690410176861632863_wp
        Alpha(9) = 0.7140902598402039924132753134777829018276_wp
        Alpha(10) = 1.4343293737380057116916412529583624291263_wp
        Alpha(11) = 2.8166420926084831063453128852103191093192_wp
        Alpha(12) = 5.5634336743866897334077903369831119562150_wp

        Xi(1) = 1.012300011019211134870310342393_wp
        Xi(2) = 1.112825973018472957513230747750_wp
        Xi(3) = 1.325327181573794575004596185686_wp
        Xi(4) = 1.673629716108549977298183830676_wp
        Xi(5) = 2.197085318013530684825782013370_wp
        Xi(6) = 2.957139278999038539425675553574_wp
        Xi(7) = 4.047776890306395496457736449258_wp
        Xi(8) = 5.611023815110407759413485573319_wp
        Xi(9) = 7.861109698646963799169296027713_wp
        Xi(10) = 11.123127573663211476763523766209_wp
        Xi(11) = 15.895376757966858063779447274300_wp
        Xi(12) = 22.951120792761833038983509958086_wp
        Xi(13) = 33.507452204478437359641551651634_wp
        Xi(14) = 49.510773333937274864879896085768_wp
        Xi(15) = 74.129402489529204321738742322623_wp
        Xi(16) = 112.624182707111735367300209276209_wp
        Xi(17) = 173.924249695524910375166172116224_wp
        Xi(18) = 273.567504803601884016961420797998_wp
        Xi(19) = 439.321187585954119836362252726758_wp
        Xi(20) = 722.226988948892116848554678654182_wp
        Xi(21) = 1218.075212719420962192984347893798_wp
        Xi(22) = 2103.529011125181807262407573944074_wp
        Xi(23) = 3636.192316312189927618447882196051_wp
        Xi(24) = 5587.788891480815288037575783164357_wp

      case (27)
        ! ----- K = 12 , R = 7E3 -----

        RIni = 7.0e3_wp

        Omega(1) = 0.0004206406183074642751176014425712815226_wp
        Omega(2) = 0.0014403628889290248813555614772544122282_wp
        Omega(3) = 0.0040830026632685866912733294292053853880_wp
        Omega(4) = 0.0105957928830076622490262384587711430584_wp
        Omega(5) = 0.0254497062877291561974345181912360303045_wp
        Omega(6) = 0.0574007983725829418998815641822508482051_wp
        Omega(7) = 0.1230084215018497156453043499257926640666_wp
        Omega(8) = 0.2527041493193299904671238892417406418645_wp
        Omega(9) = 0.5015205762704294671140183015101143837455_wp
        Omega(10) = 0.9708929815731902137373836303435581385202_wp
        Omega(11) = 1.8743958599473489759707281465672679132695_wp
        Omega(12) = 3.9322874092045371702611178399422442453215_wp

        Alpha(1) = 0.0001568012924109277522218339664798227029_wp
        Alpha(2) = 0.0010120177099265387919519793491138215380_wp
        Alpha(3) = 0.0035674354694836556680850582446914986967_wp
        Alpha(4) = 0.0104444295668740813643152079252807151732_wp
        Alpha(5) = 0.0274936281542208234292105645951709291808_wp
        Alpha(6) = 0.0669594194082923951550517273345608515456_wp
        Alpha(7) = 0.1533655507263689535599584037872666897329_wp
        Alpha(8) = 0.3340773637667244942630207643619044688421_wp
        Alpha(9) = 0.6980031233317447962984439446820772445790_wp
        Alpha(10) = 1.4095969802818261701988708511912307130842_wp
        Alpha(11) = 2.7809257068145880041100392876529667773866_wp
        Alpha(12) = 5.5148932957723285465737039778844064130681_wp

        Xi(1) = 1.012453649742454550326115070291_wp
        Xi(2) = 1.114265297694975322129185002495_wp
        Xi(3) = 1.329643492732267751835900648771_wp
        Xi(4) = 1.683064785667951284288228197550_wp
        Xi(5) = 2.215021562050113414785515453254_wp
        Xi(6) = 2.988894290072117407557755752734_wp
        Xi(7) = 4.101893533078582910252757320890_wp
        Xi(8) = 5.701352487159417659328547456354_wp
        Xi(9) = 8.010255908784789502692647022997_wp
        Xi(10) = 11.368275428512243724159247459937_wp
        Xi(11) = 16.298258484248156412882435617462_wp
        Xi(12) = 23.615328103170042141345508213135_wp
        Xi(13) = 34.609093419200483088965514255619_wp
        Xi(14) = 51.353658086930686718640792776114_wp
        Xi(15) = 77.246707733481081742388241195840_wp
        Xi(16) = 117.970525286899102543336503856608_wp
        Xi(17) = 183.248207119697166425642542719743_wp
        Xi(18) = 290.160519779868272233214199218310_wp
        Xi(19) = 469.578766911519154769738548793612_wp
        Xi(20) = 779.062326962213241576815647704279_wp
        Xi(21) = 1328.780547703719858487758642695553_wp
        Xi(22) = 2328.495107797294902951534822932445_wp
        Xi(23) = 4108.885896644536002586534095826210_wp
        Xi(24) = 6480.251750339536623624070443838718_wp

      case (28)
        ! ----- K = 12 , R = 8E3 -----

        RIni = 8.0e3_wp

        Omega(1) = 0.0003800261919516156242255321248864341488_wp
        Omega(2) = 0.0013301003573762626409333374786873871454_wp
        Omega(3) = 0.0038471561499707714797889309202294505496_wp
        Omega(4) = 0.0101197893626686987452541113124970806325_wp
        Omega(5) = 0.0245478874191187860814600566958243454962_wp
        Omega(6) = 0.0557982667134647215201250381594322291789_wp
        Omega(7) = 0.1203343865818571066086191574673325277445_wp
        Omega(8) = 0.2485237367430795088545049040251733174500_wp
        Omega(9) = 0.4954414608392572636700722926716977667638_wp
        Omega(10) = 0.9627660079132293382284460414588522780832_wp
        Omega(11) = 1.8645049667248534153727104745179588007886_wp
        Omega(12) = 3.9210143330472333653667627029193454291089_wp

        Alpha(1) = 0.0001412333942297918437753135947085141033_wp
        Alpha(2) = 0.0009235844571854584796527123490960525798_wp
        Alpha(3) = 0.0033129136408568407550036179773714217767_wp
        Alpha(4) = 0.0098462274094449256249853789230153111589_wp
        Alpha(5) = 0.0262254439890183024227846659767737769187_wp
        Alpha(6) = 0.0644660543575470885120713530891389808630_wp
        Alpha(7) = 0.1487684047413587857232059735723694871012_wp
        Alpha(8) = 0.3260898308896926300697559353292120931656_wp
        Alpha(9) = 0.6849120562819440391250265520728390811200_wp
        Alpha(10) = 1.3893988560757597640468383182188460978068_wp
        Alpha(11) = 2.7516769901723567494800787303077527212736_wp
        Alpha(12) = 5.4750676123472046341947305769792819774011_wp

        Xi(1) = 1.012581999202422624196257006091_wp
        Xi(2) = 1.115468316488554929139485893952_wp
        Xi(3) = 1.333254441146869702481021768836_wp
        Xi(4) = 1.690968210962891987714229458106_wp
        Xi(5) = 2.230070707413103697032072392226_wp
        Xi(6) = 3.015590954585679176564935732863_wp
        Xi(7) = 4.147496842777866264424108111442_wp
        Xi(8) = 5.777671787565585226429526466418_wp
        Xi(9) = 8.136642872935691993795115362076_wp
        Xi(10) = 11.576689096721222486247038219176_wp
        Xi(11) = 16.641981465350747019377175561772_wp
        Xi(12) = 24.184223921319856268757897410637_wp
        Xi(13) = 35.556695069646932102352954174762_wp
        Xi(14) = 52.946410986665998519434817026763_wp
        Xi(15) = 79.955208154598413738955997587254_wp
        Xi(16) = 122.643381238686981979379986285039_wp
        Xi(17) = 191.452633443719034231356168618277_wp
        Xi(18) = 304.874091538582232818743733560041_wp
        Xi(19) = 496.650186156782545099375170138956_wp
        Xi(20) = 830.452702523549992408291586798441_wp
        Xi(21) = 1430.183700202695451841883311772108_wp
        Xi(22) = 2538.038937035036850309310807460861_wp
        Xi(23) = 4560.218079011349314288992218280328_wp
        Xi(24) = 7364.042186608088992993970123279723_wp

      case (29)
        ! ----- K = 12 , R = 9E3 -----

        RIni = 9.0e3_wp

        Omega(1) = 0.0003476872036346924650056719599067536763_wp
        Omega(2) = 0.0012416884882636269877667056760715824648_wp
        Omega(3) = 0.0036555466665538696706320260457895310324_wp
        Omega(4) = 0.0097290950938871556673918894984909311319_wp
        Omega(5) = 0.0238023296274520649006825254168084882167_wp
        Omega(6) = 0.0544659975598365621249363505845808575145_wp
        Omega(7) = 0.1181011691502000846592970555171131508132_wp
        Omega(8) = 0.2450191564644082776303459295097475489911_wp
        Omega(9) = 0.4903290170640579521727127343400098880011_wp
        Omega(10) = 0.9559146236107875842192768689464799081179_wp
        Omega(11) = 1.8561541768321525229350210661749542850885_wp
        Omega(12) = 3.9114952214587968457818606404785555241688_wp

        Alpha(1) = 0.0001288425823778130519868665891051323635_wp
        Alpha(2) = 0.0008529882715494492601216538791176341450_wp
        Alpha(3) = 0.0031081525804036526562012583045982450081_wp
        Alpha(4) = 0.0093605603121938307665186201379781127940_wp
        Alpha(5) = 0.0251875715678334414904921377542645899439_wp
        Alpha(6) = 0.0624120873998579979934596406732083551105_wp
        Alpha(7) = 0.1449610016318818002872348751908226915930_wp
        Alpha(8) = 0.3194449964601693630022842540494565355402_wp
        Alpha(9) = 0.6739813444718496708379061599369919122182_wp
        Alpha(10) = 1.3724831527625502694328271657830953245139_wp
        Alpha(11) = 2.7271246942905251098472618753731921970029_wp
        Alpha(12) = 5.4415834861582763355009051409183484793175_wp

        Xi(1) = 1.012691548896615802415749341137_wp
        Xi(2) = 1.116495545573975540099703751551_wp
        Xi(3) = 1.336340203906648158049298713568_wp
        Xi(4) = 1.697729473523824816959086614343_wp
        Xi(5) = 2.242962729719132134592607386026_wp
        Xi(6) = 3.038499696573863620820479813922_wp
        Xi(7) = 4.186707020552541595823786702013_wp
        Xi(8) = 5.843438921887700056127801229167_wp
        Xi(9) = 8.245824891351123597446615143269_wp
        Xi(10) = 11.757222759267835202404151040412_wp
        Xi(11) = 16.940623929352614435162038830640_wp
        Xi(12) = 24.680139770350485697164577558738_wp
        Xi(13) = 36.385760954748655245094557386665_wp
        Xi(14) = 54.345565135131664386697325497266_wp
        Xi(15) = 82.345246801335715218506106083396_wp
        Xi(16) = 126.787846301940243534667196456667_wp
        Xi(17) = 198.771391179855170680546017081269_wp
        Xi(18) = 318.086545475709757263560462092755_wp
        Xi(19) = 521.148445853394496951604963896898_wp
        Xi(20) = 877.386738279595312817171759434132_wp
        Xi(21) = 1523.843537108099895016621871945972_wp
        Xi(22) = 2734.439606912585181364150344052177_wp
        Xi(23) = 4992.591587474125793377766058256384_wp
        Xi(24) = 8239.537097791827611636961137264734_wp

      case (30)
        ! ----- K = 12 , R = 1E4 -----

        RIni = 1.0e4_wp

        Omega(1) = 0.0003212728404662958500820070891672644575_wp
        Omega(2) = 0.0011690041207126430353211517156747367707_wp
        Omega(3) = 0.0034961596915281700555119354057939662361_wp
        Omega(4) = 0.0094012115265047685735779500575126643724_wp
        Omega(5) = 0.0231727232916882891201838454153794089052_wp
        Omega(6) = 0.0533354965287454047289837037043591649876_wp
        Omega(7) = 0.1161987098546097791707323865217915681569_wp
        Omega(8) = 0.2420238401722279862924658636047503534883_wp
        Omega(9) = 0.4859476215181996613078074798153238589293_wp
        Omega(10) = 0.9500306179525972926844246124744586268207_wp
        Omega(11) = 1.8489733269096180122382719335227818646672_wp
        Omega(12) = 3.9033086697822239601474852976892293554556_wp

        Alpha(1) = 0.0001187252048309153856080831843274048243_wp
        Alpha(2) = 0.0007951872127706306181007148516960703066_wp
        Alpha(3) = 0.0029393125552287440714112630489943112089_wp
        Alpha(4) = 0.0089568258624270747326619169191574965083_wp
        Alpha(5) = 0.0243187365497662530606538811017713896945_wp
        Alpha(6) = 0.0606828397924930648457404573636875255715_wp
        Alpha(7) = 0.1417406039031726232155705344495366659885_wp
        Alpha(8) = 0.3138030384443519017788944841518272710346_wp
        Alpha(9) = 0.6646708541365320590111991083670517355131_wp
        Alpha(10) = 1.3580375489597360417906926621789409637131_wp
        Alpha(11) = 2.7061158915034748547682325048313600746042_wp
        Alpha(12) = 5.4128928432449124911671201054730317991925_wp

        Xi(1) = 1.012786628273634303011412061135_wp
        Xi(2) = 1.117387423175520109368610843781_wp
        Xi(3) = 1.339021202886511108134763581301_wp
        Xi(4) = 1.703609387488905072861877787460_wp
        Xi(5) = 2.254187193152664353025527166352_wp
        Xi(6) = 3.058474594772011529754396352843_wp
        Xi(7) = 4.220953208805569629568504863570_wp
        Xi(8) = 5.900991421794251700228689516692_wp
        Xi(9) = 8.341574313890958056530444686416_wp
        Xi(10) = 11.915920290298694703308435283162_wp
        Xi(11) = 17.203828725510412148841377089070_wp
        Xi(12) = 25.118455231060140097135158221420_wp
        Xi(13) = 37.120842368979526635502930176358_wp
        Xi(14) = 55.590473138931917336524879402759_wp
        Xi(15) = 84.480151568398610562038442139965_wp
        Xi(16) = 130.506260232299767520824929079026_wp
        Xi(17) = 205.370941600141838523363446711301_wp
        Xi(18) = 330.069941392460730222868647842915_wp
        Xi(19) = 543.519262139611156137153358258729_wp
        Xi(20) = 920.592835018539883462640460720650_wp
        Xi(21) = 1610.929814205873337407126655307366_wp
        Xi(22) = 2919.449441247667238696905656070157_wp
        Xi(23) = 5407.926930663512710584228670995799_wp
        Xi(24) = 9107.045750956664901920589727524202_wp

      case (31)
        ! ----- K = 12 , R = 3E5 (Best : R = 204500) -----

        RIni = 3.0e5_wp

        Omega(1) = 0.0000605283299992814375306170671597275762_wp
        Omega(2) = 0.0003976819351504732883886811907123148835_wp
        Omega(3) = 0.0016279651361197932355453018996374603411_wp
        Omega(4) = 0.0052922029903677376701148283612285982969_wp
        Omega(5) = 0.0148989531146119075073506001806455945058_wp
        Omega(6) = 0.0379189506065104615268343633133207859487_wp
        Omega(7) = 0.0894577452373522931324033649613802765543_wp
        Omega(8) = 0.1988442747750825461078641454015780709597_wp
        Omega(9) = 0.4214479650252676439002932906596754492057_wp
        Omega(10) = 0.8619857428372385712666940704718143706486_wp
        Omega(11) = 1.7404432216810900461772745884125868087722_wp
        Omega(12) = 3.7794416490913082928154720363522756088059_wp

        Alpha(1) = 0.0000189340122962563570449109308913818346_wp
        Alpha(2) = 0.0002082931461534274049703011728266327518_wp
        Alpha(3) = 0.0011024023520154727638740112445094043725_wp
        Alpha(4) = 0.0042548734477736867134704129306960185275_wp
        Alpha(5) = 0.0136257458735436130452432714095512000796_wp
        Alpha(6) = 0.0384427107656654528163194233653099374237_wp
        Alpha(7) = 0.0988182876456668784028759822124499123674_wp
        Alpha(8) = 0.2363640171964210898842249919393765367204_wp
        Alpha(9) = 0.5337345457964074140876781227049718836497_wp
        Alpha(10) = 1.1508219943995056622990796890704245925008_wp
        Alpha(11) = 2.4001213099764144435262908094941280978674_wp
        Alpha(12) = 4.9906311509941725908581777737538232031511_wp

        Xi(1) = 1.014328418137978840909974032058_wp
        Xi(2) = 1.131893784203605395478724193747_wp
        Xi(3) = 1.382866538366302002361490908200_wp
        Xi(4) = 1.800491275910822536274885630192_wp
        Xi(5) = 2.440909708391232602314041555402_wp
        Xi(6) = 3.394698867579993121351700224864_wp
        Xi(7) = 4.805462676224806516100951370163_wp
        Xi(8) = 6.898933578374459715028960138383_wp
        Xi(9) = 10.031489549901425317561531880273_wp
        Xi(10) = 14.772521964267412364173182037774_wp
        Xi(11) = 22.046339415046315845375657360705_wp
        Xi(12) = 33.381979261537414117172462013627_wp
        Xi(13) = 51.363658254921380274388154063558_wp
        Xi(14) = 80.469189010275652261527934427932_wp
        Xi(15) = 128.677497542476539496569820641980_wp
        Xi(16) = 210.688423455113419333062374505516_wp
        Xi(17) = 354.643244517801801363177105486102_wp
        Xi(18) = 616.979417705080263045758215412206_wp
        Xi(19) = 1117.665086180729539400147132255370_wp
        Xi(20) = 2130.087451853435267956626830709865_wp
        Xi(21) = 4343.465540693967070318137757567456_wp
        Xi(22) = 9735.832337984589317514405593101401_wp
        Xi(23) = 25275.481162932050445135701011167839_wp
        Xi(24) = 88852.490738383218115359341027215123_wp

    end select

  case (13)
    ! ===== K = 13 =====

    select case (InitR)

      case default !(1:3)
        ! ----- K = 13 , R = 1E1 -----

        RIni = 1.0e1_wp

        Omega(1) = 0.0657951119706244487439494861291677807458_wp
        Omega(2) = 0.1558144970704423388032679176351535943468_wp
        Omega(3) = 0.2526931220910247708876655847731740323070_wp
        Omega(4) = 0.3612508995957037221270075066925642204296_wp
        Omega(5) = 0.4880601557364697465053824526615144918651_wp
        Omega(6) = 0.6422740638610490109374276390674651793233_wp
        Omega(7) = 0.8368561747820077732274747506213685710463_wp
        Omega(8) = 1.0903726151683377960672355344584616432257_wp
        Omega(9) = 1.4302814690796487430771155580089271097677_wp
        Omega(10) = 1.9007339128043295431360476532844927533006_wp
        Omega(11) = 2.5848157519580364366307279055945400614291_wp
        Omega(12) = 3.6846225483494229582717011117765082417463_wp
        Omega(13) = 5.9974171438876001808196414266660667635733_wp

        Alpha(1) = 0.0255868207120509142801567182071154071821_wp
        Alpha(2) = 0.1359771426930953219612682389416669082038_wp
        Alpha(3) = 0.3394815030426081711657955441996392664805_wp
        Alpha(4) = 0.6452380388667339172997609864079748831500_wp
        Alpha(5) = 1.0680388682879816996448146815268387399556_wp
        Alpha(6) = 1.6304515219187842809409408340037828111235_wp
        Alpha(7) = 2.3659824436194987933631400656153687123151_wp
        Alpha(8) = 3.3237148050771310988163453892241250287043_wp
        Alpha(9) = 4.5753547547140072191546034119369323889259_wp
        Alpha(10) = 6.2274123577515104557313085908987204675213_wp
        Alpha(11) = 8.4469671933229768618842103755639527662424_wp
        Alpha(12) = 11.5314290237107121188733782624069590383442_wp
        Alpha(13) = 16.1870932176644028131773378476054858765565_wp

        Xi(1) = 1.002367931474044091184373572734_wp
        Xi(2) = 1.021378676057969264582593604818_wp
        Xi(3) = 1.059761545282307757800607794785_wp
        Xi(4) = 1.118243092103653798814050235322_wp
        Xi(5) = 1.197920048122829643055115012018_wp
        Xi(6) = 1.300278657513724719159593778972_wp
        Xi(7) = 1.427202658643465212518104678896_wp
        Xi(8) = 1.580999499896081555939605922667_wp
        Xi(9) = 1.764419390174211692192546285796_wp
        Xi(10) = 1.980657586740019645990887320330_wp
        Xi(11) = 2.233343647407644670145610010792_wp
        Xi(12) = 2.526493258764244939900550646072_wp
        Xi(13) = 2.864395873153529338030198747589_wp
        Xi(14) = 3.251400783128782871148759747726_wp
        Xi(15) = 3.691587414387096180781733245624_wp
        Xi(16) = 4.188202780444439509138082611273_wp
        Xi(17) = 4.742824286606524062173684264110_wp
        Xi(18) = 5.354174409921217387461206405064_wp
        Xi(19) = 6.016530271216728066509954464536_wp
        Xi(20) = 6.717765304992344130297077242631_wp
        Xi(21) = 7.437326236737328808389102530185_wp
        Xi(22) = 8.144712410672528372056200307583_wp
        Xi(23) = 8.799430684699659993910914046467_wp
        Xi(24) = 9.353670018510573258216367342754_wp
        Xi(25) = 9.758432049866919944305920786576_wp
        Xi(26) = 9.972636514077775164548167463607_wp

      case (4)
        ! ----- K = 13 , R = 2E1 -----

        RIni = 2.0e1_wp

        Omega(1) = 0.0395138683042062947122689906681580751524_wp
        Omega(2) = 0.0943049708717428432632281544434338371730_wp
        Omega(3) = 0.1551950063737451642088696970867012225881_wp
        Omega(4) = 0.2269192918890779815579289424576359124330_wp
        Omega(5) = 0.3163249089900490444748123314289145469047_wp
        Omega(6) = 0.4333263112160413281254579126455794835238_wp
        Omega(7) = 0.5919767118483865042275968160190302569390_wp
        Omega(8) = 0.8118655465128660861807291526659469127480_wp
        Omega(9) = 1.1210827091354272629997526622425141340500_wp
        Omega(10) = 1.5638845436284244867254644706378030605265_wp
        Omega(11) = 2.2224139656716745058750994701313175028190_wp
        Omega(12) = 3.2940777647231293996394480005918126153119_wp
        Omega(13) = 5.5510419006212233681771262006066081085010_wp

        Alpha(1) = 0.0153527424497649253380107594452175234778_wp
        Alpha(2) = 0.0819058298666787536822611083542344800890_wp
        Alpha(3) = 0.2059765874406550866432307270170731783310_wp
        Alpha(4) = 0.3958844735513120887239815748337434797577_wp
        Alpha(5) = 0.6656749056064381429983697890939708941005_wp
        Alpha(6) = 1.0376871217418016666092506206098278198624_wp
        Alpha(7) = 1.5461507729488070815755673681479720471543_wp
        Alpha(8) = 2.2419642860926098973904652122968172989204_wp
        Alpha(9) = 3.1994930736548128349591713082133992429590_wp
        Alpha(10) = 4.5282874945454924214653658065543595512281_wp
        Alpha(11) = 6.3981387299032817330725342319652781952755_wp
        Alpha(12) = 9.1067784968735133151532257222982025268720_wp
        Alpha(13) = 13.3491215794234994217529188809123752434971_wp

        Xi(1) = 1.003264004819202082074980353354_wp
        Xi(2) = 1.029509967775600109973765938776_wp
        Xi(3) = 1.082718976902608210011180744914_wp
        Xi(4) = 1.164339482501544065229381885729_wp
        Xi(5) = 1.276584994157686558935438370099_wp
        Xi(6) = 1.422492898985620035645638381538_wp
        Xi(7) = 1.606012924664866670099172552000_wp
        Xi(8) = 1.832124226907446029599668690935_wp
        Xi(9) = 2.106975165590976750706886377884_wp
        Xi(10) = 2.438037647512079963973974372493_wp
        Xi(11) = 2.834264169129905308608385894331_wp
        Xi(12) = 3.306229242905461607193728923448_wp
        Xi(13) = 3.866220413454159565702639156370_wp
        Xi(14) = 4.528220762043572378274802758824_wp
        Xi(15) = 5.307669478395806272885693033459_wp
        Xi(16) = 6.220827499454482101270968996687_wp
        Xi(17) = 7.283455882798073766082325164462_wp
        Xi(18) = 8.508388718716954457477186224423_wp
        Xi(19) = 9.901432017101639004863500304054_wp
        Xi(20) = 11.454970491614840710781209054403_wp
        Xi(21) = 13.138975478166617705255703185685_wp
        Xi(22) = 14.890210861745023507296536990907_wp
        Xi(23) = 16.602876967717271838864689925686_wp
        Xi(24) = 18.127565187513955180520497378893_wp
        Xi(25) = 19.287724426470788330334471538663_wp
        Xi(26) = 19.918547471186226260897456086241_wp

      case (5)
        ! ----- K = 13 , R = 3E1 -----

        RIni = 3.0e1_wp

        Omega(1) = 0.0290391751465786714628172348467982644138_wp
        Omega(2) = 0.0696930970758023124331454259405838058683_wp
        Omega(3) = 0.1159194578409333255356603378793955982928_wp
        Omega(4) = 0.1723198518541611062252633113089927974215_wp
        Omega(5) = 0.2457778053882376194231619559449342204971_wp
        Omega(6) = 0.3464046083311216512354117902372507842301_wp
        Omega(7) = 0.4884592767890422960661272328009019361161_wp
        Omega(8) = 0.6916744519623734678386017815654440710205_wp
        Omega(9) = 0.9843875484296040770599081626279769352550_wp
        Omega(10) = 1.4113359869286716225889391496117752922146_wp
        Omega(11) = 2.0550145974924654234099880012465177969716_wp
        Omega(12) = 3.1115307366236419309591526571878716822539_wp
        Omega(13) = 5.3422507011769150369294434366196355767897_wp

        Alpha(1) = 0.0112757434961027098823437898357513464731_wp
        Alpha(2) = 0.0603223978722132603997598768869048413421_wp
        Alpha(3) = 0.1525007237397825909968251609227785081657_wp
        Alpha(4) = 0.2955277282819508820495540274420065429695_wp
        Alpha(5) = 0.5027902095349923103644214361640507604534_wp
        Alpha(6) = 0.7960938616773461903365106950847263078686_wp
        Alpha(7) = 1.2093501035922478453353209881804275482864_wp
        Alpha(8) = 1.7933089972016881873204813269140345255437_wp
        Alpha(9) = 2.6223634743044171358829219631303431015112_wp
        Alpha(10) = 3.8064590061885735020313220600129966442182_wp
        Alpha(11) = 5.5162604961162127548482403405216700775782_wp
        Alpha(12) = 8.0501192457232339100281492960675677750260_wp
        Alpha(13) = 12.0989895223483451391216503267855841841083_wp

        Xi(1) = 1.003823181499404824175598893721_wp
        Xi(2) = 1.034594777838683702080729842621_wp
        Xi(3) = 1.097135219703572158616612652882_wp
        Xi(4) = 1.193463503379344423376180739371_wp
        Xi(5) = 1.326679074476580561317562434454_wp
        Xi(6) = 1.501068447036459740429915898918_wp
        Xi(7) = 1.722267936530475874101417388395_wp
        Xi(8) = 1.997483259315246300019501546341_wp
        Xi(9) = 2.335760526037198537771533146667_wp
        Xi(10) = 2.748304829870090944731950144408_wp
        Xi(11) = 3.248839260751835120036648540953_wp
        Xi(12) = 3.854000274324109914661842801742_wp
        Xi(13) = 4.583734970851368934700170720831_wp
        Xi(14) = 5.461642641717418074007173167672_wp
        Xi(15) = 6.515120127022255510935294653407_wp
        Xi(16) = 7.775058708641121807837032209321_wp
        Xi(17) = 9.274648381319773452965971038964_wp
        Xi(18) = 11.046507851525320275352937127167_wp
        Xi(19) = 13.116937904918466130041454498212_wp
        Xi(20) = 15.495562270687938125762750374292_wp
        Xi(21) = 18.158489810829828919186423874521_wp
        Xi(22) = 21.024310467838543146912311243568_wp
        Xi(23) = 23.926867775241861960658362740162_wp
        Xi(24) = 26.598563146118983995316864543668_wp
        Xi(25) = 28.689894066111251145698157749564_wp
        Xi(26) = 29.849150430210857226978604650469_wp

      case (6)
        ! ----- K = 13 , R = 4E1 -----

        RIni = 4.0e1_wp

        Omega(1) = 0.0232593715227870833565752828331341106605_wp
        Omega(2) = 0.0560723340674415472326076650272375090367_wp
        Omega(3) = 0.0940721603347912478917188135174587415577_wp
        Omega(4) = 0.1417366035997742708055094698615938852981_wp
        Omega(5) = 0.2059105637639777391587059263144965370884_wp
        Omega(6) = 0.2967053050336383149450120849666134859035_wp
        Omega(7) = 0.4283329410906126902584215715430460136304_wp
        Omega(8) = 0.6205113431853984286962720717806263337479_wp
        Omega(9) = 0.9018119630642785586219049853706763997252_wp
        Omega(10) = 1.3175182943399094910369093169855148062197_wp
        Omega(11) = 1.9506750470698549813431191268797704196913_wp
        Omega(12) = 2.9969169133068959927964497458319215184019_wp
        Omega(13) = 5.2111890672789227571364001345699534795131_wp

        Alpha(1) = 0.0090268981545618437966382375560459028208_wp
        Alpha(2) = 0.0483993303389877087190048664949726386908_wp
        Alpha(3) = 0.1228819816015637557480699032508297818822_wp
        Alpha(4) = 0.2397391142273433974437717511318801655307_wp
        Alpha(5) = 0.4118217589198438482317454756298502616119_wp
        Alpha(6) = 0.6603888791946662452933894393858338389691_wp
        Alpha(7) = 1.0187846262288229165753755678380798599392_wp
        Alpha(8) = 1.5371450840589502646795633244991563515214_wp
        Alpha(9) = 2.2893425823845693278454915819963844114682_wp
        Alpha(10) = 3.3851999418594697212163413535535028131562_wp
        Alpha(11) = 4.9958531488654190407666666473573968687560_wp
        Alpha(12) = 7.4203059140681452122111083635758177479147_wp
        Alpha(13) = 11.3476308264797006841034798796385985042434_wp

        Xi(1) = 1.004232043627023784893026459031_wp
        Xi(2) = 1.038318156231480049456195047242_wp
        Xi(3) = 1.107721820878299290313862002222_wp
        Xi(4) = 1.214939458996826759928922001563_wp
        Xi(5) = 1.363815600562108043547482338909_wp
        Xi(6) = 1.559695619851743542128019726078_wp
        Xi(7) = 1.809661579665602149091796491120_wp
        Xi(8) = 2.122852714864137180228453871678_wp
        Xi(9) = 2.510866375264567018632533290656_wp
        Xi(10) = 2.988241273848355028519563347800_wp
        Xi(11) = 3.573027963140888108029002978583_wp
        Xi(12) = 4.287456156285559316643229976762_wp
        Xi(13) = 5.158681354306017041722043359187_wp
        Xi(14) = 6.219557003438503706149731442565_wp
        Xi(15) = 7.509292114577087673410460411105_wp
        Xi(16) = 9.073709985610532768873781606089_wp
        Xi(17) = 10.964534802859470019598120416049_wp
        Xi(18) = 13.236652315357856579680628783535_wp
        Xi(19) = 15.941500228811626982405869057402_wp
        Xi(20) = 19.113641092040605684587717405520_wp
        Xi(21) = 22.746480276188918878688127733767_wp
        Xi(22) = 26.753645194581302746428264072165_wp
        Xi(23) = 30.918352582718743093437296920456_wp
        Xi(24) = 34.850218727394619122605945449322_wp
        Xi(25) = 37.996806449500844848898850614205_wp
        Xi(26) = 39.768061025330197821858746465296_wp

      case (7)
        ! ----- K = 13 , R = 5E1 -----

        RIni = 5.0e1_wp

        Omega(1) = 0.0195487683014131804904302096292934010080_wp
        Omega(2) = 0.0473066906441472042096202648558955772273_wp
        Omega(3) = 0.0799532151988612554909083458831720037097_wp
        Omega(4) = 0.1218567775112953490950352779254028945388_wp
        Omega(5) = 0.1797943671268491339445844828803799941852_wp
        Omega(6) = 0.2637979780802783293352630628048061112167_wp
        Omega(7) = 0.3879558647919771652051897092849941373061_wp
        Omega(8) = 0.5719461085923810345142508093640998367846_wp
        Omega(9) = 0.8445649615904518039380573235508364859925_wp
        Omega(10) = 1.2516047076139490799731937564764905346237_wp
        Omega(11) = 1.8766592674405786687853378569279527710023_wp
        Omega(12) = 2.9152010854831847893892798895265627834306_wp
        Omega(13) = 5.1177889314922459049889136295519165287260_wp

        Alpha(1) = 0.0075835662242047441891812278435408556376_wp
        Alpha(2) = 0.0407376307940227880337270491284407114563_wp
        Alpha(3) = 0.1038080361905194638215996043306699192499_wp
        Alpha(4) = 0.2037030441964128006529597542106913010684_wp
        Alpha(5) = 0.3528297904687777034678404508571958331231_wp
        Alpha(6) = 0.5719362889920455448114489094280088465894_wp
        Alpha(7) = 0.8937509529926758017857879590195580021827_wp
        Alpha(8) = 1.3676956728177227598248349993426131732122_wp
        Alpha(9) = 2.0670104178735341665142100087138032904477_wp
        Alpha(10) = 3.1012606244403970821160720916509490052704_wp
        Alpha(11) = 4.6418880421602061927405236119170695019420_wp
        Alpha(12) = 6.9884992950388464334927907106731481690076_wp
        Alpha(13) = 10.8291419643019349878382273644206179596949_wp

        Xi(1) = 1.004554608672238489620875956376_wp
        Xi(2) = 1.041258964374473610075416685206_wp
        Xi(3) = 1.116101627887072712374399996449_wp
        Xi(4) = 1.231992672938907364978654290777_wp
        Xi(5) = 1.393424205008656586654569586869_wp
        Xi(6) = 1.606667965054614480367549400608_wp
        Xi(7) = 1.880081747489247432562724238192_wp
        Xi(8) = 2.224527584018628217563118276701_wp
        Xi(9) = 2.653899139995383925361149834998_wp
        Xi(10) = 3.185767125771570655850400966003_wp
        Xi(11) = 3.842162053775606351787089143279_wp
        Xi(12) = 4.650522429160159222378406740361_wp
        Xi(13) = 5.644807060262674138981231486412_wp
        Xi(14) = 6.866737597482240773594242000399_wp
        Xi(15) = 8.367044610287674544224501005374_wp
        Xi(16) = 10.206415511463699370003910260607_wp
        Xi(17) = 12.455507430207186489803938833631_wp
        Xi(18) = 15.192739213854361472570697610962_wp
        Xi(19) = 18.497452835697486160301816759954_wp
        Xi(20) = 22.434251096592815839445012038311_wp
        Xi(21) = 27.022081642493139871236129678778_wp
        Xi(22) = 32.180707418363440258657348636007_wp
        Xi(23) = 37.653189638622882779578349143890_wp
        Xi(24) = 42.927061612284193800603837587460_wp
        Xi(25) = 47.225778138736842323014419875449_wp
        Xi(26) = 49.677259754806401468446575009352_wp

      case (8)
        ! ----- K = 13 , R = 6E1 -----

        RIni = 6.0e1_wp

        Omega(1) = 0.0169449723784150246818773792242818210241_wp
        Omega(2) = 0.0411429334765565038551575309522245760263_wp
        Omega(3) = 0.0699893468845359939195321062610499751599_wp
        Omega(4) = 0.1077561446651243788464702259843885912005_wp
        Omega(5) = 0.1611404671865263836121094989661806096137_wp
        Omega(6) = 0.2400622452866041391230994203609405701627_wp
        Omega(7) = 0.3584665626662518737204684432517964864928_wp
        Omega(8) = 0.5359938990470181285289547057359982318303_wp
        Omega(9) = 0.8016474287450003043944621317073995214741_wp
        Omega(10) = 1.2016720578835738999752894007322367997403_wp
        Omega(11) = 1.8201701757324859294378746610298946961848_wp
        Omega(12) = 2.8525952723313601027115699038105844920210_wp
        Omega(13) = 5.0462646056798155956221729745436732628150_wp

        Alpha(1) = 0.0065710035402700543248571778392336240060_wp
        Alpha(2) = 0.0353569789480877304717082888069201729309_wp
        Alpha(3) = 0.0903879851002289691225960455156496209383_wp
        Alpha(4) = 0.1782821245318206675314421164046940759818_wp
        Alpha(5) = 0.3110700601878565301492210459777254527580_wp
        Alpha(6) = 0.5090319817929682756391281117647196197140_wp
        Alpha(7) = 0.8042946625299619084416453962393944721043_wp
        Alpha(8) = 1.2455733860384253659228312449691600249935_wp
        Alpha(9) = 1.9054792787505055919744512138436220993754_wp
        Alpha(10) = 2.8932856501032743546438646964347185530642_wp
        Alpha(11) = 4.3806489535904541733624117938461495214142_wp
        Alpha(12) = 6.6677158397230641838533682363987509233993_wp
        Alpha(13) = 10.4419368034881749653097759988895631977357_wp

        Xi(1) = 1.004820923451609358378194825079_wp
        Xi(2) = 1.043689146683935296410113136023_wp
        Xi(3) = 1.123038762755039575327691003803_wp
        Xi(4) = 1.246146262067672114007624295517_wp
        Xi(5) = 1.418079083392706159476998051261_wp
        Xi(6) = 1.645936438125727314767121622907_wp
        Xi(7) = 1.939224014751121743745966574668_wp
        Xi(8) = 2.310365837291107429926281802146_wp
        Xi(9) = 2.775354009902637464389865695757_wp
        Xi(10) = 3.354552724089791685031791690363_wp
        Xi(11) = 4.073693486997367368268596621306_wp
        Xi(12) = 4.965104191638338092164989268085_wp
        Xi(13) = 6.069197598346911939198866647871_wp
        Xi(14) = 7.436205820433130344714739434764_wp
        Xi(15) = 9.128058479981291190030745497097_wp
        Xi(16) = 11.220113416560385063112936787633_wp
        Xi(17) = 13.802057535277674121809710072739_wp
        Xi(18) = 16.976532089272953164257873304166_wp
        Xi(19) = 20.852555032717676044423660286498_wp
        Xi(20) = 25.528387830544335267146660584103_wp
        Xi(21) = 31.054840784752701937296293088764_wp
        Xi(22) = 37.367150273092381997097621848525_wp
        Xi(23) = 44.178361632359741350412063454201_wp
        Xi(24) = 50.857358679482774822117052693216_wp
        Xi(25) = 56.387973898118182392025232729793_wp
        Xi(26) = 59.578031283605855136786821901751_wp

      case (9)
        ! ----- K = 13 , R = 7E1 -----

        RIni = 7.0e1_wp

        Omega(1) = 0.0150071017032093006110648873187296548082_wp
        Omega(2) = 0.0365471975379521750101896593321271211607_wp
        Omega(3) = 0.0625365727088906125669814928258460895449_wp
        Omega(4) = 0.0971612207871765812269653460331308991726_wp
        Omega(5) = 0.1470341958518947886704400507262846531376_wp
        Omega(6) = 0.2219512087350919740772218699542683850723_wp
        Omega(7) = 0.3357153160293236112755085182035585944504_wp
        Omega(8) = 0.5079360945962475216075224238032603807369_wp
        Omega(9) = 0.7678029510712538914778829690632733218081_wp
        Omega(10) = 1.1619597933934012856226763998890305629175_wp
        Omega(11) = 1.7749717076250380731619951668598389460385_wp
        Omega(12) = 2.8023472237986139813867292325255675677909_wp
        Omega(13) = 4.9888828936432936915490454765631511691026_wp

        Alpha(1) = 0.0058175684464509904005360128084051174469_wp
        Alpha(2) = 0.0313496158111821703283461897776573934493_wp
        Alpha(3) = 0.0803767583469204337375695924861052077404_wp
        Alpha(4) = 0.1592740538724147044731408243145587277922_wp
        Alpha(5) = 0.2797464647733758327156992212758623850277_wp
        Alpha(6) = 0.4616472946686746037258698704963677528212_wp
        Alpha(7) = 0.7365358864903887340064118915972102286105_wp
        Alpha(8) = 1.1524615159229960063749093057694494746102_wp
        Alpha(9) = 1.7814423255914706925454385477713969976321_wp
        Alpha(10) = 2.7324567429500320847428340709228677951614_wp
        Alpha(11) = 4.1773180520687124769874110086220753146335_wp
        Alpha(12) = 6.4166540692766925774344344279853658008506_wp
        Alpha(13) = 10.1375583316785151648947715941062597266864_wp

        Xi(1) = 1.005047588223062141120182910736_wp
        Xi(2) = 1.045759117208935584848218669496_wp
        Xi(3) = 1.128956539525765419827198210445_wp
        Xi(4) = 1.258246110899336475371160237735_wp
        Xi(5) = 1.439214623863955571877494554744_wp
        Xi(6) = 1.679711690973732572435168919345_wp
        Xi(7) = 1.990289708987905371731855219686_wp
        Xi(8) = 2.384806754208886699812966480927_wp
        Xi(9) = 2.881194346521214070047656186091_wp
        Xi(10) = 3.502416205017879272543696034248_wp
        Xi(11) = 4.277673082152316418481224902237_wp
        Xi(12) = 5.243912828705896697362859315561_wp
        Xi(13) = 6.447699270215790543717832017379_wp
        Xi(14) = 7.947451355749782035773481903007_wp
        Xi(15) = 9.815983103290664850242019934257_wp
        Xi(16) = 12.143078404884346526738936322687_wp
        Xi(17) = 15.037406339684316336491320953428_wp
        Xi(18) = 18.626199237743337598638415641972_wp
        Xi(19) = 23.049352030501556292882425758251_wp
        Xi(20) = 28.441451052179243930498331316414_wp
        Xi(21) = 34.890186440161893889622124476091_wp
        Xi(22) = 42.354196800755224697893064345067_wp
        Xi(23) = 50.526044781888757088245123227921_wp
        Xi(24) = 58.660862952684484434434653010726_wp
        Xi(25) = 65.491295607339020265480478144582_wp
        Xi(26) = 69.471289540743901530506221320138_wp

      case (10)
        ! ----- K = 13 , R = 8E1 -----

        RIni = 8.0e1_wp

        Omega(1) = 0.0135031151724071027421650087116389293840_wp
        Omega(2) = 0.0329745931435352286415689602311340777874_wp
        Omega(3) = 0.0567263877008931249511724474038354415484_wp
        Omega(4) = 0.0888672098152087184074854157855849479120_wp
        Omega(5) = 0.1359255258845317027428887396434120660160_wp
        Omega(6) = 0.2075704507292119542759831554712413037578_wp
        Omega(7) = 0.3174713221754208590150679591967985970768_wp
        Omega(8) = 0.4852129397558113599786457775797643421356_wp
        Omega(9) = 0.7401508114959611988659983483618276522975_wp
        Omega(10) = 1.1292813870580639911116865858886626483581_wp
        Omega(11) = 1.7375901390625920424392106267141855369118_wp
        Omega(12) = 2.7606812801987436849487228540667160814337_wp
        Omega(13) = 4.9413195910534541549496978518618561793119_wp

        Alpha(1) = 0.0052329389711535952884931147208874557819_wp
        Alpha(2) = 0.0282375221936975021197891843466321049050_wp
        Alpha(3) = 0.0725906701828313244301304520134099362849_wp
        Alpha(4) = 0.1444595107707575670628235078862822149404_wp
        Alpha(5) = 0.2552626614849805273025577095902249880055_wp
        Alpha(6) = 0.4244631647399713661938588926680893109733_wp
        Alpha(7) = 0.6830922302920382192543274912921447139524_wp
        Alpha(8) = 1.0785818503932039662046235561909668376757_wp
        Alpha(9) = 1.6824002197305862953916749669147634449473_wp
        Alpha(10) = 2.6032393267837956702618479321031941253750_wp
        Alpha(11) = 4.0130294793175877959660879934489230436157_wp
        Alpha(12) = 6.2128284447499835686096192333138787944335_wp
        Alpha(13) = 9.8895184069447695093466044191643504746025_wp

        Xi(1) = 1.005244773103949057686291868041_wp
        Xi(2) = 1.047561077168103439152466171969_wp
        Xi(3) = 1.134114802087177959052585740629_wp
        Xi(4) = 1.268812673247660698181565885889_wp
        Xi(5) = 1.457715843211539361165204886639_wp
        Xi(6) = 1.709361714395951217966562740713_wp
        Xi(7) = 2.035267679186395596352063264156_wp
        Xi(8) = 2.450621129468917855538656613845_wp
        Xi(9) = 2.975160676224555998062998307674_wp
        Xi(10) = 3.634287845813459411677820432018_wp
        Xi(11) = 4.460475691095960163769229556152_wp
        Xi(12) = 5.495061529171468354872803763200_wp
        Xi(13) = 6.790494975217389227173381271996_wp
        Xi(14) = 8.413089781164437445069859133895_wp
        Xi(15) = 10.446246556745676386905585797393_wp
        Xi(16) = 12.993916949523328499549457148987_wp
        Xi(17) = 16.183625001847018949271972410031_wp
        Xi(18) = 20.167380860878859104648452849062_wp
        Xi(19) = 25.116783059549724644102242088195_wp
        Xi(20) = 31.204786827358898191817071499088_wp
        Xi(21) = 38.560022053792734751664905701318_wp
        Xi(22) = 47.171348864346439016398404575625_wp
        Xi(23) = 56.719719279352597291615767360895_wp
        Xi(24) = 66.352244188342391371937534216841_wp
        Xi(25) = 74.541686169997492819594420154772_wp
        Xi(26) = 79.357724794226199571678392885588_wp

      case (11)
        ! ----- K = 13 , R = 9E1 -----

        RIni = 9.0e1_wp

        Omega(1) = 0.0122986244453973046956789770545181994521_wp
        Omega(2) = 0.0301091511469223380805432287513068168039_wp
        Omega(3) = 0.0520540922468294390643302714527518304521_wp
        Omega(4) = 0.0821720281454225061763515028379067928199_wp
        Omega(5) = 0.1269082267776730492159524207873566581384_wp
        Omega(6) = 0.1958075833731895737561956583050459812512_wp
        Omega(7) = 0.3024160839681164305876165387942933193699_wp
        Omega(8) = 0.4662984385462345585224210736674521626810_wp
        Omega(9) = 0.7169579916993148207258503001870764137493_wp
        Omega(10) = 1.1017045394902961527984833201010417269572_wp
        Omega(11) = 1.7059067569607391338937230074179751682095_wp
        Omega(12) = 2.7252874685729666863042569557862293549988_wp
        Omega(13) = 4.9009295252479360156111498625364220060874_wp

        Alpha(1) = 0.0047648122069902020234673299496630516359_wp
        Alpha(2) = 0.0257437170231617231684525517673711725308_wp
        Alpha(3) = 0.0663430976309872055498477808055390880781_wp
        Alpha(4) = 0.1325491890132242400892432707126822322152_wp
        Alpha(5) = 0.2355255628264578346566611097290255827374_wp
        Alpha(6) = 0.3943771928109263805216702647449267260527_wp
        Alpha(7) = 0.6396459009904628332772794085947509756807_wp
        Alpha(8) = 1.0181944860684312073725518721367677699163_wp
        Alpha(9) = 1.6009832616680563971466380590591427335312_wp
        Alpha(10) = 2.4964300537331274747795267776595551367791_wp
        Alpha(11) = 3.8765534322194465295143006589029255337664_wp
        Alpha(12) = 6.0427974354698543641740970944198352299281_wp
        Alpha(13) = 9.6819261346061663729353763407914357230766_wp

        Xi(1) = 1.005419169394431866989929325040_wp
        Xi(2) = 1.049155728700803947629240087114_wp
        Xi(3) = 1.138684850126388985029103617652_wp
        Xi(4) = 1.278189650421907566299473124616_wp
        Xi(5) = 1.474168408304818242253453330903_wp
        Xi(6) = 1.735794802552608655623125888390_wp
        Xi(7) = 2.075483220568718073896843923087_wp
        Xi(8) = 2.509661483903695123297283853248_wp
        Xi(9) = 3.059764774506369659854559950318_wp
        Xi(10) = 3.753493148167162929938756033010_wp
        Xi(11) = 4.626424342029386261102019695102_wp
        Xi(12) = 5.724083056224131806945243372553_wp
        Xi(13) = 7.104569892337821482384413451783_wp
        Xi(14) = 8.841829722144629405584936865381_wp
        Xi(15) = 11.029571080202116001474366147050_wp
        Xi(16) = 13.785653703813367473331175094842_wp
        Xi(17) = 17.256289034577141021375989993203_wp
        Xi(18) = 21.618330610373903860263467002767_wp
        Xi(19) = 27.075670471000502757125127928362_wp
        Xi(20) = 33.841219752308959727027382058395_wp
        Xi(21) = 42.087911423474211412149026401153_wp
        Xi(22) = 51.840769452913119781645168160367_wp
        Xi(23) = 62.777335074704843181220281778110_wp
        Xi(24) = 73.942898924447816359772911454229_wp
        Xi(25) = 83.543802300131321615650481504645_wp
        Xi(26) = 89.237879240503543407103670404013_wp

      case (12)
        ! ----- K = 13 , R = 1E2 -----

        RIni = 1.0e2_wp

        Omega(1) = 0.0113101318768680369468646322757277022930_wp
        Omega(2) = 0.0277543331291073884246130461239676634477_wp
        Omega(3) = 0.0482051453560221898190271249545757470401_wp
        Omega(4) = 0.0766369992937893382286911568568310570981_wp
        Omega(5) = 0.1194143136132964391524333321037421029587_wp
        Omega(6) = 0.1859624116085836987492894284423883277668_wp
        Omega(7) = 0.2897143390284020331713107226923042958333_wp
        Omega(8) = 0.4502180845978160193137946470942623022893_wp
        Omega(9) = 0.6971087365781358037076544109655884540189_wp
        Omega(10) = 1.0779765276442093285824697090902191121131_wp
        Omega(11) = 1.6785411291166155571845039484735195856047_wp
        Omega(12) = 2.6946568538459302127267142035194069649151_wp
        Omega(13) = 4.8659851887432857154430765334041097958107_wp

        Alpha(1) = 0.0043806958779880428850311965480179843269_wp
        Alpha(2) = 0.0236960325297795352306781235113275485560_wp
        Alpha(3) = 0.0612068179585451655720922548714613142806_wp
        Alpha(4) = 0.1227397677409143183610193937604915248585_wp
        Alpha(5) = 0.2192288472522569340688843378406325257401_wp
        Alpha(6) = 0.3694491542866578709294195226320312031021_wp
        Alpha(7) = 0.6034892573939584233628716236630395997054_wp
        Alpha(8) = 0.9676872342224387200763924266055937550846_wp
        Alpha(9) = 1.5325340755817475489782009123729267230374_wp
        Alpha(10) = 2.4061866587862055992357024347683136511478_wp
        Alpha(11) = 3.7607303110864559931615647148106518216082_wp
        Alpha(12) = 5.8979582101501310427427160476554490742274_wp
        Alpha(13) = 9.5045784620299077691263356371109694009647_wp

        Xi(1) = 1.005575418829604012718910432955_wp
        Xi(2) = 1.050585204860276456232358321952_wp
        Xi(3) = 1.142785715858074474440136958364_wp
        Xi(4) = 1.286616261298543977507902180690_wp
        Xi(5) = 1.488980867544841086343859748453_wp
        Xi(6) = 1.759646306988846534602628779442_wp
        Xi(7) = 2.111865308076226843518596876237_wp
        Xi(8) = 2.563231938172798349100112158538_wp
        Xi(9) = 3.136781443005848844362420946297_wp
        Xi(10) = 3.862392628341886536289320752680_wp
        Xi(11) = 4.778600044854552687540599764660_wp
        Xi(12) = 5.934938395222085082310325976351_wp
        Xi(13) = 7.394949794463160231382248266385_wp
        Xi(14) = 9.239970870682225946059207277727_wp
        Xi(15) = 11.573758634168001793035562840828_wp
        Xi(16) = 14.527827017141322474454157021029_wp
        Xi(17) = 18.266882671339681290176293337879_wp
        Xi(18) = 22.992617176036376688483264807417_wp
        Xi(19) = 28.941626422796236008799786176837_wp
        Xi(20) = 36.368046303409890446234342675780_wp
        Xi(21) = 45.491932781370753489935676583400_wp
        Xi(22) = 56.379688628733474167059602777385_wp
        Xi(23) = 68.713059125646557309674555824586_wp
        Xi(24) = 81.441938730602989139628622439204_wp
        Xi(25) = 92.501408562411169524131171471026_wp
        Xi(26) = 99.112194153775850354037846301480_wp

      case (13)
        ! ----- K = 13 , R = 2E2 -----

        RIni = 2.0e2_wp

        Omega(1) = 0.0064910169387615788873851173888807064571_wp
        Omega(2) = 0.0162170689188738721855351275838441615917_wp
        Omega(3) = 0.0291812342620532922049194799948845790993_wp
        Omega(4) = 0.0489100379253506474367969164562808259689_wp
        Omega(5) = 0.0811158535922647771682666810455497596877_wp
        Omega(6) = 0.1343271900794126477908128555749378563178_wp
        Omega(7) = 0.2212386426239065575236140276382812430711_wp
        Omega(8) = 0.3612893986938155437710091383296528988467_wp
        Omega(9) = 0.5848980597832797884171908420203322975794_wp
        Omega(10) = 0.9414285557062445442618077440233292918492_wp
        Omega(11) = 1.5190239900002864945234570814847074871068_wp
        Omega(12) = 2.5149049645269635100842259278763890506525_wp
        Omega(13) = 4.6611089868946715738390329519802435243037_wp

        Alpha(1) = 0.0025091447649231517130936230551216392648_wp
        Alpha(2) = 0.0136939543239405979498931687290963488124_wp
        Alpha(3) = 0.0360053496395654930710438813845630789956_wp
        Alpha(4) = 0.0742866820971900976034460808983261870253_wp
        Alpha(5) = 0.1379542477076401043700314835716014272293_wp
        Alpha(6) = 0.2434756239225348904388823365052552105681_wp
        Alpha(7) = 0.4177689435720809181128067993915564670715_wp
        Alpha(8) = 0.7035481042365874107340738274896096982047_wp
        Alpha(9) = 1.1680019135509342026717910512090270458430_wp
        Alpha(10) = 1.9172656578193901303871007901769019099447_wp
        Alpha(11) = 3.1235654399781725293078898930687614665658_wp
        Alpha(12) = 5.0909794073166716866889192072420655676979_wp
        Alpha(13) = 8.5067809666083261223648581683676184184151_wp

        Xi(1) = 1.006601834937339819044804789527_wp
        Xi(2) = 1.059993630307594630695321169522_wp
        Xi(3) = 1.169875904016283995516881089038_wp
        Xi(4) = 1.342575070422474490475922559440_wp
        Xi(5) = 1.588006775801091085263162805408_wp
        Xi(6) = 1.920390729415649058134558935240_wp
        Xi(7) = 2.359375501626099070565795168264_wp
        Xi(8) = 2.931586599079488560494824445080_wp
        Xi(9) = 3.672655956040545165500188495855_wp
        Xi(10) = 4.629914626029690094630858476776_wp
        Xi(11) = 5.866023886463970311508364652298_wp
        Xi(12) = 7.463891992454524606723231805461_wp
        Xi(13) = 9.533304684701023087867177707722_wp
        Xi(14) = 12.219814936428096325583825265682_wp
        Xi(15) = 15.716566939691546548295286722663_wp
        Xi(16) = 20.279731535473521229079318928257_wp
        Xi(17) = 26.247922766759155376731227349296_wp
        Xi(18) = 34.064729238468355203284243515327_wp
        Xi(19) = 44.299760950361843418049412335336_wp
        Xi(20) = 57.653781313149099905374095342836_wp
        Xi(21) = 74.909832932261869960222089304125_wp
        Xi(22) = 96.740265623925491222934169854852_wp
        Xi(23) = 123.190236423527929972288053050988_wp
        Xi(24) = 152.603923740088436064588073293180_wp
        Xi(25) = 180.140007970985797022822616497706_wp
        Xi(26) = 197.594393511301768248666910210432_wp

      case (14)
        ! ----- K = 13 , R = 3E2 -----

        RIni = 3.0e2_wp

        Omega(1) = 0.0046795081858433881002059097440476387320_wp
        Omega(2) = 0.0118427641824228194680434741482832894377_wp
        Omega(3) = 0.0218565387549393473972666286295196691469_wp
        Omega(4) = 0.0379717272342794108123476340410640261780_wp
        Omega(5) = 0.0654616436269458626932447528612435405648_wp
        Omega(6) = 0.1123190471227664696216583598009641775661_wp
        Omega(7) = 0.1908397923835510966958676701277219933672_wp
        Omega(8) = 0.3203673716733569136678235617088361664173_wp
        Omega(9) = 0.5316673394828265334440160760909321879808_wp
        Omega(10) = 0.8750363462747959466658115301740394897934_wp
        Omega(11) = 1.4400626713409013551943360642582092623343_wp
        Omega(12) = 2.4250786861151264221445056357140401814831_wp
        Omega(13) = 4.5588403252510033966324320964247363008326_wp

        Alpha(1) = 0.0018063447618946470180019119185742260125_wp
        Alpha(2) = 0.0099215980419140012612502451706808503928_wp
        Alpha(3) = 0.0264254867986646808014411166865009761295_wp
        Alpha(4) = 0.0556459478438246073363606172823558893015_wp
        Alpha(5) = 0.1061317262233611031827071498334635890615_wp
        Alpha(6) = 0.1929787217317038158727240254597390389790_wp
        Alpha(7) = 0.3412478358546339905418842697315540135605_wp
        Alpha(8) = 0.5915378195802623962707214588796489351807_wp
        Alpha(9) = 1.0090348395240912329979057759921090564603_wp
        Alpha(10) = 1.6985030889793068443376425347146607691684_wp
        Alpha(11) = 2.8320126081133543201502622554954768929747_wp
        Alpha(12) = 4.7148939360311128031130745164034578920109_wp
        Alpha(13) = 8.0352971415653418711666500229284793022089_wp

        Xi(1) = 1.007193626623011663040636876065_wp
        Xi(2) = 1.065432637443806315681767149517_wp
        Xi(3) = 1.185616548254826569481698572694_wp
        Xi(4) = 1.375324414382733756770892108445_wp
        Xi(5) = 1.646491689482819092775998937039_wp
        Xi(6) = 2.016373982421509058375955203424_wp
        Xi(7) = 2.509066895179997598586130180820_wp
        Xi(8) = 3.157602591261189096385683661161_wp
        Xi(9) = 4.006736484589320281858582095680_wp
        Xi(10) = 5.116720794204446622500254138188_wp
        Xi(11) = 6.568512853202026385932704721782_wp
        Xi(12) = 8.470969485648279558714923709317_wp
        Xi(13) = 10.970782909024342312298561097350_wp
        Xi(14) = 14.266175813728105064778772792522_wp
        Xi(15) = 18.625738347419756483352992759706_wp
        Xi(16) = 24.414140711996516648618626987854_wp
        Xi(17) = 32.126557836826970551008519905167_wp
        Xi(18) = 42.432935152670238524580392791563_wp
        Xi(19) = 56.229388980517890157112548266127_wp
        Xi(20) = 74.682125611916942660439833190367_wp
        Xi(21) = 99.215116053828261460934001547685_wp
        Xi(22) = 131.305011547351796738491991334286_wp
        Xi(23) = 171.754940608600189771992816645252_wp
        Xi(24) = 218.852558605650953330035157762268_wp
        Xi(25) = 265.146181392494579703855706043214_wp
        Xi(26) = 295.708296475482044995963804012717_wp

      case (15)
        ! ----- K = 13 , R = 4E2 -----

        RIni = 4.0e2_wp

        Omega(1) = 0.0037074547827871263050682546077649326399_wp
        Omega(2) = 0.0094812003352843043722770295309021371111_wp
        Omega(3) = 0.0178578875997744910917176385661830906315_wp
        Omega(4) = 0.0318905602746983744899078681150905723030_wp
        Omega(5) = 0.0565361202845229826009982362217121121262_wp
        Omega(6) = 0.0994286622560874676419909005964381520926_wp
        Omega(7) = 0.1725966329403311167291559802272615797847_wp
        Omega(8) = 0.2952899874667847870675189969613683160787_wp
        Omega(9) = 0.4984624791364017406245824626243035027073_wp
        Omega(10) = 0.8330134160758220932072731779172158894653_wp
        Omega(11) = 1.3895433034020467885942193286830104170804_wp
        Omega(12) = 2.3672713492607711707055306860070231778082_wp
        Omega(13) = 4.4930584436212646576649931695612849580357_wp

        Alpha(1) = 0.0014294954818313334632518094187425550246_wp
        Alpha(2) = 0.0078925616602339314342899416997167971743_wp
        Alpha(3) = 0.0212436561183783878532213605996870064985_wp
        Alpha(4) = 0.0454727229384414977013764182223187759746_wp
        Alpha(5) = 0.0885341543352852344698003528300489506364_wp
        Alpha(6) = 0.1645798034473368497496029508320170009483_wp
        Alpha(7) = 0.2974049919255520110382331328002791792642_wp
        Alpha(8) = 0.5261556981667115560842747090664062170617_wp
        Alpha(9) = 0.9146013961828088281594620334580270082370_wp
        Alpha(10) = 1.5664758893112251574367133066800761298509_wp
        Alpha(11) = 2.6536401907715668653520157960201686364599_wp
        Alpha(12) = 4.4822507877281657917567514459733502008021_wp
        Alpha(13) = 7.7412394984528697886722037324602752050851_wp

        Xi(1) = 1.007606096810241348856428122982_wp
        Xi(2) = 1.069229859498175030452574163498_wp
        Xi(3) = 1.196640759740549974678158751118_wp
        Xi(4) = 1.398364305325412257721175157688_wp
        Xi(5) = 1.687871479767406181107629314919_wp
        Xi(6) = 2.084750944590363017300607695326_wp
        Xi(7) = 2.616558091060674186983936451156_wp
        Xi(8) = 3.321371593068426360815040276897_wp
        Xi(9) = 4.251226063961023316807585903376_wp
        Xi(10) = 5.476831814846624884880049721758_wp
        Xi(11) = 7.094177762738691004999747047322_wp
        Xi(12) = 9.233784044797457095897497136860_wp
        Xi(13) = 12.073668324863166596665198238014_wp
        Xi(14) = 15.857533441545550142830200979205_wp
        Xi(15) = 20.920324540554137085143771024320_wp
        Xi(16) = 27.723976175274394985470483021572_wp
        Xi(17) = 36.906997911226415293173275244953_wp
        Xi(18) = 49.351401945981475661434645374470_wp
        Xi(19) = 66.267664988518538801698376516924_wp
        Xi(20) = 89.285622974993283866573801077493_wp
        Xi(21) = 120.498395074273122885843978480125_wp
        Xi(22) = 162.287654409046177672593103125109_wp
        Xi(23) = 216.460512662666428068680701812809_wp
        Xi(24) = 281.680285519143629169303011394732_wp
        Xi(25) = 348.193887706741630805318266084214_wp
        Xi(26) = 393.535359982477614654294839624527_wp

      case (16)
        ! ----- K = 13 , R = 5E2 -----

        RIni = 5.0e2_wp

        Omega(1) = 0.0030940770041673703710907771327609516732_wp
        Omega(2) = 0.0079837694611341389841908510599521875406_wp
        Omega(3) = 0.0152993134475317661556109508948220199187_wp
        Omega(4) = 0.0279405761711479469595415187424536362926_wp
        Omega(5) = 0.0506232403794067759958039652568029254098_wp
        Omega(6) = 0.0907199318854559077520121473936498546209_wp
        Omega(7) = 0.1600583567295715970698949046080450386853_wp
        Omega(8) = 0.2778002177833187805590202390515486285949_wp
        Omega(9) = 0.4750130531377467067613400153458513841542_wp
        Omega(10) = 0.8030283835320917704462227726391887472346_wp
        Omega(11) = 1.3532170717002870102092446091823774168006_wp
        Omega(12) = 2.3255281638367188130987056604581653118657_wp
        Omega(13) = 4.4455695860020917236157811380792281852337_wp

        Alpha(1) = 0.0011918361572482997446954992404148654384_wp
        Alpha(2) = 0.0066097900285615277493251359768045016807_wp
        Alpha(3) = 0.0179527056750687483633514927167107444461_wp
        Alpha(4) = 0.0389639196029066920288412394224319967861_wp
        Alpha(5) = 0.0771529416735844446117260561623929504549_wp
        Alpha(6) = 0.1459668488904268170570487778148205393336_wp
        Alpha(7) = 0.2682605794022303309740318938425573946915_wp
        Alpha(8) = 0.4820900184432984925141964238637370954166_wp
        Alpha(9) = 0.8501377564276385293733431791629584495240_wp
        Alpha(10) = 1.4753151598618290542925743369373492441810_wp
        Alpha(11) = 2.5292704379654323263749982064751975485706_wp
        Alpha(12) = 4.3187598299444165074120749459041235240875_wp
        Alpha(13) = 7.5333877438232410093249535876225309039000_wp

        Xi(1) = 1.007920571368887845890675858396_wp
        Xi(2) = 1.072128463819201193074279188089_wp
        Xi(3) = 1.205075672577440925236844093682_wp
        Xi(4) = 1.416050087437708750507346644465_wp
        Xi(5) = 1.719766148884946971526002656994_wp
        Xi(6) = 2.137716588041077865525035894834_wp
        Xi(7) = 2.700305475636761196045476651761_wp
        Xi(8) = 3.449804264385363344186385958778_wp
        Xi(9) = 4.444350907576345230919068196584_wp
        Xi(10) = 5.763517444790797477835453399564_wp
        Xi(11) = 7.516173032776385768555238442445_wp
        Xi(12) = 9.851612969935577901643697362744_wp
        Xi(13) = 12.975339413006578889904496509899_wp
        Xi(14) = 17.171471500771985724265178419046_wp
        Xi(15) = 22.834704158258824147759158407212_wp
        Xi(16) = 30.515822412028053598614940611355_wp
        Xi(17) = 40.986238310313833299591834702369_wp
        Xi(18) = 55.327968709077348798131001927914_wp
        Xi(19) = 75.053627355245311279785536129339_wp
        Xi(20) = 102.249274562275927230592387218167_wp
        Xi(21) = 139.687387242808906465119100914762_wp
        Xi(22) = 190.713438761254829259539356201003_wp
        Xi(23) = 258.307893451777710713823665855671_wp
        Xi(24) = 341.856235332755066941290422732891_wp
        Xi(25) = 429.646147143148018854308389791186_wp
        Xi(26) = 491.120109199436753161238655707166_wp

      case (17)
        ! ----- K = 13 , R = 6E2 -----

        RIni = 6.0e2_wp

        Omega(1) = 0.0026687973590316281074251490799259523001_wp
        Omega(2) = 0.0069412729280144503670891957015920414875_wp
        Omega(3) = 0.0135041197046104785608436511801233592678_wp
        Omega(4) = 0.0251330777276360453349312445991969866554_wp
        Omega(5) = 0.0463522796265119676730132383668525619669_wp
        Omega(6) = 0.0843323236642424205105413211047693522460_wp
        Omega(7) = 0.1507401942263835080632397578348502520384_wp
        Omega(8) = 0.2646556816965149062361936649345750538487_wp
        Omega(9) = 0.4572195938687172712098800292368494524453_wp
        Omega(10) = 0.7800936606602578585276802636272464042122_wp
        Omega(11) = 1.3252656634606378833161899333248356924742_wp
        Omega(12) = 2.2933018738760703215950886857399382279254_wp
        Omega(13) = 4.4089135030774377404724773121813541365555_wp

        Alpha(1) = 0.0010271376750990705541017466865053031633_wp
        Alpha(2) = 0.0057189693059005456997162498370551597660_wp
        Alpha(3) = 0.0156583921950699245940867316423394939306_wp
        Alpha(4) = 0.0343971045134781304859457676248446489353_wp
        Alpha(5) = 0.0690933050712323722092615858880737533809_wp
        Alpha(6) = 0.1326398987674319438683032817971607641994_wp
        Alpha(7) = 0.2471536181328989768683258854142792415587_wp
        Alpha(8) = 0.4498264599858223900072807382999329206541_wp
        Alpha(9) = 0.8024646171256123415582717228744513704441_wp
        Alpha(10) = 1.4072970218638971147363456171142104267346_wp
        Alpha(11) = 2.4357683331621138168431722759166291325528_wp
        Alpha(12) = 4.1950968917859868274194967696644198440481_wp
        Alpha(13) = 7.3754691998131300008367805531861449708231_wp

        Xi(1) = 1.008173416081267910449602298595_wp
        Xi(2) = 1.074461248734802413097415285659_wp
        Xi(3) = 1.211876394948510017070264499495_wp
        Xi(4) = 1.430345761103000806172165504382_wp
        Xi(5) = 1.745630193286268445722331232872_wp
        Xi(6) = 2.180834635848444269783791282791_wp
        Xi(7) = 2.768791552967651401059509530533_wp
        Xi(8) = 3.555371145354761311125485478968_wp
        Xi(9) = 4.603991437296975227332074265263_wp
        Xi(10) = 6.001949714536108498267458211117_wp
        Xi(11) = 7.869442974468143793195795687723_wp
        Xi(12) = 10.372432825988690112753443095528_wp
        Xi(13) = 13.741047839939295708774902315952_wp
        Xi(14) = 18.295992158406104599599473203853_wp
        Xi(15) = 24.486618912601472877396679983519_wp
        Xi(16) = 32.945895338515881759638981307603_wp
        Xi(17) = 44.569663490244743889873157982606_wp
        Xi(18) = 60.629619299190485648409776331391_wp
        Xi(19) = 82.929263074030677947712231201649_wp
        Xi(20) = 114.001709620792057747751258034441_wp
        Xi(21) = 157.300713576312651373334539073312_wp
        Xi(22) = 217.172180125454940727292019175820_wp
        Xi(23) = 297.893750716248039644984402229966_wp
        Xi(24) = 399.857819251229574419870260726384_wp
        Xi(25) = 509.736103384614762573656676636347_wp
        Xi(26) = 588.491495358906647139551893133103_wp

      case (18)
        ! ----- K = 13 , R = 7E2 -----

        RIni = 7.0e2_wp

        Omega(1) = 0.0023551074366808127160831978471125514041_wp
        Omega(2) = 0.0061695608640397718077784828399900973750_wp
        Omega(3) = 0.0121659998421845802122178217818948553486_wp
        Omega(4) = 0.0230164260498364710901680885157816192077_wp
        Omega(5) = 0.0430881697404923999683470491744996522243_wp
        Omega(6) = 0.0793891480732823163042827588342653655218_wp
        Omega(7) = 0.1434523233695349266059171355891521670856_wp
        Omega(8) = 0.2542818368570532217324501461641617083842_wp
        Omega(9) = 0.4430674773326116878773266866131663732631_wp
        Omega(10) = 0.7617342359687817124900069409854097557400_wp
        Omega(11) = 1.3027813128385978228441360871059373494063_wp
        Omega(12) = 2.2673081489216234296786567359305308855255_wp
        Omega(13) = 4.3793496679197264718347293310785062203649_wp

        Alpha(1) = 0.0009057063557368288924064534393173347970_wp
        Alpha(2) = 0.0050609742799303003855303185673508714615_wp
        Alpha(3) = 0.0139578927153902635739524530460176077185_wp
        Alpha(4) = 0.0309928908733394336304705967907735519873_wp
        Alpha(5) = 0.0630365269501992592267726293886465072092_wp
        Alpha(6) = 0.1225299401529001124219763786682424466790_wp
        Alpha(7) = 0.2309880248499910385210362809282891305429_wp
        Alpha(8) = 0.4248923236469510487814480598212085737941_wp
        Alpha(9) = 0.7653181374523062194497984023211500925754_wp
        Alpha(10) = 1.3539125860336706885615754014651201941888_wp
        Alpha(11) = 2.3619293562501908043029658079703381190484_wp
        Alpha(12) = 4.0969574805805687592666353147308200277621_wp
        Alpha(13) = 7.2496932242178477282142090576400050849770_wp

        Xi(1) = 1.008384022836860419267308863756_wp
        Xi(2) = 1.076405874089302964037931364949_wp
        Xi(3) = 1.217553969104368558756812390698_wp
        Xi(4) = 1.442305417051125772213981957037_wp
        Xi(5) = 1.767324775011994340790863478041_wp
        Xi(6) = 2.217116345408709288502049905567_wp
        Xi(7) = 2.826633150094971313939090640233_wp
        Xi(8) = 3.644905456456473356267183882906_wp
        Xi(9) = 4.740013615456810919853575247984_wp
        Xi(10) = 6.206126350713595783009646211781_wp
        Xi(11) = 8.173582957816336732985440516330_wp
        Xi(12) = 10.823385901403868655443196722388_wp
        Xi(13) = 14.408058428103994915922547725984_wp
        Xi(14) = 19.281856961649636606273561234559_wp
        Xi(15) = 25.944689943726666691084514759780_wp
        Xi(16) = 35.106255469330143126338583670076_wp
        Xi(17) = 47.779719367432811779478063485271_wp
        Xi(18) = 65.417561260579817718208150978398_wp
        Xi(19) = 90.103882605797534721681074643129_wp
        Xi(20) = 124.809231064963922691601894143787_wp
        Xi(21) = 173.666481733179374452591403610313_wp
        Xi(22) = 242.045296652620456165361240152833_wp
        Xi(23) = 335.614657594662747414071546359082_wp
        Xi(24) = 456.012862937083508219071603662087_wp
        Xi(25) = 588.628705898983448308037225160660_wp
        Xi(26) = 685.670160951395615878123379616227_wp

      case (19)
        ! ----- K = 13 , R = 8E2 -----

        RIni = 8.0e2_wp

        Omega(1) = 0.0021133505242918429081988530501834411268_wp
        Omega(2) = 0.0055729095031150907330734793534293691764_wp
        Omega(3) = 0.0111249328298707799984366777733268971673_wp
        Omega(4) = 0.0213527604993536956686666673788099757303_wp
        Omega(5) = 0.0404923224591177310863340191998682726648_wp
        Omega(6) = 0.0754164751756804483472204810035055544404_wp
        Omega(7) = 0.1375432795043878238857443241371925068961_wp
        Omega(8) = 0.2458070076870906891651925169073322763325_wp
        Omega(9) = 0.4314308537768518283660353074271398554629_wp
        Omega(10) = 0.7465562294807239240076318642636010736169_wp
        Omega(11) = 1.2841170511265791665455812675666891209403_wp
        Omega(12) = 2.2456810726269699539456997650077596517804_wp
        Omega(13) = 4.3547537938385463066048863822743442142382_wp

        Alpha(1) = 0.0008121562645573330148213797650535300532_wp
        Alpha(2) = 0.0045532336041415750128387625057096999015_wp
        Alpha(3) = 0.0126416380474307741316834503668420364164_wp
        Alpha(4) = 0.0283442039758663218032650604749633771462_wp
        Alpha(5) = 0.0582897240185137544883240898196952173294_wp
        Alpha(6) = 0.1145411279601758402303711761871518604039_wp
        Alpha(7) = 0.2181089137979318667159714731362818440630_wp
        Alpha(8) = 0.4048745086967141709516273401225916472868_wp
        Alpha(9) = 0.7352886271457931505819878570395786709923_wp
        Alpha(10) = 1.3104923188652284860803570731846434682666_wp
        Alpha(11) = 2.3015610953500072652887220581163774113520_wp
        Alpha(12) = 4.0163901817262514979366139211691688615247_wp
        Alpha(13) = 7.1461275739184389006394826360235583706526_wp

        Xi(1) = 1.008563939331447717375617179059_wp
        Xi(2) = 1.078068243250622171725688902555_wp
        Xi(3) = 1.222413496521834490792998706432_wp
        Xi(4) = 1.452559832886258677549923268302_wp
        Xi(5) = 1.785967260910051081072326406574_wp
        Xi(6) = 2.248377622169493433336440180170_wp
        Xi(7) = 2.876627602813417167301399124391_wp
        Xi(8) = 3.722566318997987185697093326375_wp
        Xi(9) = 4.858458525145758260567474540181_wp
        Xi(10) = 6.384669240241603082545301939277_wp
        Xi(11) = 8.440747879163621662616801000212_wp
        Xi(12) = 11.221427886655754447252364425225_wp
        Xi(13) = 14.999833696404702469522074448349_wp
        Xi(14) = 20.161279869226432774620416665812_wp
        Xi(15) = 27.252842878380206744631797377565_wp
        Xi(16) = 37.056368544093142066570001347259_wp
        Xi(17) = 50.696266365555572926127059218970_wp
        Xi(18) = 69.797994065457069884361729350530_wp
        Xi(19) = 96.716915210622292145559164566748_wp
        Xi(20) = 134.851540808634282886391275724236_wp
        Xi(21) = 189.009172050851384269676280780459_wp
        Xi(22) = 265.598537457188825450238311987050_wp
        Xi(23) = 371.753223499904204146782049633657_wp
        Xi(24) = 510.561540063745275486173369472453_wp
        Xi(25) = 666.447772441221866601690493325805_wp
        Xi(26) = 782.671756056125553668856298372702_wp

      case (20)
        ! ----- K = 13 , R = 9E2 -----

        RIni = 9.0e2_wp

        Omega(1) = 0.0019208272844967494482146201499085658160_wp
        Omega(2) = 0.0050963883462757398966150865823182547487_wp
        Omega(3) = 0.0102886814614984720077820270524304735460_wp
        Omega(4) = 0.0200039648510271715401272413573097175288_wp
        Omega(5) = 0.0383659899437067586735312062218294926197_wp
        Omega(6) = 0.0721328305621485951787006871138707708724_wp
        Omega(7) = 0.1326220401624135987212350223929391290767_wp
        Omega(8) = 0.2387032183270266254342074560557129814242_wp
        Omega(9) = 0.4216224758355891454573733528876644527372_wp
        Omega(10) = 0.7337033242537522846355574346333128232800_wp
        Omega(11) = 1.2682562982661819158238317162279429339833_wp
        Omega(12) = 2.2272659740951444588743374231043503641558_wp
        Omega(13) = 4.3338116476561864979025195943052040092880_wp

        Alpha(1) = 0.0007376832054758342712174149782226179051_wp
        Alpha(2) = 0.0041484372456243542216096397475671997412_wp
        Alpha(3) = 0.0115892775607377264963210837522833429603_wp
        Alpha(4) = 0.0262164386240689977399393731025134535173_wp
        Alpha(5) = 0.0544514124476461614242665462692905009590_wp
        Alpha(6) = 0.1080339617046875448831168287089354862474_wp
        Alpha(7) = 0.2075428669744146560887255309468812924933_wp
        Alpha(8) = 0.3883422725388195523353682819811538706745_wp
        Alpha(9) = 0.7103391909477947375952007480481853463061_wp
        Alpha(10) = 1.2742276377577958282587972105304174874618_wp
        Alpha(11) = 2.2509169842175276083674761995467861197540_wp
        Alpha(12) = 3.9485610579775714584684770391831420965900_wp
        Alpha(13) = 7.0587123560753475440092408188519357281621_wp

        Xi(1) = 1.008720586782332077381563184559_wp
        Xi(2) = 1.079516454195575416081555319980_wp
        Xi(3) = 1.226651562695841871639991504939_wp
        Xi(4) = 1.461516346230133188595357029982_wp
        Xi(5) = 1.802281412551816476932432797842_wp
        Xi(6) = 2.275798082616062384441260602053_wp
        Xi(7) = 2.920597345898910810000487225935_wp
        Xi(8) = 3.791077854083673642901150535600_wp
        Xi(9) = 4.963300876517859849985925269067_wp
        Xi(10) = 6.543286329812640416889180094984_wp
        Xi(11) = 8.679023991891225893804961977196_wp
        Xi(12) = 11.577915343461111109119032747117_wp
        Xi(13) = 15.532178628785418812371932340000_wp
        Xi(14) = 20.956117309346031146732869387961_wp
        Xi(15) = 28.441103191377953986196391866770_wp
        Xi(16) = 38.837191986032072562456951558829_wp
        Xi(17) = 53.374743426128173573452784239635_wp
        Xi(18) = 73.845296896195637199877825906924_wp
        Xi(19) = 102.866954405769408388560126610400_wp
        Xi(20) = 144.257018853382552717179443391160_wp
        Xi(21) = 203.491298142388639003108607994363_wp
        Xi(22) = 288.027035957418353795400633998725_wp
        Xi(23) = 406.520921201055780158650776456852_wp
        Xi(24) = 563.687833666227591566943999623618_wp
        Xi(25) = 743.290311601880343383985660921098_wp
        Xi(26) = 879.508623151254246841368455989141_wp

      case (21)
        ! ----- K = 13 , R = 1E3 -----

        RIni = 1.0e3_wp

        Omega(1) = 0.0017635661455619365334288679760671447738_wp
        Omega(2) = 0.0047061107563201549891152464417942713482_wp
        Omega(3) = 0.0096001161973894923613035215503863639697_wp
        Omega(4) = 0.0188839158491304791702029624786418615656_wp
        Omega(5) = 0.0365840348403184362808691822838147444941_wp
        Omega(6) = 0.0693591882562437035396351626970190551447_wp
        Omega(7) = 0.1284376651802928805053442481987069356819_wp
        Omega(8) = 0.2326290168465691475616768935097411485913_wp
        Omega(9) = 0.4131949440169984410803783597687655060327_wp
        Omega(10) = 0.7226149903814321418991060974601481348145_wp
        Omega(11) = 1.2545308979027300954126045162695390899898_wp
        Omega(12) = 2.2113022645733460638285683419468341526226_wp
        Omega(13) = 4.3156577180022858523933404484296261216514_wp

        Alpha(1) = 0.0006768696090223793169950347299677201329_wp
        Alpha(2) = 0.0038174398927510270668774617448468222669_wp
        Alpha(3) = 0.0107265164258034905514282509111922969680_wp
        Alpha(4) = 0.0244642995291056325264016178128168021999_wp
        Alpha(5) = 0.0512717184013090078658432448971415951178_wp
        Alpha(6) = 0.1026078910917859007657991325668245252700_wp
        Alpha(7) = 0.1986759226082185785823824158979888920840_wp
        Alpha(8) = 0.3743869444419926817869005014616234916502_wp
        Alpha(9) = 0.6891677591849794330789054364050372214479_wp
        Alpha(10) = 1.2433126215196539637170003445021393417846_wp
        Alpha(11) = 2.2075757403418671217555646624219889417873_wp
        Alpha(12) = 3.8903332120321473751832497267599819679162_wp
        Alpha(13) = 6.9835030321592778258625588971852948816377_wp

        Xi(1) = 1.008859011660317063041429619386_wp
        Xi(2) = 1.080796847166073275554876165216_wp
        Xi(3) = 1.230402093767569798930631985723_wp
        Xi(4) = 1.469453070518195829537687335797_wp
        Xi(5) = 1.816762394796934359185967833294_wp
        Xi(6) = 2.300186340854779247673544961827_wp
        Xi(7) = 2.959797888000143212565082317944_wp
        Xi(8) = 3.852322283237996409458597790731_wp
        Xi(9) = 5.057300123566479988900662645435_wp
        Xi(10) = 6.685953173104684348244941549666_wp
        Xi(11) = 8.894077959709410160192566419113_wp
        Xi(12) = 11.900838381148493603373106353160_wp
        Xi(13) = 16.016291320121754701566452183670_wp
        Xi(14) = 21.681943816052676662908815785613_wp
        Xi(15) = 29.530984248548985635421382234256_wp
        Xi(16) = 40.478264345147939216873478329717_wp
        Xi(17) = 55.855439018915906790319070296391_wp
        Xi(18) = 77.613940237170021503587680200553_wp
        Xi(19) = 108.626801119276814200365066653831_wp
        Xi(20) = 153.121296855100346712141856642120_wp
        Xi(21) = 217.235117672761294776995200095371_wp
        Xi(22) = 309.479592371193537730933442730930_wp
        Xi(23) = 440.081918394029455110505466564064_wp
        Xi(24) = 615.537513789520825013834581795891_wp
        Xi(25) = 819.234884579074029875567219960431_wp
        Xi(26) = 976.190808663600207428778787743795_wp

      case (22)
        ! ----- K = 13 , R = 2E3 -----

        RIni = 2.0e3_wp

        Omega(1) = 0.0010067371248319556046560592499350406115_wp
        Omega(2) = 0.0028099546056502035573047821150408087654_wp
        Omega(3) = 0.0061880278742393199293739909291889311760_wp
        Omega(4) = 0.0131638091493266705629816147767041645267_wp
        Omega(5) = 0.0272016646696397412172476566391554619884_wp
        Omega(6) = 0.0543757528682417203673980007211374854137_wp
        Omega(7) = 0.1053434646459840612667778481092994979917_wp
        Omega(8) = 0.1984814477535834189621125403513168805603_wp
        Omega(9) = 0.3650556257206909886226832429034772076193_wp
        Omega(10) = 0.6584198791878357653255086912036375679236_wp
        Omega(11) = 1.1742485462233790930331595037117153879080_wp
        Omega(12) = 2.1173756701945022221447723076437341660494_wp
        Omega(13) = 4.2088436157456099954293771148172709217761_wp

        Alpha(1) = 0.0003845271473448446583365740753402556296_wp
        Alpha(2) = 0.0022186008736336012455931992939234254081_wp
        Alpha(3) = 0.0065187824835520443497958998062963287268_wp
        Alpha(4) = 0.0157790140744405842979894601108004081880_wp
        Alpha(5) = 0.0351723648243899866533271943852545504683_wp
        Alpha(6) = 0.0745116209124147938107417081521344215389_wp
        Alpha(7) = 0.1517718419174551602483314374780754008043_wp
        Alpha(8) = 0.2991167658475697826970206294472376384874_wp
        Alpha(9) = 0.5729793346936465232291033067291152747202_wp
        Alpha(10) = 1.0710618053186080256191509141316942077538_wp
        Alpha(11) = 1.9629781296839430280477822865492498749518_wp
        Alpha(12) = 3.5583604501642892266669027101499978016363_wp
        Alpha(13) = 6.5515589465778985072894058383496940223267_wp

        Xi(1) = 1.009725744646025334498584680176_wp
        Xi(2) = 1.088827885747662834402781084631_wp
        Xi(3) = 1.254003513448803964580417769081_wp
        Xi(4) = 1.519624546493318705529286527511_wp
        Xi(5) = 1.908828870204436245743701006106_wp
        Xi(6) = 2.456323651603291157536831934571_wp
        Xi(7) = 3.212819016710725915581328693271_wp
        Xi(8) = 4.251287847349923359335788219937_wp
        Xi(9) = 5.675912111606291277510899506709_wp
        Xi(10) = 7.635320513387641534232275208893_wp
        Xi(11) = 10.342348180933737219686086028503_wp
        Xi(12) = 14.103626293658320066393141445360_wp
        Xi(13) = 19.364251300863817621678020231002_wp
        Xi(14) = 26.775834048268365186057371474249_wp
        Xi(15) = 37.301270145058441431856754988416_wp
        Xi(16) = 52.377800508079400344102483089159_wp
        Xi(17) = 74.173699586342912999648024907628_wp
        Xi(18) = 105.996097954768730919905550535987_wp
        Xi(19) = 152.943427648277461672332577791167_wp
        Xi(20) = 222.944161912956156118559292167447_wp
        Xi(21) = 328.359385158048549313569708374416_wp
        Xi(22) = 488.176055827352966737242567774047_wp
        Xi(23) = 729.657320211492764039373781770337_wp
        Xi(24) = 1082.705180999423388521485378532816_wp
        Xi(25) = 1539.941298467155270324191462805175_wp
        Xi(26) = 1935.888468434480756519278088489955_wp

      case (23)
        ! ----- K = 13 , R = 3E3 -----

        RIni = 3.0e3_wp

        Omega(1) = 0.0007266173762481528841575934788560298117_wp
        Omega(2) = 0.0020964974323044222425239772877869626200_wp
        Omega(3) = 0.0048581767156706806801208444347023318599_wp
        Omega(4) = 0.0108225994274088841993870465686689019691_wp
        Omega(5) = 0.0231860545777698102652144510754661510532_wp
        Omega(6) = 0.0477278172991908051166795416697663512196_wp
        Omega(7) = 0.0947879329870736345213739007520281631969_wp
        Omega(8) = 0.1824733193047515683886308085481786633864_wp
        Omega(9) = 0.3419906840673959335069175685672959730255_wp
        Omega(10) = 0.6270943201002195372512591775571166863301_wp
        Omega(11) = 1.1345214719922042758727676048380317297415_wp
        Omega(12) = 2.0705197705712899576818014635293252467818_wp
        Omega(13) = 4.1555506786128121082986297540884379486670_wp

        Alpha(1) = 0.0002765305534557809014453940609423385211_wp
        Alpha(2) = 0.0016230480146192057838232835936982667580_wp
        Alpha(3) = 0.0049241941747125575466193772491280133963_wp
        Alpha(4) = 0.0123926268756996454932121443440196095764_wp
        Alpha(5) = 0.0286746047845924961946768947523844772718_wp
        Alpha(6) = 0.0627750595034835794506881956333899097444_wp
        Alpha(7) = 0.1315527145078088977931144928912310376745_wp
        Alpha(8) = 0.2657519373445610193456949132961852910739_wp
        Alpha(9) = 0.5202041093762642991671016345911482403608_wp
        Alpha(10) = 0.9911579072113548517721122455004234552689_wp
        Alpha(11) = 1.8474989877676289366098538735627698770259_wp
        Alpha(12) = 3.3994398160829219373863657738255028561980_wp
        Alpha(13) = 6.3427254429993991389005736358797093998874_wp

        Xi(1) = 1.010193884925419554091617635461_wp
        Xi(2) = 1.093175713659357331281225811370_wp
        Xi(3) = 1.266836191359841957363580045026_wp
        Xi(4) = 1.547068851020137687183458874429_wp
        Xi(5) = 1.959573740784946239198299755557_wp
        Xi(6) = 2.543178477080369750832136710450_wp
        Xi(7) = 3.355089312592423162655633217533_wp
        Xi(8) = 4.478360657916579081611052615131_wp
        Xi(9) = 6.032735215475828557344512015348_wp
        Xi(10) = 8.190928315066882576221674039374_wp
        Xi(11) = 11.203262287884983346847311835859_wp
        Xi(12) = 15.435129731786143888333517892875_wp
        Xi(13) = 21.424480376846679417074392759446_wp
        Xi(14) = 29.970988825066483688189244993794_wp
        Xi(15) = 42.276101777386887150406158397686_wp
        Xi(16) = 60.165807576485730891191483493685_wp
        Xi(17) = 86.450085911160924119489568084873_wp
        Xi(18) = 125.510614369933223281872169962980_wp
        Xi(19) = 184.274284425331046091689479737852_wp
        Xi(20) = 273.836327285173797618789137686690_wp
        Xi(21) = 412.142291932858564174058102480558_wp
        Xi(22) = 628.174180857900458452292014044360_wp
        Xi(23) = 966.981853512719977949974037301217_wp
        Xi(24) = 1487.459781101271937719943139200041_wp
        Xi(25) = 2208.800660325778965731302605490782_wp
        Xi(26) = 2885.097496793069043574675447416666_wp

      case (24)
        ! ----- K = 13 , R = 4E3 -----

        RIni = 4.0e3_wp

        Omega(1) = 0.0005772981384290036624670431752038429751_wp
        Omega(2) = 0.0017117321655301405442927299010991148975_wp
        Omega(3) = 0.0041224097445417860749515386042940434663_wp
        Omega(4) = 0.0094856813604299347655451077679367166695_wp
        Omega(5) = 0.0208318985608728884816644990815581728327_wp
        Omega(6) = 0.0437488442607272109449820674670350584279_wp
        Omega(7) = 0.0883611409328008595820522393105012071146_wp
        Omega(8) = 0.1725833026893584265797250421048403268287_wp
        Omega(9) = 0.3275606215791396407324437972663488949365_wp
        Omega(10) = 0.6072881207626289627515489555342753646983_wp
        Omega(11) = 1.1091987143511127322055218047403002401552_wp
        Omega(12) = 2.0405109967818031381416726821598217611609_wp
        Omega(13) = 4.1214134076962477036890075687836088036420_wp

        Alpha(1) = 0.0002190369434955462190035078753869678125_wp
        Alpha(2) = 0.0013041638214468583454602062556004771210_wp
        Alpha(3) = 0.0040594961584673383144200413397140181360_wp
        Alpha(4) = 0.0105190663597531257523630052186369709943_wp
        Alpha(5) = 0.0249976696275790752605172303016378698715_wp
        Alpha(6) = 0.0559909911522194053871775108743857884974_wp
        Alpha(7) = 0.1196433236803296841193190551411174027407_wp
        Alpha(8) = 0.2457744585688548566063766839806525865697_wp
        Alpha(9) = 0.4881524088078709740592432686501211946961_wp
        Alpha(10) = 0.9420361988247535020048195608666219413863_wp
        Alpha(11) = 1.7757837240382841738437630363378616493719_wp
        Alpha(12) = 3.2999574598283268581262739349568846591865_wp
        Alpha(13) = 6.2112597218473327084775625905876950128004_wp

        Xi(1) = 1.010507171655304307824468912358_wp
        Xi(2) = 1.096089317567576409528234959723_wp
        Xi(3) = 1.275457771083344649999148645581_wp
        Xi(4) = 1.565572209208788321383419572808_wp
        Xi(5) = 1.993939949609717315263772430267_wp
        Xi(6) = 2.602319097392235636677371157788_wp
        Xi(7) = 3.452578553195592656619231086701_wp
        Xi(8) = 4.635076890885197571732373456044_wp
        Xi(9) = 6.280951878486133464589424857394_wp
        Xi(10) = 8.580757164513207110694004153828_wp
        Xi(11) = 11.812928826767320134721872548766_wp
        Xi(12) = 16.387505908343608500635224523023_wp
        Xi(13) = 22.913986916926745118078256879812_wp
        Xi(14) = 32.307870360351605198978575117508_wp
        Xi(15) = 45.960257704975180259587830278178_wp
        Xi(16) = 66.011654195450042745263452559357_wp
        Xi(17) = 95.801090043883571842342483648736_wp
        Xi(18) = 140.614788668700952972701045950998_wp
        Xi(19) = 208.954038513489973927228504635423_wp
        Xi(20) = 314.711822968892776908322872486679_wp
        Xi(21) = 480.915486201436502888828528057275_wp
        Xi(22) = 745.985517155986802229072907266527_wp
        Xi(23) = 1172.664620532656124862747049064637_wp
        Xi(24) = 1851.445761013825762719697820557485_wp
        Xi(25) = 2839.909796921835096306097057095030_wp
        Xi(26) = 3825.735247723769567373963695899874_wp

      case (25)
        ! ----- K = 13 , R = 5E3 -----

        RIni = 5.0e3_wp

        Omega(1) = 0.0004834183698374707752262757922247574105_wp
        Omega(2) = 0.0014675492248497519941626034676619705355_wp
        Omega(3) = 0.0036457480254511605207782252470460893079_wp
        Omega(4) = 0.0085991534896893102134834345300073987772_wp
        Omega(5) = 0.0192417810195062726691189129019088355221_wp
        Omega(6) = 0.0410222078190040722932101485004308116800_wp
        Omega(7) = 0.0839043742330625100714195686857355127586_wp
        Omega(8) = 0.1656549505050655346304446305372159997660_wp
        Omega(9) = 0.3173632327682835929877694369771745641629_wp
        Omega(10) = 0.5931886289713927987946734399216097699536_wp
        Omega(11) = 1.0910702058875274777192954100968336206279_wp
        Omega(12) = 2.0189563667345757479870682615796795289498_wp
        Omega(13) = 4.0968895055939679610869375014914339772076_wp

        Alpha(1) = 0.0001829255424868854810194156694383785244_wp
        Alpha(2) = 0.0011029602809106096935085972524093200775_wp
        Alpha(3) = 0.0035082040776387238560814207932736508155_wp
        Alpha(4) = 0.0093055582700188832209547366153268299627_wp
        Alpha(5) = 0.0225756890794564490266432572190824501490_wp
        Alpha(6) = 0.0514532208102668484849516201268260218171_wp
        Alpha(7) = 0.1115701158044498115593629559982868926227_wp
        Alpha(8) = 0.2320750581708474017403678879567507920001_wp
        Alpha(9) = 0.4659542555149446310105640356447054273303_wp
        Alpha(10) = 0.9077269511701410156984982935401973236367_wp
        Alpha(11) = 1.7253408354432977452660308292919921768771_wp
        Alpha(12) = 3.2295976939344447805002591422862678882666_wp
        Alpha(13) = 6.1179182378765278489674006578979970072396_wp

        Xi(1) = 1.010738974857806489449606190956_wp
        Xi(2) = 1.098247200440898018098702382250_wp
        Xi(3) = 1.281854567869962779197434699086_wp
        Xi(4) = 1.579334652848051773958541754705_wp
        Xi(5) = 2.019580321691430035979664125989_wp
        Xi(6) = 2.646610736278654934594892211841_wp
        Xi(7) = 3.525915057422893787502857221305_wp
        Xi(8) = 4.753560317982870825001540104893_wp
        Xi(9) = 6.469652372559513124877089706999_wp
        Xi(10) = 8.878906761062757563816860706396_wp
        Xi(11) = 12.282266232036800817327415469293_wp
        Xi(12) = 17.125848188178688999080967469979_wp
        Xi(13) = 24.077550125482101208915231005392_wp
        Xi(14) = 34.148424906016304918032755466584_wp
        Xi(15) = 48.887862478507531315691236883936_wp
        Xi(16) = 70.702197987010334083446405628592_wp
        Xi(17) = 103.383810322159083130988399545913_wp
        Xi(18) = 153.005728623073744595894929432234_wp
        Xi(19) = 229.461785443632748524023234892866_wp
        Xi(20) = 349.166407012105063090157131000524_wp
        Xi(21) = 539.826624204983897770127043713728_wp
        Xi(22) = 848.788293460028481829393598445677_wp
        Xi(23) = 1356.143188866657152247263695699075_wp
        Xi(24) = 2185.311920916363763067735703771177_wp
        Xi(25) = 3440.872267032228991823217256751377_wp
        Xi(26) = 4758.866855376139633815313345621689_wp

      case (26)
        ! ----- K = 13 , R = 6E3 -----

        RIni = 6.0e3_wp

        Omega(1) = 0.0004184727948542464259221414018489332287_wp
        Omega(2) = 0.0012972732263054492916603540686795725811_wp
        Omega(3) = 0.0033075235899535997919634227638907564284_wp
        Omega(4) = 0.0079584618577688491727290605488864816408_wp
        Omega(5) = 0.0180763356026030094760514380786142751845_wp
        Omega(6) = 0.0390017527131794468388801579161739851997_wp
        Omega(7) = 0.0805718846011344139821795835854079115279_wp
        Omega(8) = 0.1604343330103549718852664479618042037146_wp
        Omega(9) = 0.3096284138665091020304904328996897788784_wp
        Omega(10) = 0.5824345405064120297111175672544192138957_wp
        Omega(11) = 1.0771838414761236184984463659652931255550_wp
        Omega(12) = 2.0024038613880578551390554409294963988941_wp
        Omega(13) = 4.0780542604105885617221671779475400398951_wp

        Alpha(1) = 0.0001579639475325221557856034346471914287_wp
        Alpha(2) = 0.0009633488427175769010574386298773053028_wp
        Alpha(3) = 0.0031222338920011592445869299822930331700_wp
        Alpha(4) = 0.0084447683196101985288653372398182117564_wp
        Alpha(5) = 0.0208344862398892867293701729528021537874_wp
        Alpha(6) = 0.0481516957484454969102964129457067343765_wp
        Alpha(7) = 0.1056356213489921477369248327404793741380_wp
        Alpha(8) = 0.2219158164417327056515405470094037809758_wp
        Alpha(9) = 0.4493679596948363069274911402173344754374_wp
        Alpha(10) = 0.8819265141437007679081974687029799042648_wp
        Alpha(11) = 1.6872058594615989730869376472810472478159_wp
        Alpha(12) = 3.1761840167015348132453084195603310035949_wp
        Alpha(13) = 6.0468509031626904615293327527325573100825_wp

        Xi(1) = 1.010920946649155907691268463378_wp
        Xi(2) = 1.099942443889555764068369225761_wp
        Xi(3) = 1.286886753704524820416527086664_wp
        Xi(4) = 1.590181497923492496236741455817_wp
        Xi(5) = 2.039836727934589126953399129683_wp
        Xi(6) = 2.681703103067590013806886783954_wp
        Xi(7) = 3.584216057170782720412566257906_wp
        Xi(8) = 4.848110980743221974827161790955_wp
        Xi(9) = 6.620874502625305116267767635563_wp
        Xi(10) = 9.118938881467079838986922180055_wp
        Xi(11) = 12.662000733272805380776049322122_wp
        Xi(12) = 17.726453775662492189091334893192_wp
        Xi(13) = 25.029577491848180788650002348561_wp
        Xi(14) = 35.663907097702922704279826504603_wp
        Xi(15) = 51.315002423162425583547108232096_wp
        Xi(16) = 74.620138063550451049121825519705_wp
        Xi(17) = 109.769787631884041319285660875948_wp
        Xi(18) = 163.536101042984578043948395986718_wp
        Xi(19) = 247.066473507943357573735099208534_wp
        Xi(20) = 379.078988115023749611021131045163_wp
        Xi(21) = 591.629447221732486350553870124713_wp
        Xi(22) = 940.530548868877992685355593494023_wp
        Xi(23) = 1522.796122253376686295212039112812_wp
        Xi(24) = 2495.439757517349457005551016663958_wp
        Xi(25) = 4016.574435451630723736826666936395_wp
        Xi(26) = 5685.191791248119010493411451534485_wp

      case (27)
        ! ----- K = 13 , R = 7E3 -----

        RIni = 7.0e3_wp

        Omega(1) = 0.0003706377943815102771234427431927896390_wp
        Omega(2) = 0.0011709747407894105638609296825974892542_wp
        Omega(3) = 0.0030528344477778092929221660702804319598_wp
        Omega(4) = 0.0074687037676975781414815910332072323996_wp
        Omega(5) = 0.0171753370381771652815008260065110512471_wp
        Omega(6) = 0.0374259815736839124110510099163739994310_wp
        Omega(7) = 0.0779539684179911716450906387920749551768_wp
        Omega(8) = 0.1563078510967816640077907366923604115527_wp
        Omega(9) = 0.3034823620318470197650042446957385777750_wp
        Omega(10) = 0.5738515410841367624301435912403945849292_wp
        Omega(11) = 1.0660630760791806437943682328572947426437_wp
        Omega(12) = 1.9891211408960819676710121695073496539408_wp
        Omega(13) = 4.0629379599812607197321934204836679782602_wp

        Alpha(1) = 0.0001395911103949207554129102029963870046_wp
        Alpha(2) = 0.0008602478231696831677067798698540362912_wp
        Alpha(3) = 0.0028349335336120280063744556083016323811_wp
        Alpha(4) = 0.0077968208961486511058163503384149661812_wp
        Alpha(5) = 0.0195091641305399062398872678594785590178_wp
        Alpha(6) = 0.0456141150523087043678921882056132020011_wp
        Alpha(7) = 0.1010362835063562390359743342616027916847_wp
        Alpha(8) = 0.2139863759050597113292851297883245820231_wp
        Alpha(9) = 0.4363438111546551760502443001454864202060_wp
        Alpha(10) = 0.8615632167206826681543282986375231757847_wp
        Alpha(11) = 1.6569797098202836513770180859594916000788_wp
        Alpha(12) = 3.1337076649686691487417533608095254749060_wp
        Alpha(13) = 5.9902047769526498423135751902890433484572_wp

        Xi(1) = 1.011069519030782586544066348200_wp
        Xi(2) = 1.101327355506823330727472765034_wp
        Xi(3) = 1.291002214027701274585632762459_wp
        Xi(4) = 1.599065635925675331612157148431_wp
        Xi(5) = 2.056459251051908898632117339567_wp
        Xi(6) = 2.710566638540833956814643523892_wp
        Xi(7) = 3.632298569107429986758256434953_wp
        Xi(8) = 4.926328628237788356120097077451_wp
        Xi(9) = 6.746394341811706720727498387546_wp
        Xi(10) = 9.318910594218486549526747619865_wp
        Xi(11) = 12.979626574112289267258490976786_wp
        Xi(12) = 18.231006536335286163583169205538_wp
        Xi(13) = 25.833110196498794945349497531595_wp
        Xi(14) = 36.949525543467812566111874161834_wp
        Xi(15) = 53.385468217286557508771682734050_wp
        Xi(16) = 77.982689877503673842040754493610_wp
        Xi(17) = 115.287289250844873882140273479990_wp
        Xi(18) = 172.701987284767743080493573870626_wp
        Xi(19) = 262.517016506791535490616951165066_wp
        Xi(20) = 405.576384631176343198433187353658_wp
        Xi(21) = 638.005573073923045068767123666476_wp
        Xi(22) = 1023.675579114166468042146362904532_wp
        Xi(23) = 1676.071938308362667457984684915573_wp
        Xi(24) = 2786.083260147426660102709661259723_wp
        Xi(25) = 4570.464376150974832757611920897034_wp
        Xi(26) = 6605.212628370586382064999497742974_wp

      case (28)
        ! ----- K = 13 , R = 8E3 -----

        RIni = 8.0e3_wp

        Omega(1) = 0.0003338078810012542241301752767718147297_wp
        Omega(2) = 0.0010731162787732948328650156013738814487_wp
        Omega(3) = 0.0028528419239137359014562672328632619845_wp
        Omega(4) = 0.0070792216098078710114682905469446483337_wp
        Omega(5) = 0.0164520713096193200454373959870646437764_wp
        Omega(6) = 0.0361518092567141136475328353633296174507_wp
        Omega(7) = 0.0758243845754235437842788979112551572825_wp
        Omega(8) = 0.1529339780982985849710521775168814428980_wp
        Omega(9) = 0.2984353231132846150487972214371268364630_wp
        Omega(10) = 0.5667775223001799543716926199632410998674_wp
        Omega(11) = 1.0568715828439863723851518950702654819906_wp
        Omega(12) = 1.9781243638999431445598189971413205512363_wp
        Omega(13) = 4.0504218514964760687602218780511975637637_wp

        Alpha(1) = 0.0001254533213413407806652386661250275779_wp
        Alpha(2) = 0.0007806785450082557440315181957954382419_wp
        Alpha(3) = 0.0026116176695613647849582669062742934507_wp
        Alpha(4) = 0.0072882286878179625655993782891439725091_wp
        Alpha(5) = 0.0184589798304450120202980243147572458895_wp
        Alpha(6) = 0.0435867985980898777018804901572934795695_wp
        Alpha(7) = 0.0973362507445847350491291601060073190865_wp
        Alpha(8) = 0.2075698167314758098343872166902102094355_wp
        Alpha(9) = 0.4257518170980607442161446074546660156557_wp
        Alpha(10) = 0.8449324268358560348162432374063968154587_wp
        Alpha(11) = 1.6322074462041672078278092827829937050410_wp
        Alpha(12) = 3.0988005415161087707093257037627154204529_wp
        Alpha(13) = 5.9435640977043874238865128756259537112783_wp

        Xi(1) = 1.011194271189963009511050828770_wp
        Xi(2) = 1.102490805240993707340632634661_wp
        Xi(3) = 1.294462592735715565834576923354_wp
        Xi(4) = 1.606544960882145392944836403970_wp
        Xi(5) = 2.070475367190860802339355539736_wp
        Xi(6) = 2.734950809647740910864749386278_wp
        Xi(7) = 3.673009910906543813149602306645_wp
        Xi(8) = 4.992723021119104396658289202193_wp
        Xi(9) = 6.853239939403569421123851856947_wp
        Xi(10) = 9.489649798672527634089968184838_wp
        Xi(11) = 13.251721282740746303031864616884_wp
        Xi(12) = 18.664785660215381921100963324989_wp
        Xi(13) = 26.526631345696071123327342000664_wp
        Xi(14) = 38.063867020101055941433410012564_wp
        Xi(15) = 55.188416375678922345948684835548_wp
        Xi(16) = 80.925743847941144552826653324473_wp
        Xi(17) = 120.143624167390607002103841693952_wp
        Xi(18) = 180.819948808945976589934723222086_wp
        Xi(19) = 276.296939379611788206725364602789_wp
        Xi(20) = 429.395624268682294860655090928958_wp
        Xi(21) = 680.071425164502705251745595660395_wp
        Xi(22) = 1099.888095936666530638703420663660_wp
        Xi(23) = 1818.354039085222022542609465745045_wp
        Xi(24) = 3060.280942425657703997998737577291_wp
        Xi(25) = 5105.124648034188095113705685434979_wp
        Xi(26) = 7519.310923709670903658519591772347_wp

      case (29)
        ! ----- K = 13 , R = 9E3 -----

        RIni = 9.0e3_wp

        Omega(1) = 0.0003044979986515639643271668334694249336_wp
        Omega(2) = 0.0009947872650128499371167273019472832019_wp
        Omega(3) = 0.0026908314022329150792039797546531532113_wp
        Omega(4) = 0.0067602481509608796317302924997610924507_wp
        Omega(5) = 0.0158549983554854806457170895636621388292_wp
        Omega(6) = 0.0350934163362851217193838764679680863878_wp
        Omega(7) = 0.0740464043548204659236423343104949168492_wp
        Omega(8) = 0.1501049592454127755924118969854674787712_wp
        Omega(9) = 0.2941877084622041017272428553575025489408_wp
        Omega(10) = 0.5608055691529981708954612218409607748981_wp
        Omega(11) = 1.0490934946574624696659719380598119187198_wp
        Omega(12) = 1.9688053441501122312702262706629596777930_wp
        Omega(13) = 4.0398142712789524011451902651970158331096_wp

        Alpha(1) = 0.0001142078282379464811224502862054778346_wp
        Alpha(2) = 0.0007172191780088991689513650538458089390_wp
        Alpha(3) = 0.0024323501465652189386784643148481377395_wp
        Alpha(4) = 0.0068763947394231850905671140933989038402_wp
        Alpha(5) = 0.0176015596004510661499911205324364615876_wp
        Alpha(6) = 0.0419199056873468953450891961998170387460_wp
        Alpha(7) = 0.0942759616337734391531286495813368730978_wp
        Alpha(8) = 0.2022361057922115201056700514725683603956_wp
        Alpha(9) = 0.4169099038558705398640601841320219023146_wp
        Alpha(10) = 0.8309996676277631378801889761032128944862_wp
        Alpha(11) = 1.6113925491144701489546961492571597318602_wp
        Alpha(12) = 3.0694021532196554213033079516748102832935_wp
        Alpha(13) = 5.9042205649169941920374748978161960621946_wp

        Xi(1) = 1.011301249609793416798196596940_wp
        Xi(2) = 1.103488913440093922109161428402_wp
        Xi(3) = 1.297433455503520915194463902775_wp
        Xi(4) = 1.612973024208289563020089774081_wp
        Xi(5) = 2.082537319964158808397597377748_wp
        Xi(6) = 2.755969232365370307673638206047_wp
        Xi(7) = 3.708168547919199563040609790576_wp
        Xi(8) = 5.050184365461531847338316758567_wp
        Xi(9) = 6.945930525094032291413131963154_wp
        Xi(10) = 9.638154665146328028101085916290_wp
        Xi(11) = 13.489050539430503977342268306927_wp
        Xi(12) = 19.044302644665340680518728078141_wp
        Xi(13) = 27.135426454652572650361186390455_wp
        Xi(14) = 39.045608645104961920840880296169_wp
        Xi(15) = 56.783157011132844863704471904953_wp
        Xi(16) = 83.540273286663584033362450753657_wp
        Xi(17) = 124.478716923536892970114653422797_wp
        Xi(18) = 188.105625063055856172056223840627_wp
        Xi(19) = 288.738723310025514223742248987037_wp
        Xi(20) = 451.049222218557127439542142610662_wp
        Xi(21) = 718.613432280905455484809607469288_wp
        Xi(22) = 1170.358028656877776629841036992730_wp
        Xi(23) = 1951.379160595497792130004199862015_wp
        Xi(24) = 3320.304950255201992259657117756433_wp
        Xi(25) = 5622.572944123117865355965250273584_wp
        Xi(26) = 8427.788308099427949215964872564655_wp

      case (30)
        ! ----- K = 13 , R = 1E4 -----

        RIni = 1.0e4_wp

        Omega(1) = 0.0002805676165532984518155677907628314793_wp
        Omega(2) = 0.0009304915861452454389846289861329409199_wp
        Omega(3) = 0.0025563936765912246068727965708777816189_wp
        Omega(4) = 0.0064930225459359853644045396710987594346_wp
        Omega(5) = 0.0153513195022753170445123046573832059636_wp
        Omega(6) = 0.0341957745875822269461215881930504778552_wp
        Omega(7) = 0.0725318011194240101511339531559219295787_wp
        Omega(8) = 0.1476859942928149881200428766847565498210_wp
        Omega(9) = 0.2905441729112577614005819698350663315978_wp
        Omega(10) = 0.5556692375929922942615762915252730635984_wp
        Omega(11) = 1.0423899529883427146929528173124879231182_wp
        Omega(12) = 1.9607638827744524425474720308315568217949_wp
        Omega(13) = 4.0306601210273864796182485825681851565605_wp

        Alpha(1) = 0.0001050303259562613398418802709865550737_wp
        Alpha(2) = 0.0006653037355264198210729591502401308878_wp
        Alpha(3) = 0.0022848087471415481963090890601617011235_wp
        Alpha(4) = 0.0065347902968938500829803239441928575104_wp
        Alpha(5) = 0.0168851672754504879700796068572876684755_wp
        Alpha(6) = 0.0405185835722869734366836588779214878286_wp
        Alpha(7) = 0.0916899642061235559744062847964407581003_wp
        Alpha(8) = 0.1977094597844340329815268977875497569130_wp
        Alpha(9) = 0.4093783172749800212085469208744825664326_wp
        Alpha(10) = 0.8190948779971189729672692569906189419271_wp
        Alpha(11) = 1.5935618858617478106045814434388319114078_wp
        Alpha(12) = 3.0441685261321626066507300967245441825071_wp
        Alpha(13) = 5.8704038773647556391857560420532990974607_wp

        Xi(1) = 1.011394502750584783925397180937_wp
        Xi(2) = 1.104359278856419517966108712059_wp
        Xi(3) = 1.300025824648590792929729598182_wp
        Xi(4) = 1.618587101892998377492861872273_wp
        Xi(5) = 2.093083986917413600670548334826_wp
        Xi(6) = 2.774373089345218839317508408016_wp
        Xi(7) = 3.739004345991395365962628449807_wp
        Xi(8) = 5.100674537215651497233659394936_wp
        Xi(9) = 7.027542888986806526429446106974_wp
        Xi(10) = 9.769204956923797807805120996250_wp
        Xi(11) = 13.698998182122122726882273990867_wp
        Xi(12) = 19.380928351820645237635343249849_wp
        Xi(13) = 27.676970013468871121414816016681_wp
        Xi(14) = 39.921665490508533677416203033772_wp
        Xi(15) = 58.211130279764656602486949665831_wp
        Xi(16) = 85.890337944122038482674952319940_wp
        Xi(17) = 128.391748565267824450097933208781_wp
        Xi(18) = 194.712981765983032325650903260339_wp
        Xi(19) = 300.082023274835592047127263981565_wp
        Xi(20) = 470.910027685282373155484236804114_wp
        Xi(21) = 754.209698745703329469947107099870_wp
        Xi(22) = 1235.971801574550370084182304708520_wp
        Xi(23) = 2076.462566934585095124177200887061_wp
        Xi(24) = 3567.919536174050756160980313325126_wp
        Xi(25) = 6124.428667863334515342188524300582_wp
        Xi(26) = 9330.891138495153310117302680737339_wp

      case (31)
        ! ----- K = 13 , R = 4E5 (Best : R = 373800) -----

        RIni = 4.0e5_wp

        Omega(1) = 0.0000330785848121602154387056886249976126_wp
        Omega(2) = 0.0002173321916008653986926517817884589590_wp
        Omega(3) = 0.0008896788976872138107598869801936891655_wp
        Omega(4) = 0.0028921756053065667819714998094254854166_wp
        Omega(5) = 0.0081422349627770869181416228850534100303_wp
        Omega(6) = 0.0207224876586965230132556534461962804983_wp
        Omega(7) = 0.0488864342146971603968207738949436347298_wp
        Omega(8) = 0.1086418495611125869620928671710480273305_wp
        Omega(9) = 0.2300332300492383233344846973911224097264_wp
        Omega(10) = 0.4682950733039983514044771184536841701629_wp
        Omega(11) = 0.9262055359666433552523825967206505538343_wp
        Omega(12) = 1.8198087510110939301862312267310528568487_wp
        Omega(13) = 3.8700490617962436596440761737980551515648_wp

        Alpha(1) = 0.0000103473916611944270948840590255708282_wp
        Alpha(2) = 0.0001138316914487578993839338792673759188_wp
        Alpha(3) = 0.0006024601498553105686272644218436783659_wp
        Alpha(4) = 0.0023252777266436541401638786073433062818_wp
        Alpha(5) = 0.0074464345169693261373645629380008115206_wp
        Alpha(6) = 0.0210088043449285925962484672738228574929_wp
        Alpha(7) = 0.0540032192609735148525727538998397747605_wp
        Alpha(8) = 0.1291621153315104471087301774012701116590_wp
        Alpha(9) = 0.2915643750527972055479851432036042524487_wp
        Alpha(10) = 0.6276946181536831419200961335036481614225_wp
        Alpha(11) = 1.3003150067055842798151993333632958638191_wp
        Alpha(12) = 2.6217712688276107583541796319792638314539_wp
        Alpha(13) = 5.2973357440769386740922475276249770104187_wp

        Xi(1) = 1.013181302065140132755323432168_wp
        Xi(2) = 1.121092899838553857147852022003_wp
        Xi(3) = 1.350178171338921877860789200199_wp
        Xi(4) = 1.728133062090680845384466080006_wp
        Xi(5) = 2.301135413611220813115756933520_wp
        Xi(6) = 3.142313650650617387529578428129_wp
        Xi(7) = 4.365283442146903240734662210443_wp
        Xi(8) = 6.144672947757623488775069464651_wp
        Xi(9) = 8.749086754792229351194998709929_wp
        Xi(10) = 12.595210582443065289517947213938_wp
        Xi(11) = 18.337551132071687531172043961192_wp
        Xi(12) = 27.019638911679229103732224892553_wp
        Xi(13) = 40.333934960276501911075053286027_wp
        Xi(14) = 61.079107103866465810354524634818_wp
        Xi(15) = 93.984385911458973639209801831385_wp
        Xi(16) = 147.243788330935086616779727819448_wp
        Xi(17) = 235.457674684953069135784353704821_wp
        Xi(18) = 385.524479321568202178083240028172_wp
        Xi(19) = 648.938204148598712983631031647747_wp
        Xi(20) = 1128.970068389656095986239847661636_wp
        Xi(21) = 2045.142077976131562699890764633892_wp
        Xi(22) = 3897.707449713617865683090712991543_wp
        Xi(23) = 7947.823233342667208045639881675015_wp
        Xi(24) = 17814.961325541782590775596872845199_wp
        Xi(25) = 46249.942302022233448610677442047745_wp
        Xi(26) = 162585.334114312755389164522057399154_wp

    end select

  case (14)
    ! ===== K = 14 =====

    select case (InitR)

      case default !(1:3)
        ! ----- K = 14 , R = 1E1 -----

        RIni = 1.0e1_wp

        Omega(1) = 0.0611672710980397459221895393316481559509_wp
        Omega(2) = 0.1445085337560793091400733814677614930133_wp
        Omega(3) = 0.2333087043102459606655702320510137326437_wp
        Omega(4) = 0.3312553013320872669935073656644064499233_wp
        Omega(5) = 0.4432467435331534960001129835882593965835_wp
        Omega(6) = 0.5759071772555116923647880455838787838729_wp
        Omega(7) = 0.7383746794403462614239316841313609529607_wp
        Omega(8) = 0.9433758854644169632359430865964355916731_wp
        Omega(9) = 1.2089176341155743880147882540754267211014_wp
        Omega(10) = 1.5616549458497541315966580799390328593290_wp
        Omega(11) = 2.0450074532252133611975875027155780117027_wp
        Omega(12) = 2.7419850943350311993709778057848325261148_wp
        Omega(13) = 3.8566755067690577972808196882681386341574_wp
        Omega(14) = 6.1981813314547232786344554966717623756267_wp

        Alpha(1) = 0.0237936689536894477217231907983929772854_wp
        Alpha(2) = 0.1262969464732095495004306207054156629965_wp
        Alpha(3) = 0.3146136033888996736719578795060670017847_wp
        Alpha(4) = 0.5959506165914683579803642898831128604797_wp
        Alpha(5) = 0.9817869160077900206459386445168036061659_wp
        Alpha(6) = 1.4893061716231274493058359120745137715858_wp
        Alpha(7) = 2.1434977755714617149093714765228924079565_wp
        Alpha(8) = 2.9801766485096160056150055295276501965418_wp
        Alpha(9) = 4.0503389263368282892027738562035210634349_wp
        Alpha(10) = 5.4269062114355195174282553427502762133372_wp
        Alpha(11) = 7.2168007000185059830396627766901929135201_wp
        Alpha(12) = 9.5870452772388343133311283672526315058349_wp
        Alpha(13) = 12.8357494149077693088717433411716228874866_wp
        Alpha(14) = 17.6753312545577532001461751320903204032220_wp

        Xi(1) = 1.002048743182274802569937943719_wp
        Xi(2) = 1.018505136379617066234859468210_wp
        Xi(3) = 1.051666186927454282981804489783_wp
        Xi(4) = 1.102090275357523667431117186943_wp
        Xi(5) = 1.170601129974442013636712345281_wp
        Xi(6) = 1.258281223804727011797953417638_wp
        Xi(7) = 1.366533237763279598291597805026_wp
        Xi(8) = 1.497085087079080443811691936151_wp
        Xi(9) = 1.651984139160994388261655130279_wp
        Xi(10) = 1.833520904503394571990158856689_wp
        Xi(11) = 2.044409233122751011176368440836_wp
        Xi(12) = 2.287701715895157765848222375382_wp
        Xi(13) = 2.566570076646193390842651260719_wp
        Xi(14) = 2.884590301423110433052046652769_wp
        Xi(15) = 3.245079970134787466628373264754_wp
        Xi(16) = 3.651382264167655078293886483642_wp
        Xi(17) = 4.106056771269647577473488331279_wp
        Xi(18) = 4.610542740487232127050837293680_wp
        Xi(19) = 5.164333945273492986003521787453_wp
        Xi(20) = 5.763862439972125912591993657230_wp
        Xi(21) = 6.401371093462892352017601371728_wp
        Xi(22) = 7.063340611288946490654849785784_wp
        Xi(23) = 7.729450216803472793313661837189_wp
        Xi(24) = 8.371749672479516976228430813478_wp
        Xi(25) = 8.955490792852686543143447828186_wp
        Xi(26) = 9.441787161827626983709449426385_wp
        Xi(27) = 9.792471414911000586467348938413_wp
        Xi(28) = 9.976558470699145049660760964372_wp

      case (4)
        ! ----- K = 14 , R = 2E1 -----

        RIni = 2.0e1_wp

        Omega(1) = 0.0366971334946315666002201842654795171939_wp
        Omega(2) = 0.0872726090345815495787507726221932102817_wp
        Omega(3) = 0.1426593255181061152699582839087355523588_wp
        Omega(4) = 0.2064271217737803232469221553779803457473_wp
        Omega(5) = 0.2835897386182783272239298771877358262827_wp
        Omega(6) = 0.3812489108142730704557703258483769559462_wp
        Omega(7) = 0.5093454281471228856223096292943353091687_wp
        Omega(8) = 0.6815036880705534611782532572465953535357_wp
        Omega(9) = 0.9164311767479403860099142209438838335700_wp
        Omega(10) = 1.2411293792614432520554634820086903346237_wp
        Omega(11) = 1.6988916309365062128847873035297766364238_wp
        Omega(12) = 2.3714793690419589983642972841337837053288_wp
        Omega(13) = 3.4577518634250880304702169265596012337483_wp
        Omega(14) = 5.7402472869474302982761848568316054297611_wp

        Alpha(1) = 0.0142641020572255696012418284615574926022_wp
        Alpha(2) = 0.0759638522130805713449886763799501210315_wp
        Alpha(3) = 0.1903985303505704626254955297293136595727_wp
        Alpha(4) = 0.3640607520646051586846007885878151455472_wp
        Alpha(5) = 0.6076945939410506078003776364582222413446_wp
        Alpha(6) = 0.9380409019477502496897614037152379751205_wp
        Alpha(7) = 1.3802978464353806456422699144859222997184_wp
        Alpha(8) = 1.9713646649512173690283606264372906480276_wp
        Alpha(9) = 2.7641415826304564767185167895036101981532_wp
        Alpha(10) = 3.8339843611569064936537426513396553673374_wp
        Alpha(11) = 5.2903788510434682675065420909277236205526_wp
        Alpha(12) = 7.3023268339139556773118022725554965290939_wp
        Alpha(13) = 10.1671211920942247573448891095893031888409_wp
        Alpha(14) = 14.5837970745518835380405420210081501863897_wp

        Xi(1) = 1.002831485574532416013039670677_wp
        Xi(2) = 1.025583475486455251675550759405_wp
        Xi(3) = 1.071623521123393540607250662422_wp
        Xi(4) = 1.142033044953567116550074977965_wp
        Xi(5) = 1.238460861552316845781221421596_wp
        Xi(6) = 1.363160575873700177534919575351_wp
        Xi(7) = 1.519045447931094941019028454754_wp
        Xi(8) = 1.709763123619948054702881445888_wp
        Xi(9) = 1.939784035727543322910962353411_wp
        Xi(10) = 2.214503147865410618948256438188_wp
        Xi(11) = 2.540345053631744031948438977686_wp
        Xi(12) = 2.924868399997331741167402863191_wp
        Xi(13) = 3.376852039801018171699675818331_wp
        Xi(14) = 3.906342613885526484241578204504_wp
        Xi(15) = 4.524615371611106158771659080386_wp
        Xi(16) = 5.243987583175885720549597168727_wp
        Xi(17) = 6.077362373687328661337608837201_wp
        Xi(18) = 7.037344840674978575442710670984_wp
        Xi(19) = 8.134677080171074696306188211992_wp
        Xi(20) = 9.375679283081342884828557338039_wp
        Xi(21) = 10.758327198877061139925537247564_wp
        Xi(22) = 12.266718311791904568418265508711_wp
        Xi(23) = 13.864098226244888211779182629613_wp
        Xi(24) = 15.485650623782400917526069772379_wp
        Xi(25) = 17.033982474508729805171003413022_wp
        Xi(26) = 18.382163434717883752217426618358_wp
        Xi(27) = 19.389421462003029458104341742342_wp
        Xi(28) = 19.930475888186686651490364674544_wp

      case (5)
        ! ----- K = 14 , R = 3E1 -----

        RIni = 3.0e1_wp

        Omega(1) = 0.0269485719324102415012518437278932204038_wp
        Omega(2) = 0.0643920828356638264082208514271776778060_wp
        Omega(3) = 0.1062051071686407342615728922885498519690_wp
        Omega(4) = 0.1558226242684818538188507713870833981673_wp
        Omega(5) = 0.2182387103483475439470078496850824478770_wp
        Omega(6) = 0.3006760473398981716450836104703725482068_wp
        Omega(7) = 0.4132531737378125141432489569881525426354_wp
        Omega(8) = 0.5697250681720736762347553905350139302755_wp
        Omega(9) = 0.7889239435759886847178770641342282488040_wp
        Omega(10) = 1.0981226631535513495272621264930990037101_wp
        Omega(11) = 1.5410283271380334405411052789425241371646_wp
        Omega(12) = 2.1995131318883793108964752605061221402138_wp
        Omega(13) = 3.2707215533727314348867720728009089725674_wp
        Omega(14) = 5.5255489337075930690069969219990753117600_wp

        Alpha(1) = 0.0104692054336327858570128424009482159818_wp
        Alpha(2) = 0.0558852823007056333488396047265434063434_wp
        Alpha(3) = 0.1406960623012082067513454919360782469084_wp
        Alpha(4) = 0.2708801030466686922220081318313233964545_wp
        Alpha(5) = 0.4565826558074826535668297444736651868880_wp
        Alpha(6) = 0.7139990688724309940101954430335950974040_wp
        Alpha(7) = 1.0679417676762392550460917250987336046819_wp
        Alpha(8) = 1.5550829406200727021095803204175922473951_wp
        Alpha(9) = 2.2282123272354860115274793885831172701728_wp
        Alpha(10) = 3.1627623992935905163602361422903186394251_wp
        Alpha(11) = 4.4686527799694548118009107540160584903788_wp
        Alpha(12) = 6.3156455909790721985064276822186002391391_wp
        Alpha(13) = 9.0012119220790479849972354209342029207619_wp
        Alpha(14) = 13.2194183434275530303683710009465812618146_wp

        Xi(1) = 1.003321224832383769774712956036_wp
        Xi(2) = 1.030030492626788905276280505774_wp
        Xi(3) = 1.084196012045892534110189120433_wp
        Xi(4) = 1.167327983796548967749921732473_wp
        Xi(5) = 1.281737891987097182266638895065_wp
        Xi(6) = 1.430605741312938948063028521407_wp
        Xi(7) = 1.618081766492557762378463881436_wp
        Xi(8) = 1.849424363655329838873435377433_wp
        Xi(9) = 2.131171039029589531687766679191_wp
        Xi(10) = 2.471340346121211647804905009451_wp
        Xi(11) = 2.879661595395037482474770440533_wp
        Xi(12) = 3.367828706396157146000722160117_wp
        Xi(13) = 3.949772365114953853433912844828_wp
        Xi(14) = 4.641923498521994057713913051089_wp
        Xi(15) = 5.463422478780356666104578433618_wp
        Xi(16) = 6.436187086040242192729737835677_wp
        Xi(17) = 7.584677428480115245635478915531_wp
        Xi(18) = 8.935085516080642547599721670082_wp
        Xi(19) = 10.513520727690377768587248752397_wp
        Xi(20) = 12.342501004752789365084830741637_wp
        Xi(21) = 14.434812903450735910131086381369_wp
        Xi(22) = 16.783634779342583071351513446245_wp
        Xi(23) = 19.348159783798424478026833916999_wp
        Xi(24) = 22.035594549402970817805758052543_wp
        Xi(25) = 24.684372986728322205488006524376_wp
        Xi(26) = 27.059826381799241903219899185729_wp
        Xi(27) = 28.878537944622666046659564820231_wp
        Xi(28) = 29.871565401147938460107988944969_wp

      case (6)
        ! ----- K = 14 , R = 4E1 -----

        RIni = 4.0e1_wp

        Omega(1) = 0.0215714048457843105687122258612431480174_wp
        Omega(2) = 0.0517388459374235217105636105983101913353_wp
        Omega(3) = 0.0859543366519704896800508839294074903137_wp
        Omega(4) = 0.1275363375090655946496537645473701161336_wp
        Omega(5) = 0.1814236824781505632518110657236398708392_wp
        Omega(6) = 0.2548345086504729486455694126290083545427_wp
        Omega(7) = 0.3578645442824938114104692254979767085388_wp
        Omega(8) = 0.5042194995002234180077847758294495861264_wp
        Omega(9) = 0.7127864027157676590968661545222317954540_wp
        Omega(10) = 1.0111326193554143029794747943306276738440_wp
        Omega(11) = 1.4434560698218156258925901069822828048927_wp
        Omega(12) = 2.0919781635689984765260479582238417606277_wp
        Omega(13) = 3.1530482964361891962123962684927391819656_wp
        Omega(14) = 5.3905535884065372137508020955465326551348_wp

        Alpha(1) = 0.0083766354672433427827797441066193240289_wp
        Alpha(2) = 0.0447991964410272469703747522801196012665_wp
        Alpha(3) = 0.1131902344655730421101795350746765933536_wp
        Alpha(4) = 0.2191456758318374283477873026848570248148_wp
        Alpha(5) = 0.3723391762342706555802534368337930459347_wp
        Alpha(6) = 0.5884689484069488065202599424985407949862_wp
        Alpha(7) = 0.8918422272547547164332770730510446810513_wp
        Alpha(8) = 1.3185844672473211036797455242286503107607_wp
        Alpha(9) = 1.9209270418492403716473337915715546841966_wp
        Alpha(10) = 2.7739127897433361309795923910215265095758_wp
        Alpha(11) = 3.9874784350916229929209788629407285043271_wp
        Alpha(12) = 5.7318680938431347069578203345940892177168_wp
        Alpha(13) = 8.3049530509144087178294291184954545315122_wp
        Alpha(14) = 12.3982499699347220156048221539890619169455_wp

        Xi(1) = 1.003680405687570364558873492467_wp
        Xi(2) = 1.033296098396227069050878899503_wp
        Xi(3) = 1.093451374628653129691784051669_wp
        Xi(4) = 1.186016540882130367202951237715_wp
        Xi(5) = 1.313861639271574527181030145329_wp
        Xi(6) = 1.480953050500121769743680499865_wp
        Xi(7) = 1.692500867408396253975012835102_wp
        Xi(8) = 1.955159670970068617155968038190_wp
        Xi(9) = 2.277280556392905020443931385365_wp
        Xi(10) = 2.669213858930995028877902752740_wp
        Xi(11) = 3.143665520773058335776695138009_wp
        Xi(12) = 3.716111334016190307109536727914_wp
        Xi(13) = 4.405269811350842221794182895778_wp
        Xi(14) = 5.233616831145244719519782350403_wp
        Xi(15) = 6.227898500017066587240710529016_wp
        Xi(16) = 7.419548529613404719285085775482_wp
        Xi(17) = 8.844822521574932413200575975765_wp
        Xi(18) = 10.544315793569040863066066382459_wp
        Xi(19) = 12.561248743920816494691605758049_wp
        Xi(20) = 14.937519478743447998332627424389_wp
        Xi(21) = 17.705931018766468342320874462814_wp
        Xi(22) = 20.876409995041283034974965993058_wp
        Xi(23) = 24.413903521863484842591152101932_wp
        Xi(24) = 28.207393563142253014447424774858_wp
        Xi(25) = 32.035512194776539631607681357650_wp
        Xi(26) = 35.546799003083486633014498323746_wp
        Xi(27) = 38.287076400773418147910343023455_wp
        Xi(28) = 39.802914803067470653902271138236_wp

      case (7)
        ! ----- K = 14 , R = 5E1 -----

        RIni = 5.0e1_wp

        Omega(1) = 0.0181203903997484738327760711920633163174_wp
        Omega(2) = 0.0436009472996198242532490691597546472735_wp
        Omega(3) = 0.0728818458472537549459944854202131381271_wp
        Omega(4) = 0.1091824331810118182446581527877960127171_wp
        Omega(5) = 0.1573753385008842204718405854979579316932_wp
        Omega(6) = 0.2246218843896930941441336324060351614662_wp
        Omega(7) = 0.3209240269289367617541047783547014660144_wp
        Omega(8) = 0.4598968526369810663615485951671857378642_wp
        Omega(9) = 0.6604755354010544363412178736627566877360_wp
        Omega(10) = 0.9505074044318574636065755090186257802998_wp
        Omega(11) = 1.3746503676746198249368144383986134471343_wp
        Omega(12) = 2.0155108372394360859505835215088609402301_wp
        Omega(13) = 3.0690175985474151918331459087951884612266_wp
        Omega(14) = 5.2942233398720185285311889966664011808462_wp

        Alpha(1) = 0.0070339795370923024902860712457425274025_wp
        Alpha(2) = 0.0376784577605089209654284747308405911781_wp
        Alpha(3) = 0.0954894503195428311095546182463156270614_wp
        Alpha(4) = 0.1857638802321160366508258007800513667007_wp
        Alpha(5) = 0.3177927469496878407117027021122623864358_wp
        Alpha(6) = 0.5068344028592784806185565782410407109637_wp
        Alpha(7) = 0.7766828152420551294394038266410973392340_wp
        Alpha(8) = 1.1628491895241590386436345005982673228573_wp
        Alpha(9) = 1.7169175230748922364020642161719365503814_wp
        Alpha(10) = 2.5134448279597254472406825387409412542183_wp
        Alpha(11) = 3.6622634753022948194984415026809188020707_wp
        Alpha(12) = 5.3339687858463377414207362559039893312729_wp
        Alpha(13) = 7.8268675578464850916410311398863086651545_wp
        Alpha(14) = 11.8309580846968924782228183012477984448196_wp

        Xi(1) = 1.003964463629397876258224353574_wp
        Xi(2) = 1.035881220231276358544232618630_wp
        Xi(3) = 1.100792113768499243535610443434_wp
        Xi(4) = 1.200880066059467387315842779039_wp
        Xi(5) = 1.339501843116432876668216167548_wp
        Xi(6) = 1.521312512319487506799037257466_wp
        Xi(7) = 1.752457273050489124110486172103_wp
        Xi(8) = 2.040833248721928070649328512687_wp
        Xi(9) = 2.396420482118169978597368996454_wp
        Xi(10) = 2.831683015171900737443760887047_wp
        Xi(11) = 3.362052647165408762221228133882_wp
        Xi(12) = 4.006504479687130676219403158100_wp
        Xi(13) = 4.788240572094590569770383714499_wp
        Xi(14) = 5.735469840795347543169896420423_wp
        Xi(15) = 6.882254158139940615219326547347_wp
        Xi(16) = 8.269329729885666711332314893212_wp
        Xi(17) = 9.944709440199711880832567079125_wp
        Xi(18) = 11.963673873112535328555594560385_wp
        Xi(19) = 14.387430710146393157342070701077_wp
        Xi(20) = 17.279103242847792755790270469163_wp
        Xi(21) = 20.694861929304298055198874095595_wp
        Xi(22) = 24.666812271173920334543794297133_wp
        Xi(23) = 29.173367242468087497109241468252_wp
        Xi(24) = 34.094177311204777226089079888993_wp
        Xi(25) = 39.154260818376700605675244659665_wp
        Xi(26) = 43.881478345259495987135123229450_wp
        Xi(27) = 47.629809930857014223931633623010_wp
        Xi(28) = 49.726209480431178468901043387973_wp

      case (8)
        ! ----- K = 14 , R = 6E1 -----

        RIni = 6.0e1_wp

        Omega(1) = 0.0156994377913550977882427587759006648582_wp
        Omega(2) = 0.0378817333353865612529927462542866933859_wp
        Omega(3) = 0.0636654818251124950969320470672840883708_wp
        Omega(4) = 0.0961847375948002209744796923984733183488_wp
        Omega(5) = 0.1402432889797341093583065221861438942597_wp
        Omega(6) = 0.2029227810849076473230035225214784588843_wp
        Omega(7) = 0.2941076299401057839134291521654418488652_wp
        Omega(8) = 0.4273186187383292571376983831621387821542_wp
        Omega(9) = 0.6215386798463879908845791666394120511541_wp
        Omega(10) = 0.9048675733944338747389950217137055687999_wp
        Omega(11) = 1.3223740652817814597227985551342044345802_wp
        Omega(12) = 1.9570374697495082172253588748134234265308_wp
        Omega(13) = 3.0045539951549261944042240601859816706565_wp
        Omega(14) = 5.2203736700979702428301876260263725271216_wp

        Alpha(1) = 0.0060922817824678279624866941797250996515_wp
        Alpha(2) = 0.0326796427354352025450941101464286120404_wp
        Alpha(3) = 0.0830432040368440788216186990156053582268_wp
        Alpha(4) = 0.1622373410799410406180889529514921321152_wp
        Alpha(5) = 0.2792335371332518192781513288736050526495_wp
        Alpha(6) = 0.4489001056044332305485552309098906675899_wp
        Alpha(7) = 0.6945418267921845624675303287887118131039_wp
        Alpha(8) = 1.0510655723020722144027058742032920690690_wp
        Alpha(9) = 1.5694187175490266713574760704119626097963_wp
        Alpha(10) = 2.3236707816357991059248722098828920934466_wp
        Alpha(11) = 3.4235072283402712242823817678782916118507_wp
        Alpha(12) = 5.0397939065273830204841343238797435333254_wp
        Alpha(13) = 7.4712575541170140043349678915518552457797_wp
        Alpha(14) = 11.4069124477270112142865410298497863550438_wp

        Xi(1) = 1.004199463362554230845199998434_wp
        Xi(2) = 1.038021580056463033818123231633_wp
        Xi(3) = 1.106879359284786768771388643096_wp
        Xi(4) = 1.213233295898756222762175649432_wp
        Xi(5) = 1.360873325556787383238711852673_wp
        Xi(6) = 1.555070342390879033667334951918_wp
        Xi(7) = 1.802810809432306055760664165177_wp
        Xi(8) = 2.113117747631799232766944274253_wp
        Xi(9) = 2.497457179134984294995372455794_wp
        Xi(10) = 2.970237139378840199180689674030_wp
        Xi(11) = 3.549413976207429963476841505710_wp
        Xi(12) = 4.257233936341095928250488622524_wp
        Xi(13) = 5.121126723365594167296233063524_wp
        Xi(14) = 6.174761118797634141912300531629_wp
        Xi(15) = 7.459241468024081396503977003931_wp
        Xi(16) = 9.024367421522351983935450459029_wp
        Xi(17) = 10.929764591262393068696323616962_wp
        Xi(18) = 13.245466038343728465073723121748_wp
        Xi(19) = 16.051122814181819527246108947338_wp
        Xi(20) = 19.432246835564534259005453620262_wp
        Xi(21) = 23.470700729134137326251097199759_wp
        Xi(22) = 28.224802961012600523854620959696_wp
        Xi(23) = 33.692641973277278354126362813759_wp
        Xi(24) = 39.752409716617594977877692130619_wp
        Xi(25) = 46.082213176208074587730800786289_wp
        Xi(26) = 52.088252836165963022274638660747_wp
        Xi(27) = 56.916242705365995533522971427942_wp
        Xi(28) = 59.642539771599628304482587637381_wp

      case (9)
        ! ----- K = 14 , R = 7E1 -----

        RIni = 7.0e1_wp

        Omega(1) = 0.0138981094100516390688979258792590254501_wp
        Omega(2) = 0.0336195382439525929142991496345604929274_wp
        Omega(3) = 0.0567778323204290500227434113078572863742_wp
        Omega(4) = 0.0864325676208157085079661872961676039040_wp
        Omega(5) = 0.1273193360592727064610528778243470071629_wp
        Omega(6) = 0.1864307073569864624219220156087217787899_wp
        Omega(7) = 0.2735287270934270563058857833604875509081_wp
        Omega(8) = 0.4020466010199495706441564363275986693225_wp
        Omega(9) = 0.5910134830489486492820401519754369701332_wp
        Omega(10) = 0.8687528181682690328266338575247118569678_wp
        Omega(11) = 1.2806972849123447405830789058711616235087_wp
        Omega(12) = 1.9101752773557870700724473089771038303297_wp
        Omega(13) = 2.9527569215812303133391475418179084044823_wp
        Omega(14) = 5.1610696744092252451902025356389458465856_wp

        Alpha(1) = 0.0053917358661889226640618331173188115812_wp
        Alpha(2) = 0.0289579546036361528682647826019036507716_wp
        Alpha(3) = 0.0737635463848466191102309398586189814750_wp
        Alpha(4) = 0.1446605225882026802256994854611882317386_wp
        Alpha(5) = 0.2503473594396809384600086079153413720633_wp
        Alpha(6) = 0.4053437838072211329130435653045338995071_wp
        Alpha(7) = 0.6324982596023326382591797645282838402636_wp
        Alpha(8) = 0.9661477674983821334483112541757776625673_wp
        Alpha(9) = 1.4566423523519519287123050177434890883887_wp
        Alpha(10) = 2.1775887642399564558294555727258057231666_wp
        Alpha(11) = 3.2385103860715660794612757999999530511559_wp
        Alpha(12) = 4.8104879927941818903644466409730284794932_wp
        Alpha(13) = 7.1926391193233954741807767430117337426054_wp
        Alpha(14) = 11.0733054471847641789250538124633749248460_wp

        Xi(1) = 1.004399828902913362942486696916_wp
        Xi(2) = 1.039847728421465022181566228987_wp
        Xi(3) = 1.112079776788726031958899620466_wp
        Xi(4) = 1.223806877783323803594514589399_wp
        Xi(5) = 1.379210529911305202163153416084_wp
        Xi(6) = 1.584120511386782501994478777885_wp
        Xi(7) = 1.846290820870384225254462462473_wp
        Xi(8) = 2.175777180328766481588395609315_wp
        Xi(9) = 2.585417910500366631696214647640_wp
        Xi(10) = 3.091425783935278747945399335784_wp
        Xi(11) = 3.714118769055528738315397063730_wp
        Xi(12) = 4.478822257191948924461705194844_wp
        Xi(13) = 5.416976525759794772445915311820_wp
        Xi(14) = 6.567471281122377282490204519760_wp
        Xi(15) = 7.978203918950735546670310904815_wp
        Xi(16) = 9.707800066685734122554074509992_wp
        Xi(17) = 11.827316057772773032728819142534_wp
        Xi(18) = 14.421495083963127516527014115866_wp
        Xi(19) = 17.588671250208150659971595430875_wp
        Xi(20) = 21.437511045352986821521668048263_wp
        Xi(21) = 26.077244143316060470683126482072_wp
        Xi(22) = 31.595543488792589763985985129580_wp
        Xi(23) = 38.015251733818948200877008147813_wp
        Xi(24) = 45.220081559448178532001350049541_wp
        Xi(25) = 52.847944494740012696953890269924_wp
        Xi(26) = 60.184168246765109535484361558133_wp
        Xi(27) = 66.153095322911794506914873181813_wp
        Xi(28) = 69.552680159615619150115239932575_wp

      case (10)
        ! ----- K = 14 , R = 8E1 -----

        RIni = 8.0e1_wp

        Omega(1) = 0.0125004212646736390650997489820683838957_wp
        Omega(2) = 0.0303076914188914629644759765256245565013_wp
        Omega(3) = 0.0514124614377946843495937895990310906313_wp
        Omega(4) = 0.0788083970102124296733085706545551829549_wp
        Omega(5) = 0.1171649099103936408845837505166009862023_wp
        Omega(6) = 0.1733827405509416185201481894195829625005_wp
        Omega(7) = 0.2571046068008916202663350497603644839728_wp
        Omega(8) = 0.3816850457961544672508616415917970243754_wp
        Omega(9) = 0.5661965543161639033875771886972216861977_wp
        Omega(10) = 0.8391601113814530699314998829496659027427_wp
        Omega(11) = 1.2463316648187273651864417911561133678333_wp
        Omega(12) = 1.8713633171352733962876369044892044257722_wp
        Omega(13) = 2.9097642414871805181288133557160335840308_wp
        Omega(14) = 5.1118712221355903872060943982091885118280_wp

        Alpha(1) = 0.0048482592516001103827690205463896155891_wp
        Alpha(2) = 0.0260686293011070024490058189697849755362_wp
        Alpha(3) = 0.0665500407740831386566547622773981629507_wp
        Alpha(4) = 0.1309719715774267940870418530652230515443_wp
        Alpha(5) = 0.2277953400254295355462910363519846157487_wp
        Alpha(6) = 0.3712257195456134759318968935337679226905_wp
        Alpha(7) = 0.5836889969345770847559615179811487450934_wp
        Alpha(8) = 0.8989929313055353704434505268672239708394_wp
        Alpha(9) = 1.3669355474682264575334023071206956956303_wp
        Alpha(10) = 2.0606917616485738874079064464339694495720_wp
        Alpha(11) = 3.0896188502677947973051181262960085405211_wp
        Alpha(12) = 4.6249719171086872476425511102604559710016_wp
        Alpha(13) = 6.9662281803325074125247950984629596860032_wp
        Alpha(14) = 10.8012531761208309200386623594170032447437_wp

        Xi(1) = 1.004574409398721246216255875705_wp
        Xi(2) = 1.041439794412749675648829916774_wp
        Xi(3) = 1.116618705551927111024532179862_wp
        Xi(4) = 1.233050686296803127601356964149_wp
        Xi(5) = 1.395275330904291830914909500194_wp
        Xi(6) = 1.609635232881006502774975941428_wp
        Xi(7) = 1.884591731760469213901101892539_wp
        Xi(8) = 2.231157831100469954212911494196_wp
        Xi(9) = 2.663449323495960571750451606476_wp
        Xi(10) = 3.199369365994019141514331794340_wp
        Xi(11) = 3.861458454225360590857454856639_wp
        Xi(12) = 4.677958450114408868458770740517_wp
        Xi(13) = 5.684132759612934091084796017768_wp
        Xi(14) = 6.923881926522082252452305883850_wp
        Xi(15) = 8.451669851273091271619264386317_wp
        Xi(16) = 10.334721829133737200367981756699_wp
        Xi(17) = 12.655331836263332695967365193468_wp
        Xi(18) = 15.512852820690443205256525693869_wp
        Xi(19) = 19.024412347503992242944703683349_wp
        Xi(20) = 23.322348472599268891930490710251_wp
        Xi(21) = 28.544519675407910743250572416230_wp
        Xi(22) = 34.810434964067422023564768451820_wp
        Xi(23) = 42.171975004832018905920332230153_wp
        Xi(24) = 50.524527995838974355619077627466_wp
        Xi(25) = 59.472387311023984672597730849475_wp
        Xi(26) = 68.181905048529348800645522032937_wp
        Xi(27) = 75.345409801972011369641801792341_wp
        Xi(28) = 79.457213912610988523765875868321_wp

      case (11)
        ! ----- K = 14 , R = 9E1 -----

        RIni = 9.0e1_wp

        Omega(1) = 0.0113813041251049466054515212423692283039_wp
        Omega(2) = 0.0276524776127700123882970518335300269541_wp
        Omega(3) = 0.0471009994681223302796281311027870586372_wp
        Omega(4) = 0.0726615318115644792144815385104283933515_wp
        Omega(5) = 0.1089398770918274977868313822815249380938_wp
        Omega(6) = 0.1627455965234474854629216633572497130444_wp
        Omega(7) = 0.2436082848337518060347853948721663641663_wp
        Omega(8) = 0.3648123709988585392199455376127659178564_wp
        Omega(9) = 0.5454701187758314865272266103168874451512_wp
        Omega(10) = 0.8142773411254590846261938730243912232254_wp
        Omega(11) = 1.2172789422143520286171552990950317507668_wp
        Omega(12) = 1.8384268358954554908619299924410483981774_wp
        Omega(13) = 2.8732112163151237922177616557206647485145_wp
        Omega(14) = 5.0700605746327023072399231562457089239615_wp

        Alpha(1) = 0.0044131687819247254553858990627074332380_wp
        Alpha(2) = 0.0237540107528117159956443154786298599390_wp
        Alpha(3) = 0.0607645778668153227434893005887150962963_wp
        Alpha(4) = 0.1199747097378512308168442557221311517424_wp
        Alpha(5) = 0.2096355719615132448280930832495982940600_wp
        Alpha(6) = 0.3436674309974069099489057260710112018387_wp
        Alpha(7) = 0.5441054324421764173047230683799568851100_wp
        Alpha(8) = 0.8442683196867919211809315471750636561410_wp
        Alpha(9) = 1.2934459449065358992029053930572501940333_wp
        Alpha(10) = 1.9644120588566827564046268772557368720300_wp
        Alpha(11) = 2.9663588112266798511602178267665408384346_wp
        Alpha(12) = 4.4706848820515293859746697169299523011432_wp
        Alpha(13) = 6.7771978107945288916785786526020274322946_wp
        Alpha(14) = 10.5734181763829215875902245436179782700492_wp

        Xi(1) = 1.004729032416783593298706400709_wp
        Xi(2) = 1.042850578143463038927572783532_wp
        Xi(3) = 1.120644850395562548629919963350_wp
        Xi(4) = 1.241262023107741417605152933756_wp
        Xi(5) = 1.409572102304617733877813157140_wp
        Xi(6) = 1.632392438605882554612976076402_wp
        Xi(7) = 1.918841959530300752970535804742_wp
        Xi(8) = 2.280827243721750823869653945764_wp
        Xi(9) = 2.733662096629159128126362876543_wp
        Xi(10) = 3.296841212189401015180204024269_wp
        Xi(11) = 3.995010396018193446401184365513_wp
        Xi(12) = 4.859187146991676860535019910614_wp
        Xi(13) = 5.928293590676148453692123618808_wp
        Xi(14) = 7.251053897509312391560232224919_wp
        Xi(15) = 8.888293048457661208426594523502_wp
        Xi(16) = 10.915623346908362838354744628333_wp
        Xi(17) = 13.426381142357288123005687463518_wp
        Xi(18) = 16.534400662015035598020884499704_wp
        Xi(19) = 20.375627698379672405812335078679_wp
        Xi(20) = 25.106432824227256042637357325020_wp
        Xi(21) = 30.894266557075042121949248752344_wp
        Xi(22) = 37.892497473926202138788665862990_wp
        Xi(23) = 46.185670325825824523990892345182_wp
        Xi(24) = 55.686378646269722496320309090834_wp
        Xi(25) = 65.971578519795400462333301305762_wp
        Xi(26) = 76.091301002290827611151957654556_wp
        Xi(27) = 84.497138578197352687393362202783_wp
        Xi(28) = 89.356599271330439969440728731342_wp

      case (12)
        ! ----- K = 14 , R = 1E2 -----

        RIni = 1.0e2_wp

        Omega(1) = 0.0104630611822745417864471734366305089026_wp
        Omega(2) = 0.0254712518740332508723116320542567692087_wp
        Omega(3) = 0.0435516943672130260874577646623695414974_wp
        Omega(4) = 0.0675857109039668863917122050566366198154_wp
        Omega(5) = 0.1021182439661924676755234281388950989822_wp
        Omega(6) = 0.1538700796419602742751185989189544045530_wp
        Omega(7) = 0.2322649456760444986580917484109143344995_wp
        Omega(8) = 0.3505246753474143100429063352185821145213_wp
        Omega(9) = 0.5277975088592344113380683268132997909561_wp
        Omega(10) = 0.7929346987934133738815632719276749185155_wp
        Omega(11) = 1.1922412421604374170450504899498866961949_wp
        Omega(12) = 1.8099472513036932155502944929992281686282_wp
        Omega(13) = 2.8415520082506048565700040864356878955732_wp
        Omega(14) = 5.0338617706525944036667630232528836131678_wp

        Alpha(1) = 0.0040562245376395091996092839934512364053_wp
        Alpha(2) = 0.0218539743092570181921173661301893709208_wp
        Alpha(3) = 0.0560102456526223489891057934902951487288_wp
        Alpha(4) = 0.1109232905691386661652868099448010497099_wp
        Alpha(5) = 0.1946568577743407214288798995349694109791_wp
        Alpha(6) = 0.3208703362477143840834903354419793686247_wp
        Alpha(7) = 0.5112372035499269371829067498680387870991_wp
        Alpha(8) = 0.7986245539444243275793537673035160651125_wp
        Alpha(9) = 1.2318541070578792897237396797294195494032_wp
        Alpha(10) = 1.8833267887511218623055458953885477058066_wp
        Alpha(11) = 2.8620729340610342446380581771236961685645_wp
        Alpha(12) = 4.3396108706176149492211857605639124813024_wp
        Alpha(13) = 6.6160519092730605914681740742366855556611_wp
        Alpha(14) = 10.3786631164987534587870743330029199569253_wp

        Xi(1) = 1.004867744422024420096646590839_wp
        Xi(2) = 1.044116774458187907196130006504_wp
        Xi(3) = 1.124261616172937912989590603097_wp
        Xi(4) = 1.248647883970209255914685098077_wp
        Xi(5) = 1.422452844938755765931415186998_wp
        Xi(6) = 1.652936457790613973312283946360_wp
        Xi(7) = 1.949832966263220590725536995880_wp
        Xi(8) = 2.325887705682776740489028921566_wp
        Xi(9) = 2.797544455295651174555823792467_wp
        Xi(10) = 3.385805976423070912410470678644_wp
        Xi(11) = 4.117318890875324560556236530573_wp
        Xi(12) = 5.025753177986356100021697601576_wp
        Xi(13) = 6.153544413960178281319118864801_wp
        Xi(14) = 7.554072825888110332380465106183_wp
        Xi(15) = 9.294337225410822821147771577444_wp
        Xi(16) = 11.458138137699406099850873985702_wp
        Xi(17) = 14.149662656228357445614696175085_wp
        Xi(18) = 17.497079141823863175989051121917_wp
        Xi(19) = 21.655133140059325434420567368932_wp
        Xi(20) = 26.804466342392660601137710507658_wp
        Xi(21) = 33.142875404431976070485088570194_wp
        Xi(22) = 40.859263677785791095409262396743_wp
        Xi(23) = 50.073963117031069433582324279541_wp
        Xi(24) = 60.721740000069154011980154805883_wp
        Xi(25) = 72.358214095924009510996111060877_wp
        Xi(26) = 83.920238156193154091533337179953_wp
        Xi(27) = 93.611467602814374969200095222277_wp
        Xi(28) = 99.251208591006958893709910540792_wp

      case (13)
        ! ----- K = 14 , R = 2E2 -----

        RIni = 2.0e2_wp

        Omega(1) = 0.0059897754697514995871548493591718376194_wp
        Omega(2) = 0.0147993270032761131379042685018987413059_wp
        Omega(3) = 0.0260524770457164332042342743539498073346_wp
        Omega(4) = 0.0422725413703181347437271356271004663085_wp
        Omega(5) = 0.0675264601300846596407885698287731912615_wp
        Omega(6) = 0.1078388793634545032024542147996992724757_wp
        Omega(7) = 0.1719019292760687385337829702347534066575_wp
        Omega(8) = 0.2725409542163217569029059345053767060563_wp
        Omega(9) = 0.4291033113493526942206369306775926020237_wp
        Omega(10) = 0.6713790246552777997717138813715820333528_wp
        Omega(11) = 1.0473599692425859896240461499594687211356_wp
        Omega(12) = 1.6432750177593764012327556822690155513556_wp
        Omega(13) = 2.6552117669183682784842664981184157113603_wp
        Omega(14) = 4.8210711548907294781221577473218076193007_wp

        Alpha(1) = 0.0023182288230558957666211321703778058811_wp
        Alpha(2) = 0.0125823731486630252544487145599516608740_wp
        Alpha(3) = 0.0327195618552320785285395833563715228820_wp
        Alpha(4) = 0.0663255367617940227539688295699260578431_wp
        Alpha(5) = 0.1202545590709986545781143632560805656340_wp
        Alpha(6) = 0.2063694428248624243463042737811719007368_wp
        Alpha(7) = 0.3438043885629947898343802825671389200579_wp
        Alpha(8) = 0.5622945792714698096202013966049548798765_wp
        Alpha(9) = 0.9074155937680068548679274131885108545248_wp
        Alpha(10) = 1.4488825900310234888381660800504846520198_wp
        Alpha(11) = 2.2943634981023110239271162180685337261821_wp
        Alpha(12) = 3.6159269289529930701404536463883232499938_wp
        Alpha(13) = 5.7157741285866358522428642974233525819727_wp
        Alpha(14) = 9.2805653437018404599070930949267221876653_wp

        Xi(1) = 1.005783423474271097430854382537_wp
        Xi(2) = 1.052489365036918946808182362407_wp
        Xi(3) = 1.148254951461565999556134476389_wp
        Xi(4) = 1.297874142975835953014301926700_wp
        Xi(5) = 1.508814844920841332504002352266_wp
        Xi(6) = 1.791671683393926679740866536594_wp
        Xi(7) = 2.160876387478386478034619866406_wp
        Xi(8) = 2.635675720287350149405314336803_wp
        Xi(9) = 3.241395772617858590889522441003_wp
        Xi(10) = 4.011075652554957635467780541205_wp
        Xi(11) = 4.987610740587797831645938684275_wp
        Xi(12) = 6.226585113565167666025562676069_wp
        Xi(13) = 7.800007865684829249600801270859_wp
        Xi(14) = 9.801220827275853267335214979994_wp
        Xi(15) = 12.351309172252640383726784989449_wp
        Xi(16) = 15.607391642043947972695594472725_wp
        Xi(17) = 19.773157609552126628277290976143_wp
        Xi(18) = 25.111797925531063319440350234402_wp
        Xi(19) = 31.960759802441204940198771211612_wp
        Xi(20) = 40.745860291856650380742932426870_wp
        Xi(21) = 51.987661755080454287908375832217_wp
        Xi(22) = 66.282418961142906481398551221673_wp
        Xi(23) = 84.217970766470203421161233592329_wp
        Xi(24) = 106.145066720760969825043673608889_wp
        Xi(25) = 131.677581455554635031512944465248_wp
        Xi(26) = 158.832561217851184115490426052020_wp
        Xi(27) = 183.129616610208101168399963398770_wp
        Xi(28) = 197.985220406843973842136996665886_wp

      case (14)
        ! ----- K = 14 , R = 3E2 -----

        RIni = 3.0e2_wp

        Omega(1) = 0.0043105055503077696991761779028717871398_wp
        Omega(2) = 0.0107629963032836868978568190848010832639_wp
        Omega(3) = 0.0193446992531293919894731792030739825350_wp
        Omega(4) = 0.0323686242376243212328105949143752795294_wp
        Omega(5) = 0.0535802550401220360397575113303725302671_wp
        Omega(6) = 0.0885650729153326647007452021576057177299_wp
        Omega(7) = 0.1456078542584206715610048096798401218166_wp
        Omega(8) = 0.2373064176090271693307697312147475088295_wp
        Omega(9) = 0.3830591074603137403096781571365525564943_wp
        Omega(10) = 0.6131010801761819402886689867049341273741_wp
        Omega(11) = 0.9763482420250424030530415897555229776117_wp
        Omega(12) = 1.5602761224141383718288025872489299672452_wp
        Omega(13) = 2.5616606516288392130387824296988696914923_wp
        Omega(14) = 4.7144075044545905714063471769748048245674_wp

        Alpha(1) = 0.0016663613283814120935658878094876023113_wp
        Alpha(2) = 0.0090917173500048858559324685239821928207_wp
        Alpha(3) = 0.0238907650081464097515017040371243517427_wp
        Alpha(4) = 0.0492457144648069480704570110435325247522_wp
        Alpha(5) = 0.0913375036628434229659745727247788238401_wp
        Alpha(6) = 0.1609683530978087451934364568439939091604_wp
        Alpha(7) = 0.2757734097349819249304393731225815145081_wp
        Alpha(8) = 0.4636622369593021248334284434289287446518_wp
        Alpha(9) = 0.7682880833268509717520755586406266957056_wp
        Alpha(10) = 1.2576815493486096930600762200391784517706_wp
        Alpha(11) = 2.0385090183531704907931442205537564404949_wp
        Alpha(12) = 3.2829551393289981401930194282456909604662_wp
        Alpha(13) = 5.2944340379242350253310989138810782606015_wp
        Alpha(14) = 8.7598988337524299023495455429610956343822_wp

        Xi(1) = 1.006315237360364457479267041862_wp
        Xi(2) = 1.057363459691149750168531751271_wp
        Xi(3) = 1.162285463353864100699390504712_wp
        Xi(4) = 1.326845259291127352113508208742_wp
        Xi(5) = 1.560057635766641579593873301324_wp
        Xi(6) = 1.874801561952667291717847253274_wp
        Xi(7) = 2.288784347834112163069461565179_wp
        Xi(8) = 2.825870501566240824541792475699_wp
        Xi(9) = 3.517815375625272366266402257473_wp
        Xi(10) = 4.406544277999540099900743772565_wp
        Xi(11) = 5.547208178319016709696687161824_wp
        Xi(12) = 7.012296877113417459346372817564_wp
        Xi(13) = 8.897174186778193184614582644798_wp
        Xi(14) = 11.327513582117413042105180664976_wp
        Xi(15) = 14.469279784670279704293649469804_wp
        Xi(16) = 18.542064035176400008803065944107_wp
        Xi(17) = 23.836755324296568880546032254131_wp
        Xi(18) = 30.738562568365602854433826784941_wp
        Xi(19) = 39.755848693557840636825240210328_wp
        Xi(20) = 51.553351851322890719853120344851_wp
        Xi(21) = 66.982720580029443922798204624769_wp
        Xi(22) = 87.088441063746228787412206884255_wp
        Xi(23) = 113.031639670528592740705864372330_wp
        Xi(24) = 145.797237152944107990704125654702_wp
        Xi(25) = 185.418779279689918140783255751103_wp
        Xi(26) = 229.381732373208163597300135450041_wp
        Xi(27) = 270.443464948497017869000202949792_wp
        Xi(28) = 296.422536960799892280515877018843_wp

      case (15)
        ! ----- K = 14 , R = 4E2 -----

        RIni = 4.0e2_wp

        Omega(1) = 0.0034102793689438034177186941271753273242_wp
        Omega(2) = 0.0085877067644438304675559613457491714428_wp
        Omega(3) = 0.0156948222369874568904316372575102889186_wp
        Omega(4) = 0.0268969875261082530000641618861401127560_wp
        Omega(5) = 0.0457049261615041172641710383317548149762_wp
        Omega(6) = 0.0774022197228828850578286752126544456587_wp
        Omega(7) = 0.1300039887810861542234578194787708582680_wp
        Omega(8) = 0.2159412580728695688411919104132685731656_wp
        Omega(9) = 0.3546111560919620659575526855356031319388_wp
        Omega(10) = 0.5765112081449787561986640438416173992664_wp
        Omega(11) = 0.9311714010393723479734606751634373722482_wp
        Omega(12) = 1.5069617811410725934963272809596901424811_wp
        Omega(13) = 2.5012628008061577539536401992847913788864_wp
        Omega(14) = 4.6455970530622682453970873073245684281574_wp

        Alpha(1) = 0.0013171250509087271421003036641315941946_wp
        Alpha(2) = 0.0072165930695399487539990672577833175083_wp
        Alpha(3) = 0.0191248799561856207320913745087431756531_wp
        Alpha(4) = 0.0399561872177887720431496785394864801333_wp
        Alpha(5) = 0.0754352695183603340863052848230818625552_wp
        Alpha(6) = 0.1356331022754389202230121402870111069205_wp
        Alpha(7) = 0.2371598253031562126397734091565183689454_wp
        Alpha(8) = 0.4066773886612077479861277612416259330530_wp
        Alpha(9) = 0.6865007575761530051600134427758348465431_wp
        Alpha(10) = 1.1434436672367446008805047519274467049399_wp
        Alpha(11) = 1.8833899153099708375855292352696324087447_wp
        Alpha(12) = 3.0785150126590599113738516701133107744681_wp
        Alpha(13) = 5.0330591708377733087612093676455060631270_wp
        Alpha(14) = 8.4343872183413754663877659467630110157188_wp

        Xi(1) = 1.006687768458224417687622842532_wp
        Xi(2) = 1.060782789639744152446892977437_wp
        Xi(3) = 1.172156295232698616976853289717_wp
        Xi(4) = 1.347309446295021649543828534679_wp
        Xi(5) = 1.596439169299369165676283355282_wp
        Xi(6) = 1.934186091099623896723616600646_wp
        Xi(7) = 2.380813041319493150813763060469_wp
        Xi(8) = 2.963827965478099218249452562191_wp
        Xi(9) = 3.720118524505239503532075251435_wp
        Xi(10) = 4.698795731552833488016418517219_wp
        Xi(11) = 5.965055525664543634948439043697_wp
        Xi(12) = 7.605444403980453190773636240607_wp
        Xi(13) = 9.735034757760511523079494100585_wp
        Xi(14) = 12.507213603001360598121483036493_wp
        Xi(15) = 16.127036255524257069016158538943_wp
        Xi(16) = 20.869417193176430210324601155492_wp
        Xi(17) = 27.103866884064095709241937193212_wp
        Xi(18) = 35.327692313473632475945285236207_wp
        Xi(19) = 46.209591656963769922689122537918_wp
        Xi(20) = 60.643879284988232058334345353501_wp
        Xi(21) = 79.809678786824841463520829876188_wp
        Xi(22) = 105.211917927531603808577465031249_wp
        Xi(23) = 138.634083789398782127211262604760_wp
        Xi(24) = 181.819035653450962422561509868046_wp
        Xi(25) = 235.465404790081247646260642625293_wp
        Xi(26) = 296.867425186533363201668578312820_wp
        Xi(27) = 356.119449512395370216477630265217_wp
        Xi(28) = 394.631056323149155384832909021497_wp

      case (16)
        ! ----- K = 14 , R = 5E2 -----

        RIni = 5.0e2_wp

        Omega(1) = 0.0028426663110938020009905725362861494965_wp
        Omega(2) = 0.0072103241609037110839534076393853023745_wp
        Omega(3) = 0.0133657275195878244502213833856674174427_wp
        Omega(4) = 0.0233612862120284995966408049528073576084_wp
        Omega(5) = 0.0405263650731162850112884732389773034811_wp
        Omega(6) = 0.0699215304569996237322461983543231411886_wp
        Omega(7) = 0.1193636598582501377941627110357281082997_wp
        Omega(8) = 0.2011495081893048014103422826970302139671_wp
        Omega(9) = 0.3346541988448492033192665368046547769154_wp
        Omega(10) = 0.5505486506765955056260514188792853929044_wp
        Omega(11) = 0.8988127429424152543108693380613516410449_wp
        Omega(12) = 1.4685085046890018607669212147293080761301_wp
        Omega(13) = 2.4575387649296610392650463960961815246264_wp
        Omega(14) = 4.5958060777879913907928433047800353961065_wp

        Alpha(1) = 0.0010970334583767882200461239185738937518_wp
        Alpha(2) = 0.0060323406803296099254263007774512495018_wp
        Alpha(3) = 0.0161030878430710059278171573590721088465_wp
        Alpha(4) = 0.0340296553738375912029475930942190586848_wp
        Alpha(5) = 0.0651971871104269767386034557798968336328_wp
        Alpha(6) = 0.1191293407601383996688264310659377542834_wp
        Alpha(7) = 0.2116738402433355986876012747033293237564_wp
        Alpha(8) = 0.3685611864183791356642822673039816550045_wp
        Alpha(9) = 0.6310911840532924818396830202171798873678_wp
        Alpha(10) = 1.0651302197216859181036485715310391242383_wp
        Alpha(11) = 1.7759219386721268611663976377812446116877_wp
        Alpha(12) = 2.9355854175023709227727169990274092015170_wp
        Alpha(13) = 4.8489747171770592204000094316285185414017_wp
        Alpha(14) = 8.2038646820468288378691568141221068799496_wp

        Xi(1) = 1.006972908128617223309439476786_wp
        Xi(2) = 1.063402835956595894996781770381_wp
        Xi(3) = 1.179735523494981054852710766756_wp
        Xi(4) = 1.363068745404108952586942626883_wp
        Xi(5) = 1.624560616562614529271788010867_wp
        Xi(6) = 1.980294227041889729644917605977_wp
        Xi(7) = 2.452641125258710084024518782719_wp
        Xi(8) = 3.072141302588350317932450184166_wp
        Xi(9) = 3.879990181548534547211967704605_wp
        Xi(10) = 4.931386633383035074384981300000_wp
        Xi(11) = 6.300121770460542964392702813825_wp
        Xi(12) = 8.084896558266141935317206090872_wp
        Xi(13) = 10.418015657705428383787704671448_wp
        Xi(14) = 13.477368131632375061398909643628_wp
        Xi(15) = 17.502983446103226866999436062144_wp
        Xi(16) = 22.819894694028703964819304594869_wp
        Xi(17) = 29.869754296451002642878713722041_wp
        Xi(18) = 39.254168332327877319332110062078_wp
        Xi(19) = 51.793290567937741317172672950164_wp
        Xi(20) = 68.601960242918330327988041261733_wp
        Xi(21) = 91.180267185159977576247580799418_wp
        Xi(22) = 121.496116109682911303468966224273_wp
        Xi(23) = 161.981503757656784783791614756865_wp
        Xi(24) = 215.217085564527138091817093368263_wp
        Xi(25) = 282.746156105924997453682223635951_wp
        Xi(26) = 361.962294243132206655877425305334_wp
        Xi(27) = 440.461597656433940656262038260138_wp
        Xi(28) = 492.647718336578408993009503547000_wp

      case (17)
        ! ----- K = 14 , R = 6E2 -----

        RIni = 6.0e2_wp

        Omega(1) = 0.0024493762987277453332710217166756638818_wp
        Omega(2) = 0.0062525417642427215862345631341295826289_wp
        Omega(3) = 0.0117353802142847542872562097322763907670_wp
        Omega(4) = 0.0208593785557110738476465586904717852690_wp
        Omega(5) = 0.0368082225463150505831368304587858375498_wp
        Omega(6) = 0.0644690856910460385444995058679129584789_wp
        Omega(7) = 0.1115033107571565517817388860088989588348_wp
        Omega(8) = 0.1900944563293023498363630216512198956025_wp
        Omega(9) = 0.3195868584534482722821040201477416076159_wp
        Omega(10) = 0.5307743481279922103440588321277004979493_wp
        Omega(11) = 0.8739864827482794822316285177077332946283_wp
        Omega(12) = 1.4388462588113037831857865533891072118422_wp
        Omega(13) = 2.4237120872801188467896005285240335069830_wp
        Omega(14) = 4.5572977662850002100389246173506307968637_wp

        Alpha(1) = 0.0009445997794898587881681367728076842427_wp
        Alpha(2) = 0.0052106503828448096642906380189519399693_wp
        Alpha(3) = 0.0139993998059489099533507614099570037780_wp
        Alpha(4) = 0.0298816538645328309941592631257534229405_wp
        Alpha(5) = 0.0579751202785818767532666391517626180985_wp
        Alpha(6) = 0.1073721136819342762436799501579720583777_wp
        Alpha(7) = 0.1933219116222184735299789021256700038975_wp
        Alpha(8) = 0.3408201283025550454113244275289318352407_wp
        Alpha(9) = 0.5903549288936290426622194305039670325641_wp
        Alpha(10) = 1.0070203076871469613376777751945212457940_wp
        Alpha(11) = 1.6955193869260442238178221430189296370372_wp
        Alpha(12) = 2.8278951803735827743701419789701390072878_wp
        Alpha(13) = 4.7094833640003784113077511808143071903032_wp
        Alpha(14) = 8.0284416387115672035859614830144437291892_wp

        Xi(1) = 1.007202914800316681347147618197_wp
        Xi(2) = 1.065518106264580068900417464395_wp
        Xi(3) = 1.185864472955849129422525867206_wp
        Xi(4) = 1.375842005664037344045413924221_wp
        Xi(5) = 1.647420211021003179883981371034_wp
        Xi(6) = 2.017906664482518768518465313022_wp
        Xi(7) = 2.511474653003032098032981656033_wp
        Xi(8) = 3.161271359374571949391041769672_wp
        Xi(9) = 4.012222430049692694243451684954_wp
        Xi(10) = 5.124834585716500822295121908567_wp
        Xi(11) = 6.580454177572546470287306874525_wp
        Xi(12) = 8.488553460285218935435502807252_wp
        Xi(13) = 10.996840695516867314300568647667_wp
        Xi(14) = 14.305306950756526665653745045148_wp
        Xi(15) = 18.685818917109797742195920200459_wp
        Xi(16) = 24.509491622579109354126969400056_wp
        Xi(17) = 32.284978109522561379868266229209_wp
        Xi(18) = 42.711832230096845788164250379282_wp
        Xi(19) = 56.754102786836676555054514992094_wp
        Xi(20) = 75.738934911144978306962727288010_wp
        Xi(21) = 101.480179268624615393501464666315_wp
        Xi(22) = 136.406985035249315341143727664530_wp
        Xi(23) = 183.614613895448215277750314555760_wp
        Xi(24) = 246.576710827109510951182791416159_wp
        Xi(25) = 327.819419939240517175571554275848_wp
        Xi(26) = 425.085411433247325113660153306228_wp
        Xi(27) = 523.664787470510884104246684955797_wp
        Xi(28) = 590.496446832911009994226958497165_wp

      case (18)
        ! ----- K = 14 , R = 7E2 -----

        RIni = 7.0e2_wp

        Omega(1) = 0.0021594492963257279436261964165877369481_wp
        Omega(2) = 0.0055442742030058087287469853303434330627_wp
        Omega(3) = 0.0105226822930893956613025014279472202361_wp
        Omega(4) = 0.0189804496356877784811717068541581099339_wp
        Omega(5) = 0.0339808497464531976975593910439421563296_wp
        Omega(6) = 0.0602711111109771843840981434686776552212_wp
        Omega(7) = 0.1053851584135917551345658534078797430311_wp
        Omega(8) = 0.1814084463685945920356301988562996996279_wp
        Omega(9) = 0.3076509595262928073694085745470871984253_wp
        Omega(10) = 0.5149978645662088606329252760129833177416_wp
        Omega(11) = 0.8540615398602765621634096138148350974006_wp
        Omega(12) = 1.4149345700297016986375989167079580965947_wp
        Omega(13) = 2.3963776420822999551306364818259453386418_wp
        Omega(14) = 4.5261871826738335603847185328874047627323_wp

        Alpha(1) = 0.0008322695037266466425591101576351993341_wp
        Alpha(2) = 0.0046041779357021047074279007649266759472_wp
        Alpha(3) = 0.0124421375785095599149015859440081666065_wp
        Alpha(4) = 0.0267964431863254235673560974331786077585_wp
        Alpha(5) = 0.0525661414626129625800126082130203641896_wp
        Alpha(6) = 0.0984912639251911154608665199816641688813_wp
        Alpha(7) = 0.1793334990446931190370130279609917067773_wp
        Alpha(8) = 0.3194864137129840428105942429226615786320_wp
        Alpha(9) = 0.5587657706361327789360525508666199812069_wp
        Alpha(10) = 0.9616156945673737321627315421945780826718_wp
        Alpha(11) = 1.6322722752828760647229172642269645621127_wp
        Alpha(12) = 2.7426945317859525586955821152557177811104_wp
        Alpha(13) = 4.5986103520545233802434281766835511007230_wp
        Alpha(14) = 7.8885292427604850038803940837084383019828_wp

        Xi(1) = 1.007395036308558245916576490142_wp
        Xi(2) = 1.067286191578751572569226657716_wp
        Xi(3) = 1.190994301131639571968666435620_wp
        Xi(4) = 1.386553281899055304693738066568_wp
        Xi(5) = 1.666635507624293386177462272801_wp
        Xi(6) = 2.049614185777143367337554957786_wp
        Xi(7) = 2.561238663574738795656454692917_wp
        Xi(8) = 3.236949124687638550817991855979_wp
        Xi(9) = 4.124968053904481232480733199353_wp
        Xi(10) = 5.290525092357551192184805799457_wp
        Xi(11) = 6.821730658703404222547445057145_wp
        Xi(12) = 8.837769165998975847906016234390_wp
        Xi(13) = 11.500328233957221379241131486992_wp
        Xi(14) = 15.029616778645906094089013116388_wp
        Xi(15) = 19.726843000593919557131283859519_wp
        Xi(16) = 26.005941186914756785286684248604_wp
        Xi(17) = 34.438341860266177999611025839499_wp
        Xi(18) = 45.816220339762934548460959760519_wp
        Xi(19) = 61.240996678954597205735144171967_wp
        Xi(20) = 82.244714540690540029166388791282_wp
        Xi(21) = 110.947761605146295177393156450307_wp
        Xi(22) = 150.236827224452757556516502290833_wp
        Xi(23) = 203.878680969143173770663501898071_wp
        Xi(24) = 276.279675957493381005525279192625_wp
        Xi(25) = 371.058887831935104106850786820360_wp
        Xi(26) = 486.525575742089060909956188538672_wp
        Xi(27) = 605.866530293086994141127377133671_wp
        Xi(28) = 688.194268309426277874241151266688_wp

      case (19)
        ! ----- K = 14 , R = 8E2 -----

        RIni = 8.0e2_wp

        Omega(1) = 0.0019361209248683744086110819081388012552_wp
        Omega(2) = 0.0049971806200044293842755464135629850375_wp
        Omega(3) = 0.0095809917510057843200821196816274749608_wp
        Omega(4) = 0.0175087402037209378964898722971788025404_wp
        Omega(5) = 0.0317419702329472813214915772467294985404_wp
        Omega(6) = 0.0569117246472132654479920369633427768008_wp
        Omega(7) = 0.1004443163474548991628788538466032775887_wp
        Omega(8) = 0.1743385660966713684343412649191407126636_wp
        Omega(9) = 0.2978690083437383052901090141340034733730_wp
        Omega(10) = 0.5019910235145395863078261744494312779352_wp
        Omega(11) = 0.8375524167267300298727814023003190868621_wp
        Omega(12) = 1.3950481721401585319787888583853430191084_wp
        Omega(13) = 2.3735982301456124165563166261705418946804_wp
        Omega(14) = 4.5002652071230410369372110235275386003195_wp

        Alpha(1) = 0.0007457708419311785665749849310712099593_wp
        Alpha(2) = 0.0041365105869129268884829301808347956637_wp
        Alpha(3) = 0.0112381125521925794856191731414807311040_wp
        Alpha(4) = 0.0244007484568962324762367169586618764754_wp
        Alpha(5) = 0.0483397900781959522131445410736416690156_wp
        Alpha(6) = 0.0914999138377235021933283655448665427912_wp
        Alpha(7) = 0.1682345234675917510766334980742975346857_wp
        Alpha(8) = 0.3024303189839334995167932640391939003166_wp
        Alpha(9) = 0.5333315311260418145351130947418027972162_wp
        Alpha(10) = 0.9248228623483728231971859323756746107392_wp
        Alpha(11) = 1.5807297911278333924899736073754752396781_wp
        Alpha(12) = 2.6729249179396884245497184595663497930218_wp
        Alpha(13) = 4.5074643277945692153123313783424919165554_wp
        Alpha(14) = 7.7731799603489856813863523310459413551143_wp

        Xi(1) = 1.007559565169001995960963713461_wp
        Xi(2) = 1.068801236800597058798113769829_wp
        Xi(3) = 1.195394983364988433522518862517_wp
        Xi(4) = 1.395756730501326540322404312633_wp
        Xi(5) = 1.683179314898664369359350401201_wp
        Xi(6) = 2.076980012415584565246481907330_wp
        Xi(7) = 2.604310762598350100383709504470_wp
        Xi(8) = 3.302660903902071089493358568845_wp
        Xi(9) = 4.223214606111398587273003668585_wp
        Xi(10) = 5.435463824124180364123770647922_wp
        Xi(11) = 7.033658176068005277344064030132_wp
        Xi(12) = 9.145843285778266678123782362775_wp
        Xi(13) = 11.946552820585227116384485768208_wp
        Xi(14) = 15.674667998361805276372193951318_wp
        Xi(15) = 20.658703454100486614566922760616_wp
        Xi(16) = 27.352677294407203746964829527144_wp
        Xi(17) = 36.387268219058119207709456333077_wp
        Xi(18) = 48.642682895969507131384013121078_wp
        Xi(19) = 65.352026262405143154643472058751_wp
        Xi(20) = 88.245534595886393806896119684779_wp
        Xi(21) = 119.743166062967725131460650089821_wp
        Xi(22) = 163.184430354296004489489391175994_wp
        Xi(23) = 223.011753425256878488602829690990_wp
        Xi(24) = 304.593940685787349376134613976319_wp
        Xi(25) = 412.732558070269537980889751338509_wp
        Xi(26) = 546.494984986318698416685180063723_wp
        Xi(27) = 687.169966216998286734352063831466_wp
        Xi(28) = 785.754038180450499972984346186422_wp

      case (20)
        ! ----- K = 14 , R = 9E2 -----

        RIni = 9.0e2_wp

        Omega(1) = 0.0017583561916395429822060017142557053482_wp
        Omega(2) = 0.0045606051634619621230909035022871300669_wp
        Omega(3) = 0.0088258952559276655398936639942975013184_wp
        Omega(4) = 0.0163192947691258584884085088700933496852_wp
        Omega(5) = 0.0299149031953826850005912505250926791689_wp
        Omega(6) = 0.0541451570576183548560143289629653651218_wp
        Omega(7) = 0.0963434223496242410426622718166989045585_wp
        Omega(8) = 0.1684309556033862256650820640735410904654_wp
        Omega(9) = 0.2896469662987799439663860051608068602036_wp
        Omega(10) = 0.4910021987669276186312151966095740363016_wp
        Omega(11) = 0.8235447309834208091407735385924837601124_wp
        Omega(12) = 1.3781205638528317495824018767969221244130_wp
        Omega(13) = 2.3541736567665158426625876542104265354283_wp
        Omega(14) = 4.4781636543138096371620415769854162135744_wp

        Alpha(1) = 0.0006769402078263382473447450288824706055_wp
        Alpha(2) = 0.0037638910093806651992535250829743920242_wp
        Alpha(3) = 0.0102764752572631166838897109373773819385_wp
        Alpha(4) = 0.0224797445212613789045557948984010998572_wp
        Alpha(5) = 0.0449316043808094025504196091652664435401_wp
        Alpha(6) = 0.0858240735352995001098547200868438267207_wp
        Alpha(7) = 0.1591614518697570559540828763478437224421_wp
        Alpha(8) = 0.2883947892598595928509083718971339749260_wp
        Alpha(9) = 0.5122730073838949713159809928875176865404_wp
        Alpha(10) = 0.8941908057078154255367316571856406426377_wp
        Alpha(11) = 1.5376077062076304076279967270046711291798_wp
        Alpha(12) = 2.6143101743202715571759431334442069783108_wp
        Alpha(13) = 4.4306346423300903875248935026576191376080_wp
        Alpha(14) = 7.6757091479698113497057154752667429420399_wp

        Xi(1) = 1.007703131510779039070602181027_wp
        Xi(2) = 1.070123938594893971674483490464_wp
        Xi(3) = 1.199240777802830972800958442281_wp
        Xi(4) = 1.403810760739595360794759737466_wp
        Xi(5) = 1.697682166981737988302805786311_wp
        Xi(6) = 2.101020309966497421998166128532_wp
        Xi(7) = 2.642241663843272448134943952702_wp
        Xi(8) = 3.360690390160581039399878555685_wp
        Xi(9) = 4.310241741204164041701557552599_wp
        Xi(10) = 5.564277827666065487106222819946_wp
        Xi(11) = 7.222677715972033789856160401754_wp
        Xi(12) = 9.421658387554165060145949794901_wp
        Xi(13) = 12.347645846092340555892608033872_wp
        Xi(14) = 16.256927789513155738557181884119_wp
        Xi(15) = 21.503582659418284004845811629991_wp
        Xi(16) = 28.579431675928376918940920781154_wp
        Xi(17) = 38.171324390074186028859459796081_wp
        Xi(18) = 51.243525771498495510747339665159_wp
        Xi(19) = 69.155797265770647161420292547973_wp
        Xi(20) = 93.830428988181079658081884531384_wp
        Xi(21) = 127.980389928026863324839190738658_wp
        Xi(22) = 175.392737748039292997037641441693_wp
        Xi(23) = 241.187173599354448605414091844068_wp
        Xi(24) = 331.717331821975056599383080424559_wp
        Xi(25) = 453.041911439697934982362426126201_wp
        Xi(26) = 605.156632010292442058752016009748_wp
        Xi(27) = 767.655839022134085070092623936944_wp
        Xi(28) = 883.185893111137792332865359412608_wp

      case (21)
        ! ----- K = 14 , R = 1E3 -----

        RIni = 1.0e3_wp

        Omega(1) = 0.0016132124497670363360598869378707831412_wp
        Omega(2) = 0.0042033179673286300773109411439698579471_wp
        Omega(3) = 0.0082051681426387509475769890116003679736_wp
        Omega(4) = 0.0153343743590184841799105644466472142540_wp
        Omega(5) = 0.0283888068001417349352845679796852884635_wp
        Omega(6) = 0.0518157306160420973214076993673682380859_wp
        Omega(7) = 0.0928668402052448009646951412876525466800_wp
        Omega(8) = 0.1633932015987313481476632393007974997090_wp
        Omega(9) = 0.2825994389709872368541231091310095280278_wp
        Omega(10) = 0.4815408164843763549618337743707385811831_wp
        Omega(11) = 0.8114387138683526611471381884133080575339_wp
        Omega(12) = 1.3634496747327067076157799951907634294912_wp
        Omega(13) = 2.3373123988437141795165841173798071395140_wp
        Omega(14) = 4.4589806097080054219858324948688732547453_wp

        Alpha(1) = 0.0006207557929943959426693132709531436353_wp
        Alpha(2) = 0.0034593749933095318074656649744158531945_wp
        Alpha(3) = 0.0094888451804602900570287317590090214736_wp
        Alpha(4) = 0.0209005466717163312898746831136609358737_wp
        Alpha(5) = 0.0421152120539183988705766396171137344595_wp
        Alpha(6) = 0.0811052525009715691059583760058604440246_wp
        Alpha(7) = 0.1515714966963109508647647785672951670222_wp
        Alpha(8) = 0.2765844306591897970798790695745594803157_wp
        Alpha(9) = 0.4944571460594123973116172099384613147777_wp
        Alpha(10) = 0.8681492267517379013527963060870007439007_wp
        Alpha(11) = 1.5007904436022078916747324317526590675698_wp
        Alpha(12) = 2.5640827986459930748486257812146504875273_wp
        Alpha(13) = 4.3646061500119444805101398543456525658257_wp
        Alpha(14) = 7.5917614754962411591741877625594270284637_wp

        Xi(1) = 1.007830250880272755483260671561_wp
        Xi(2) = 1.071295646757362046513038789097_wp
        Xi(3) = 1.202650501122817269295840747478_wp
        Xi(4) = 1.410960228390045112100044388814_wp
        Xi(5) = 1.710575956024697986393098836366_wp
        Xi(6) = 2.122432998568170951173700311543_wp
        Xi(7) = 2.676099822013581461949138784462_wp
        Xi(8) = 3.412616073335494759882074999879_wp
        Xi(9) = 4.388324774083330777341149797621_wp
        Xi(10) = 5.680190837086783292326358507651_wp
        Xi(11) = 7.393300078694071852079827689153_wp
        Xi(12) = 9.671454294457193075207346311117_wp
        Xi(13) = 12.712182348927038999172878241950_wp
        Xi(14) = 16.788080861856720044436586114500_wp
        Xi(15) = 22.277332668577900672696179285026_wp
        Xi(16) = 29.707546328054613487754109790728_wp
        Xi(17) = 39.819093475399266270770004538804_wp
        Xi(18) = 53.656795027205435311135062192989_wp
        Xi(19) = 72.702510799078205239176497798326_wp
        Xi(20) = 99.065061737622908025202583814917_wp
        Xi(21) = 135.744157475194193174239387644775_wp
        Xi(22) = 186.968934242911004162124299909920_wp
        Xi(23) = 258.536220789325584867857088511300_wp
        Xi(24) = 357.801698996791233825520706091083_wp
        Xi(25) = 492.144128900693246547737302876158_wp
        Xi(26) = 662.639973155541552485292555729757_wp
        Xi(27) = 847.389394762888743639628330583946_wp
        Xi(28) = 980.498069961306024933112013286518_wp

      case (22)
        ! ----- K = 14 , R = 2E3 -----

        RIni = 2.0e3_wp

        Omega(1) = 0.0009157569104258143407238324450962002610_wp
        Omega(2) = 0.0024721937700800369844148039050191201227_wp
        Omega(3) = 0.0051478470787629635160354869827248336378_wp
        Omega(4) = 0.0103540882355857959877279915981984625972_wp
        Omega(5) = 0.0204407992989296262054837484412050940819_wp
        Omega(6) = 0.0393608842827519610437262635033173197030_wp
        Omega(7) = 0.0738597067220314193268905847422622912291_wp
        Omega(8) = 0.1353151948841131961152797064107389246601_wp
        Omega(9) = 0.2426483067657189412894350177829494441539_wp
        Omega(10) = 0.4271026836177867425840912890544487368061_wp
        Omega(11) = 0.7409052083663800540276958561669573555264_wp
        Omega(12) = 1.2771569837351899401670096700911471998552_wp
        Omega(13) = 2.2376078746285902639889114906779354896571_wp
        Omega(14) = 4.3455745783663723949725998796367321119760_wp

        Alpha(1) = 0.0003510368022941439417818036022346517289_wp
        Alpha(2) = 0.0019913702978629040273133015293699488168_wp
        Alpha(3) = 0.0056609555525626375320288213334463112147_wp
        Alpha(4) = 0.0131208337665729531203289977107917518850_wp
        Alpha(5) = 0.0279788053246742183667025499413383293756_wp
        Alpha(6) = 0.0569166589761033866168001092017414421775_wp
        Alpha(7) = 0.1118433745957511314911107802583245529604_wp
        Alpha(8) = 0.2135401128441132648114143474660231447615_wp
        Alpha(9) = 0.3976338058430778222402356136822021426269_wp
        Alpha(10) = 0.7243220713184966184569613112387287401361_wp
        Alpha(11) = 1.2945431454910699557757716715578055755032_wp
        Alpha(12) = 2.2792966515798136375829663347047926436062_wp
        Alpha(13) = 3.9865887787029058773966699869006902190449_wp
        Alpha(14) = 7.1077410216557896983392839107551708366373_wp

        Xi(1) = 1.008632189108310501157381167125_wp
        Xi(2) = 1.078699122282745128027260650416_wp
        Xi(3) = 1.224259205672618162972879740646_wp
        Xi(4) = 1.456458985960483048919568427326_wp
        Xi(5) = 1.793066161802028811575113187438_wp
        Xi(6) = 2.260302624265208433149143552043_wp
        Xi(7) = 2.895737656080540215267071779692_wp
        Xi(8) = 3.752322270362356595877276199680_wp
        Xi(9) = 4.903961828884370521917240859366_wp
        Xi(10) = 6.453465919941870727969540633140_wp
        Xi(11) = 8.544038831798453964040673813685_wp
        Xi(12) = 11.375914498639907963990769435014_wp
        Xi(13) = 15.230562231000130036899509189396_wp
        Xi(14) = 20.506116255519682754424048987119_wp
        Xi(15) = 27.769634485956145848392639940982_wp
        Xi(16) = 37.834861577064350164623096617333_wp
        Xi(17) = 51.878864203640160643327128298097_wp
        Xi(18) = 71.618450266567320973420951446542_wp
        Xi(19) = 99.578904116495437867651041585759_wp
        Xi(20) = 139.504710893621180989310204267895_wp
        Xi(21) = 196.984703432065028222419655890008_wp
        Xi(22) = 280.380132645586132661419398459657_wp
        Xi(23) = 402.098544586103016335032478423273_wp
        Xi(24) = 579.907368731582759324716391802212_wp
        Xi(25) = 836.447018090643819976381934111487_wp
        Xi(26) = 1189.547781771136276462996761438262_wp
        Xi(27) = 1612.179929966378417849348636536888_wp
        Xi(28) = 1948.167189464106106577823140924011_wp

      case (23)
        ! ----- K = 14 , R = 3E3 -----

        RIni = 3.0e3_wp

        Omega(1) = 0.0006582780968990024278469099211063414456_wp
        Omega(2) = 0.0018238955102013432973480472581821743017_wp
        Omega(3) = 0.0039689243401756151851419822385036439982_wp
        Omega(4) = 0.0083471503569080813148364263176492983121_wp
        Omega(5) = 0.0170917347246665210408657223709050221316_wp
        Omega(6) = 0.0339122562898936415295059101232544307436_wp
        Omega(7) = 0.0652812337962773135973011673360844042691_wp
        Omega(8) = 0.1222991048085924378522010118330864969494_wp
        Omega(9) = 0.2236897234768833826398792830048023461131_wp
        Omega(10) = 0.4007368414404818431865263478330874136191_wp
        Omega(11) = 0.7061527331295446461775777713398838386638_wp
        Omega(12) = 1.2340831943571578578362074596341813048639_wp
        Omega(13) = 2.1874725831200839232075494589224717856268_wp
        Omega(14) = 4.2885600188927733692791088859763704022043_wp

        Alpha(1) = 0.0002516319551055987430042176262621686500_wp
        Alpha(2) = 0.0014463966580943881962319767136085357784_wp
        Alpha(3) = 0.0042192863421776700386364223650803584320_wp
        Alpha(4) = 0.0101195652062268073377622880186510467837_wp
        Alpha(5) = 0.0223521001472474348827746510812164437709_wp
        Alpha(6) = 0.0469651666520016426651089758853574451791_wp
        Alpha(7) = 0.0949761173357285567214378617373338187235_wp
        Alpha(8) = 0.1859949564330128747916253545424680737597_wp
        Alpha(9) = 0.3542301980022188864283879533578769382984_wp
        Alpha(10) = 0.6583664765398028476052656732253609561667_wp
        Alpha(11) = 1.1980745909205334337986614823279296615510_wp
        Alpha(12) = 2.1438544125498438426297848069523865888186_wp
        Alpha(13) = 3.8044083660706669346510122586835223046364_wp
        Alpha(14) = 6.8722249361801352844636436856262662331574_wp

        Xi(1) = 1.009070286187634341219278300006_wp
        Xi(2) = 1.082752262413102746821912114594_wp
        Xi(3) = 1.236136394203963409640575910053_wp
        Xi(4) = 1.481606971602632887518267990057_wp
        Xi(5) = 1.838982070848160912945878919800_wp
        Xi(6) = 2.337698296741422804029958415484_wp
        Xi(7) = 3.020264301605273550002314641993_wp
        Xi(8) = 3.947093129638576850838627141371_wp
        Xi(9) = 5.203271703513132750897746126029_wp
        Xi(10) = 6.908359966048753764058382165203_wp
        Xi(11) = 9.230736714816151771909047418774_wp
        Xi(12) = 12.408663241603011520922117405519_wp
        Xi(13) = 16.781362732081296097394473854081_wp
        Xi(14) = 22.835273644526014512334821660033_wp
        Xi(15) = 31.273460901970722243189970690480_wp
        Xi(16) = 43.120824444532904872601219992134_wp
        Xi(17) = 59.885230718427018205274325879373_wp
        Xi(18) = 83.806858436595493028198333718137_wp
        Xi(19) = 118.248187319778805599856941199732_wp
        Xi(20) = 168.309049111282949179324575084138_wp
        Xi(21) = 241.801324455166201238265344386491_wp
        Xi(22) = 350.781042435316185057470050878692_wp
        Xi(23) = 513.865591281148380464749436669081_wp
        Xi(24) = 759.273411353206116425340610476269_wp
        Xi(25) = 1126.646365958049686772923791977519_wp
        Xi(26) = 1657.203863519424259065360161002900_wp
        Xi(27) = 2333.185126236751510520051056118973_wp
        Xi(28) = 2907.925413012566687065429960057372_wp

      case (24)
        ! ----- K = 14 , R = 4E3 -----

        RIni = 4.0e3_wp

        Omega(1) = 0.0005212677491514954218130321807623400732_wp
        Omega(2) = 0.0014754241414154630869342410938796494868_wp
        Omega(3) = 0.0033215711612985019576517681585969832270_wp
        Omega(4) = 0.0072120880495609244083537060082678138428_wp
        Omega(5) = 0.0151457452900341282069651715515290479885_wp
        Omega(6) = 0.0306764151466120554822278928414069909536_wp
        Omega(7) = 0.0600935751154314152806118487504249969788_wp
        Omega(8) = 0.1143044814464072772076160683374368431942_wp
        Omega(9) = 0.2118856020224212764887453813722473228154_wp
        Omega(10) = 0.3841247050738851271734289327675071490376_wp
        Omega(11) = 0.6840362336317067256372982664291981791393_wp
        Omega(12) = 1.2064609599820673001974169480376986030024_wp
        Omega(13) = 2.1551812818521882252759125364605097274762_wp
        Omega(14) = 4.2518383632143199927187759978863823562278_wp

        Alpha(1) = 0.0001987978935030617574186219645329502803_wp
        Alpha(2) = 0.0011552666132272471463946275898465598786_wp
        Alpha(3) = 0.0034409962587201822635370277456492993373_wp
        Alpha(4) = 0.0084711218688381321973243252496099664484_wp
        Alpha(5) = 0.0191961514636296384957847961996724173162_wp
        Alpha(6) = 0.0412654467728940122616526891413668565178_wp
        Alpha(7) = 0.0851279932993591171189553569453867254424_wp
        Alpha(8) = 0.1696344295121822774941250724922703341235_wp
        Alpha(9) = 0.3280570368244356148541580192601330168145_wp
        Alpha(10) = 0.6180615250656186619336238119437609839224_wp
        Alpha(11) = 1.1384402924603920947326310408875826851727_wp
        Alpha(12) = 2.0593131981714612325010116400392234936589_wp
        Alpha(13) = 3.6898183294620708629583949456076652495540_wp
        Alpha(14) = 6.7232681806355749491201090339842494358891_wp

        Xi(1) = 1.009365790770963671081628110215_wp
        Xi(2) = 1.085489655663102161295592573698_wp
        Xi(3) = 1.244177037777732447985414021563_wp
        Xi(4) = 1.498688212725789823277083856468_wp
        Xi(5) = 1.870299327967785467711180069639_wp
        Xi(6) = 2.390754196886843693792279452737_wp
        Xi(7) = 3.106133356644637027663136130684_wp
        Xi(8) = 4.082298770277047605600784230262_wp
        Xi(9) = 5.412577757664872825458385330322_wp
        Xi(10) = 7.229007754815780065899000206286_wp
        Xi(11) = 9.718940188868680564102109364910_wp
        Xi(12) = 13.149632487915218144979012659945_wp
        Xi(13) = 17.904924436335444478188283490283_wp
        Xi(14) = 24.540393461737699832073134054156_wp
        Xi(15) = 33.867127427260030704536664103443_wp
        Xi(16) = 47.080324238993898469662902073196_wp
        Xi(17) = 65.959033777349492022601396001846_wp
        Xi(18) = 93.180058469906382231928709813928_wp
        Xi(19) = 132.817624606849302215660912906969_wp
        Xi(20) = 191.147594376094313806113689224730_wp
        Xi(21) = 277.954733203012668507625448910403_wp
        Xi(22) = 408.658974179089547557053307968999_wp
        Xi(23) = 607.704138264774603617190962268069_wp
        Xi(24) = 913.506450357726390631984969559198_wp
        Xi(25) = 1383.292352342555111777144816187501_wp
        Xi(26) = 2085.366194992934200325152005461860_wp
        Xi(27) = 3022.047473583720196943502855901897_wp
        Xi(28) = 3861.321520022056027920953624743561_wp

      case (25)
        ! ----- K = 14 , R = 5E3 -----

        RIni = 5.0e3_wp

        Omega(1) = 0.0004352427581767983618992876244154638377_wp
        Omega(2) = 0.0012548576180906004206129180809578538280_wp
        Omega(3) = 0.0029046595378673015925571838967769733308_wp
        Omega(4) = 0.0064645586114702151104412458735432034374_wp
        Omega(5) = 0.0138392720069440783444312203197268384969_wp
        Omega(6) = 0.0284704841665357742747430444194001175617_wp
        Omega(7) = 0.0565119417861124794805877374209712726838_wp
        Omega(8) = 0.1087244476673139367912316981257259129734_wp
        Omega(9) = 0.2035678909477675178652271090196101965830_wp
        Omega(10) = 0.3723216049428287995005839444706552399111_wp
        Omega(11) = 0.6682119500204294828692337671061096671110_wp
        Omega(12) = 1.1865913419286377433877963105679498312384_wp
        Omega(13) = 2.1318813562162843423634928097243346201140_wp
        Omega(14) = 4.2253409304934068796422153013736533466727_wp

        Alpha(1) = 0.0001656550983736506384406475398318389636_wp
        Alpha(2) = 0.0009719040519362571361550429919085730690_wp
        Alpha(3) = 0.0029465717920729978218602967228109679176_wp
        Alpha(4) = 0.0074093762335677205316918764437666977685_wp
        Alpha(5) = 0.0171307513468537186462675638715102977017_wp
        Alpha(6) = 0.0374775400711816419706658501387330773014_wp
        Alpha(7) = 0.0784921372444625901570558305919700359254_wp
        Alpha(8) = 0.1584754097352566251801593219528108136274_wp
        Alpha(9) = 0.3100135420433681162971292777363885306841_wp
        Alpha(10) = 0.5900153839041758170916827930252424039281_wp
        Alpha(11) = 1.0966083910583347686045277424859989423567_wp
        Alpha(12) = 1.9996081150871290993799658797236418195098_wp
        Alpha(13) = 3.6084591233871420245474942456453959493956_wp
        Alpha(14) = 6.6171039433964087585204882913103574537672_wp

        Xi(1) = 1.009585803629730712833306727205_wp
        Xi(2) = 1.087529567163898100564875526253_wp
        Xi(3) = 1.250179023934667543824922342210_wp
        Xi(4) = 1.511468018403788665334798457796_wp
        Xi(5) = 1.893799457175245379075487206499_wp
        Xi(6) = 2.430707808847820036333639559523_wp
        Xi(7) = 3.171067336009214557681137880429_wp
        Xi(8) = 4.185021458606984867505118597109_wp
        Xi(9) = 5.572423445176040247405618854870_wp
        Xi(10) = 7.475265965197596537954866091624_wp
        Xi(11) = 10.096154571210241612626290486077_wp
        Xi(12) = 13.725868979493426130789046624159_wp
        Xi(13) = 18.784768317560292855486481222727_wp
        Xi(14) = 25.885571274187428361321372527470_wp
        Xi(15) = 35.929568389254655503411983730189_wp
        Xi(16) = 50.255736750058273287766752446259_wp
        Xi(17) = 70.874805007675333271000006618578_wp
        Xi(18) = 100.841377956485743032366553251222_wp
        Xi(19) = 144.854072196638575767413392725302_wp
        Xi(20) = 210.236116070503484667963967069682_wp
        Xi(21) = 308.558668847189461026392365283755_wp
        Xi(22) = 458.344739094307165150343763571072_wp
        Xi(23) = 689.531694272874184525878860085868_wp
        Xi(24) = 1050.420345319029590491766157356324_wp
        Xi(25) = 1615.975892449078113122240552002040_wp
        Xi(26) = 2483.892857072607095014760147932975_wp
        Xi(27) = 3685.033596941953352832399559702026_wp
        Xi(28) = 4809.210718299243535689413420186611_wp

      case (26)
        ! ----- K = 14 , R = 6E3 -----

        RIni = 6.0e3_wp

        Omega(1) = 0.0003757964231191720894239006572337333245_wp
        Omega(2) = 0.0011013938591955864320308362087549869379_wp
        Omega(3) = 0.0026102631943494946954748291189155051661_wp
        Omega(4) = 0.0059271464833728378313226672468444489184_wp
        Omega(5) = 0.0128860046532219340670945327900764887374_wp
        Omega(6) = 0.0268419977564718356200304527625632999843_wp
        Omega(7) = 0.0538421401369425328202976670927126789934_wp
        Omega(8) = 0.1045302971427407760522036543615609360813_wp
        Omega(9) = 0.1972705244360379961236773294075419471483_wp
        Omega(10) = 0.3633288826899753685680059117624196574070_wp
        Omega(11) = 0.6560911744653233455696131337653298487567_wp
        Omega(12) = 1.1713098067261392495495495036372801678226_wp
        Omega(13) = 2.1139192519073143272049369878473612516245_wp
        Omega(14) = 4.2049130034262412448052559366118430261849_wp

        Alpha(1) = 0.0001427695429054092805663457412044918537_wp
        Alpha(2) = 0.0008448608471347996179937467201622597335_wp
        Alpha(3) = 0.0026014627740257254616775862976485456102_wp
        Alpha(4) = 0.0066596448160948629553583478710154786029_wp
        Alpha(5) = 0.0156533799168818287282428300399850229496_wp
        Alpha(6) = 0.0347350649212882243940553762123757408631_wp
        Alpha(7) = 0.0736359011260507862679553121865172471416_wp
        Alpha(8) = 0.1502321267433053322815604546658363460665_wp
        Alpha(9) = 0.2965751784347268078659894813275954561504_wp
        Alpha(10) = 0.5689780633687246379497255699408952978047_wp
        Alpha(11) = 1.0650374633237884148097632253104904975771_wp
        Alpha(12) = 1.9543163700723141009687675029304898544069_wp
        Alpha(13) = 3.5464904138887233075831673101419028171222_wp
        Alpha(14) = 6.5360082020505321270982534986870859938790_wp

        Xi(1) = 1.009759423645731589002511041642_wp
        Xi(2) = 1.089140442439909128185572539937_wp
        Xi(3) = 1.254924746323060942934136952065_wp
        Xi(4) = 1.521590832368567632591239979778_wp
        Xi(5) = 1.912455651212389301420863096936_wp
        Xi(6) = 2.462512683048163704109881355642_wp
        Xi(7) = 3.222921273351894386541283132885_wp
        Xi(8) = 4.267348507491065362409038419500_wp
        Xi(9) = 5.701040713877533227735283638538_wp
        Xi(10) = 7.674266617894350726684915553477_wp
        Xi(11) = 10.402397579327837314616678199286_wp
        Xi(12) = 14.196024279887322836792018510010_wp
        Xi(13) = 19.506473792871007164745345008328_wp
        Xi(14) = 26.995290139305076636613844875967_wp
        Xi(15) = 37.641456365932051436606897709680_wp
        Xi(16) = 52.908858369821286385392689055607_wp
        Xi(17) = 75.011344579328146309205038733126_wp
        Xi(18) = 107.338076152353949305195968122462_wp
        Xi(19) = 155.146627326469985913570681645979_wp
        Xi(20) = 226.709021208764812074787364792883_wp
        Xi(21) = 335.235835316554947133083075527793_wp
        Xi(22) = 502.140572942818161933375975536364_wp
        Xi(23) = 762.565150234866601430816501761001_wp
        Xi(24) = 1174.378489516354740151982127827068_wp
        Xi(25) = 1830.239271394546196281005734363134_wp
        Xi(26) = 2858.756200324219738595488138344081_wp
        Xi(27) = 4326.171464965746435549220905159018_wp
        Xi(28) = 5752.152994166337293346913384084473_wp

      case (27)
        ! ----- K = 14 , R = 7E3 -----

        RIni = 7.0e3_wp

        Omega(1) = 0.0003320523602446911839109026524973281025_wp
        Omega(2) = 0.0009877883512064985389095867038440623720_wp
        Omega(3) = 0.0023894838975166594517009308485527840915_wp
        Omega(4) = 0.0055180549376580071153728247312225674293_wp
        Omega(5) = 0.0121516088991089741446678684257121094703_wp
        Omega(6) = 0.0255755558339171696876826857594555697517_wp
        Omega(7) = 0.0517496526777221388414870067995465774402_wp
        Omega(8) = 0.1012210601197211070544036345952854105690_wp
        Omega(9) = 0.1922728458422537548244387120566223359219_wp
        Omega(10) = 0.3561559590722259835079716722150422469895_wp
        Omega(11) = 0.6463818938212136978069093096177510915368_wp
        Omega(12) = 1.1590285100821415355297830385161717003939_wp
        Omega(13) = 2.0994562104528666176585238734020322226570_wp
        Omega(14) = 4.1884637421121013621180817487754666217370_wp

        Alpha(1) = 0.0001259399039924061092209764387331825952_wp
        Alpha(2) = 0.0007511606113890968361609065965991266500_wp
        Alpha(3) = 0.0023452526412624827559112299655078026639_wp
        Alpha(4) = 0.0060974369005672144819378609290755344219_wp
        Alpha(5) = 0.0145334809715014825981461901033275907480_wp
        Alpha(6) = 0.0326353621703552626704661256823447601505_wp
        Alpha(7) = 0.0698852450077041354036693800166890611081_wp
        Alpha(8) = 0.1438170738021750018223541417450306312276_wp
        Alpha(9) = 0.2860481254040681115523043348547460595910_wp
        Alpha(10) = 0.5524038197218400129224688543949639552011_wp
        Alpha(11) = 1.0400418005232401251697674227081336084666_wp
        Alpha(12) = 1.9183099739324458423090150049894475614565_wp
        Alpha(13) = 3.4970663804887805921079585447230897443660_wp
        Alpha(14) = 6.4711800508423940707369848013286173227243_wp

        Xi(1) = 1.009901823360759665873694068683_wp
        Xi(2) = 1.090462394415279327318686608539_wp
        Xi(3) = 1.258823195298802149295555596886_wp
        Xi(4) = 1.529918399423402316388877208997_wp
        Xi(5) = 1.927830548167165567374423118530_wp
        Xi(6) = 2.488781324253722668400634621122_wp
        Xi(7) = 3.265859157088403253062225406467_wp
        Xi(8) = 4.335716246503774000658343323877_wp
        Xi(9) = 5.808189936932905584644759588997_wp
        Xi(10) = 7.840628772672288646430893699346_wp
        Xi(11) = 10.659370863600591177247634000480_wp
        Xi(12) = 14.592121141847086769789687554777_wp
        Xi(13) = 20.117119280001803086413425525336_wp
        Xi(14) = 27.938587894110425768195993256882_wp
        Xi(15) = 39.103863869794267798390974277822_wp
        Xi(16) = 55.187487095897049659692923384569_wp
        Xi(17) = 78.584645758482612597217720207254_wp
        Xi(18) = 112.985602168826408296664620678484_wp
        Xi(19) = 164.155421665906010489366373406028_wp
        Xi(20) = 241.236176889214221849555563892409_wp
        Xi(21) = 358.957972477610030737071866724364_wp
        Xi(22) = 541.446129752940204127309442583282_wp
        Xi(23) = 828.794889700391907649823508563713_wp
        Xi(24) = 1288.136425684413921111115541862091_wp
        Xi(25) = 2029.678388672859771224565861302835_wp
        Xi(26) = 3213.976430433580548928773623629240_wp
        Xi(27) = 4948.304895005864610268986325536389_wp
        Xi(28) = 6690.550796528218577385871412843699_wp

      case (28)
        ! ----- K = 14 , R = 8E3 -----

        RIni = 8.0e3_wp

        Omega(1) = 0.0002983989963563436384249721199839536223_wp
        Omega(2) = 0.0008999186728744597156233732903295174932_wp
        Omega(3) = 0.0022167291373876102643848982681404335970_wp
        Omega(4) = 0.0051938397507288718906493820131470284807_wp
        Omega(5) = 0.0115637280016053537487651287941903266443_wp
        Omega(6) = 0.0245537990526819006199658618899706041816_wp
        Omega(7) = 0.0500504653739266480064575622760392192845_wp
        Omega(8) = 0.0985188628895173317413357788607797260738_wp
        Omega(9) = 0.1881721889260020816905761772974692291882_wp
        Omega(10) = 0.3502457688554311079920117882879893045356_wp
        Omega(11) = 0.6383535275892021627723582621882769672084_wp
        Omega(12) = 1.1488458057525466396702612525970721435442_wp
        Omega(13) = 2.0874455976907589924498886979620237980271_wp
        Omega(14) = 4.1748030920369040923899317530043617807678_wp

        Alpha(1) = 0.0001129997667730506017741767990734003818_wp
        Alpha(2) = 0.0006789271013530558473627197612482175870_wp
        Alpha(3) = 0.0021465653166690079212435335711053885355_wp
        Alpha(4) = 0.0056575785323294799351513835721644296584_wp
        Alpha(5) = 0.0136491045391653851625568256815401735338_wp
        Alpha(6) = 0.0309631884975518033795702840110797282591_wp
        Alpha(7) = 0.0668762843902247179151202005620291402010_wp
        Alpha(8) = 0.1386378928877147248863816939024040664208_wp
        Alpha(9) = 0.2775023855512674345210038379105910166800_wp
        Alpha(10) = 0.5388849900439262755134223192854392436857_wp
        Alpha(11) = 1.0195707670958165299299538752997307256010_wp
        Alpha(12) = 1.8887208600165844695623063032030586327892_wp
        Alpha(13) = 3.4563421263845536630585075776878056785790_wp
        Alpha(14) = 6.4176615144620159664895497897418863431085_wp

        Xi(1) = 1.010021876087461640343796720476_wp
        Xi(2) = 1.091577382099120778133349851657_wp
        Xi(3) = 1.262114131007728140706179720265_wp
        Xi(4) = 1.536956603474498366575934360512_wp
        Xi(5) = 1.940844488310749692749554440319_wp
        Xi(6) = 2.511056295354720912420612433102_wp
        Xi(7) = 3.302346913890639810290617006672_wp
        Xi(8) = 4.393953532044695498536002187251_wp
        Xi(9) = 5.899705249240140257378089971851_wp
        Xi(10) = 7.983125009359271711563421636271_wp
        Xi(11) = 10.880166731198383961096154415316_wp
        Xi(12) = 14.933595748408965791370617282041_wp
        Xi(13) = 20.645449204978822688172357402436_wp
        Xi(14) = 28.757883568506492941857510814430_wp
        Xi(15) = 40.379322561448514686266442197393_wp
        Xi(16) = 57.183783216903668136094074370135_wp
        Xi(17) = 81.730459890198066286171041383568_wp
        Xi(18) = 117.983894588977728669354494428489_wp
        Xi(19) = 172.174916684593830182303797471377_wp
        Xi(20) = 254.250726574617395767119454319527_wp
        Xi(21) = 380.360409158257794598467782520856_wp
        Xi(22) = 577.188226714989345456086056174172_wp
        Xi(23) = 889.557458674596053360694725142821_wp
        Xi(24) = 1393.576918824347966596732817379234_wp
        Xi(25) = 2216.802704432670242828962159364892_wp
        Xi(26) = 3552.448127287603106605473612944479_wp
        Xi(27) = 5553.570815507726670556820636193152_wp
        Xi(28) = 7624.710568402647891339540819899412_wp

      case (29)
        ! ----- K = 14 , R = 9E3 -----

        RIni = 9.0e3_wp

        Omega(1) = 0.0002716356155158930017149898823204667853_wp
        Omega(2) = 0.0008296965023838813597571069922455289092_wp
        Omega(3) = 0.0020772138991470568411898948340058024264_wp
        Omega(4) = 0.0049290823178676729135637619473586656937_wp
        Omega(5) = 0.0110795308513761144015132356632769017324_wp
        Omega(6) = 0.0237065990394166789337213269509008206626_wp
        Omega(7) = 0.0486337493334717352138057283594818613892_wp
        Omega(8) = 0.0962551893464257460184723205753876129620_wp
        Omega(9) = 0.1847228594633214214786737228379820408009_wp
        Omega(10) = 0.3452565701047216811597154878521465093399_wp
        Omega(11) = 0.6315558344955877121112280958570295297250_wp
        Omega(12) = 1.1402040860907487576951166441396878781234_wp
        Omega(13) = 2.0772388473542544424198241781454044030397_wp
        Omega(14) = 4.1631936182286419828764423289868545907666_wp

        Alpha(1) = 0.0001027140684192060395268882053794370218_wp
        Alpha(2) = 0.0006213754330576066933856348433341115767_wp
        Alpha(3) = 0.0019873999833585964459191006408617097279_wp
        Alpha(4) = 0.0053024100910098172180148767829166445154_wp
        Alpha(5) = 0.0129291547668094474290547747120927457587_wp
        Alpha(6) = 0.0295919326841182352423028972462915042740_wp
        Alpha(7) = 0.0643932006648959625745554623266198035481_wp
        Alpha(8) = 0.1343406420206151061187216724790438604487_wp
        Alpha(9) = 0.2703785774775263907039428434853078897504_wp
        Alpha(10) = 0.5275698886607571892748413944218555116095_wp
        Alpha(11) = 1.0023772597521848588342577301979474668769_wp
        Alpha(12) = 1.8637971960121483040468542968781662239053_wp
        Alpha(13) = 3.4219610092511837523554518725177331361920_wp
        Alpha(14) = 6.3724060819922240706154881184630767165800_wp

        Xi(1) = 1.010125201560989707718490882016_wp
        Xi(2) = 1.092537381742422711457772555033_wp
        Xi(3) = 1.264949688541450199060107995042_wp
        Xi(4) = 1.543027040277458050254559551728_wp
        Xi(5) = 1.952083353517166921121256395466_wp
        Xi(6) = 2.530322643622601130625887622116_wp
        Xi(7) = 3.333963617006754299749299308431_wp
        Xi(8) = 4.444519836938108185535323269022_wp
        Xi(9) = 5.979346217389822784517988685238_wp
        Xi(10) = 8.107437542282512363787938536319_wp
        Xi(11) = 11.073296154614077048405906822381_wp
        Xi(12) = 15.233137042066466934446544723158_wp
        Xi(13) = 21.110326514049565119895968123842_wp
        Xi(14) = 29.481168875419115939229275369371_wp
        Xi(15) = 41.509331343108100988853470525441_wp
        Xi(16) = 58.959240998940860505050931550386_wp
        Xi(17) = 84.540024245553541636100103318086_wp
        Xi(18) = 122.468349414466947236435068191440_wp
        Xi(19) = 179.406107152853319552932020997105_wp
        Xi(20) = 266.050786687141597564032480249807_wp
        Xi(21) = 399.884748285057179534307536528104_wp
        Xi(22) = 610.018072850112541394285159412902_wp
        Xi(23) = 945.803693317025847397960802709349_wp
        Xi(24) = 1492.058339914736600273492683754739_wp
        Xi(25) = 2393.457106528536410872121109605359_wp
        Xi(26) = 3876.353956281850738907479581030202_wp
        Xi(27) = 6143.641436161321587317019066176726_wp
        Xi(28) = 8554.875233509914672858087669737870_wp

      case (30)
        ! ----- K = 14 , R = 1E4 -----

        RIni = 1.0e4_wp

        Omega(1) = 0.0002497975537203257548106469194482484597_wp
        Omega(2) = 0.0007721386329534672507321604916288859677_wp
        Omega(3) = 0.0019617570439813717404042820542306446718_wp
        Omega(4) = 0.0047078208097291976323748132534502408575_wp
        Omega(5) = 0.0106718522991160866575884361671933131532_wp
        Omega(6) = 0.0229891216461233004154999766746603206968_wp
        Omega(7) = 0.0474281748230232914284056327386407403424_wp
        Omega(8) = 0.0943209541135432382564325257123805101855_wp
        Omega(9) = 0.1817649964446676749473897141684552991592_wp
        Omega(10) = 0.3409650001887978614726996962058791496020_wp
        Omega(11) = 0.6256934039371224431642752594751044625809_wp
        Omega(12) = 1.1327363996341790508852118146698728651245_wp
        Omega(13) = 2.0684084060135147842881403468240364418307_wp
        Omega(14) = 4.1531491773382185369971331123650770678069_wp

        Alpha(1) = 0.0000943250416165216137989618163403691270_wp
        Alpha(2) = 0.0005743348523820048554984620892485680521_wp
        Alpha(3) = 0.0018566487446835667285619298253439291013_wp
        Alpha(4) = 0.0050085418932687886600417590931416667033_wp
        Alpha(5) = 0.0123291275938582904669519159715496137153_wp
        Alpha(6) = 0.0284417237162022576080406578713510867829_wp
        Alpha(7) = 0.0622988806916012419749193937401621212757_wp
        Alpha(8) = 0.1306990307731739439390259219231360532376_wp
        Alpha(9) = 0.2643170553011205327234404982572613107550_wp
        Alpha(10) = 0.5179082172316745659077473484810383297372_wp
        Alpha(11) = 0.9876519994633035921089390940519336936632_wp
        Alpha(12) = 1.8423980004705068490734182207191338420671_wp
        Alpha(13) = 3.3923836281365733233243708877324706918444_wp
        Alpha(14) = 6.3334194940727337611377500969922493823105_wp

        Xi(1) = 1.010215573145741349797392671928_wp
        Xi(2) = 1.093377313015473438918140347553_wp
        Xi(3) = 1.267432169816699997848784275956_wp
        Xi(4) = 1.548346261731285797183348129646_wp
        Xi(5) = 1.961942329403673889397750895736_wp
        Xi(6) = 2.547246361548702414407535643548_wp
        Xi(7) = 3.361779147118380815274785167190_wp
        Xi(8) = 4.489086559209831966184983675916_wp
        Xi(9) = 6.049674242423618930695333739678_wp
        Xi(10) = 8.217449116291621392417499070149_wp
        Xi(11) = 11.244603278428059654064918859717_wp
        Xi(12) = 15.499491141598152142873356940100_wp
        Xi(13) = 21.524801679202383516281704700646_wp
        Xi(14) = 30.127891145613154240401332373267_wp
        Xi(15) = 42.522868854636181872030942230367_wp
        Xi(16) = 60.557087013139544104389777601227_wp
        Xi(17) = 87.077791252557945979317910456530_wp
        Xi(18) = 126.535224264338712396593678022327_wp
        Xi(19) = 185.992772931924037566053797831955_wp
        Xi(20) = 276.851457123846353164386968614963_wp
        Xi(21) = 417.852288133038502038196781995794_wp
        Xi(22) = 640.413851445001130913148301715410_wp
        Xi(23) = 998.239223748491957155515308386384_wp
        Xi(24) = 1584.601549504330486595193860921427_wp
        Xi(25) = 2561.050952771884144043568198867433_wp
        Xi(26) = 4187.400841055936970835915644784109_wp
        Xi(27) = 6719.869978176123153978949176234892_wp
        Xi(28) = 9481.242392326531422952484717825428_wp

      case (31)
        ! ----- K = 14 , R = 7E5 (Best : R = 669250) -----

        RIni = 7.0e5_wp

        Omega(1) = 0.0000184753234608245182832378360746668905_wp
        Omega(2) = 0.0001213861645056389429640852281589141065_wp
        Omega(3) = 0.0004969107802932144441376842338830427392_wp
        Omega(4) = 0.0016153616951963146559925689656200820110_wp
        Omega(5) = 0.0045476671955009153315327798984546436856_wp
        Omega(6) = 0.0115740755102117943190983846007968693925_wp
        Omega(7) = 0.0273041506300428478152468418638232350304_wp
        Omega(8) = 0.0606755970684782298768761375079572317759_wp
        Omega(9) = 0.1284366973240136942301060571369752949522_wp
        Omega(10) = 0.2611402694575218794929144516814290000184_wp
        Omega(11) = 0.5137270825249222130524581753086010849074_wp
        Omega(12) = 0.9871481159900666525102222448317945691088_wp
        Omega(13) = 1.8941325485709733519130645862382777977473_wp
        Omega(14) = 3.9547772395130827950086166433507628426014_wp

        Alpha(1) = 0.0000057793103608868070374277361758280072_wp
        Alpha(2) = 0.0000635782132566041314606087872052329196_wp
        Alpha(3) = 0.0003364910016085842384488450805936701760_wp
        Alpha(4) = 0.0012987332474693207431947977541740468777_wp
        Alpha(5) = 0.0041590436292552776221117746564132078291_wp
        Alpha(6) = 0.0117340040293272211121894041396807040201_wp
        Alpha(7) = 0.0301622120920925923808104951451269926110_wp
        Alpha(8) = 0.0721391388152867440501062219759820237641_wp
        Alpha(9) = 0.1628287566381473312857475904680271128200_wp
        Alpha(10) = 0.3504007079130461677302175310666498830869_wp
        Alpha(11) = 0.7245973001913729933439606289979906250664_wp
        Alpha(12) = 1.4504311395006661014822971211835067606444_wp
        Alpha(13) = 2.8398376292459660175278302052248591280659_wp
        Alpha(14) = 5.5949042429760907193039831675207551597850_wp

        Xi(1) = 1.012202011747291133833374920936_wp
        Xi(2) = 1.111908257588833184056724412425_wp
        Xi(3) = 1.322577398019902154033093644170_wp
        Xi(4) = 1.667625646862291010296373838795_wp
        Xi(5) = 2.185688370457748422445373726042_wp
        Xi(6) = 2.936997174736573466445263558278_wp
        Xi(7) = 4.013522612873748518242528193767_wp
        Xi(8) = 5.553984255286706629445997451811_wp
        Xi(9) = 7.767175898508397968895977658832_wp
        Xi(10) = 10.969177894241897386513884260495_wp
        Xi(11) = 15.643182022852918929868570874220_wp
        Xi(12) = 22.536802248036572025868817803484_wp
        Xi(13) = 32.822981032538749612131878663490_wp
        Xi(14) = 48.370851797949895544553866599813_wp
        Xi(15) = 72.211172446181928784303405421952_wp
        Xi(16) = 109.355112104990656117509129074961_wp
        Xi(17) = 168.270238675717422119348931630611_wp
        Xi(18) = 263.627460109292646645018365347823_wp
        Xi(19) = 421.567669726816232317334609547288_wp
        Xi(20) = 690.250435615596447780362154844624_wp
        Xi(21) = 1161.871689858553022167697577060608_wp
        Xi(22) = 2021.330299815971881960408040868060_wp
        Xi(23) = 3661.662830963089881430860828004370_wp
        Xi(24) = 6978.532556574037406083022005987004_wp
        Xi(25) = 14229.940028609742074472421791142551_wp
        Xi(26) = 31896.259376488910900704354389745276_wp
        Xi(27) = 82806.812140749924971316886512795463_wp
        Xi(28) = 291096.000256029446035199725884012878_wp

    end select

  case (15)
    ! ===== K = 15 =====

    select case (InitR)

      case default !(1:3)
        ! ----- K = 15 , R = 1E1 -----

        RIni = 1.0e1_wp

        Omega(1) = 0.0571511962479834348032161923242044920812_wp
        Omega(2) = 0.1347609820353712418484776885962972414745_wp
        Omega(3) = 0.2167924173410872770034491657797914854200_wp
        Omega(4) = 0.3061300554975517251767719306698722903093_wp
        Omega(5) = 0.4065242973198527684909695661952255818505_wp
        Omega(6) = 0.5229143871001192803616049176174129797801_wp
        Omega(7) = 0.6619515007536857417172124207205285983946_wp
        Omega(8) = 0.8326876570348027339560359472159234428545_wp
        Omega(9) = 1.0475551694593583445472978477042147460452_wp
        Omega(10) = 1.3240406119874787986990388732877477195871_wp
        Omega(11) = 1.6881745466647175900418847982464853885176_wp
        Omega(12) = 2.1829224353081294340579027180027082977176_wp
        Omega(13) = 2.8914979429762138795137471625196212698938_wp
        Omega(14) = 4.0201557355540847760351497175435042663594_wp
        Omega(15) = 6.3896647738393501912181060564677181901061_wp

        Alpha(1) = 0.0222363742099818770173831009723253160359_wp
        Alpha(2) = 0.1179176691974456763197594451331173814879_wp
        Alpha(3) = 0.2932181611149396961626262259059672032890_wp
        Alpha(4) = 0.5539290777205830730968785791112907190836_wp
        Alpha(5) = 0.9091507799137611410024428781895267093205_wp
        Alpha(6) = 1.3722908032211439577525979860439520052751_wp
        Alpha(7) = 1.9625022213863007779699118571237193009438_wp
        Alpha(8) = 2.7067233856900886972921688178317367601267_wp
        Alpha(9) = 3.6425339141137162792850878734185471330420_wp
        Alpha(10) = 4.8222835928813742919357698824001090542879_wp
        Alpha(11) = 6.3196587298171715231173695137556478584884_wp
        Alpha(12) = 8.2417775368720227408339540353665597649524_wp
        Alpha(13) = 10.7556773355908089108423641988565577776171_wp
        Alpha(14) = 14.1604753425802106564165838697988419880858_wp
        Alpha(15) = 19.1750635168336970078767356184812342689838_wp

        Xi(1) = 1.001742002519599818990622686066_wp
        Xi(2) = 1.016256515869695547051899942037_wp
        Xi(3) = 1.044956843621521282270032948691_wp
        Xi(4) = 1.089034086654614941376550962815_wp
        Xi(5) = 1.148324279348726706211081260989_wp
        Xi(6) = 1.224653365407459770287414146939_wp
        Xi(7) = 1.317686476933486784375929024016_wp
        Xi(8) = 1.430516793639161399090235782250_wp
        Xi(9) = 1.562968661557137332461693557395_wp
        Xi(10) = 1.717382025458436712304390037165_wp
        Xi(11) = 1.896294827894254538696618139770_wp
        Xi(12) = 2.101892136857211478787738068164_wp
        Xi(13) = 2.335688311894927313107014898996_wp
        Xi(14) = 2.600941080685604287186379890429_wp
        Xi(15) = 2.902340297336362463006403933363_wp
        Xi(16) = 3.238207746322455995438213682824_wp
        Xi(17) = 3.617001587679771039580967073590_wp
        Xi(18) = 4.036043893618174631734984991738_wp
        Xi(19) = 4.497724617924908842477027537399_wp
        Xi(20) = 5.002668369534430500150667953108_wp
        Xi(21) = 5.550378837989716308571469527422_wp
        Xi(22) = 6.132242691307296785559882712491_wp
        Xi(23) = 6.741834101360709671866416892172_wp
        Xi(24) = 7.363480746778404697865105799082_wp
        Xi(25) = 7.980453048090865568724855405591_wp
        Xi(26) = 8.562938576889232315803024420120_wp
        Xi(27) = 9.083638083161566919784823959105_wp
        Xi(28) = 9.511320235181596404253068222090_wp
        Xi(29) = 9.820249490631424653386272183297_wp
        Xi(30) = 9.979618292913723588861973912145_wp

      case (4)
        ! ----- K = 15 , R = 2E1 -----

        RIni = 2.0e1_wp

        Omega(1) = 0.0342570755840630594360332952796888150715_wp
        Omega(2) = 0.0812389574552505602667430532892733907602_wp
        Omega(3) = 0.1320880642088497519990664477718134151019_wp
        Omega(4) = 0.1895620574511707937518821195932439138687_wp
        Omega(5) = 0.2574347552632545563418026840496466434161_wp
        Omega(6) = 0.3409324413871414226546215781032955760566_wp
        Omega(7) = 0.4472725024651988329552326079108226508652_wp
        Omega(8) = 0.5862378479869452498800599937744948420004_wp
        Omega(9) = 0.7709416195632693878593728886716718307071_wp
        Omega(10) = 1.0193233364053463209462219163370377827960_wp
        Omega(11) = 1.3575648828950585544081755551282242322486_wp
        Omega(12) = 1.8283241651243346893081914705803114884475_wp
        Omega(13) = 2.5132971297146048884809527157457864632306_wp
        Omega(14) = 3.6130225748024470322147805312340551608941_wp
        Omega(15) = 5.9202936131776174579974258982417723018443_wp

        Alpha(1) = 0.0133199863613047550980807848253206771005_wp
        Alpha(2) = 0.0708357967215275064081273469207555848470_wp
        Alpha(3) = 0.1770754400387099266971340644216681425860_wp
        Alpha(4) = 0.3372094560469260752524033042387685554786_wp
        Alpha(5) = 0.5596497412025823315457305484610373014220_wp
        Alpha(6) = 0.8572651840652747578944495310260265341640_wp
        Alpha(7) = 1.2491004136013469878524126399632621087221_wp
        Alpha(8) = 1.7626491360944358240212676003366709664988_wp
        Alpha(9) = 2.4367709098923107514760022507260828206199_wp
        Alpha(10) = 3.3256722098160102919339048410662940113980_wp
        Alpha(11) = 4.5052027161924701617948252341250281460816_wp
        Alpha(12) = 6.0845927692710795584886407372238181778812_wp
        Alpha(13) = 8.2321905231666911532950603458402838441543_wp
        Alpha(14) = 11.2452663855440287169959456825552024383796_wp
        Alpha(15) = 15.8274909305672764536429553139385006943485_wp

        Xi(1) = 1.002479499060150240174853208952_wp
        Xi(2) = 1.022391750455047769625960274453_wp
        Xi(3) = 1.062625160624124051296848780801_wp
        Xi(4) = 1.124003499356398210091978473457_wp
        Xi(5) = 1.207780198723366982564361715724_wp
        Xi(6) = 1.315663377120945382275744861555_wp
        Xi(7) = 1.449851096400061798569192106356_wp
        Xi(8) = 1.613080412284010575524502428824_wp
        Xi(9) = 1.808683092560154019481748199905_wp
        Xi(10) = 2.040659750638416931444590884936_wp
        Xi(11) = 2.313741520040654406787511421228_wp
        Xi(12) = 2.633475245137824016479835553817_wp
        Xi(13) = 3.006282198555712859098371092514_wp
        Xi(14) = 3.439522060833853566710546778573_wp
        Xi(15) = 3.941501159331839765527505381471_wp
        Xi(16) = 4.521441417355159078737131261860_wp
        Xi(17) = 5.189323080388879948134789321657_wp
        Xi(18) = 5.955554020001711075342487600182_wp
        Xi(19) = 6.830364172019600505891129271774_wp
        Xi(20) = 7.822753285351927012274825257165_wp
        Xi(21) = 8.938836769060568735595362355184_wp
        Xi(22) = 10.179342570236754901037701248612_wp
        Xi(23) = 11.536099533668944362198305331191_wp
        Xi(24) = 12.987495627551309394787093509649_wp
        Xi(25) = 14.493369771815821333595108999504_wp
        Xi(26) = 15.990565620993585975624651229765_wp
        Xi(27) = 17.391553165581567757386194372771_wp
        Xi(28) = 18.589285048398366650967972191211_wp
        Xi(29) = 19.470958874305620065273125796068_wp
        Xi(30) = 19.939964616609519680590079104832_wp

      case (5)
        ! ----- K = 15 , R = 3E1 -----

        RIni = 3.0e1_wp

        Omega(1) = 0.0251404605998490003396624243095502926337_wp
        Omega(2) = 0.0598615044552078848093701769067509488309_wp
        Omega(3) = 0.0980768773006056569061677542775701255096_wp
        Omega(4) = 0.1424173537496608342521175333272864804712_wp
        Omega(5) = 0.1966064249578906485626476202677181959189_wp
        Omega(6) = 0.2659327500342726060200086030116395363621_wp
        Omega(7) = 0.3577519270311487996349963935927007696591_wp
        Omega(8) = 0.4819746544793883449321874462567549812775_wp
        Omega(9) = 0.6518014776973908524604177372463453821183_wp
        Omega(10) = 0.8853077937671970994051688641501840493220_wp
        Omega(11) = 1.2089526907881880205550620410726025966142_wp
        Omega(12) = 1.6657167955120668526415930732476056164160_wp
        Omega(13) = 2.3371854613987042021544526804177621670533_wp
        Omega(14) = 3.4218312791854920487666369410817424068227_wp
        Omega(15) = 5.7000508866800705478662669722922373694018_wp

        Alpha(1) = 0.0097706839879553685810991379978585058552_wp
        Alpha(2) = 0.0520656002503765860991156218945397071707_wp
        Alpha(3) = 0.1306478367526141272201992485357813222890_wp
        Alpha(4) = 0.2502493155231965290500196347212025216322_wp
        Alpha(5) = 0.4187480144640803076555059247798595833956_wp
        Alpha(6) = 0.6484848684705118514832664078362256532273_wp
        Alpha(7) = 0.9580829940330154019002828480466860128217_wp
        Alpha(8) = 1.3747544144022870453610007945854931676877_wp
        Alpha(9) = 1.9371818079908711005216251366256585697556_wp
        Alpha(10) = 2.6994958848433685664504333434621230480843_wp
        Alpha(11) = 3.7376744793080215497997620399672769053723_wp
        Alpha(12) = 5.1613963890033491706786639152682027997798_wp
        Alpha(13) = 7.1396285747356587861496557412266383835231_wp
        Alpha(14) = 9.9693937759796579598653432086052816885058_wp
        Alpha(15) = 14.3484779274818628541296461342824386520078_wp

        Xi(1) = 1.002911899669123286312107101725_wp
        Xi(2) = 1.026313689141005041976714706742_wp
        Xi(3) = 1.073687816287695194548794386247_wp
        Xi(4) = 1.146186639545510977267889218822_wp
        Xi(5) = 1.245569476402107386529827248722_wp
        Xi(6) = 1.374246811369371118623748662113_wp
        Xi(7) = 1.535346148506875486221615045856_wp
        Xi(8) = 1.732801356476310413292155065967_wp
        Xi(9) = 1.971463807015423310570190196245_wp
        Xi(10) = 2.257234657950460307053452968518_wp
        Xi(11) = 2.597215293815418540781386758098_wp
        Xi(12) = 2.999875704093432549372441608959_wp
        Xi(13) = 3.475238154369281948002576299039_wp
        Xi(14) = 4.035064925614428033885661140090_wp
        Xi(15) = 4.693036802769865503334734135876_wp
        Xi(16) = 5.464884344761582315574272028513_wp
        Xi(17) = 6.368412773388223142682906807721_wp
        Xi(18) = 7.423316204099501691357487764833_wp
        Xi(19) = 8.650621011282775081954743434665_wp
        Xi(20) = 10.071483516655537285462707064454_wp
        Xi(21) = 11.704963697503043124764632310075_wp
        Xi(22) = 13.564220944764207000057809615612_wp
        Xi(23) = 15.650479657737154826936676332849_wp
        Xi(24) = 17.944198940371834598284106476740_wp
        Xi(25) = 20.393466886603605053157606263881_wp
        Xi(26) = 22.901278629381184189645059845475_wp
        Xi(27) = 25.316342800409824157631399899060_wp
        Xi(28) = 27.435805077602629868660510936706_wp
        Xi(29) = 29.029563787005013233977201991820_wp
        Xi(30) = 29.889333975110803413854920762560_wp

      case (6)
        ! ----- K = 15 , R = 4E1 -----

        RIni = 4.0e1_wp

        Omega(1) = 0.0201134538823361043219286464767137090348_wp
        Omega(2) = 0.0480468900161741623712583371813567012509_wp
        Omega(3) = 0.0792047491252899234882378989852647066527_wp
        Omega(4) = 0.1161121701594685450795478229291113336785_wp
        Omega(5) = 0.1624290885458093495466100342206061668549_wp
        Omega(6) = 0.2234294134956040576680180373414508210317_wp
        Omega(7) = 0.3064598852976081112429659891738253918447_wp
        Omega(8) = 0.4213902130844775926945405819523671198112_wp
        Omega(9) = 0.5814003222278704058117657083037954635074_wp
        Omega(10) = 0.8046842905645213672423746165129898599844_wp
        Omega(11) = 1.1180166798382869076467285851528288276313_wp
        Omega(12) = 1.5647904440413159076047819318056042448006_wp
        Omega(13) = 2.2267586807525969559012901655137284251396_wp
        Omega(14) = 3.3013295338455457977617363285460783117742_wp
        Omega(15) = 5.5613724683790860987830928507946737227030_wp

        Alpha(1) = 0.0078140833353863054459905053937551766552_wp
        Alpha(2) = 0.0417064349886099036901964884349602868951_wp
        Alpha(3) = 0.1049725240802041791791758551521951403629_wp
        Alpha(4) = 0.2020201597069699714137374843558880854744_wp
        Alpha(5) = 0.3403153505965562537207650534165637168371_wp
        Alpha(6) = 0.5317496824765323231408442605783193357638_wp
        Alpha(7) = 0.7944899302065638654196666479379018710461_wp
        Alpha(8) = 1.1552671367902653680502858546397249028814_wp
        Alpha(9) = 1.6522434451894113991397838869801262262627_wp
        Alpha(10) = 2.3390686809253648662312863315548838727409_wp
        Alpha(11) = 3.2914576993509841271613042268384674571280_wp
        Alpha(12) = 4.6192066650403851603423621696720147156157_wp
        Alpha(13) = 6.4917334024130807878898152818436528832535_wp
        Alpha(14) = 9.2062044495678152173159802806878815317759_wp
        Alpha(15) = 13.4572444522132288824770030011279686732451_wp

        Xi(1) = 1.003229857361806261315147204005_wp
        Xi(2) = 1.029200765720327604368054819073_wp
        Xi(3) = 1.081849244748823186002689711316_wp
        Xi(4) = 1.162604179074884219757292369568_wp
        Xi(5) = 1.273652369294515127365850781782_wp
        Xi(6) = 1.418000961648649064670776875907_wp
        Xi(7) = 1.599575428591748234297131858739_wp
        Xi(8) = 1.823342794824100966287125369369_wp
        Xi(9) = 2.095484842818377407619098717895_wp
        Xi(10) = 2.423588140806265554252560412429_wp
        Xi(11) = 2.816875262889290515129839254094_wp
        Xi(12) = 3.286470548855907088461880682928_wp
        Xi(13) = 3.845709652766320670209954091234_wp
        Xi(14) = 4.510476878570599915639627308650_wp
        Xi(15) = 5.299566074429211033798570174014_wp
        Xi(16) = 6.235029436377318096031835414905_wp
        Xi(17) = 7.342446775895706162582549092477_wp
        Xi(18) = 8.650996654635645408820687762841_wp
        Xi(19) = 10.193114846508562773216843844182_wp
        Xi(20) = 12.003382121351368796123570970025_wp
        Xi(21) = 14.116058058878114126952207918464_wp
        Xi(22) = 16.560385872556048922343840956728_wp
        Xi(23) = 19.352313942618907987094911682391_wp
        Xi(24) = 22.481563079780628417597698742725_wp
        Xi(25) = 25.892502301700924332086839463329_wp
        Xi(26) = 29.460835413519626530334805103450_wp
        Xi(27) = 32.971840195575683459677795994480_wp
        Xi(28) = 36.115528534276363340632665099861_wp
        Xi(29) = 38.519215416158373282290439121311_wp
        Xi(30) = 39.830469749008866213457746852100_wp

      case (7)
        ! ----- K = 15 , R = 5E1 -----

        RIni = 5.0e1_wp

        Omega(1) = 0.0168880504457705811679482047621414508853_wp
        Omega(2) = 0.0404523657107779312077841787215204405470_wp
        Omega(3) = 0.0670338169082042983486528428538342438969_wp
        Omega(4) = 0.0990695634544053111477518447358647790679_wp
        Omega(5) = 0.1401553278525078226080533352893020548890_wp
        Omega(6) = 0.1955184392167879767383559946880566826621_wp
        Omega(7) = 0.2724397623186262082145472213090542368263_wp
        Omega(8) = 0.3806988587933995549900632882067696982631_wp
        Omega(9) = 0.5334384650052095012398231665962100578326_wp
        Omega(10) = 0.7489681402627489304650934553997387865820_wp
        Omega(11) = 1.0543592358076193700325631774639134619065_wp
        Omega(12) = 1.4933981322491246728281777400582086556824_wp
        Omega(13) = 2.1480751959787540564857794000985791171843_wp
        Omega(14) = 3.2151626673716206042397125486331788124517_wp
        Omega(15) = 5.4623017093067981523289711187629791311338_wp

        Alpha(1) = 0.0065589737028395636394760526925035160062_wp
        Alpha(2) = 0.0350550853541004113653417693236002961044_wp
        Alpha(3) = 0.0884595857610457954417980917793951078920_wp
        Alpha(4) = 0.1709281913314045605651768020005754067370_wp
        Alpha(5) = 0.2895971214269864041769390883285240079204_wp
        Alpha(6) = 0.4559750943112586425958864672081105595680_wp
        Alpha(7) = 0.6877928532962935922914597675070780269380_wp
        Alpha(8) = 1.0112643890368854897570533024619976458780_wp
        Alpha(9) = 1.4639620939737501783715553638209883047239_wp
        Alpha(10) = 2.0989787920680469870005763199927173445758_wp
        Alpha(11) = 2.9916804481492304287719441457937818995561_wp
        Alpha(12) = 4.2518719910654838274946532017395384173142_wp
        Alpha(13) = 6.0493272187711193718477609149175577840651_wp
        Alpha(14) = 8.6814711877106877273638230185781594627770_wp
        Alpha(15) = 12.8409564351942392032568385218382900347933_wp

        Xi(1) = 1.003481855447011282978127255650_wp
        Xi(2) = 1.031490732158271747716249411297_wp
        Xi(3) = 1.088333640314139787261954461783_wp
        Xi(4) = 1.175680240722669799534327861323_wp
        Xi(5) = 1.296090183590884389874808846344_wp
        Xi(6) = 1.453094491106130258157615176007_wp
        Xi(7) = 1.651319610191417435515054412676_wp
        Xi(8) = 1.896656937573268690349834186826_wp
        Xi(9) = 2.196477388512006007244281580171_wp
        Xi(10) = 2.559890491008365413306072100319_wp
        Xi(11) = 2.998053898728377583606657941218_wp
        Xi(12) = 3.524539770453997259303663414087_wp
        Xi(13) = 4.155765896611457538723649607348_wp
        Xi(14) = 4.911493744258011896839516419178_wp
        Xi(15) = 5.815386349036163648757585820093_wp
        Xi(16) = 6.895596746028621872211250565243_wp
        Xi(17) = 8.185322874284366840257065600639_wp
        Xi(18) = 9.723200314621305568110742889587_wp
        Xi(19) = 11.553289726430778155269463169930_wp
        Xi(20) = 13.724215799603990733647462807454_wp
        Xi(21) = 16.286719270811650639094181336475_wp
        Xi(22) = 19.288370022171881075501165270225_wp
        Xi(23) = 22.763612784503022020907159195247_wp
        Xi(24) = 26.716677090822341378295590175185_wp
        Xi(25) = 31.095063981479428792328856268057_wp
        Xi(26) = 35.753698060397073880262164280452_wp
        Xi(27) = 40.417221907703723179916543983836_wp
        Xi(28) = 44.661862908451292482336913991503_wp
        Xi(29) = 47.952660699750523112111499912658_wp
        Xi(30) = 49.764818282386813565881800158763_wp

      case (8)
        ! ----- K = 15 , R = 6E1 -----

        RIni = 6.0e1_wp

        Omega(1) = 0.0146259217872521891768382426221584147186_wp
        Omega(2) = 0.0351175064449876853134877106171751748320_wp
        Omega(3) = 0.0584601957105404397711031095480338493076_wp
        Omega(4) = 0.0870165685389104654571547879660098345767_wp
        Omega(5) = 0.1243207801068556728234714911507374779376_wp
        Omega(6) = 0.1755396894057373440045134727061082458022_wp
        Omega(7) = 0.2478661231331974945581598374078069468851_wp
        Omega(8) = 0.3509786266364572828043398577957390216397_wp
        Omega(9) = 0.4979850090944644447094481060078763334786_wp
        Omega(10) = 0.7073027698955994890068696423490735014639_wp
        Omega(11) = 1.0062683560617502864092953629437943163794_wp
        Omega(12) = 1.4390235276810452051459918165754459096206_wp
        Omega(13) = 2.0878094629421765564096513667635690580937_wp
        Omega(14) = 3.1489876316118781847657598360612496435351_wp
        Omega(15) = 5.3862800842515326351159221651698771893280_wp

        Alpha(1) = 0.0056788702569020912235681335794579283061_wp
        Alpha(2) = 0.0303873265325022603842866977807979367299_wp
        Alpha(3) = 0.0768546492022988189511158163558235401069_wp
        Alpha(4) = 0.1490327845948825036780094333010993068456_wp
        Alpha(5) = 0.2537854202184757155798342648234111607053_wp
        Alpha(6) = 0.4022900808658428740494442110708916970907_wp
        Alpha(7) = 0.6118736806237610867582367435435486413553_wp
        Alpha(8) = 0.9082474519754037504135416036810113382671_wp
        Alpha(9) = 1.3284052305443362714974450677907213957951_wp
        Alpha(10) = 1.9248935323363437296549302568671180324600_wp
        Alpha(11) = 2.7727204966996773344810323091991222099750_wp
        Alpha(12) = 3.9816530453020171492761747344957257155329_wp
        Alpha(13) = 5.7217548099966258393599127796136372126057_wp
        Alpha(14) = 8.2907395887501191293275648597216331836535_wp
        Alpha(15) = 12.3799185668548834054689122829984171403339_wp

        Xi(1) = 1.003690691737168430098040006193_wp
        Xi(2) = 1.033389874851190462783263879221_wp
        Xi(3) = 1.093718642683819189430696749987_wp
        Xi(4) = 1.186560911089355885273365076582_wp
        Xi(5) = 1.314808791861358351865489424437_wp
        Xi(6) = 1.482461815451560535735961443571_wp
        Xi(7) = 1.694778672608641367043277869975_wp
        Xi(8) = 1.958485097627332217938914138422_wp
        Xi(9) = 2.282037250536437880633552910403_wp
        Xi(10) = 2.675942969701724920197211110562_wp
        Xi(11) = 3.153148192149421274663168013319_wp
        Xi(12) = 3.729502671580447342855255121741_wp
        Xi(13) = 4.424317147737756152590937741564_wp
        Xi(14) = 5.261024415054952740791044396573_wp
        Xi(15) = 6.267940203075530144691263068069_wp
        Xi(16) = 7.479106120763681706323233733258_wp
        Xi(17) = 8.935155916674657633827494329637_wp
        Xi(18) = 10.684074934984131811585572613410_wp
        Xi(19) = 12.781590917452321275529780741920_wp
        Xi(20) = 15.290692656127236188391693705313_wp
        Xi(21) = 18.279391780428630885288465712790_wp
        Xi(22) = 21.815153227757841315065379106386_wp
        Xi(23) = 25.953542579470569560268211617426_wp
        Xi(24) = 30.717517687357415171606200843257_wp
        Xi(25) = 36.063340223198843699503601811784_wp
        Xi(26) = 41.831395934728776656724713589597_wp
        Xi(27) = 47.689420141103013205813310193548_wp
        Xi(28) = 53.096033085676186246554264513975_wp
        Xi(29) = 57.338084472108027285497389158309_wp
        Xi(30) = 59.693320802696889410682290488808_wp

      case (9)
        ! ----- K = 15 , R = 7E1 -----

        RIni = 7.0e1_wp

        Omega(1) = 0.0129431385760811314218342326677613218777_wp
        Omega(2) = 0.0311433962175995538834011311379867947835_wp
        Omega(3) = 0.0520576228698372043838453297748269932299_wp
        Omega(4) = 0.0779840434358426823802275251740212524965_wp
        Omega(5) = 0.1123987204649185051407301975945340899443_wp
        Omega(6) = 0.1604023598955141054406888651007179191765_wp
        Omega(7) = 0.2290926497594118356042900211400059617972_wp
        Omega(8) = 0.3280493759577526712706285749732160184067_wp
        Omega(9) = 0.4703507500452662837091537956224485128587_wp
        Omega(10) = 0.6745122915667454001169278665628326052683_wp
        Omega(11) = 0.9681041644892160188224858319294696684665_wp
        Omega(12) = 1.3955858527294683469168259937198683928727_wp
        Omega(13) = 2.0394448609688786832206536248079942197364_wp
        Omega(14) = 3.0957651278694305178353085583609072273248_wp
        Omega(15) = 5.3251821433448008865155220981080219644355_wp

        Alpha(1) = 0.0050242744369184408194854523071332108586_wp
        Alpha(2) = 0.0269131431088499060335896212806106042592_wp
        Alpha(3) = 0.0682062998374950968293799029307233894315_wp
        Alpha(4) = 0.1326861736517988391465287453807597728428_wp
        Alpha(5) = 0.2269855619840041827220715117685578832152_wp
        Alpha(6) = 0.3619911699308799959734668505451082864965_wp
        Alpha(7) = 0.5546589943832142908380911094923249038402_wp
        Alpha(8) = 0.8302275837852374761613796771975160027068_wp
        Alpha(9) = 1.2251474742957150349236677966402453421324_wp
        Alpha(10) = 1.7914531635436964926780256668337187875295_wp
        Alpha(11) = 2.6038082335791434180922304664207445057400_wp
        Alpha(12) = 3.7719182048923451738235007590915870423487_wp
        Alpha(13) = 5.4660873571676506076782997567420352424961_wp
        Alpha(14) = 7.9843185170675084285675515705094085205928_wp
        Alpha(15) = 12.0169576607587050556466556905377274233615_wp

        Xi(1) = 1.003869021713253574957375180432_wp
        Xi(2) = 1.035012555911422882045681836960_wp
        Xi(3) = 1.098325070301832839846437051001_wp
        Xi(4) = 1.195883988344786225056955164536_wp
        Xi(5) = 1.330882547499336633839775445587_wp
        Xi(6) = 1.507745732205119922205403815152_wp
        Xi(7) = 1.732309131445018534701044132795_wp
        Xi(8) = 2.012063648685074870913272460093_wp
        Xi(9) = 2.356466640738977198201928309196_wp
        Xi(10) = 2.777322490733142419227358344003_wp
        Xi(11) = 3.289245690693270381815827008154_wp
        Xi(12) = 3.910223848851538200723301774708_wp
        Xi(13) = 4.662302835038631008881399031551_wp
        Xi(14) = 5.572409455073852398761941051575_wp
        Xi(15) = 6.673321239676869499890532466679_wp
        Xi(16) = 8.004772939028161932335792005944_wp
        Xi(17) = 9.614651387741289116466159025975_wp
        Xi(18) = 11.560153941240544802661338597005_wp
        Xi(19) = 13.908637463799372543243881850117_wp
        Xi(20) = 16.737614017371478167955922700116_wp
        Xi(21) = 20.132875726972504146217168852218_wp
        Xi(22) = 24.182884524734062060835615426413_wp
        Xi(23) = 28.966358540259132967054078733327_wp
        Xi(24) = 34.528318026877213037934000183782_wp
        Xi(25) = 40.838604370819314472174088592737_wp
        Xi(26) = 47.728776377028992096429593772200_wp
        Xi(27) = 54.813878058326717286619089719579_wp
        Xi(28) = 61.432882561411230437631303402668_wp
        Xi(29) = 66.681268392919390133877222126557_wp
        Xi(30) = 69.616639040442253115459347867500_wp

      case (10)
        ! ----- K = 15 , R = 8E1 -----

        RIni = 8.0e1_wp

        Omega(1) = 0.0116376959297197557926148675337685300946_wp
        Omega(2) = 0.0280565631287540469228494266831774872628_wp
        Omega(3) = 0.0470734626814240072829236081336512498297_wp
        Omega(4) = 0.0709302331808647406336388978320695031243_wp
        Omega(5) = 0.1030483747950790365550260974103480293707_wp
        Omega(6) = 0.1484608343730583948622383030913063350908_wp
        Omega(7) = 0.2141699383993229090569895486928508887559_wp
        Omega(8) = 0.3096632904070938441579673161552221927195_wp
        Omega(9) = 0.4479943584047823081127973066095648846385_wp
        Omega(10) = 0.6477664122253113216059854095885128799637_wp
        Omega(11) = 0.9367556700653058413311678742640964401289_wp
        Omega(12) = 1.3597062225198743081475485916520540286001_wp
        Omega(13) = 1.9993413337952957502209258566772120957467_wp
        Omega(14) = 3.0515526253989090257456168187033540561970_wp
        Omega(15) = 5.2744587859215596903524059246848310067435_wp

        Alpha(1) = 0.0045165373581371195146511878480699575533_wp
        Alpha(2) = 0.0242166963980014430311528883174587711835_wp
        Alpha(3) = 0.0614863974926384792736035710125630515677_wp
        Alpha(4) = 0.1199639025272229584994981496648414065476_wp
        Alpha(5) = 0.2060825487666099330608549672350093828754_wp
        Alpha(6) = 0.3304703765697770167825315923426643394123_wp
        Alpha(7) = 0.5097429488165322793180068983165398321944_wp
        Alpha(8) = 0.7686998504455993988624014356947355963712_wp
        Alpha(9) = 1.1432889758299511601645492842038720482378_wp
        Alpha(10) = 1.6850715529907800840176898549138684302307_wp
        Alpha(11) = 2.4683859599626540547168129791089086211286_wp
        Alpha(12) = 3.6028637238078778854723227542677932433435_wp
        Alpha(13) = 5.2590120105798029950978922375526281030034_wp
        Alpha(14) = 7.7351118438454861121582328831181030182051_wp
        Alpha(15) = 11.7207866995992235304921424976498656178592_wp

        Xi(1) = 1.004024611678951089564423848621_wp
        Xi(2) = 1.036429045999876775410904194974_wp
        Xi(3) = 1.102350204991204553946898547778_wp
        Xi(4) = 1.204042429449802128474042439699_wp
        Xi(5) = 1.344974603354369209972339616055_wp
        Xi(6) = 1.529962545222354810929565882560_wp
        Xi(7) = 1.765373718388322756846740835268_wp
        Xi(8) = 2.059407784772918241066735900446_wp
        Xi(9) = 2.422453711469328537604292495011_wp
        Xi(10) = 2.867528498348190244774530421878_wp
        Xi(11) = 3.410814667030763296540901707843_wp
        Xi(12) = 4.072320586331676467613249759125_wp
        Xi(13) = 4.876690443441787359973860205287_wp
        Xi(14) = 5.854191786781036360796243167570_wp
        Xi(15) = 7.041895129523705379546300031635_wp
        Xi(16) = 8.485050052605370951054608585906_wp
        Xi(17) = 10.238621125821589888903184739632_wp
        Xi(18) = 12.368868448315555632656403961533_wp
        Xi(19) = 14.954699426572277638769481455938_wp
        Xi(20) = 18.088216348532182266989343233377_wp
        Xi(21) = 21.873325591020155268112690194471_wp
        Xi(22) = 26.420300894221018182458915468658_wp
        Xi(23) = 31.832600608164070669220402365340_wp
        Xi(24) = 38.180038355519728431042292271513_wp
        Xi(25) = 45.450194896975374420772242345379_wp
        Xi(26) = 53.471031272100935448354785961556_wp
        Xi(27) = 61.809268988957774559195312846782_wp
        Xi(28) = 69.683444375138042757578116237482_wp
        Xi(29) = 75.986543115418312147046897564451_wp
        Xi(30) = 79.535272972414583214639804964463_wp

      case (11)
        ! ----- K = 15 , R = 9E1 -----

        RIni = 9.0e1_wp

        Omega(1) = 0.0105926344340905414396767953051636812489_wp
        Omega(2) = 0.0255826077357476880509386859413600845414_wp
        Omega(3) = 0.0430707967103431928921556792050351347712_wp
        Omega(4) = 0.0652489962492777935605691753961909995496_wp
        Omega(5) = 0.0954874765697908190449443542602292467336_wp
        Omega(6) = 0.1387519341256913148869266139140421500997_wp
        Omega(7) = 0.2019522835098053309253350248719804937991_wp
        Omega(8) = 0.2944915569107425864375533058892031590403_wp
        Omega(9) = 0.4294022509808315279000896780292606536023_wp
        Omega(10) = 0.6253657255484110376233392036837699379248_wp
        Omega(11) = 0.9103406991822887626931645743688648053649_wp
        Omega(12) = 1.3293276205302154366078709601950436081097_wp
        Omega(13) = 1.9652732153735464103519459122537682560505_wp
        Omega(14) = 3.0139343774001751512237046171094334567897_wp
        Omega(15) = 5.2313239633687951277499217361821592930937_wp

        Alpha(1) = 0.0041101270238754405812767959548653662338_wp
        Alpha(2) = 0.0220571300018926773574333508737277398382_wp
        Alpha(3) = 0.0560989284387557866623367271857736326979_wp
        Alpha(4) = 0.1097490263250038257858073875716797829227_wp
        Alpha(5) = 0.1892657547100838417368333377321931720871_wp
        Alpha(6) = 0.3050446106979405895738306081499402466761_wp
        Alpha(7) = 0.4733880519672660061917222884542155725285_wp
        Alpha(8) = 0.7186895877224460323670456041167398097969_wp
        Alpha(9) = 1.0764340399712193814540928116230134037323_wp
        Alpha(10) = 1.5977464735297438055234772735779813501722_wp
        Alpha(11) = 2.3566601731303012598472967908946884563193_wp
        Alpha(12) = 3.4627257320720155233783471571129553012725_wp
        Alpha(13) = 5.0866235331733391220911111307856344865286_wp
        Alpha(14) = 7.5268981403874479869584024349649098439841_wp
        Alpha(15) = 11.4726163407158323241999409169977752753766_wp

        Xi(1) = 1.004162582029408795234945439567_wp
        Xi(2) = 1.037685695187326312513738379995_wp
        Xi(3) = 1.105924293733365469899303235035_wp
        Xi(4) = 1.211295943883145047174389186129_wp
        Xi(5) = 1.357524220232578484029301035818_wp
        Xi(6) = 1.549786974818863132582341668453_wp
        Xi(7) = 1.794946143148746320523413688708_wp
        Xi(8) = 2.101863015200717751560996471305_wp
        Xi(9) = 2.481799495764264427933795675862_wp
        Xi(10) = 2.948914285225161044465963522399_wp
        Xi(11) = 3.520871665147989925835278479660_wp
        Xi(12) = 4.219599171866740438662651824231_wp
        Xi(13) = 5.072222186071038071975003447278_wp
        Xi(14) = 6.112213647380998372739963064859_wp
        Xi(15) = 7.380786110372765320471882821707_wp
        Xi(16) = 8.928541269462804010233203388935_wp
        Xi(17) = 10.817355927958625714732643352534_wp
        Xi(18) = 13.122403041454299945506312141674_wp
        Xi(19) = 15.934039739813314534577770587020_wp
        Xi(20) = 19.358966647485198544348383720148_wp
        Xi(21) = 23.519425979528879025112764189487_wp
        Xi(22) = 28.548115907535518760707038410374_wp
        Xi(23) = 34.574509264277191612074746629446_wp
        Xi(24) = 41.695498324481614336317614188943_wp
        Xi(25) = 49.920019397288935589923619673414_wp
        Xi(26) = 59.077229894963870584362330973249_wp
        Xi(27) = 68.689937397766962591527661885493_wp
        Xi(28) = 77.856287843186458938471439239493_wp
        Xi(29) = 85.257300548335234142982574923053_wp
        Xi(30) = 89.449614211362739858712789242645_wp

      case (12)
        ! ----- K = 15 , R = 1E2 -----

        RIni = 1.0e2_wp

        Omega(1) = 0.0097353035250252180240644063989596812547_wp
        Omega(2) = 0.0235509356290765406682197842178072555441_wp
        Omega(3) = 0.0397775784086478460848312552200489733423_wp
        Omega(4) = 0.0605621311489281332204545355707425713376_wp
        Omega(5) = 0.0892266961232228339703288245987611304599_wp
        Omega(6) = 0.1306713372983350476879077853298305456065_wp
        Omega(7) = 0.1917178993499250586664836917827337003928_wp
        Omega(8) = 0.2816922494803035432217423739142958538650_wp
        Omega(9) = 0.4136087055236950441158751828885087320486_wp
        Omega(10) = 0.6062180228309940962258366148329713496423_wp
        Omega(11) = 0.8876415482272522335343949151376108375189_wp
        Omega(12) = 1.3031121145509258732637924982356025793706_wp
        Omega(13) = 1.9357875430219505574541347114880807112058_wp
        Omega(14) = 2.9813305239671294517497801557226466684369_wp
        Omega(15) = 5.1939564819246461309550089158193486582604_wp

        Alpha(1) = 0.0037767636667319992393943901441352717363_wp
        Alpha(2) = 0.0202847833501038873942951728038264214149_wp
        Alpha(3) = 0.0516732608069974953441424840583362509960_wp
        Alpha(4) = 0.1013462127489563033004463070418932346684_wp
        Alpha(5) = 0.1754064696475346228572754839203362564604_wp
        Alpha(6) = 0.2840385871952036498256505270276761621062_wp
        Alpha(7) = 0.4432561489279913867446111797043784008565_wp
        Alpha(8) = 0.6770770588279350320204397206680368981324_wp
        Alpha(9) = 1.0205594158683729179987187407618876022752_wp
        Alpha(10) = 1.5244258129485720603974591158547013947100_wp
        Alpha(11) = 2.2624235257354227235197796641585910037975_wp
        Alpha(12) = 3.3440185071922190686473375098586302556214_wp
        Alpha(13) = 4.9400405999879862563306520684136557974853_wp
        Alpha(14) = 7.3492824024054162557542668743337799241999_wp
        Alpha(15) = 11.2603722455184587251594829915291029465152_wp

        Xi(1) = 1.004286491374597387635052292421_wp
        Xi(2) = 1.038814735511330400170686150130_wp
        Xi(3) = 1.109137987553001864350459004793_wp
        Xi(4) = 1.217825521188258722335602535214_wp
        Xi(5) = 1.368837939395539215060686177861_wp
        Xi(6) = 1.567690825204025039415860198755_wp
        Xi(7) = 1.821708863181571224059557279062_wp
        Xi(8) = 2.140374694778018385991555372350_wp
        Xi(9) = 2.535773253686438767889183010240_wp
        Xi(10) = 3.023142394209262089803738615679_wp
        Xi(11) = 3.621555637363941586087320856358_wp
        Xi(12) = 4.354770211928382454922431543665_wp
        Xi(13) = 5.252288561299911683315916333115_wp
        Xi(14) = 6.350669235945679522679918616035_wp
        Xi(15) = 7.695131241623746833676461553964_wp
        Xi(16) = 9.341479685750591948657783891719_wp
        Xi(17) = 11.358348485559450371032430560092_wp
        Xi(18) = 13.829675812307640164582067665577_wp
        Xi(19) = 16.857155155790341884147753859224_wp
        Xi(20) = 20.562053289041963013983327357437_wp
        Xi(21) = 25.085103676531217059553213744039_wp
        Xi(22) = 30.581891161466432309951057888497_wp
        Xi(23) = 37.208942577635263544016952508287_wp
        Xi(24) = 45.092168656907410474732333938164_wp
        Xi(25) = 54.265042940772657668185852131160_wp
        Xi(26) = 64.562322586547332983752944812750_wp
        Xi(27) = 75.467256659206015750873675074217_wp
        Xi(28) = 85.958283875905530331540393973455_wp
        Xi(29) = 94.496276014416706254261946185125_wp
        Xi(30) = 99.359980127812882196025601189149_wp

      case (13)
        ! ----- K = 15 , R = 2E2 -----

        RIni = 2.0e2_wp

        Omega(1) = 0.0055614445793698374450548323373599846065_wp
        Omega(2) = 0.0136224431019640811167206750388614100444_wp
        Omega(3) = 0.0235750487763078402954483245301808835848_wp
        Omega(4) = 0.0372729718543466537764236382046512119359_wp
        Omega(5) = 0.0576750231360127275212945803517516907277_wp
        Omega(6) = 0.0891547071205389258431880071531505649318_wp
        Omega(7) = 0.1378921395737442449544712247400113369622_wp
        Omega(8) = 0.2127087840449548338320195917106048000278_wp
        Omega(9) = 0.3264936972926197210798267278608619790248_wp
        Omega(10) = 0.4983953354087468980441934934821901492796_wp
        Omega(11) = 0.7575386633983470467809326343022746641509_wp
        Omega(12) = 1.1507045886255240824038587055255788982322_wp
        Omega(13) = 1.7626459217391767304210573041878262756654_wp
        Omega(14) = 2.7889482630945156801157563952742179935740_wp
        Omega(15) = 4.9738042967548596619796474893604454337037_wp

        Alpha(1) = 0.0021545257831798762967911889214256415670_wp
        Alpha(2) = 0.0116436500968750751715380365412799701375_wp
        Alpha(3) = 0.0300215936149693332600643181154453920101_wp
        Alpha(4) = 0.0600297576601244908521570232962094593177_wp
        Alpha(5) = 0.1067865632474907834297927870792488391771_wp
        Alpha(6) = 0.1790529516984699501594684861727202473958_wp
        Alpha(7) = 0.2908199897036503363398005084228259420343_wp
        Alpha(8) = 0.4634860058463758151179896077609399185349_wp
        Alpha(9) = 0.7291591909739004999271676232819316965106_wp
        Alpha(10) = 1.1357391474843713363284541939535188248556_wp
        Alpha(11) = 1.7548471860364001617118775988579670865875_wp
        Alpha(12) = 2.6951161158064083005030109330135701384279_wp
        Alpha(13) = 4.1281869152446315049571901045766253446345_wp
        Alpha(14) = 6.3546659737279791273223406022907511214726_wp
        Alpha(15) = 10.0614643180308471551165361290713917696849_wp

        Xi(1) = 1.005107922869164440056455323003_wp
        Xi(2) = 1.046310545611985985594433179457_wp
        Xi(3) = 1.130535493023324185518987960997_wp
        Xi(4) = 1.261482275618622608988339539593_wp
        Xi(5) = 1.444886048402232851394118295918_wp
        Xi(6) = 1.688814172466229769696381690647_wp
        Xi(7) = 2.004129361118252874301992605410_wp
        Xi(8) = 2.405128123552147019655056414678_wp
        Xi(9) = 2.910359053439188724114738970883_wp
        Xi(10) = 3.543659615510939683282951029319_wp
        Xi(11) = 4.335486604412859002518482176569_wp
        Xi(12) = 5.324633615376642651901423208827_wp
        Xi(13) = 6.560455318249405778224803142962_wp
        Xi(14) = 8.105735776721632048456844810147_wp
        Xi(15) = 10.040372369390934146130001103359_wp
        Xi(16) = 12.466079169573248256776276665647_wp
        Xi(17) = 15.512331663565415596807428588733_wp
        Xi(18) = 19.343746232804397278473573784652_wp
        Xi(19) = 24.168954687767984453039549208597_wp
        Xi(20) = 30.250590684210382804233963049967_wp
        Xi(21) = 37.914995874992405845704812605845_wp
        Xi(22) = 47.557900487020893991879644246978_wp
        Xi(23) = 59.637264311460038802154004500267_wp
        Xi(24) = 74.634464513473317601177647873101_wp
        Xi(25) = 92.946581539845967906421364546077_wp
        Xi(26) = 114.646564926806013524906990852514_wp
        Xi(27) = 139.035396209208599857531929444576_wp
        Xi(28) = 163.995366802160568442392474253211_wp
        Xi(29) = 185.507461262228860315870448971509_wp
        Xi(30) = 198.288101188496654087023429724468_wp

      case (14)
        ! ----- K = 15 , R = 3E2 -----

        RIni = 3.0e2_wp

        Omega(1) = 0.0039963763487845303396661785985677362021_wp
        Omega(2) = 0.0098751009059687778298861852294132646080_wp
        Omega(3) = 0.0173873966346357320469510942054225388631_wp
        Omega(4) = 0.0282209747987641637653863282281241353644_wp
        Omega(5) = 0.0450958506221992216564387057076146447798_wp
        Omega(6) = 0.0720399636167920214333383957527878749261_wp
        Omega(7) = 0.1148595378783730234880018118703848273299_wp
        Omega(8) = 0.1820981220823806225609297823608834221432_wp
        Omega(9) = 0.2865444576764013857471304763668840109858_wp
        Omega(10) = 0.4475017573487411423637828689070516929860_wp
        Omega(11) = 0.6945970376068834315798350420578799457871_wp
        Omega(12) = 1.0754922855491432295968121102269776656613_wp
        Omega(13) = 1.6759871195093390723466000857833080317505_wp
        Omega(14) = 2.6919852463147228120721177191398965078406_wp
        Omega(15) = 4.8630599466714542812063870602656834307709_wp

        Alpha(1) = 0.0015467042849176954962383268563953786501_wp
        Alpha(2) = 0.0083952816472138907269735634032736815158_wp
        Alpha(3) = 0.0218334981051808871674497786879998173504_wp
        Alpha(4) = 0.0442655123280556494751321848502012556992_wp
        Alpha(5) = 0.0802749348423698695900101000855197952433_wp
        Alpha(6) = 0.1377946023687582154495026576235261472902_wp
        Alpha(7) = 0.2296169463850606289392897174650443048449_wp
        Alpha(8) = 0.3756070866312024175714696233585598861282_wp
        Alpha(9) = 0.6061451901696484984233617709925567851315_wp
        Alpha(10) = 0.9674225764379703575634658618831451803999_wp
        Alpha(11) = 1.5296736237746061503727651453843350282114_wp
        Alpha(12) = 2.4008394966710964383953264000837179992232_wp
        Alpha(13) = 3.7528692640339534856985803656925781979226_wp
        Alpha(14) = 5.8874881288607278330068939276031869667349_wp
        Alpha(15) = 9.4913359153416843577016903665821700997185_wp

        Xi(1) = 1.005588000700964566532825961165_wp
        Xi(2) = 1.050700460413906080670183584846_wp
        Xi(3) = 1.143117239043653853361459660221_wp
        Xi(4) = 1.287300366525003658874949652446_wp
        Xi(5) = 1.490190781701487541133915870617_wp
        Xi(6) = 1.761611321882728561330662409024_wp
        Xi(7) = 2.114898299916266595326053145953_wp
        Xi(8) = 2.567770977709930953040984302760_wp
        Xi(9) = 3.143453961412974615504539732669_wp
        Xi(10) = 3.872120270522848010279551300705_wp
        Xi(11) = 4.792775182696712790025861394838_wp
        Xi(12) = 5.955736699951859882025406722761_wp
        Xi(13) = 7.425902118476742126782635944338_wp
        Xi(14) = 9.287043146829721957692671541196_wp
        Xi(15) = 11.647443240130051000068289734912_wp
        Xi(16) = 14.647289559199265777077181294175_wp
        Xi(17) = 18.468311302909319914986041588634_wp
        Xi(18) = 23.346241742412311582360873174480_wp
        Xi(19) = 29.586656240765413485027668372140_wp
        Xi(20) = 37.584421748846409350175434838093_wp
        Xi(21) = 47.845911426365556187029204338046_wp
        Xi(22) = 61.010313042861925997284444633806_wp
        Xi(23) = 77.859417012040960090479835287169_wp
        Xi(24) = 99.289393374895234027632806572683_wp
        Xi(25) = 126.185078099482405285247654802561_wp
        Xi(26) = 159.077326811573789830389635824304_wp
        Xi(27) = 197.392752284127000569902143922718_wp
        Xi(28) = 238.160369496851600151776651159707_wp
        Xi(29) = 274.652937169310398690669217103277_wp
        Xi(30) = 296.972486408022178033805360541919_wp

      case (15)
        ! ----- K = 15 , R = 4E2 -----

        RIni = 4.0e2_wp

        Omega(1) = 0.0031580665533451926515155902194798009930_wp
        Omega(2) = 0.0078585696681408003474823896310208493077_wp
        Omega(3) = 0.0140296686661902682585376881535532689327_wp
        Omega(4) = 0.0232449783409302702253388566225322797720_wp
        Omega(5) = 0.0380499461297272989146402329130314257100_wp
        Omega(6) = 0.0622288885678109713890226312171805034268_wp
        Omega(7) = 0.1013387615850393206381584859654410024632_wp
        Omega(8) = 0.1637323506241231536829580650582727230358_wp
        Omega(9) = 0.2621075905769351542702605417645145280403_wp
        Omega(10) = 0.4158369767907242735846887193851273423206_wp
        Omega(11) = 0.6548585051238816585948525339500037034668_wp
        Omega(12) = 1.0274341017750817477149904988209527800791_wp
        Omega(13) = 1.6201338736773287715897673422205116366968_wp
        Omega(14) = 2.6292153158930104208963057788039918705181_wp
        Omega(15) = 4.7914402331023540574511665468548926583026_wp

        Alpha(1) = 0.0012213088418495537004662592817962052383_wp
        Alpha(2) = 0.0066522128401389255792803818605196664038_wp
        Alpha(3) = 0.0174210360526302611842069728229076552850_wp
        Alpha(4) = 0.0357153196385866714841468647773092826014_wp
        Alpha(5) = 0.0657608477231415411610950146059639820351_wp
        Alpha(6) = 0.1149208824288120524064957250764962282119_wp
        Alpha(7) = 0.1951651703896346168066186846368204044211_wp
        Alpha(8) = 0.3253123101170385742929996863992414546374_wp
        Alpha(9) = 0.5345514314848428274692428152636836102829_wp
        Alpha(10) = 0.8678657429638241914264008836088493126226_wp
        Alpha(11) = 1.3944649527164186738345877381028969921317_wp
        Alpha(12) = 2.2217212147441035808518444705228489510773_wp
        Alpha(13) = 3.5217221726634938554385823028525237532449_wp
        Alpha(14) = 5.5969790205896105357227798826613707205979_wp
        Alpha(15) = 9.1341806528607349079107446954139959416352_wp

        Xi(1) = 1.005925747734232925958107462971_wp
        Xi(2) = 1.053792964858323062660692970294_wp
        Xi(3) = 1.152003033768909901531779516048_wp
        Xi(4) = 1.305600412397938297755055347960_wp
        Xi(5) = 1.522451809161981181595102441406_wp
        Xi(6) = 1.813738335860105252558480137814_wp
        Xi(7) = 2.194730223645267108874279404773_wp
        Xi(8) = 2.685850738406786398105446922457_wp
        Xi(9) = 3.314060265518352417545358545858_wp
        Xi(10) = 4.114650008002854532883352245420_wp
        Xi(11) = 5.133618600349286557708861078453_wp
        Xi(12) = 6.430834442002402406585864769184_wp
        Xi(13) = 8.084251009730162189359514712805_wp
        Xi(14) = 10.195517362799900396887198272822_wp
        Xi(15) = 12.897451390950307111536721205169_wp
        Xi(16) = 16.363992429124257377975881766474_wp
        Xi(17) = 20.823427766632239457184061492967_wp
        Xi(18) = 26.575895448033863759790262903948_wp
        Xi(19) = 34.016317083385521735650058872125_wp
        Xi(20) = 43.663749518874066371521003304679_wp
        Xi(21) = 56.197286985580479544422072279986_wp
        Xi(22) = 72.495673530117732012711950773109_wp
        Xi(23) = 93.669558817274358585025506584998_wp
        Xi(24) = 121.054980507308547643674501159694_wp
        Xi(25) = 156.089802970307589546905546740163_wp
        Xi(26) = 199.898086658202700019448805335287_wp
        Xi(27) = 252.257887331837507352561011941816_wp
        Xi(28) = 309.590264512515617623122921031609_wp
        Xi(29) = 362.414818019072917348877282961439_wp
        Xi(30) = 395.470504041337537898259668622813_wp

      case (16)
        ! ----- K = 15 , R = 5E2 -----

        RIni = 5.0e2_wp

        Omega(1) = 0.0026298460390556482285692014265465754619_wp
        Omega(2) = 0.0065832442531841001603625115022599967318_wp
        Omega(3) = 0.0118917603535314008359361306367552035823_wp
        Omega(4) = 0.0200429042853591930952703069682141467922_wp
        Omega(5) = 0.0334464729571471454632868719469840357306_wp
        Omega(6) = 0.0557039547836739697006937059586828553392_wp
        Omega(7) = 0.0921899550749297709830386814144898011136_wp
        Omega(8) = 0.1511113941004667048442766663640668411972_wp
        Omega(9) = 0.2450839575131379099846460831169103755656_wp
        Omega(10) = 0.3935114036889438912799558251132125974436_wp
        Omega(11) = 0.6265467178768755856219964395226185160936_wp
        Omega(12) = 0.9928992278647892220011170194560179425025_wp
        Omega(13) = 1.5797450704374967286620218276382843214378_wp
        Omega(14) = 2.5836775326140330290639046673106804519193_wp
        Omega(15) = 4.7395149729995405176591338758385063556489_wp

        Alpha(1) = 0.0010163654596489672351419041398425113787_wp
        Alpha(2) = 0.0055523279214756996604944981610341159239_wp
        Alpha(3) = 0.0146272132671542568534833739235523353273_wp
        Alpha(4) = 0.0302730130416823495081177306659236947439_wp
        Alpha(5) = 0.0564510822625912328077502164852541355344_wp
        Alpha(6) = 0.1000982261047091139777786769626821694601_wp
        Alpha(7) = 0.1725709802348770900411265710738817347192_wp
        Alpha(8) = 0.2919086142773966053448348595411587780291_wp
        Alpha(9) = 0.4864039250725653168189871466964291357726_wp
        Alpha(10) = 0.8001129679399844055404447673573997690255_wp
        Alpha(11) = 1.3014354844107493782605702858923280018644_wp
        Alpha(12) = 2.0972637457991409630150836385098500613822_wp
        Alpha(13) = 3.3597478567908137022518522130098972411361_wp
        Alpha(14) = 5.3919954067373262818554935371651026798645_wp
        Alpha(15) = 8.8808449536686026783843805887386224640068_wp

        Xi(1) = 1.006185134482228618699878319731_wp
        Xi(2) = 1.056170294653800441364056850180_wp
        Xi(3) = 1.158846606707571684513093102087_wp
        Xi(4) = 1.319732090518601833723476290938_wp
        Xi(5) = 1.547448630422906595046489419687_wp
        Xi(6) = 1.854292454339231302159508862903_wp
        Xi(7) = 2.257132513234196740390183344438_wp
        Xi(8) = 2.778646613633750587623455485797_wp
        Xi(9) = 3.448931732848780591389054039908_wp
        Xi(10) = 4.307615916055986267797078825836_wp
        Xi(11) = 5.406675184063368204672589412141_wp
        Xi(12) = 6.814220012068835528786653332745_wp
        Xi(13) = 8.619583540384217638929964522276_wp
        Xi(14) = 10.940160941995193255929508246993_wp
        Xi(15) = 13.930605331835086672584433920008_wp
        Xi(16) = 17.795230093903935139273975174490_wp
        Xi(17) = 22.804705821009911772753731007413_wp
        Xi(18) = 29.318505212387904381582393043004_wp
        Xi(19) = 37.814909504949985735722250979052_wp
        Xi(20) = 48.930435047373750816201987134946_wp
        Xi(21) = 63.510051366181364620316918134790_wp
        Xi(22) = 82.666614648042439313091467312233_wp
        Xi(23) = 107.839124642511327741012383540919_wp
        Xi(24) = 140.815561601189142976187085309903_wp
        Xi(25) = 183.626414330187036447195048083358_wp
        Xi(26) = 238.081941187831344838854441547937_wp
        Xi(27) = 304.485675751855637494180939484067_wp
        Xi(28) = 378.875066774860891433140963613369_wp
        Xi(29) = 449.051122170990711673921325086667_wp
        Xi(30) = 493.813224037873900545969618747222_wp

      case (17)
        ! ----- K = 15 , R = 6E2 -----

        RIni = 6.0e2_wp

        Omega(1) = 0.0022640577225830361791103777398259655484_wp
        Omega(2) = 0.0056973303057685757995652294817523486614_wp
        Omega(3) = 0.0103981057606504449930884324652946570922_wp
        Omega(4) = 0.0177852165866605353940134871121303561381_wp
        Omega(5) = 0.0301587536348224576396694421309696565459_wp
        Omega(6) = 0.0509766295013259356908627128318611809732_wp
        Omega(7) = 0.0854716363601831396576056186498160371912_wp
        Omega(8) = 0.1417322693899287700447422316818446574871_wp
        Omega(9) = 0.2322996708593134201984711437183328541778_wp
        Omega(10) = 0.3765894209406675952208319591019147765110_wp
        Omega(11) = 0.6049134328425471823472217569594988617609_wp
        Omega(12) = 0.9663336751410720168236831439934064746922_wp
        Omega(13) = 1.5485236223280244059015073321283750829025_wp
        Omega(14) = 2.5483852865162736312746455391575750581978_wp
        Omega(15) = 4.6992901048617441548023443420589728702907_wp

        Alpha(1) = 0.0008744958294475131544913113867782683553_wp
        Alpha(2) = 0.0047897439227468614920500548145371189079_wp
        Alpha(3) = 0.0126845371835364479919219847861566741187_wp
        Alpha(4) = 0.0264715244466399859632374210252625523765_wp
        Alpha(5) = 0.0499047966992862909300129447415228511886_wp
        Alpha(6) = 0.0895847119517327264103770563530293458143_wp
        Alpha(7) = 0.1563859893279255842447883484425119604566_wp
        Alpha(8) = 0.2677347395888903809732690952660760785875_wp
        Alpha(9) = 0.4512116353157648715921281684648747045685_wp
        Alpha(10) = 0.7501246982213588193133882442431570325425_wp
        Alpha(11) = 1.2322057652185713550905751811903598991194_wp
        Alpha(12) = 2.0039333213152113665964093591753680811962_wp
        Alpha(13) = 3.2374803343641888637039621956503765431989_wp
        Alpha(14) = 5.2364298101630870137419837551107093531755_wp
        Alpha(15) = 8.6878037141362704012803597208858263911679_wp

        Xi(1) = 1.006394949628333896555869297895_wp
        Xi(2) = 1.058094769159536815278377375193_wp
        Xi(3) = 1.164394712621561042784762429747_wp
        Xi(4) = 1.331212710369555388589986322234_wp
        Xi(5) = 1.567809937489857727765593664593_wp
        Xi(6) = 1.887431734725242153064700412202_wp
        Xi(7) = 2.308315630383043569546011597460_wp
        Xi(8) = 2.855079921818981010138080445238_wp
        Xi(9) = 3.560540732373147095567167408348_wp
        Xi(10) = 4.468107410429185140376961982156_wp
        Xi(11) = 5.635007538206450348854115173225_wp
        Xi(12) = 7.136647265147478675067355124995_wp
        Xi(13) = 9.072510834617849169968228917327_wp
        Xi(14) = 11.574152398492492466522407745799_wp
        Xi(15) = 14.816034673758016742585452707459_wp
        Xi(16) = 19.030255105054156827479316937968_wp
        Xi(17) = 24.526614737872571635896035147084_wp
        Xi(18) = 31.719891037520669218804925115762_wp
        Xi(19) = 41.166832617257441157693209632384_wp
        Xi(20) = 53.615734679994806032715670696120_wp
        Xi(21) = 70.071262088330390674362213587756_wp
        Xi(22) = 91.874723600885937958848703743797_wp
        Xi(23) = 120.790763010020509561626145256241_wp
        Xi(24) = 159.065016092999220592707487753614_wp
        Xi(25) = 209.346561553769042726691473887968_wp
        Xi(26) = 274.199941891441432817444479042024_wp
        Xi(27) = 354.593579173934398657275224309160_wp
        Xi(28) = 446.383309832086461438338531593217_wp
        Xi(29) = 534.726829720322987760106059340615_wp
        Xi(30) = 592.020706939949882929852265078807_wp

      case (18)
        ! ----- K = 15 , R = 7E2 -----

        RIni = 7.0e2_wp

        Omega(1) = 0.0019945380638779841443940742378576613802_wp
        Omega(2) = 0.0050427907113674821369335269944517152751_wp
        Omega(3) = 0.0092889724596230179876020223266042119548_wp
        Omega(4) = 0.0160950930477551877167139218749447593382_wp
        Omega(5) = 0.0276699612914006832440879171675085501647_wp
        Omega(6) = 0.0473548764340851198299548602266284547113_wp
        Omega(7) = 0.0802675698972774872053700771779016776009_wp
        Omega(8) = 0.1343967267489980976953589664191213159938_wp
        Omega(9) = 0.2222156818745986620039630995981916328219_wp
        Omega(10) = 0.3631409974153666139469551904506161577046_wp
        Omega(11) = 0.5876075064254369875347758234163109136716_wp
        Omega(12) = 0.9449657115229299850505013946833088311905_wp
        Omega(13) = 1.5233094274329127094458682933897364364384_wp
        Omega(14) = 2.5198228955401023394702203495043590919522_wp
        Omega(15) = 4.6667463424862905396056644846680683258455_wp

        Alpha(1) = 0.0007699970490877884595638549179458937077_wp
        Alpha(2) = 0.0042272630399960448487346850905688189215_wp
        Alpha(3) = 0.0112479671026250682619917944243814744709_wp
        Alpha(4) = 0.0236490750857580692246098265363318802201_wp
        Alpha(5) = 0.0450157333067533311525749463444245890287_wp
        Alpha(6) = 0.0816731939775164522392499576175506348363_wp
        Alpha(7) = 0.1441034743319927689098438119974510129850_wp
        Alpha(8) = 0.2492317095072272726277934698502214416749_wp
        Alpha(9) = 0.4240516392151889676529700340390505175492_wp
        Alpha(10) = 0.7112470016203497646706428725327242545973_wp
        Alpha(11) = 1.1779826380950750911152241640067472872033_wp
        Alpha(12) = 1.9303738608286636095276445068691373307956_wp
        Alpha(13) = 3.1405939910687392466887013497967018338386_wp
        Alpha(14) = 5.1126181849943693725167592756264411946177_wp
        Alpha(15) = 8.5336606461802805142710659502824910305208_wp

        Xi(1) = 1.006570622484552558046320969165_wp
        Xi(2) = 1.059707104699230406316158570323_wp
        Xi(3) = 1.169048586031044273956039369899_wp
        Xi(4) = 1.340859496970923495562030836048_wp
        Xi(5) = 1.584956307384439728644896783205_wp
        Xi(6) = 1.915411687398691456107185926427_wp
        Xi(7) = 2.351662563056146397271958803366_wp
        Xi(8) = 2.920036522584414024174617741103_wp
        Xi(9) = 3.655754474561851317167965325083_wp
        Xi(10) = 4.605592026462496899573417774576_wp
        Xi(11) = 5.831476847449754423620416066143_wp
        Xi(12) = 7.415385607590361347497137645135_wp
        Xi(13) = 9.466007129254602976697885452495_wp
        Xi(14) = 12.127816592998331805819400841528_wp
        Xi(15) = 15.593475081790864815271813181852_wp
        Xi(16) = 20.120808972607252549172351940854_wp
        Xi(17) = 26.056100760949900310703419847869_wp
        Xi(18) = 33.866121529556390601367610315720_wp
        Xi(19) = 44.181978651464860967379433276392_wp
        Xi(20) = 57.858847845664609296778557734342_wp
        Xi(21) = 76.055565702005308074962197650848_wp
        Xi(22) = 100.336391411221727130997916077604_wp
        Xi(23) = 132.787913973737655648488242832173_wp
        Xi(24) = 176.115722850641647431735314910384_wp
        Xi(25) = 233.605229592472310090611742339206_wp
        Xi(26) = 308.627273030173973150658284225756_wp
        Xi(27) = 402.929885356120519035449945022265_wp
        Xi(28) = 512.369532146612567280463679253444_wp
        Xi(29) = 619.558212002491375869350775928979_wp
        Xi(30) = 690.107176530776788569987445498555_wp

      case (19)
        ! ----- K = 15 , R = 8E2 -----

        RIni = 8.0e2_wp

        Omega(1) = 0.0017870212715594367768755893768027609436_wp
        Omega(2) = 0.0045376002737008805430341375295594241734_wp
        Omega(3) = 0.0084290300565706590425532336412206557696_wp
        Omega(4) = 0.0147750621270927640636918989729131634903_wp
        Omega(5) = 0.0257069116625649650275796821025098282121_wp
        Omega(6) = 0.0444686874865422564499697205553374601550_wp
        Omega(7) = 0.0760818511689729804129528854816921779047_wp
        Omega(8) = 0.1284487982735185641241029697035003209749_wp
        Omega(9) = 0.2139808776813502482990782260774409451187_wp
        Omega(10) = 0.3520892514244785430257284053467969897611_wp
        Omega(11) = 0.5733069875972591514191531447686145384068_wp
        Omega(12) = 0.9272271445141610459101701169348785924740_wp
        Omega(13) = 1.5023065731054268420827796370886630938912_wp
        Omega(14) = 2.4959879568923889199376364711469022950041_wp
        Omega(15) = 4.6395958902014889061078684040495545559679_wp

        Alpha(1) = 0.0006895611127582455120514943612726466160_wp
        Alpha(2) = 0.0037937712630365261380920618690740231393_wp
        Alpha(3) = 0.0101383023615175421161443470620400209015_wp
        Alpha(4) = 0.0214609580225112899923365760573856775295_wp
        Alpha(5) = 0.0412052639215424465863278884886211628213_wp
        Alpha(6) = 0.0754655648230115803606118246582568076519_wp
        Alpha(7) = 0.1343950641624070088567081959762816723014_wp
        Alpha(8) = 0.2344981727691005559934206772809694996340_wp
        Alpha(9) = 0.4022718771382511915063635854483692355643_wp
        Alpha(10) = 0.6798660151460762128333356613119065059436_wp
        Alpha(11) = 1.1339537645619051447164193913152985260240_wp
        Alpha(12) = 1.8703271767807595980228838050685169491771_wp
        Alpha(13) = 3.0611464206734718420353791312749081043876_wp
        Alpha(14) = 5.0107183382202085176505834684945739354589_wp
        Alpha(15) = 8.4064479210506800003349647809613998106215_wp

        Xi(1) = 1.006721380456283441583338611469_wp
        Xi(2) = 1.061091517463179429944276910280_wp
        Xi(3) = 1.173048717630027889353636072478_wp
        Xi(4) = 1.349163118214007318451490291533_wp
        Xi(5) = 1.599742715330653725991708058451_wp
        Xi(6) = 1.939594452789270081160231951056_wp
        Xi(7) = 2.389224015730618983291422274817_wp
        Xi(8) = 2.976488293178805926266286552284_wp
        Xi(9) = 3.738771278074708967953040938959_wp
        Xi(10) = 4.725886599094059556565305957321_wp
        Xi(11) = 6.004028122378091374480857389528_wp
        Xi(12) = 7.661167731594584548535792345447_wp
        Xi(13) = 9.814433817982832590487429769155_wp
        Xi(14) = 12.620225238511970131116690652107_wp
        Xi(15) = 16.288094165448597394618701628133_wp
        Xi(16) = 21.099879428735870950245567811976_wp
        Xi(17) = 27.436164885629690362786603508027_wp
        Xi(18) = 35.812882916516620902813716753599_wp
        Xi(19) = 46.932016422984952682545545243897_wp
        Xi(20) = 61.751301777339101615332817019066_wp
        Xi(21) = 81.578810242068995624287186529955_wp
        Xi(22) = 108.196517656471522729477729996006_wp
        Xi(23) = 144.008829305045290447573336223286_wp
        Xi(24) = 192.181578658054551173028912103291_wp
        Xi(25) = 256.648884145825445735056291596266_wp
        Xi(26) = 341.628757450284282204400199134398_wp
        Xi(27) = 449.745499932875416104760546431862_wp
        Xi(28) = 577.021055192766996244646549030222_wp
        Xi(29) = 703.632253977297076108499140900676_wp
        Xi(30) = 788.083377984554580786280553184042_wp

      case (20)
        ! ----- K = 15 , R = 9E2 -----

        RIni = 9.0e2_wp

        Omega(1) = 0.0016219084670863559936738798323944965585_wp
        Omega(2) = 0.0041347531000776380092693500570288911433_wp
        Omega(3) = 0.0077404596712594351285001176721928262126_wp
        Omega(4) = 0.0137109856821038252554885092382441769132_wp
        Omega(5) = 0.0241104755830180065159318390094478345986_wp
        Omega(6) = 0.0421003703750971199021118068078761709216_wp
        Omega(7) = 0.0726196964790066316062285695087652470647_wp
        Omega(8) = 0.1234948615707579749908406711866870608674_wp
        Omega(9) = 0.2070802540837517613519927831000622120428_wp
        Omega(10) = 0.3427777482480647935046431246108866730538_wp
        Omega(11) = 0.5612008838336487104238477219286806985110_wp
        Omega(12) = 0.9121508407153058013770545453535731894590_wp
        Omega(13) = 1.4844032778764126925818978008919657440856_wp
        Omega(14) = 2.4756385101249463417168866774531466035114_wp
        Omega(15) = 4.6164204896316331934967569505801066043205_wp

        Alpha(1) = 0.0006255780084365577944930178347707205866_wp
        Alpha(2) = 0.0034485635069017050178818035175223877253_wp
        Alpha(3) = 0.0092527900519063576451213835871723034643_wp
        Alpha(4) = 0.0197090055472030227261737069660574306340_wp
        Alpha(5) = 0.0381394934269198958134755497034751314800_wp
        Alpha(6) = 0.0704408742294657181496112495489025917550_wp
        Alpha(7) = 0.1264852816914941078895286488692484283547_wp
        Alpha(8) = 0.2224163364079585390541528647223579184811_wp
        Alpha(9) = 0.3843020768245482766045303946178535170475_wp
        Alpha(10) = 0.6538272042959846833674354693588526288295_wp
        Alpha(11) = 1.0972315718134093568336337454205420272046_wp
        Alpha(12) = 1.8200161955372063935561927072726007281744_wp
        Alpha(13) = 2.9943196982235439904725810000840624525154_wp
        Alpha(14) = 4.9247349165318251985705433337869862953085_wp
        Alpha(15) = 8.2988518304647784879987670869638805015711_wp

        Xi(1) = 1.006853176555457203339080662907_wp
        Xi(2) = 1.062302369921885844810532928406_wp
        Xi(3) = 1.176550488474161181560485867070_wp
        Xi(4) = 1.356441416966551330685990250924_wp
        Xi(5) = 1.612724004189677166377914208084_wp
        Xi(6) = 1.960865988460154367379814632777_wp
        Xi(7) = 2.422337979587958453273066616873_wp
        Xi(8) = 3.026382766910526195009573102723_wp
        Xi(9) = 3.812350949066554055581421134313_wp
        Xi(10) = 4.832831650830515514156299206050_wp
        Xi(11) = 6.157930403730347043711468302263_wp
        Xi(12) = 7.881141126194877331075788751136_wp
        Xi(13) = 10.127409080838835894050609098826_wp
        Xi(14) = 13.064222849896538832144277098024_wp
        Xi(15) = 16.916925988773785146171424464967_wp
        Xi(16) = 21.989942371927765676561938867906_wp
        Xi(17) = 28.696263145898042862258492391447_wp
        Xi(18) = 37.598595376417315718636791288532_wp
        Xi(19) = 49.466694830718208695380377903916_wp
        Xi(20) = 65.357108688094277620939109851861_wp
        Xi(21) = 86.722585457944927449236427463575_wp
        Xi(22) = 115.557983430261986559917097849848_wp
        Xi(23) = 154.581192990693060101969891206863_wp
        Xi(24) = 207.417307737267877260078918766339_wp
        Xi(25) = 278.657932364932122315526186184798_wp
        Xi(26) = 373.401149784383102586682312562516_wp
        Xi(27) = 495.229957264965882840490607463835_wp
        Xi(28) = 640.482160013576152790548690063588_wp
        Xi(29) = 787.017026953280821321978777405093_wp
        Xi(30) = 885.957745722293947610026521033433_wp

      case (21)
        ! ----- K = 15 , R = 1E3 -----

        RIni = 1.0e3_wp

        Omega(1) = 0.0014871445606266374690648170305527775437_wp
        Omega(2) = 0.0038052862730260423568116905828143115009_wp
        Omega(3) = 0.0071751648320479873574389824506664092851_wp
        Omega(4) = 0.0128319872157548835371414751305163548523_wp
        Omega(5) = 0.0227811357330662754630274790396127215786_wp
        Omega(6) = 0.0401125718299626029395140922040319253483_wp
        Omega(7) = 0.0696934521423355049618672714395462008952_wp
        Omega(8) = 0.1192823247668556892327976909351505696577_wp
        Omega(9) = 0.2011809721541527566244979968734796216268_wp
        Omega(10) = 0.3347795767086457424199872484615880807723_wp
        Omega(11) = 0.5507588922017481879883117612539678020767_wp
        Omega(12) = 0.8991015649090148173744156678832695206438_wp
        Omega(13) = 1.4688669259558777996929959663852116591443_wp
        Omega(14) = 2.4579547964873604457523847965916274915799_wp
        Omega(15) = 4.5962843562528531344592297447348983041593_wp

        Alpha(1) = 0.0005733678386173403022237547407066043936_wp
        Alpha(2) = 0.0031665856795662918647250551806501817076_wp
        Alpha(3) = 0.0085280855110995089536301909903526219381_wp
        Alpha(4) = 0.0182707634739886854995414644867168796338_wp
        Alpha(5) = 0.0356113865418937814485283999133002907911_wp
        Alpha(6) = 0.0662745665478196662695844522394938991283_wp
        Alpha(7) = 0.1198882170585382153727761235517679239138_wp
        Alpha(8) = 0.2122814420767166918117005783295248022569_wp
        Alpha(9) = 0.3691460181815040449419991486212921927290_wp
        Alpha(10) = 0.6317554902189304127609927042019677401186_wp
        Alpha(11) = 1.0659628718482381493862926058469042800425_wp
        Alpha(12) = 1.7770046979357533404068544835574527951394_wp
        Alpha(13) = 2.9369927346261072438182021526742460082460_wp
        Alpha(14) = 4.8507701846577407742318321925978352737729_wp
        Alpha(15) = 8.2061044496867307255702495005778018821729_wp

        Xi(1) = 1.006970073202688794916821735104_wp
        Xi(2) = 1.063376780618819135437735923233_wp
        Xi(3) = 1.179660120678657380158781131652_wp
        Xi(4) = 1.362911892369016381935248327650_wp
        Xi(5) = 1.624280629534539719662986079207_wp
        Xi(6) = 1.979835147831775864603678916076_wp
        Xi(7) = 2.451926363891509618288602756486_wp
        Xi(8) = 3.071065244455815873613271915765_wp
        Xi(9) = 3.878407421790135628687778979184_wp
        Xi(10) = 4.929098830293953303199883109187_wp
        Xi(11) = 6.296863303020070916100198177290_wp
        Xi(12) = 8.080323533238966634585642712096_wp
        Xi(13) = 10.411711474326583890155639711139_wp
        Xi(14) = 13.468900109276450950505921166922_wp
        Xi(15) = 17.492097426706340018709440187195_wp
        Xi(16) = 22.807059904425902811114790225133_wp
        Xi(17) = 29.857586391445431261287835056351_wp
        Xi(18) = 39.251016323276317603746532469700_wp
        Xi(19) = 51.822205030656594941518289232363_wp
        Xi(20) = 68.723077412386748143158232693395_wp
        Xi(21) = 91.547000401919210600676368727591_wp
        Xi(22) = 122.497060792932926868992193902841_wp
        Xi(23) = 164.600407649826112041480108416636_wp
        Xi(24) = 221.939527402204062070389944949511_wp
        Xi(25) = 299.769727474212866719094350287378_wp
        Xi(26) = 404.095987991380787812678576642611_wp
        Xi(27) = 539.531615295786462271898642484302_wp
        Xi(28) = 702.867585330041904578823874771842_wp
        Xi(29) = 869.767525781364489290137242960554_wp
        Xi(30) = 983.737118578837241467116569992868_wp

      case (22)
        ! ----- K = 15 , R = 2E3 -----

        RIni = 2.0e3_wp

        Omega(1) = 0.0008404149095994821547922493110149844142_wp
        Omega(2) = 0.0022127172568996186612089172029982364620_wp
        Omega(3) = 0.0044044157718100057294363658400015992811_wp
        Omega(4) = 0.0084252777388026441757196138837571197655_wp
        Omega(5) = 0.0159295973722688583018580743555503431708_wp
        Omega(6) = 0.0295938800680031685597093371598020894453_wp
        Omega(7) = 0.0538511112433189692572099716016964521259_wp
        Omega(8) = 0.0960186198421827258320249143253022339195_wp
        Omega(9) = 0.1680232026699381431278368381754262372851_wp
        Omega(10) = 0.2891110509850164378420345201448071748018_wp
        Omega(11) = 0.4903019557844095488441382713062921538949_wp
        Omega(12) = 0.8226597473598823428986293038178700953722_wp
        Omega(13) = 1.3770544599011862452186960581457242369652_wp
        Omega(14) = 2.3529519089229520290018626837991178035736_wp
        Omega(15) = 4.4767751352432112454948764934670180082321_wp

        Alpha(1) = 0.0003230240034515234528317284556919730676_wp
        Alpha(2) = 0.0018095714502588270203298215221821010346_wp
        Alpha(3) = 0.0050161890329175661332183899787651171209_wp
        Alpha(4) = 0.0112212887847932644980186722705184365623_wp
        Alpha(5) = 0.0230170410722087548371916909673018381000_wp
        Alpha(6) = 0.0451147565214759307949954347805032739416_wp
        Alpha(7) = 0.0857044983738741855683329617932031396776_wp
        Alpha(8) = 0.1587370998704414615243507569175562821329_wp
        Alpha(9) = 0.2876082176004111645717387091281125321984_wp
        Alpha(10) = 0.5110180594894182526388703990960493683815_wp
        Alpha(11) = 0.8923202702949009079702591407112777233124_wp
        Alpha(12) = 1.5349453620910749052086430310737341642380_wp
        Alpha(13) = 2.6106708645312535210791793360840529203415_wp
        Alpha(14) = 4.4258490015686451357623809599317610263824_wp
        Alpha(15) = 7.6696259660669774405050702625885605812073_wp

        Xi(1) = 1.007712242881242170611244546308_wp
        Xi(2) = 1.070207907157320281754664392793_wp
        Xi(3) = 1.199485052179940947070235779393_wp
        Xi(4) = 1.404322731537668511592187314818_wp
        Xi(5) = 1.698605010097730350082656514132_wp
        Xi(6) = 2.102551993204441416196459613275_wp
        Xi(7) = 2.644662161603932748477063796599_wp
        Xi(8) = 3.364400549289334740876711293822_wp
        Xi(9) = 4.315818878600615526863748527830_wp
        Xi(10) = 5.572556903872480482675655366620_wp
        Xi(11) = 7.234871749462114642881260806462_wp
        Xi(12) = 9.439540609574310181528744578827_wp
        Xi(13) = 12.373830976687880678355213603936_wp
        Xi(14) = 16.295323830542574938817779184319_wp
        Xi(15) = 21.560143014460372512530739186332_wp
        Xi(16) = 28.663489744676368786713283043355_wp
        Xi(17) = 38.298132791621640080848010256886_wp
        Xi(18) = 51.439448936787506738710362697020_wp
        Xi(19) = 69.469806604976923836147761903703_wp
        Xi(20) = 94.361356140622973498466308228672_wp
        Xi(21) = 128.945200347420382058771792799234_wp
        Xi(22) = 177.305932388017566836424521170557_wp
        Xi(23) = 245.348832019322287578688701614738_wp
        Xi(24) = 341.570568196029398677637800574303_wp
        Xi(25) = 477.944629773722795107460115104914_wp
        Xi(26) = 670.340593697570739095681346952915_wp
        Xi(27) = 936.217380956610213615931570529938_wp
        Xi(28) = 1282.582934856395695533137768507004_wp
        Xi(29) = 1669.807290680049618458724580705166_wp
        Xi(30) = 1957.242597377555057391873560845852_wp

      case (23)
        ! ----- K = 15 , R = 3E3 -----

        RIni = 3.0e3_wp

        Omega(1) = 0.0006021998451342176979288824166872129418_wp
        Omega(2) = 0.0016187261267440271098155723703514163869_wp
        Omega(3) = 0.0033452393365352763509459692835290578472_wp
        Omega(4) = 0.0066742118282262347194911810029094301200_wp
        Omega(5) = 0.0130868192116284632019591839188765725055_wp
        Omega(6) = 0.0250590910423157118485320117920456262084_wp
        Omega(7) = 0.0467969004387706694784250429502980850316_wp
        Omega(8) = 0.0853675374355371224287312818812534231938_wp
        Omega(9) = 0.1524652909186951293615819302829095249763_wp
        Omega(10) = 0.2672108204861978688227520245535018261762_wp
        Omega(11) = 0.4607495497177949561194883199011229635289_wp
        Omega(12) = 0.7846876596876935452571852347425362950162_wp
        Omega(13) = 1.3308927436829058854042204251832970385294_wp
        Omega(14) = 2.2998057567870609319091862454698116380314_wp
        Omega(15) = 4.4163157384143941802841226706988209116389_wp

        Alpha(1) = 0.0002309479118724317000477037857141156263_wp
        Alpha(2) = 0.0013072884159532110557924429287968148827_wp
        Alpha(3) = 0.0037002878105363975695076878989417390464_wp
        Alpha(4) = 0.0085257832976107738285915377263179948386_wp
        Alpha(5) = 0.0180659572668002429355059738555477899524_wp
        Alpha(6) = 0.0365339066158719588569810457490127930669_wp
        Alpha(7) = 0.0714083393886806190101732873254380962180_wp
        Alpha(8) = 0.1356885479689384899732677164374550216053_wp
        Alpha(9) = 0.2515721314522706339163009292558825791275_wp
        Alpha(10) = 0.4563724471127592911205272246943920322337_wp
        Alpha(11) = 0.8120423512111979215421543598640852223980_wp
        Alpha(12) = 1.4209334533330517537754247858750034083641_wp
        Alpha(13) = 2.4545240138803349828119232389767034874239_wp
        Alpha(14) = 4.2199332485013572769001755080608973003109_wp
        Alpha(15) = 7.4072221155356338874627131207972752235946_wp

        Xi(1) = 1.008121593556959773850013883933_wp
        Xi(2) = 1.073982970148324786159935162466_wp
        Xi(3) = 1.210481226968480986810883237581_wp
        Xi(4) = 1.427410523604222229959552492584_wp
        Xi(5) = 1.740314102870956596263302129657_wp
        Xi(6) = 2.171961277409063295878305366315_wp
        Xi(7) = 2.754678113087707313870466929728_wp
        Xi(8) = 3.533583750261958335986411938734_wp
        Xi(9) = 4.570995213173794194903842846855_wp
        Xi(10) = 5.952600854260180783094486001161_wp
        Xi(11) = 7.796249000786654706398465242145_wp
        Xi(12) = 10.264479569558108359300652168145_wp
        Xi(13) = 13.582447085113456232072226548269_wp
        Xi(14) = 18.063705211190198193435541007545_wp
        Xi(15) = 24.147529218143520997330586652652_wp
        Xi(16) = 32.453334343040926238815879756316_wp
        Xi(17) = 43.860699401935573860794592704337_wp
        Xi(18) = 59.627926216284838630693476702049_wp
        Xi(19) = 81.569212009265394253132441804155_wp
        Xi(20) = 112.321436402374068085074920730904_wp
        Xi(21) = 155.748476914038984877941551587810_wp
        Xi(22) = 217.555340133537157429333142033556_wp
        Xi(23) = 306.215300407791237269350403948920_wp
        Xi(24) = 434.329701855966633927685904836835_wp
        Xi(25) = 620.453803362545718458331833744523_wp
        Xi(26) = 890.859483103901161116233708980872_wp
        Xi(27) = 1278.348735289984839380572623213084_wp
        Xi(28) = 1807.026081072200998534071914036758_wp
        Xi(29) = 2432.807995726973129935188921990630_wp
        Xi(30) = 2924.603774735733815237637145401095_wp

      case (24)
        ! ----- K = 15 , R = 4E3 -----

        RIni = 4.0e3_wp

        Omega(1) = 0.0004756369084263004952714521495968358344_wp
        Omega(2) = 0.0013003520804791979888086181946053876413_wp
        Omega(3) = 0.0027672864591600532465548082207765223650_wp
        Omega(4) = 0.0056928470782110608657607554033974750496_wp
        Omega(5) = 0.0114501120106099499038605055677153554683_wp
        Omega(6) = 0.0223884810149639784851015745364977149734_wp
        Omega(7) = 0.0425633913822724507076359495849615655061_wp
        Omega(8) = 0.0788704848951632592053109967189516993358_wp
        Omega(9) = 0.1428378098730392881229780271956641968245_wp
        Omega(10) = 0.2534844400338091604662094785682668884874_wp
        Omega(11) = 0.4420178490944162802921656163501751279910_wp
        Omega(12) = 0.7603902042987330134218675048440871933053_wp
        Omega(13) = 1.3011427010528306348786931057048832371947_wp
        Omega(14) = 2.2654170754188351810838597533148686125060_wp
        Omega(15) = 4.3772021220065278606509151515524536080193_wp

        Alpha(1) = 0.0001820782633968936153992700384776116940_wp
        Alpha(2) = 0.0010395111289733385836581521692936758150_wp
        Alpha(3) = 0.0029925180411342110725723009850840483637_wp
        Alpha(4) = 0.0070545320321262105725314023752653413979_wp
        Alpha(5) = 0.0153116412226620900789464700328390556194_wp
        Alpha(6) = 0.0316633381176961214803716824356394354822_wp
        Alpha(7) = 0.0631367385810755350547128668392549144528_wp
        Alpha(8) = 0.1221173957421378924965505233246609861908_wp
        Alpha(9) = 0.2300168021593796421436729435217038997052_wp
        Alpha(10) = 0.4232217341282906642363405985696189759437_wp
        Alpha(11) = 0.7627288970210382018644068580215389374644_wp
        Alpha(12) = 1.3501284949179842941177284720843942977808_wp
        Alpha(13) = 2.3566519253922217683704098956098960115924_wp
        Alpha(14) = 4.0899108319618529120732797199622154948884_wp
        Alpha(15) = 7.2406374424386336823358800707239879557164_wp

        Xi(1) = 1.008399536089696037796485661442_wp
        Xi(2) = 1.076549175536408965432426387832_wp
        Xi(3) = 1.217972693848358699828318685299_wp
        Xi(4) = 1.443188471259828498842806010494_wp
        Xi(5) = 1.768929003052049432273551410333_wp
        Xi(6) = 2.219804307045161887888481966336_wp
        Xi(7) = 2.830928382091189915957529787960_wp
        Xi(8) = 3.651573113310687771718848759761_wp
        Xi(9) = 4.750178905489354046470101833322_wp
        Xi(10) = 6.221452381810463394973959028178_wp
        Xi(11) = 8.196544979260530701045683876771_wp
        Xi(12) = 10.857702771770388778041493971216_wp
        Xi(13) = 14.459395840240733399975625594891_wp
        Xi(14) = 19.359032227629489508052840207597_wp
        Xi(15) = 26.061870515171585991134883819598_wp
        Xi(16) = 35.287269826709305186457354963636_wp
        Xi(17) = 48.067285203290741328607804661033_wp
        Xi(18) = 65.894695807480156861390963030090_wp
        Xi(19) = 90.947445315536196770722643378804_wp
        Xi(20) = 126.432109629553380517053007281447_wp
        Xi(21) = 177.114281615756022761343224658503_wp
        Xi(22) = 250.142546888428821502681209665298_wp
        Xi(23) = 356.329570250085996191424797530090_wp
        Xi(24) = 512.113092873752612621540691861810_wp
        Xi(25) = 742.394997837475936641737916943384_wp
        Xi(26) = 1083.912131223594253581765656235802_wp
        Xi(27) = 1585.976659734769377396723655238020_wp
        Xi(28) = 2294.047326326232192128529163710482_wp
        Xi(29) = 3168.504659462954313076110679503472_wp
        Xi(30) = 3887.092106025369653066192654478073_wp

      case (25)
        ! ----- K = 15 , R = 5E3 -----

        RIni = 5.0e3_wp

        Omega(1) = 0.0003962677505883613429462231368986466218_wp
        Omega(2) = 0.0010992903894092523588024720520260613110_wp
        Omega(3) = 0.0023969412766162124992569204063111443226_wp
        Omega(4) = 0.0050508717428056814312016932575627459201_wp
        Omega(5) = 0.0103583403993260641099671804342408076849_wp
        Omega(6) = 0.0205783676303624219185660869368970082860_wp
        Omega(7) = 0.0396555901930640382722231557927061729174_wp
        Omega(8) = 0.0743565312997897404507499294346040130677_wp
        Omega(9) = 0.1360810125724631901750509219151119566504_wp
        Omega(10) = 0.2437641389423446734556217424527702064552_wp
        Omega(11) = 0.4286480516510043328097610132454420295289_wp
        Omega(12) = 0.7429320310955790694976141885774723050417_wp
        Omega(13) = 1.2796585097760807599870380713014128559735_wp
        Omega(14) = 2.2405122931894283274528723026719490007963_wp
        Omega(15) = 4.3488779989433205140809957089942372476798_wp

        Alpha(1) = 0.0001514563506470363833414023745690263345_wp
        Alpha(2) = 0.0008711260387687643076721269984070278314_wp
        Alpha(3) = 0.0025442361670791670130245825005854687362_wp
        Alpha(4) = 0.0061115723260027562432065333444808352681_wp
        Alpha(5) = 0.0135201559930968971137509428495637742174_wp
        Alpha(6) = 0.0284475633694058737973404341753402846393_wp
        Alpha(7) = 0.0575988032150850329750802099326723038075_wp
        Alpha(8) = 0.1129165520128868640980959563158148029061_wp
        Alpha(9) = 0.2152382813196922147522087420679071101404_wp
        Alpha(10) = 0.4002657944347219287585179503530241618137_wp
        Alpha(11) = 0.7282787055024561484635470254289657532354_wp
        Alpha(12) = 1.3002835551871088776154650190619577188045_wp
        Alpha(13) = 2.2873048529384315526256227224521921925771_wp
        Alpha(14) = 3.9973073734529804231820071525049797855900_wp
        Alpha(15) = 7.1215488770918427764884672814815758101759_wp

        Xi(1) = 1.008607548065553133798295804002_wp
        Xi(2) = 1.078471333272938473789304136830_wp
        Xi(3) = 1.223592701433025819340340289987_wp
        Xi(4) = 1.455050730582966612980388987353_wp
        Xi(5) = 1.790501748102732074976670140209_wp
        Xi(6) = 2.255993869817991644447993659917_wp
        Xi(7) = 2.888831206626912981782692724231_wp
        Xi(8) = 3.741566112838891348035472161904_wp
        Xi(9) = 4.887511008943874586674116500262_wp
        Xi(10) = 6.428593700672319048283059883531_wp
        Xi(11) = 8.506705591175765453736123955508_wp
        Xi(12) = 11.320115966500891183620092217410_wp
        Xi(13) = 15.147348828771383073138467878493_wp
        Xi(14) = 20.382100579139743773068871313114_wp
        Xi(15) = 27.584741503126332868786607654954_wp
        Xi(16) = 37.558943283391386352976226703504_wp
        Xi(17) = 51.466733475857053891483472440882_wp
        Xi(18) = 71.002995935133440194575449311287_wp
        Xi(19) = 98.662941095424492379817316134449_wp
        Xi(20) = 138.156390239855341187102943933951_wp
        Xi(21) = 195.056436564772982242277343800652_wp
        Xi(22) = 277.823760078639739795614005402058_wp
        Xi(23) = 399.432469610330092707473070845481_wp
        Xi(24) = 579.931423862097276067384399311777_wp
        Xi(25) = 850.331708131169334008436777594397_wp
        Xi(26) = 1257.744948155719920124262500848999_wp
        Xi(27) = 1868.606025334137808568968353029049_wp
        Xi(28) = 2752.647564697004047618023037102830_wp
        Xi(29) = 3882.125886910784884786806969714235_wp
        Xi(30) = 4845.405490236185681585112661196035_wp

      case (26)
        ! ----- K = 15 , R = 6E3 -----

        RIni = 6.0e3_wp

        Omega(1) = 0.0003414757206547954683262053514423289835_wp
        Omega(2) = 0.0009596636276879566365309104716829757820_wp
        Omega(3) = 0.0021365303957790902949651025335281451234_wp
        Omega(4) = 0.0045917780797535652267040849790202816649_wp
        Omega(5) = 0.0095656214187066115749196642051299654952_wp
        Omega(6) = 0.0192478522560639908174721359954140176285_wp
        Omega(7) = 0.0374963311723699641913535706566618088686_wp
        Omega(8) = 0.0709749985723212378787802014079577617167_wp
        Omega(9) = 0.1309799832669097009464056688421429441860_wp
        Omega(10) = 0.2363753024370086878894019200014753323558_wp
        Omega(11) = 0.4184236484184497331640338602110062993233_wp
        Omega(12) = 0.7295129765209068313947665673779852113512_wp
        Omega(13) = 1.2630808671191903606347006761012607967132_wp
        Omega(14) = 2.2212530740729632757098077622259779673186_wp
        Omega(15) = 4.3269755853895644857193270027551079692785_wp

        Alpha(1) = 0.0001303310340679438245633309440604102880_wp
        Alpha(2) = 0.0007546147404157980901201372922210836025_wp
        Alpha(3) = 0.0022321298968959612833828091071550203672_wp
        Alpha(4) = 0.0054484287226041676528279819168715247812_wp
        Alpha(5) = 0.0122450024578017911861159351845362852629_wp
        Alpha(6) = 0.0261311329097184906015960675212950192758_wp
        Alpha(7) = 0.0535658283228252219509534903102432323863_wp
        Alpha(8) = 0.1061504639425160218839578642191501245406_wp
        Alpha(9) = 0.2042761733917355039167294827873178064692_wp
        Alpha(10) = 0.3831072245356744632616575324535190816277_wp
        Alpha(11) = 0.7023544400177715455249605192022954724962_wp
        Alpha(12) = 1.2625539646581452843951756670470842891518_wp
        Alpha(13) = 2.2345533125141977257629294362217819980287_wp
        Alpha(14) = 3.9265876434223163664229833136332104004396_wp
        Alpha(15) = 7.0303439652448463047761106903266181689105_wp

        Xi(1) = 1.008772408116241343792665596801_wp
        Xi(2) = 1.079995719377837837043773716594_wp
        Xi(3) = 1.228055056080842786158865975743_wp
        Xi(4) = 1.464485279493109726482377719314_wp
        Xi(5) = 1.807695931805995871169065458695_wp
        Xi(6) = 2.284912192165491337525473625192_wp
        Xi(7) = 2.935238146811181312451077829451_wp
        Xi(8) = 3.813937210931115801395077657965_wp
        Xi(9) = 4.998363927309790924642662890420_wp
        Xi(10) = 6.596474533666179375021587638628_wp
        Xi(11) = 8.759169045930881813272206315446_wp
        Xi(12) = 11.698258701309225672700098375145_wp
        Xi(13) = 15.712703638637112711108623219758_wp
        Xi(14) = 21.227257307300300803731829546450_wp
        Xi(15) = 28.849805069089485847280696084738_wp
        Xi(16) = 39.457250974277042272070348261082_wp
        Xi(17) = 54.325448841075961631946755758804_wp
        Xi(18) = 75.327824329634581163039896978262_wp
        Xi(19) = 105.242487364381162057158647371580_wp
        Xi(20) = 148.232500152187237532630526004596_wp
        Xi(21) = 210.606114800487202343148851468868_wp
        Xi(22) = 302.032690156142679560602459787333_wp
        Xi(23) = 437.503310897037233806461387075615_wp
        Xi(24) = 640.486655729559991812038788339123_wp
        Xi(25) = 947.880459891898872692994615363205_wp
        Xi(26) = 1417.021223231072227566329502224107_wp
        Xi(27) = 2131.793653271545705640832579774724_wp
        Xi(28) = 3188.324684138302722358204732699960_wp
        Xi(29) = 4577.025028872135528068554322089767_wp
        Xi(30) = 5799.999146855984597070943209473626_wp

      case (27)
        ! ----- K = 15 , R = 7E3 -----

        RIni = 7.0e3_wp

        Omega(1) = 0.0003011910603773180930263604154978676619_wp
        Omega(2) = 0.0008564729701576365269785544678295130527_wp
        Omega(3) = 0.0019419490399198079889074133375960473646_wp
        Omega(4) = 0.0042438127675699630689655581644581516798_wp
        Omega(5) = 0.0089572919260013106854156125086197093310_wp
        Omega(6) = 0.0182166528963363917419782502318814643161_wp
        Omega(7) = 0.0358089973084950708752329136264069830986_wp
        Omega(8) = 0.0683137230812789659293985701504769991743_wp
        Omega(9) = 0.1269403467537262602578143969034307758648_wp
        Omega(10) = 0.2304915026927615025837889350945886235422_wp
        Omega(11) = 0.4102423245034995822537286780384846451852_wp
        Omega(12) = 0.7187313324855581339514060124074035229569_wp
        Omega(13) = 1.2497198359483893011483840451703031249053_wp
        Omega(14) = 2.2057032208277742770657914306298152951058_wp
        Omega(15) = 4.3092920548625502363826456075912574306130_wp

        Alpha(1) = 0.0001148082111075384276075927101853132312_wp
        Alpha(2) = 0.0006687803525803460729510951823621744960_wp
        Alpha(3) = 0.0020009370081711234145035142742777289904_wp
        Alpha(4) = 0.0049528792338325030370404979812581913734_wp
        Alpha(5) = 0.0112823279837394440386832625278223929399_wp
        Alpha(6) = 0.0243649356433474875607402693189484921277_wp
        Alpha(7) = 0.0504631660845739493800269498402499834810_wp
        Alpha(8) = 0.1009037065377226192512357706709824611835_wp
        Alpha(9) = 0.1957159032289775951447916707559571136699_wp
        Alpha(10) = 0.3696251344057429575890982159647180083084_wp
        Alpha(11) = 0.6818738938022625279658690822248701124408_wp
        Alpha(12) = 1.2326061877222592045449944819779375393409_wp
        Alpha(13) = 2.1925153876058049134522426326654453987430_wp
        Alpha(14) = 3.8700528921889102919491609799962361648795_wp
        Alpha(15) = 6.9572668947263066707535628108161063209991_wp

        Xi(1) = 1.008908127362487775098565112586_wp
        Xi(2) = 1.081251304022287459743675852586_wp
        Xi(3) = 1.231734114590510359627840986096_wp
        Xi(4) = 1.472274286191225951586809828076_wp
        Xi(5) = 1.821915543211667064802707083349_wp
        Xi(6) = 2.308876722519944008678999258777_wp
        Xi(7) = 2.973788827437679623075933443843_wp
        Xi(8) = 3.874220815397069276600147680334_wp
        Xi(9) = 5.090980897802434691420958579045_wp
        Xi(10) = 6.737194374631522964923940888227_wp
        Xi(11) = 8.971532815073269094031938042377_wp
        Xi(12) = 12.017528487409227991485061171062_wp
        Xi(13) = 16.191946054764509448434250948878_wp
        Xi(14) = 21.946733172542159535575723161571_wp
        Xi(15) = 29.931619394205071591624767179951_wp
        Xi(16) = 41.088415407792347463361570802931_wp
        Xi(17) = 56.794532837777393287359561924177_wp
        Xi(18) = 79.083834064138912312014184635700_wp
        Xi(19) = 110.990594989350655014126445507827_wp
        Xi(20) = 157.091701137116315778463615515648_wp
        Xi(21) = 224.372574949777182182208967731185_wp
        Xi(22) = 323.626790368828476635032487251920_wp
        Xi(23) = 471.741810947603555898011151725768_wp
        Xi(24) = 695.440590353829698411214366160493_wp
        Xi(25) = 1037.302986492132237716923270909319_wp
        Xi(26) = 1564.716173465391656383083329728834_wp
        Xi(27) = 2379.177951955116393278899522556458_wp
        Xi(28) = 3604.806434049069072456106255231134_wp
        Xi(29) = 5255.568707099384338388148307785741_wp
        Xi(30) = 6751.198981108692237729229645992746_wp

      case (28)
        ! ----- K = 15 , R = 8E3 -----

        RIni = 8.0e3_wp

        Omega(1) = 0.0002702223025865613888936383161923707186_wp
        Omega(2) = 0.0007767771880651208602113168201919873868_wp
        Omega(3) = 0.0017901805651350507902784794304722193470_wp
        Omega(4) = 0.0039690416370657308611156728964108886260_wp
        Omega(5) = 0.0084718887508346092019530172885499741398_wp
        Omega(6) = 0.0173869880469622443877852598428934705055_wp
        Omega(7) = 0.0344420673061209607173889860054494249653_wp
        Omega(8) = 0.0661449935763909648413303582481503894996_wp
        Omega(9) = 0.1236312094677251008099428709885447830175_wp
        Omega(10) = 0.2256494857804086714162189811205827538743_wp
        Omega(11) = 0.4034823918551539799601159813624562389123_wp
        Omega(12) = 0.7097924619431462531682526517773368368580_wp
        Omega(13) = 1.2386136266646492081790198458435270367772_wp
        Omega(14) = 2.1927583985724614741227855851946060283808_wp
        Omega(15) = 4.2945711898528852608868111584428106652922_wp

        Alpha(1) = 0.0001028812551570597490250812923917502623_wp
        Alpha(2) = 0.0006026772000226355159545637738775086811_wp
        Alpha(3) = 0.0018220066643375311307608615195817447496_wp
        Alpha(4) = 0.0045663474136871346808813545287142332540_wp
        Alpha(5) = 0.0105247448017949373921011953572635100329_wp
        Alpha(6) = 0.0229632113568895219124569835947302642865_wp
        Alpha(7) = 0.0479820641000258561322216847855326449235_wp
        Alpha(8) = 0.0966799894828725857719316748306592756990_wp
        Alpha(9) = 0.1887842738725176050963022733597540536721_wp
        Alpha(10) = 0.3586516625412090184996626029967359272632_wp
        Alpha(11) = 0.6651285739047628965572682424589601168918_wp
        Alpha(12) = 1.2080240390876300740933754762629348533665_wp
        Alpha(13) = 2.1578952260979739972545504711476382908586_wp
        Alpha(14) = 3.8233719362259296979233286206678599228326_wp
        Alpha(15) = 6.8968130432491910890464847927461278231931_wp

        Xi(1) = 1.009022925974474242508983901079_wp
        Xi(2) = 1.082313804505225142828066242640_wp
        Xi(3) = 1.234849928760580450729303358948_wp
        Xi(4) = 1.478878287117131369089208237089_wp
        Xi(5) = 1.833989011332974671409506983721_wp
        Xi(6) = 2.329259458577866743252821057908_wp
        Xi(7) = 3.006643989535890256316677526804_wp
        Xi(8) = 3.925714789753387274793758332514_wp
        Xi(9) = 5.170291987700855087205231708225_wp
        Xi(10) = 6.858027474045047244648187767524_wp
        Xi(11) = 9.154418019810454968006319109719_wp
        Xi(12) = 12.293343604743625133678130989523_wp
        Xi(13) = 16.607340928101278137302143633747_wp
        Xi(14) = 22.572571322579393216303800251410_wp
        Xi(15) = 30.876210603182207016167981628030_wp
        Xi(16) = 42.518450554766183475258189972124_wp
        Xi(17) = 58.968566035945209131974298344403_wp
        Xi(18) = 82.406414797020995630094652995012_wp
        Xi(19) = 116.100887969858598533501314875593_wp
        Xi(20) = 165.010501379140145580826093407723_wp
        Xi(21) = 236.750066134443987078683768743304_wp
        Xi(22) = 343.166569791899711577842779774983_wp
        Xi(23) = 502.940653239856677730879042087508_wp
        Xi(24) = 745.904639267252305201427731162767_wp
        Xi(25) = 1120.132762222368250637494213606260_wp
        Xi(26) = 1702.880464722715736392011365296639_wp
        Xi(27) = 2613.319609582309821060874810427777_wp
        Xi(28) = 4004.791123529432345540257642824145_wp
        Xi(29) = 5919.534383978480728494275808770908_wp
        Xi(30) = 7699.252399038355023197510718091507_wp

      case (29)
        ! ----- K = 15 , R = 9E3 -----

        RIni = 9.0e3_wp

        Omega(1) = 0.0002456102897028886932859597773967999523_wp
        Omega(2) = 0.0007131730524152647808370791436462926560_wp
        Omega(3) = 0.0016679632044389639519804240932673855546_wp
        Omega(4) = 0.0037453590318413544034607354053789673998_wp
        Omega(5) = 0.0080731774364872896340873794848724820383_wp
        Omega(6) = 0.0167006499760849031391811295513272206392_wp
        Omega(7) = 0.0333046051797057354854048570602653711603_wp
        Omega(8) = 0.0643311699365383485790261496040098876392_wp
        Omega(9) = 0.1208512826462217910815865294649196215460_wp
        Omega(10) = 0.2215658405811709383422761915349852301915_wp
        Omega(11) = 0.3977616018252502192744084891229405798185_wp
        Omega(12) = 0.7022056961576497009455168074598674365916_wp
        Omega(13) = 1.2291664486410005130033115672283372532547_wp
        Omega(14) = 2.1817332841699960041797340393188164853200_wp
        Omega(15) = 4.2820334602316590202376878071532928515808_wp

        Alpha(1) = 0.0000934068720182621284586280672491975796_wp
        Alpha(2) = 0.0005500573606271143280303282958182409246_wp
        Alpha(3) = 0.0016789264798342002106741011556965970097_wp
        Alpha(4) = 0.0042550759956803765393242131205575606856_wp
        Alpha(5) = 0.0099098689489630684085013073020276053171_wp
        Alpha(6) = 0.0218171234245755086953886744960406929295_wp
        Alpha(7) = 0.0459401420899353520498271160471537299941_wp
        Alpha(8) = 0.0931839435803971711893048911201642425794_wp
        Alpha(9) = 0.1830179615796721870422253397592271539907_wp
        Alpha(10) = 0.3494826683901751471369133417166352728600_wp
        Alpha(11) = 0.6510827012489291688011362746468790874133_wp
        Alpha(12) = 1.1873356150225313516023001314536600148131_wp
        Alpha(13) = 2.1286767960585158574945974230274714500410_wp
        Alpha(14) = 3.7838867200618920482932394655506413982948_wp
        Alpha(15) = 6.8455960408957202303729949743882343682344_wp

        Xi(1) = 1.009122023844475767246145481160_wp
        Xi(2) = 1.083231329301445447217504269855_wp
        Xi(3) = 1.237542464500154619695161672954_wp
        Xi(4) = 1.484590695549148624049137779046_wp
        Xi(5) = 1.844444936881460817912695149268_wp
        Xi(6) = 2.346937643605285799550466685481_wp
        Xi(7) = 3.035188530648495507276163341004_wp
        Xi(8) = 3.970540103055201205722521407004_wp
        Xi(9) = 5.239480434723596254803501537367_wp
        Xi(10) = 6.963684535560758376900730715064_wp
        Xi(11) = 9.314733335740850671963242390916_wp
        Xi(12) = 12.535767831380382032226139532582_wp
        Xi(13) = 16.973493003353009556458119710953_wp
        Xi(14) = 23.125903476756856811877294255453_wp
        Xi(15) = 31.714084046546791672768761394252_wp
        Xi(16) = 43.791335855977624912788170874478_wp
        Xi(17) = 60.910906328881381961715124617740_wp
        Xi(18) = 85.386803918233194286846465814733_wp
        Xi(19) = 120.704718237470903959007717531904_wp
        Xi(20) = 172.177979834423158303491341314384_wp
        Xi(21) = 248.010207216664290336138876114092_wp
        Xi(22) = 361.041457459874378377318038246813_wp
        Xi(23) = 531.655741626293901425093224588636_wp
        Xi(24) = 792.666241711624215693365869128684_wp
        Xi(25) = 1197.468966272857073263047311684204_wp
        Xi(26) = 1833.004145658207951763429832681140_wp
        Xi(27) = 2836.113866479956576727161632334173_wp
        Xi(28) = 4390.328081800878949003674733830849_wp
        Xi(29) = 6570.312494082755212954083390286542_wp
        Xi(30) = 8644.355539147717990111630115279695_wp

      case (30)
        ! ----- K = 15 , R = 1E4 -----

        RIni = 1.0e4_wp

        Omega(1) = 0.0002255397353676053207922755160881253533_wp
        Omega(2) = 0.0006611042867166782100040605708868302059_wp
        Omega(3) = 0.0015670812397325413366538885838637196191_wp
        Omega(4) = 0.0035589310274069557899554296174354783844_wp
        Omega(5) = 0.0077382539600035213085181345048224610395_wp
        Omega(6) = 0.0161205385911687138160550050031138979989_wp
        Omega(7) = 0.0323382459969524620993670815938259011091_wp
        Omega(8) = 0.0627833806660786870391163100313391964846_wp
        Omega(9) = 0.1184699102140543149042604953447010984746_wp
        Omega(10) = 0.2180557225882068321363592075523563096340_wp
        Omega(11) = 0.3928295685505218414498677936830262069634_wp
        Omega(12) = 0.6956484406594744318900157276086559932082_wp
        Omega(13) = 1.2209854833881973573119980591705768802058_wp
        Omega(14) = 2.1721753157989217952565652591800926529686_wp
        Omega(15) = 4.2711641588735770264442792587544772686670_wp

        Alpha(1) = 0.0000856839825970028654958073832074039872_wp
        Alpha(2) = 0.0005070831754268762750049198586958787427_wp
        Alpha(3) = 0.0015615827771326924017788347867803644409_wp
        Alpha(4) = 0.0039981485316142193332287851047224247480_wp
        Alpha(5) = 0.0093987693813625961427162067846063919774_wp
        Alpha(6) = 0.0208582420419701535417784151599240871633_wp
        Alpha(7) = 0.0442219155511724033146426626056380015939_wp
        Alpha(8) = 0.0902273275897956522284483645013342112406_wp
        Alpha(9) = 0.1781199701205341328968468142535108711400_wp
        Alpha(10) = 0.3416644402189818329634331262045776611558_wp
        Alpha(11) = 0.6390657647645775942392247503942570574509_wp
        Alpha(12) = 1.1695841604775098689517759398270868587133_wp
        Alpha(13) = 2.1035450462970592940079722632518155478465_wp
        Alpha(14) = 3.7498584977918745053497573405110188105027_wp
        Alpha(15) = 6.8013961440006963335598577646834428378497_wp

        Xi(1) = 1.009208933910929662631193737266_wp
        Xi(2) = 1.084036269593066855094522227265_wp
        Xi(3) = 1.239906037642474827937220938079_wp
        Xi(4) = 1.489609395458397731208319281215_wp
        Xi(5) = 1.853640840419345385482914612130_wp
        Xi(6) = 2.362505482913046469129608739834_wp
        Xi(7) = 3.060362938579087490513844227458_wp
        Xi(8) = 4.010140331786234245935174325126_wp
        Xi(9) = 5.300718507532951047356295504187_wp
        Xi(10) = 7.057389212797474001108433139251_wp
        Xi(11) = 9.457224604316827683628265965865_wp
        Xi(12) = 12.751739902731012975614643956135_wp
        Xi(13) = 17.300504874605341535526159546521_wp
        Xi(14) = 23.621403717169431058994777394844_wp
        Xi(15) = 32.466521527743094306661664205649_wp
        Xi(16) = 44.937910036206841608852347036418_wp
        Xi(17) = 62.666224902694419925108793023583_wp
        Xi(18) = 88.089710166909260147249938199820_wp
        Xi(19) = 124.895778451849706613963064683048_wp
        Xi(20) = 178.729665715905407158414774926314_wp
        Xi(21) = 258.349246559639066311842015011280_wp
        Xi(22) = 377.534671007663655301067606728793_wp
        Xi(23) = 558.294837208122016192213266094768_wp
        Xi(24) = 836.308132639007831943267490260041_wp
        Xi(25) = 1270.133443643806666956130868584296_wp
        Xi(26) = 1956.215704141925878389862702988466_wp
        Xi(27) = 3049.016702403769551121825998052373_wp
        Xi(28) = 4763.025285042322653161761536466656_wp
        Xi(29) = 7209.027731618097082133544972748496_wp
        Xi(30) = 9586.668367498586480479616511729546_wp

      case (31)
        ! ----- K = 15 , R = 2E6 (Best : R = 1175000) -----

        RIni = 2.0e6_wp

        Omega(1) = 0.0000105228176670261742840088471752704995_wp
        Omega(2) = 0.0000691367855732188023467920262056778807_wp
        Omega(3) = 0.0002830208384686293506941606593681898252_wp
        Omega(4) = 0.0009200464898989470742820623522587042409_wp
        Omega(5) = 0.0025901722444491988862050644388743125823_wp
        Omega(6) = 0.0065921351784279965347346512578863286791_wp
        Omega(7) = 0.0155513212399444333759425873262016715159_wp
        Omega(8) = 0.0345577940336581698850743678969044125893_wp
        Omega(9) = 0.0731454596883880376191437722166455159822_wp
        Omega(10) = 0.1486679642694432365698700010403499049971_wp
        Omega(11) = 0.2920249462823511854634838530442308979218_wp
        Omega(12) = 0.5577584133433053689807464003980186362242_wp
        Omega(13) = 1.0451182163084206676361212484671625588817_wp
        Omega(14) = 1.9640378140134062713833412727737481873191_wp
        Omega(15) = 4.0343871534325909029329582100587003878900_wp

        Alpha(1) = 0.0000032916678993378630075590744267626528_wp
        Alpha(2) = 0.0000362116499423094770823232623331534505_wp
        Alpha(3) = 0.0001916520413872042188449850493459775125_wp
        Alpha(4) = 0.0007397073815295242332734412912982613353_wp
        Alpha(5) = 0.0023688276459558680546775311656984563946_wp
        Alpha(6) = 0.0066832262130187145182690696548766240426_wp
        Alpha(7) = 0.0171791919033504756800207575670458304273_wp
        Alpha(8) = 0.0410873650857746208155845454162991003955_wp
        Alpha(9) = 0.0927379070681148618668409103205085841637_wp
        Alpha(10) = 0.1995445819374171568023499864774628065334_wp
        Alpha(11) = 0.4124340749566497427288855326388272715121_wp
        Alpha(12) = 0.8239287603035586122677234877098584320265_wp
        Alpha(13) = 1.6008068838318576734705903086286582492903_wp
        Alpha(14) = 3.0544270604919671998137009616414161428111_wp
        Alpha(15) = 5.8841569972355831702123807325222060171654_wp

        Xi(1) = 1.011356431856024272897959237483_wp
        Xi(2) = 1.104003921877126792355956752534_wp
        Xi(3) = 1.298967172564399300840019824133_wp
        Xi(4) = 1.616293799078014142671377850746_wp
        Xi(5) = 2.088774395352814420342285606225_wp
        Xi(6) = 2.766850077561704469375838955081_wp
        Xi(7) = 3.726393897993726684247581260045_wp
        Xi(8) = 5.080016080628945361781501066645_wp
        Xi(9) = 6.994131337891847674612805230154_wp
        Xi(10) = 9.715521417601100629193866531352_wp
        Xi(11) = 13.612939230868090786583102236307_wp
        Xi(12) = 19.242846449322077027194177745173_wp
        Xi(13) = 27.454670425888878579692709003268_wp
        Xi(14) = 39.561788791582659235990249158021_wp
        Xi(15) = 57.624125410856096875938936463513_wp
        Xi(16) = 84.923776643453565603214538981547_wp
        Xi(17) = 126.782233409247627554938819827157_wp
        Xi(18) = 191.997991084500353370367875527336_wp
        Xi(19) = 295.438020506877174947701902851804_wp
        Xi(20) = 462.860698788542593656858414874478_wp
        Xi(21) = 740.162676662045127262690158431724_wp
        Xi(22) = 1211.899635043392013278307217660767_wp
        Xi(23) = 2039.943611755279275210561706899171_wp
        Xi(24) = 3548.928812212822196281436504250451_wp
        Xi(25) = 6428.924951436530012660597321882960_wp
        Xi(26) = 12252.483153235444802930942387320101_wp
        Xi(27) = 24984.063501351675157025056250859052_wp
        Xi(28) = 56001.512880335759721361910123960115_wp
        Xi(29) = 145387.166005767926492353581124916673_wp
        Xi(30) = 511088.658350937068661323792184703052_wp

    end select

  case (16)
    ! ===== K = 16 =====

    select case (InitR)

      case default !(1:3)
        ! ----- K = 16 , R = 1E1 -----

        RIni = 1.0e1_wp

        Omega(1) = 0.0544693993323789184580487575393359378495_wp
        Omega(2) = 0.1282577437959811045254613498522999748275_wp
        Omega(3) = 0.2057916770589411241046537068277721971299_wp
        Omega(4) = 0.2894365923450766548276173739662198158840_wp
        Omega(5) = 0.3822032310241595546359150842308238793521_wp
        Omega(6) = 0.4879433905317224986922328838767626280060_wp
        Omega(7) = 0.6116909136127484010198723796136732744344_wp
        Omega(8) = 0.7600887458718382676324358537822689640961_wp
        Omega(9) = 0.9419406889769544439193482698868820079952_wp
        Omega(10) = 1.1690433766271810673556194593025736594427_wp
        Omega(11) = 1.4577672298738048011889495425563723074447_wp
        Omega(12) = 1.8325870880315658182236476791970858357672_wp
        Omega(13) = 2.3346947673352790063924094843272882826568_wp
        Omega(14) = 3.0457271705138052335284232219336786329222_wp
        Omega(15) = 4.1712794331204028318305088163242544396780_wp
        Omega(16) = 6.5349837313441176047529301840199877915438_wp

        Alpha(1) = 0.0211963589052251348014477066658556303480_wp
        Alpha(2) = 0.1123242174600216008732342896914158103527_wp
        Alpha(3) = 0.2789480531208277305635963999597137785713_wp
        Alpha(4) = 0.5259381094628534172061122553554213254756_wp
        Alpha(5) = 0.8608525827545032450003117951986553180177_wp
        Alpha(6) = 1.2946560518738649671419832509755565297382_wp
        Alpha(7) = 1.8427240850238871151941730253120965699054_wp
        Alpha(8) = 2.5262309525576912049363298429227597807767_wp
        Alpha(9) = 3.3740205529595078720768441815636151659419_wp
        Alpha(10) = 4.4251473628903414138900662688769216401852_wp
        Alpha(11) = 5.7325633870510128630923474268854533875128_wp
        Alpha(12) = 7.3692105865439201666140800472959426770103_wp
        Alpha(13) = 9.4397718200387649243460197467925354430918_wp
        Alpha(14) = 12.1071304391623297451799357027368841954740_wp
        Alpha(15) = 15.6650471589833527241361621484827537642559_wp
        Alpha(16) = 20.8276833299618638964401107216417585732415_wp

        Xi(1) = 1.001642773023285848659619245993_wp
        Xi(2) = 1.020632283267804189482753918217_wp
        Xi(3) = 1.069694067115581228568851890337_wp
        Xi(4) = 1.079893351595470750436955853502_wp
        Xi(5) = 1.165776128629064865328823774604_wp
        Xi(6) = 1.215427455094397498291079739996_wp
        Xi(7) = 1.309820334060649646076845975529_wp
        Xi(8) = 1.396768543697270465332033184680_wp
        Xi(9) = 1.524016153482674934462742788455_wp
        Xi(10) = 1.669566103906232756729291488718_wp
        Xi(11) = 1.739123483722660599612962195604_wp
        Xi(12) = 1.897139976316912667996666952419_wp
        Xi(13) = 2.012573117182326204888803689030_wp
        Xi(14) = 2.225100967753168139297986760639_wp
        Xi(15) = 2.561199438850251413647696185905_wp
        Xi(16) = 2.800852853736864682930585157550_wp
        Xi(17) = 3.121498587220625926938119554244_wp
        Xi(18) = 3.385597622280114333756517730478_wp
        Xi(19) = 3.809646075125514618885808060256_wp
        Xi(20) = 4.262560966502159083644368786636_wp
        Xi(21) = 4.990680098969088600031202007656_wp
        Xi(22) = 5.235494914747392084051341720752_wp
        Xi(23) = 5.679028495515885146399659433314_wp
        Xi(24) = 6.310242140042951990500097458892_wp
        Xi(25) = 6.906489335891794362123213835147_wp
        Xi(26) = 7.346756507326352338659025686951_wp
        Xi(27) = 8.143084406071635463104929630873_wp
        Xi(28) = 8.661425262176749761446059583392_wp
        Xi(29) = 9.103434878560661631767547685712_wp
        Xi(30) = 9.468566991168702428292858197523_wp
        Xi(31) = 9.852011085479216269573943254301_wp
        Xi(32) = 9.982569464695079830551804889183_wp

      case (4)
        ! ----- K = 16 , R = 2E1 -----

        RIni = 2.0e1_wp

        Omega(1) = 0.0321225885764125043171516096784934291009_wp
        Omega(2) = 0.0760016922055371523816134915474407307556_wp
        Omega(3) = 0.1230393637303320996076211497893826418704_wp
        Omega(4) = 0.1754101491069476755860646194351559756797_wp
        Omega(5) = 0.2360231401909021565013475224548145092740_wp
        Omega(6) = 0.3088188185834776517540989360099956684280_wp
        Omega(7) = 0.3991595234159666612846605743269989829969_wp
        Omega(8) = 0.5142450102957285566133809051603265061203_wp
        Omega(9) = 0.6635919448742423670719453587807379335572_wp
        Omega(10) = 0.8598082538447276580283878955057730308909_wp
        Omega(11) = 1.1202030710426284491497070372467703691655_wp
        Omega(12) = 1.4703548620534354144902614014434050204727_wp
        Omega(13) = 1.9525233214943743469778000831915676371864_wp
        Omega(14) = 2.6485569231618074975340149102009945636382_wp
        Omega(15) = 3.7608140990416505055043278060722400368832_wp
        Omega(16) = 6.0921817077078821980537504110486679564929_wp

        Alpha(1) = 0.0124933559834041096275154218014913354295_wp
        Alpha(2) = 0.0663634793404057955643017205349387666047_wp
        Alpha(3) = 0.1655401210940494760050483574742585801687_wp
        Alpha(4) = 0.3142122161559275833923269433123337535108_wp
        Alpha(5) = 0.5190964445166673073154064987910771833413_wp
        Alpha(6) = 0.7903036143386039961630795680491701205028_wp
        Alpha(7) = 1.1425620524652800269428329360543727943877_wp
        Alpha(8) = 1.5968455073257302928363063343475403144112_wp
        Alpha(9) = 2.1824392199329476567555563537581519995001_wp
        Alpha(10) = 2.9396026075790025980805780791982328992162_wp
        Alpha(11) = 3.9233621214215907357317908443761211856327_wp
        Alpha(12) = 5.2097574852041275347835913578364852583036_wp
        Alpha(13) = 6.9076901985583718684459653136542556239874_wp
        Alpha(14) = 9.1850123784656421332614484320799874694785_wp
        Alpha(15) = 12.3393159714999913713004420223739998618839_wp
        Alpha(16) = 17.0792800750442479908869364280121772026177_wp

        Xi(1) = 1.002189500000000066393113229424_wp
        Xi(2) = 1.019762947701370624990595270543_wp
        Xi(3) = 1.055229999999999890292201598641_wp
        Xi(4) = 1.109217220959847719375518382279_wp
        Xi(5) = 1.182708024034756535448897896146_wp
        Xi(6) = 1.277016834403164474774465542151_wp
        Xi(7) = 1.393834079811942746533574843948_wp
        Xi(8) = 1.535250464041688117709620609030_wp
        Xi(9) = 1.703819073036006557890498225305_wp
        Xi(10) = 1.902554471272783690361136976765_wp
        Xi(11) = 2.135029299496591970435738216949_wp
        Xi(12) = 2.405404258639855907819793401359_wp
        Xi(13) = 2.718495459904630815730838810929_wp
        Xi(14) = 3.079773970864370405582655099153_wp
        Xi(15) = 3.495481619359764200501844433333_wp
        Xi(16) = 3.972546210309521930170717496367_wp
        Xi(17) = 4.518617597172095341935549317469_wp
        Xi(18) = 5.141874855214699761678243117302_wp
        Xi(19) = 5.850850279044855657955420680238_wp
        Xi(20) = 6.653940585562659133268825772944_wp
        Xi(21) = 7.558725347853058861621183828206_wp
        Xi(22) = 8.570838617733095778083995952557_wp
        Xi(23) = 9.692320370862275815890263253038_wp
        Xi(24) = 10.919306587316860122915951336608_wp
        Xi(25) = 12.239073715550893941585164359509_wp
        Xi(26) = 13.626418554740222651225356464977_wp
        Xi(27) = 15.040170469656139274304551545214_wp
        Xi(28) = 16.420733253759619497263999576830_wp
        Xi(29) = 17.690605814385193066823331875526_wp
        Xi(30) = 18.759784666594276307874378240115_wp
        Xi(31) = 19.537293771777199353351095467701_wp
        Xi(32) = 19.947636191155441776762247130250_wp

      case (5)
        ! ----- K = 16 , R = 3E1 -----

        RIni = 3.0e1_wp

        Omega(1) = 0.0235609197261937656794168142321277414908_wp
        Omega(2) = 0.0559413134016891635680453990359106697383_wp
        Omega(3) = 0.0911632628103368471841698970903333787419_wp
        Omega(4) = 0.1312859571784798062951855890378993763079_wp
        Omega(5) = 0.1791533047258759359988446180356191916871_wp
        Omega(6) = 0.2387270433782989137932529014052285276648_wp
        Omega(7) = 0.3154669497444695528201585865701872535283_wp
        Omega(8) = 0.4166893847022847066912006053218675560856_wp
        Omega(9) = 0.5519940219255625856332320156028004021209_wp
        Omega(10) = 0.7340636247311380832005975327625435511436_wp
        Omega(11) = 0.9803606385561895881488800319480958478380_wp
        Omega(12) = 1.3167257512871589496435131838580900875968_wp
        Omega(13) = 1.7856400770476758351097690269426720988122_wp
        Omega(14) = 2.4686440241868266001670079168306415340339_wp
        Omega(15) = 3.5657298034729491065849410480481651575246_wp
        Omega(16) = 5.8667042716208066843210089391646988588036_wp

        Alpha(1) = 0.0091597849242320682433974644333363190896_wp
        Alpha(2) = 0.0487412456562638064932481939128816605944_wp
        Alpha(3) = 0.1219811140457653242323674165625835286164_wp
        Alpha(4) = 0.2326929954185220091424698279913840792688_wp
        Alpha(5) = 0.3871221911571684245307288507742526917355_wp
        Alpha(6) = 0.5948850330998255694957624339913593303208_wp
        Alpha(7) = 0.8702777453847756691535791917502251635597_wp
        Alpha(8) = 1.2339548618200209037900141884058768937393_wp
        Alpha(9) = 1.7149841111626251076958948882378308553598_wp
        Alpha(10) = 2.3534803627430483296753549460333942988655_wp
        Alpha(11) = 3.2044326031056090044613765543957129011687_wp
        Alpha(12) = 4.3440541494478195263390096769740011950489_wp
        Alpha(13) = 5.8816849055963979658358253832517448245198_wp
        Alpha(14) = 7.9856357540459984491881817447733737935778_wp
        Alpha(15) = 10.9528456420038435440225677020009698026115_wp
        Alpha(16) = 15.4852730910841890682103083598519788210979_wp

        Xi(1) = 1.002573734850334118379015790889_wp
        Xi(2) = 1.023246810174743838886546321021_wp
        Xi(3) = 1.065036071547796444800547399634_wp
        Xi(4) = 1.128835801500153776153433093921_wp
        Xi(5) = 1.216010039164406659876506433537_wp
        Xi(6) = 1.328419331760795168940678112612_wp
        Xi(7) = 1.468467515067892203615117219240_wp
        Xi(8) = 1.639160723274115317202141717257_wp
        Xi(9) = 1.844179561582496653629643923455_wp
        Xi(10) = 2.087971282324882142059779832621_wp
        Xi(11) = 2.375850111715012873744162780199_wp
        Xi(12) = 2.714107645002727208730780583679_wp
        Xi(13) = 3.110143980894210012731054826318_wp
        Xi(14) = 3.572596476374702785354375844662_wp
        Xi(15) = 4.111475675606219205675262440280_wp
        Xi(16) = 4.738284434211328140576502399028_wp
        Xi(17) = 5.466101472116655788074240929220_wp
        Xi(18) = 6.309585592081037487641875793898_wp
        Xi(19) = 7.284826671866312525639025610147_wp
        Xi(20) = 8.408955894819123388396475693440_wp
        Xi(21) = 9.699317473423044616044619292694_wp
        Xi(22) = 11.172013401263362592820416185901_wp
        Xi(23) = 12.839446921794874787066398358348_wp
        Xi(24) = 14.706508605280594661687820767781_wp
        Xi(25) = 16.764986980257974451269387117236_wp
        Xi(26) = 18.986058317882165179593667581059_wp
        Xi(27) = 21.311359792364128411257251372035_wp
        Xi(28) = 23.644551464645229681485472106317_wp
        Xi(29) = 25.847317894115123357301477824421_wp
        Xi(30) = 27.745714371433567072000592190761_wp
        Xi(31) = 29.152259241143720490396784583709_wp
        Xi(32) = 29.903656300768179401167179420895_wp

      case (6)
        ! ----- K = 16 , R = 4E1 -----

        RIni = 4.0e1_wp

        Omega(1) = 0.0188412226242910377874828628183556666897_wp
        Omega(2) = 0.0448605493543371719201160244261394893783_wp
        Omega(3) = 0.0734930731016188330131737836692362719759_wp
        Omega(4) = 0.1067040806278245791750857764113025183406_wp
        Omega(5) = 0.1472715918877373489289132543755833637533_wp
        Omega(6) = 0.1991328541485264322634819028623454073568_wp
        Omega(7) = 0.2677423025669556750315552368091331914002_wp
        Omega(8) = 0.3603915702110762296888485378687505544804_wp
        Omega(9) = 0.4866411695538599008474967694271029472475_wp
        Omega(10) = 0.6591954549030602137586060118579212030454_wp
        Omega(11) = 0.8956803685351774997266960376407496369211_wp
        Omega(12) = 1.2222440383488672153728532565608588811301_wp
        Omega(13) = 1.6816930580726245238578145380081707571662_wp
        Omega(14) = 2.3555756876213544785941128578699022000364_wp
        Omega(15) = 3.4425924383299705466756485128243525650760_wp
        Omega(16) = 5.7245565688195948557577541482288552288082_wp

        Alpha(1) = 0.0073225601418134040001580017595350646786_wp
        Alpha(2) = 0.0390193033202446146977791525262135863272_wp
        Alpha(3) = 0.0979067844659002423532922175508641515762_wp
        Alpha(4) = 0.1875228759783119811222970585440350532735_wp
        Alpha(5) = 0.3137542183667890708276444755586709334239_wp
        Alpha(6) = 0.4858136923194132932740142705174690718195_wp
        Alpha(7) = 0.7175823469735936248163621353768348853919_wp
        Alpha(8) = 1.0292869363680080104167871568598968678998_wp
        Alpha(9) = 1.4495317958082096967270535614602522400673_wp
        Alpha(10) = 2.0179516117564629451488350531107585084101_wp
        Alpha(11) = 2.7891432547193736283131643149957312743936_wp
        Alpha(12) = 3.8391679753262131513397853455060726446391_wp
        Alpha(13) = 5.2775471129346672432419707021722388162743_wp
        Alpha(14) = 7.2729818547780006043890210776226012967527_wp
        Alpha(15) = 10.1223022282442616168185445246763265458867_wp
        Alpha(16) = 14.5237447268137015426309188725895182869863_wp

        Xi(1) = 1.002857171005857524789632584383_wp
        Xi(2) = 1.025817267547144013986951116557_wp
        Xi(3) = 1.072287351569654696801375814985_wp
        Xi(4) = 1.143377809232345467026562002566_wp
        Xi(5) = 1.240784578294497208822595724076_wp
        Xi(6) = 1.366830740138873189946054631516_wp
        Xi(7) = 1.524530992111522788128030736488_wp
        Xi(8) = 1.717677462397433716854121266149_wp
        Xi(9) = 1.950950040217017091492487934623_wp
        Xi(10) = 2.230046143600699522918651140913_wp
        Xi(11) = 2.561835105220026761584789509740_wp
        Xi(12) = 2.954535579269154725980464637480_wp
        Xi(13) = 3.417917275365817277900382098466_wp
        Xi(14) = 3.963529860969144456893722971991_wp
        Xi(15) = 4.604950601382712837249255866112_wp
        Xi(16) = 5.358041946476412298316915894247_wp
        Xi(17) = 6.241191647301613178774187806752_wp
        Xi(18) = 7.275491251863214980759309025160_wp
        Xi(19) = 8.484770406912843634071597076485_wp
        Xi(20) = 9.895351467251455564524964536588_wp
        Xi(21) = 11.535301674615324042487418676473_wp
        Xi(22) = 13.432831884931616845367530199695_wp
        Xi(23) = 15.613323064388788258792517549267_wp
        Xi(24) = 18.094252362824670242805180642165_wp
        Xi(25) = 20.877156702410411971693182664467_wp
        Xi(26) = 23.935868424235407380187834469609_wp
        Xi(27) = 27.201080567697457033085606248335_wp
        Xi(28) = 30.543444434751223785090568085820_wp
        Xi(29) = 33.761395182338625502865836480737_wp
        Xi(30) = 36.584850427056804168107762009754_wp
        Xi(31) = 38.707611387791347672820974068486_wp
        Xi(32) = 39.852625297309455229544106202866_wp

      case (7)
        ! ----- K = 16 , R = 5E1 -----

        RIni = 5.0e1_wp

        Omega(1) = 0.0158137200328224617952739810416984411745_wp
        Omega(2) = 0.0377409803223574240874571214532373630846_wp
        Omega(3) = 0.0621065805258739136550242328910220734883_wp
        Omega(4) = 0.0907986308703175472160397385668062497643_wp
        Omega(5) = 0.1265341518126833013878351774650354855112_wp
        Omega(6) = 0.1732081162442415632880766615531165086850_wp
        Omega(7) = 0.2362277390277097480729417977585171684041_wp
        Omega(8) = 0.3228120296047016722347598044473571121671_wp
        Omega(9) = 0.4424551929503152977274198220936085590438_wp
        Omega(10) = 0.6078787063087151399222787806486678618967_wp
        Omega(11) = 0.8368663078342602444094876223967816031291_wp
        Omega(12) = 1.1558545652842125910997675419800145846239_wp
        Omega(13) = 1.6079713225351366139492456652781982029410_wp
        Omega(14) = 2.2748723034512419898574980825500801984163_wp
        Omega(15) = 3.3544410264652535695693164052855195222946_wp
        Omega(16) = 5.6229103116070464694724317933349766462925_wp

        Alpha(1) = 0.0061442796289501532090269618393424400082_wp
        Alpha(2) = 0.0327791122423217869165921502239147145019_wp
        Alpha(3) = 0.0824314340868094741013598521217620174184_wp
        Alpha(4) = 0.1584254595616176457430332100151382235254_wp
        Alpha(5) = 0.2663633767071307630763289664432846848285_wp
        Alpha(6) = 0.4151230097894758880718111410113024817292_wp
        Alpha(7) = 0.6182082205518974109899506264031998625796_wp
        Alpha(8) = 0.8954087492618855389798354638131172578142_wp
        Alpha(9) = 1.2748171403732771564832476629547386437480_wp
        Alpha(10) = 1.7955250328439662241622465366219785209978_wp
        Alpha(11) = 2.5116713611733493800946132035178948171961_wp
        Alpha(12) = 3.4990867106993718221950168123512980855594_wp
        Alpha(13) = 4.8673792671259520892755590093514683758258_wp
        Alpha(14) = 6.7855822825558887831549903513206345451181_wp
        Alpha(15) = 9.5506056509052685583177622063999478996266_wp
        Alpha(16) = 13.8582962625927245264856324524771480355412_wp

        Xi(1) = 1.003082187031355445487948441219_wp
        Xi(2) = 1.027859687592568978427298720391_wp
        Xi(3) = 1.078057395890052983585156998192_wp
        Xi(4) = 1.154974233682103314344198630170_wp
        Xi(5) = 1.260596638410348168985071481973_wp
        Xi(6) = 1.397653353732658535499115692868_wp
        Xi(7) = 1.569698023830484381250897907289_wp
        Xi(8) = 1.781222464089847532854073153263_wp
        Xi(9) = 2.037799168843202316306340016183_wp
        Xi(10) = 2.346254768002126228762169168895_wp
        Xi(11) = 2.714874319136810658543990104619_wp
        Xi(12) = 3.153641949979435070254563111547_wp
        Xi(13) = 3.674522836983694047859050346716_wp
        Xi(14) = 4.291788356200312000952867830250_wp
        Xi(15) = 5.022386334308347026218061204261_wp
        Xi(16) = 5.886344711498311887357820504363_wp
        Xi(17) = 6.907189247874329671422061194264_wp
        Xi(18) = 8.112326812021093857402176396931_wp
        Xi(19) = 9.533307696033266082948076647341_wp
        Xi(20) = 11.205810438298586911333443172367_wp
        Xi(21) = 13.169075652993514335790947722415_wp
        Xi(22) = 15.464342471304390101358161579981_wp
        Xi(23) = 18.131571280598456772509274870231_wp
        Xi(24) = 21.203392018446972654144300207690_wp
        Xi(25) = 24.694852990502020423715934249742_wp
        Xi(26) = 28.587421489567784713137221785928_wp
        Xi(27) = 32.806440728453080516585638193305_wp
        Xi(28) = 37.194018687447342474267086842588_wp
        Xi(29) = 41.485465403523819716974374571805_wp
        Xi(30) = 45.306503275734785429057493999494_wp
        Xi(31) = 48.214455529116414321533223841243_wp
        Xi(32) = 49.795804671837560206881789781619_wp

      case (8)
        ! ----- K = 16 , R = 6E1 -----

        RIni = 6.0e1_wp

        Omega(1) = 0.0136908377299396044207035376752412680901_wp
        Omega(2) = 0.0327417346325856678237932895908501862436_wp
        Omega(3) = 0.0540912157388128640461038339898314220022_wp
        Omega(4) = 0.0795625932745488592432261433506912595703_wp
        Omega(5) = 0.1118172832095265544061061559200087600630_wp
        Omega(6) = 0.1547008231015822485966216062736044989379_wp
        Omega(7) = 0.2135556027713844064370735803537471042546_wp
        Omega(8) = 0.2955133019832536639664305466723703830212_wp
        Omega(9) = 0.4100001524248211071165069047106399580116_wp
        Omega(10) = 0.5697558701772300694178043090065699516344_wp
        Omega(11) = 0.7927078885207336509722587936099813532564_wp
        Omega(12) = 1.1055499871864973630798331671876155724021_wp
        Omega(13) = 1.5517056878524865918015859800327405082498_wp
        Omega(14) = 2.2129740071556539273648495091961763137078_wp
        Omega(15) = 3.2866778023493446055612599021600317428238_wp
        Omega(16) = 5.5448487278535236997580037954236331643187_wp

        Alpha(1) = 0.0053182066261874213429777932208436519446_wp
        Alpha(2) = 0.0284011328189939031852584557999441550180_wp
        Alpha(3) = 0.0715605398648006798946008370776450391304_wp
        Alpha(4) = 0.1379484239253294434551745314965209843194_wp
        Alpha(5) = 0.2329335881567227507148122242486287802876_wp
        Alpha(6) = 0.3651092254541286646527496750902486155610_wp
        Alpha(7) = 0.5476393721885706778232903724212832230478_wp
        Alpha(8) = 0.7998966011997658137069010242381494890651_wp
        Alpha(9) = 1.1494732580738143478760529547777480274817_wp
        Alpha(10) = 1.6349308451367353002678037277028977314330_wp
        Alpha(11) = 2.3099578735226667029582781065499830219778_wp
        Alpha(12) = 3.2501382678703844913373904867981423194578_wp
        Alpha(13) = 4.5651196769680073232061667809489335922990_wp
        Alpha(14) = 6.4242183428527932921407728183993413040298_wp
        Alpha(15) = 9.1244977187338398849636433851628680713475_wp
        Alpha(16) = 13.3601334728373712015109253670175348815974_wp

        Xi(1) = 1.003268954923555123567163738230_wp
        Xi(2) = 1.029556008797791147313382964512_wp
        Xi(3) = 1.082855440022723218199303396592_wp
        Xi(4) = 1.164634111432245486771573650397_wp
        Xi(5) = 1.277138088539143851948311214617_wp
        Xi(6) = 1.423459235187697745068750865283_wp
        Xi(7) = 1.607636573966127172224775676579_wp
        Xi(8) = 1.834794677819060208920007304378_wp
        Xi(9) = 2.111319404311678209987951371218_wp
        Xi(10) = 2.445071479099704985470109530610_wp
        Xi(11) = 2.845641348624389181144744842555_wp
        Xi(12) = 3.324652574133972351196758543423_wp
        Xi(13) = 3.896122654636821812119268360064_wp
        Xi(14) = 4.576888108617195779874442518320_wp
        Xi(15) = 5.387098391416639992478054832503_wp
        Xi(16) = 6.350773440036542298418875684973_wp
        Xi(17) = 7.496409682465592422793976279394_wp
        Xi(18) = 8.857587984687720121523835103972_wp
        Xi(19) = 10.473497990380090252350575585893_wp
        Xi(20) = 12.389204257258051825214828500776_wp
        Xi(21) = 14.655349583419857167551625387603_wp
        Xi(22) = 17.326755222748656678002920905612_wp
        Xi(23) = 20.459029270439212126395100987963_wp
        Xi(24) = 24.101782182788760300687358473226_wp
        Xi(25) = 28.286424404647261036938044043154_wp
        Xi(26) = 33.006081233208052379268782061672_wp
        Xi(27) = 38.185652963770639254509342563892_wp
        Xi(28) = 43.643238869459938741174331866546_wp
        Xi(29) = 49.052443340562168410529109152662_wp
        Xi(30) = 53.929290807526017108297633484426_wp
        Xi(31) = 57.679914983451419423221562254867_wp
        Xi(32) = 59.734007012493412575038220779788_wp

      case (9)
        ! ----- K = 16 , R = 7E1 -----

        RIni = 7.0e1_wp

        Omega(1) = 0.0121119422284885791480239553181974532237_wp
        Omega(2) = 0.0290189532715633909560086142742640902270_wp
        Omega(3) = 0.0481093108632741291783420454347686145979_wp
        Omega(4) = 0.0711508422286992170256074777268118225493_wp
        Omega(5) = 0.1007542599873194719787834576496532434930_wp
        Omega(6) = 0.1407132096601951245975342663641072249447_wp
        Omega(7) = 0.1962985762182013438169455796233897615366_wp
        Omega(8) = 0.2745528922234677530263453892889202734295_wp
        Omega(9) = 0.3848395792100061791577055025914688712874_wp
        Omega(10) = 0.5399170453962834127802951356134286697852_wp
        Omega(11) = 0.7578406304494023196824219190670390844389_wp
        Omega(12) = 1.0655310964555628802218761874698316205468_wp
        Omega(13) = 1.5066799629278595433305720785455150689813_wp
        Omega(14) = 2.1632416780817367598071382372992843556858_wp
        Omega(15) = 3.2321339880593698366123556198203914391343_wp
        Omega(16) = 5.4820672068151955966253119356679235352203_wp

        Alpha(1) = 0.0047039031601346258206606154757267734823_wp
        Alpha(2) = 0.0251434613504408551858001997711400399282_wp
        Alpha(3) = 0.0634624580084015806394608788520983466697_wp
        Alpha(4) = 0.1226699481548256966310602757153080943908_wp
        Alpha(5) = 0.2079381850759855593054406530661992746900_wp
        Alpha(6) = 0.3276138429438437771952941784947554992868_wp
        Alpha(7) = 0.4945540693857812940433232168357946534343_wp
        Alpha(8) = 0.7277422312582489759670914541977992939792_wp
        Alpha(9) = 1.0543005913145042581890997723803593544289_wp
        Alpha(10) = 1.5122947031342729613812014832774366368540_wp
        Alpha(11) = 2.1549896726867369830225873439388806218631_wp
        Alpha(12) = 3.0577259786550075454369945759935944806784_wp
        Alpha(13) = 4.3301629301893997856486018260824266690179_wp
        Alpha(14) = 6.1418596932948431384342857786151625987259_wp
        Alpha(15) = 8.7900619662313506002052077015207487420412_wp
        Alpha(16) = 12.9677092977387981298509345151614979840815_wp

        Xi(1) = 1.003428650850012487471922539850_wp
        Xi(2) = 1.031007206247950144946921857692_wp
        Xi(3) = 1.086964401585403750308787829137_wp
        Xi(4) = 1.172919035756665565180777155430_wp
        Xi(5) = 1.291352523950442493017899681096_wp
        Xi(6) = 1.445686855421073847708171578041_wp
        Xi(7) = 1.640403888683373654668040675375_wp
        Xi(8) = 1.881208547679956308621394833747_wp
        Xi(9) = 2.175236252473233881888844964969_wp
        Xi(10) = 2.531305723494001962961194585056_wp
        Xi(11) = 2.960223167304340141474383596432_wp
        Xi(12) = 3.475147698324568867932551907707_wp
        Xi(13) = 4.092030634851775970119158731819_wp
        Xi(14) = 4.830140859978980539916892933405_wp
        Xi(15) = 5.712684090346128815401244294137_wp
        Xi(16) = 6.767518437426713942377254351612_wp
        Xi(17) = 8.027955257164587068649530543496_wp
        Xi(18) = 9.533606681635384107534325615063_wp
        Xi(19) = 11.331193698942411123876949918809_wp
        Xi(20) = 13.475135845050428147137711043513_wp
        Xi(21) = 16.027585820352931588947398289235_wp
        Xi(22) = 19.057298239731036980024181382021_wp
        Xi(23) = 22.636277701185040909029333455749_wp
        Xi(24) = 26.832476998127376190475534833979_wp
        Xi(25) = 31.695907140097002918296964502076_wp
        Xi(26) = 37.234675578505159788261780207108_wp
        Xi(27) = 43.377548401359464658177955698193_wp
        Xi(28) = 49.922998632993908999649912416885_wp
        Xi(29) = 56.485050482773292216537308618740_wp
        Xi(30) = 62.466232288618900061333283346698_wp
        Xi(31) = 67.109010911477774460887069096771_wp
        Xi(32) = 69.667807344614608493216234563761_wp

      case (10)
        ! ----- K = 16 , R = 8E1 -----

        RIni = 8.0e1_wp

        Omega(1) = 0.0108873068861372562665142246438043649448_wp
        Omega(2) = 0.0261282663223343776516766154977000624626_wp
        Omega(3) = 0.0434552850078543593980904234663764640345_wp
        Omega(4) = 0.0645878894987122491969946079526376792046_wp
        Omega(5) = 0.0920903496023600561648904741507415394608_wp
        Omega(6) = 0.1297042298984791642354075930088086465730_wp
        Omega(7) = 0.1826274887652090326948953538654940587094_wp
        Omega(8) = 0.2578168051849726095697996897104786739874_wp
        Omega(9) = 0.3645792437854544994267711605084869574966_wp
        Omega(10) = 0.5156912792016989570496182659908868117782_wp
        Omega(11) = 0.7293212975503779889590444396318957842595_wp
        Omega(12) = 1.0325907226271625882763061099467449821532_wp
        Omega(13) = 1.4694338205014248436176133028041590478097_wp
        Omega(14) = 2.1219626037538278065432834074144352598523_wp
        Omega(15) = 3.1867916413473558830040527345417444848863_wp
        Omega(16) = 5.4299136597940658287980708607278756971937_wp

        Alpha(1) = 0.0042274941327305500965867126814123899692_wp
        Alpha(2) = 0.0226156451548393151356538145063668210355_wp
        Alpha(3) = 0.0571724048914840694760443415983752757370_wp
        Alpha(4) = 0.1107855007815737983603598428017367183429_wp
        Alpha(5) = 0.1884582617757677853434408049571047172321_wp
        Alpha(6) = 0.2983204706033127306074020229642584922658_wp
        Alpha(7) = 0.4529507102131609971406960013506903806046_wp
        Alpha(8) = 0.6709720585108713596447782556708716583671_wp
        Alpha(9) = 0.9790711090780677519932947983072324404930_wp
        Alpha(10) = 1.4148557088957122064446295106598938673415_wp
        Alpha(11) = 2.0311982614953904605591128440522652454092_wp
        Alpha(12) = 2.9032052654708691739274956100658187096997_wp
        Alpha(13) = 4.1405299017812756882765679034008599046501_wp
        Alpha(14) = 5.9129417512000719144536764471098422291107_wp
        Alpha(15) = 8.5178777748218142407132802418345818296075_wp
        Alpha(16) = 12.6473263671162415074794571623328920395579_wp

        Xi(1) = 1.003568146725075832287506216733_wp
        Xi(2) = 1.032275421951231643576639418480_wp
        Xi(3) = 1.090558429595056287738329137138_wp
        Xi(4) = 1.180175123042378194666152113612_wp
        Xi(5) = 1.303822609129984488006426257556_wp
        Xi(6) = 1.465226302116329349235594736722_wp
        Xi(7) = 1.669276518959938414949847884650_wp
        Xi(8) = 1.922215602583746199344058580571_wp
        Xi(9) = 2.231876104194821583978983192686_wp
        Xi(10) = 2.607971760466308256722067770994_wp
        Xi(11) = 3.062449793423694131182591360307_wp
        Xi(12) = 3.609917387686896413740919586921_wp
        Xi(13) = 4.268159232181218664618077562878_wp
        Xi(14) = 5.058762368842223879333158675919_wp
        Xi(15) = 6.007861986970951318539485574988_wp
        Xi(16) = 7.147016996702025953477899200550_wp
        Xi(17) = 8.514211259033158024460419266433_wp
        Xi(18) = 10.154950044864726479108829604314_wp
        Xi(19) = 12.123369696286200031601354432098_wp
        Xi(20) = 14.483180337865166581283460356389_wp
        Xi(21) = 17.308084416454160245693594255556_wp
        Xi(22) = 20.680995355834386979776429171807_wp
        Xi(23) = 24.690858225061783201223497563603_wp
        Xi(24) = 29.425016953835080223134124111084_wp
        Xi(25) = 34.953889483568182535572210767327_wp
        Xi(26) = 41.303361521813087238463868544613_wp
        Xi(27) = 48.409874077820592781079556132795_wp
        Xi(28) = 56.056430700655069294252585976324_wp
        Xi(29) = 63.799997587513382624191393688307_wp
        Xi(30) = 70.927009405821734140673573421054_wp
        Xi(31) = 76.505504680732721367597548578487_wp
        Xi(32) = 79.597637690856747637313528542791_wp

      case (11)
        ! ----- K = 16 , R = 9E1 -----

        RIni = 9.0e1_wp

        Omega(1) = 0.0099070942053584087306917073216527835910_wp
        Omega(2) = 0.0238121984392840396955234177900351255630_wp
        Omega(3) = 0.0397197022388286855612516898506036966410_wp
        Omega(4) = 0.0593065074157296978656821531683540627000_wp
        Omega(5) = 0.0850941121309136351742469922543321558805_wp
        Omega(6) = 0.1207729487547143815568675626059835792603_wp
        Omega(7) = 0.1714692310620858691768775630692722700132_wp
        Omega(8) = 0.2440589639770963612403755979562980371611_wp
        Omega(9) = 0.3477989365091591634755827794345250936203_wp
        Omega(10) = 0.4954823766957529564093284935033523197490_wp
        Omega(11) = 0.7053775625652202247361718945306563455233_wp
        Omega(12) = 1.0047843223684194823453613709673959419888_wp
        Omega(13) = 1.4378576455423156901531483997658256157592_wp
        Omega(14) = 2.0868647293083933649988914726947086819564_wp
        Omega(15) = 3.1481875148259403213429202805784257179766_wp
        Omega(16) = 5.3855376972745262007663535896995199436788_wp

        Alpha(1) = 0.0038462156898872527546131037468404856305_wp
        Alpha(2) = 0.0205915676059204840877094912221512146289_wp
        Alpha(3) = 0.0521312367564103167914054282744418244988_wp
        Alpha(4) = 0.1012481384043264893461623772641377883019_wp
        Alpha(5) = 0.1727980588057935402480653347412342668576_wp
        Alpha(6) = 0.2747175043007444109066390175444283272554_wp
        Alpha(7) = 0.4193307805231102605074498118620240916243_wp
        Alpha(8) = 0.6249280273155943181786521611886797700208_wp
        Alpha(9) = 0.9177937594965698797329703428893310501735_wp
        Alpha(10) = 1.3351147388188597902019399965567458821170_wp
        Alpha(11) = 1.9294001189108118012168965882402460465528_wp
        Alpha(12) = 2.7755336610437487396881722778729795209074_wp
        Alpha(13) = 3.9831504408499844558950303552435912024521_wp
        Alpha(14) = 5.7222039389716857524063875484188201880897_wp
        Alpha(15) = 8.2903222456109351128777928963131671480369_wp
        Alpha(16) = 12.3787400479130052528137229117533024691511_wp

        Xi(1) = 1.003691976916175615866522186082_wp
        Xi(2) = 1.033401659367964465748371272191_wp
        Xi(3) = 1.093752631786901880362600225638_wp
        Xi(4) = 1.186631336597799261695258821181_wp
        Xi(5) = 1.314934451540161665524485135048_wp
        Xi(6) = 1.482668631894253555097117069916_wp
        Xi(7) = 1.695103987050068696678696567037_wp
        Xi(8) = 1.958984485408930278200166641422_wp
        Xi(9) = 2.282795600848628430170131342614_wp
        Xi(10) = 2.677092955809774674647699699293_wp
        Xi(11) = 3.154901093114184340918115800889_wp
        Xi(12) = 3.732200333833114767791844235134_wp
        Xi(13) = 4.428521835716363213783508534505_wp
        Xi(14) = 5.267671209050016853294579055600_wp
        Xi(15) = 6.278601228720268359086154053017_wp
        Xi(16) = 7.496448588479154036007950923093_wp
        Xi(17) = 8.963739143018941191859427775057_wp
        Xi(18) = 10.731740206947994127444079293099_wp
        Xi(19) = 12.861885824280574017186573598792_wp
        Xi(20) = 15.427096091462101036875798065751_wp
        Xi(21) = 18.512620100932913286967407273664_wp
        Xi(22) = 22.215675943845266181869702037943_wp
        Xi(23) = 26.642549964022028703009015160319_wp
        Xi(24) = 31.900809183562342526141453191713_wp
        Xi(25) = 38.082764344011690707536565980718_wp
        Xi(26) = 45.234470509062490239243770062672_wp
        Xi(27) = 53.303434122760849408345640298990_wp
        Xi(28) = 62.061106950823038484743587872572_wp
        Xi(29) = 71.010132921310621989530176279004_wp
        Xi(30) = 79.319141105841021607836260898239_wp
        Xi(31) = 85.872336017797033375087423223704_wp
        Xi(32) = 89.523836686630239280193510609251_wp

      case (12)
        ! ----- K = 16 , R = 1E2 -----

        RIni = 1.0e2_wp

        Omega(1) = 0.0091030846063941373947308146122314731485_wp
        Omega(2) = 0.0219107094627195211883731798541141699843_wp
        Omega(3) = 0.0366477217677682609360426211481120972735_wp
        Omega(4) = 0.0549529774439788038923107842820270363404_wp
        Omega(5) = 0.0793083736311463438007816027786045509629_wp
        Omega(6) = 0.1133547294055116984719212488985640163719_wp
        Omega(7) = 0.1621489723348042820026335131267103406572_wp
        Omega(8) = 0.2324921039907123551090722610168448625245_wp
        Omega(9) = 0.3335959219735489658447899485871346314525_wp
        Omega(10) = 0.4782687636968508076529098260731842628957_wp
        Omega(11) = 0.6848674256397950935236311476028703282282_wp
        Omega(12) = 0.9808514234763911221752670510642957424352_wp
        Omega(13) = 1.4105774826377881514018197695037315497757_wp
        Omega(14) = 2.0564635512053020538481007406517164781690_wp
        Omega(15) = 3.1147098104802878920822906128051954510738_wp
        Omega(16) = 5.3470752639906505886482901424017200042726_wp

        Alpha(1) = 0.0035335098097274502238478871302823636213_wp
        Alpha(2) = 0.0189307470173197830927468501838184522512_wp
        Alpha(3) = 0.0479913291058656280866289659675638645808_wp
        Alpha(4) = 0.0934063314619771164137255331505471644959_wp
        Alpha(5) = 0.1599009758991468320513942499849768807962_wp
        Alpha(6) = 0.2552377205651998341359980998488410364189_wp
        Alpha(7) = 0.3915075062326464324703213792933098602589_wp
        Alpha(8) = 0.5866925245564957031702522927130161178866_wp
        Alpha(9) = 0.8667062212594494830723672318395500724364_wp
        Alpha(10) = 1.2683475596560744691043004506170177592139_wp
        Alpha(11) = 1.8437892243513303981803513886816858757811_wp
        Alpha(12) = 2.6677035867305717634339473631754913185432_wp
        Alpha(13) = 3.8496992950460036169921251847370058385422_wp
        Alpha(14) = 5.5598919048823978901474607283716977690347_wp
        Alpha(15) = 8.0960961548375747339964525473021694779163_wp
        Alpha(16) = 12.1489357832949838493855065912896407098742_wp

        Xi(1) = 1.003803294380646768694728687699_wp
        Xi(2) = 1.034414459604964373409807798510_wp
        Xi(3) = 1.096627116167862222044157571865_wp
        Xi(4) = 1.192447330001823254546668151566_wp
        Xi(5) = 1.324957592558943276833324786956_wp
        Xi(6) = 1.498427125021005234173018805244_wp
        Xi(7) = 1.718481564209594093881744436558_wp
        Xi(8) = 1.992335948611406212685297212328_wp
        Xi(9) = 2.329090952993635296611532603528_wp
        Xi(10) = 2.740098264961065148729787988024_wp
        Xi(11) = 3.239404450340402310919152295554_wp
        Xi(12) = 3.844298179747094311095589880267_wp
        Xi(13) = 4.575980668529397538298042480420_wp
        Xi(14) = 5.460388213371944769389254581426_wp
        Xi(15) = 6.529191337570760817875931136811_wp
        Xi(16) = 7.820993488808169717772106332987_wp
        Xi(17) = 9.382742725529808837207912386447_wp
        Xi(18) = 11.271345248068873026031644357658_wp
        Xi(19) = 13.555416450399780177622777177504_wp
        Xi(20) = 16.316997190581468463524417167321_wp
        Xi(21) = 19.652855825131246635811832490504_wp
        Xi(22) = 23.674600448775685208879782095437_wp
        Xi(23) = 28.506167398102847601246145892162_wp
        Xi(24) = 34.276011964813860175699256060966_wp
        Xi(25) = 41.099581606579022291514391440614_wp
        Xi(26) = 49.045173894371357652122167536390_wp
        Xi(27) = 58.074399576032853211721151964753_wp
        Xi(28) = 67.950839319041810432375783079806_wp
        Xi(29) = 78.125663428646631089580232298886_wp
        Xi(30) = 87.648656297381219167086374710607_wp
        Xi(31) = 95.211876155386198909702510917441_wp
        Xi(32) = 99.446678144653096610705578228817_wp

      case (13)
        ! ----- K = 16 , R = 2E2 -----

        RIni = 2.0e2_wp

        Omega(1) = 0.0051910050391620038140489551154593828031_wp
        Omega(2) = 0.0126277612849116577841408043912652114926_wp
        Omega(3) = 0.0215606407131872894500943058787578721081_wp
        Omega(4) = 0.0333846640063611133550348052112521912704_wp
        Omega(5) = 0.0502953701646382063748634289740424563320_wp
        Omega(6) = 0.0755353808513282801251971683431651882756_wp
        Omega(7) = 0.1136336810394631888142757090987000623272_wp
        Omega(8) = 0.1708798850219694833184108304080517370949_wp
        Omega(9) = 0.2561954432329190103000918915521211260966_wp
        Omega(10) = 0.3824592769646401152450979760155647113606_wp
        Omega(11) = 0.5685478192743596877059900074335985209473_wp
        Omega(12) = 0.8429296732276469761584343498572735597918_wp
        Omega(13) = 1.2513442211943193828945636303373589726107_wp
        Omega(14) = 1.8774322383842335613957086337677537812851_wp
        Omega(15) = 2.9167470741005022236320975981271885757451_wp
        Omega(16) = 5.1200362418007544676283593343413258480723_wp

        Alpha(1) = 0.0020125660928861264092960586986891247108_wp
        Alpha(2) = 0.0108393025800470222817957488548046285359_wp
        Alpha(3) = 0.0277607372768341410946277888344146944632_wp
        Alpha(4) = 0.0549153548548539399454552697998899191134_wp
        Alpha(5) = 0.0962139067787890126429270649943692461648_wp
        Alpha(6) = 0.1582685859013296521361499960556251664912_wp
        Alpha(7) = 0.2515517622186365606119784477101575248525_wp
        Alpha(8) = 0.3918898940757439454808490342241356074737_wp
        Alpha(9) = 0.6026248579910799004323920802228542470402_wp
        Alpha(10) = 0.9178653621745612205739747735400158035191_wp
        Alpha(11) = 1.3873828979151186895080183347772617707960_wp
        Alpha(12) = 2.0842135216668545253260874661194179680024_wp
        Alpha(13) = 3.1175469236537975187186638725833631724527_wp
        Alpha(14) = 4.6584561428876707965887615792155429517152_wp
        Alpha(15) = 7.0062440250979758607670355718965993219172_wp
        Alpha(16) = 10.8487716247577870394211418947350011876551_wp

        Xi(1) = 1.004543963223910764100479942051_wp
        Xi(2) = 1.041162209364920160662923431794_wp
        Xi(3) = 1.115827713131373046559317863125_wp
        Xi(4) = 1.231441193971959971410409795389_wp
        Xi(5) = 1.392482124242125950618585716878_wp
        Xi(6) = 1.605208525760429899716628943107_wp
        Xi(7) = 1.877967940703037661690118220736_wp
        Xi(8) = 2.221625241230130186993393937023_wp
        Xi(9) = 2.650109627811283131634811915944_wp
        Xi(10) = 3.181098745057285561051163558233_wp
        Xi(11) = 3.836876209790531579027841879714_wp
        Xi(12) = 4.645422392240050664391493029370_wp
        Xi(13) = 5.641798797641391978452407585110_wp
        Xi(14) = 6.869906621767563695887121033579_wp
        Xi(15) = 8.384711159585907115816827417376_wp
        Xi(16) = 10.255043690692088445559626563863_wp
        Xi(17) = 12.567108023170744343297022815875_wp
        Xi(18) = 15.428825091440629301529907646540_wp
        Xi(19) = 18.975125907624490443834086583763_wp
        Xi(20) = 23.374196278342412529116511876737_wp
        Xi(21) = 28.834431487391838569780966317069_wp
        Xi(22) = 35.611266762769640249058911152247_wp
        Xi(23) = 44.011799056629079002905813311486_wp
        Xi(24) = 54.392539367342489734641342025157_wp
        Xi(25) = 67.140738711682966172178410602100_wp
        Xi(26) = 82.620841953775863229780807017733_wp
        Xi(27) = 101.054169044903487922892892925120_wp
        Xi(28) = 122.286579577677779903654986526362_wp
        Xi(29) = 145.409716504364367142820846368068_wp
        Xi(30) = 168.301902381474210376732791871746_wp
        Xi(31) = 187.425817576227922295339922698076_wp
        Xi(32) = 198.527536791589821002501636826310_wp

      case (14)
        ! ----- K = 16 , R = 3E2 -----

        RIni = 3.0e2_wp

        Omega(1) = 0.0037255564169910067277597943676339942165_wp
        Omega(2) = 0.0091302987056417856438241247712950254822_wp
        Omega(3) = 0.0158171136039656348763288119460557190621_wp
        Omega(4) = 0.0250466904073013040228428641859048653373_wp
        Omega(5) = 0.0388327073275191414696656724142576067038_wp
        Omega(6) = 0.0601510240904654430165875329816760341828_wp
        Omega(7) = 0.0932081506757678266748639271666032612984_wp
        Omega(8) = 0.1440089851094002000250499789801850170079_wp
        Omega(9) = 0.2213025039554186542865247405709183681211_wp
        Omega(10) = 0.3379614198919181375997851379811010019694_wp
        Omega(11) = 0.5130898999076860187416350422839883549386_wp
        Omega(12) = 0.7756846839561412798483112929215366193603_wp
        Omega(13) = 1.1723016787490888693809912290966224190925_wp
        Omega(14) = 1.7874363147483472139985180571208900346392_wp
        Omega(15) = 2.8166376845013903116275721938421838785871_wp
        Omega(16) = 5.0054814386377507826747512709175680356566_wp

        Alpha(1) = 0.0014432114855624366430164606540237359766_wp
        Alpha(2) = 0.0078015317798097004564905892088494399061_wp
        Alpha(3) = 0.0201254698329391623609277558668018137666_wp
        Alpha(4) = 0.0402748735892814804047793765529039689000_wp
        Alpha(5) = 0.0717270043310115766298511348619815208849_wp
        Alpha(6) = 0.1204378422197703271513485011323663087524_wp
        Alpha(7) = 0.1959225086659849674689676257699488814978_wp
        Alpha(8) = 0.3127387451926994034500425362343989377223_wp
        Alpha(9) = 0.4927264320226538121989812157863752872800_wp
        Alpha(10) = 0.7683981194605783673250501419893510046677_wp
        Alpha(11) = 1.1880336160297062512341062512355449598545_wp
        Alpha(12) = 1.8235512816513078762680269417906231410598_wp
        Alpha(13) = 2.7837030596469976369451704290014504294959_wp
        Alpha(14) = 4.2399974599818059570709161132384679149254_wp
        Alpha(15) = 6.4927254010870050415435428270427564712008_wp
        Alpha(16) = 10.2289376129522341478395031089121403056197_wp

        Xi(1) = 1.004979210341979526917646770023_wp
        Xi(2) = 1.045134743033515391371843228363_wp
        Xi(3) = 1.127172042944499245854458879812_wp
        Xi(4) = 1.254599420629432444105645638732_wp
        Xi(5) = 1.432850797631794785135656011832_wp
        Xi(6) = 1.669557895055314880638011598712_wp
        Xi(7) = 1.974975462656173537455243927052_wp
        Xi(8) = 2.362565861687259726907003876128_wp
        Xi(9) = 2.849750598893955735825198050115_wp
        Xi(10) = 3.458858686311279327825485285786_wp
        Xi(11) = 4.218338541652051693585379066320_wp
        Xi(12) = 5.164320825089354523133866603501_wp
        Xi(13) = 6.342640395703774114453260546398_wp
        Xi(14) = 7.811444110011478200034096275317_wp
        Xi(15) = 9.644553699667076237037055541634_wp
        Xi(16) = 11.935793682202448600347677309941_wp
        Xi(17) = 14.804545904806895795929877479580_wp
        Xi(18) = 18.402846576447768963888385052030_wp
        Xi(19) = 22.924367560072551247563055198952_wp
        Xi(20) = 28.615602890090455294111460027295_wp
        Xi(21) = 35.789364298836023884492485436226_wp
        Xi(22) = 44.840078460752617432738542646575_wp
        Xi(23) = 56.258867284198758970670262158364_wp
        Xi(24) = 70.642906086139508434196265795890_wp
        Xi(25) = 88.685979443723195511650736477804_wp
        Xi(26) = 111.122037340557960498343970101587_wp
        Xi(27) = 138.565411870331778212972473340869_wp
        Xi(28) = 171.151311059692604610305011192395_wp
        Xi(29) = 207.857682225517994667463028690690_wp
        Xi(30) = 245.514446222413214526714142493802_wp
        Xi(31) = 278.044557713935721832143599385745_wp
        Xi(32) = 297.404850809861197785677333627063_wp

      case (15)
        ! ----- K = 16 , R = 4E2 -----

        RIni = 4.0e2_wp

        Omega(1) = 0.0029411704155330162737393910176580957838_wp
        Omega(2) = 0.0072506379760886996575654608095801156864_wp
        Omega(3) = 0.0127075060808783997139066778217508790760_wp
        Omega(4) = 0.0204817457415186981432056153099030559872_wp
        Omega(5) = 0.0324552336489367649763478854835208053942_wp
        Omega(6) = 0.0514120457313516375765709114087353981404_wp
        Omega(7) = 0.0813399962135207765340184568603199011250_wp
        Omega(8) = 0.1280531259067858902434824605598429059228_wp
        Omega(9) = 0.2001715994871604321903024725770592340268_wp
        Omega(10) = 0.3105368092541307178716582931832235203728_wp
        Omega(11) = 0.4783756936706182759703152862496011721305_wp
        Omega(12) = 0.7330239927551988701227847489150235560373_wp
        Omega(13) = 1.1216059739542480910455199416553284663678_wp
        Omega(14) = 1.7292644544768973737056821926572069969552_wp
        Omega(15) = 2.7516812909019973754434773560717530926922_wp
        Omega(16) = 4.9312407184718789086505519314584944368107_wp

        Alpha(1) = 0.0011386076066376620912603070696206275514_wp
        Alpha(2) = 0.0061729978126926382157131298344943104439_wp
        Alpha(3) = 0.0160169098352160544780869826098079755639_wp
        Alpha(4) = 0.0323524323481128600355502880625557793337_wp
        Alpha(5) = 0.0583702583176669546740700841164217749224_wp
        Alpha(6) = 0.0995777894307948640905759143948650091716_wp
        Alpha(7) = 0.1648315575629028856323660470750702700116_wp
        Alpha(8) = 0.2678221908723548319683927659839994817048_wp
        Alpha(9) = 0.4293602681830135254224031088332580452516_wp
        Alpha(10) = 0.6808451714576150000748401558414713008460_wp
        Alpha(11) = 1.0694838116053031123542718661134642843535_wp
        Alpha(12) = 1.6663482146929018885318785070381863988587_wp
        Alpha(13) = 2.5798010052893875195326178761945357109653_wp
        Alpha(14) = 3.9815934666964298322029691679091456535389_wp
        Alpha(15) = 6.1727341131020466791570755482609911268810_wp
        Alpha(16) = 9.8399722788013861806549753552531001332682_wp

        Xi(1) = 1.005286566415155632557738851762_wp
        Xi(2) = 1.047943318089983249109115037712_wp
        Xi(3) = 1.135210829091119112297866844408_wp
        Xi(4) = 1.271063636152130822644032437374_wp
        Xi(5) = 1.461671315433593978721713013691_wp
        Xi(6) = 1.715731492895732926699976783613_wp
        Xi(7) = 2.044992742814055935356501447764_wp
        Xi(8) = 2.464972706047913258813053527518_wp
        Xi(9) = 2.995883140373143346769577166633_wp
        Xi(10) = 3.663810016421250367836190409143_wp
        Xi(11) = 4.502235087972342951999349391912_wp
        Xi(12) = 5.554025676793504075721974455604_wp
        Xi(13) = 6.874030818543077030104843094804_wp
        Xi(14) = 8.532470992427824087525611851390_wp
        Xi(15) = 10.619360983407052111127111260203_wp
        Xi(16) = 13.250281458211545110477402875659_wp
        Xi(17) = 16.573901537800292628824228025053_wp
        Xi(18) = 20.781764408042251687444568197805_wp
        Xi(19) = 26.120951319989963723688930485878_wp
        Xi(20) = 32.910312071354180987053927864849_wp
        Xi(21) = 41.560819307413242679194453899072_wp
        Xi(22) = 52.600107250088324689629715180672_wp
        Xi(23) = 66.699641276715551117815206083606_wp
        Xi(24) = 84.698923618478650476859570517263_wp
        Xi(25) = 107.611537314413924532052746485533_wp
        Xi(26) = 136.576865916434009806668115061257_wp
        Xi(27) = 172.678517322811636527934986418131_wp
        Xi(28) = 216.477712329128258686439245650490_wp
        Xi(29) = 267.033458290465157913828875280160_wp
        Xi(30) = 320.286188490862217620458451960985_wp
        Xi(31) = 367.483131109660798974125839322369_wp
        Xi(32) = 396.127659252490586572204023241284_wp

      case (16)
        ! ----- K = 16 , R = 5E2 -----

        RIni = 5.0e2_wp

        Omega(1) = 0.0024472131936283410200325512872873701120_wp
        Omega(2) = 0.0060630961621274316105539587545836255345_wp
        Omega(3) = 0.0107312550216339942659886158224380459103_wp
        Omega(4) = 0.0175540514820946540683208264913006679819_wp
        Omega(5) = 0.0283110517763022486080823217693969162667_wp
        Omega(6) = 0.0456402961035302483781047472310987700439_wp
        Omega(7) = 0.0733688359935863446185900713092387093184_wp
        Omega(8) = 0.1171684766792688380638524231480701587316_wp
        Omega(9) = 0.1855546600295324656461782436567276022288_wp
        Omega(10) = 0.2913293199019893982065418369664655529050_wp
        Omega(11) = 0.4537929108325465425504644662035858004856_wp
        Omega(12) = 0.7025225505971376122721881729571435926118_wp
        Omega(13) = 1.0850730482430842483310834412080225774844_wp
        Omega(14) = 1.6871056785919717904784903561576925312693_wp
        Omega(15) = 2.7044720053839088518346511769152584747644_wp
        Omega(16) = 4.8773253290853238362152766161017325430294_wp

        Alpha(1) = 0.0009468599039160355319218457159548968383_wp
        Alpha(2) = 0.0051461608873027619992851695342184203952_wp
        Alpha(3) = 0.0134185711537655366997178743235230591324_wp
        Alpha(4) = 0.0273192349904864276411927976467852730025_wp
        Alpha(5) = 0.0498287460342099060398328089216413161466_wp
        Alpha(6) = 0.0861193605778245913263567110074170329881_wp
        Alpha(7) = 0.1445556956959863234243347795970890956596_wp
        Alpha(8) = 0.2381830158803071012242570311800982096884_wp
        Alpha(9) = 0.3870414798533882194718838687386153196712_wp
        Alpha(10) = 0.6216847902584319523577698063387941829205_wp
        Alpha(11) = 0.9884865630628114253224993090984895616202_wp
        Alpha(12) = 1.5578388987534183036812521216951665792294_wp
        Alpha(13) = 2.4377619194799008527779760058251667942386_wp
        Alpha(14) = 3.8001562576159002308697626437705707758141_wp
        Alpha(15) = 5.9465865778449154555040934155840659514070_wp
        Alpha(16) = 9.5636980300421801154051038018621966330102_wp

        Xi(1) = 1.005523302139058109761285408901_wp
        Xi(2) = 1.050108461063545093423440834890_wp
        Xi(3) = 1.141418414873557743926223151387_wp
        Xi(4) = 1.283808123736142587463168740491_wp
        Xi(5) = 1.484048964244604059629223380412_wp
        Xi(6) = 1.751715896121533585778722030835_wp
        Xi(7) = 2.099794962973845620325327732125_wp
        Xi(8) = 2.545518284900301884694090448491_wp
        Xi(9) = 3.111443809952679282479839906195_wp
        Xi(10) = 3.826838263523562360529650661078_wp
        Xi(11) = 4.729483845715898698500168073622_wp
        Xi(12) = 5.868051273980675961849245991786_wp
        Xi(13) = 7.305226942313801762572678111241_wp
        Xi(14) = 9.121828661475524921073421147355_wp
        Xi(15) = 11.422223160509220610259184525148_wp
        Xi(16) = 14.341464492801081471659574440025_wp
        Xi(17) = 18.054709049381179824730891958495_wp
        Xi(18) = 22.789617009219566576630477960919_wp
        Xi(19) = 28.842636541390743329382750914647_wp
        Xi(20) = 36.600283997038238807675281094589_wp
        Xi(21) = 46.566491235160331074033335241324_wp
        Xi(22) = 59.396774874873065046959608181965_wp
        Xi(23) = 75.938463956485049821676369674606_wp
        Xi(24) = 97.271796448065084995937557010848_wp
        Xi(25) = 124.735614845963427271824386366461_wp
        Xi(26) = 159.895284599243062084727995397770_wp
        Xi(27) = 204.353454151061048646442763754294_wp
        Xi(28) = 259.196002266412646453108692412570_wp
        Xi(29) = 323.720422719421398571881809402839_wp
        Xi(30) = 393.136876103394547282698923140742_wp
        Xi(31) = 455.962930007404209614696455332705_wp
        Xi(32) = 494.722418110257492418480040896611_wp

      case (17)
        ! ----- K = 16 , R = 6E2 -----

        RIni = 6.0e2_wp

        Omega(1) = 0.0021053197367968382995978809512499507761_wp
        Omega(2) = 0.0052388810322415410763938214706164631096_wp
        Omega(3) = 0.0093527407563119556425679798974948120360_wp
        Omega(4) = 0.0154958369642747481409480558682545403570_wp
        Omega(5) = 0.0253648437706326759556062959768962805640_wp
        Omega(6) = 0.0414818886569631436331208503009992938360_wp
        Omega(7) = 0.0675490862157180216787254363464132467243_wp
        Omega(8) = 0.1091252814085100464563775346515939190795_wp
        Omega(9) = 0.1746371114059133875472378666371398026058_wp
        Omega(10) = 0.2768450406296747271536577705841608576520_wp
        Omega(11) = 0.4350961478436266127289150373869119903247_wp
        Omega(12) = 0.6791506110084174656967709315935621816607_wp
        Omega(13) = 1.0569067323081625278495543351908736440237_wp
        Omega(14) = 1.6544567764087873979741635799278753893304_wp
        Omega(15) = 2.6678292216763005312036433069877716661722_wp
        Omega(16) = 4.8355004456198803571183308935843569997814_wp

        Alpha(1) = 0.0008141838379395475550614482643935199135_wp
        Alpha(2) = 0.0044346795523015622871485191640317236050_wp
        Alpha(3) = 0.0116136389856696046680349871174642295557_wp
        Alpha(4) = 0.0238092630968598776858307806622572222466_wp
        Alpha(5) = 0.0438383776182723896082556441083921683344_wp
        Alpha(6) = 0.0766089872341315043370959560198318882840_wp
        Alpha(7) = 0.1300986481276461124610092229270108532546_wp
        Alpha(8) = 0.2168455261816722999849375158598618895667_wp
        Alpha(9) = 0.3562805725175744084559214391871861948857_wp
        Alpha(10) = 0.5782804460787271140227543131384635444192_wp
        Alpha(11) = 0.9285403059074989504475573787178888096605_wp
        Alpha(12) = 1.4768849008164217853503533128822766684607_wp
        Alpha(13) = 2.3310307802189951843514076346153274243989_wp
        Alpha(14) = 3.6629754895839110103317876054873636348930_wp
        Alpha(15) = 5.7747339253388481326335901488988611163222_wp
        Alpha(16) = 9.3529386861084840504645954162299403833458_wp

        Xi(1) = 1.005715257024971536078082612331_wp
        Xi(2) = 1.051865261327573555262537841060_wp
        Xi(3) = 1.146461986814843870500363331999_wp
        Xi(4) = 1.294182500205798182950703290039_wp
        Xi(5) = 1.502309193265005543408731336630_wp
        Xi(6) = 1.781165246115183020637548483300_wp
        Xi(7) = 2.144797001799990852910493543959_wp
        Xi(8) = 2.611915402446914854497558855506_wp
        Xi(9) = 3.207111396122765457862222038266_wp
        Xi(10) = 3.962428457963935397650664937963_wp
        Xi(11) = 4.919423697530920916876767678971_wp
        Xi(12) = 6.131899175222163090887700365172_wp
        Xi(13) = 7.669520950434395173585244709358_wp
        Xi(14) = 9.622613902487533195546920428143_wp
        Xi(15) = 12.108520278443703821480426086055_wp
        Xi(16) = 15.280043428652557609137563332524_wp
        Xi(17) = 19.336671649158674922072687074603_wp
        Xi(18) = 24.539526368547471881290111817009_wp
        Xi(19) = 31.231211134348567231081017503413_wp
        Xi(20) = 39.862071504431753059261023253157_wp
        Xi(21) = 51.024602316323522666063894348554_wp
        Xi(22) = 65.497477446429079608880563512230_wp
        Xi(23) = 84.299401550706269052593011537056_wp
        Xi(24) = 108.748525176769570574353274139412_wp
        Xi(25) = 140.510815862846447976686192760098_wp
        Xi(26) = 181.590258826201451869408742822998_wp
        Xi(27) = 234.143479946847548694988105921766_wp
        Xi(28) = 299.856260581767213896364765446378_wp
        Xi(29) = 378.395246141377767284952327031533_wp
        Xi(30) = 464.391670671157115651839220049624_wp
        Xi(31) = 543.625273039097820670928484787510_wp
        Xi(32) = 593.206156763596342440525432948562_wp

      case (18)
        ! ----- K = 16 , R = 7E2 -----

        RIni = 7.0e2_wp

        Omega(1) = 0.0018535144254540457011458598956740728703_wp
        Omega(2) = 0.0046303911519224061630527157149006534098_wp
        Omega(3) = 0.0083305478664809014863852024840307208819_wp
        Omega(4) = 0.0139590364258402108302233078987453218645_wp
        Omega(5) = 0.0231433556018734578973674491722545276673_wp
        Omega(6) = 0.0383107456954635583907262455505282616741_wp
        Omega(7) = 0.0630622860656869282578156771235100563899_wp
        Omega(8) = 0.1028633169367928892955037223355674314007_wp
        Omega(9) = 0.1660632135759762743307298496486801298033_wp
        Omega(10) = 0.2653813785938354123422784625052628371122_wp
        Omega(11) = 0.4201953469238532187909809095804902767668_wp
        Omega(12) = 0.6604101630611989827640898043092931857245_wp
        Omega(13) = 1.0342079766011074785361442684994415230904_wp
        Omega(14) = 1.6280489633923891337925760058880086944555_wp
        Omega(15) = 2.6381353298352225220622935264458419624134_wp
        Omega(16) = 4.8016213243881142968931541314958622024278_wp

        Alpha(1) = 0.0007164949012989691333421935718478446020_wp
        Alpha(2) = 0.0039101863821661210702473291268149724331_wp
        Alpha(3) = 0.0102801072464593958679624180301004354732_wp
        Alpha(4) = 0.0212070464670612541331542327815995463425_wp
        Alpha(5) = 0.0393748784721239822396434901441431009061_wp
        Alpha(6) = 0.0694755195241083988366356805066459401132_wp
        Alpha(7) = 0.1191707702350980214441944840531029115027_wp
        Alpha(8) = 0.2005851272064654881112409981480659837416_wp
        Alpha(9) = 0.3326494625651600446408686501120754996919_wp
        Alpha(10) = 0.5446788598533015113940455864671719155012_wp
        Alpha(11) = 0.8817981251009138986794722026019144323072_wp
        Alpha(12) = 1.4133459060584151906767372275375294066180_wp
        Alpha(13) = 2.2467669668753898573089478052366985139088_wp
        Alpha(14) = 3.5541234904565744520850423260327488605981_wp
        Alpha(15) = 5.6378062888905456332029608734757175625418_wp
        Alpha(16) = 9.1844824112248478770875736998391403176356_wp

        Xi(1) = 1.005876309005510832483322458675_wp
        Xi(2) = 1.053340084448622940402513770142_wp
        Xi(3) = 1.150700637067471238149732326583_wp
        Xi(4) = 1.302914865532209731527682472496_wp
        Xi(5) = 1.517710126543378411898867208496_wp
        Xi(6) = 1.806062687551509105790721598783_wp
        Xi(7) = 2.182950165131476624822554066796_wp
        Xi(8) = 2.668385994970611529292225938548_wp
        Xi(9) = 3.288763099913594485949538959169_wp
        Xi(10) = 4.078594878129303225207380467765_wp
        Xi(11) = 5.082817271925541693228245421210_wp
        Xi(12) = 6.359850953453854601350042630159_wp
        Xi(13) = 7.985681648838154654285875766462_wp
        Xi(14) = 10.059296353077444365585468855073_wp
        Xi(15) = 12.709929661761720182631962838826_wp
        Xi(16) = 16.106761007607586591930393105088_wp
        Xi(17) = 20.471892668824389804188568220411_wp
        Xi(18) = 26.097744473175567116521311739064_wp
        Xi(19) = 33.370410916040415448546863075308_wp
        Xi(20) = 42.800865993629198257702794805368_wp
        Xi(21) = 55.066376719235140806824047032819_wp
        Xi(22) = 71.064550369022659426221366629761_wp
        Xi(23) = 91.981228843948959053578029454457_wp
        Xi(24) = 119.369175386683191796011183072324_wp
        Xi(25) = 155.221425507736776452927252023528_wp
        Xi(26) = 201.988642197331957670036750585041_wp
        Xi(27) = 262.407106944497138439897909734100_wp
        Xi(28) = 338.822143677454432147966301158704_wp
        Xi(29) = 431.380209614188204869655862694344_wp
        Xi(30) = 534.275379812551198699388521617948_wp
        Xi(31) = 630.569530725390331893365214455116_wp
        Xi(32) = 691.590930201107461305909396287461_wp

      case (19)
        ! ----- K = 16 , R = 8E2 -----

        RIni = 8.0e2_wp

        Omega(1) = 0.0016597110979603259690510122363084344332_wp
        Omega(2) = 0.0041610632624492284243526794146328384727_wp
        Omega(3) = 0.0075390201458541255498857405065171732872_wp
        Omega(4) = 0.0127615706769678872249652000492314840585_wp
        Omega(5) = 0.0213972122588712194549739453036663050511_wp
        Omega(6) = 0.0357936419467514498914443264732110350224_wp
        Omega(7) = 0.0594677802105419486269292234092986859650_wp
        Omega(8) = 0.0978053035397581935553312781905788142467_wp
        Omega(9) = 0.1590871684802288474303803150755598494470_wp
        Omega(10) = 0.2559931002217505570557609662074582956848_wp
        Omega(11) = 0.4079207043380653206527015897098209507021_wp
        Omega(12) = 0.6448932537552953053673794603373181644201_wp
        Omega(13) = 1.0153335202525769788688830841394405979372_wp
        Omega(14) = 1.6060220091447995983740745784551506858406_wp
        Omega(15) = 2.6133276848398222708992622020929275095114_wp
        Omega(16) = 4.7733264900750258710043771870346063224133_wp

        Alpha(1) = 0.0006413268094817407344158084552508680698_wp
        Alpha(2) = 0.0035061734626053672457980382422618181870_wp
        Alpha(3) = 0.0092508507448277612239634858613823720219_wp
        Alpha(4) = 0.0191923279358940297880222261572846775834_wp
        Alpha(5) = 0.0359033684018405671501577344079114695319_wp
        Alpha(6) = 0.0638944507354568748034272240854392066467_wp
        Alpha(7) = 0.1105628524589580313801850594557230067494_wp
        Alpha(8) = 0.1876862369295839864030217905566111369353_wp
        Alpha(9) = 0.3137737097128846019478739387542631789074_wp
        Alpha(10) = 0.5176624942979187261925602681067459798214_wp
        Alpha(11) = 0.8439866686004004246268554723098276326709_wp
        Alpha(12) = 1.3616601833297931429742066988808346650330_wp
        Alpha(13) = 2.1778821657413933766695574023586345902004_wp
        Alpha(14) = 3.4647582889019940316412821124103516012838_wp
        Alpha(15) = 5.5250005152878595270776740377982605423313_wp
        Alpha(16) = 9.0453357535582769796689284191870683571324_wp

        Xi(1) = 1.006014770006011066765146122481_wp
        Xi(2) = 1.054608649958689026268805755127_wp
        Xi(3) = 1.154349912463906832247283185122_wp
        Xi(4) = 1.310443042542520201824453318640_wp
        Xi(5) = 1.531009803243571506885765953765_wp
        Xi(6) = 1.827607135648407961814494726571_wp
        Xi(7) = 2.216043517152185178759396522175_wp
        Xi(8) = 2.717499782918300837942712777640_wp
        Xi(9) = 3.359988834639446199283735472463_wp
        Xi(10) = 4.180256279270505156194642770728_wp
        Xi(11) = 5.226303654756044108319057439616_wp
        Xi(12) = 6.560763652753605687864557616251_wp
        Xi(13) = 8.265412865630715556590490944444_wp
        Xi(14) = 10.447215904516594989909217261470_wp
        Xi(15) = 13.246427732457084346096021221229_wp
        Xi(16) = 16.847480533181990430824370719165_wp
        Xi(17) = 21.493649889102426278966739836562_wp
        Xi(18) = 27.506867419054068542652591311537_wp
        Xi(19) = 35.314460273588206010164647352667_wp
        Xi(20) = 45.485262313086833711045509076598_wp
        Xi(21) = 58.778035298133601598696840184743_wp
        Xi(22) = 76.205516395195813147966923395416_wp
        Xi(23) = 99.116695413420177446228720441468_wp
        Xi(24) = 129.295537243414167843935658197552_wp
        Xi(25) = 169.060988432439411974228526958086_wp
        Xi(26) = 221.315575358555647347835737548394_wp
        Xi(27) = 289.394717428705165518376674071988_wp
        Xi(28) = 376.351687843882334871814521193301_wp
        Xi(29) = 482.908399988066930896835060593730_wp
        Xi(30) = 602.953743961086938318860717345160_wp
        Xi(31) = 716.870063847640778764702673697684_wp
        Xi(32) = 789.885808266521592280184904666385_wp

      case (20)
        ! ----- K = 16 , R = 9E2 -----

        RIni = 9.0e2_wp

        Omega(1) = 0.0015055631595250562666732919406449575650_wp
        Omega(2) = 0.0037870450506321845648883704562454033393_wp
        Omega(3) = 0.0069059686538486601540820237609547005420_wp
        Omega(4) = 0.0117983671303826509692803882652090141647_wp
        Omega(5) = 0.0199815601654048778368355770290731854288_wp
        Omega(6) = 0.0337352970754505528657785692198123328467_wp
        Omega(7) = 0.0565047866336671157529018046788982232442_wp
        Omega(8) = 0.0936063868568455301861445244648862740178_wp
        Omega(9) = 0.1532596650407880263462844297461806775118_wp
        Omega(10) = 0.2481064208093561478975230816246266840608_wp
        Omega(11) = 0.3975573296197845333881885215854268267321_wp
        Omega(12) = 0.6317344227996357830978886582773412783354_wp
        Omega(13) = 0.9992684300875393019073303357835413862631_wp
        Omega(14) = 1.5872230602253215233114411186754466598359_wp
        Omega(15) = 2.5921258825123527603454670620308775141893_wp
        Omega(16) = 4.7491508522890109713146533643168822891312_wp

        Alpha(1) = 0.0005815528588767195223618469517284229608_wp
        Alpha(2) = 0.0031845872823828756917046448266309099928_wp
        Alpha(3) = 0.0084300984840787595718698821389835984519_wp
        Alpha(4) = 0.0175811659326021210709409302655470419552_wp
        Alpha(5) = 0.0331156506410349317250628193437522739373_wp
        Alpha(6) = 0.0593885926005389108910754916020868776627_wp
        Alpha(7) = 0.1035710999862279087105690345005371000298_wp
        Alpha(8) = 0.1771438940187624976182675928804144405149_wp
        Alpha(9) = 0.2982529920600857122872564403115447362325_wp
        Alpha(10) = 0.4953210601193777047124896378837988208943_wp
        Alpha(11) = 0.8125524576330856438071900682551529371267_wp
        Alpha(12) = 1.3184844662501273938664744789228677745996_wp
        Alpha(13) = 2.1200923357897898953016574408181327271450_wp
        Alpha(14) = 3.3895107039352260140489814244091348882648_wp
        Alpha(15) = 5.4297307903033270120272957282026027314714_wp
        Alpha(16) = 8.9275533544404011261977349711571605439531_wp

        Xi(1) = 1.006136012409362691789709021162_wp
        Xi(2) = 1.055719931445410426961231675769_wp
        Xi(3) = 1.157549330324401512287875448681_wp
        Xi(4) = 1.317050855184951755109078186390_wp
        Xi(5) = 1.542700605008748187488981651327_wp
        Xi(6) = 1.846578984154591802132466815678_wp
        Xi(7) = 2.245245130934919511583175433245_wp
        Xi(8) = 2.760939294470852008438893632203_wp
        Xi(9) = 3.423148362551289448007726945455_wp
        Xi(10) = 4.270657235986691423797884104019_wp
        Xi(11) = 5.354279482436550032398853149473_wp
        Xi(12) = 6.740525940807517171326351812510_wp
        Xi(13) = 8.516531260420975283156885815572_wp
        Xi(14) = 10.796675502208701058828865804173_wp
        Xi(15) = 13.731498439456119056018001955977_wp
        Xi(16) = 17.519740105992699359324649144298_wp
        Xi(17) = 22.424651605042061531039565558387_wp
        Xi(18) = 28.796117372762449860418287883590_wp
        Xi(19) = 37.100772684586712478471692833182_wp
        Xi(20) = 47.962900980259937141342296840207_wp
        Xi(21) = 62.219816526658292429180496796448_wp
        Xi(22) = 80.996014694373807858707170481694_wp
        Xi(23) = 105.799881084980490091040916666998_wp
        Xi(24) = 138.643125702413568964277423845033_wp
        Xi(25) = 182.168901573139119215438697096943_wp
        Xi(26) = 239.734799690210702946546739156020_wp
        Xi(27) = 315.290477426446540731852508088195_wp
        Xi(28) = 412.637447746722082458914471203570_wp
        Xi(29) = 533.156399397003743667600161870723_wp
        Xi(30) = 670.554597887061645367090534364252_wp
        Xi(31) = 802.584897091940211777405522752815_wp
        Xi(32) = 888.097908151572883517221868032721_wp

      case (21)
        ! ----- K = 16 , R = 1E3 -----

        RIni = 1.0e3_wp

        Omega(1) = 0.0013797888502005507772685364165782262447_wp
        Omega(2) = 0.0034813296515988117510506102492487401356_wp
        Omega(3) = 0.0063868138275758572432797531299303317454_wp
        Omega(4) = 0.0110042750893626431830035284145652418886_wp
        Omega(5) = 0.0188060416343505131900348294193836551358_wp
        Omega(6) = 0.0320129266794980219734613902798914253367_wp
        Omega(7) = 0.0540079284084068265886003669395964976729_wp
        Omega(8) = 0.0900461102030354742912533760890170242419_wp
        Omega(9) = 0.1482913872171991381761256903470957979607_wp
        Omega(10) = 0.2413494780456323231487792899119249057094_wp
        Omega(11) = 0.3886392366518692786925542187459470255817_wp
        Omega(12) = 0.6203667158670121572648301078167776267946_wp
        Omega(13) = 0.9853451315581527809373853232788320610780_wp
        Omega(14) = 1.5708916426203869327361817287425083122798_wp
        Omega(15) = 2.5736841409626707883207680227499736247410_wp
        Omega(16) = 4.7281271127462638696574426244723099443945_wp

        Alpha(1) = 0.0005327914492434099262305651010639823140_wp
        Alpha(2) = 0.0029220140733091449616117170340334752154_wp
        Alpha(3) = 0.0077588419683956976036238583228216159071_wp
        Alpha(4) = 0.0162599925842890798076650731018855111643_wp
        Alpha(5) = 0.0308208827514734286988985810865071535147_wp
        Alpha(6) = 0.0556612598231712853493206773908275053486_wp
        Alpha(7) = 0.0977557047572212667619902075333282454039_wp
        Alpha(8) = 0.1683264855900836211966649022531150592386_wp
        Alpha(9) = 0.2852019800763537923861574985284761396542_wp
        Alpha(10) = 0.4764397276834995646332005653222907426425_wp
        Alpha(11) = 0.7858625488052698108542432064993477069947_wp
        Alpha(12) = 1.2816696415764265190410930994602267674054_wp
        Alpha(13) = 2.0706307332283275014512408107592023043253_wp
        Alpha(14) = 3.3248991112367181318450887195226073345111_wp
        Alpha(15) = 5.3477122672426787952475180087930084482650_wp
        Alpha(16) = 8.8259523750929880087090495521806587930769_wp

        Xi(1) = 1.006243706349369666569124048827_wp
        Xi(2) = 1.056707400670955050996265012220_wp
        Xi(3) = 1.160394330619598642897680185193_wp
        Xi(4) = 1.322932700258313029866713561589_wp
        Xi(5) = 1.553120395588735876829522553066_wp
        Xi(6) = 1.863514697251501866875032786819_wp
        Xi(7) = 2.271360233615394187169089779132_wp
        Xi(8) = 2.799867055723205390514035650895_wp
        Xi(9) = 3.479877277249979421582318228623_wp
        Xi(10) = 4.352054420590895917123064329246_wp
        Xi(11) = 5.469813169292706701418022019467_wp
        Xi(12) = 6.903265503925697304816316846043_wp
        Xi(13) = 8.744539676385018759155465239985_wp
        Xi(14) = 11.114951975751633667273410299714_wp
        Xi(15) = 14.174706306404383061338381200045_wp
        Xi(16) = 18.136049297776665786829930482327_wp
        Xi(17) = 23.281151209986575804575248405115_wp
        Xi(18) = 29.986529513007290685677741493009_wp
        Xi(19) = 38.756415836380956937284869567861_wp
        Xi(20) = 50.268415213302309007409318653004_wp
        Xi(21) = 65.435796873362536800466937592091_wp
        Xi(22) = 85.491662295261926479694736258352_wp
        Xi(23) = 112.100398226085366817916799675459_wp
        Xi(24) = 147.498040056546388928970792164819_wp
        Xi(25) = 194.649827778459253449505617084014_wp
        Xi(26) = 257.370255453781749288566516042920_wp
        Xi(27) = 340.235188693453923319820475512643_wp
        Xi(28) = 447.828326319095989110907751751256_wp
        Xi(29) = 582.262922250848045713222944641529_wp
        Xi(30) = 737.179619506007345774989403253130_wp
        Xi(31) = 887.760886989793165269269081818493_wp
        Xi(32) = 986.233000349415966367949692994443_wp

      case (22)
        ! ----- K = 16 , R = 2E3 -----

        RIni = 2.0e3_wp

        Omega(1) = 0.0007768847420693003804794456927489132436_wp
        Omega(2) = 0.0020065660238064882712827270748803809575_wp
        Omega(3) = 0.0038523102169686026524150345307778708914_wp
        Omega(4) = 0.0070518785440354660497554411917320305037_wp
        Omega(5) = 0.0128051535867035469926666934638304073957_wp
        Omega(6) = 0.0229904356331946649850896287337054246791_wp
        Omega(7) = 0.0406223196249400557508322986283166855515_wp
        Omega(8) = 0.0705684431621601551419043808766051739667_wp
        Omega(9) = 0.1206153238244931053891120797949554344086_wp
        Omega(10) = 0.2030907673714956750333658283547144662862_wp
        Omega(11) = 0.3373950010841398567469802405316414706249_wp
        Omega(12) = 0.5541903301702226011934623861598936400696_wp
        Omega(13) = 0.9034013813719590036484426154306248690773_wp
        Omega(14) = 1.4739942115085553971543658180465286022809_wp
        Omega(15) = 2.4637954099354931818913888585953486654034_wp
        Omega(16) = 4.6029358609429098189193274581754167229519_wp

        Alpha(1) = 0.0002992241208797295031346425477894110379_wp
        Alpha(2) = 0.0016602631356552019942533424123887186852_wp
        Alpha(3) = 0.0045138170003323857566210179376869904289_wp
        Alpha(4) = 0.0098111412969511009946667898576344590822_wp
        Alpha(5) = 0.0194616438373382750691399038084350436861_wp
        Alpha(6) = 0.0368864680879441863142482582044179295622_wp
        Alpha(7) = 0.0679045674012652014929730497072579709084_wp
        Alpha(8) = 0.1222041350629109780914034397159007028222_wp
        Alpha(9) = 0.2156923865273337429540557660945232498761_wp
        Alpha(10) = 0.3741677069120103561001774994387858441769_wp
        Alpha(11) = 0.6390292853494859240956818546841589068208_wp
        Alpha(12) = 1.0762526604724020765427305867945051431889_wp
        Alpha(13) = 1.7911592740081455582632635747941662884841_wp
        Alpha(14) = 2.9558684609147227069503305330044895526953_wp
        Alpha(15) = 4.8751392319118493660565216707425406639231_wp
        Alpha(16) = 8.2366778447943210109474798663597994163865_wp

        Xi(1) = 1.006931202362213098188033211233_wp
        Xi(2) = 1.063019469384929592374600115789_wp
        Xi(3) = 1.178625720298270394695641649641_wp
        Xi(4) = 1.360758809358223196716089486991_wp
        Xi(5) = 1.620433485216211504163601186956_wp
        Xi(6) = 1.973517348237369513938403720044_wp
        Xi(7) = 2.442066143829587792279597890932_wp
        Xi(8) = 3.056165694352967220071004561532_wp
        Xi(9) = 3.856366173641724338222447343583_wp
        Xi(10) = 4.896955873372914460044807993810_wp
        Xi(11) = 6.250445492056464633773282812967_wp
        Xi(12) = 8.013740354402243216795642677397_wp
        Xi(13) = 10.316640688505583998285652924665_wp
        Xi(14) = 13.333568352935279436623228210834_wp
        Xi(15) = 17.299840002468369401858461209542_wp
        Xi(16) = 22.534299577395717753200421462623_wp
        Xi(17) = 29.471029029283412697948207892296_wp
        Xi(18) = 38.703928997986791538110651345050_wp
        Xi(19) = 51.049828735696785243086681660429_wp
        Xi(20) = 67.638080241728973672488134383229_wp
        Xi(21) = 90.038414614460910108395541584514_wp
        Xi(22) = 120.443318367013035181944946572230_wp
        Xi(23) = 161.927067404595660893296127369467_wp
        Xi(24) = 218.808000562085702991876878797939_wp
        Xi(25) = 297.133859138640721020419022124770_wp
        Xi(26) = 405.264591321704776033518768940667_wp
        Xi(27) = 554.347801774285637022021688835594_wp
        Xi(28) = 757.909757688632282901508574468608_wp
        Xi(29) = 1028.168681040490993860991864039534_wp
        Xi(30) = 1363.125723846814662976179022280121_wp
        Xi(31) = 1716.193156804617456989703327963070_wp
        Xi(32) = 1964.134483811259450303232654277963_wp

      case (23)
        ! ----- K = 16 , R = 3E3 -----

        RIni = 3.0e3_wp

        Omega(1) = 0.0005552467027450946637557560830837644072_wp
        Omega(2) = 0.0014584383837619173779302032801195865286_wp
        Omega(3) = 0.0028902827511056797591154930498688613305_wp
        Omega(4) = 0.0055003742951535241132696206836638630477_wp
        Omega(5) = 0.0103516891504137027865415699758033785116_wp
        Omega(6) = 0.0191571445523666846210585805801440706375_wp
        Omega(7) = 0.0347438857823840648833720046173567652659_wp
        Omega(8) = 0.0617661025400090076408908448118983969266_wp
        Omega(9) = 0.1077870257402970902431589561576430469358_wp
        Omega(10) = 0.1849494100974524521819718908544061264365_wp
        Omega(11) = 0.3125950216953475513045630079789205524321_wp
        Omega(12) = 0.5215821469062483484689651813681621206342_wp
        Omega(13) = 0.8624104031684323868472974439214340236504_wp
        Omega(14) = 1.4249765739881087528526798502426231607387_wp
        Omega(15) = 2.4078703105469664194453194294709419409628_wp
        Omega(16) = 4.5392702166529680200823815283683870802633_wp

        Alpha(1) = 0.0002134705217666186853756709926430330804_wp
        Alpha(2) = 0.0011944382121612197912085697013191698534_wp
        Alpha(3) = 0.0033030929481546051993269118990381727485_wp
        Alpha(4) = 0.0073633574898127585851037601484547767683_wp
        Alpha(5) = 0.0150440187979848748675706449420830246311_wp
        Alpha(6) = 0.0293726979281437198497910646272091028663_wp
        Alpha(7) = 0.0555989091923544578093484801184498422799_wp
        Alpha(8) = 0.1026407267113996364378634279512780835830_wp
        Alpha(9) = 0.1854143263568294130013086942754263475308_wp
        Alpha(10) = 0.3285173216429233088295078224483347639762_wp
        Alpha(11) = 0.5720171780433248305921568876986071927604_wp
        Alpha(12) = 0.9806147524370287133508097832113037384261_wp
        Alpha(13) = 1.6587347438742525073882005948178175458452_wp
        Alpha(14) = 2.7783675668309569231542027756276524996792_wp
        Alpha(15) = 4.6450720342179955234251931273092850460671_wp
        Alpha(16) = 7.9472026835846368108921822237533660882036_wp

        Xi(1) = 1.007313518843293634764252120206_wp
        Xi(2) = 1.066535862309835448508317512495_wp
        Xi(3) = 1.188816594963225726860021891351_wp
        Xi(4) = 1.382003997960557570312024877612_wp
        Xi(5) = 1.658469578868685293773950484297_wp
        Xi(6) = 2.036129981586481010272932090643_wp
        Xi(7) = 2.540058670835582692445683838933_wp
        Xi(8) = 3.204711797327240542206452933449_wp
        Xi(9) = 4.076896440729057241592059313007_wp
        Xi(10) = 5.219814824401254844260522602895_wp
        Xi(11) = 6.718675885724491570871047541047_wp
        Xi(12) = 8.688507703690568208937383687385_wp
        Xi(13) = 11.285044312162827710721568763574_wp
        Xi(14) = 14.719953669818344563213763365184_wp
        Xi(15) = 19.282243192993958297992707073831_wp
        Xi(16) = 25.368500856855381066223831254547_wp
        Xi(17) = 33.525907356905569393218380369603_wp
        Xi(18) = 44.513791278037670100242717552419_wp
        Xi(19) = 59.392255900397037244459186311474_wp
        Xi(20) = 79.650749218774877007109846971389_wp
        Xi(21) = 107.395501977541313345931861533700_wp
        Xi(22) = 145.624118898835682209269570819288_wp
        Xi(23) = 198.628445768431519641294968891998_wp
        Xi(24) = 272.582126817676001551138398326657_wp
        Xi(25) = 376.379417355359462415975713156513_wp
        Xi(26) = 522.761652063993322581136879989572_wp
        Xi(27) = 729.578496466861973479556269239765_wp
        Xi(28) = 1020.272640770886480754420233552082_wp
        Xi(29) = 1420.156514258006317574789534319279_wp
        Xi(30) = 1938.106130483476449266611041366559_wp
        Xi(31) = 2513.149382258735478057332102252985_wp
        Xi(32) = 2937.145917315011803161439729592530_wp

      case (24)
        ! ----- K = 16 , R = 4E3 -----

        RIni = 4.0e3_wp

        Omega(1) = 0.0004376510700857689912567819958853429507_wp
        Omega(2) = 0.0011653666641048586782173284792998857462_wp
        Omega(3) = 0.0023680456391125560362919923651156550726_wp
        Omega(4) = 0.0046379834344294636243877781069111509993_wp
        Omega(5) = 0.0089518482939726288836161530518276130408_wp
        Omega(6) = 0.0169190097024326531614162125866096531013_wp
        Omega(7) = 0.0312441751584375854244819475588612611716_wp
        Omega(8) = 0.0564366963334440862069471545048315874737_wp
        Omega(9) = 0.0999033373206869644461464337326095375147_wp
        Omega(10) = 0.1736501546243619733021290932328817291364_wp
        Omega(11) = 0.2969613931046048855015746792407327347973_wp
        Omega(12) = 0.5008065845528724652582254484034507413526_wp
        Omega(13) = 0.8360592730493393466692161886122391933895_wp
        Omega(14) = 1.3932536792220334117067825907554379227804_wp
        Omega(15) = 2.3715445629975790188277012804363153009035_wp
        Omega(16) = 4.4979306792040948293105728783558561190148_wp

        Alpha(1) = 0.0001680125928201301219977655484224925153_wp
        Alpha(2) = 0.0009465404888159955446108676782243640524_wp
        Alpha(3) = 0.0026538938383495571741655440775944918563_wp
        Alpha(4) = 0.0060343511107069091400544535389732780928_wp
        Alpha(5) = 0.0126044257319316648984378688609642171770_wp
        Alpha(6) = 0.0251439404415568449864206877417127650176_wp
        Alpha(7) = 0.0485422220818330998734133324007400389632_wp
        Alpha(8) = 0.0912233302740963573004037527602871904264_wp
        Alpha(9) = 0.1674571159436080884457508088924981848322_wp
        Alpha(10) = 0.3010446184572926708151400371216155349430_wp
        Alpha(11) = 0.5311533323793717916816263524548702434913_wp
        Alpha(12) = 0.9216025899928024695028153101539203362336_wp
        Alpha(13) = 1.5761719145679200359405625064823652792256_wp
        Alpha(14) = 2.6667207963614877679840026170055011789373_wp
        Alpha(15) = 4.4993322323555962252011886093328030256089_wp
        Alpha(16) = 7.7628668850489034132175691915733750647632_wp

        Xi(1) = 1.007574569294012870328443309820_wp
        Xi(2) = 1.068939445826771577870498153295_wp
        Xi(3) = 1.195796686120764102658972638515_wp
        Xi(4) = 1.396597582035634397060186451256_wp
        Xi(5) = 1.684692532946183825921446231000_wp
        Xi(6) = 2.079486669305312379792127774891_wp
        Xi(7) = 2.608262938092970148690544207604_wp
        Xi(8) = 3.308703009241117815606228469427_wp
        Xi(9) = 4.232270880775881090156420594184_wp
        Xi(10) = 5.448864721892181405080851641642_wp
        Xi(11) = 7.053326496492226941638359338782_wp
        Xi(12) = 9.174574041637437877769190031074_wp
        Xi(13) = 11.988440511077821115822061859291_wp
        Xi(14) = 15.735778913915754973276706074614_wp
        Xi(15) = 20.748179556649722623737353011109_wp
        Xi(16) = 27.484647354795787521641869943778_wp
        Xi(17) = 36.584363092090761336583337737238_wp
        Xi(18) = 48.943045590636358243424863800897_wp
        Xi(19) = 65.824300738152503126299208702221_wp
        Xi(20) = 89.023331061222415204470248539792_wp
        Xi(21) = 121.109255467692712819605116436605_wp
        Xi(22) = 165.786268551979793922646067017013_wp
        Xi(23) = 228.434385063764427259425104921320_wp
        Xi(24) = 316.918677395432000221120460992097_wp
        Xi(25) = 442.786666521232530724372367103570_wp
        Xi(26) = 622.972120263308429599646842689253_wp
        Xi(27) = 881.953039206869549737621838403356_wp
        Xi(28) = 1253.450674381069734564242423857650_wp
        Xi(29) = 1777.482136531094566778143928331701_wp
        Xi(30) = 2478.162137296291906984890829335200_wp
        Xi(31) = 3286.862161306532813975778140047623_wp
        Xi(32) = 3906.325460007557433206315522511431_wp

      case (25)
        ! ----- K = 16 , R = 5E3 -----

        RIni = 5.0e3_wp

        Omega(1) = 0.0003639842736266964178310734200202589128_wp
        Omega(2) = 0.0009806443211126502088166863421898888920_wp
        Omega(3) = 0.0020348019995182053730111034189204088918_wp
        Omega(4) = 0.0040773537715263094949365708756505721766_wp
        Omega(5) = 0.0080241245702683121638685410058344782414_wp
        Omega(6) = 0.0154111678077861830196034474173283346232_wp
        Omega(7) = 0.0288537422675381651965184062400004982862_wp
        Omega(8) = 0.0527529528046750570404127642781810791917_wp
        Omega(9) = 0.0943962899936182988163920794255634660885_wp
        Omega(10) = 0.1656821898372087954162722514550587504800_wp
        Omega(11) = 0.2858429295176569531538798840664128420030_wp
        Omega(12) = 0.4859200114387559206441911940910216571865_wp
        Omega(13) = 0.8170578475224640865501833653095076215322_wp
        Omega(14) = 1.3702697547901459369443233349272759369342_wp
        Omega(15) = 2.3451567673310105685210991133260449714726_wp
        Omega(16) = 4.4679066019423948753590269955537905843812_wp

        Alpha(1) = 0.0001395561408272046871737763002385354638_wp
        Alpha(2) = 0.0007908754043432692235792866254553284966_wp
        Alpha(3) = 0.0022437319728956972643010923506440290609_wp
        Alpha(4) = 0.0051861449510816475890272018740151382943_wp
        Alpha(5) = 0.0110265465390968265123897368643057781412_wp
        Alpha(6) = 0.0223694637981097021065608927703483033156_wp
        Alpha(7) = 0.0438480579093586883351212936388296981249_wp
        Alpha(8) = 0.0835312227650857650027189926777992923235_wp
        Alpha(9) = 0.1552186994022961743486038396766524272152_wp
        Alpha(10) = 0.2821252956665503958794648242669467208543_wp
        Alpha(11) = 0.5027477680571566028976289441931868395841_wp
        Alpha(12) = 0.8802381938407273885690770831580209687672_wp
        Alpha(13) = 1.5178750221968237785891142621608196350280_wp
        Alpha(14) = 2.5873972191280157082240998001054776977981_wp
        Alpha(15) = 4.3952693534984634562111371902659584520734_wp
        Alpha(16) = 7.6307623342543931983794625040218306821771_wp

        Xi(1) = 1.007770795780931430678933891087_wp
        Xi(2) = 1.070747566096610958611204389435_wp
        Xi(3) = 1.201055232665560991260622625276_wp
        Xi(4) = 1.407614329043466570032597240392_wp
        Xi(5) = 1.704539598973179336037324371755_wp
        Xi(6) = 2.112404195000717510837667800772_wp
        Xi(7) = 2.660234480703984382820442844420_wp
        Xi(8) = 3.388271982928642332612984011675_wp
        Xi(9) = 4.351697965542117357059387350304_wp
        Xi(10) = 5.625791153667208028144736120524_wp
        Xi(11) = 7.313189251692526079984268960743_wp
        Xi(12) = 9.554136392540210479841500390563_wp
        Xi(13) = 12.540973361192514306389211853343_wp
        Xi(14) = 16.538737305833064141630250887260_wp
        Xi(15) = 21.914584017855716394432286087124_wp
        Xi(16) = 29.180138251998641833392933442326_wp
        Xi(17) = 39.052865128266937247425438783921_wp
        Xi(18) = 52.545679566284073297444123795685_wp
        Xi(19) = 71.098914463001920728546512151524_wp
        Xi(20) = 96.776240058579681938000227603425_wp
        Xi(21) = 132.558018298638639761999336030840_wp
        Xi(22) = 182.784080056747836526276351776232_wp
        Xi(23) = 253.826595804328427108398358313934_wp
        Xi(24) = 355.115530315099602431949676883960_wp
        Xi(25) = 500.693591776281145433458164006879_wp
        Xi(26) = 711.511993913223033003667694629257_wp
        Xi(27) = 1018.549699793377793732851444019616_wp
        Xi(28) = 1465.938683276471933170093109310983_wp
        Xi(29) = 2109.391908448062716407989114486554_wp
        Xi(30) = 2991.442293428318250025910174372257_wp
        Xi(31) = 4041.741988572836854709535714391677_wp
        Xi(32) = 4872.251104496301511215250457098591_wp

      case (26)
        ! ----- K = 16 , R = 6E3 -----

        RIni = 6.0e3_wp

        Omega(1) = 0.0003131740893025812162727879048524162742_wp
        Omega(2) = 0.0008525743777120362815273941689813569367_wp
        Omega(3) = 0.0018013124768756676217053899063946786896_wp
        Omega(4) = 0.0036784434068537452545763754294463887540_wp
        Omega(5) = 0.0073538954069976066201639480247875813035_wp
        Omega(6) = 0.0143079355017790147446262523540927436017_wp
        Omega(7) = 0.0270861317219600750889917963049047244795_wp
        Omega(8) = 0.0500039451671855890466740404004530873294_wp
        Omega(9) = 0.0902531713884734725832168639181318781084_wp
        Omega(10) = 0.1596439284804761997286527697159641547842_wp
        Omega(11) = 0.2773620927927093378633112419340278620439_wp
        Omega(12) = 0.4744994636619417472235635951927079645429_wp
        Omega(13) = 0.8024095949928360991892060949393084001713_wp
        Omega(14) = 1.3524864442578875393420612116557322224253_wp
        Omega(15) = 2.3246983701212194409983086140769614758028_wp
        Omega(16) = 4.4446320157087978023648100478482092512422_wp

        Alpha(1) = 0.0001199406337778443636857985098989262673_wp
        Alpha(2) = 0.0006832918618089672388053348428115646440_wp
        Alpha(3) = 0.0019587716841422183862100490567281774190_wp
        Alpha(4) = 0.0045917452487347416349730230313019863075_wp
        Alpha(5) = 0.0099085570350233827150601239978910950157_wp
        Alpha(6) = 0.0203808716667070842385535477461755426987_wp
        Alpha(7) = 0.0404466664137481064933844331771295443900_wp
        Alpha(8) = 0.0779018628897185319790168363252025329757_wp
        Alpha(9) = 0.1461816622304678183863046320167811131796_wp
        Alpha(10) = 0.2680422516895201304264022279522095004722_wp
        Alpha(11) = 0.4814506743542380286259886601474278222668_wp
        Alpha(12) = 0.8490259511837918704729027963296061898291_wp
        Alpha(13) = 1.4736387032169466420683870411245663945010_wp
        Alpha(14) = 2.5269188569364048871281752672857123798167_wp
        Alpha(15) = 4.3156266381570576305161390795461784364306_wp
        Alpha(16) = 7.5293759519457498950376073132417786837323_wp

        Xi(1) = 1.007926880878180067591110002923_wp
        Xi(2) = 1.072186661890797544074635516687_wp
        Xi(3) = 1.205245264343925420112181579047_wp
        Xi(4) = 1.416406408431956687773815850573_wp
        Xi(5) = 1.720410532419148070897539637425_wp
        Xi(6) = 2.138790611290749044993975136464_wp
        Xi(7) = 2.702011854768343456596399465575_wp
        Xi(8) = 3.452437690046877283951351489399_wp
        Xi(9) = 4.448344337372049695205483699212_wp
        Xi(10) = 5.769514515083047219781708925268_wp
        Xi(11) = 7.525145864411813474790374778323_wp
        Xi(12) = 9.865065558558741913922074395771_wp
        Xi(13) = 12.995677729521920621379160731834_wp
        Xi(14) = 17.202730517444799317877679456501_wp
        Xi(15) = 22.884064885755052969756873082474_wp
        Xi(16) = 30.597010199494975324699397312145_wp
        Xi(17) = 41.127511184094246686482332364676_wp
        Xi(18) = 55.591839301494131185543690065742_wp
        Xi(19) = 75.587438261880676916570909185111_wp
        Xi(20) = 103.418740278164994830789602531240_wp
        Xi(21) = 142.438347995312840466031190089780_wp
        Xi(22) = 197.567170711043371827964065801098_wp
        Xi(23) = 276.094099660891465264356270381541_wp
        Xi(24) = 388.911863069203033865806062863157_wp
        Xi(25) = 552.425595514388325379950828164510_wp
        Xi(26) = 791.446501710242535265749097561638_wp
        Xi(27) = 1143.313849906627009955784046724148_wp
        Xi(28) = 1662.593202381695404956651884731400_wp
        Xi(29) = 2421.343267206676987290947522524220_wp
        Xi(30) = 3482.956847843122484276179307016719_wp
        Xi(31) = 4780.612701110991521691317984732450_wp
        Xi(32) = 5835.298406764837678473156756808748_wp

      case (27)
        ! ----- K = 16 , R = 7E3 -----

        RIni = 7.0e3_wp

        Omega(1) = 0.0002758456244211198484029873463641013842_wp
        Omega(2) = 0.0007580596968981613001793861046119504810_wp
        Omega(3) = 0.0016273891617054744144489056203312093984_wp
        Omega(4) = 0.0033773574743788997862165452432461898447_wp
        Omega(5) = 0.0068416530214521522122995110266191476356_wp
        Omega(6) = 0.0134560408443224474344041604462196559666_wp
        Omega(7) = 0.0257094629782190379569734766176947182714_wp
        Omega(8) = 0.0478470046371534687721445607000436694989_wp
        Omega(9) = 0.0869809853452546726990633751963599706869_wp
        Omega(10) = 0.1548469094737899646985722357239723123712_wp
        Omega(11) = 0.2705890951533227137845803028648461463490_wp
        Omega(12) = 0.4653363336543718229733595376229615681041_wp
        Omega(13) = 0.7906106418418388792081728799754358760765_wp
        Omega(14) = 1.3381198015624958747374351197834130289266_wp
        Omega(15) = 2.3081433515491244952422267289193769101985_wp
        Omega(16) = 4.4257998198618129855226588365013640213874_wp

        Alpha(1) = 0.0001055373074657432837801094713465251823_wp
        Alpha(2) = 0.0006041148903459930011925159410151459083_wp
        Alpha(3) = 0.0017480808776888785328212407021997165657_wp
        Alpha(4) = 0.0041489235133947849417641688934271781974_wp
        Alpha(5) = 0.0090677748285274670356467308952874706307_wp
        Alpha(6) = 0.0188708736505370171843322260954445290793_wp
        Alpha(7) = 0.0378405483771319748233371519502421520542_wp
        Alpha(8) = 0.0735534743950406653969728044961673418811_wp
        Alpha(9) = 0.1391499701720789750611969454174854377015_wp
        Alpha(10) = 0.2570126196928731625095890939158849164414_wp
        Alpha(11) = 0.4646737063801709214359327015930745119476_wp
        Alpha(12) = 0.8243107746878442959105083576165640124600_wp
        Alpha(13) = 1.4384517778893632151309203703881678393373_wp
        Alpha(14) = 2.4786279831485562544397815987196054265951_wp
        Alpha(15) = 4.2518384035905261455166870288024938417948_wp
        Alpha(16) = 7.4479905527901974220290892692020179310930_wp

        Xi(1) = 1.008055779207782207780239036587_wp
        Xi(2) = 1.073375671584851511462278816911_wp
        Xi(3) = 1.208710313926687793728217967804_wp
        Xi(4) = 1.423686547017540950078005879131_wp
        Xi(5) = 1.733573479654473688000432263223_wp
        Xi(6) = 2.160717628625757772385662658365_wp
        Xi(7) = 2.736808038339744198266800800923_wp
        Xi(8) = 3.506018171653663538555115164463_wp
        Xi(9) = 4.529277527399502834264816142351_wp
        Xi(10) = 5.890241476602233601424923881140_wp
        Xi(11) = 7.703775160018586064866047413702_wp
        Xi(12) = 10.128028644161589948494150981162_wp
        Xi(13) = 13.381664314000179407879165260908_wp
        Xi(14) = 17.768600388391513886232608854243_wp
        Xi(15) = 23.713719700629780573475358096402_wp
        Xi(16) = 31.814861739974554624352243159535_wp
        Xi(17) = 42.919065660375737857046285483875_wp
        Xi(18) = 58.235365994749826740878306807758_wp
        Xi(19) = 79.503158873587560423468190151652_wp
        Xi(20) = 109.245995836587721304622444762344_wp
        Xi(21) = 151.157855596998021305288339988238_wp
        Xi(22) = 210.696927268345853390130173465877_wp
        Xi(23) = 296.007248588640691083773148761793_wp
        Xi(24) = 419.359245334154272927262852022068_wp
        Xi(25) = 599.406688185068775298613985569318_wp
        Xi(26) = 864.679841780689235786194757338308_wp
        Xi(27) = 1258.735317870830061504072716616065_wp
        Xi(28) = 1846.539033092342188324685992029117_wp
        Xi(29) = 2716.937133125949431988743754118332_wp
        Xi(30) = 3956.119117622416815338581841388077_wp
        Xi(31) = 5505.466470836707039815394182369346_wp
        Xi(32) = 6795.735316804062661422136670807959_wp

      case (28)
        ! ----- K = 16 , R = 8E3 -----

        RIni = 8.0e3_wp

        Omega(1) = 0.0002471688129707959323570824706692294903_wp
        Omega(2) = 0.0006851572503816519147648152698003298777_wp
        Omega(3) = 0.0014921087303356729843801153825161298627_wp
        Omega(4) = 0.0031404544653767500959818303931520011574_wp
        Omega(5) = 0.0064343087795187726742063023116502407106_wp
        Omega(6) = 0.0127727320392762643041777051271246090636_wp
        Omega(7) = 0.0245972718527957374620458025349378772262_wp
        Omega(8) = 0.0460935944652694077526772567510171541016_wp
        Omega(9) = 0.0843063411502943351981940491024491990402_wp
        Omega(10) = 0.1509066144125360964657698442659139459465_wp
        Omega(11) = 0.2650012689108123447917410991925279972747_wp
        Omega(12) = 0.4577473064340514511837056804699486178833_wp
        Omega(13) = 0.7808066064884537147545127966363764926427_wp
        Omega(14) = 1.3261526510468241279311341074276242579799_wp
        Omega(15) = 2.2943342451891662763831247762169596171589_wp
        Omega(16) = 4.4100922622450336430498241124809055691003_wp

        Alpha(1) = 0.0000944773691119611902426450944308088109_wp
        Alpha(2) = 0.0005431933718162297242877610526333442831_wp
        Alpha(3) = 0.0015852907600157036040126309677997240577_wp
        Alpha(4) = 0.0038044511901996254332009391894639516174_wp
        Alpha(5) = 0.0084083088894682432245410836323118441982_wp
        Alpha(6) = 0.0176766622119811988063381317029532091567_wp
        Alpha(7) = 0.0357636322027086082402298240767724024636_wp
        Alpha(8) = 0.0700642034135144223044259353949669844042_wp
        Alpha(9) = 0.1334728750534660843214728245809119755450_wp
        Alpha(10) = 0.2480589770841020254066094583023094344298_wp
        Alpha(11) = 0.4509880134678298451529274148974479885510_wp
        Alpha(12) = 0.8040623195001615163262503882535270349763_wp
        Alpha(13) = 1.4095152073454303062174583494048363263573_wp
        Alpha(14) = 2.4387882648489325467747873554813509144878_wp
        Alpha(15) = 4.1990791711593300687514751512452448878321_wp
        Alpha(16) = 7.3805513886962790482890173215935192274628_wp

        Xi(1) = 1.008165110648253801062246282072_wp
        Xi(2) = 1.074384597037245452273192258819_wp
        Xi(3) = 1.211652804915005441363864968629_wp
        Xi(4) = 1.429875405649688178402398341937_wp
        Xi(5) = 1.744778502878409746543707237176_wp
        Xi(6) = 2.179413638528453895943509666111_wp
        Xi(7) = 2.766533085561071399509011636830_wp
        Xi(8) = 3.551889198475691804398821083133_wp
        Xi(9) = 4.598729763476446650693335582716_wp
        Xi(10) = 5.994109407066016876903385979247_wp
        Xi(11) = 7.857885718551512398057784514194_wp
        Xi(12) = 10.355562038617222487928148932923_wp
        Xi(13) = 13.716691201702225162521731061105_wp
        Xi(14) = 18.261379363073626068761368124171_wp
        Xi(15) = 24.438738851383566025310400959825_wp
        Xi(16) = 32.883065798256496028301176792752_wp
        Xi(17) = 44.496644198750315540857869223146_wp
        Xi(18) = 60.572853238664348046044194617821_wp
        Xi(19) = 82.980927397131630282589576097507_wp
        Xi(20) = 114.446023333009322210662705998629_wp
        Xi(21) = 158.978262123559724389165737079566_wp
        Xi(22) = 222.536874297237396921889107659354_wp
        Xi(23) = 314.069373225468011751049246527145_wp
        Xi(24) = 447.151233986197667480411155338516_wp
        Xi(25) = 642.585980326371911719451901490174_wp
        Xi(26) = 932.495580759485205857295397890994_wp
        Xi(27) = 1366.515654488450761561146862277383_wp
        Xi(28) = 2019.946039647338810851806556456722_wp
        Xi(29) = 2998.725418032073378071800107136369_wp
        Xi(30) = 4413.413095771435769254509295933531_wp
        Xi(31) = 6217.794961798526415908128228693386_wp
        Xi(32) = 7753.764950412754636488443793496117_wp

      case (29)
        ! ----- K = 16 , R = 9E3 -----

        RIni = 9.0e3_wp

        Omega(1) = 0.0002243920605751591336424850472996972428_wp
        Omega(2) = 0.0006270417067617411212975996796879091244_wp
        Omega(3) = 0.0013834406617184846414476335802086481852_wp
        Omega(4) = 0.0029481984836296284451914099904168420352_wp
        Omega(5) = 0.0061006853878017267386307657429933559712_wp
        Omega(6) = 0.0122089286671955113017460086040355538017_wp
        Omega(7) = 0.0236739223810270169472055054940562435206_wp
        Omega(8) = 0.0446301312678028930030701970643747245049_wp
        Omega(9) = 0.0820634642574863223097821930684947488999_wp
        Omega(10) = 0.1475885080271769715372191800195622590763_wp
        Omega(11) = 0.2602781043968390923865237340795175668973_wp
        Omega(12) = 0.4513113627618848883095870744286415288116_wp
        Omega(13) = 0.7724689548824036639928375391761505852628_wp
        Omega(14) = 1.3159538709954992786576449814717193476099_wp
        Omega(15) = 2.2825517326481741162656768917571525889798_wp
        Omega(16) = 4.3966905735612235627332577969639260118129_wp

        Alpha(1) = 0.0000856966014984952755532602313326851623_wp
        Alpha(2) = 0.0004947372328309368735756184785531952564_wp
        Alpha(3) = 0.0014553149185678176834364362982575380467_wp
        Alpha(4) = 0.0035277193978652813810531198258566010217_wp
        Alpha(5) = 0.0078746243803636643179873623281384009687_wp
        Alpha(6) = 0.0167031895938322041453679344877314427720_wp
        Alpha(7) = 0.0340593424378246069270895646924346422679_wp
        Alpha(8) = 0.0671839347162209137161951316707941828099_wp
        Alpha(9) = 0.1287618816032252760863264368085268074537_wp
        Alpha(10) = 0.2405941184184795421716193328243171833947_wp
        Alpha(11) = 0.4395302550885714915464373003795017780249_wp
        Alpha(12) = 0.7870474597304489113502520836185283314990_wp
        Alpha(13) = 1.3851212211488514633971058653827412854298_wp
        Alpha(14) = 2.4051111576427746828119036415216669411166_wp
        Alpha(15) = 4.1543839778031023445351022083116276917281_wp
        Alpha(16) = 7.3233295014735437632874548441463957715314_wp

        Xi(1) = 1.008259723879726138789231226767_wp
        Xi(2) = 1.075258004261145553404760422112_wp
        Xi(3) = 1.214201731916668916913017162162_wp
        Xi(4) = 1.435241413958034460847910940284_wp
        Xi(5) = 1.754505053750317001420855334004_wp
        Xi(6) = 2.195665201139894474446218053387_wp
        Xi(7) = 2.792414039715915654400399392721_wp
        Xi(8) = 3.591902005587370458631821246165_wp
        Xi(9) = 4.659435744122105359579788519397_wp
        Xi(10) = 6.085097732474222439243843885137_wp
        Xi(11) = 7.993205967130740433566893354822_wp
        Xi(12) = 10.555856697719727954609680864451_wp
        Xi(13) = 14.012398949955290462440760101970_wp
        Xi(14) = 18.697561378800773526642764466033_wp
        Xi(15) = 25.082410883299436812898797199978_wp
        Xi(16) = 33.834433326273254095950537845283_wp
        Xi(17) = 45.906429557701034872840839540231_wp
        Xi(18) = 62.669241316123736631332752367030_wp
        Xi(19) = 86.111937187298724968864682693948_wp
        Xi(20) = 119.146741352932677100728842134458_wp
        Xi(21) = 166.078778721825472852735572359961_wp
        Xi(22) = 233.337675973566279430393421989720_wp
        Xi(23) = 330.630228643207634497969138465123_wp
        Xi(24) = 472.773955560221967292511280334111_wp
        Xi(25) = 682.634311241447989671016216561839_wp
        Xi(26) = 995.810103863546314306720574904830_wp
        Xi(27) = 1467.882979342089860841369386434963_wp
        Xi(28) = 2184.401544057028329515190989695839_wp
        Xi(29) = 3268.607022735769193300114920930355_wp
        Xi(30) = 4856.732194515447875726721349565196_wp
        Xi(31) = 6918.766716127081207066140677852673_wp
        Xi(32) = 8709.547740942938796493422159983311_wp

      case (30)
        ! ----- K = 16 , R = 1E4 -----

        RIni = 1.0e4_wp

        Omega(1) = 0.0002058282962339966570492447403323450772_wp
        Omega(2) = 0.0005795163034704684972347108991904546249_wp
        Omega(3) = 0.0012939465354750554289245856494361408551_wp
        Omega(4) = 0.0027884026266817487573487446437733860982_wp
        Omega(5) = 0.0058211432932475670054928791521259290320_wp
        Omega(6) = 0.0117334510546047818806762968839307870184_wp
        Omega(7) = 0.0228910200924036787596406146774730139271_wp
        Omega(8) = 0.0433834862866021760988372111406130571254_wp
        Omega(9) = 0.0801450248378290806266642694322133166906_wp
        Omega(10) = 0.1447399726413308090276202064194954743925_wp
        Omega(11) = 0.2562100884259844294811581111304121805006_wp
        Omega(12) = 0.4457521692955299713254091464875727979233_wp
        Omega(13) = 0.7652495837273190427605361330076760850716_wp
        Omega(14) = 1.3071066983128932045181772036990253127442_wp
        Omega(15) = 2.2723201246218485212409465834326738331583_wp
        Omega(16) = 4.3850533669438823197381238649938950402429_wp

        Alpha(1) = 0.0000785427260255526335381718684768102140_wp
        Alpha(2) = 0.0004551924617865591627534382120762179724_wp
        Alpha(3) = 0.0013488674597178575663637117702277345987_wp
        Alpha(4) = 0.0032998006170175148886844929563971762221_wp
        Alpha(5) = 0.0074321614793527533236597911597831611630_wp
        Alpha(6) = 0.0158908890315543802752948788353792863859_wp
        Alpha(7) = 0.0326288715086645033290298819478314484854_wp
        Alpha(8) = 0.0647538075936886720816677495693447497160_wp
        Alpha(9) = 0.1247687659706603474505303905117692409021_wp
        Alpha(10) = 0.2342408160070870709053942812505333392892_wp
        Alpha(11) = 0.4297430785088648099107402900109775600868_wp
        Alpha(12) = 0.7724665951175793187429181874126271623027_wp
        Alpha(13) = 1.3641581106829700108717604023489400333347_wp
        Alpha(14) = 2.3761018937333028314910082778865785257949_wp
        Alpha(15) = 4.1158110002476520194319986600817173894029_wp
        Alpha(16) = 7.2738777917555991910612822604065286213881_wp

        Xi(1) = 1.008342889724298546520742103638_wp
        Xi(2) = 1.076025969828900324021421497367_wp
        Xi(3) = 1.216444223911958485550427466748_wp
        Xi(4) = 1.439966112440204014421077371999_wp
        Xi(5) = 1.763077677382567077742948902230_wp
        Xi(6) = 2.210006274817696659672899839322_wp
        Xi(7) = 2.815285079582492041780361624603_wp
        Xi(8) = 3.627318434126628444785431226727_wp
        Xi(9) = 4.713263842711685328631249225495_wp
        Xi(10) = 6.165933069442765533749867268298_wp
        Xi(11) = 8.113673080220766232681373075764_wp
        Xi(12) = 10.734560325839581347226459051569_wp
        Xi(13) = 14.276844759455947475243475786044_wp
        Xi(14) = 19.088598358000106149784391718072_wp
        Xi(15) = 25.660981461558588253082935359828_wp
        Xi(16) = 34.691962651683783212114287053396_wp
        Xi(17) = 47.180913850895391293410741084813_wp
        Xi(18) = 64.570411096551821504563317688508_wp
        Xi(19) = 88.960948595539069416737731899048_wp
        Xi(20) = 123.439503538840588771352546615390_wp
        Xi(21) = 172.588155100055261406688522640707_wp
        Xi(22) = 243.280490027515989531758933139827_wp
        Xi(23) = 345.944111231481443657909125022343_wp
        Xi(24) = 496.583338359364417052832507692983_wp
        Xi(25) = 720.047273570623065530593720495744_wp
        Xi(26) = 1055.305720940396589591081522030436_wp
        Xi(27) = 1563.760916773524789480376284700469_wp
        Xi(28) = 2341.113443378716730380872945715964_wp
        Xi(29) = 3528.049719157751417775159552547848_wp
        Xi(30) = 5287.568848380935123021373556184699_wp
        Xi(31) = 7609.323569634084918344996140149306_wp
        Xi(32) = 9663.214336891674749452363357704598_wp

      case (31)
        ! ----- K = 16 , R = 3E6 (Best : R = 2080000) -----

        RIni = 3.0e6_wp

        Omega(1) = 0.0000061003972863863402651743415894584292_wp
        Omega(2) = 0.0000400806962962503114856435918063846946_wp
        Omega(3) = 0.0001640757836358783800369896591054251489_wp
        Omega(4) = 0.0005333789180131260076594652082184605524_wp
        Omega(5) = 0.0015016015641512588418497568549386800285_wp
        Omega(6) = 0.0038216606296993014147883241512712593213_wp
        Omega(7) = 0.0090155651757342924576852727716613777886_wp
        Omega(8) = 0.0200340931971503289075645743919085184359_wp
        Omega(9) = 0.0424034320065246971366340524717264415244_wp
        Omega(10) = 0.0861757293982789379928571675548720776305_wp
        Omega(11) = 0.1691970111509645595203711956888703582536_wp
        Omega(12) = 0.3225876471954512571087281269655022697407_wp
        Omega(13) = 0.6004239149409639519677957253396272108148_wp
        Omega(14) = 1.1003842126562718536008853020824460600124_wp
        Omega(15) = 2.0300383954052895140417189789872054461739_wp
        Omega(16) = 4.1094986543741171998445071977812403929420_wp

        Alpha(1) = 0.0000019082799452525089727123312353175870_wp
        Alpha(2) = 0.0000209929942749567198929778781085970021_wp
        Alpha(3) = 0.0001111065144474894622336291847425338020_wp
        Alpha(4) = 0.0004288308553325246431604475684908409083_wp
        Alpha(5) = 0.0013732813892174006607116380621926021721_wp
        Alpha(6) = 0.0038744692237789735938431894164690749705_wp
        Alpha(7) = 0.0099592966839161908000036156629608541380_wp
        Alpha(8) = 0.0238195511381459912370105267559461381666_wp
        Alpha(9) = 0.0537624744822230793628780806669986702673_wp
        Alpha(10) = 0.1156769011058406200660730878126969400910_wp
        Alpha(11) = 0.2390543762941218424247519919179083558447_wp
        Alpha(12) = 0.4772836329010108079079401782740532667049_wp
        Alpha(13) = 0.9252667046693414284173374861897087839679_wp
        Alpha(14) = 1.7511652086513542147475780419441093727073_wp
        Alpha(15) = 3.2656590749026312514666919861738847430388_wp
        Alpha(16) = 6.1657963486574848834508022488165579488850_wp

        Xi(1) = 1.010619076865009477754139832761_wp
        Xi(2) = 1.097130811991455252374719209207_wp
        Xi(3) = 1.278543967005584003045551289013_wp
        Xi(4) = 1.572208435851447816183554251346_wp
        Xi(5) = 2.006295143517123847655414525804_wp
        Xi(6) = 2.623643844654697781593535865774_wp
        Xi(7) = 3.487852850749272319965180177981_wp
        Xi(8) = 4.692003643084871500334998017934_wp
        Xi(9) = 6.371504907418426590353516258958_wp
        Xi(10) = 8.723643303268770113692576195774_wp
        Xi(11) = 12.037534791178474512587937006458_wp
        Xi(12) = 16.740308053138351525312987888583_wp
        Xi(13) = 23.469074329074754535831570656512_wp
        Xi(14) = 33.184443208950699554654617529081_wp
        Xi(15) = 47.352164865604347267213913852402_wp
        Xi(16) = 68.238125166270907559495917382719_wp
        Xi(17) = 99.395846543504521516942951109286_wp
        Xi(18) = 146.486945151395616310230352041799_wp
        Xi(19) = 218.690811018864780301829142672432_wp
        Xi(20) = 331.184415657320374265504270283600_wp
        Xi(21) = 509.612454553907175014426655934585_wp
        Xi(22) = 798.406621704696891683727955069116_wp
        Xi(23) = 1276.735904045949239304924560656218_wp
        Xi(24) = 2090.453775548968897268764521868434_wp
        Xi(25) = 3518.779759495649122946758780017262_wp
        Xi(26) = 6121.688303599042539016039654597989_wp
        Xi(27) = 11089.508083923279960991692405514186_wp
        Xi(28) = 21134.795018521869595318207757372875_wp
        Xi(29) = 43096.003825583163386880869438755326_wp
        Xi(30) = 96599.234684056726585765773052116856_wp
        Xi(31) = 250784.099339365001043233860400505364_wp
        Xi(32) = 881597.133850978159500755282351747155_wp

    end select

  case (17)
    ! ===== K = 17 =====

    select case (InitR)

      case default !(1:4)
        ! ----- K = 17 , R = 2E1 -----

        Omega(1) = 0.0302397699486285961049871363198748852597_wp
        Omega(2) = 0.0714111729693463297876233723815442289151_wp
        Omega(3) = 0.1151983413308698069545316956807656794126_wp
        Omega(4) = 0.1633463455699484639498933122303547804677_wp
        Omega(5) = 0.2181445810861671799195816295879879476161_wp
        Omega(6) = 0.2826292106846990995726309758850902653649_wp
        Omega(7) = 0.3608685543158396038924441706630119597321_wp
        Omega(8) = 0.4582722819283937388363473625085831031356_wp
        Omega(9) = 0.5819243715708696990151008177072355920245_wp
        Omega(10) = 0.7410321325985205146835016476458690704021_wp
        Omega(11) = 0.9477492570824826021358075478495663901413_wp
        Omega(12) = 1.2188845101312763704280717202976802582270_wp
        Omega(13) = 1.5795805617706500316010237439101615564141_wp
        Omega(14) = 2.0718714460365349940701440445423031633254_wp
        Omega(15) = 2.7779101419370363975497317809981723257806_wp
        Omega(16) = 3.9019593558207467727961398340497112258163_wp
        Omega(17) = 6.2568211133691469119991190872553943336243_wp

        Alpha(1) = 0.0117636499669930626207187650267282919003_wp
        Alpha(2) = 0.0624282057373425173215709168411668983367_wp
        Alpha(3) = 0.1554499074025039259292976789872398057923_wp
        Alpha(4) = 0.2942728713790734211899757788444276229711_wp
        Alpha(5) = 0.4843511156615427304611767866937555027107_wp
        Alpha(6) = 0.7337793906644703758050847508531688845324_wp
        Alpha(7) = 1.0541795508192826758128365916000745983183_wp
        Alpha(8) = 1.4618861427706882925641948545170123452408_wp
        Alpha(9) = 1.9794504252494227664656983867175199520716_wp
        Alpha(10) = 2.6375217858538901655098690701706232175638_wp
        Alpha(11) = 3.4773350777344393465646821272940769631532_wp
        Alpha(12) = 4.5544058764435397814417105788464823490358_wp
        Alpha(13) = 5.9447852881084881178017087499654280691175_wp
        Alpha(14) = 7.7570395281938202330582199373054663738003_wp
        Alpha(15) = 10.1586829246453062003396894397155847400427_wp
        Alpha(16) = 13.4479116052396606220686869903602200793102_wp
        Alpha(17) = 18.3386844672637758790756379312369972467422_wp

        Xi(1) = 1.001944992041143801786949596444_wp
        Xi(2) = 1.017573053775527585772427141553_wp
        Xi(3) = 1.049054516638138171043259649906_wp
        Xi(4) = 1.096933820715218840607073358395_wp
        Xi(5) = 1.161946276668325532619580053861_wp
        Xi(6) = 1.245136996763966193602253540451_wp
        Xi(7) = 1.347820890830599597641666076431_wp
        Xi(8) = 1.471549417805818207633135330514_wp
        Xi(9) = 1.618489190546147302065919937331_wp
        Xi(10) = 1.790745220583274777289249646284_wp
        Xi(11) = 1.991282425182284036521175585310_wp
        Xi(12) = 2.223219853512135877953923013273_wp
        Xi(13) = 2.490140598951331849485180125470_wp
        Xi(14) = 2.796239012409017054942850233346_wp
        Xi(15) = 3.146287033789434546071986376070_wp
        Xi(16) = 3.545673539110263757087246339061_wp
        Xi(17) = 4.000046161168884129629569335052_wp
        Xi(18) = 4.515993532901835928825701182987_wp
        Xi(19) = 5.100193050568134361683753086325_wp
        Xi(20) = 5.759757577923325344917970802783_wp
        Xi(21) = 6.501659436276989377859192309472_wp
        Xi(22) = 7.332452451966076140913580738712_wp
        Xi(23) = 8.256903948714259114556590635203_wp
        Xi(24) = 9.277899381065644061374053652713_wp
        Xi(25) = 10.393855072285290004792390750055_wp
        Xi(26) = 11.597510360067749908899448207933_wp
        Xi(27) = 12.872698832376013395629998914416_wp
        Xi(28) = 14.192600960322230821755717933286_wp
        Xi(29) = 15.516222048844071365082675217906_wp
        Xi(30) = 16.789073787817390622650881581990_wp
        Xi(31) = 17.942660053233430389116320746723_wp
        Xi(32) = 18.901622259349553395457732563045_wp
        Xi(33) = 19.591956786301745608094848627445_wp
        Xi(34) = 19.953925505681302989685610427273_wp

      case (5)
        ! ----- K = 17 , R = 3E1 -----

        RIni = 3.0e1_wp

        Omega(1) = 0.0221689954347680469674537837715400101501_wp
        Omega(2) = 0.0525136316982886899500462558193492057512_wp
        Omega(3) = 0.0852024918822239950270025672651730275220_wp
        Omega(4) = 0.1218775877465483974658195381879366436806_wp
        Omega(5) = 0.1647575454691382380742687221997222479786_wp
        Omega(6) = 0.2168726754863256642955884542778299817201_wp
        Omega(7) = 0.2823530993770712014016236596214426413098_wp
        Omega(8) = 0.3667057338149049161731257012730722522065_wp
        Omega(9) = 0.4770978458889288356561779458342442694629_wp
        Omega(10) = 0.6227933403263984189356279597138410508705_wp
        Omega(11) = 0.8160260402127402867920175077998123924772_wp
        Omega(12) = 1.0737693823036798862393531583947492435982_wp
        Omega(13) = 1.4213969799562429773562635682004895443242_wp
        Omega(14) = 1.9010620007560267858684119324230721304048_wp
        Omega(15) = 2.5944356141359594832267870812003707214899_wp
        Omega(16) = 3.7031544721692649809428876661598906139261_wp
        Omega(17) = 6.0263079220428109991809784151683970776503_wp

        Alpha(1) = 0.0086209575744671239913921589881340423744_wp
        Alpha(2) = 0.0458206720026962791509413542440087852015_wp
        Alpha(3) = 0.1144226474825498924414854048925427321137_wp
        Alpha(4) = 0.2175483194524849357882889666626624602941_wp
        Alpha(5) = 0.3602371351111766857621462184502192371838_wp
        Alpha(6) = 0.5501295551911949390714966223736581696357_wp
        Alpha(7) = 0.7984222655089383742250258579886690313288_wp
        Alpha(8) = 1.1211082157416934496768634588192981027532_wp
        Alpha(9) = 1.5404908164489909015368307443338835582836_wp
        Alpha(10) = 2.0870398051392735324152671871900111000286_wp
        Alpha(11) = 2.8018727807909891906572308339917753983173_wp
        Alpha(12) = 3.7405073580248508196156170813040375833225_wp
        Alpha(13) = 4.9792014542644435345572495599242301977938_wp
        Alpha(14) = 6.6269271013874555863523707621709490922512_wp
        Alpha(15) = 8.8514862857485870194923505138717700901907_wp
        Alpha(16) = 11.9500387904471270811984640047853645228315_wp
        Alpha(17) = 16.6290535199854737982705366050595330307260_wp

        Xi(1) = 1.002291568893694707334182658620_wp
        Xi(2) = 1.020685218992778364949583491672_wp
        Xi(3) = 1.057827576467378746490413665704_wp
        Xi(4) = 1.114413507840623477089178616772_wp
        Xi(5) = 1.191528289563124709841621107209_wp
        Xi(6) = 1.290624288731503517633353306238_wp
        Xi(7) = 1.413584180873620381587926764233_wp
        Xi(8) = 1.562739803779250570992670477111_wp
        Xi(9) = 1.740949844069554654397979920333_wp
        Xi(10) = 1.951622386802114737921684528565_wp
        Xi(11) = 2.198829269235813942259469278007_wp
        Xi(12) = 2.487340809060435524880522395463_wp
        Xi(13) = 2.822746253318904027405575329901_wp
        Xi(14) = 3.211534714095589648817988259744_wp
        Xi(15) = 3.661191678398682642367814588091_wp
        Xi(16) = 4.180297123045072366521318629751_wp
        Xi(17) = 4.778612938301576774490636578818_wp
        Xi(18) = 5.467118211613049514989437493995_wp
        Xi(19) = 6.258040356889422471675654691126_wp
        Xi(20) = 7.164671300680670553828049973433_wp
        Xi(21) = 8.201177916486034297043061780386_wp
        Xi(22) = 9.381898130731370829930859667201_wp
        Xi(23) = 10.720507977526170022460061392877_wp
        Xi(24) = 12.228219924802319389968430218829_wp
        Xi(25) = 13.911459749000607544026919537572_wp
        Xi(26) = 15.768042709301844005176373997301_wp
        Xi(27) = 17.782453404170050157914739763498_wp
        Xi(28) = 19.919486830047647271321742401540_wp
        Xi(29) = 22.117910203168493023240248618322_wp
        Xi(30) = 24.285100625197177140815463225465_wp
        Xi(31) = 26.296600456185529319508287215967_wp
        Xi(32) = 28.003802500104567398467314554011_wp
        Xi(33) = 29.253240477906523528406346201791_wp
        Xi(34) = 29.915367900093881785988925159359_wp

      case (6)
        ! ----- K = 17 , R = 4E1 -----

        RIni = 4.0e1_wp

        Omega(1) = 0.0177211717809415518640002830744184514344_wp
        Omega(2) = 0.0420804499893616642852874065727775132473_wp
        Omega(3) = 0.0685892521116945304979141580237200770398_wp
        Omega(4) = 0.0988069670354228814801726861472008778264_wp
        Omega(5) = 0.1348877148729152561400842363914520660728_wp
        Omega(6) = 0.1798302894659236093774208956053151098331_wp
        Omega(7) = 0.2377570966773244348510746651315095334667_wp
        Omega(8) = 0.3141621682693044888387470109369914439412_wp
        Omega(9) = 0.4161787636643034244527360188126152706900_wp
        Omega(10) = 0.5530505032238192121801616252252387084809_wp
        Omega(11) = 0.7370736963599462467392961495527003989991_wp
        Omega(12) = 0.9854093744573996703451854162203460418823_wp
        Omega(13) = 1.3237121967377334781591421974944466910529_wp
        Omega(14) = 1.7943741588876145828801433879107207758352_wp
        Omega(15) = 2.4789350252740321232762521486137075044098_wp
        Omega(16) = 3.5775455854296866070336086140812881239981_wp
        Omega(17) = 5.8808761560584260018845481621241333414218_wp

        Alpha(1) = 0.0068893777628398834758633286282081265028_wp
        Alpha(2) = 0.0366619080357459382424421129309521916184_wp
        Alpha(3) = 0.0917603715074955805243960005324055373421_wp
        Alpha(4) = 0.1750703781562064611029708519063241567437_wp
        Alpha(5) = 0.2913192809176047665088108568065550230131_wp
        Alpha(6) = 0.4477872044599345537009733864142191350766_wp
        Alpha(7) = 0.6552926902022263624925622893524490564232_wp
        Alpha(8) = 0.9294428610717238099423430364165454875547_wp
        Alpha(9) = 1.2921313629933525301152902331480731845659_wp
        Alpha(10) = 1.7733859123770382129527817260949973388051_wp
        Alpha(11) = 2.4138975752788709510602421781655380073062_wp
        Alpha(12) = 3.2688815574094914609068474042885554808890_wp
        Alpha(13) = 4.4145341217529551320307434902900922679692_wp
        Alpha(14) = 5.9600424918313773198566551947408242995152_wp
        Alpha(15) = 8.0735188590298359763430813273998865042813_wp
        Alpha(16) = 11.0517695651505124913860367286133623565547_wp
        Alpha(17) = 15.5970252200462439961733962512369089381536_wp

        Xi(1) = 1.002545391012707700273062783047_wp
        Xi(2) = 1.022989700653836702077051357129_wp
        Xi(3) = 1.064312733992968879720951436152_wp
        Xi(4) = 1.127389942221722018284024713353_wp
        Xi(5) = 1.213557477075983313937954100048_wp
        Xi(6) = 1.324639555837989256551964767361_wp
        Xi(7) = 1.462990572970611438197510723036_wp
        Xi(8) = 1.631561119558584901771160213801_wp
        Xi(9) = 1.833961456632768107833687809549_wp
        Xi(10) = 2.074562680292959938746846781932_wp
        Xi(11) = 2.358595575819610036267387487285_wp
        Xi(12) = 2.692272446945528910008937906184_wp
        Xi(13) = 3.082923335884153877208685434752_wp
        Xi(14) = 3.539158400714016342253689928476_wp
        Xi(15) = 4.071029753543150548570489499767_wp
        Xi(16) = 4.690223124743389164839624250192_wp
        Xi(17) = 5.410239559896043670846055295698_wp
        Xi(18) = 6.246568023375087223749563269948_wp
        Xi(19) = 7.216801269083062290998764609640_wp
        Xi(20) = 8.340650732776270932442019834241_wp
        Xi(21) = 9.639766686236018726485763608025_wp
        Xi(22) = 11.137225750524117250973843429307_wp
        Xi(23) = 12.856445313725064712000178968054_wp
        Xi(24) = 14.819256411371155610079419773228_wp
        Xi(25) = 17.042638573235718358819812401350_wp
        Xi(26) = 19.533631219514500466927220401203_wp
        Xi(27) = 22.281835512529508855003590461763_wp
        Xi(28) = 25.249330361463563417168121105760_wp
        Xi(29) = 28.358601719426040955324141989990_wp
        Xi(30) = 31.481047768266854563340428896367_wp
        Xi(31) = 34.431395403085794500175342847115_wp
        Xi(32) = 36.976004843928678710723856326581_wp
        Xi(33) = 38.862502856911729828826018717791_wp
        Xi(34) = 39.870707818741036188719961685933_wp

      case (7)
        ! ----- K = 17 , R = 5E1 -----

        RIni = 5.0e1_wp

        Omega(1) = 0.0148686660388008807532681768322213589073_wp
        Omega(2) = 0.0353795804560981431076025719578703743196_wp
        Omega(3) = 0.0578913354783317874690097646489528315783_wp
        Omega(4) = 0.0838960166520241401523667879036594285935_wp
        Omega(5) = 0.1154909243595618800322876416897255325011_wp
        Omega(6) = 0.1556347058668119319956545110184009672594_wp
        Omega(7) = 0.2084132074967702798830348001613010922028_wp
        Omega(8) = 0.2792663830727312734858187176500976534044_wp
        Omega(9) = 0.3752605918790274696684745175945252526617_wp
        Omega(10) = 0.5056075720898941724132727437712730989006_wp
        Omega(11) = 0.6826650143959483072971829165087598312311_wp
        Omega(12) = 0.9237725888705498888692907444308843878389_wp
        Omega(13) = 1.2548501525981011695464947819367296233395_wp
        Omega(14) = 1.7185396923469535971663868512138151345425_wp
        Omega(15) = 2.3963756361848144599020066225492087141902_wp
        Omega(16) = 3.4875378011820311698922003618150711190538_wp
        Omega(17) = 5.7767967602680691920416511386804359062808_wp

        Alpha(1) = 0.0057790612680969588938700127415254437224_wp
        Alpha(2) = 0.0307848490694564320059887834943324058656_wp
        Alpha(3) = 0.0771991247235623547496272990608989772454_wp
        Alpha(4) = 0.1477253347274733382305531362299610975697_wp
        Alpha(5) = 0.2468449877068841172305822470600666918017_wp
        Alpha(6) = 0.3815442037677632649477019871619987156919_wp
        Alpha(7) = 0.5623109577640680386439488158689670171952_wp
        Alpha(8) = 0.8043777658932390478303126046810689331323_wp
        Alpha(9) = 1.1292008564500714781243806417165842503891_wp
        Alpha(10) = 1.5663159266865059626207268284048268469633_wp
        Alpha(11) = 2.1559328328586830575506333440571893333981_wp
        Alpha(12) = 2.9529057092101003143859438981877474361681_wp
        Alpha(13) = 4.0332967734518776884745927535647069817060_wp
        Alpha(14) = 5.5064292403364440953682235058153082718491_wp
        Alpha(15) = 7.5407037530380250545287978969355435765465_wp
        Alpha(16) = 10.4328308991523239354290431002425521000987_wp
        Alpha(17) = 14.8822667559735207229790199257024596590782_wp

        Xi(1) = 1.002747516529837833957478077096_wp
        Xi(2) = 1.024822470846904157604911889123_wp
        Xi(3) = 1.069480378717986563189001891772_wp
        Xi(4) = 1.137747078914491229606649469464_wp
        Xi(5) = 1.231188066139299916012389557451_wp
        Xi(6) = 1.351946282217304978420270578088_wp
        Xi(7) = 1.502798888520101475369319943276_wp
        Xi(8) = 1.687235795879166483216861360805_wp
        Xi(9) = 1.909553499731967626589987374608_wp
        Xi(10) = 2.174978286234729048720035193831_wp
        Xi(11) = 2.489802479293736500644967923890_wp
        Xi(12) = 2.861547482390371150220986784518_wp
        Xi(13) = 3.299155511449299699481760295150_wp
        Xi(14) = 3.813203198815979375144863405822_wp
        Xi(15) = 4.416157890949495515124645361738_wp
        Xi(16) = 5.122646892908169101530507782982_wp
        Xi(17) = 5.949761203893470298213058677916_wp
        Xi(18) = 6.917353421690117099788347454492_wp
        Xi(19) = 8.048309539958687430956374431190_wp
        Xi(20) = 9.368725948256321439319704857951_wp
        Xi(21) = 10.907896603463473500852942699169_wp
        Xi(22) = 12.697927722935548186175558349476_wp
        Xi(23) = 14.772704548227969807705495242089_wp
        Xi(24) = 17.165788871667563860942684605959_wp
        Xi(25) = 19.906596463527166234255827426125_wp
        Xi(26) = 23.014021556751030934614976608543_wp
        Xi(27) = 26.486506433188673752254782556292_wp
        Xi(28) = 30.287761291592585566334605573502_wp
        Xi(29) = 34.328413013627400472010942067413_wp
        Xi(30) = 38.446466616132194189792148364404_wp
        Xi(31) = 42.394153189670192732191322804169_wp
        Xi(32) = 45.844150908129016632613916826244_wp
        Xi(33) = 48.429519596735885651056507938961_wp
        Xi(34) = 49.821049698729165840227484096658_wp

      case (8)
        ! ----- K = 17 , R = 6E1 -----

        RIni = 6.0e1_wp

        Omega(1) = 0.0128688641747443431470488014144171184938_wp
        Omega(2) = 0.0306759431049708882359357839733626782674_wp
        Omega(3) = 0.0503652577739043404976375644527264441308_wp
        Omega(4) = 0.0733727538326730014271097150835299061100_wp
        Omega(5) = 0.1017455651731470538470542691600329021639_wp
        Omega(6) = 0.1383996020235092485434983539116515771639_wp
        Omega(7) = 0.1873716447221103687041350929765037847119_wp
        Omega(8) = 0.2540325726980336789554305171323633771863_wp
        Omega(9) = 0.3453754544025244804892216948788430386230_wp
        Omega(10) = 0.4705821147337007801468838213976297879526_wp
        Omega(11) = 0.6420685909054469359276938467129269838551_wp
        Omega(12) = 0.8773350492032141751536089557816922024358_wp
        Omega(13) = 1.2025388662231339082864592371358014588623_wp
        Omega(14) = 1.6605588690218929312633275308108693479880_wp
        Omega(15) = 2.3329791005813414442489339695185890377616_wp
        Omega(16) = 3.4182913257221446416862786765733517313492_wp
        Omega(17) = 5.6968099048043257253033655462814976999653_wp

        Alpha(1) = 0.0050007674880163409831774297322620759232_wp
        Alpha(2) = 0.0266626502692728805931916390842273401063_wp
        Alpha(3) = 0.0669742870912714411999280307574711201823_wp
        Alpha(4) = 0.1284926354669767219635620816853283088221_wp
        Alpha(5) = 0.2154985496957080209131252940268907991594_wp
        Alpha(6) = 0.3347316344380094074533425752004944797591_wp
        Alpha(7) = 0.4963890092080094549025631733840668857738_wp
        Alpha(8) = 0.7153535743415229191874415981544643727830_wp
        Alpha(9) = 1.0126577716550769446817586993958570928953_wp
        Alpha(10) = 1.4173574351365213142483687192019203848758_wp
        Alpha(11) = 1.9691940995077596654678001031335554671386_wp
        Alpha(12) = 2.7226634022245192986218936814779567612277_wp
        Alpha(13) = 3.7536719703944716284172572473565310247068_wp
        Alpha(14) = 5.1716341738480789531028081196240009376197_wp
        Alpha(15) = 7.1452085392034963683832726744071806024294_wp
        Alpha(16) = 9.9711244754619540453521864264985197223723_wp
        Alpha(17) = 14.3468619350247799807199222676956651412183_wp

        Xi(1) = 1.002915499109157203240158540147_wp
        Xi(2) = 1.026346630989126865705907132309_wp
        Xi(3) = 1.073782732619288837945005499197_wp
        Xi(4) = 1.146383008420414932799526186447_wp
        Xi(5) = 1.245918789443741162893154739155_wp
        Xi(6) = 1.374819018650134055969415369791_wp
        Xi(7) = 1.536240708054906436551972914728_wp
        Xi(8) = 1.734161982673499102910671942901_wp
        Xi(9) = 1.973503827098548688815000951013_wp
        Xi(10) = 2.260274464933970315228678460429_wp
        Xi(11) = 2.601744180092282813371307170769_wp
        Xi(12) = 3.006647453358342283145920870169_wp
        Xi(13) = 3.485425441718501938766563985617_wp
        Xi(14) = 4.050507024804020406432564227472_wp
        Xi(15) = 4.716636777895760899905613916516_wp
        Xi(16) = 5.501248259550373111061855047055_wp
        Xi(17) = 6.424879686084404700145006339440_wp
        Xi(18) = 7.511615350987158668590731869230_wp
        Xi(19) = 8.789522694599506495369278802343_wp
        Xi(20) = 10.291020111543749447104156846677_wp
        Xi(21) = 12.053065680014858375435071735637_wp
        Xi(22) = 14.116964761606295349419259999024_wp
        Xi(23) = 16.527467776378566692921379122083_wp
        Xi(24) = 19.330626819368343607963289798590_wp
        Xi(25) = 22.569577309641494939143613329691_wp
        Xi(26) = 26.277070914326466520224978573594_wp
        Xi(27) = 30.463249836486540004371059708888_wp
        Xi(28) = 35.097156787527594493797389318956_wp
        Xi(29) = 40.081587760813652453195654601359_wp
        Xi(30) = 45.224179044115468292103221514822_wp
        Xi(31) = 50.214331710936857079696782335532_wp
        Xi(32) = 54.624695735134070985472565240570_wp
        Xi(33) = 57.960537309272782375418886502416_wp
        Xi(34) = 59.767106150282154120034761035640_wp

      case (9)
        ! ----- K = 17 , R = 7E1 -----

        RIni = 7.0e1_wp

        Omega(1) = 0.0113817599792177474883739024189011068700_wp
        Omega(2) = 0.0271743593440448312882964931880280801124_wp
        Omega(3) = 0.0447515467109071490308292058657446688130_wp
        Omega(4) = 0.0655014490394985475404379324237957860078_wp
        Omega(5) = 0.0914265035459931564443681988041401353939_wp
        Omega(6) = 0.1253997678698674680237111489500279049025_wp
        Omega(7) = 0.1714039191196946684581396344593251512833_wp
        Omega(8) = 0.2347367696287157425250264122795762489204_wp
        Omega(9) = 0.3223205037789023950413076075038887324808_wp
        Omega(10) = 0.4433112013591309012565128827576543812938_wp
        Omega(11) = 0.6101784217262353503577997082540917972437_wp
        Omega(12) = 0.8405640412966286443554805862810042071942_wp
        Omega(13) = 1.1608360910097696578864925887053516362357_wp
        Omega(14) = 1.6140922256599763499581093317836177902791_wp
        Omega(15) = 2.2819924145495760149562197716122113888559_wp
        Omega(16) = 3.3625148353540114170776020863229405222228_wp
        Omega(17) = 5.6324406740100084454471451866908182637417_wp

        Alpha(1) = 0.0044220826490264052288574705452950386331_wp
        Alpha(2) = 0.0235959842136256114419402048903995350315_wp
        Alpha(3) = 0.0593600889932688150471323720175353955142_wp
        Alpha(4) = 0.1141499987320069049107963167488666300642_wp
        Alpha(5) = 0.1920783153601949629779317013644757139446_wp
        Alpha(6) = 0.2996733309241805377580707830942330360813_wp
        Alpha(7) = 0.4468736581015504913947430226661694518953_wp
        Alpha(8) = 0.6482397686822209835390967702650755200011_wp
        Alpha(9) = 0.9244066057150568037304935731235389084759_wp
        Alpha(10) = 1.3039805390235891479049773478138263271831_wp
        Alpha(11) = 1.8262656613661149404472666679666303934937_wp
        Alpha(12) = 2.5454181581362931795740889295842634965084_wp
        Alpha(13) = 3.5371843477475413788143482340942114205973_wp
        Alpha(14) = 4.9110406570491912229582676108208261211985_wp
        Alpha(15) = 6.8358741981674408794145136880615609697998_wp
        Alpha(16) = 9.6084889100046543333744075710001197876409_wp
        Alpha(17) = 13.9248776999019524772821521096943797601853_wp

        Xi(1) = 1.003059306218734375798842806127_wp
        Xi(2) = 1.027652007831690134770884215154_wp
        Xi(3) = 1.077470832132708706109849861043_wp
        Xi(4) = 1.153795992652397408304368320309_wp
        Xi(5) = 1.258585453713018532614399680458_wp
        Xi(6) = 1.394528608539600317045743793631_wp
        Xi(7) = 1.565128237977414441656730248464_wp
        Xi(8) = 1.774811888247491756297538612053_wp
        Xi(9) = 2.029073788523009899860405358396_wp
        Xi(10) = 2.334647524299378020017428636557_wp
        Xi(11) = 2.699711470502097960741383353422_wp
        Xi(12) = 3.134134981167966560499163475306_wp
        Xi(13) = 3.649768331014000715285111597908_wp
        Xi(14) = 4.260789544679054438418297490188_wp
        Xi(15) = 4.984111036106225256676893753394_wp
        Xi(16) = 5.839850856509280158622432477245_wp
        Xi(17) = 6.851869206681649118566290734833_wp
        Xi(18) = 8.048356757322034505178209862208_wp
        Xi(19) = 9.462448801068054722064604167286_wp
        Xi(20) = 11.132800568353531299632674933520_wp
        Xi(21) = 13.104007107330691340313366666237_wp
        Xi(22) = 15.426648606116144035911574849962_wp
        Xi(23) = 18.156587826522455474120842922048_wp
        Xi(24) = 21.352892433095730553849200816785_wp
        Xi(25) = 25.073360231710072847446935107030_wp
        Xi(26) = 29.366141741389968937078758237647_wp
        Xi(27) = 34.255394316438924642853613988791_wp
        Xi(28) = 39.718638441445942799390422806027_wp
        Xi(29) = 45.654496890002611110548347284066_wp
        Xi(30) = 51.843356484674942806500919090240_wp
        Xi(31) = 57.912297647083303200887804962349_wp
        Xi(32) = 63.329143135346385548589687175536_wp
        Xi(33) = 67.459959939941881126856948469594_wp
        Xi(34) = 69.709379552572190680281849495259_wp

      case (10)
        ! ----- K = 17 , R = 8E1 -----

        RIni = 8.0e1_wp

        Omega(1) = 0.0102284904090335036154122382023867743328_wp
        Omega(2) = 0.0244561615921032835244597249818587769710_wp
        Omega(3) = 0.0403860818642373749303667741089585163650_wp
        Omega(4) = 0.0593649342842641226417022774081155844783_wp
        Omega(5) = 0.0833549435940638370153336032808422118023_wp
        Omega(6) = 0.1151872860033455609065451349937170277826_wp
        Omega(7) = 0.1587891810315065413005027050719064618534_wp
        Omega(8) = 0.2193862206997807445173166371321649137371_wp
        Omega(9) = 0.3038348393048518613376949432103302228825_wp
        Omega(10) = 0.4212691673947276489384239361868367268471_wp
        Omega(11) = 0.5842068690607538775159783550616765523955_wp
        Omega(12) = 0.8104146081742157019375469295674463410251_wp
        Omega(13) = 1.1264476636364262366555438754645024346246_wp
        Omega(14) = 1.5756047826778999358795715912684443082981_wp
        Omega(15) = 2.2396347826394381254518084478632999889669_wp
        Omega(16) = 3.3161182846423878898840104101353176702105_wp
        Omega(17) = 5.5789379346348069691093052302477417470072_wp

        Alpha(1) = 0.0039733564241690171219145038573883343958_wp
        Alpha(2) = 0.0212168432499748764094165158142213378767_wp
        Alpha(3) = 0.0534476838381714172498222412133195291517_wp
        Alpha(4) = 0.1029986320484459273082384136999145596292_wp
        Alpha(5) = 0.1738382188340671212229554543071685657196_wp
        Alpha(6) = 0.2723103794523001048417075453667024476090_wp
        Alpha(7) = 0.4081217168328174226845077993131027938034_wp
        Alpha(8) = 0.5955361059459275052869499378349615881234_wp
        Alpha(9) = 0.8548199579708161325143980779905916733696_wp
        Alpha(10) = 1.2141643235424078605712294720575528117479_wp
        Alpha(11) = 1.7124696860301575545649024889804934446147_wp
        Alpha(12) = 2.4035753152187838668947528208974517838215_wp
        Alpha(13) = 3.3630688929233138084913534715525429419358_wp
        Alpha(14) = 4.7004686318565189542817428502985421800986_wp
        Alpha(15) = 6.5848644957529141393155758255062437456218_wp
        Alpha(16) = 9.3131630039612661184703079619850996095920_wp
        Alpha(17) = 13.5801936423355342027294301310291757545201_wp

        Xi(1) = 1.003185049342236287504663561077_wp
        Xi(2) = 1.028793990287334054328782784182_wp
        Xi(3) = 1.080699748461550579949237704813_wp
        Xi(4) = 1.160293481503308493326710737481_wp
        Xi(5) = 1.269704561755699912472226953408_wp
        Xi(6) = 1.411861685687981412011191006961_wp
        Xi(7) = 1.590586957135407310866964669671_wp
        Xi(8) = 1.810723906681402168216896142461_wp
        Xi(9) = 2.078299602443752740980192905695_wp
        Xi(10) = 2.400724593438739941321166670996_wp
        Xi(11) = 2.787029119941703737437996291071_wp
        Xi(12) = 3.248150538939510043046077214512_wp
        Xi(13) = 3.797272462402132554371361750079_wp
        Xi(14) = 4.450236221724152312160388555995_wp
        Xi(15) = 5.226024730023481232668580709344_wp
        Xi(16) = 6.147335538281357032123347616803_wp
        Xi(17) = 7.241237963009023085800275332247_wp
        Xi(18) = 8.539913938899198962186043448952_wp
        Xi(19) = 10.081456310334636464025315216730_wp
        Xi(20) = 11.910664040008737896841894965405_wp
        Xi(21) = 14.079717653115598969063015921677_wp
        Xi(22) = 16.648495244969107535737418146482_wp
        Xi(23) = 19.684131952622105739963975779006_wp
        Xi(24) = 23.259078536592308302996534585816_wp
        Xi(25) = 27.446514278793275869172685332842_wp
        Xi(26) = 32.311219585524162591777530550985_wp
        Xi(27) = 37.893276819644083831112313376366_wp
        Xi(28) = 44.181388382686948908262269952729_wp
        Xi(29) = 51.073333552978532983573911607778_wp
        Xi(30) = 58.325346325847006980630515826647_wp
        Xi(31) = 65.503209781983478085520822276067_wp
        Xi(32) = 71.966253063518017804578441598551_wp
        Xi(33) = 76.931402511047548453682054514502_wp
        Xi(34) = 79.647416004628414316901530156656_wp

      case (11)
        ! ----- K = 17 , R = 9E1 -----

        RIni = 9.0e1_wp

        Omega(1) = 0.0093055452799273397626321252702752406805_wp
        Omega(2) = 0.0222788871640996847180458744971942586233_wp
        Omega(3) = 0.0368837610336946389418195154513702149757_wp
        Omega(4) = 0.0544304246766022368402351827576524101460_wp
        Omega(5) = 0.0768446068104137880862107568002983271072_wp
        Omega(6) = 0.1069170503299141731774844071845009807475_wp
        Omega(7) = 0.1485200449520121921163440595337057459346_wp
        Omega(8) = 0.2068099190615291706755096354242162703940_wp
        Omega(9) = 0.2885830163742097357704001481248923255407_wp
        Omega(10) = 0.4029545064879059731695679080765160051669_wp
        Omega(11) = 0.5624848391441751290571911259252857462343_wp
        Omega(12) = 0.7850508763672496834792186026774629681313_wp
        Omega(13) = 1.0973753054331811402624316520437730559934_wp
        Omega(14) = 1.5429418443905634444977131858323105007003_wp
        Omega(15) = 2.2035942157384123282846261071554749833012_wp
        Omega(16) = 3.2765969845607466565205051489684251464496_wp
        Omega(17) = 5.5333942237095084396361621692506105318898_wp

        Alpha(1) = 0.0036142847601028185659230223744642351846_wp
        Alpha(2) = 0.0193121979363516552036001035219658916731_wp
        Alpha(3) = 0.0487106256560613163639824208656081694357_wp
        Alpha(4) = 0.0940536129957006666649200164742605778656_wp
        Alpha(5) = 0.1591842932859885571041666865732544522416_wp
        Alpha(6) = 0.2502835048715152584130651247651400126415_wp
        Alpha(7) = 0.3768478515919123995111129216128986740841_wp
        Alpha(8) = 0.5528681138914166540328232191114921079134_wp
        Alpha(9) = 0.7982702270194615650324501676671218319825_wp
        Alpha(10) = 1.1408628740479669352262531711872384221351_wp
        Alpha(11) = 1.6191750225339964947305834708224381301989_wp
        Alpha(12) = 2.2867510241830227718257823044112342358858_wp
        Alpha(13) = 3.2190241844884033011083002140395592505229_wp
        Alpha(14) = 4.5255397708238691554562260765948167318129_wp
        Alpha(15) = 6.3755685758961003579622928150172356254188_wp
        Alpha(16) = 9.0661336228983590247373913761919084208785_wp
        Alpha(17) = 13.2911275294674962061530765211614379950333_wp

        Xi(1) = 1.003296779115244159925370015518_wp
        Xi(2) = 1.029808878893360909144438775087_wp
        Xi(3) = 1.083571537207537143713415028756_wp
        Xi(4) = 1.166078406934612099118964145816_wp
        Xi(5) = 1.279617310379891032901325531590_wp
        Xi(6) = 1.427339244833476762154035599739_wp
        Xi(7) = 1.613363210094094657583684016000_wp
        Xi(8) = 1.842920671002319024460967822332_wp
        Xi(9) = 2.122538146114561116153604536372_wp
        Xi(10) = 2.460261767966067304952773597826_wp
        Xi(11) = 2.865926179907826641841961379242_wp
        Xi(12) = 3.351478973975718224625336416622_wp
        Xi(13) = 3.931372321603539599131973969470_wp
        Xi(14) = 4.623035084321579500730298173394_wp
        Xi(15) = 5.447437774494031566870771987610_wp
        Xi(16) = 6.429762708165225281592425332633_wp
        Xi(17) = 7.600186972482000267829771456007_wp
        Xi(18) = 8.994777496566889835212521564500_wp
        Xi(19) = 10.656479258892964868260522237620_wp
        Xi(20) = 12.636141833151536209220444639811_wp
        Xi(21) = 14.993463705687141253493355197435_wp
        Xi(22) = 17.797615773195354276284274597231_wp
        Xi(23) = 21.127093376418113003512022629593_wp
        Xi(24) = 25.068012988698169147411265456071_wp
        Xi(25) = 29.709490481022010189768312038439_wp
        Xi(26) = 35.133926553450054650062917005471_wp
        Xi(27) = 41.398943821547749412348427000552_wp
        Xi(28) = 48.506794614204075954955186489315_wp
        Xi(29) = 56.357422262427981987653602402588_wp
        Xi(30) = 64.685922339974741085422138553440_wp
        Xi(31) = 72.998129197201843718079139478050_wp
        Xi(32) = 80.542045616521516282482195947523_wp
        Xi(33) = 86.376471319539266387199649699369_wp
        Xi(34) = 89.584005199299913783927973298660_wp

      case (12)
        ! ----- K = 17 , R = 1E2 -----

        RIni = 1.0e2_wp

        Omega(1) = 0.0085486027112414137163487210663292970025_wp
        Omega(2) = 0.0204917524282931772402181497955142575051_wp
        Omega(3) = 0.0340047708324275376429266393857320238681_wp
        Omega(4) = 0.0503655153946454480856859473036912788757_wp
        Omega(5) = 0.0714663291839928747361285026051813673575_wp
        Omega(6) = 0.1000591584195262242389448670287777076737_wp
        Omega(7) = 0.1399628625763894228896172186593638286922_wp
        Omega(8) = 0.1962682990482164567346392083924122573535_wp
        Omega(9) = 0.2757171017009835744322916728510364237081_wp
        Omega(10) = 0.3874077253288334146360573362333212799058_wp
        Omega(11) = 0.5439384928815124032198263792636794278224_wp
        Omega(12) = 0.7632840773608892314056398320509799759748_wp
        Omega(13) = 1.0723178584501672561731661081907418520132_wp
        Omega(14) = 1.5146942602000858187939746901662374511943_wp
        Omega(15) = 2.1723542263902079296842423383750997345487_wp
        Omega(16) = 3.2423058243498171153255948961913190942141_wp
        Omega(17) = 5.4939008854835959206816542454987484234152_wp

        Alpha(1) = 0.0033198247340284218243658512269636529091_wp
        Alpha(2) = 0.0177496313316968076455022815945961056627_wp
        Alpha(3) = 0.0448214559364190146659623521785409394624_wp
        Alpha(4) = 0.0867017057150739512905103685558305670611_wp
        Alpha(5) = 0.1471228721624464812976198699323582275156_wp
        Alpha(6) = 0.2321198066274996981334143367203814989352_wp
        Alpha(7) = 0.3509975895016110316833920362844523310741_wp
        Alpha(8) = 0.5174948868641361680777851494994479253364_wp
        Alpha(9) = 0.7512229986159361620886784627337817710213_wp
        Alpha(10) = 1.0796383396633986272031627273726428484224_wp
        Alpha(11) = 1.5409273193095968162644182619658295152476_wp
        Alpha(12) = 2.1883595496344136345067549198617484762508_wp
        Alpha(13) = 3.0972191923966147315740865542110782371310_wp
        Alpha(14) = 4.3770675431933018500324505950516140728723_wp
        Alpha(15) = 6.1973374899988810981035780578451976907672_wp
        Alpha(16) = 8.8551748034756879732545820083089438412571_wp
        Alpha(17) = 13.0437004598485047808520409517996085924096_wp

        Xi(1) = 1.003397304594680938902989675299_wp
        Xi(2) = 1.030722356774342372267780354456_wp
        Xi(3) = 1.086157905417406153926636325213_wp
        Xi(4) = 1.171293105508434075306793686888_wp
        Xi(5) = 1.288563641148398332533439591963_wp
        Xi(6) = 1.441327846477146027641229319993_wp
        Xi(7) = 1.633983056430369042058385609106_wp
        Xi(8) = 1.872124856731489036393289993043_wp
        Xi(9) = 2.162750234354115420063249808358_wp
        Xi(10) = 2.514506306061629360530662591344_wp
        Xi(11) = 2.937989383136705290235773935237_wp
        Xi(12) = 3.446109522244970960495533129908_wp
        Xi(13) = 4.054529740004681303677108727967_wp
        Xi(14) = 4.782200741792271876315356671228_wp
        Xi(15) = 5.652005271429134055077153631075_wp
        Xi(16) = 6.691527239140495126549362803203_wp
        Xi(17) = 7.933959967695566292342213010613_wp
        Xi(18) = 9.419157721786756060078360519583_wp
        Xi(19) = 11.194818659897883121080719437401_wp
        Xi(20) = 13.317749785224979691862501329780_wp
        Xi(21) = 15.855096568877923896359605215878_wp
        Xi(22) = 18.885291875017808239933803093891_wp
        Xi(23) = 22.498252544423162078632283389368_wp
        Xi(24) = 26.793960055960475894615280267885_wp
        Xi(25) = 31.877911764722284731002344138062_wp
        Xi(26) = 37.850957461647661145276444472074_wp
        Xi(27) = 44.789632960802394250066083003503_wp
        Xi(28) = 52.711817335957233870979976586568_wp
        Xi(29) = 61.522385107045003675707572909914_wp
        Xi(30) = 70.938197102818303187155013489473_wp
        Xi(31) = 80.406727242843933162186331742305_wp
        Xi(32) = 89.062467800429165006026455131405_wp
        Xi(33) = 95.798208021227316426304820851101_wp
        Xi(34) = 99.516891069178648451809543473701_wp

      case (13)
        ! ----- K = 17 , R = 2E2 -----

        RIni = 2.0e2_wp

        Omega(1) = 0.0048673410480669797478294228646605867894_wp
        Omega(2) = 0.0117746330456780427921052094124287901167_wp
        Omega(3) = 0.0198870791428627719878467153963308433617_wp
        Omega(4) = 0.0302776004275052840479694193920323641578_wp
        Omega(5) = 0.0446058663311346083678147773216027260901_wp
        Omega(6) = 0.0653179331698807654382928078451442388541_wp
        Omega(7) = 0.0958124822046568227828589395245195703410_wp
        Omega(8) = 0.1407135799336433006312104335155943601876_wp
        Omega(9) = 0.2064043244858754213232835544866716759316_wp
        Omega(10) = 0.3018653172872777795817614970852460487549_wp
        Omega(11) = 0.4399011184015803755254902379645542964681_wp
        Omega(12) = 0.6390746417174834501334461378529994135533_wp
        Omega(13) = 0.9272349393169635067444775411438229184569_wp
        Omega(14) = 1.3492433066001499579337000178647087977879_wp
        Omega(15) = 1.9879267827114792103635962372543133369618_wp
        Omega(16) = 3.0391568135442928416493912902041074630688_wp
        Omega(17) = 5.2603872308116383059439780112143125734292_wp

        Alpha(1) = 0.0018882627435518682409133786278578048812_wp
        Alpha(2) = 0.0101417169947529204940798877934704513848_wp
        Alpha(3) = 0.0258347276183185878667663681190758140360_wp
        Alpha(4) = 0.0506687265340891762473672903926980026768_wp
        Alpha(5) = 0.0876936725177892334700592048835066094625_wp
        Alpha(6) = 0.1419979493343611348524571647011782715708_wp
        Alpha(7) = 0.2215771340255664241270525463556317902203_wp
        Alpha(8) = 0.3384059441793975349984519018775586118863_wp
        Alpha(9) = 0.5099057820568959907602327175091261324269_wp
        Alpha(10) = 0.7610963644983120443511179220585916027630_wp
        Alpha(11) = 1.1277658774087937613474219578080237624818_wp
        Alpha(12) = 1.6611792947718912516258046863093511547049_wp
        Alpha(13) = 2.4354020371117624408838220340101088368101_wp
        Alpha(14) = 3.5598923622483053235548977610847032337915_wp
        Alpha(15) = 5.2051002196070909459370279481760235285037_wp
        Alpha(16) = 7.6693061305436458293012524922716011133161_wp
        Alpha(17) = 11.6418874847292572833820645250568759365706_wp

        Xi(1) = 1.004068323206283402568815554812_wp
        Xi(2) = 1.036827203872648411951511160378_wp
        Xi(3) = 1.103482813671758877896965367338_wp
        Xi(4) = 1.206341745306593884090769208672_wp
        Xi(5) = 1.348954822983449604962766232852_wp
        Xi(6) = 1.536255201143464148468009700110_wp
        Xi(7) = 1.774772272621446738815470522965_wp
        Xi(8) = 2.072925566564174586865407223790_wp
        Xi(9) = 2.441400100516846734125323004072_wp
        Xi(10) = 2.893611175249084627690565607594_wp
        Xi(11) = 3.446278483039645467800662914648_wp
        Xi(12) = 4.120142952274416699338932534857_wp
        Xi(13) = 4.940862870667321987173947750449_wp
        Xi(14) = 5.940136329222954380841620336540_wp
        Xi(15) = 7.157101804351796531852153426811_wp
        Xi(16) = 8.640080069197455927242079720685_wp
        Xi(17) = 10.448731184678607597157706798896_wp
        Xi(18) = 12.656708312032312059060290554413_wp
        Xi(19) = 15.354889430827064356749334095298_wp
        Xi(20) = 18.655246473177264695392230109405_wp
        Xi(21) = 22.695344424535778063423130390674_wp
        Xi(22) = 27.643302536301633016632695039050_wp
        Xi(23) = 33.702703932991579259537129331648_wp
        Xi(24) = 41.116236408418016124582949899491_wp
        Xi(25) = 50.165481532705346403439028435400_wp
        Xi(26) = 61.161682714892204943551989870798_wp
        Xi(27) = 74.417925209841545380884397786758_wp
        Xi(28) = 90.186015098394796865588496359578_wp
        Xi(29) = 108.532839143091825333253819252377_wp
        Xi(30) = 129.127827549526113151689976632497_wp
        Xi(31) = 150.937426751899529994704884927614_wp
        Xi(32) = 171.918608256554023894224769719585_wp
        Xi(33) = 188.993468008481036918277951031087_wp
        Xi(34) = 198.720074371188938050503125509749_wp

      case (14)
        ! ----- K = 17 , R = 3E2 -----

        RIni = 3.0e2_wp

        Omega(1) = 0.0034895570058752979874977019104920206161_wp
        Omega(2) = 0.0084953771076717173810761269368840853389_wp
        Omega(3) = 0.0145270414019809661616185474282414435976_wp
        Omega(4) = 0.0225468027510196970941968317228565332755_wp
        Omega(5) = 0.0340719403754830647104202574065867636932_wp
        Omega(6) = 0.0513432024006063979310802818101233668813_wp
        Omega(7) = 0.0774918652736211760145260303567216197962_wp
        Omega(8) = 0.1168740336961956888882471945756424958063_wp
        Omega(9) = 0.1756713275412030964462047066820815643950_wp
        Omega(10) = 0.2627731353169696485491246135124399074812_wp
        Omega(11) = 0.3910483078099100866600618814361567388005_wp
        Omega(12) = 0.5793369494164638594719575481750695189476_wp
        Omega(13) = 0.8560194482301063125494200445331216542399_wp
        Omega(14) = 1.2666966821386286369066978552133662105916_wp
        Omega(15) = 1.8948707090313091034932888034880704708485_wp
        Omega(16) = 2.9361280759129321055959233044418965619116_wp
        Omega(17) = 5.1422560459812323464812156004200005554594_wp

        Alpha(1) = 0.0013527924373737048622635492731852702164_wp
        Alpha(2) = 0.0072886986335998001675672063135597023020_wp
        Alpha(3) = 0.0186812974634128143710875636154811196832_wp
        Alpha(4) = 0.0369992735644177996947072010330836988601_wp
        Alpha(5) = 0.0649352701797229282028745683819082046284_wp
        Alpha(6) = 0.1070488859072480633863842791608789184465_wp
        Alpha(7) = 0.1705675762987403684515803209409412488640_wp
        Alpha(8) = 0.2664238485789205659784017327096350413740_wp
        Alpha(9) = 0.4107577410338992014303578781397874308823_wp
        Alpha(10) = 0.6271579934161993041645544177953297548811_wp
        Alpha(11) = 0.9499639456298539084868996773547422662887_wp
        Alpha(12) = 1.4291576240431078479321130858537003405218_wp
        Alpha(13) = 2.1379244969668702457035042163724369856936_wp
        Alpha(14) = 3.1854886763815114252181015519482798481476_wp
        Alpha(15) = 4.7428116771175643337672689359152400356834_wp
        Alpha(16) = 7.1090076869049263646495318003104557647021_wp
        Alpha(17) = 10.9721317820403240841931058469072013394907_wp

        Xi(1) = 1.004464527867190501192636020722_wp
        Xi(2) = 1.040437796402030025942099522585_wp
        Xi(3) = 1.113762374064386074688434713487_wp
        Xi(4) = 1.227234872778009348317194981970_wp
        Xi(5) = 1.385171811777678287005283186417_wp
        Xi(6) = 1.593598093484232945454479035696_wp
        Xi(7) = 1.860540064834858015887196491800_wp
        Xi(8) = 2.196428736571339649253503667481_wp
        Xi(9) = 2.614615919234772979185335572971_wp
        Xi(10) = 3.132018132737269137076674407894_wp
        Xi(11) = 3.769923724965650096204589081239_wp
        Xi(12) = 4.555016764559704660342287496633_wp
        Xi(13) = 5.520677093343362143960939558696_wp
        Xi(14) = 6.708633343311841956951163679079_wp
        Xi(15) = 8.171059739554667077608696157398_wp
        Xi(16) = 9.973232095840790818221111013742_wp
        Xi(17) = 12.196886306702664104495426045105_wp
        Xi(18) = 14.944453145867676252013522830708_wp
        Xi(19) = 18.344366609672621490356725360016_wp
        Xi(20) = 22.557671615395216426283586663715_wp
        Xi(21) = 27.786098499632169645662260748509_wp
        Xi(22) = 34.281663399831593923988704020189_wp
        Xi(23) = 42.357475313742274179024116520509_wp
        Xi(24) = 52.398581781765994624944227098240_wp
        Xi(25) = 64.869836265945337555605121337976_wp
        Xi(26) = 80.313962712675186078614686380206_wp
        Xi(27) = 99.325475447809962840917119564210_wp
        Xi(28) = 122.472696031008108440152337692552_wp
        Xi(29) = 150.118835284941462365759790031916_wp
        Xi(30) = 182.070852861453252896639654068167_wp
        Xi(31) = 216.997814200839051826386061350149_wp
        Xi(32) = 251.709878456813249000822985124159_wp
        Xi(33) = 280.812119844836972720747425569243_wp
        Xi(34) = 297.750869555545535610763252520883_wp

      case (15)
        ! ----- K = 17 , R = 4E2 -----

        RIni = 4.0e2_wp

        Omega(1) = 0.0027525568817549190527555180154836711459_wp
        Omega(2) = 0.0067349258709659411096330717226565720779_wp
        Omega(3) = 0.0116306948133806012482164378082916122992_wp
        Omega(4) = 0.0183283970860098548055533602962885453280_wp
        Omega(5) = 0.0282435886802854051323057125305604486698_wp
        Omega(6) = 0.0434676806844395435869331332373599607877_wp
        Omega(7) = 0.0669468256839370023971434210674491538384_wp
        Omega(8) = 0.1028581020016681330867523193328683106529_wp
        Omega(9) = 0.1572421655776665259268178032048268732979_wp
        Omega(10) = 0.2389084736932060266468083598034866099624_wp
        Omega(11) = 0.3607421430923923736982985438492832486190_wp
        Omega(12) = 0.5417453516089880428139094470374459433515_wp
        Omega(13) = 0.8106492863971687715097212689752126379972_wp
        Omega(14) = 1.2135804900968095800016868790471846750734_wp
        Omega(15) = 1.8345705538017225324620013138243734829302_wp
        Omega(16) = 2.8691445129195687073660192556445736045134_wp
        Omega(17) = 5.0655577565800045543829843719407790558762_wp

        Alpha(1) = 0.0010664804747085546700840461053008678682_wp
        Alpha(2) = 0.0057604517182599801614652213193473739850_wp
        Alpha(3) = 0.0148368170868551522151713821517926561455_wp
        Alpha(4) = 0.0296165628394039036889355889781150210638_wp
        Alpha(5) = 0.0525586001957675487815282826253859838062_wp
        Alpha(6) = 0.0878641811279735020215526410114481592473_wp
        Alpha(7) = 0.1422323164806631419021867857033214477269_wp
        Alpha(8) = 0.2258815020393660163592957842992881012378_wp
        Alpha(9) = 0.3540758486226656344322133440627453637717_wp
        Alpha(10) = 0.5494155210276604147149291657026282109655_wp
        Alpha(11) = 0.8452157720214318415504102521840223971594_wp
        Alpha(12) = 1.2905193491966397324613793906600278660335_wp
        Alpha(13) = 1.9578266197559235022680398929928458073846_wp
        Alpha(14) = 2.9561204345082620304971388058845604973612_wp
        Alpha(15) = 4.4566738162125549430406246109725998394424_wp
        Alpha(16) = 6.7592308752209600986732285221592064772267_wp
        Alpha(17) = 10.5512095381578559893237523326092741626780_wp

        Xi(1) = 1.004745236963215977580091142762_wp
        Xi(2) = 1.042998580297516816742589562672_wp
        Xi(3) = 1.121068111086431744638676477610_wp
        Xi(4) = 1.242128093989788244493228941412_wp
        Xi(5) = 1.411087079820837677005505861327_wp
        Xi(6) = 1.634819396399565815530818824186_wp
        Xi(7) = 1.922526097392019043858142623993_wp
        Xi(8) = 2.286231454190448127280696488661_wp
        Xi(9) = 2.741418827416767944708808624199_wp
        Xi(10) = 3.307830635984425216499990596297_wp
        Xi(11) = 4.010481879731350666161426854472_wp
        Xi(12) = 4.880956652461628126563747409605_wp
        Xi(13) = 5.959074289729046970329195964666_wp
        Xi(14) = 7.295027725893324937675443742080_wp
        Xi(15) = 8.952125467199831437489776497785_wp
        Xi(16) = 11.010306209724394013727000185554_wp
        Xi(17) = 13.570642194880133932916033145943_wp
        Xi(18) = 16.761101456619901009861828278247_wp
        Xi(19) = 20.743903594996919827544989978207_wp
        Xi(20) = 25.724864105597058641997088024311_wp
        Xi(21) = 31.965130803874801450684128845126_wp
        Xi(22) = 39.795659506903283832157658395090_wp
        Xi(23) = 49.634435468996615056780363772759_wp
        Xi(24) = 62.005576615130705262629584595402_wp
        Xi(25) = 77.557292073570342674782018832502_wp
        Xi(26) = 97.070850150577827536446662293201_wp
        Xi(27) = 121.442787348079176142079393940776_wp
        Xi(28) = 151.602315904855365624737117968834_wp
        Xi(29) = 188.291057731508376715812147494944_wp
        Xi(30) = 231.583575803770799733660723518369_wp
        Xi(31) = 280.010071452744729342265728178063_wp
        Xi(32) = 329.320421846918235664336549461950_wp
        Xi(33) = 371.614835336650621178122833043744_wp
        Xi(34) = 396.651647865055578978665806744175_wp

      case (16)
        ! ----- K = 17 , R = 5E2 -----

        RIni = 5.0e2_wp

        Omega(1) = 0.0022886748517332675470407850120899140300_wp
        Omega(2) = 0.0056236828345040892299872893091150505285_wp
        Omega(3) = 0.0097928733037316318863112012647886839289_wp
        Omega(4) = 0.0156304076643632762245599053876898842930_wp
        Omega(5) = 0.0244733657769561815717288856542044328535_wp
        Omega(6) = 0.0382981921967690830207087411551913902485_wp
        Omega(7) = 0.0599135544832279009344676128090423361527_wp
        Omega(8) = 0.0933646423079849068792416232065445313992_wp
        Omega(9) = 0.1445828113357693121241191389070301909214_wp
        Omega(10) = 0.2223064676335977724112554290702981063532_wp
        Omega(11) = 0.3394170592751431185637913529040687876659_wp
        Omega(12) = 0.5150227219028765968935820729335262058157_wp
        Omega(13) = 0.7781099899771061128650809224360784810415_wp
        Omega(14) = 1.1752094119195155936738461655011178663699_wp
        Omega(15) = 1.7907855704489536393959198123937426316843_wp
        Omega(16) = 2.8203860853302943445357342522328281120281_wp
        Omega(17) = 5.0097774964476492884010716544196384347742_wp

        Alpha(1) = 0.0008863304499981981883709391222252865772_wp
        Alpha(2) = 0.0047974786871564526520171547335769091092_wp
        Alpha(3) = 0.0124079377825491436129434178088637441562_wp
        Alpha(4) = 0.0249336715434232541519198869325868450630_wp
        Alpha(5) = 0.0446635089640648912607059656948438330915_wp
        Alpha(6) = 0.0755321053695595088337704815784068301809_wp
        Alpha(7) = 0.1238433200134318948399856329112278530147_wp
        Alpha(8) = 0.1992838302546514328686183201155834865403_wp
        Alpha(9) = 0.3164642939227481399687771923545653862675_wp
        Alpha(10) = 0.4972394929080259180147020880990993418891_wp
        Alpha(11) = 0.7741388461621440731687260328186539481976_wp
        Alpha(12) = 1.1954666247789219785157069364522897103598_wp
        Alpha(13) = 1.8331629335494571537935534766283751650917_wp
        Alpha(14) = 2.7959834160555098192275230717385170464695_wp
        Alpha(15) = 4.2554107039169150219874715457279990005190_wp
        Alpha(16) = 6.5116887625723135928947071793260192862363_wp
        Alpha(17) = 10.2518864229407280098255172973154003557283_wp

        Xi(1) = 1.004962001499464992540500141160_wp
        Xi(2) = 1.044977580151059749096788265010_wp
        Xi(3) = 1.126722704247385133135596346765_wp
        Xi(4) = 1.253680609469466672076763680277_wp
        Xi(5) = 1.431245779640285588959341955828_wp
        Xi(6) = 1.666993061062927686429013807157_wp
        Xi(7) = 1.971098133211909397659311604656_wp
        Xi(8) = 2.356915394487746377542858322585_wp
        Xi(9) = 2.841721367904107638409361147325_wp
        Xi(10) = 3.447652619593606006893546367387_wp
        Xi(11) = 4.202903960451469030090709821934_wp
        Xi(12) = 5.143275280425625909166975624487_wp
        Xi(13) = 6.314173305843842203075727548267_wp
        Xi(14) = 7.773199104086760474578710056193_wp
        Xi(15) = 9.593492509756293343606281531777_wp
        Xi(16) = 11.868059118960482900030639630273_wp
        Xi(17) = 14.715365041488026962707147404785_wp
        Xi(18) = 18.286576135684652033921993030674_wp
        Xi(19) = 22.774907593839932982981988374149_wp
        Xi(20) = 28.427674789536183496546639659641_wp
        Xi(21) = 35.561705766153302608889585911811_wp
        Xi(22) = 44.582789009799400105349409528799_wp
        Xi(23) = 56.009584642621762607533897337930_wp
        Xi(24) = 70.501591449897556128367792638301_wp
        Xi(25) = 88.888433213876038553602487723992_wp
        Xi(26) = 112.192237130073681747322034141234_wp
        Xi(27) = 141.622499745848777277790198070306_wp
        Xi(28) = 178.497019942348070675741666946124_wp
        Xi(29) = 223.992181977449276178426806893640_wp
        Xi(30) = 278.548065385866678078663483120181_wp
        Xi(31) = 340.690436025609171000194663747607_wp
        Xi(32) = 405.210313026239249023507937863542_wp
        Xi(33) = 461.593492653376612833149295056501_wp
        Xi(34) = 495.445182885228008445865910402972_wp

      case (17)
        ! ----- K = 17 , R = 6E2 -----

        RIni = 6.0e2_wp

        Omega(1) = 0.0019677354874388323126610430549229824315_wp
        Omega(2) = 0.0048530017461425127139060600733877492985_wp
        Omega(3) = 0.0085126416938917754928699342491238422070_wp
        Omega(4) = 0.0137382055891998971393217409894815794758_wp
        Omega(5) = 0.0218033447099512608183946712689382785300_wp
        Omega(6) = 0.0345924123273276516630462141393964614622_wp
        Omega(7) = 0.0548067294939218097934313395774852395448_wp
        Omega(8) = 0.0863880720233744228825074722155630269071_wp
        Omega(9) = 0.1351781625335746134860138295552545173450_wp
        Omega(10) = 0.2098517476303381749149759225026024012095_wp
        Omega(11) = 0.3232773411303883744831282054388310598370_wp
        Omega(12) = 0.4946371851374084474160450208835015928344_wp
        Omega(13) = 0.7531150452829313788889568914708405600322_wp
        Omega(14) = 1.1455673736824923487822441847683307969419_wp
        Omega(15) = 1.7568239419680497032582797145572328645358_wp
        Omega(16) = 2.7824917704868392244292596426902264283854_wp
        Omega(17) = 4.9664538413426434467282111206998251873301_wp

        Alpha(1) = 0.0007617277641131662537388027423620323608_wp
        Alpha(2) = 0.0041306178060617515049843023657721863273_wp
        Alpha(3) = 0.0107221676748737044701690972713092886437_wp
        Alpha(4) = 0.0216724338810658121518888273235314834153_wp
        Alpha(5) = 0.0391384125653803920169484121399650078388_wp
        Alpha(6) = 0.0668450206227083926173036972084862128440_wp
        Alpha(7) = 0.1107848373700356455916118253446356334280_wp
        Alpha(8) = 0.1802267143396127727230466783470141933776_wp
        Alpha(9) = 0.2892662900064681361111430801802413270707_wp
        Alpha(10) = 0.4591652019510440568275059375347169066117_wp
        Alpha(11) = 0.7218187960719698762304154249225263129119_wp
        Alpha(12) = 1.1249245553693138130989248590907436664565_wp
        Alpha(13) = 1.7399490705067022409090193857394979204400_wp
        Alpha(14) = 2.6754381832962549421679743222313163641957_wp
        Alpha(15) = 4.1030244474367591057777093777048094125348_wp
        Alpha(16) = 6.3233632286307359252411353089939893834526_wp
        Alpha(17) = 10.0233206331863437229048874321790663088905_wp

        Xi(1) = 1.005138133768391998923430463719_wp
        Xi(2) = 1.046586623977852883809157302775_wp
        Xi(3) = 1.131325789969696443541292973922_wp
        Xi(4) = 1.263101222153942230315079586767_wp
        Xi(5) = 1.447720944925196325621039039611_wp
        Xi(6) = 1.693358206297682679746831324596_wp
        Xi(7) = 2.011024911865095328574945532374_wp
        Xi(8) = 2.415224231346351723174678549633_wp
        Xi(9) = 2.924788043159102948189906334697_wp
        Xi(10) = 3.563941057090348815379957092553_wp
        Xi(11) = 4.363669818757127066496948053498_wp
        Xi(12) = 5.363498547425370321407867013619_wp
        Xi(13) = 6.613801940837145031115706927949_wp
        Xi(14) = 8.178813554929238998671137128582_wp
        Xi(15) = 10.140539593635202664755057266888_wp
        Xi(16) = 12.603854383814765595904494599466_wp
        Xi(17) = 15.703138793478456480823268748281_wp
        Xi(18) = 19.610947775420321985764826600018_wp
        Xi(19) = 24.549307826606576128647407841754_wp
        Xi(20) = 30.804424423886318250981330990612_wp
        Xi(21) = 38.745748023126332211518718651178_wp
        Xi(22) = 48.850437586964519650484239932098_wp
        Xi(23) = 61.734121123044980544020132740002_wp
        Xi(24) = 78.188094893991741805849837732012_wp
        Xi(25) = 99.220802039093330987329366621452_wp
        Xi(26) = 126.095335720632240765481935085290_wp
        Xi(27) = 160.340420935767275320560365514666_wp
        Xi(28) = 203.681021008724827389690936740863_wp
        Xi(29) = 257.769764353614304480677432707125_wp
        Xi(30) = 323.489331488499119365842560114288_wp
        Xi(31) = 399.476596947846901403122998885920_wp
        Xi(32) = 479.667550385982437655130183884467_wp
        Xi(33) = 550.870382015631614713857544529674_wp
        Xi(34) = 594.146087176383738637230891299623_wp

      case (18)
        ! ----- K = 17 , R = 7E2 -----

        RIni = 7.0e2_wp

        Omega(1) = 0.0017314513798382410054079192989959223858_wp
        Omega(2) = 0.0042844078892813092310028655514131390447_wp
        Omega(3) = 0.0075644514054617839744789589868922963234_wp
        Omega(4) = 0.0123283620338962461271574487346736059834_wp
        Omega(5) = 0.0197968832453667010148765309854312910431_wp
        Omega(6) = 0.0317784158517143378902449889905623692243_wp
        Omega(7) = 0.0508873401078118539732429281147174648936_wp
        Omega(8) = 0.0809808457726174684535610896751745002575_wp
        Omega(9) = 0.1278245489500008482867217038392304573335_wp
        Omega(10) = 0.2000357218363171675452405820494483634775_wp
        Omega(11) = 0.3104654243775561357797983935702390567712_wp
        Omega(12) = 0.4783501053541849459382461623446758380851_wp
        Omega(13) = 0.7330321104341378530416806214020652987529_wp
        Omega(14) = 1.1216396204718298984515101324177521746606_wp
        Omega(15) = 1.7293175542134885413515787977445370415808_wp
        Omega(16) = 2.7517493678958247474344162020187809503113_wp
        Omega(17) = 4.9313241617443771235007599074151585227810_wp

        Alpha(1) = 0.0006700144767931550113013450524924313756_wp
        Alpha(2) = 0.0036392566808145860040259373692494992980_wp
        Alpha(3) = 0.0094776144199785698788994253889683694325_wp
        Alpha(4) = 0.0192575417621667895166508209861544465014_wp
        Alpha(5) = 0.0350294747050865863994238702527894702143_wp
        Alpha(6) = 0.0603470189797091049485968945048641742801_wp
        Alpha(7) = 0.1009485795206532420984716497125432965731_wp
        Alpha(8) = 0.1657623660703450222379416387208372896112_wp
        Alpha(9) = 0.2684625388540077777345480708470759623196_wp
        Alpha(10) = 0.4298211884370964110113622069775374257006_wp
        Alpha(11) = 0.6812043753394366205154014659584049695695_wp
        Alpha(12) = 1.0697957652746826374837510464921308539488_wp
        Alpha(13) = 1.6666522422174740947426255233132508237759_wp
        Alpha(14) = 2.5801264890974790111501796507909034517070_wp
        Alpha(15) = 3.9819627692099408776925384412237463038764_wp
        Alpha(16) = 6.1731642832369503933430399111870201522834_wp
        Alpha(17) = 9.8404776509214518590007436671385221416131_wp

        Xi(1) = 1.005286176376527788943297803126_wp
        Xi(2) = 1.047939757148727411405902953945_wp
        Xi(3) = 1.135200657171558425253578283876_wp
        Xi(4) = 1.271042874503931015338180354757_wp
        Xi(5) = 1.461635169709039568314637436153_wp
        Xi(6) = 1.715674062618721101738765433264_wp
        Xi(7) = 2.044906733392796121693765765137_wp
        Xi(8) = 2.464849224935444491719691018083_wp
        Xi(9) = 2.995711766438607066538948875412_wp
        Xi(10) = 3.663579554499805840438522719360_wp
        Xi(11) = 4.501935846331491267342805873142_wp
        Xi(12) = 5.553655006528252460317601757467_wp
        Xi(13) = 6.873605486015021498891891527094_wp
        Xi(14) = 8.532053582891235763742410203037_wp
        Xi(15) = 10.619115462439049015104519657626_wp
        Xi(16) = 13.250588265517025468587242187368_wp
        Xi(17) = 16.575595739671516882368651479851_wp
        Xi(18) = 20.786626434963571747813193546506_wp
        Xi(19) = 26.132728277360467324802462130151_wp
        Xi(20) = 32.936842115132468303273505227935_wp
        Xi(21) = 41.618481612528138428863666575808_wp
        Xi(22) = 52.723195580967399619803925858719_wp
        Xi(23) = 66.960280462928516452547622606062_wp
        Xi(24) = 85.249489622445436716458022630150_wp
        Xi(25) = 108.775285677342645002108056928591_wp
        Xi(26) = 139.040746643135324389639606579294_wp
        Xi(27) = 177.897411385237572731266020298335_wp
        Xi(28) = 227.490570181090303772397653858661_wp
        Xi(29) = 289.980762853297981490596768594514_wp
        Xi(30) = 366.756282166663774346737625364767_wp
        Xi(31) = 456.665521755188760472110587329553_wp
        Xi(32) = 552.891549678987294980103683883499_wp
        Xi(33) = 639.531338122269526369478853666806_wp
        Xi(34) = 692.764683404033109026531889185208_wp

      case (19)
        ! ----- K = 17 , R = 8E2 -----

        RIni = 8.0e2_wp

        Omega(1) = 0.0015496548496931517238359808240517889733_wp
        Omega(2) = 0.0038461093612214815545657090782930009887_wp
        Omega(3) = 0.0068310098587557051586795740794055831202_wp
        Omega(4) = 0.0112319347377321021818114455022297182296_wp
        Omega(5) = 0.0182244744017269463180161729806694204115_wp
        Omega(6) = 0.0295529712709816183638784859930015436191_wp
        Omega(7) = 0.0477594128931904872012654313800927852185_wp
        Omega(8) = 0.0766296847109545783380572456233714717655_wp
        Omega(9) = 0.1218632320861166369499028333533985346548_wp
        Omega(10) = 0.1920250573615980014055559799146521982038_wp
        Omega(11) = 0.2999465851088126892951424978805974319584_wp
        Omega(12) = 0.4649051786447570906084334613206365816040_wp
        Omega(13) = 0.7163744310297716986601218591701467630628_wp
        Omega(14) = 1.1017145944498874467112198294316272040305_wp
        Omega(15) = 1.7063473740013236616295741465698654337757_wp
        Omega(16) = 2.7260403665683967831465106534949427441461_wp
        Omega(17) = 4.9019578207789707797667477784386846906273_wp

        Alpha(1) = 0.0005994659006691419248704380757022947002_wp
        Alpha(2) = 0.0032609293287467020271499801992165501474_wp
        Alpha(3) = 0.0085176833686744144688460190703582775029_wp
        Alpha(4) = 0.0173898979557887154129785069536784369859_wp
        Alpha(5) = 0.0318392619475952310470554928317765686074_wp
        Alpha(6) = 0.0552756247918590500062133510916773104782_wp
        Alpha(7) = 0.0932243061540419517489296924450670189799_wp
        Alpha(8) = 0.1543281517252368470748428649907957321830_wp
        Alpha(9) = 0.2519068290953717686368826950138455345041_wp
        Alpha(10) = 0.4063175936906405903053032813154743507766_wp
        Alpha(11) = 0.6484737266112313892212938581938175275354_wp
        Alpha(12) = 1.0251141636009288428225960654316395448404_wp
        Alpha(13) = 1.6069349856798574150253469672300354886829_wp
        Alpha(14) = 2.5021106354488860776426328857979797248845_wp
        Alpha(15) = 3.8824710537171935030055502924284382970654_wp
        Alpha(16) = 6.0493196223628172046929418748106854764046_wp
        Alpha(17) = 9.6893344958846452834658191655137216002913_wp

        Xi(1) = 1.005413654823237422299316012886_wp
        Xi(2) = 1.049105445525520472144567762918_wp
        Xi(3) = 1.138541603761564504531181296620_wp
        Xi(4) = 1.277898631250481934688682794654_wp
        Xi(5) = 1.473665476796978640312667241385_wp
        Xi(6) = 1.735004750826417141526709919663_wp
        Xi(7) = 2.074320476489633381916161125602_wp
        Xi(8) = 2.508036732772011240949483479312_wp
        Xi(9) = 3.057604990244894586828156013425_wp
        Xi(10) = 3.750790110032816935918542622375_wp
        Xi(11) = 4.623342547573089557008674255201_wp
        Xi(12) = 5.721189601102591461810609185790_wp
        Xi(13) = 7.103314083744329841818565540734_wp
        Xi(14) = 8.845533412284985419656024863144_wp
        Xi(15) = 11.045464746113298605198482782441_wp
        Xi(16) = 13.829058959167053787300827316642_wp
        Xi(17) = 17.359208391703357896376780367120_wp
        Xi(18) = 21.847130258128196100136930724034_wp
        Xi(19) = 27.567408406556421137170720392362_wp
        Xi(20) = 34.877879052724904032634078987485_wp
        Xi(21) = 44.245909146905266665777967105555_wp
        Xi(22) = 56.282865179722242190973879871763_wp
        Xi(23) = 71.788758502353512427895942948908_wp
        Xi(24) = 91.808680306975094569110762421360_wp
        Xi(25) = 117.700284592173750844446011853961_wp
        Xi(26) = 151.205130977462468416794294512329_wp
        Xi(27) = 194.499468462767073267949413661881_wp
        Xi(28) = 250.158506656163433998218970089056_wp
        Xi(29) = 320.875806062613576408182325394591_wp
        Xi(30) = 408.597624661581310334668160066940_wp
        Xi(31) = 512.472510487944985790864649288778_wp
        Xi(32) = 625.029325618174673440119448741825_wp
        Xi(33) = 727.640516882454204627972416119519_wp
        Xi(34) = 791.308726273244767590586690175769_wp

      case (20)
        ! ----- K = 17 , R = 9E2 -----

        RIni = 9.0e2_wp

        Omega(1) = 0.0014051007766722965632607598800672688011_wp
        Omega(2) = 0.0034970054384386864617748899098162440602_wp
        Omega(3) = 0.0062449864228303118897558520328933129306_wp
        Omega(4) = 0.0103515614024025429393009094074240561412_wp
        Omega(5) = 0.0169531084327195187580564770049584844003_wp
        Omega(6) = 0.0277389784964547285802419175279598384520_wp
        Omega(7) = 0.0451895634357045944527622814007594698182_wp
        Omega(8) = 0.0730292751149963780156601153570328222031_wp
        Omega(9) = 0.1168990257280485887503122810482381055408_wp
        Omega(10) = 0.1853159622757152636730610534288032908989_wp
        Omega(11) = 0.2910909270989168994567871523448943094081_wp
        Omega(12) = 0.4535328802956832996539759256249269014916_wp
        Omega(13) = 0.7022264571587609558595000081027137639467_wp
        Omega(14) = 1.0847337738721048024060841785676245763170_wp
        Omega(15) = 1.6867230015056864829165600427884896816977_wp
        Omega(16) = 2.7040488024480957571574202269459874514723_wp
        Omega(17) = 4.8768459270337208614175261711665143593564_wp

        Alpha(1) = 0.0005433808812488208877660859951255756251_wp
        Alpha(2) = 0.0029599069426816397451533816346722072232_wp
        Alpha(3) = 0.0077526842167645549894012190709352561058_wp
        Alpha(4) = 0.0158978482337048581532544348771107678431_wp
        Alpha(5) = 0.0292814998075957901791841440919395012088_wp
        Alpha(6) = 0.0511903626026614958172220731791313141912_wp
        Alpha(7) = 0.0869674734537953759302649804630025442975_wp
        Alpha(8) = 0.1450115984650592445485799142357308255669_wp
        Alpha(9) = 0.2383379920535064063299339823243094826921_wp
        Alpha(10) = 0.3869453058931429341514400543911023078181_wp
        Alpha(11) = 0.6213522337346994388177813528617576821489_wp
        Alpha(12) = 0.9879063014460309506540626467430143975434_wp
        Alpha(13) = 1.5569814737710108527510288745077104977099_wp
        Alpha(14) = 2.4365870992827620088692908151273286421201_wp
        Alpha(15) = 3.7986202418785148370722337674010304908734_wp
        Alpha(16) = 5.9446475204785969056846262059679020239855_wp
        Alpha(17) = 9.5613115630708263590134543541410039324546_wp

        Xi(1) = 1.005525438161108560115690468972_wp
        Xi(2) = 1.050128008171806777404887200955_wp
        Xi(3) = 1.141474520126940505292123528847_wp
        Xi(4) = 1.283923503974767335675949420093_wp
        Xi(5) = 1.484252016805530110469396598560_wp
        Xi(6) = 1.752043382746376302071776576152_wp
        Xi(7) = 2.100295591246834598243148350960_wp
        Xi(8) = 2.546257639463843358606023881130_wp
        Xi(9) = 3.112511136419654979041321540834_wp
        Xi(10) = 3.828356034168831911095390307942_wp
        Xi(11) = 4.731621677436173593108564849885_wp
        Xi(12) = 5.871046812130794240614656276378_wp
        Xi(13) = 7.309418426379093028117522878873_wp
        Xi(14) = 9.127707868617433438138131229778_wp
        Xi(15) = 11.430524980468207067556274392217_wp
        Xi(16) = 14.353326518503221412466352990833_wp
        Xi(17) = 18.071966015722866323589546055928_wp
        Xi(18) = 22.815364200027887427515982388115_wp
        Xi(19) = 28.882348796409434076373767830148_wp
        Xi(20) = 36.664087349041270317423490965325_wp
        Xi(21) = 46.673877167377457425567754967233_wp
        Xi(22) = 59.586577118723999042793071367896_wp
        Xi(23) = 76.290295684899303726989039375894_wp
        Xi(24) = 97.952583465486922853493467755470_wp
        Xi(25) = 126.101502760988925619378697717821_wp
        Xi(26) = 162.715255360101360385116642248704_wp
        Xi(27) = 210.295647220470167421124685347422_wp
        Xi(28) = 271.854921729895204929317031883329_wp
        Xi(29) = 350.639961678737222844359777695900_wp
        Xi(30) = 449.199535578446821737497174353848_wp
        Xi(31) = 567.061310696411741250066995689849_wp
        Xi(32) = 696.194453365939922184768562374302_wp
        Xi(33) = 815.247943933204043731599597322202_wp
        Xi(34) = 889.784294879880762019208617630284_wp

      case (21)
        ! ----- K = 17 , R = 1E3 -----

        RIni = 1.0e3_wp

        Omega(1) = 0.0012871872080564161462051941541180621886_wp
        Omega(2) = 0.0032117941613826283106286114564162370577_wp
        Omega(3) = 0.0057648310783511739021668651705709862654_wp
        Omega(4) = 0.0096269441442492903807703398012807216588_wp
        Omega(5) = 0.0158999861928829474555648982106448308116_wp
        Omega(6) = 0.0262254171298502744835983314236615981940_wp
        Omega(7) = 0.0430303066441239794127156421954527587559_wp
        Omega(8) = 0.0699851440381639299686579447357104299954_wp
        Omega(9) = 0.1126783784596098899648199294365458200673_wp
        Omega(10) = 0.1795830790806989507682770931817906046035_wp
        Omega(11) = 0.2834892399462224134027831662363539066973_wp
        Omega(12) = 0.4437306366958131367177545212321376766340_wp
        Omega(13) = 0.6899874875249782166914368219945430382722_wp
        Omega(14) = 1.0700000486583649838851634017444780511141_wp
        Omega(15) = 1.6696584075417829916331349804359263089282_wp
        Omega(16) = 2.6849046641728076820557213899576254334534_wp
        Omega(17) = 4.8549914587310428900311654043520093182451_wp

        Alpha(1) = 0.0004976403321071418149077351185567696934_wp
        Alpha(2) = 0.0027142127504787087851420031682850297905_wp
        Alpha(3) = 0.0071273804248462571307361505725669337252_wp
        Alpha(4) = 0.0146754850874545199092890209076334162575_wp
        Alpha(5) = 0.0271791137483763283698965946416636896288_wp
        Alpha(6) = 0.0478178274002143230170551110869148203619_wp
        Alpha(7) = 0.0817762146607146612231381728463919245087_wp
        Alpha(8) = 0.1372408151319940840213518554180360808914_wp
        Alpha(9) = 0.2269611748749099650447700465827338689451_wp
        Alpha(10) = 0.3706210741259242424699895529505155877814_wp
        Alpha(11) = 0.5983902924464396798375810659909035393866_wp
        Alpha(12) = 0.9562673114382848842509113329590775265387_wp
        Alpha(13) = 1.5143353451316332415133372735205341541587_wp
        Alpha(14) = 2.3804501862485811080477615675476954493206_wp
        Alpha(15) = 3.7265623220764432311016833843098083889345_wp
        Alpha(16) = 5.8544721956401703294101124264159352605930_wp
        Alpha(17) = 9.4508090016707621099090186334024110692553_wp

        Xi(1) = 1.005624857260242493496062154978_wp
        Xi(2) = 1.051037776031361623777643510280_wp
        Xi(3) = 1.144085632307707296544639630209_wp
        Xi(4) = 1.289292354443618857074392580753_wp
        Xi(5) = 1.493697110192759300046488890601_wp
        Xi(6) = 1.767266808180364684585303414188_wp
        Xi(7) = 2.123542280620193702794784584853_wp
        Xi(8) = 2.580529014972632514133069059348_wp
        Xi(9) = 3.161846604169519828978623943172_wp
        Xi(10) = 3.898211775734842492223755638925_wp
        Xi(11) = 4.829376120796933432970099442372_wp
        Xi(12) = 6.006688148249366330632542654300_wp
        Xi(13) = 7.496478917611433989438146641149_wp
        Xi(14) = 9.384536767254948655750390607011_wp
        Xi(15) = 11.782034789642450507002324755490_wp
        Xi(16) = 14.833390401753585599586848786657_wp
        Xi(17) = 18.726715611333547695585566295051_wp
        Xi(18) = 23.707743580957714729304242062824_wp
        Xi(19) = 30.098449477587686113830289791338_wp
        Xi(20) = 38.321934724149690952577085312214_wp
        Xi(21) = 48.935716099509238622561335674277_wp
        Xi(22) = 62.676122825621992301625695631628_wp
        Xi(23) = 80.516920971003582085945282642570_wp
        Xi(24) = 103.745447494515662077707585098096_wp
        Xi(25) = 134.057539620691033771526790019379_wp
        Xi(26) = 173.666021406432282733756089498911_wp
        Xi(27) = 225.398301251110790052734067501206_wp
        Xi(28) = 292.709122982317816563657686401712_wp
        Xi(29) = 379.415345545763830259478055495492_wp
        Xi(30) = 488.706631794636579269042186979277_wp
        Xi(31) = 620.560744841132646776316050818423_wp
        Xi(32) = 766.477278600852961865097512372813_wp
        Xi(33) = 902.393918911106110325448526054970_wp
        Xi(34) = 988.196307010837813322723377496004_wp

      case (22)
        ! ----- K = 17 , R = 2E3 -----

        RIni = 2.0e3_wp

        Omega(1) = 0.0007225227877812019577221340623993106611_wp
        Omega(2) = 0.0018383479865653803184383474319044116108_wp
        Omega(3) = 0.0034284305708949454600021375209883967017_wp
        Omega(4) = 0.0060419410942769143976676456293806172226_wp
        Omega(5) = 0.0105698943758729995303125600951410500983_wp
        Omega(6) = 0.0183726837344387461827650235429709724144_wp
        Omega(7) = 0.0315655497760095967917321011647624473540_wp
        Omega(8) = 0.0534865300050024236877814921148344140533_wp
        Omega(9) = 0.0893792795913145530351104003918827345387_wp
        Omega(10) = 0.1474044567948891908891169769124651622860_wp
        Omega(11) = 0.2401668464046654445621063744820133933899_wp
        Omega(12) = 0.3870900495373424629347736819484637749156_wp
        Omega(13) = 0.6183966764795467233061833367546711315299_wp
        Omega(14) = 0.9829329898408706898222872427872687239869_wp
        Omega(15) = 1.5680614875995368081608438703966612592922_wp
        Omega(16) = 2.5704877366016645511351268460487062839093_wp
        Omega(17) = 4.7244846417528871085171615629150210224907_wp

        Alpha(1) = 0.0002787396918418467402997800634687625143_wp
        Alpha(2) = 0.0015350921123070499986556722033653743509_wp
        Alpha(3) = 0.0041106907929045474245271303732895962213_wp
        Alpha(4) = 0.0087293105825525938079909689035207343011_wp
        Alpha(5) = 0.0168277602925986761491498238177411117533_wp
        Alpha(6) = 0.0309525536197961524115517548474962872262_wp
        Alpha(7) = 0.0553564991160549083722630051701552034160_wp
        Alpha(8) = 0.0969722741399121798317492330887557017149_wp
        Alpha(9) = 0.1669550624771232238892663610196898105187_wp
        Alpha(10) = 0.2830599836152463728714337443115311998554_wp
        Alpha(11) = 0.4732691834136237801051137752406106073977_wp
        Alpha(12) = 0.7813315385337428963744704768501492253563_wp
        Alpha(13) = 1.2753824306906016022595401482320198738307_wp
        Alpha(14) = 2.0621523950909896566018736363545826861809_wp
        Alpha(15) = 3.3137957600148759240276524762336407547991_wp
        Alpha(16) = 5.3335920256305480212713487908615661581280_wp
        Alpha(17) = 8.8084384918989444088893714379651100898627_wp

        Xi(1) = 1.006262575581306089461675057350_wp
        Xi(2) = 1.056880453784974475159134932323_wp
        Xi(3) = 1.160893119463695852570592204067_wp
        Xi(4) = 1.323964518230208178947888975063_wp
        Xi(5) = 1.554949652075657058614144290232_wp
        Xi(6) = 1.866490580437049123926560034725_wp
        Xi(7) = 2.275954039898536564092562661799_wp
        Xi(8) = 2.806723223467505106808878023728_wp
        Xi(9) = 3.489882831473003923280357119197_wp
        Xi(10) = 4.366433744535447066502376056718_wp
        Xi(11) = 5.490259669265066431835531357564_wp
        Xi(12) = 6.932125403204579439705307342123_wp
        Xi(13) = 8.785070986148338006115066090018_wp
        Xi(14) = 11.171691225593531198410912619323_wp
        Xi(15) = 14.253998010224109346290310984529_wp
        Xi(16) = 18.246815733898321411007859005338_wp
        Xi(17) = 23.436036615298173094515399306204_wp
        Xi(18) = 30.203661117211572928464846654606_wp
        Xi(19) = 39.062191382856482409213771944678_wp
        Xi(20) = 50.702182707955384358133876787633_wp
        Xi(21) = 66.058100327335996439226395438027_wp
        Xi(22) = 86.399863409337979494595494145415_wp
        Xi(23) = 113.459959898623741085532667938196_wp
        Xi(24) = 149.609365930734299551807531258873_wp
        Xi(25) = 198.097441030071139300705596042462_wp
        Xi(26) = 263.369032168224073198192769496018_wp
        Xi(27) = 351.451765851933036560739154197108_wp
        Xi(28) = 470.335400337133903991837513558494_wp
        Xi(29) = 630.048224645127467491434458679578_wp
        Xi(30) = 841.541948179044426603390149921324_wp
        Xi(31) = 1112.097155178210803438965115219617_wp
        Xi(32) = 1432.683428867649838234221704169613_wp
        Xi(33) = 1753.896398834346094686154060582339_wp
        Xi(34) = 1969.488625452609237731493863066135_wp

      case (23)
        ! ----- K = 17 , R = 3E3 -----

        RIni = 3.0e3_wp

        Omega(1) = 0.0005152964505271765564264614505052275462_wp
        Omega(2) = 0.0013294206789940208072407632671277366576_wp
        Omega(3) = 0.0025467655464825499513393278032402644584_wp
        Omega(4) = 0.0046491772207381486777121753529995462628_wp
        Omega(5) = 0.0084201053589488962952905467228281732872_wp
        Omega(6) = 0.0150835542282891992472523725980770858968_wp
        Omega(7) = 0.0265996778777616203332499495576302095401_wp
        Omega(8) = 0.0461282368955676308087802441060265579154_wp
        Omega(9) = 0.0787150521253964747129335151332174547179_wp
        Omega(10) = 0.1323274885931812435205506134350272873235_wp
        Omega(11) = 0.2194327004038306663140974611669875571351_wp
        Omega(12) = 0.3594562657790958322705145849829122539631_wp
        Omega(13) = 0.5828720549538151937197030805215547388798_wp
        Omega(14) = 0.9391138289162590412268750983404430598966_wp
        Omega(15) = 1.5163956265285225614948202976783875328692_wp
        Omega(16) = 2.5119855345932346698829135123176570232317_wp
        Omega(17) = 4.6578206954465259386106990024956076013041_wp

        Alpha(1) = 0.0001984953473166476890669774536704421386_wp
        Alpha(2) = 0.0011007426306861563136964176821096239589_wp
        Alpha(3) = 0.0029892040618499704110536525870564505958_wp
        Alpha(4) = 0.0064860166922959568544541561745803148753_wp
        Alpha(5) = 0.0128390780889647865655977241640395281053_wp
        Alpha(6) = 0.0242824227240725237057252482164582119140_wp
        Alpha(7) = 0.0446106978535022276731599431831565283346_wp
        Alpha(8) = 0.0801320845535391342235372741376053795648_wp
        Alpha(9) = 0.1411881555222306924580457851425840942738_wp
        Alpha(10) = 0.2445218306526257932983440396279561923620_wp
        Alpha(11) = 0.4169320182125273602864908017995304589931_wp
        Alpha(12) = 0.7009081577243787522144523316480757557656_wp
        Alpha(13) = 1.1634433815360944908419427168411175443907_wp
        Alpha(14) = 1.9105417240992761178331946214647985016200_wp
        Alpha(15) = 3.1143693332951915468762710270667071199568_wp
        Alpha(16) = 5.0790073624075776853939445665275798091898_wp
        Alpha(17) = 8.4917280230121232331708114315915736369789_wp

        Xi(1) = 1.006619739302944042599885499101_wp
        Xi(2) = 1.060158072508502472160694940495_wp
        Xi(3) = 1.170351216673867388072885487027_wp
        Xi(4) = 1.343562385359251028964580609859_wp
        Xi(5) = 1.589766797573395107931286396319_wp
        Xi(6) = 1.923274091941266551882526447681_wp
        Xi(7) = 2.363865822398624000994990512758_wp
        Xi(8) = 2.938362404751409532894373954903_wp
        Xi(9) = 3.682681603539253699331965097308_wp
        Xi(10) = 4.644577889497503419766188947904_wp
        Xi(11) = 5.887354711068112623172099429070_wp
        Xi(12) = 7.494926230627882583583410092132_wp
        Xi(13) = 9.578720935855206871159528425608_wp
        Xi(14) = 12.287110066764472922178075275923_wp
        Xi(15) = 15.818347089976447673344284172714_wp
        Xi(16) = 20.438364951931099148391135500447_wp
        Xi(17) = 26.505412846161769114400330593639_wp
        Xi(18) = 34.504301665878660387160037359422_wp
        Xi(19) = 45.094273068701129687091411213373_wp
        Xi(20) = 59.176257812111765835988430595194_wp
        Xi(21) = 77.987888646878960580144379832745_wp
        Xi(22) = 103.238237950210646302895423076507_wp
        Xi(23) = 137.299727635986278995172682471093_wp
        Xi(24) = 183.481260528500435366683340987493_wp
        Xi(25) = 246.415335594635525343076842830214_wp
        Xi(26) = 332.597352445678344112778290764254_wp
        Xi(27) = 451.104268816801100616675768151254_wp
        Xi(28) = 614.448774360691819917956735253028_wp
        Xi(29) = 839.251117416957251426978814379254_wp
        Xi(30) = 1145.547130135411393769828691802104_wp
        Xi(31) = 1551.143749141165920857332594096079_wp
        Xi(32) = 2052.236901422372904013613492679724_wp
        Xi(33) = 2578.487415807036629455950560441124_wp
        Xi(34) = 2946.808860665281275537097371852724_wp

      case (24)
        ! ----- K = 17 , R = 4E3 -----

        RIni = 4.0e3_wp

        Omega(1) = 0.0004054777725141607820470602288826303394_wp
        Omega(2) = 0.0010578850551656629063238917705727648588_wp
        Omega(3) = 0.0020701844217904609134742508575952046890_wp
        Omega(4) = 0.0038805823916970897530412485266354538638_wp
        Omega(5) = 0.0072040529016436389688217401504333192719_wp
        Omega(6) = 0.0131795047771428243183253594186798984822_wp
        Omega(7) = 0.0236672199996987954476532082029316406135_wp
        Omega(8) = 0.0417073892595867673098236280693207334025_wp
        Omega(9) = 0.0722090454618636848398106550805941594717_wp
        Omega(10) = 0.1230011275658273160014282744667557523144_wp
        Omega(11) = 0.2064444437019107227604831217293579470606_wp
        Omega(12) = 0.3419477228333598406494284877021883062298_wp
        Omega(13) = 0.5601360518319130894055422886790296388426_wp
        Omega(14) = 0.9108314795281586088711127391359667626602_wp
        Omega(15) = 1.4828393697982242696933874448461665451759_wp
        Omega(16) = 2.4738623767994982998851688504871049190115_wp
        Omega(17) = 4.6143996077690712896701707990132490522228_wp

        Alpha(1) = 0.0001560036351335973021506571756710987642_wp
        Alpha(2) = 0.0008699543072648038101912277729119149505_wp
        Alpha(3) = 0.0023894041062470810685414011236352149758_wp
        Alpha(4) = 0.0052733900201038298668545503070786972799_wp
        Alpha(5) = 0.0106504372402373795990909981917793558637_wp
        Alpha(6) = 0.0205576655730831200731881756860186527547_wp
        Alpha(7) = 0.0385007191243840527126681728964374684665_wp
        Alpha(8) = 0.0703892927052088953276346577903055390379_wp
        Alpha(9) = 0.1260376002707051351875260081225604835709_wp
        Alpha(10) = 0.2215216365171963732773070115955604819646_wp
        Alpha(11) = 0.3828472658379700372418934228857523294209_wp
        Alpha(12) = 0.6516434522803097441432523817539390620368_wp
        Alpha(13) = 1.0941030700653212205794226341559749471344_wp
        Alpha(14) = 1.8156960891012023867965852197414733382175_wp
        Alpha(15) = 2.9885558338476809002014905614075246376160_wp
        Alpha(16) = 4.9172983979235954268664776645181291314657_wp
        Alpha(17) = 8.2895297475037491357827135196600920608034_wp

        Xi(1) = 1.006864795993849638036828075816_wp
        Xi(2) = 1.062409149010022496205027375815_wp
        Xi(3) = 1.176859446125301238692852090484_wp
        Xi(4) = 1.357084034823194351152171599484_wp
        Xi(5) = 1.613871195281853859293356989735_wp
        Xi(6) = 1.962747970103677018363988460159_wp
        Xi(7) = 2.425271773097594276043242889074_wp
        Xi(8) = 3.030810585827873909542037877962_wp
        Xi(9) = 3.818893481084848249857560675835_wp
        Xi(10) = 4.842363229090031085700973623176_wp
        Xi(11) = 6.171686397737105312689964708994_wp
        Xi(12) = 7.900873653821401534826790458155_wp
        Xi(13) = 10.155616445514733573417254230975_wp
        Xi(14) = 13.104493655119210267917528867088_wp
        Xi(15) = 16.974474146642133375387229143882_wp
        Xi(16) = 22.072463034425087974321755268647_wp
        Xi(17) = 28.815386232648649457790956951442_wp
        Xi(18) = 37.772481771556383908783338654302_wp
        Xi(19) = 49.725034894025339403389462944460_wp
        Xi(20) = 65.751329581906675474101131584348_wp
        Xi(21) = 87.348188737073600956817287510603_wp
        Xi(22) = 116.605765035151659789192102323341_wp
        Xi(23) = 156.460461679568579723698107386554_wp
        Xi(24) = 211.061770030358098179656956006056_wp
        Xi(25) = 286.304065218168480866367531234573_wp
        Xi(26) = 390.590591863310777898110615069527_wp
        Xi(27) = 535.900462295627864062197431849199_wp
        Xi(28) = 739.168563354513936447531818885182_wp
        Xi(29) = 1023.694540816035043528930970069268_wp
        Xi(30) = 1419.221922589301406736339572489669_wp
        Xi(31) = 1955.984512322953893015409221334266_wp
        Xi(32) = 2639.525469739646505784591568044561_wp
        Xi(33) = 3383.202563703130898886684008175507_wp
        Xi(34) = 3921.048508992051748967000435186492_wp

      case (25)
        ! ----- K = 17 , R = 5E3 -----

        RIni = 5.0e3_wp

        Omega(1) = 0.0003367471971412925717325832021585910181_wp
        Omega(2) = 0.0008870248350176340236647966525952713202_wp
        Omega(3) = 0.0017671250240551048071597017657084760511_wp
        Omega(4) = 0.0033837180880403332873391218128811264165_wp
        Omega(5) = 0.0064032001961904855644885665305610089604_wp
        Omega(6) = 0.0119045361137674933479232920914558935266_wp
        Omega(7) = 0.0216757082525989423132315976963657710996_wp
        Omega(8) = 0.0386681299913834293774785120373360669532_wp
        Omega(9) = 0.0676873864573128068182418688725210387247_wp
        Omega(10) = 0.1164553957031184914033762785312564602691_wp
        Omega(11) = 0.1972469731352802093331645985618827410235_wp
        Omega(12) = 0.3294489379855182067209293239606893166638_wp
        Omega(13) = 0.5437893337218036256047390408774333536712_wp
        Omega(14) = 0.8903748071315684746939410343369303291183_wp
        Omega(15) = 1.4584594866065747734195159890369097865914_wp
        Omega(16) = 2.4460979032403840764155200648843901944929_wp
        Omega(17) = 4.5827860157812262045101681096070933563169_wp

        Alpha(1) = 0.0001294266036966938828865212042745595766_wp
        Alpha(2) = 0.0007252109691484226381374347412869102447_wp
        Alpha(3) = 0.0020112414165164148180440470718363665803_wp
        Alpha(4) = 0.0045022054999761172525343116707780000141_wp
        Alpha(5) = 0.0092419249574203509365501597698500102496_wp
        Alpha(6) = 0.0181282009167901023078187366238678279728_wp
        Alpha(7) = 0.0344616528102501758019026670970541559313_wp
        Alpha(8) = 0.0638665778171708784202221555192924284938_wp
        Alpha(9) = 0.1157751886157385535694583009180824717532_wp
        Alpha(10) = 0.2057748375160446075110651925621496616259_wp
        Alpha(11) = 0.3592834836980452450894661653446604532292_wp
        Alpha(12) = 0.6172838037451011228393289720361991612663_wp
        Alpha(13) = 1.0453574126507787800952412049504403057654_wp
        Alpha(14) = 1.7485535610644321547141669115355000485579_wp
        Alpha(15) = 2.8989593755717999216340446944073505619599_wp
        Alpha(16) = 4.8015854096117595514736642314801429165527_wp
        Alpha(17) = 8.1443259141638506999616442527667459216900_wp

        Xi(1) = 1.007049693449281561061572909388_wp
        Xi(2) = 1.064108820854260653381538614948_wp
        Xi(3) = 1.181780171940103476018076034126_wp
        Xi(4) = 1.367327182203664896384918969741_wp
        Xi(5) = 1.632175495044025791526245972030_wp
        Xi(6) = 1.992811385457244366305878946477_wp
        Xi(7) = 2.472199028997921004978041703559_wp
        Xi(8) = 3.101735172081476675466291115235_wp
        Xi(9) = 3.923839983683448578120900318567_wp
        Xi(10) = 4.995456261526767576409274740357_wp
        Xi(11) = 6.392860937076527319544716476685_wp
        Xi(12) = 8.218312354975754578829960639297_wp
        Xi(13) = 10.609233067579239642025390466529_wp
        Xi(14) = 13.750953258771116952639823605153_wp
        Xi(15) = 17.894435370012756714511170663684_wp
        Xi(16) = 23.381093615075234671066284342089_wp
        Xi(17) = 30.677711474343777363696372795232_wp
        Xi(18) = 40.425903615724536832587432400032_wp
        Xi(19) = 53.512558827882373595902221197207_wp
        Xi(20) = 71.170966653390076259166896477382_wp
        Xi(21) = 95.126796142367268369066657385247_wp
        Xi(22) = 127.810396758401282120420905386027_wp
        Xi(23) = 172.667488498757912884773446648978_wp
        Xi(24) = 234.615989443755118035173090618173_wp
        Xi(25) = 320.719243800078100348205367708943_wp
        Xi(26) = 441.173611081260894029032471053142_wp
        Xi(27) = 610.730229110299273231898808944607_wp
        Xi(28) = 850.631048009837712475800231004541_wp
        Xi(29) = 1190.843552362957778911756179240911_wp
        Xi(30) = 1671.155079313161797238684869171266_wp
        Xi(31) = 2335.490910002118279287230961926980_wp
        Xi(32) = 3201.896500983435303044899455926497_wp
        Xi(33) 