// Copyright 2023 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "ash/shelf/shelf.h"
#include "ash/system/unified/date_tray.h"
#include "ash/test/ash_test_base.h"
#include "ash/test/pixel/ash_pixel_differ.h"
#include "ash/test/pixel/ash_pixel_test_helper.h"
#include "ash/test/pixel/ash_pixel_test_init_params.h"

namespace ash {

class DateTrayPixelTest
    : public AshTestBase,
      public testing::WithParamInterface</*enable_system_blur=*/bool> {
 public:
  DateTrayPixelTest() = default;
  DateTrayPixelTest(const DateTrayPixelTest&) = delete;
  DateTrayPixelTest& operator=(const DateTrayPixelTest&) = delete;
  ~DateTrayPixelTest() override = default;

  // AshTestBase:
  std::optional<pixel_test::InitParams> CreatePixelTestInitParams()
      const override {
    pixel_test::InitParams init_params;
    init_params.system_blur_enabled = GetParam();
    return init_params;
  }

 protected:
  DateTray* GetDateTray() {
    return GetPrimaryShelf()->GetStatusAreaWidget()->date_tray();
  }
};

INSTANTIATE_TEST_SUITE_P(
    /* no prefix */,
    DateTrayPixelTest,
    testing::Bool());

// Tests the inactive date tray UI for bottom shelf alignment and side shelf
// alignment.
TEST_P(DateTrayPixelTest, InactiveDateTrayInBottomAndSideShelfPositions) {
  auto* shelf = GetPrimaryShelf();

  // Tests the bottom shelf.
  shelf->SetAlignment(ShelfAlignment::kBottom);
  ASSERT_EQ(shelf->alignment(), ShelfAlignment::kBottom);
  auto* bottom_date_tray = GetDateTray();
  // By default `bottom_date_tray` should be inactive.
  ASSERT_FALSE(bottom_date_tray->is_active());
  EXPECT_TRUE(GetPixelDiffer()->CompareUiComponentsOnPrimaryScreen(
      GenerateScreenshotName("bottom_shelf_inactive_date_tray"),
      /*revision_number=*/pixel_test_helper()->IsSystemBlurEnabled() ? 1 : 0,
      bottom_date_tray));

  // Tests the side shelf.
  shelf->SetAlignment(ShelfAlignment::kLeft);
  ASSERT_EQ(shelf->alignment(), ShelfAlignment::kLeft);
  auto* side_date_tray = GetDateTray();
  // `side_date_tray` should remain inactive.
  ASSERT_FALSE(side_date_tray->is_active());
  EXPECT_TRUE(GetPixelDiffer()->CompareUiComponentsOnPrimaryScreen(
      GenerateScreenshotName("side_shelf_inactive_date_tray"),
      /*revision_number=*/pixel_test_helper()->IsSystemBlurEnabled() ? 1 : 0,
      side_date_tray));
}

// Tests the active date tray UI for bottom shelf alignment and side shelf
// alignment.
TEST_P(DateTrayPixelTest, ActiveDateTrayInBottomAndSideShelfPositions) {
  auto* shelf = GetPrimaryShelf();

  // Tests the bottom shelf.
  shelf->SetAlignment(ShelfAlignment::kBottom);
  ASSERT_EQ(shelf->alignment(), ShelfAlignment::kBottom);
  auto* bottom_date_tray = GetDateTray();
  // Sets `bottom_date_tray` to be active in the bottom shelf.
  bottom_date_tray->SetIsActive(/*is_active=*/true);
  ASSERT_TRUE(bottom_date_tray->is_active());
  EXPECT_TRUE(GetPixelDiffer()->CompareUiComponentsOnPrimaryScreen(
      GenerateScreenshotName("bottom_shelf_active_date_tray"),
      /*revision_number=*/pixel_test_helper()->IsSystemBlurEnabled() ? 1 : 0,
      bottom_date_tray));

  // Tests the side shelf.
  shelf->SetAlignment(ShelfAlignment::kLeft);
  ASSERT_EQ(shelf->alignment(), ShelfAlignment::kLeft);
  auto* side_date_tray = GetDateTray();
  // Sets `side_date_tray` to be active in the side shelf.
  side_date_tray->SetIsActive(/*is_active=*/true);
  ASSERT_TRUE(side_date_tray->is_active());
  EXPECT_TRUE(GetPixelDiffer()->CompareUiComponentsOnPrimaryScreen(
      GenerateScreenshotName("side_shelf_active_date_tray"),
      /*revision_number=*/pixel_test_helper()->IsSystemBlurEnabled() ? 1 : 0,
      side_date_tray));
}

}  // namespace ash
